/* NVTV NV TV-Xbox I2C access -- Milosch Meriac <xboxlinux@meriac.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 *
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * I2C-related by code - Andy Green <andy@warmcat.com>
 *
 * Adapted for nvtv - "Milosch Meriac" <xboxlinux@meriac.de>
 *  
 */

#include <sys/io.h>

#include "local.h"
#include "xbox.h"

/* FIXME TODO Can the IO_BASE be read from the PCI registers? */

const unsigned short I2C_IO_BASE=0xC000;

inline void IoOutputWord(unsigned short Port,unsigned short Data)
{
    outw(Data,I2C_IO_BASE+Port);
};

inline void IoOutputByte(unsigned short Port,unsigned char Data)
{
    outb(Data,I2C_IO_BASE+Port);
};

inline unsigned short IoInputWord(unsigned short Port)
{
    return inw(I2C_IO_BASE+Port);
};

inline unsigned char IoInputByte(unsigned short Port)
{
    return inb(I2C_IO_BASE+Port);
};

// ----------------------------  I2C -----------------------------------------------------------
//
// get a value from a given device address
// errors will have b31 set, ie, will be negative, otherwise fetched byte in LSB of return

int I2CTransmitCmdGetReturn(unsigned char bPicAddressI2cFormat, unsigned char Cmd)
{
	unsigned int dwRetriesToLive=4;

	while(dwRetriesToLive--) {
		unsigned int dwSpinsToLive=0x8000000;

		IoOutputByte(4, bPicAddressI2cFormat|1);
		IoOutputByte(8, Cmd);
		IoOutputWord(0, IoInputWord(0));
		IoOutputByte(2, 0x0a);

		{
			unsigned char b=8;
			while( (b & 8) && (dwSpinsToLive--) ) { b=IoInputByte(0); }
			if(dwSpinsToLive==0) return ERR_I2C_ERROR_TIMEOUT;
			if(b&0x2) continue; // retry
			if(b&0x24) return ERR_I2C_ERROR_BUS;
			if(!(b&0x10)) return ERR_I2C_ERROR_BUS;

				// we are okay, fetch returned byte
			return (int)IoInputByte(6);

		}
	}
	return ERR_I2C_ERROR_BUS;
}

// transmit a word, no returned data from I2C device

int I2CTransmitCmdData(unsigned char bPicAddressI2cFormat, unsigned char Cmd, unsigned char Data, int fMode)
{
	unsigned int dwRetriesToLive=4;

	while(dwRetriesToLive--) {
		unsigned int dwSpinsToLive=0x8000000;

		IoOutputByte(4, bPicAddressI2cFormat & ~1);
		IoOutputByte(8, Cmd);
		IoOutputWord(6, Data);
		IoOutputWord(0, IoInputWord(0));
		if(fMode) {
			IoOutputByte(2, 0x1a);
		} else {
			IoOutputByte(2, 0x0a);
		}

		{
			unsigned char b=8;
			while( (b & 8) && (dwSpinsToLive--) ) { b=IoInputByte(0); }
			if(dwSpinsToLive==0) return ERR_I2C_ERROR_TIMEOUT;
			if(b&0x2) continue; // retry
			if(b&0x24) return ERR_I2C_ERROR_BUS;
			if(!(b&0x10)) return ERR_I2C_ERROR_BUS;

				// we are okay, return happy code
			return ERR_SUCCESS;
		}
	}
	return ERR_I2C_ERROR_BUS;
}

void I2CTransmitStop(void)
{
    IoOutputWord(2, (IoInputWord(2) & 0x10)|0x20);
}

