# ifndef _SKIT_QMR_H
# define _SKIT_QMR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cmath>

/*Class:qmr
NAME: @code{qmr} -- quasi-minimal residual algoritm 
@findex qmr
@cindex quasi-minimal residual algorithm
@cindex iterative solver
SYNOPSIS:
  @example
	template <class Matrix, class Vector, class Preconditioner1,
            class Preconditioner2, class Real>
    	int qmr (const Matrix &A, Vector &x, const Vector &b,
    	    const Preconditioner1 &M1, const Preconditioner2 &M2,
    	    int &max_iter, Real &tol);
  @end example

EXAMPLE:
  @noindent
  The simplest call to 'qmr' has the folling form:
  @example
  	int status = qmr(a, x, b, EYE, EYE, 100, 1e-7);
  @end example

DESCRIPTION:       

  @noindent
  @code{qmr} solves the unsymmetric linear system Ax = b 
  using the the quasi-minimal residual method.

  @noindent
  The return value indicates convergence within max_iter (input)
  iterations (0), or no convergence within max_iter iterations (1).
  Upon successful return, output arguments have the following values:
  @table @code
    @itemx x
	approximate solution to Ax = b

    @itemx max_iter
	the number of iterations performed before the tolerance was reached

    @itemx tol
	the residual after the final iteration
  @end table

  @noindent
  A return value of 1 indicates that the method did not reach the
  specified convergence tolerance in the maximum numbefr of iterations.
  A return value of 2 indicates that a breackdown associated with @code{rho} occurred. 
  A return value of 3 indicates that a breackdown associated with @code{beta} occurred. 
  A return value of 4 indicates that a breackdown associated with @code{gamma} occurred. 
  A return value of 5 indicates that a breackdown associated with @code{delta} occurred. 
  A return value of 6 indicates that a breackdown associated with @code{epsilon} occurred. 
  A return value of 7 indicates that a breackdown associated with @code{xi} occurred. 
   
NOTE: 

  @noindent
  @code{qmr} is an iterative template routine.

  @noindent
  @code{qmr} follows the algorithm described on p. 24 in
  @quotation
	Templates for the Solution of Linear Systems: Building Blocks for Iterative Methods, 
	2nd Edition, 
        R. Barrett, M. Berry, T. F. Chan, J. Demmel, J. Donato, J. Dongarra, V. Eijkhout,
	R. Pozo, C. Romine, H. Van der Vorst,
        SIAM, 1994, 
	@url{ftp.netlib.org/templates/templates.ps}.
  @end quotation

  @noindent
  The present implementation is inspired from @code{IML++ 1.2} iterative method library,
  @url{http://math.nist.gov/iml++}.

AUTHOR: 

    Pierre Saramito
    | Pierre.Saramito@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France

DATE: 
    
    12 march 1997

End:
*/
// ========== [ method implementation ] ====================================
namespace rheolef { 

template < class Matrix, class Vector, class Preconditioner1,
           class Preconditioner2, class Real >
int 
qmr(const Matrix &A, Vector &x, const Vector &b, const Preconditioner1 &M1, 
    const Preconditioner2 &M2, int &max_iter, Real &tol)
{
  Real resid;

  Vector rho(1), rho_1(1), xi(1), gamma(1), gamma_1(1), theta(1), theta_1(1);
  Vector eta(1), delta(1), ep(1), beta(1);

  Vector r, v_tld, y, w_tld, z;
  Vector v, w, y_tld, z_tld;
  Vector p, q, p_tld, d, s;

  Real normb = norm(b);

  r = b - A * x;

  if (normb == Real(0))
    normb = 1;

  if ((resid = norm(r) / normb) <= tol) {
    tol = resid;
    max_iter = 0;
    return 0;
  }

  v_tld = r;
  y = M1.solve(v_tld);
  rho(0) = norm(y);

  w_tld = r;
  z = M2.trans_solve(w_tld);
  xi(0) = norm(z);

  gamma(0) = 1.0;
  eta(0) = -1.0;
  theta(0) = 0.0;

  for (int i = 1; i <= max_iter; i++) {

    if (rho(0) == Real(0))
      return 2;                        // return on breakdown

    if (xi(0) == Real(0))
      return 7;                        // return on breakdown

    v = (1. / rho(0)) * v_tld;
    y = (1. / rho(0)) * y;

    w = (1. / xi(0)) * w_tld;
    z = (1. / xi(0)) * z;

    delta(0) = dot(z, y);
    if (delta(0) == Real(0))
      return 5;                        // return on breakdown

    y_tld = M2.solve(y);               // apply preconditioners
    z_tld = M1.trans_solve(z);

    if (i > 1) {
      p = y_tld - (xi(0) * delta(0) / ep(0)) * p;
      q = z_tld - (rho(0) * delta(0) / ep(0)) * q;
    } else {
      p = y_tld;
      q = z_tld;
    }

    p_tld = A * p;
    ep(0) = dot(q, p_tld);
    if (ep(0) == Real(0))
      return 6;                        // return on breakdown

    beta(0) = ep(0) / delta(0);
    if (beta(0) == Real(0))
      return 3;                        // return on breakdown

    v_tld = p_tld - beta(0) * v;
    y = M1.solve(v_tld);

    rho_1(0) = rho(0);
    rho(0) = norm(y);
    w_tld = A.trans_mult(q) - beta(0) * w;
    z = M2.trans_solve(w_tld);

    xi(0) = norm(z);

    gamma_1(0) = gamma(0);
    theta_1(0) = theta(0);

    theta(0) = rho(0) / (gamma_1(0) * beta(0));
    gamma(0) = 1.0 / ::sqrt(1.0 + theta(0) * theta(0));

    if (gamma(0) == Real(0))
      return 4;                        // return on breakdown

    eta(0) = -eta(0) * rho_1(0) * gamma(0) * gamma(0) / 
      (beta(0) * gamma_1(0) * gamma_1(0));

    if (i > 1) {
      d = eta(0) * p + (theta_1(0) * theta_1(0) * gamma(0) * gamma(0)) * d;
      s = eta(0) * p_tld + (theta_1(0) * theta_1(0) * gamma(0) * gamma(0)) * s;
    } else {
      d = eta(0) * p;
      s = eta(0) * p_tld;
    }

    x += d;                            // update approximation vector
    r -= s;                            // compute residual

    if ((resid = norm(r) / normb) <= tol) {
      tol = resid;
      max_iter = i;
      return 0;
    }
  }

  tol = resid;
  return 1;                            // no convergence
}
}// namespace rheolef
# endif // _SKIT_QMR_H
