/*
** Copyright 1998 - 2001 Double Precision, Inc.
** See COPYING for distribution information.
*/

#include	"config.h"
#include	"maildirquota.h"
#include	"maildircreate.h"
#include	"maildirmisc.h"
#include	"quotawarnmsg.h"
#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<sys/types.h>
#include	<errno.h>
#if HAVE_SYS_STAT_H
#include	<sys/stat.h>
#endif
#if HAVE_UNISTD_H
#include	<unistd.h>
#endif
#if	HAVE_FCNTL_H
#include	<fcntl.h>
#endif
#include	<time.h>
#if	HAVE_SYSEXITS_H
#include	<sysexits.h>
#endif
#include	"rfc822/rfc822.h"
#ifndef	BUFSIZE
#define	BUFSIZE	8192
#endif

#ifndef	EX_OSERR
#define	EX_OSERR	71
#endif

#ifndef	EX_IOERR
#define	EX_IOERR	74
#endif

#ifndef	EX_TEMPFAIL
#define EX_TEMPFAIL	75
#endif

#ifndef	EX_NOPERM
#define	EX_NOPERM	77
#endif

static const char rcsid[]="$Id: deliverquota.c,v 1.15 2001/02/18 20:58:06 mrsam Exp $";

static long deliver(int fd, const char *dir, const char *q, long s,
		    int auto_create, int quota_warn_percent, const char *pfix);

static void do_deliver_warning(const char *, const char *);

static void deliver_warning(const char *dir, const char *q)
{
	size_t l;
	char *p;
	struct stat	sb;

	/* If we delivered to a folder, dump the warning message into INBOX */

	l = strlen(dir)+sizeof("/maildirfolder");
	if ((p = malloc(l)) == 0)
	{
		perror("malloc");
		exit(EX_TEMPFAIL);
	}

	strcat(strcpy(p, dir), "/maildirfolder");

	/* If delivering to a folder, find quotawarn in its parent directory */

	if (stat(p, &sb) == 0)
	{
		strcat(strcpy(p, dir), "/..");
		do_deliver_warning(p, q);
		free(p);
		return;
	}
	free(p);
	do_deliver_warning(dir, q);
}

static void do_deliver_warning(const char *dir, const char *q)
{
int	fdin, fd;
size_t	l;
char	*qname = 0, *tname = 0, *nname = 0;
time_t	t;
struct stat	sb;
char	hostname[256];
char	buf[4096];
size_t	n;

	if ((fdin=open(QUOTAWARNMSG, O_RDONLY)) < 0)
		return;

	l = strlen(dir)+sizeof("/quotawarn");

	/* Send only one warning every 24 hours */
	if ((qname = malloc(l)) == 0)
	{
		close(fdin);
		perror("malloc");
		exit(EX_TEMPFAIL);
	}

	strcat(strcpy(qname, dir), "/quotawarn");
	time(&t);
	if (stat(qname, &sb) == 0 && ((sb.st_mtime + 86400) > t))
	{
		free(qname);
		close(fdin);
		return;
	}

	fd = open(qname, O_WRONLY|O_CREAT|O_TRUNC, 0644);
	if (!fd)
	{
		free(qname);
		close(fdin);
		perror("open");
		exit(EX_IOERR);
	}

	write(fd, buf, 0);
	close(fd);

	if (maildir_try_create(dir, "quota", fstat(fdin, &sb) ? 0UL :
			       (unsigned long)sb.st_size, &tname, &nname))
	{
		free(qname);
		close(fdin);
		perror("maildir_try_create");
		exit(EX_IOERR);
	}

	if ((fd=maildir_safeopen(tname, O_RDWR|O_CREAT|O_EXCL, 0644)) < 0)
	{
		if (tname)
			free(tname);
		if (nname)
			free(nname);
		free(qname);
		close(fdin);
		perror(tname);
		exit(EX_IOERR);
	}

	strcpy(buf, "Date: ");
	rfc822_mkdate_buf(t, buf+strlen(buf));
	strcat(buf, "\n");
	write(fd, buf, strlen(buf));
	hostname[0]=0;
	hostname[sizeof(hostname)-1]=0;
	gethostname(hostname, sizeof(hostname)-1);

	sprintf(buf, "Message-Id: <%lu.overquota@%-1.256s>\n",
		(unsigned long)t, hostname);
	l=strlen(buf);
	write(fd, buf, l);
	while ((n=read(fdin, buf, sizeof(buf))) > 0)
	{
	char	*p = buf;

		while (n)
		{
		int	l;

			if ((l=write(fd, p, n)) < 0)
			{
				close(fd);
				if (tname)
					free(tname);
				if (nname)
					free(nname);
				free(qname);
				close(fdin);
				perror("write");
				exit(EX_IOERR);
			}
			p += l;
			n -= l;
		}
	}

	close(fdin);
	if (lseek(fd, 0L, SEEK_SET) >= 0)
	{
		/* Make sure the quota message's size itself is factored into
		** the quota.
		*/
		deliver(fd, dir, q, 0, 0, -1, "warn");
	}
	close(fd);
	unlink(tname);

	if (tname) free (tname);
	if (nname) free (nname);
}

static long deliver(int fdin, const char *dir, const char *q, long s,
	int auto_create, int quota_warn_percent, const char *pfix)
{
char	*tname, *nname;
char	buf[BUFSIZ];
int	n;
long	ss=0;
int	fd;
int	rc;

	while ((rc=maildir_try_create(dir, pfix, s, &tname, &nname)) != 0)
	{
		if (rc < 0)
		{
			perror("maildir_try_create");
			exit(EX_TEMPFAIL);
		}
		sleep(3);
	}

	fd=maildir_safeopen(tname, O_WRONLY|O_CREAT|O_EXCL, 0644);
	if (fd < 0 && errno == ENOENT && auto_create
	    && maildir_mkdir(dir) == 0)
	{
		fd=maildir_safeopen(tname, O_WRONLY|O_CREAT|O_EXCL, 0644);
	}
	if (fd < 0)
	{
		perror(tname);
		exit(EX_TEMPFAIL);
	}

	while ((n=read(fdin, buf, sizeof(buf))) > 0)
	{
	char	*p=buf;

		ss += n;
		while (n)
		{
		int	l;

			if ((l=write(fd, p, n)) < 0)
			{
				close(fd);
				unlink(tname);
				perror(tname);
				exit(EX_IOERR);
			}
			p += l;
			n -= l;
		}
	}
	close(fd);
	if (n < 0)
	{
		unlink(tname);
		perror(tname);
		exit(EX_IOERR);
	}

	if (s != ss)
	{
	char	*qq;
	int	quotafd;

		if (s)	*strrchr(nname, ',')=0;	/* Zap incorrect size */
		qq=malloc(strlen(nname)+100);
		if (!qq)
		{
			unlink(tname);
			perror(tname);
			exit(EX_OSERR);
		}
		sprintf(qq, "%s,S=%ld", nname, ss-s);
		free(nname);
		nname=qq;

		if (*q)
		{
			if (maildir_checkquota(dir, &quotafd, q, ss-s, 1)
				&& errno != EAGAIN)
			{
				if (quotafd >= 0)	close(quotafd);
				unlink(tname);
				printf("Mail quota exceeded.\n");
				exit(EX_NOPERM);
			}
			maildir_addquota(dir, quotafd, q, ss-s, 1);
			if (quotafd >= 0)	close(quotafd);
		}
	}

	if (rename(tname, nname))
	{
		unlink(tname);
		perror(tname);
		exit(EX_IOERR);
	}
	if (quota_warn_percent >= 0 && maildir_readquota(dir, q) >= quota_warn_percent)
	{
		deliver_warning(dir, q);
	}
	return (ss);
}

int main(int argc, char **argv)
{
const char *dir;
const char *q;
struct	stat	stat_buf;
int	quotafd;
int	auto_create = 0;
int	quota_warn_percent = -1;

	while (argc >= 3) {
		if (!strcmp(argv[1], "-c"))
		{
			auto_create = 1;
			argv++, argc--;
		}
		else if (!strcmp(argv[1], "-w"))
		{
			quota_warn_percent = atoi(argv[2]);
			argv+=2, argc-=2;
		}
		else break;
	}
	if (argc < 3 || quota_warn_percent < -1 || quota_warn_percent > 100)
	{
		fprintf(stderr, "Usage: %s [-c] [-w percent] maildir quota\n", argv[0]);
		exit(73);
	}

	dir=argv[1];
	q=argv[2];

	if (fstat(0, &stat_buf) == 0 && S_ISREG(stat_buf.st_mode) &&
		stat_buf.st_size > 0 && *q)
	{
		if (maildir_checkquota(dir, &quotafd, q, stat_buf.st_size, 1)
			&& errno != EAGAIN)
		{
			if (quotafd >= 0)	close(quotafd);
			printf("Mail quota exceeded.\n");
			exit(77);
		}
		maildir_addquota(dir, quotafd, q, stat_buf.st_size, 1);
		if (quotafd >= 0)       close(quotafd);
		deliver(0, dir, q, stat_buf.st_size,
			auto_create, quota_warn_percent, NULL);
		exit(0);
	}
	deliver(0, dir, q, 0, auto_create, quota_warn_percent, NULL);
	exit(0);
}
