/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB

   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.

   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.	The License grants you the right to
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* drop and alter of tables */

#include "mysql_priv.h"
#include "handler.h"
#include <hash.h>
#ifdef __WIN32__
#include <io.h>
#endif

extern HASH open_cache;

static bool check_if_keyname_exists(const char *name,KEY *start, KEY *end);
static char *make_unique_key_name(char *field_name,KEY *start,KEY *end);
static int copy_data_between_tables(TABLE *from,TABLE *to,
				    List<create_field> &create,
				    enum enum_duplicates handle_duplicates,
				    ulong *copied,ulong *deleted);

/*****************************************************************************
** Remove all possbile tables and give a compact errormessage for all
** wrong tables.
*****************************************************************************/

int mysql_rm_table(THD *thd,TABLE_LIST *tables)
{
  char	path[FN_REFLEN];
  String wrong_tables;
  bool some_tables_deleted=0;
  DBUG_ENTER("mysql_rm_table");

  /* mark for close and remove all cached entries */

  VOID(pthread_mutex_lock(&LOCK_open));
  for (TABLE_LIST *table=tables ; table ; table=table->next)
  {
    remove_table_from_cache(thd->db,table->real_name);

    /* remove form file and isam files */
    (void) sprintf(path,"%s/%s/%s%s",mysql_data_home,thd->db,table->real_name,
		   reg_ext);
    (void) unpack_filename(path,path);
    if (access(path,F_OK))
    {
      if (wrong_tables.length())
	wrong_tables.append(',');
      wrong_tables.append(String(table->real_name));
    }
    else
    {
      VOID(my_delete(path,MYF(MY_WME)));
      sprintf(path,"%s/%s/%s",mysql_data_home,thd->db,table->real_name);
      VOID(ha_fdelete(DB_TYPE_ISAM,path));	// Is fixed here
      some_tables_deleted=1;
    }
  }
  if (some_tables_deleted)
    mysql_update_log.write(thd->query);
  VOID(pthread_mutex_unlock(&LOCK_open));
  if (wrong_tables.length())
  {
    my_error(ER_BAD_TABLE_ERROR,MYF(0),wrong_tables.c_ptr());
    DBUG_RETURN(-1);
  }
  send_ok(&thd->net);
  DBUG_RETURN(0);
}


int quick_rm_table(THD *thd,enum db_type base,const char *table_name)
{
  char path[FN_REFLEN];
  int error=0;
  (void) sprintf(path,"%s/%s/%s%s",mysql_data_home,thd->db,table_name,reg_ext);
  unpack_filename(path,path);
  if (my_delete(path,MYF(0)))
    error=1; /* purecov: inspected */
  sprintf(path,"%s/%s/%s",mysql_data_home,thd->db,table_name);
  return ha_fdelete(base,path) || error;
}


int mysql_create_table(THD *thd,const char *table_name,
		       List<create_field> &fields,
		       List<Key> &keys,bool tmp_table)
{
  char		path[FN_REFLEN],*key_name;
  create_field	*sql_field,*dup_field;
  int		error;
  uint		db_options,field;
  ulong		pos;
  KEY	*key_info,*key_info_buffer;
  KEY_PART_INFO *key_part_info;
  int		auto_increment=0;
  DBUG_ENTER("mysql_create_table");

  /*
  ** Check for dupplicate fields and check type of table to create
  */

  if (!fields.elements)
  {
    my_error(ER_TABLE_MUST_HAVE_COLUMNS,MYF(0));
    DBUG_RETURN(-1);
  }
  List_iterator<create_field> it(fields),it2(fields);
  pos=1;					// Offset if fixed size record
  db_options=HA_OPTION_PACK_KEYS;
  while ((sql_field=it++))
  {
    if (sql_field->sql_type == FIELD_TYPE_BLOB ||
	sql_field->sql_type == FIELD_TYPE_VAR_STRING)
    {
      pos=0;					// Can use first pos in table
      db_options|=HA_OPTION_PACK_RECORD;
    }
    while ((dup_field=it2++) != sql_field)
    {
      if (my_strcasecmp(sql_field->field_name, dup_field->field_name) == 0)
      {
	my_error(ER_DUP_FIELDNAME,MYF(0),sql_field->field_name);
	DBUG_RETURN(-1);
      }
    }
    it2.rewind();
  }
  it.rewind();
  while ((sql_field=it++))
  {
    switch (sql_field->sql_type) {
    case FIELD_TYPE_BLOB:
    case FIELD_TYPE_MEDIUM_BLOB:
    case FIELD_TYPE_TINY_BLOB:
    case FIELD_TYPE_LONG_BLOB:
      sql_field->pack_flag=FIELDFLAG_BLOB |
	pack_length_to_packflag(sql_field->pack_length - sizeof(char*));
      if (sql_field->flags & BINARY_FLAG)
	sql_field->pack_flag|=FIELDFLAG_BINARY;
      sql_field->length=8;			// Unireg field length
      sql_field->unireg_check=Field::BLOB_FIELD;
      break;
    case FIELD_TYPE_VAR_STRING:
    case FIELD_TYPE_STRING:
      sql_field->pack_flag=0;
      if (sql_field->flags & BINARY_FLAG)
	sql_field->pack_flag|=FIELDFLAG_BINARY;
      break;
    case FIELD_TYPE_ENUM:
      sql_field->pack_flag=pack_length_to_packflag(sql_field->pack_length) |
	FIELDFLAG_INTERVAL;
      sql_field->unireg_check=Field::INTERVAL_FIELD;
      break;
    case FIELD_TYPE_SET:
      sql_field->pack_flag=pack_length_to_packflag(sql_field->pack_length) |
	FIELDFLAG_BITFIELD;
      sql_field->unireg_check=Field::BIT_FIELD;
      break;
    case FIELD_TYPE_DATE:			// Rest of string types
    case FIELD_TYPE_TIME:
    case FIELD_TYPE_DATETIME:
    case FIELD_TYPE_NULL:
      sql_field->pack_flag=f_settype((uint) sql_field->sql_type);
      break;
    default:
      sql_field->pack_flag=(FIELDFLAG_NUMBER |
			    (sql_field->flags & UNSIGNED_FLAG ? 0 :
			     FIELDFLAG_DECIMAL) |
			    (sql_field->flags & ZEROFILL_FLAG ?
			     FIELDFLAG_ZEROFILL : 0) |
			    f_settype((uint) sql_field->sql_type) |
			    (sql_field->decimals << FIELDFLAG_DEC_SHIFT));
      break;
    }
    if (!(sql_field->flags & NOT_NULL_FLAG))
      sql_field->pack_flag|=FIELDFLAG_MAYBE_NULL;
    sql_field->offset= pos;
    if (MTYP_TYPENR(sql_field->unireg_check) == Field::NEXT_NUMBER)
      auto_increment++;
    pos+=sql_field->pack_length;
  }
  if (auto_increment > 1)
  {
    my_error(ER_WRONG_AUTO_KEY,MYF(0));
    DBUG_RETURN(-1);
  }

  /* Create keys */
  List_iterator<Key> key_iterator(keys);
  uint key_parts=0,key_count=keys.elements;
  bool primary_key=0;
  Key *key;

  if (key_count > MAX_KEY)
  {
    my_error(ER_TOO_MANY_KEYS,MYF(0),MAX_KEY);
    DBUG_RETURN(-1);
  }
  while ((key=key_iterator++))
  {
    if (key->columns.elements > MAX_REF_PARTS)
    {
      my_error(ER_TOO_MANY_KEY_PARTS,MYF(0),MAX_REF_PARTS);
      DBUG_RETURN(-1);
    }
    key_parts+=key->columns.elements;
  }
  key_info_buffer=key_info=(KEY*) sql_calloc(sizeof(KEY)*key_count);
  key_part_info=(KEY_PART_INFO*) sql_calloc(sizeof(KEY_PART_INFO)*key_parts);
  key_iterator.rewind();
  for (; (key=key_iterator++) ; key_info++)
  {
    uint key_length=0;
    key_part_spec *column;
    if (key->type == Key::PRIMARY)
    {
      if (primary_key)
      {
	my_error(ER_MULTIPLE_PRI_KEY,MYF(0));
	DBUG_RETURN(-1);
      }
      primary_key=1;
    }
    key_info->dupp_key=test(key->type == Key::MULTIPLE);
    key_info->key_parts=(uint8) key->columns.elements;
    key_info->key_part=key_part_info;

    if (key->type == Key::PRIMARY)
      key_name="PRIMARY";
    else if (!(key_name = key->name()))
      key_name=make_unique_key_name(key->columns.head()->field_name,
				    key_info_buffer,key_info);
    if (check_if_keyname_exists(key_name,key_info_buffer,key_info))
    {
      my_error(ER_DUP_KEYNAME,MYF(0),key_name);
      DBUG_RETURN(-1);
    }
    key_info->name=key_name;

    List_iterator<key_part_spec> cols(key->columns);
    for (uint column_nr=0 ; (column=cols++) ; column_nr++)
    {
      it.rewind();
      field=0;
      while ((sql_field=it++) &&
	     my_strcasecmp(column->field_name,sql_field->field_name))
	field++;
      if (!sql_field)
      {
	my_error(ER_KEY_COLUMN_DOES_NOT_EXITS,MYF(0),column->field_name);
	DBUG_RETURN(-1);
      }
      if (f_is_blob(sql_field->pack_flag))
      {
	my_error(ER_BLOB_USED_AS_KEY,MYF(0),column->field_name);
	DBUG_RETURN(-1);
      }
      // NULL keys not allowed
      sql_field->pack_flag&= ~FIELDFLAG_MAYBE_NULL;
      sql_field->flags|=NOT_NULL_FLAG;
      if (MTYP_TYPENR(sql_field->unireg_check) == Field::NEXT_NUMBER &&
	  column_nr == 0)
	auto_increment--;			// Fields is used
      key_part_info->fieldnr= field;
      key_part_info->offset=  (uint16) sql_field->offset;
      key_part_info->key_type=sql_field->pack_flag;
      uint length=sql_field->pack_length;
      if (column->length)
      {
	if (column->length > length ||
	    (f_is_packed(sql_field->pack_flag) && column->length != length))
	{
	  my_error(ER_WRONG_SUB_KEY,MYF(0));
	  DBUG_RETURN(-1);
	}
	length=column->length;
      }
      key_part_info->length=(uint8) length;
      key_length+=length;
      key_part_info++;
    }
    key_info->key_length=(uint16) key_length;
    if (key_length > ha_max_key_length)
    {
      my_error(ER_TOO_LONG_KEY,MYF(0),ha_max_key_length);
      DBUG_RETURN(-1);
    }
  }
  if (auto_increment > 0)
  {
    my_error(ER_WRONG_AUTO_KEY,MYF(0));
    DBUG_RETURN(-1);
  }

      /* Check if table exists */
  (void) sprintf(path,"%s/%s/%s%s",mysql_data_home,thd->db,table_name,reg_ext);
  unpack_filename(path,path);
  VOID(pthread_mutex_lock(&LOCK_open));
  if (!tmp_table)
 {
    if (!access(path,F_OK))
    {
      VOID(pthread_mutex_unlock(&LOCK_open));
      my_error(ER_TABLE_EXISTS_ERROR,MYF(0),table_name);
      DBUG_RETURN(-1);
    }
  }
  if (rea_create_table(path,DB_TYPE_ISAM,db_options,0L,0L,
		       fields,key_count,key_info_buffer))
  {
    /* my_error(ER_CANT_CREATE_TABLE,MYF(0),table_name,my_errno); */
    error=-1;
    VOID(pthread_mutex_unlock(&LOCK_open));
    goto end;
  }
  mysql_update_log.write(thd->query);
  VOID(pthread_mutex_unlock(&LOCK_open));
  error=0;
end:
  DBUG_RETURN(error);
}

/*
** Give the key name after the first field with an optional '_#' after
**/

static bool
check_if_keyname_exists(const char *name, KEY *start, KEY *end)
{
  for (KEY *key=start ; key != end ; key++)
    if (!my_strcasecmp(name,key->name))
      return 1;
  return 0;
}


static char *
make_unique_key_name(char *field_name,KEY *start,KEY *end)
{
  char buff[MAX_FIELD_NAME],*buff_end;

  if (!check_if_keyname_exists(field_name,start,end))
    return field_name;				// Use fieldname
  buff_end=strmake(buff,field_name,MAX_FIELD_NAME-4);
  for (uint i=2 ; ; i++)
  {
    sprintf(buff_end,"_%d",i);
    if (!check_if_keyname_exists(buff,start,end))
      return sql_strdup(buff);
  }
}


/****************************************************************************
** Alter a table definition
****************************************************************************/

static int
mysql_rename_table(THD *thd,enum db_type base,const char * old_name,
		   const char * new_name)
{
  char from[FN_REFLEN],to[FN_REFLEN];
  DBUG_ENTER("mysql_rename_table");
  (void) sprintf(from,"%s/%s/%s",mysql_data_home,thd->db,old_name);
  (void) sprintf(to,"%s/%s/%s",mysql_data_home,thd->db,new_name);

  if (rename_file_ext(from,to,reg_ext) ||
      ha_frename(base,(const char*) from,(const char *) to) ||
      fix_frm_ref(to))
   DBUG_RETURN(1);
 DBUG_RETURN(0);
}


int mysql_alter_table(THD *thd,char *new_name, TABLE_LIST *table_list,
		      List<create_field> &fields,
		      List<Key> &keys,List<Alter_drop> &drop_list,
		      List<Alter_column> &alter_list,
		      bool drop_primary,
		      enum enum_duplicates handle_duplicates)
{
  TABLE *table,*new_table;
  int error;
  char tmp_name[80],old_name[32],new_name_buff[FN_REFLEN],
    *table_name=table_list->real_name;
  bool use_timestamp=0;
  ulong copied,deleted;
  DBUG_ENTER("mysql_alter_table");

  /* Check that we are not trying to rename to an existing table */
  if (new_name)
  {
    strmov(new_name_buff,new_name);
    new_name=new_name_buff;
    fn_same(new_name_buff,table_name,3);
    if (!strcmp(new_name_buff,table_name))	// Check if name changed
      new_name=table_name;			// No. Make later check easier
    else
    {
      if (!access(fn_format(new_name_buff,new_name_buff,"",reg_ext,0),F_OK))
      {
	my_error(ER_TABLE_EXISTS_ERROR,MYF(0),new_name);
	DBUG_RETURN(-1);
      }
    }
  }
  else
    new_name=table_name;
    
  if (!(table=open_ltable(thd,table_list,0)))
    DBUG_RETURN(-1);

  /* Check if the user only wants to do a simple RENAME */

  if (new_name != table_name &&
      !fields.elements && !keys.elements && ! drop_list.elements &&
      !alter_list.elements && !drop_primary)
  {

    /* Then do a 'simple' rename of the table */
    error=0;
    VOID(pthread_mutex_lock(&LOCK_open));
    if (!access(new_name,F_OK))
    {
      my_error(ER_TABLE_EXISTS_ERROR,MYF(0),new_name);
      error= -1;
    }
    else
    {
      *fn_ext(new_name)=0;
      remove_table_from_cache(thd->db,table->real_name);
      if (mysql_rename_table(thd,table->db_type,table_name,new_name))
	error= -1;
    }
    if (!error)
    {
      if (thd->lock)
      {
	mysql_unlock_tables(thd->lock); thd->lock=0;	// Start locked threads
      }
      else
	mysql_lock_remove(thd->locked_tables,table);
      thd->open_tables=unlink_open_table(thd->open_tables,table);
      VOID(hash_delete(&open_cache,(byte*) table)); 	// Close table	
      send_ok(&thd->net);
    }
    VOID(pthread_mutex_unlock(&LOCK_open));
    DBUG_RETURN(error);
  }


  /* Full alter table */
  restore_record(table,2);			// Empty record for DEFAULT
  List_iterator<Alter_drop> drop_it(drop_list);
  List_iterator<create_field> def_it(fields);
  List_iterator<Alter_column> alter_it(alter_list);
  List<create_field> create_list;		// Add new fields here
  List<Key> key_list;				// Add new keys here

  /*
  ** First collect all fields from table which isn't in drop_list
  */

  create_field *def;
  Field **f_ptr,*field;
  for (f_ptr=table->field ; (field= *f_ptr) ; f_ptr++)
  {
    /* Check if field should be droped */
    Alter_drop *drop;
    drop_it.rewind();
    while ((drop=drop_it++))
    {
      if (drop->type == Alter_drop::COLUMN &&
	  !my_strcasecmp(field->field_name, drop->name))
	break;
    }
    if (drop)
    {
      drop_it.remove();
      continue;
    }
    /* Check if field is changed */
    def_it.rewind();
    while ((def=def_it++))
    {
      if (def->change && !my_strcasecmp(field->field_name, def->change))
	break;
    }
    if (def)
    {						// Field is changed
      def->field=field;
      if (def->sql_type == FIELD_TYPE_TIMESTAMP)
	use_timestamp=1;
      create_list.push_back(def);
      def_it.remove();
    }
    else
    {						// Use old field value
      create_list.push_back(def=new create_field(field));
      if (def->sql_type == FIELD_TYPE_TIMESTAMP)
	use_timestamp=1;

      alter_it.rewind();			// Change default if ALTER
      Alter_column *alter;
      while ((alter=alter_it++))
      {
	if (!my_strcasecmp(field->field_name, alter->name))
	  break;
      }
      if (alter)
      {
	def->def=alter->def;			// Use new default
	alter_it.remove();
      }
    }
  }
  def_it.rewind();
  while ((def=def_it++))			// Add new columns
  {
    if (def->change)
    {
      my_error(ER_BAD_FIELD_ERROR,MYF(0),def->change,table_name);
      DBUG_RETURN(-1);
    }
    create_list.push_back(def);
  }
  if (alter_list.elements)
  {
    my_error(ER_BAD_FIELD_ERROR,MYF(0),alter_list.head()->name,table_name);
    DBUG_RETURN(-1);
  }
  if (!create_list.elements)
  {
    my_error(ER_CANT_REMOVE_ALL_FIELDS,MYF(0));
    DBUG_RETURN(-1);
  }

  /*
  ** Collect all keys which isn't in drop list. Add only those
  ** for which some fields exists.
  */

  List_iterator<Key> key_it(keys);
  List_iterator<create_field> field_it(create_list);
  List<key_part_spec> key_parts;

  KEY *key_info=table->key_info;
  for (uint i=0 ; i < table->keys ; i++,key_info++)
  {
    if (drop_primary && !key_info->dupp_key)
    {
      drop_primary=0;
      continue;
    }

    char *key_name=key_info->name;
    Alter_drop *drop;
    drop_it.rewind();
    while ((drop=drop_it++))
    {
      if (drop->type == Alter_drop::KEY && !my_strcasecmp(key_name, drop->name))
	break;
    }
    if (drop)
    {
      drop_it.remove();
      continue;
    }

    KEY_PART_INFO *key_part= key_info->key_part;
    key_parts.empty();
    for (uint j=0 ; j < key_info->key_parts ; j++,key_part++)
    {
      if (!key_part->field)
	continue;				// Wrong field (from UNIREG)
      char *key_part_name=key_part->field->field_name;
      create_field *field;
      field_it.rewind();
      while ((field=field_it++))
      {
	if (!my_strcasecmp(key_part_name, field->change))
	  break;
      }
      if (!field)
	continue;				// Field is removed
      uint key_part_length=key_part->length;
      if (field->field)				// Not new field
      {						// Check if sub key
	if (field->field->pack_length() == key_part_length ||
	    field->length != field->pack_length)
	  key_part_length=0;			// Use hole field
      }
      key_parts.push_back(new key_part_spec(field->field_name,key_part_length));
    }
    if (key_parts.elements)
      key_list.push_back(new Key(key_info->dupp_key ? Key::MULTIPLE :
				 Key::UNIQUE,
				 key_name,key_parts));
  }
  key_it.rewind();
  Key *key;
  while ((key=key_it++))			// Add new keys
    key_list.push_back(key);

  if (drop_list.elements)
  {
    my_error(ER_CANT_DROP_FIELD_OR_KEY,MYF(0),drop_list.head()->name);
    DBUG_RETURN(-1);
  }
  if (alter_list.elements)
  {
    my_error(ER_CANT_DROP_FIELD_OR_KEY,MYF(0),alter_list.head()->name);
    DBUG_RETURN(-1);
  }

  (void) sprintf(tmp_name,"A-%lx", thd->thread_id);
  if ((error=mysql_create_table(thd,tmp_name,create_list,key_list,1)))
    DBUG_RETURN(error);
  if (!(new_table=open_tmp_table(thd,thd->db,tmp_name)))
  {
    VOID(quick_rm_table(thd,DB_TYPE_ISAM,tmp_name));
    DBUG_RETURN(-1);
  }

  uint save_time_stamp=new_table->time_stamp;
  if (use_timestamp)
    new_table->time_stamp=0;
  new_table->next_number_field=new_table->found_next_number_field;
  thd->count_cuted_fields=1;			/* calc cuted fields */
  thd->cuted_fields=0L;
  error=copy_data_between_tables(table,new_table,create_list,handle_duplicates,
				 &copied,&deleted);
  thd->count_cuted_fields=0;			/* Don`t calc cuted fields */
  new_table->time_stamp=save_time_stamp;

  intern_close_table(new_table);		/* close temporary table */
  my_free((gptr) new_table,MYF(0));
  VOID(pthread_mutex_lock(&LOCK_open));
  if (error)
  {
    if (thd->lock)
    {
      mysql_unlock_tables(thd->lock);
      thd->lock=0;
    }
    VOID(pthread_mutex_unlock(&LOCK_open));
    VOID(quick_rm_table(thd,DB_TYPE_ISAM,tmp_name));
    DBUG_RETURN(-1);
  }
  /*
  ** Data is copied.  Now we rename the old table to a temp name,
  ** rename the new one to the old name, remove all entries from the old table
  ** from the cash, free all locks, close the old table and remove it.
  */

  sprintf(old_name,"B-%lx", thd->thread_id);
  error=0;
  if (new_name != table_name)
  {
    if (!access(new_name,F_OK))
    {
      error=1;
      my_error(ER_TABLE_EXISTS_ERROR,MYF(0),new_name);
    }
    else
      *fn_ext(new_name)=0;			// remove extension
  }
  if (!error && mysql_rename_table(thd,table->db_type,table_name,old_name))
  {
    error=1;
    VOID(quick_rm_table(thd,DB_TYPE_ISAM,tmp_name));
  }
  else if (mysql_rename_table(thd,DB_TYPE_ISAM,tmp_name,new_name))
  {						// Try to get everything back
    error=1;
    VOID(quick_rm_table(thd,DB_TYPE_ISAM,new_name));
    VOID(quick_rm_table(thd,DB_TYPE_ISAM,tmp_name));
    VOID(mysql_rename_table(thd,table->db_type,old_name,table_name));
  }
  if (error)
  {
    if (thd->lock)
    {
      mysql_unlock_tables(thd->lock);
      thd->lock=0;
    }
    thd->open_tables=unlink_open_table(thd->open_tables,table);
    remove_table_from_cache(thd->db,table_name);
    VOID(hash_delete(&open_cache,(byte*) table));	 // Close table
    VOID(pthread_mutex_unlock(&LOCK_open));
    DBUG_RETURN(-1);
  }
  VOID(ha_extra(table,HA_EXTRA_FORCE_REOPEN));		// Use new file
  if (thd->lock || new_name != table_name)
  {
    thd->open_tables=unlink_open_table(thd->open_tables,table);
    if (thd->lock)
    {
      mysql_unlock_tables(thd->lock); thd->lock=0;	// Start locked threads
    }
    else
      mysql_lock_remove(thd->locked_tables,table);
    remove_table_from_cache(thd->db,table_name);
    VOID(quick_rm_table(thd,table->db_type,old_name));
    VOID(hash_delete(&open_cache,(byte*) table)); 	// Close table
  }
  else
  {							// Using LOCK TABLES
    mysql_lock_remove(thd->locked_tables,table);
    VOID(quick_rm_table(thd,table->db_type,old_name));
    if (reopen_tables(thd,table_name))
    {							// This shouldn't happen
      thd->open_tables=unlink_open_table(thd->open_tables,table);
      VOID(hash_delete(&open_cache,(byte*) table));	// Close table
      VOID(pthread_mutex_unlock(&LOCK_open));
      DBUG_RETURN(-1);
    }
  }

  mysql_update_log.write(thd->query);
  VOID(pthread_mutex_unlock(&LOCK_open));

  sprintf(tmp_name,ER(ER_INSERT_INFO),copied+deleted,deleted,
	  thd->cuted_fields);
  send_ok(&thd->net,copied+deleted,0L,tmp_name);
  thd->some_tables_deleted=0;
  DBUG_RETURN(0);
}


static int
copy_data_between_tables(TABLE *from,TABLE *to,List<create_field> &create,
			 enum enum_duplicates handle_duplicates,
			 ulong *copied,ulong *deleted)
{
  int error;
  Copy_field *copy,*copy_end;
  ulong found_count,delete_count;
  DBUG_ENTER("copy_data_between_tables");

  ha_lock(to,F_WRLCK);
  ha_extra(to,HA_EXTRA_WRITE_CACHE);

  if (!(copy= new Copy_field[to->fields]))
    DBUG_RETURN(-1);				/* purecov: inspected */

  List_iterator<create_field> it(create);
  create_field *def;
  copy_end=copy;
  for (Field **ptr=to->field ; *ptr ; ptr++)
  {
    def=it++;
    if (def->field)
      (copy_end++)->set(*ptr,def->field);
  }

  READ_RECORD info;
  init_read_record(&info, from, (SQL_SELECT *) 0);

  TABLE *error_table=from;
  found_count=delete_count=0;
  Field *next_field=to->next_number_field;
  while ((error=info.read_record(&info)) <= 0)
  {
    if (!error)
    {
      found_count++;
      if (next_field)
	next_field->reset();
      for (Copy_field *ptr=copy ; ptr != copy_end ; ptr++)
	ptr->do_copy(ptr);
      if ((error=ha_write(to,(byte*) to->record[0])))
      {
	if (handle_duplicates != DUP_IGNORE ||
	    error != HA_ERR_FOUND_DUPP_KEY)
	{
	  error_table=to;
	  break;
	}
	delete_count++;
      }
    }
  }
  end_read_record(&info);
  delete [] copy;
  uint tmp_error;
  if ((tmp_error=ha_extra(to,HA_EXTRA_NO_CACHE)))
  {
    error=tmp_error;
    error_table=to;
  }
  if (error != HA_ERR_END_OF_FILE)
  {
    ha_error(error_table,error,MYF(0));
    DBUG_RETURN(-1);
  }
  *copied= found_count;
  *deleted=delete_count;
  DBUG_RETURN(0);
}
