/*
 **************************************************************************
 *
 * Boot-ROM-Code to load an operating system across a TCP/IP network.
 *
 * Module:  bootp.c
 * Purpose: Get information for client with BOOTP protocol
 * Entries: bootp
 *
 **************************************************************************
 *
 * Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../../headers/general.h"
#include "../public/net.h"
#include "../public/arpa.h"
#include "../public/romlib.h"
#include "./arpapriv.h"
#include "./bootp.h"



/*
 **************************************************************************
 * 
 * We allocate enough space for a bootp record and an extension file of
 * up to 2048 bytes, which is 4 TFTP buffers.
 */
#define BOOT_SIZE	(MAX_EXT_BLOCKS * SEGSIZE + sizeof(struct bootp) + 2)



/*
 **************************************************************************
 * 
 * Global variables:
 */
static struct bootp *boot_rec;			/* bootp record		*/
static unsigned long boot_xid;			/* bootp transaction ID	*/



/*
 **************************************************************************
 * 
 * Return information from the vendor area of the BOOTP record. Note
 * that this will work only with RFC1048-compliant messages.
 *
 */
unsigned char *get_vend(struct bootp *brec, int id)
{
  unsigned char *cp, *endp;

  /*
   * If the bootp record has been modified, there is no ending pointer. In
   * that case we assume that the vendor information field is correctly
   * terminated with an ending tag.
   */
  if (brec == NULL)
	brec = boot_rec;
  endp = (brec == boot_rec) ? (unsigned char *)boot_rec + BOOT_SIZE : NULL;
  cp = brec->bp_vend;

  /* Check if vendor area is RFC1048-compliant */
  if (memcmp(cp, VM_RFC1048, VM_SIZE))
	return(NULL);

  /* Now scan the vendor information field */
  cp += VM_SIZE;
  while (*cp != VEND_END && (endp == NULL || cp < endp)) {
	if (*cp != VEND_NOP) {
		if (*cp == id)
			break;
		cp += *(cp + 1) + 1;
	}
	cp++;
  }
  if (*cp == id)
	return(cp + 1);
  return(NULL);
}



/*
 **************************************************************************
 * 
 * Send a BOOTP request via UDP.
 *
 */
static void bootp_send(void)
{
  /* Assemble the BOOTP request */
  memset(boot_rec, 0, sizeof(struct bootp));
  memcpy(&(boot_rec->bp_vend), VM_RFC1048, VM_SIZE);
  get_hw_addr((char *)&(boot_rec->bp_chaddr));
  boot_rec->bp_op = BOOTP_REQUEST;
  boot_rec->bp_hwtype = BOOTP_ETHER;
  boot_rec->bp_hlen = ETH_ALEN;
  boot_rec->bp_xid = boot_xid;
  boot_rec->bp_ciaddr = htonl(IP_ANY);
  boot_rec->bp_siaddr = htonl(IP_BROADCAST);

  /* Send the bootp record as a broadcast message to all servers */
  udp_write((char *)boot_rec, sizeof(struct bootp));
}



/*
 **************************************************************************
 * 
 * Get a BOOTP record from the server.
 *
 */
static int bootp_get(void)
{
  char errch;
  int mask, timeout, i, boot_try;

  /*
   * Setup the IP interface information. Our own address has to be
   * IP_ANY, so that the IP layer will not discard the incoming packets.
   * Then open a UDP socket.
   */
  if_config(IP_ANY, IP_CLASS_A);
  (void)udp_open(IP_BROADCAST, BOOTP_C_PORT, BOOTP_S_PORT);

  /*
   * Now loop until receiving a reply from a server. The retry time is
   * computed as suggested in RFC951.
   */
  mask = 0x01;
  timeout = 1;
  boot_try = 0;
  printf("BOOTP: Sending request: ");
  while (++boot_try < BOOTP_RETRY) {
	boot_xid = get_ticks() + random();
	bootp_send();
	i = udp_read((char *)boot_rec, sizeof(struct bootp), timeout*18, CHR_ESC);
	if (i < 0) {				/* user pressed ESC */
		boot_try = BOOTP_RETRY;
		break;
	} else if (i == 0)			/* timeout */
		errch = '.';
	else if (boot_rec->bp_xid != boot_xid)	/* rule out duplicate packet */
		errch = '?';
	else
		break;
	if (mask < 0x3f)
		mask = (mask << 1) | 1;
	if ((timeout = mask & random()) < 1)
		timeout = 1;
	printf("%c", errch);
  }

  if (boot_try >= BOOTP_RETRY) {
	printf("\nBOOTP: No reply\n");
	return(FALSE);
  }

  if (get_vend(NULL, VEND_END) == NULL) {
	printf("\nBOOTP: Invalid record\n");
	return(FALSE);
  }

  printf("ok\n");
  return(TRUE);
}



#ifndef NOBPEXT
/*
 **************************************************************************
 * 
 * Get vendor extensions file with TFTP.
 */
static int get_ext(unsigned char *ftag)
{
  int size, i;
  char *inbuf;
  unsigned char *cp;

  /* Open the TFTP connection */
  printf("BOOTP: Loading %ls: ", ftag + 1, *ftag);
  if ((inbuf = tftp_open(ntohl(boot_rec->bp_siaddr), ftag + 1, *ftag)) == NULL)
	return(FALSE);

  /*
   * Read all blocks from vendor extension file. The file has the RFC
   * vendor ID at the beginning, so we have to take special care with
   * the first block.
   */
  cp = get_vend(NULL, VEND_END) - 1;
  for (i = 0; i < MAX_EXT_BLOCKS; i++) {
	if ((size = tftp_get()) < 0)
		return(FALSE);
	if (i == 0) {
		if (size < (VM_SIZE + 1) ||
		    memcmp(inbuf, VM_RFC1048, VM_SIZE)) {
			printf("Invalid file\n");
			return(FALSE);
		}
		memcpy(cp, (unsigned char *)inbuf + VM_SIZE, size - VM_SIZE);
		cp += size - VM_SIZE;
	} else if (size > 0) {
		memcpy(cp, (unsigned char *)inbuf, size);
		cp += size;
	}
	if (size < SEGSIZE)
		break;
  }

  /* If the file is too large, we are probably missing the end identifier */
  if (i == MAX_EXT_BLOCKS) {
	printf("File too large\n");
	return(FALSE);
  }

  /* Finally check if we have an end identifier at all */
  if (get_vend(NULL, VEND_END) == NULL) {
	printf("No end tag\n");
	return(FALSE);
  }
  printf("ok\n");
  return(TRUE);
}
#endif



#ifdef BPDEBUG
/*
 **************************************************************************
 * 
 * Print the complete BOOTP record
 */
static void print_rec(void)
{
  unsigned char *cp, *end;
  int i, c, adr;

  adr = 0;
  cp = (unsigned char *)boot_rec;
  if ((end = get_vend(NULL, VEND_END)) == NULL)
	end = cp + sizeof(struct bootp);
  printf("\n\n");
  while(cp < end) {
	if (adr < 16) printf("00%x  ", adr);
	else if (adr < 256) printf("0%x  ", adr);
	else printf("%x  ", adr);

	for (i = 0; i < 16 && cp <= end; i++, cp++) {
		c = *cp & 0xff;
		if (c < 16) printf("0%x ", c);
		else printf("%x ", c);
	}
	printf("\n");
	adr += 16;
  }
  if (getkey() == 0)
	(void)getkey();
  printf("\n\n");
}
#endif



/*
 **************************************************************************
 * 
 * Handle BOOTP protocol.
 */
struct bootp *bootp(void)
{
  unsigned char *cp;
  t_ipaddr netmask;

  /* Get BOOTP record from server */
  if (!bootp_get())
	return(NULL);

  /* Setup the network interface with the values received from BOOTP server */
  netmask = IP_ANY;
  if ((cp = get_vend(NULL, VEND_SUBNET)) != NULL)
	netmask = ntohl(*((t_ipaddr *)(cp + 1)));
  if_config(ntohl(boot_rec->bp_yiaddr), netmask);

  /* Also setup the default gateway so that we can reach the server. */
  if ((cp = get_vend(NULL, VEND_ROUTER)) != NULL)
	set_gateway(ntohl(*((t_ipaddr *)(cp + 1))));

#ifndef NOBPEXT
  /* Next try to get the extensions file if there is any specified */
  if ((cp = get_vend(NULL, VEND_EXTFILE)) != NULL && !get_ext(cp))
	return(NULL);
#endif

#ifdef BPDEBUG
  print_rec();
#endif

  return(boot_rec);
}



/*
 **************************************************************************
 * 
 * Initialize BOOTP protocol.
 */
int init_bootp(void)
{
  /* Set name of module for error messages */
  arpa_module_name = "bootp";

  /* Allocate space for BOOTP buffer */
  if ((boot_rec = malloc(BOOT_SIZE)) == NULL)
	return(FALSE);
  return(TRUE);
}
