;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  inflate.asm
;* Purpose: Decompress the rom image
;* Entries: inflate
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Load the boot rom segment layout and assembler macros.
;
include ..\headers\asm\macros.inc
include ..\headers\asm\layout.inc
include ..\headers\asm\memory.inc
include .\loadpriv.inc


;
;**************************************************************************
;
; Definitions
;
; Bit definitions for the first byte after compression magic number:
;
BIT_MASK	equ	01Fh		; mask for number of compression bits
BLOCK_MASK	equ	080h		; mask for block compression bit
;
; Number of compression bits
;
BITS		equ	15		; this is fixed, must be 15 bits
INIT_BITS	equ	9		; initial number of bits/code
;
; Define special places in the tables:
;
FIRST		equ	257		; first free entry
CLEAR		equ	256		; table clear output code
;
; Size of output and code table
;
HSIZE_C		equ	35023			; size of htab in characters
HSIZE_P		equ	(HSIZE_C shr 4) + 1	; size of htab in paragraphs
CSIZE_S		equ	32768			; size of ctab in short words
CSIZE_P		equ	(CSIZE_S shr 3) + 1	; size of ctab in paragraphs
 

;
;**************************************************************************
;
; Define the data segment
;
data_start

clr_flag	db	0			; clear flag
n_bits		db	INIT_BITS		; number of bits/code
maxcode		dw	(1 shl INIT_BITS) - 1	; maximum code, given n_bits
free_ent	dw	0			; first free entry
htab		dw	0			; pointer to output table
ctab		dw	0			; pointer to code table

; Static variables used within decompression routine:

finchar		db	0			; input character
oldcode		dw	0			; last input code

; Static variables used by code reading routine:

buf_off		db	0			; buffer bit offset
buf_size	db	0			; buffer bit size
buffer		db	BITS dup (0)		; buffer

		even				; align to word boundary

data_end


;
;**************************************************************************
;
text_start

	public	inflate

	extrn	malloc:near
	extrn	getbyte:near
	extrn	putbyte:near


;
;**************************************************************************
;
; Get next code from input stream
; Input:  none
; Output: AX  -  next code, or -1 if end of file
; Registers changed: AX, BX, CX, DX, SI
;
getcode		proc	near

; If the next entry will be too big for the current code size, then
; we must increase the size. This implies reading a new buffer full.
; Also, we have to read a new buffer if it's empty.

	mov	bx,free_ent
	test	clr_flag,0FFh		; check if buffer cleared
	jnz	short getc1
	mov	al,buf_off
	cmp	al,buf_size		; check if buffer empty
	jae	short getc1
	cmp	bx,maxcode		; check if next entry too big
	jbe	short getc7

getc1:	cmp	bx,maxcode
	jbe	short getc3
	mov	cl,n_bits		; if the next entry is too big we
	inc	cl			; have to increase the code table
	mov	ch,BITS			; size
	mov	ax,(1 shl BITS)
	cmp	cl,ch			; don't increase more than the
	jae	short getc2		; maximum
	mov	ch,cl
	mov	ax,1
	shl	ax,cl			; determine new maximum code
	dec	ax
getc2:	mov	maxcode,ax
	mov	n_bits,ch

getc3:	test	clr_flag,0FFh
	jz	short getc4
	mov	cl,INIT_BITS		; if the code table got cleared
	mov	n_bits,cl		; reset the bit number to the
	mov	ax,1			; initial value
	shl	ax,cl
	dec	ax
	mov	maxcode,ax		; also update the maximum code
	mov	clr_flag,0

getc4:	xor	bx,bx
getc5:	cmp	bl,n_bits		; read as much bytes from the
	jae	short getc6		; input stream as we have bits
	call	getbyte			; in the code table
	or	ah,ah
	jnz	short getc6
	mov	buffer[bx],al		; save read byte into buffer
	inc	bx
	jmp	short getc5
getc6:	mov	ax,-1
	or	bx,bx			; check for end-of-file
	jz	short getc9
	mov	buf_off,bh		; BH is 0 (shorter than immediate)
	shift	shl,bl,3		; compute number of bits in the buffer
	sub	bl,n_bits		; round size down to integral number
	inc	bl			; of codes
	mov	buf_size,bl

; Get next code out of the buffer. If we are running on a 386 we can use
; 32 bit functions.

ifdef IS386
getc7:	mov	cl,n_bits
	xor	dx,dx
	mov	ax,0FFFFh			; compute bit mask for next code
	shld	dx,ax,cl
	movzx	si,buf_off			; get byte index into buffer
	mov	cx,si
	shr	si,3
	mov	eax,dword ptr buffer[si]	; read next code
	and	cl,07h
	shr	eax,cl			; rotate it into the lower bits of AX
	and	ax,dx			; and mask out the unused high bits

else

getc7:	mov	cl,buf_off
	mov	ch,n_bits
	mov	si,offset dgroup:buffer
	mov	al,cl
	xor	ah,ah
	shift	shr,ax,3		; get to the first byte
	add	si,ax
	and	cl,07h			; CL contains the bit offset into
	lodsb				; the current byte
	xor	ah,ah
	shr	ax,cl
	mov	bx,ax			; BX now contains the next code bits

	mov	dl,8
	sub	dl,cl
	mov	cl,dl			; offset into code word
	sub	ch,cl			; number of bits remaining

	cmp	ch,8			; have to get another byte?
	jb	short getc8
	lodsb				; get any 8 bit parts in the middle
	xor	ah,ah
	shl	ax,cl			; adjust it for inclusion into code
	or	bx,ax			; word
	add	cl,8			; new offset into code word
	sub	ch,8			; number of bits remaining

getc8:	xor	dl,dl
getca:	or	ch,ch
	jz	short getcb
	stc				; compute mask for missing bits
	rcl	dl,1
	dec	ch
	jmp	short getca
getcb:	lodsb				; get high order bits
	and	al,dl			; clear all bits we don't want
	xor	ah,ah
	shl	ax,cl			; adjust bits for inclusion into
	or	ax,bx			; code word
endif

	mov	cl,n_bits
	add	buf_off,cl		; advance bit offset into buffer
getc9:	ret

getcode		endp


;
;**************************************************************************
;
; Actually decompress the image
; Input:  none
; Output: none
; Registers changed: AX, BX, CX, DX, SI, DI, ES
;
decompress	proc	near

	cld
	push	bp
	mov	bp,(1 shl BITS)		; initialize stack pointer
	call	getcode			; get first code from input stream
	or	ax,ax			; oops, already reaching the end
	js	short decom8
	mov	finchar,al		; save it for later
	mov	oldcode,ax
	call	putbyte			; must be 8 bits = char

decom1:	call	getcode			; get next code from input stream
	or	ax,ax			; check for end of compressed input
	jns	short decom7
decom8:	jmp	decom9

; Check if we have to clear the code table

decom7:	cmp	ax,CLEAR
	jne	short decom2
	mov	es,ctab
	mov	cx,256			; clear first 256 entries of code table
	xor	di,di
	xor	ax,ax
	rep	stosw
	inc	clr_flag		; indicate that the table was cleared
	mov	free_ent,FIRST - 1
	call	getcode			; get next code
	or	ax,ax			; oops, at the end?
	js	short decom8
decom2: push	ax			; save code

; Special case for KwKwK string

	mov	bx,ax
	mov	es,htab
	cmp	ax,free_ent
	jb	short decom3
	mov	al,finchar
	mov	es:[bp],al
	inc	bp
	mov	bx,oldcode

; Generate output characters in reverse order

decom3:	cmp	bx,256			; check if code run has ended
	jb	short decom4
	mov	al,es:[bx]		; code is offset into code table
	mov	es:[bp],al		; so get corresponding character
	inc	bp			; and save it into stack area
	push	es
	mov	es,ctab
	shl	bx,1
	mov	bx,es:[bx]		; get next code out of code table
	pop	es
	jmp	short decom3

decom4:	mov	al,es:[bx]		; get the final character onto
	mov	es:[bp],al		; the stack
	inc	bp
	mov	finchar,al		; save last character for later

; Put the output characters out in forward order

decom5:	dec	bp
	mov	al,es:[bp]		; get next character from stack
	call	putbyte			; and write it out
	cmp	bp,(1 shl BITS)		; reached bottom of stack
	ja	short decom5

; Generate new entry in code table

	mov	bx,free_ent
	cmp	bx,(1 shl BITS)		; end of code table reached?
	jae	short decom6
	mov	es,ctab
	mov	ax,oldcode
	shl	bx,1
	mov	es:[bx],ax		; save old code into code table
	mov	es,htab
	mov	bx,free_ent
	mov	al,finchar		; and save corresponding character
	mov	es:[bx],al
	inc	free_ent

; Remember previous code and continue with the next one

decom6:	pop	ax
	mov	oldcode,ax
	jmp	decom1

decom9:	pop	bp
	ret

decompress	endp


;
;**************************************************************************
;
; Inflate the compressed image
; Input:  none
; Output: AL  -  0=no error, -1=error
; Registers changed: AX, BX, CX, DX, SI, DI
;
inflate		proc	near

	push	es
	call	getbyte			; get value indicating number of bits
	test	al,BLOCK_MASK		; can only decompress version 3.0+
	jz	short infl8
	and	al,BIT_MASK
	cmp	al,BITS			; check that number of bits is
	jne	short infl8		; correct

; Generate output table and code tables

	mov	ax,HSIZE_P
	call	malloc			; assign memory for output table
	mov	htab,es
	mov	bx,255			; initialize the first 256 entries
infl1:	mov	es:[bx],bl
	or	bx,bx
	jz	short infl2
	dec	bx
	jmp	short infl1

infl2:	mov	ax,CSIZE_P
	call	malloc			; assign memory for code table
	mov	ctab,es
	mov	cx,256
	xor	ax,ax			; initialize the first 256 entries
	xor	di,di
	cld
	rep	stosw

; Now actually decompress the image

infl3:	mov	free_ent,FIRST
	call	decompress
	xor	al,al			; return without error
	jmp	short infl9

infl8:	mov	al,-1			; return with error
infl9:	pop	es
	ret

inflate		endp


;
;**************************************************************************
;
text_end			; end text segment

	end
