/*
 * afport.c
 *
 * This module supports serial port channels
 *
 * $Id: afport.c,v 1.9 1996/08/29 07:17:39 ron Exp $
 *
 */
#include <termios.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include "afos.h"

int port_init ( AF_CHANNEL *chan , void (*read_callback)() ) ;
int port_write ( AF_CHANNEL *chan, unsigned char *dat, int size ) ;
int port_getchar ( AF_CHANNEL *chan ) ;
void afport_read_select_handler( void *private, int event ) ;

/*
	parse_port parse out the 'port' line in the config file
*/
int
parse_port( char *line )
{
	static char args[8][80] ;
	AF_CHANNEL *chan ;
	PORT_STRUCT *mystuff ;
	int count ;

	count = sscanf ( line,  "%[^ 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]",
		args[0], args[1], args[2], args[3], 
		args[4], args[5], args[6], args[7]) ;

	if ( count < 3 )
	{
		fprintf ( stderr, "\r\nAFOSD: Unable to parse the following line:\n%s\n"
			"Insufficient parameters for port definition\n"
			"Usage:\n"
	"port <CHANNEL NAME> <DEVICE NAME> [BAUD[,DATABITS[,PARITY[,STOPBITS]]]]\n",
			line);
		exit (-1);
	}

	if (!verify_unique_portname( args[1] ) )
	{
		fprintf(stderr,"\r\nERROR in port configuration file on LINE:\n%s\n"
					   "Port '%s' already defined\n",
						line, args[1] ) ;
		exit (-1);
	}

#if 0
	printf( "config line: '%s'\n"
			"channel name: '%s'\n"
			"device name: '%s'\n"
			"Baud:'%s'\n"
			"Parity:'%s'\n"
			"Stops:'%s'\n"
			"count = %d\n\n",
			line, args[1], args[2], args[3], args[4], args[5], count ) ;
#endif

	chan = (AF_CHANNEL *)malloc( sizeof(*chan) ) ;
	if (!chan)
	{
		fprintf(stderr, "Unable to allocate new channel definition space\n");
		exit (-1);
	}
	
	/* Put new channel on the channel list */
	chan->next = channel_list ;
	channel_list = chan ;
	strncpy ( chan->name, args[1], AFCHAN_NAME_SIZE ) ;

	/* Allocate our private data structure and fill it out with our
	   configuration info */
	mystuff = (PORT_STRUCT *)malloc( sizeof(*mystuff) ) ;
	if (!mystuff)
	{
		fprintf(stderr, "Unable to allocate new channel definition space\n");
		exit (-1);
	}
	strncpy ( mystuff->device_name, args[2], AFCHAN_NAME_SIZE ) ;
	mystuff->baud = 9600 ;
	mystuff->parity = 'N' ;
	mystuff->bits = 8 ;
	mystuff->stops = 1 ;
	if ( count > 3 )
	{
		mystuff->baud = atoi ( args[3] ) ;
		if ( count > 4 )
		{
			mystuff->bits = atoi( args[4]) ;
			if ( count > 5 )
			{
				mystuff->parity = args[5][0] ;
				if ( count > 6 )
					mystuff->stops = atoi ( args[6] );
			}
		}
	}
	chan->private = mystuff ;

	/* Fill out the rest of the channel info */
	chan->fd = -1 ;
	chan->init = port_init ;
	chan->chanwrite = port_write ;
	chan->changetchar = port_getchar ;
#if 0
	printf( "config line: '%s'\n"
			"channel name: '%s'\n"
			"device name: '%s'\n"
			"Baud:'%d'\n"
			"Data bits:'%d'\n"
			"Parity:'%c'\n"
			"Stops:'%d'\n\n",
			line, chan->name, mystuff->device_name, 
			mystuff->baud, mystuff->bits, mystuff->parity, mystuff->stops );
#endif


}

static int
convert_stops ( int dat )
{
	switch (dat)
	{
		case 1:
			return 0 ;
		case 2:
			return CSTOPB ;
		default:
			return 0 ;
	}
}

static int
convert_bits ( int dat )
{
	switch (dat)
	{
		case 5:
			return CS5 ;
		case 6:
			return CS6 ;
		case 7:
			return CS7 ;
		case 8:
		default:
			return CS8 ;
	}
}

static int
convert_parity ( int dat )
{
	switch (dat)
	{
		case 'O':
		case 'o':
			return PARENB | PARODD ;
		case 'E':
		case 'e':
			return PARENB ;
		case 'N':
		case 'n':
		default:
			return 0 ;
	}
}

static int
convert_baud ( int dat )
{
	if ( dat >= 230400 )
		return B230400 ;
	else if (dat >= 115200)
		return B115200 ;
	else if (dat >= 57600)
		return B57600 ;
	else if (dat >= 38400)
		return B38400 ;
	else if (dat >= 19200)
		return B19200 ;
	else if (dat >= 9600)
		return B9600 ;
	else if (dat >= 4800)
		return B4800 ;
	else if (dat >= 2400)
		return B2400 ;
	else if (dat >= 1800)
		return B1800 ;
	else if (dat >= 1200)
		return B1200 ;
	else if (dat >= 600)
		return B600 ;
	else if (dat >= 300)
		return B300 ;
	else
		return B9600 ;
}

/*
	port_init initializes a serial port
*/
int
port_init ( AF_CHANNEL *chan , void (*read_callback)())
{
	PORT_STRUCT *mystuff = chan->private ;
	struct termios tparms;
	int ret ;

	/* Check to see if previously initialized */
	if ( chan->fd != -1 )
	{
		/* Already initialized */

		/* 
			If read_callback not already defined, and provided on this
			call, take this pointer
		*/
		if ( (read_callback) && (!mystuff->read_callback) )
		{
			mystuff->read_callback = read_callback ;
			afloop_select_list_add( chan->fd, 
									afport_read_select_handler,
									chan ) ;
		}

		return 0 ;
	}

	chan->fd  = open (mystuff->device_name, O_RDWR) ;

	if (read_callback)
	{
		/* 
			The port channel can just let the read_callback function
			handle read events
		*/
		mystuff->read_callback = read_callback ;
		afloop_select_list_add( chan->fd, 
								afport_read_select_handler,
								chan ) ;
	}
	if (chan->fd  < 0 )
	{
		perror ("afosd unable to open AFOS channel's serial port");
		exit(-1);
	}


	if ( isatty (chan->fd ) )
	{
		/* Configure serial port if this is really a serial port */
		/* The chan->fd  might point to a test data file */
		ret = tcgetattr(chan->fd , &tparms);
		if ( ret == -1 )
			return -1 ;

		tparms.c_iflag = IGNBRK ; /* |  IGNCR ; */
		tparms.c_oflag = 0 ;
		tparms.c_cflag = ( convert_parity(mystuff->parity) | 
						   convert_bits(mystuff->bits) |
						   convert_stops(mystuff->stops) | CREAD | CLOCAL ) ;
		tparms.c_lflag = 0 ;
		cfsetospeed ( &tparms, convert_baud(mystuff->baud)) ;
		cfsetispeed ( &tparms, convert_baud(mystuff->baud)) ;
		ret = tcsetattr(chan->fd ,TCSANOW,&tparms);
		if (ret == -1)
		{
			perror("Unable to ioctl the AFOS port");
			return -1 ;
		}

#if 0
		parm  = TIOCM_DTR ;
		ret = ioctl(chan->fd ,TIOCMBIC, &parm ) ;

		if (ret == -1)
			return -1 ;
#endif
	}
}

/*
	port_write writes a message out the serial port
*/
int
port_write ( AF_CHANNEL *chan, unsigned char *dat, int size )
{
	chan->write_count++ ;
	chan->bytes_written += size ;
	return ( write ( chan->fd, dat, size ) ) ;
}


void
afport_read_select_handler( void *private, int event )
{
	int ret ;
	unsigned char buffer ; 
	AF_CHANNEL *chan = private ;
	PORT_STRUCT *mystuff = chan->private ;

#if 0
	fprintf(stderr, "afport_read_select_handler got char for '%s'\n",
		chan->name );
#endif

	if ( (ret = read ( chan->fd, &buffer, 1 )) <= 0 )
	{
		if 	( ( errno == EAGAIN ) || ( errno == EINTR )  )
			return ;
	}
	else
		mystuff->read_callback( chan, buffer) ;
}

/*
	port_getchar reads one character from the serial port.  This call is
	non-blocking and may return a negative value if an error occurs
*/
int
port_getchar ( AF_CHANNEL *chan )
{
	int i, result;

	i = 0 ;
	result = read ( chan->fd, &i, 1 ) ;

	if ( result <= 0 )
		return -1 ;

	chan->bytes_read++ ;
	return 0xff & i ;
}
