
/*
 * DREADART [-h] [-v] 'messageid'
 * DREADART [-h] [-v] < file
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

int LookupHash(hash_t hv, const char *msgid);

int VerifyOnly = 0;
int HeadOnly = 0;
int ForceOpt = 0;
FILE *LogFo;

int
main(int ac, char **av)
{
    int r = 0;
    int i;
    char *arg = NULL;
    char *file = NULL;

    LogFo = stderr;

    LoadDiabloConfig(ac, av);

    for (i = 1; i < ac; ++i) {
	char *ptr = av[i];

	if (*ptr != '-') {
	    arg = ptr;
	    continue;
	}
	ptr += 2;
	switch(ptr[-1]) {
	case 'C':
	    if (*ptr == 0)
		++i;
	    break;
	case 'h':
	    HeadOnly = 1;
	    break;
	case 'v':
	    VerifyOnly = 1;
	    LogFo = stdout;
	    break;
	case 'f':
	    file = (*ptr) ? ptr : av[++i];
	    break;
	case 'F':
	    ForceOpt = 1;
	    break;
	default:
	    fprintf(stderr, "dreadart: Illegal option: %s\n", ptr - 2);
	    exit(1);
	}
    }

    if (arg == NULL && file == NULL) {
	printf("dreadart <message-id>\n");
	printf("dreadart hv1.hv2\n");
	printf("dreadart [-v] [-h] -f -/file\n");
	exit(0);
    }

    HistoryOpen(NULL, 0);

    if (arg == NULL) {
	char buf[1024];
	FILE *fi = (strcmp(file, "-") == 0) ? stdin : fopen(file, "r");

	if (fi) {
	    while (fgets(buf, sizeof(buf), fi) != NULL) {
		hash_t hv;
		char path[256];
		char msgid[256];

		path[0] = 0;

		if (strncmp(buf, "DUMP ", 5) == 0) {
		    hv.h1 = 0;
		    hv.h2 = 0;
		    if (sscanf(buf + 5, "%s", msgid) == 1) {
			char *p;

			hv.h1 = (int32)strtoul(msgid, &p, 16);
			if (*p == '.') {
			    hv.h2 = (int32)strtoul(p + 1, NULL, 16);
			}
			LookupHash(hv, NULL);
		    }
		} else if (sscanf(buf, "%s %s", path, msgid) == 2) {
		    hv = hhash(msgid);
		    LookupHash(hv, msgid);
		}
	    }
	    if (fi != stdin)
		fclose(fi);
	} else {
	    fprintf(stderr, "Unable to open %s\n", file);
	}
    } else {
	hash_t hv;
	char *msgid = NULL;

	if (arg[0] == '<') {
	    msgid = arg;
	    hv = hhash(arg);
	} else if (arg[0] == 'D' && arg[1] == '.') {
	    int32 dummy;

	    if (sscanf(arg + 2, "%x/%x.%x", &dummy, &hv.h1, &hv.h2) != 3) {
		fprintf(stderr, "argument error\n");
		exit(1);
	    }
	} else if (sscanf(arg, "%x.%x", &hv.h1, &hv.h2) != 2) {
	    fprintf(stderr, "argument error\n");
	    exit(1);
	}
	LookupHash(hv, msgid);
    }
    exit(r);
}

int
LookupHash(hash_t hv, const char *msgid)
{
    History h;
    int rv = 0;

    if (msgid)
	fprintf(LogFo, "%60s\t", msgid);
    else
	fprintf(LogFo, "%08x.%08x\t", hv.h1, hv.h2);

    if (HistoryLookupByHash(hv, &h) == 0) {
	char buf[8192];

	if (ForceOpt || (
		H_EXP(h.exp) != H_EXP((unsigned short)-1) && 
		(h.boffset || h.bsize)
	    )
	) {
	    int fd;
	    int headOnly = (int)(h.exp & EXPF_HEADONLY);

	    snprintf(buf, sizeof(buf), SpoolHomePat, NewsHome);
	    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "/D.%08x/B.%04x", 
		h.gmt - h.gmt % 10,
		h.iter
	    );

	    if (HeadOnly == 0 && headOnly) {
		fprintf(LogFo, "Article stored as header-only, use -h\n");
		rv = 1;
	    } else if ((fd = open(buf, O_RDONLY)) >= 0) {
		const char *base;
		const char *ptr;
		int extra = (h.boffset == 0) ? 0 : 1;

		errno = 0;

		base = xmap(
		    NULL, 
		    h.bsize + extra + 1, 
		    PROT_READ,
		    MAP_SHARED, 
		    fd, 
		    h.boffset - extra
		);

		if (base == NULL) {
		    fprintf(LogFo, "Unable to map file %s: %s (%d,%d,%d)\n", buf, strerror(errno), (int)(h.boffset - extra), (int)(h.bsize + extra + 1), extra);
		    rv = 1;
		}

		/*
		 * check for prior terminating zero, article body does not 
		 * begin with a null, and post terminating zero
		 */

		ptr = base;

		if (rv == 0 && extra) {
		    if (*ptr != 0) {
			fprintf(LogFo, " missingPreNul");
			rv = 1;
		    }
		    ++ptr;
		}
		if (rv == 0 && ptr[0] == 0) {
		    fprintf(LogFo, " nullArtBody");
		    rv = 1;
		}

		if (rv == 0 && ptr[h.bsize] != 0) {
		    fprintf(LogFo, " missingPostNul");
		    rv = 1;
		}

		/*
		 * Locate Message-ID header and test
		 */

		if (rv == 0 && msgid) {
		    const char *l;
		    int haveMsgId = 0;

		    for (l = ptr - 1; l; l = strchr(l, '\n')) {
			++l;
			if (strncasecmp(l, "Message-ID:", 11) == 0) {
			    int i;

			    haveMsgId = 1;

			    l += 11;
			    while (*l && *l != '<' && *l != '\n')
				++l;
			    for (i = 0; l[i] && l[i] != '>' && l[i] != '\n'; ++i) {
				if (msgid[i] != l[i])
				    rv = 1;
			    }
			    if (msgid[i] != l[i])
				rv = 1;
			}
			if (l[0] == '\n')	/* end of headers */
			    break;
		    }
		    if (rv) {
			fprintf(LogFo, " messageID-MisMatch");
		    } else if (haveMsgId == 0) {
			fprintf(LogFo, " missing-MessageID");
			rv = 1;
		    }
		}
		if (rv == 0)
		    fprintf(LogFo, "OK");
		fprintf(LogFo, "\n");
		if (rv == 0 && VerifyOnly == 0) {
		    fflush(LogFo);
		    fflush(stdout);

		    if (HeadOnly) {
			int i;
			int lastNl = 1;

			for (i = 0; i < h.bsize; ++i) {
			    if (ptr[i] == '\n') {
				if (lastNl)
				    break;
				lastNl = 1;
			    } else {
				lastNl = 0;
			    }
			}
			write(1, ptr, i);
		    } else {
			write(1, ptr, h.bsize);
		    }
		    fflush(stdout);
		}
		xunmap((void *)base, h.bsize + extra + 1);
		close(fd);
	    } else {
		fprintf(LogFo, "Unable to open %s\n", buf);
		rv = 1;
	    }
	} else if (h.boffset || h.bsize) {
	    fprintf(LogFo, "Article expired\n");
	    rv = 1;
	} else {
	    fprintf(LogFo, "Article pre-expired\n");
	    rv = 1;
	}
    } else {
	fprintf(LogFo, "Article not found in history\n");
	rv = 1;
    }
    return(rv);
}

