/* Debugging for `mmalloc'.
   Copyright 1990, 1991, 1992 Free Software Foundation

   Written May 1989 by Mike Haertel.
   Heavily modified Mar 1992 by Fred Fish (fnf@cygnus.com)

The GNU C Library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

The GNU C Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the GNU C Library; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.

 The author may be reached (Email) at the address mike@ai.mit.edu,
 or (US mail) as Mike Haertel c/o Free Software Foundation. */

/* substantially re-written by Julian Assange <proff@iq.org> */

#include <string.h>

#include "mmprivate.h"

/* Default function to call when something awful happens.  The application
   can specify an alternate function to be called instead (and probably will
   want to). */

extern void abort PARAMS ((void));

/* Arbitrary magical numbers.  */

#define MAGICWORD	(unsigned int) 0xadeadfed	/* Active chunk */
#define MAGICWORDFREE	(unsigned int) 0xbabecafe	/* Inactive chunk */
#define MAGICBYTE	((char) 0xd7)

/* Each memory allocation is bounded by a header structure and a trailer
   byte.  I.E.

	<size><magicword><user's allocation><magicbyte>

   The pointer returned to the user points to the first byte in the
   user's allocation area.  The magic word can be tested to detect
   buffer underruns and the magic byte can be tested to detect overruns. */

/* keep this header 64 bit aligned ! */

struct hdr
  {
    union
    {
	void *desc;		/* source.c:num */
	char pad[8];
    } u;
    unsigned int size;		/* Exact size requested by user.  */
    unsigned int magic;         /* Magic number to check header integrity.  */
  };

/* Check the magicword and magicbyte, and if either is corrupted then
   call the emergency abort function specified for the heap in use. */

static void
checkhdr (mdp, hdr)
  struct mdesc *mdp;
  CONST struct hdr *hdr;
{
  if (hdr -> magic == MAGICWORDFREE)
    {
	(*mdp->abortfunc)((void*)(&hdr[1]), hdr->u.desc, -1);
    }
  else
  if (hdr -> magic != MAGICWORD)
    {
	(*mdp->abortfunc)((void*)(&hdr[1]), hdr->u.desc, 0);
    }
  else
  if (((char*)&hdr[1])[hdr -> size] != MAGICBYTE)
    {
	(*mdp->abortfunc)((void *)(&hdr[1]), hdr->u.desc, 1);
    }
}

void
mfree_check (md, ptr, desc)
  PTR md;
  PTR ptr;
  char *desc;
{
  struct hdr *hdr = ((struct hdr *) ptr) - 1;
  struct mdesc *mdp;

  mdp = MD_TO_MDP (md);
  checkhdr (mdp, hdr);
  hdr -> magic = MAGICWORDFREE;
  mfree (md, (PTR)hdr);
}

PTR
mmalloc_check (md, size, desc)
  PTR md;
  size_t size;
  char *desc;
{
  struct hdr *hdr;
  struct mdesc *mdp;
  size_t nbytes;

  mdp = MD_TO_MDP (md);
  nbytes = sizeof (struct hdr) + size + 1;
  hdr = (struct hdr *) mmalloc (md, nbytes);
  if (hdr != NULL)
    {
      hdr -> size = size;
      hdr -> u.desc = desc;
      hdr -> magic = MAGICWORD;
      hdr++;
      *((char *) hdr + size) = MAGICBYTE;
    }
  return ((PTR) hdr);
}

PTR
mcalloc_check (md, num, size, desc)
  PTR md;
  size_t num;
  size_t size;
  char *desc;
{
  register PTR result;
  if ((result = mmalloc_check (md, num * size, desc)) != NULL)
    {
      bzero (result, num * size);
    }
  return (result);
}

PTR
mrealloc_check (md, ptr, size, desc)
  PTR md;
  PTR ptr;
  size_t size;
  char *desc;
{
  struct hdr *hdr;
  struct mdesc *mdp;
  size_t nbytes;

  if (ptr == NULL)
      return mmalloc_check (md, size, desc);
  hdr = ((struct hdr *) ptr) - 1;
  mdp = MD_TO_MDP (md);
  checkhdr (mdp, hdr);
  nbytes = sizeof (struct hdr) + size + 1;
  hdr = (struct hdr *) mrealloc (md, (PTR) hdr, nbytes);
  if (hdr != NULL)
    {
      hdr -> size = size;
      hdr -> u.desc = desc;
      hdr++;
      *((char *) hdr + size) = MAGICBYTE;
    }
  return ((PTR) hdr);
}

static void default_abort (void *p, char *desc, int overflow)
{
    abort();
}

void
mmcheck (md, func)
  PTR md;
  void (*func) PARAMS ((void *, char *, int));
{
  struct mdesc *mdp;

  mdp = MD_TO_MDP (md);

  /* We can safely set or update the abort function at any time, regardless
     of whether or not we successfully do anything else. */

  mdp -> abortfunc = (func != NULL ? func : default_abort);
  mdp -> flags |= MMALLOC_MMCHECK_USED;
}
