/* ===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*            National Center for Biotechnology Information (NCBI)
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government do not place any restriction on its use or reproduction.
*  We would, however, appreciate having the NCBI and the author cited in
*  any work or product based on this material
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
* ===========================================================================
*
* File Name:  salsa.c
*
* Author:  Colombe Chappey
*
* Version Creation Date:   1/27/96
*
* $Revision: 6.42 $
*
* File Description: 
*
* Modifications:  
* --------------------------------------------------------------------------
* Date     Name        Description of modification
* -------  ----------  -----------------------------------------------------
*
*
* ==========================================================================
*/
#include <salsa.h>
#include <saledit.h>
#include <salutil.h>
#include <salsap.h>
#include <salstruc.h>
#include <salpanel.h>
#include <saled.h>
#include <saldist.h>
#include <salparam.h>
#include <salfiles.h>
#include <dlogutil.h>
#include <fstyle.h>
#include <satutil.h>
#include <picture.h>
#include <viewer.h>
#include <drawseq.h>

#define OBJ_VIRT 254
#define MARGINLEFT15 16
#define MARGINLEFT25 22

static EditAlignDataPtr SeqAlignToEditAlignData (EditAlignDataPtr adp, SeqAlignPtr salp, Int4 start, SeqIdPtr mastersip, Uint1 showfeature);
static void CleanupAlignDataPanel (GraphiC g, VoidPtr data);
static void setup_aacolor (EditAlignDataPtr adp, SeqEditViewProcsPtr svpp);

static void get_client_rect (PaneL p, RectPtr prc)
{
  ObjectRect (p, prc);
  InsetRect (prc, HRZ_BORDER_WIDTH, VER_BORDER_WIDTH);
}

static Int2 getsize_seqids (ValNodePtr sloc, Uint2 printid)
{
  ValNodePtr  vnp;
  SeqLocPtr   slp;
  SeqIdPtr    sip;
  Char        str [52];
  Int4        lens, 
              max = 15;

  if (printid < 1)
     return 15;

  for (vnp=sloc; vnp!=NULL; vnp=vnp->next) 
  {
     slp = (SeqLocPtr) vnp->data.ptrvalue;
     if (slp!=NULL) {
        sip = SeqLocId (slp);
        if (sip!=NULL) {
           SeqIdWrite (sip, str, printid, 50);
           lens = (Int4)StringLen (str);
           if (lens > max)
              max = lens;
        }
     }
  }
  return max;
}

static ValNodePtr CreateCopyAlign (ValNodePtr anp_list, Int4 length)
{
  ValNodePtr     copyalign = NULL;
  SelEdStructPtr sesp;
  EditCellPtr    ecp;
  AlignNodePtr   anp;
  ValNodePtr     vnp;
  Int4           k;

  for (vnp = anp_list; vnp!=NULL; vnp=vnp->next) {
     anp = (AlignNodePtr)vnp->data.ptrvalue;
     if (anp!=NULL) {
        ecp= (EditCellPtr)MemNew ((size_t)((length+100)*sizeof(EditCell)));
        for (k=0; k < (length+100); k++) {
           ecp[k].r=0;
           ecp[k].g=0;
           ecp[k].b=0;
        }
        sesp = new_seledstruct (anp->seq_entityID, anp->bsp_itemID, OBJ_BIOSEQ,  anp->bsp_itemID, 0, 0, anp->sip, Seq_strand_plus, FALSE, NULL, (Pointer) ecp, 0, 0);  
        ValNodeAddPointer (&(copyalign), 0, (Pointer)sesp);
     }
  }
  return copyalign;
}

static void SetColorEditCell (Uint2 entityID, Uint2 itemID, ValNodePtr copyalign, Int4 from, Int4 to, Uint1Ptr rgb)
{
  EditCellPtr    ecp;
  ValNodePtr vnp;
  SelEdStructPtr sesp;
  Int4 j;

  for (vnp=copyalign; vnp!=NULL; vnp=vnp->next) {
     sesp = (SelEdStructPtr) vnp->data.ptrvalue;
     if (sesp->entityID == entityID && sesp->itemID==itemID) {
        ecp=(EditCellPtr)(sesp->data->data.ptrvalue);
        if (ecp!=NULL) {
           for (j=from; j<=to; j++) {
             ecp[j].r=rgb[0];
             ecp[j].g=rgb[1];
             ecp[j].b=rgb[2];
           }
        }
      }
  }
}

/**************************************************
***  AlignDataInit
***************************************************/
static EditAlignDataPtr AdpOptionsInit (EditAlignDataPtr adp, Int2 marginleft, Uint1 columnpcell, FonT font, Uint1 display_panel)
{
  Uint2  j;
  Uint4  carColor;

  adp->marginleft = marginleft;
  adp->intersalpwidth = 15;
  adp->interline = 0;

  if (font == NULL) {
#ifdef WIN_MAC
  font = ParseFont ("Monaco, 9");
#endif
#ifdef WIN_MSWIN
  font = ParseFont ("Courier, 9");
#endif
#ifdef WIN_MOTIF
  font = ParseFont ("fixed, 12");
#endif
  }
  adp->font = (Handle)(font);
  SelectFont (font);
  adp->charw   = CharWidth ('0');
  adp->ascent  = Ascent ();
  adp->leading = Leading ();
  adp->lineheight   = LineHeight () + adp->interline + 1;
  adp->scaleheight  = 2 * adp->lineheight;
  adp->margin.left  = adp->marginleft * adp->charw;
  adp->margin.right = EDIT_MARGIN_RIGHT;
  adp->margin.bottom= EDIT_MARGIN_BOT;
  SelectFont (systemFont);

if (!BLACKNWHITE) 
  carColor = GetColorRGB(0,0,0);
else
  carColor = GetColorRGB(255, 255, 255);
  for (j=0; j<256; j++) adp->colorRefs[j] = carColor;
  adp->colorRefs[COLOR_SCALE] = GetColorRGB(255,0,255);
  adp->colorRefs[COLOR_ID] = carColor;
  adp->colorRefs[COLOR_ID_MASTER] = GetColorRGB(255, 128, 0);
  adp->colorRefs[COLOR_CDS] = GetColorRGB(0, 255, 0);
  adp->colorRefs[COLOR_SELECT] = GetColorRGB(0, 0, 0);
  adp->colorRefs[COLOR_GAP] = GetColorRGB(0, 0, 0);
  adp->colorRefs[COLOR_STAR] = GetColorRGB (255, 0, 0);
/*
  adp->colorRefs[(Uint1) ('M' - '*')] = GetColorRGB (255, 128, 0);
*/
  adp->col[0].r=0; adp->col[0].g=0; adp->col[0].b=0;
  adp->col[1].r=0; adp->col[1].g=0; adp->col[1].b=0;
  adp->col[2].r=0; adp->col[2].g=0; adp->col[2].b=0;
  adp->col[3].r=0; adp->col[3].g=0; adp->col[3].b=0;
  adp->col[4].r=255; adp->col[4].g=255; adp->col[4].b=0;
  adp->col[5].r=255; adp->col[5].g=255; adp->col[5].b=0;
  adp->col[6].r=0; adp->col[6].g=255; adp->col[6].b=255;
  adp->col[7].r=0; adp->col[7].g=255; adp->col[7].b=255;
  adp->col[8].r=183; adp->col[8].g=199; adp->col[8].b=242;
  adp->col[9].r=183; adp->col[9].g=199; adp->col[9].b=242;
  adp->popcolor[0] = adp->popcolor[1] = adp->popcolor[2] = adp->popcolor[3] = 1;  
  adp->popcolor[4] = adp->popcolor[5] = 6;
  adp->popcolor[6] = adp->popcolor[7] = 4;
  adp->popcolor[8] = adp->popcolor[9] = 3;

  adp->marginwithindex = FALSE;
  adp->marginwithpos = TRUE;
  adp->marginwithIds = FALSE;
  adp->marginwithfeatid = TRUE;
  adp->marginwithgroup = FALSE;

  adp->styleNum = GetMuskCurrentSt();
  adp->newstyle = NULL;
  adp->showfeat = FALSE;
  adp->drawrfp = adp->drawrfm = adp->drawcomplt = FALSE;

  adp->draw_scale = FALSE; 
  adp->draw_bars = FALSE; 
  adp->nscaleline = 0;
  if ( adp->draw_scale ) adp->nscaleline ++;
  if ( adp->draw_bars )  adp->nscaleline ++;
  adp->columnpcell = columnpcell; 
  adp->rowpcell = 0;
  adp->displaytype = TRUE;
  adp->charmode = FALSE;
  adp->edit_pos = -1;
  adp->feat_pos = -1;
  adp->click_feat = 0;
  adp->clickwhat = 0;
  adp->ondrag = FALSE;

  adp->align_format = SALSA_FASTA;

  adp->printid = 0;
  adp->prot_mode = ALLPROTAA;
  adp->stoptransl = TRUE;
  adp->spliteditmode = FALSE;
  adp->gap_choice = IGNORE_GAP_CHOICE;

  adp->edit_mode = SEQ_EDIT;
  adp->display_panel = display_panel;
  adp->all_sequences = FALSE;
  adp->draw_emptyline = TRUE;
  adp->tmpfile [0] = '\0';

  return adp;
}

static EditAlignDataPtr AdpFieldsInit (EditAlignDataPtr adp, SelStructPtr master, Int2 width, Int2 height)
{
  Int2         x, y;

  adp->seqnumber = 0;
  adp->length = 0;
  adp->seg_bioseq = FALSE;
  adp->master.entityID = 0;   /*master->entityID;*/
  adp->master.itemID = 0;     /*master->itemID;*/
  adp->master.itemtype = 0;   /*master->itemtype;*/
  adp->master.region = NULL;
  adp->master.regiontype = 0;

  adp->sqloc_list = NULL;
  adp->bsqloc_list = NULL;
  adp->size_labels = 15;

  adp->sap_align = NULL;
  adp->sap_original = NULL;
  adp->blocks = NULL;

  adp->dirty = FALSE;

  adp->anp_list = NULL;
  adp->anp_graph = NULL;
  MemSet((Pointer)(adp->featOrder), 0, (size_t)(FEATDEF_ANY*sizeof(Uint1)));
  MemSet((Pointer)(adp->groupOrder),0, (size_t)(FEATDEF_ANY*sizeof(Uint1)));

  adp->voffset = 0;
  adp->hoffset = 0;

  adp->pnlLine = height - adp->margin.bottom/adp->lineheight;
  adp->pnlWidth= width - adp->margin.right/adp->charw;

  y = 0; x = 0;
  if (adp->columnpcell > 0) {
     y = (Int2) (adp->pnlWidth -adp->marginleft) / (Int2) adp->columnpcell;
     x = (Int2) (adp->pnlWidth -adp->marginleft -y) % (Int2)(adp->columnpcell);
     if (x == 9)  
        x = -1; 
  }
  adp->visibleWidth  = (Int2) (adp->pnlWidth -adp->marginleft -y -x);

  adp->vPage = adp->pnlLine - 3;  
  adp->hPage = adp->visibleWidth - 1; 
  adp->curalignline = 0;
  adp->nlines = 0;
  adp->numberalignline = 0;
  adp->item_id = adp->seqEntity_id = adp->alignline = NULL;
  adp->itemtype =NULL;
  adp->itemsubtype =NULL;
  
  adp->gr.left = 0;
  adp->gr.right = 0;

  adp->edit_item.entityID = 0;
  adp->edit_item.itemID = 0;
  adp->edit_item.itemtype = 0;
  adp->firstssp = NULL;
  adp->buffer = NULL;
  adp->linebuff = NULL;
  adp->buffertail = NULL;
  adp->copyalign = NULL;
  adp->bufferlength= 0;
  adp->minbufferlength= 0;
  adp->bufferstart = 0; 
  adp->editbuffer  = TMP_EDITBUFFER;
  adp->colonne = NULL;
  adp->select_block = NULL;

  adp->caret.entityID= adp->master.entityID;
  adp->caret.itemID = adp->master.itemID;
  adp->caret.itemtype= adp->master.itemtype;
  adp->caret.regiontype = 0;
  adp->caret.region = NULL;
  adp->caret_line= 0;
  adp->feat_line= 0;
  adp->caret_orig= 0;

  adp->params = NULL;

  adp->feat= NULL;
  adp->seqfeat= NULL;
  adp->nfeat= 0;

  adp->mol_type = Seq_mol_na;

  adp->current_pattern = NULL;
  adp->match_pat = NULL;
  adp->cur_pat = NULL;
  adp->ngroup = 0;
  adp->extra_data = NULL;
  return adp;
}


static EditAlignDataPtr EditAlignDataInit (SelStructPtr master, Int2 width, Int2 height, Int2 marginleft, Int2 columnpcell, Handle font, Boolean size_pixel, Uint1 display_panel)
{
  EditAlignDataPtr adp;

  adp = (EditAlignDataPtr) MemNew (sizeof(EditAlignData));
  AdpOptionsInit (adp, marginleft, columnpcell, (FonT)font, display_panel);
  if (size_pixel)
     adp =AdpFieldsInit(adp, master, width/adp->charw, height/adp->lineheight);
  else 
     adp =AdpFieldsInit(adp, master, width, height);
  return adp;
}

/******************************
***  Proc FreeEditAlignData
*******************************/
static EditAlignDataPtr AdpFieldsFree (EditAlignDataPtr adp)
{
  ValNodePtr       vnp;
  SeqParamPtr      prm;

  FreeAlignNode (adp->anp_list);
  adp->anp_list = NULL; 
  if (adp->anp_graph!=NULL)
     FreeAlignNode (adp->anp_graph);
  adp->anp_graph = NULL; 

  adp->sqloc_list = ValNodeFreeType (&(adp->sqloc_list), TypeSeqLoc);
  adp->bsqloc_list = ValNodeFreeType (&(adp->bsqloc_list), TypeSeqLoc);
  adp->size_labels = 15;

  if (adp->master.region != NULL) 
        adp->master.region = SeqLocFree ((SeqLocPtr) adp->master.region);
  
  adp->sap_align = CompSeqAnnotFree (adp->sap_align);
  adp->sap_original = NULL;
  adp->blocks = NULL;
 
  adp->firstssp = NULL;
  adp->lastses  = NULL;
  
  adp->item_id = MemFree (adp->item_id);
  adp->seqEntity_id = MemFree (adp->seqEntity_id);
  adp->itemtype = MemFree (adp->itemtype);
  adp->itemsubtype = MemFree (adp->itemsubtype); 
  adp->alignline = MemFree (adp->alignline);

  adp->buffer = BufferFree (adp->buffer);
  adp->buffertail = NULL;

  adp->linebuff = FreeTextAlignList(adp->linebuff);
  adp->colonne = MemFree (adp->colonne);
  adp->newstyle = NULL;
  
  adp->select_block = NULL;
  if (adp->caret.region != NULL)
        adp->caret.region = SeqLocFree ((SeqLocPtr) adp->caret.region);
  
  adp->feat = NULL;
  adp->seqfeat = NULL;

  for (vnp = adp->params; vnp != NULL; vnp = vnp = vnp->next) {
        prm = (SeqParamPtr) vnp->data.ptrvalue;
        if (prm != NULL) MemFree (prm);
        vnp->data.ptrvalue = NULL;
  }
  adp->params = ValNodeFree (adp->params);
 
  if (adp->current_pattern != NULL)
     MemFree (adp->current_pattern);
  adp->current_pattern = NULL;
  if (adp->match_pat != NULL) 
     SelStructDelList (adp->match_pat);
  adp->match_pat = NULL;
  adp->cur_pat = NULL;

  adp->extra_data = NULL;
  return adp;
}

static EditAlignDataPtr EditAlignDataFree (EditAlignDataPtr adp)
{
  if (adp!=NULL) {
     adp = AdpFieldsFree (adp);
     adp = MemFree (adp);
  }
  return NULL;
}

/*********************************************************
***
***  ResizeAlignDataWindow: 
***            CallBack for DocumentWindow
**********************************************************/
static void ResizeAlignDataWindow (WindoW w)
{
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  WindoW             temport;

  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  if (wdp == NULL) return;
  temport = SavePort(w);
  Select (wdp->pnl);
  GetPanelExtra (wdp->pnl, &adp);
  if ( adp == NULL ) return;
  do_resize_window (wdp->pnl, adp, FALSE);
  inval_panel (wdp->pnl, -1, -1);
  RestorePort (temport);
  Update ();
  return;
}

/*********************************************************
***
***   RESET FUNCTIONS
***    update sequence length
***
**********************************************************/
typedef struct updatesegstruc {
  BioseqSetPtr      parts;
  BioseqPtr         segseq;
  BioseqSetPtr      segset;
} UpdateSegStruc, PNTR UpdateSegStrucPtr;

static void AddBspToSegSet (BioseqPtr segseq, BioseqPtr bsp)

{
  SeqIdPtr   sip;
  SeqLocPtr  slp;

  if (segseq == NULL) return;
  slp = ValNodeNew ((ValNodePtr) segseq->seq_ext);
  if (slp == NULL) return;
  if (segseq->seq_ext == NULL) {
    segseq->seq_ext = (Pointer) slp;
  }
  if (bsp != NULL && bsp->length > 0) {
    segseq->length += bsp->length;
    slp->choice = SEQLOC_WHOLE;
    sip = SeqIdFindBest (bsp->id, 0);
    slp->data.ptrvalue = (Pointer) SeqIdDup (sip);
  } else {
    slp->choice = SEQLOC_NULL;
  }
}


static void DoUpdateSegSet (BioseqPtr segseq, BioseqSetPtr parts)
 
{
  BioseqPtr    bsp;
  Boolean      notFirst;
  Boolean      nullsBetween;
  SeqFeatPtr   sfp;
  SeqFeatPtr   sfpnext;
  SeqLocPtr    slp;
  SeqEntryPtr  tmp;

  if (segseq == NULL || parts == NULL || parts->seq_set == NULL) return;
  tmp = parts->seq_set;
  notFirst = FALSE;
  nullsBetween = FALSE;
  segseq->length = 0;
  switch (segseq->seq_ext_type) {
    case 1:    /* seg-ext */
      slp = (ValNodePtr) segseq->seq_ext;
      while (slp != NULL) {
        if (slp->choice == SEQLOC_NULL) {
          nullsBetween = TRUE;
        }
        slp = slp->next;
      }  
      SeqLocSetFree ((ValNodePtr) segseq->seq_ext);
      break;
    case 2:   /* reference */
      SeqLocFree ((ValNodePtr) segseq->seq_ext);
      break;
    case 3:   /* map */
      sfp = (SeqFeatPtr) segseq->seq_ext;
      while (sfp != NULL) {
        sfpnext = sfp->next;
        SeqFeatFree (sfp);
        sfp = sfpnext;
      }
      break;
    default:
      break;
  }
  segseq->seq_ext = NULL;
  while (tmp != NULL) {
    if (nullsBetween && notFirst) {
      AddBspToSegSet (segseq, NULL);
    }
    bsp = (BioseqPtr) tmp->data.ptrvalue;
    if (bsp != NULL) {
      AddBspToSegSet (segseq, bsp);
    }      
    notFirst = TRUE;
    tmp = tmp->next;
  }
}
 
 
static void FindSegSetComponentsCallback (SeqEntryPtr sep, Pointer mydata,
                                          Int4 index, Int2 indent)
 
{
  BioseqPtr          bsp;
  BioseqSetPtr       bssp;
  UpdateSegStrucPtr  ussp;
 
  if (sep != NULL && sep->data.ptrvalue && mydata != NULL) {
    ussp = (UpdateSegStrucPtr) mydata;
    if (IS_Bioseq(sep)) {
      bsp = (BioseqPtr) sep->data.ptrvalue;
      if (ISA_na (bsp->mol)) {
        ussp->segseq = bsp;
      }  
    } else if (IS_Bioseq_set(sep)) {
      bssp = (BioseqSetPtr) sep->data.ptrvalue;
      if (bssp->_class == 2) {
        ussp->segset = bssp;
      } else if (bssp->_class == 4) {
        ussp->parts = bssp;
      }  
    }
  }
}
 
static Int4 UpdateSegList (SeqEntryPtr sep, Pointer mydata,
                           SeqEntryFunc mycallback,
                           Int4 index, Int2 indent)

{
  BioseqSetPtr  bssp;

  if (sep == NULL) return index;
  if (mycallback != NULL)
    (*mycallback) (sep, mydata, index, indent);
  index++;
  if (IS_Bioseq (sep)) return index;
  if (Bioseq_set_class (sep) == 4) return index;
  bssp = (BioseqSetPtr) sep->data.ptrvalue;
  sep = bssp->seq_set;
  indent++;
  while (sep != NULL) {
    index = UpdateSegList (sep, mydata, mycallback, index, indent);
    sep = sep->next;
  }
  return index;
}

#define UpdateSegExplore(a,b,c) UpdateSegList(a, b, c, 0L, 0);

static EditAlignDataPtr EditAlignDataChangeLength (EditAlignDataPtr adp, Int4 cutlength, SelStructPtr ssp)
{
  SeqLocPtr    slp_master;
  SeqIntPtr    sit_master;
  DenseSegPtr  dsp;
  SeqAnnotPtr  sap;
  SeqAlignPtr  salp;

  SeqEntryPtr  sep;
  UpdateSegStruc  uss;


  adp->length -= cutlength;

  sap = SeqAnnotBoolSegToDenseSeg (adp->sap_align);
  salp = (SeqAlignPtr) sap->data;
  dsp = (DenseSegPtr) salp->segs;
  *(dsp->lens) -= cutlength;
  CompSeqAnnotFree (adp->sap_align);
  adp->sap_align = SeqAnnotDenseSegToBoolSeg (sap);

  adp->sqloc_list = ValNodeFreeType (&(adp->sqloc_list), TypeSeqLoc);
  adp->bsqloc_list= ValNodeFreeType (&(adp->bsqloc_list), TypeSeqLoc); 
  adp->sqloc_list = SeqLocListFromSeqAlign (salp); 
  adp->bsqloc_list = WholeSeqLocListFromSeqAlign (salp);

  slp_master = (SeqLocPtr) adp->master.region;
  sit_master = (SeqIntPtr) slp_master->data.ptrvalue;
  sit_master->to -= cutlength;

  FreeAlignNode(adp->anp_list);
  if (adp->display_panel==0 || adp->display_panel==1 || adp->display_panel==2) 
  {
     adp->anp_list = (ValNodePtr) CollAlignFromSeqAnnot (sap, (SeqLocPtr) adp->master.region, adp->featOrder, adp->groupOrder, COLLECT_MD, FALSE, FALSE, FALSE);
  } 
  else if (adp->display_panel == 3) {
     adp->anp_list = (ValNodePtr) CollAlignFromSeqAnnot (sap, (SeqLocPtr) adp->master.region, adp->featOrder, adp->groupOrder, COLLECT_MP, TRUE, FALSE, FALSE);
  }
  if (adp->display_panel==2) {
     adp->anp_graph = (ValNodePtr) CollAlignFromSeqAnnot (sap, (SeqLocPtr)adp->master.region, adp->featOrder, adp->groupOrder, COLLECT_MP, TRUE, FALSE, FALSE);
  }
  sap = SeqAnnotFree (sap);
  clean_annot_for_anp(&(adp->anp_list));
  OrderFeatProc (adp->anp_list);
 
  adp->caret_orig = SeqLocStart((SeqLocPtr)adp->caret.region); 
  adp->feat   =update_featpept (adp, adp->feat, NULL, ssp, (-cutlength), 255);
  adp->seqfeat=update_featpept (adp, adp->seqfeat,NULL,ssp,(-cutlength), 255);
 
  adp->bufferlength = MIN((Int4)(adp->length),(Int4) adp->minbufferlength);
  if (abs((adp->bufferstart +adp->bufferlength)-adp->length) < 100)
               adp->bufferlength = adp->length - adp->bufferstart;

  if ( adp->seg_bioseq ) {
     sep = GetBestTopParentForItemID (adp->master.entityID, adp->master.itemID, adp->master.itemtype);
     uss.segseq = NULL;
     uss.parts = NULL;
     uss.segset = NULL;
     UpdateSegExplore (sep, (Pointer) &uss, FindSegSetComponentsCallback);
     if (uss.segseq != NULL && uss.parts != NULL && uss.segset != NULL) {
        DoUpdateSegSet (uss.segseq, uss.parts);
     }
  }
  return adp;
}

static void reset_features (PaneL pnl, EditAlignDataPtr adp)
{
 if (adp->showfeat) {
    Select (pnl);
    HideFeatureFunc (adp);
    adp->showfeat = FALSE;
    ShowFeatureProc (pnl, FALSE);
 }
}

static void reset_features_afterlengthchange(PaneL pnl, EditAlignDataPtr adp) { 
  reset_features (pnl, adp);
}

static void update_select_region (SelStructPtr ommsp, PaneL pnl, EditAlignDataPtr adp, Boolean update_caret)
{
  SeqAlignPtr salp;
  SeqIdPtr    sip;
  SeqLocPtr   slp;
  BaR         vsb;
  RecT        rp;
  Int4        from, to;
  Int4        froms, tos;
  Int2        chklocp, chklocp2;
 
  get_client_rect (pnl, &rp);
  salp = (SeqAlignPtr) adp->sap_align->data;
  slp = (SeqLocPtr) ommsp->region;
  sip = SeqLocId (slp);
  chklocp = chkloc (sip, SeqLocStart (slp), adp->sqloc_list, &froms);
  from = SeqCoordToAlignCoord (froms, sip, salp, 0, chklocp);
  chklocp2 = chkloc (sip, SeqLocStop (slp), adp->sqloc_list, &tos);
  to = SeqCoordToAlignCoord (tos, sip, salp, 0, chklocp2);
  if(chklocp>-1 && chklocp2>-1) {
     if (to < adp->hoffset || from > adp->hoffset+ adp->visibleLength)  
     {
        vsb = GetSlateVScrollBar ((SlatE) pnl);
        adp->voffset = hoffset2voffset(adp, adp->anp_list, adp->visibleWidth, 0, adp->length-1, from);
        ResetClip ();
        data_collect_arrange (adp, TRUE);
        SetCorrectVBarMax (pnl, adp->nlines, adp->voffset);
     }
  }
  if ( adp->caret.regiontype == OM_REGION_SEQLOC ) {
     inval_selstructpos(adp,adp->caret.entityID, adp->caret.itemID, adp->caret.itemtype, &rp, SeqLocStart (adp->caret.region));
  }
  if (update_caret && (from != SeqLocStart(adp->caret.region))) {
     locate_region (&(adp->caret), from, to, sip, Seq_strand_plus, FALSE);
  }
  if (!adp->display_panel) {
     if (update_caret) {
        if (chklocp2==APPEND_RESIDUE)
           slp = SeqLocIntNew (froms, tos-1, 0, sip);
        else
           slp = SeqLocIntNew (froms, tos, 0, sip);
/************* BEFORE = REPLACE to do*****/
        ommsp->region = slp;
        to_update_prompt (pnl, ommsp, NULL, NULL, TRUE);
     }
     else {
        to_update_prompt (pnl, &(adp->caret), NULL, NULL, TRUE); 
     }
     update_edititem (pnl);
  }
  inval_selstructloc (adp, ommsp->entityID, ommsp->itemID, ommsp->itemtype, 255, &rp, from, to);
  return;
}


static EditAlignDataPtr update_length (PaneL pnl, EditAlignDataPtr adp, Int4 cutlength)
{
  SelStructPtr     ssp;
  RecT             rp;
  SeqLocPtr        slp;
  ValNodePtr       vnp;
  AlignNodePtr     anp;
  SeqAlignPtr      salp;
  Int4             from, to;
  Boolean          localssp=FALSE;

  if (adp!=NULL && cutlength != 0) 
  {
     get_client_rect (pnl, &rp);
     slp = (SeqLocPtr)(adp->caret.region);
     ssp = getOMselect_for_itemtype (OBJ_BIOSEQ);
     if (ssp == NULL) {
        localssp = TRUE;
        from = adp->caret_orig - cutlength;
        to = adp->caret_orig;
        for (vnp = adp->anp_list; vnp != NULL; vnp = vnp->next) {
           anp = (AlignNodePtr) vnp->data.ptrvalue;
           if (SeqIdForSameBioseq(anp->sip, SeqLocId(slp))) 
               break;
        }
        ssp = SelStructNew (anp->seq_entityID, anp->bsp_itemID, OBJ_BIOSEQ, from, to, anp->sip, Seq_strand_plus, FALSE);
     }
     adp = EditAlignDataChangeLength (adp, cutlength, ssp);
     if (localssp)
        SelStructDel (ssp);
     salp = (SeqAlignPtr) adp->sap_align->data;
     if (!adp->display_panel)
        to_update_prompt (pnl, &(adp->caret), salp, adp->sqloc_list, FALSE);
     data_collect_arrange (adp, TRUE);
     SetCorrectVBarMax (pnl, adp->nlines, adp->voffset);
  }
  return adp;
}


/*********************************************************
***
***   RESET FUNCTIONS
***     Reset data structure 
***
**********************************************************/
static EditAlignDataPtr EditAlignDataReset (EditAlignDataPtr adp, Int2 width, Int2 height, SeqAlignPtr salp)
{
  SeqAlignPtr  newsalp;
  Uint2        choice = 1;
  SeqIdPtr     mastersip;
  SeqAlignPtr  blocks;

  if (salp != NULL) {
     newsalp = salp;
  } else {
     newsalp = SeqAlignBoolSegToDenseSeg ((SeqAlignPtr)adp->sap_align->data);
  }
  if (newsalp == NULL)
     return adp;

  mastersip =SeqIdDup (SeqIdFindBest(SeqLocId((SeqLocPtr)adp->master.region), 0));
  if (adp->blocks!=NULL) {
     blocks=adp->blocks;
     adp->blocks=NULL;
  }
  adp = AdpFieldsFree (adp);
  adp = AdpFieldsInit (adp, NULL, width, height); 
  adp = SeqAlignToEditAlignData (adp, newsalp, adp->hoffset, mastersip, (Uint1)(adp->showfeat));
  if (adp->blocks!=NULL) {
     SeqAlignFree (blocks);
  }
  else 
     adp->blocks=blocks;
  return adp;
}

/******************************************
***
***  MESSAGE FUNCTIONS
***
***  SalsaFormMessage 
***       sends update message when the windows is closed
***
***  BioseqEditMsgFunc
***       picks up update messages 
***       if itemtype is OBJ_SEQFEAT: recollect data structure
***
*******************************************/
static void SalsaFormMessage (ForM f, Int2 mssg)
{
  SeqEditViewFormPtr  wdp;

  wdp = (SeqEditViewFormPtr) GetObjectExtra (f);
  if (wdp != NULL) {
    switch (mssg) {
      case VIB_MSG_CLOSE :
        Beep ();
        break;
      default :
        if (wdp->appmessage != NULL) {
          wdp->appmessage (f, mssg);
        }
        break;
    }
  }
}

static void visible_on (ValNodePtr params, Uint2 entityID, Uint2 itemID)
{
  ValNodePtr  vnp;
  SeqParamPtr prm;

  for (vnp = params; vnp != NULL; vnp = vnp->next)
  {
     prm = (SeqParamPtr) vnp->data.ptrvalue;
     if (prm->entityID == entityID && prm->itemID == itemID) {
        prm->visible = TRUE;
     }
  }
}

static void visible_off (ValNodePtr params, Uint2 entityID, Uint2 itemID)
{
  ValNodePtr  vnp;
  SeqParamPtr prm;

  for (vnp = params; vnp != NULL; vnp = vnp->next)
  {
     prm = (SeqParamPtr) vnp->data.ptrvalue;
     if (prm->entityID == entityID && prm->itemID == itemID) {
        prm->visible = FALSE;
     }
  }
}

/*************************************************************
**************************************************************/
static Int2 LIBCALLBACK BioseqEditMsgFunc (OMMsgStructPtr ommsp)
{
  WindoW             temport;
  OMUserDataPtr      omudp;
  SeqEditViewFormPtr wdp = NULL;
  EditAlignDataPtr   adp;
  SeqAlignPtr        salp,
                     newsalp = NULL;
  SelStructPtr       ssptmp;
  ValNodePtr         vnp;
  SeqLocPtr          slp;
  BioseqPtr          bsp;
  SeqIdPtr           sip;
  Int4               cutlength;
  Int4               from , to;
  Int2               width;
  Int2               chklocp;
  RecT               rp;

  omudp = (OMUserDataPtr)(ommsp->omuserdata);
  if (omudp == NULL) return OM_MSG_RET_ERROR;
  wdp = (SeqEditViewFormPtr) omudp->userdata.ptrvalue;
  if (wdp == NULL) return OM_MSG_RET_ERROR;
  if ( ( adp = GetAlignDataPanel (wdp->pnl) ) == NULL ) 
         return OM_MSG_RET_ERROR;
  if ( adp->seqnumber == 0 ) 
         return OM_MSG_RET_ERROR;

  temport = SavePort (ParentWindow (wdp->pnl));
  UseWindow (ParentWindow(wdp->pnl));
  Select (wdp->pnl);
  get_client_rect (wdp->pnl, &rp);
  width = adp->visibleWidth;
  if (adp->columnpcell > 0)
     width += (Int2) adp->visibleWidth / (Int2) adp->columnpcell;
  salp = (SeqAlignPtr) adp->sap_align->data;
  switch (ommsp->message) 
  {
      case OM_MSG_UPDATE:
          if (! Visible (ParentWindow (wdp->pnl))) return OM_MSG_RET_OK;
          cutlength = 0;
          for (vnp=adp->bsqloc_list; vnp!=NULL; vnp=vnp->next) 
          {
             slp = (SeqLocPtr) vnp->data.ptrvalue;
             bsp = BioseqLockById (SeqLocId(slp));
             if (bsp != NULL) {
                cutlength = (Int4) SeqLocLen (slp) - bsp->length;
                BioseqUnlock (bsp);
                if (cutlength !=  0) break;
             }
          }
          if (cutlength != 0 && adp->input_format == OBJ_BIOSEQ) 
          {
             update_length (wdp->pnl, adp, cutlength);
          }
          if (cutlength != 0 && adp->input_format == OBJ_SEQALIGN) 
          {
             adp = EditAlignDataReset (adp, (rp.right-rp.left), (rp.bottom-rp.top), NULL);
             if (adp == NULL) {
                return  OM_MSG_RET_ERROR;
             }
             do_resize_window (wdp->pnl, adp, TRUE);
          }
          if (cutlength!=0 && (adp->input_format==OBJ_BIOSEQ || adp->input_format==OBJ_SEQALIGN)) 
          {
             Select (wdp->pnl);
             reset_features_afterlengthchange (wdp->pnl, adp);
             if (cutlength < 0)
                inval_selstructpos_tobottom(adp, adp->caret.entityID, adp->caret.itemID, adp->caret.itemtype, &rp, adp->edit_pos);
             else if (cutlength > 0 && adp->caret.region!=NULL) {
                inval_selstructpos_tobottom(adp, adp->caret.entityID, adp->caret.itemID, adp->caret.itemtype, &rp, SeqLocStart((SeqLocPtr)adp->caret.region));
             }
             else 
                inval_panel (wdp->pnl, -1, -1);
             adp->edit_pos = SeqLocStart((SeqLocPtr)adp->caret.region);
          }
          if (cutlength == 0 && ommsp->itemtype == OBJ_BIOSEQ) 
          {
             if (adp->input_format == OBJ_BIOSEQ) 
             {
                if (ommsp->regiontype == OM_REGION_SEQLOC) 
                {
                   sip = SeqLocId ((SeqLocPtr) ommsp->region);
                   from = SeqLocStart ((SeqLocPtr) ommsp->region);
                   from = SeqCoordToAlignCoord(from, sip, salp, 0, 0);
                   to = SeqLocStop ((SeqLocPtr) ommsp->region);
                   chklocp =chkloc(sip, to, adp->sqloc_list, &to);
                   to = SeqCoordToAlignCoord(to, sip, salp, 0, chklocp);
                   inval_selstructloc (adp, ommsp->entityID, ommsp->itemID, ommsp->itemtype, 255, &rp, from, to);
                } 
                else {
                    if (adp->curfeat != NULL) {
                       adp->feat = del_ssp_fromid (adp->feat, (Uint2) SEQFEAT_CDREGION, adp->curfeat);
                       adp->curfeat = NULL;
                    }
                    reset_features (wdp->pnl, adp);
                }
             }
             else if (adp->input_format == OBJ_SEQALIGN) {
                Select (wdp->pnl);
                EditAlignDataRepopulateFromSeqAlign (wdp->pnl, adp, NULL);
             }
          }
          else if (cutlength == 0 && ommsp->itemtype == OBJ_VIRT)
          {
             data_collect_arrange (adp, TRUE);
             SetCorrectVBarMax (wdp->pnl, adp->nlines, adp->voffset);
             reset_features (wdp->pnl, adp);
             inval_panel (wdp->pnl, -1, -1);
          }
          else if (cutlength == 0 && ommsp->itemtype == OBJ_SEQFEAT)
          {
             reset_features (wdp->pnl, adp);
             inval_panel (wdp->pnl, -1, -1);
          }
          else {
             inval_panel (wdp->pnl, -1, -1);
          }
          break;
      case OM_MSG_SETCOLOR:
          if (ommsp->itemtype == OBJ_BIOSEQ)
          {
             if (ommsp->regiontype == OM_REGION_SEQLOC) {
                if (is_id_in (SeqLocId((SeqLocPtr)ommsp->region), adp->sqloc_list)) {
                   sip = SeqLocId ((SeqLocPtr) ommsp->region);
                   from = SeqLocStart ((SeqLocPtr) ommsp->region);
                   chklocp =chkloc(sip, from, adp->sqloc_list, &from);
                   from = SeqCoordToAlignCoord(from, sip, salp, 0, chklocp);
                   to = SeqLocStop ((SeqLocPtr) ommsp->region);
                   chklocp =chkloc(sip, to, adp->sqloc_list, &to);
                   to = SeqCoordToAlignCoord(to, sip, salp, 0, chklocp);
                   SetColorEditCell (ommsp->entityID, ommsp->itemID, adp->copyalign, from, to, ommsp->rgb);
                   inval_selstructloc (adp, ommsp->entityID, ommsp->itemID, ommsp->itemtype, 255, &rp, from, to);
                }
             }
          }
          break;
      case OM_MSG_DESELECT:
          if (ommsp->itemtype == OBJ_BIOSEQ)
          {
             if (ommsp->regiontype == OM_REGION_SEQLOC) {
                if (is_id_in (SeqLocId((SeqLocPtr)ommsp->region), adp->sqloc_list)) {
                   sip = SeqLocId ((SeqLocPtr) ommsp->region);
                   from = SeqLocStart ((SeqLocPtr) ommsp->region);
                   from = SeqCoordToAlignCoord(from, sip, salp, 0, 0);
                   to = SeqLocStop ((SeqLocPtr) ommsp->region);
                   chklocp =chkloc(sip, to, adp->sqloc_list, &to);
                   to = SeqCoordToAlignCoord(to, sip, salp, 0, chklocp);
                   inval_selstructloc (adp, ommsp->entityID, ommsp->itemID, ommsp->itemtype, 255, &rp, from, to);
                   if ( adp->caret.regiontype == OM_REGION_SEQLOC ) {
                      from = SeqLocStart ((SeqLocPtr)adp->caret.region); 
                      if (from != SeqLocStop((SeqLocPtr)adp->caret.region))
                         setposition_tossp (&(adp->caret), from, from);
                      if (!adp->display_panel)
                         to_update_prompt (wdp->pnl, &(adp->caret), (SeqAlignPtr) adp->sap_align->data, adp->sqloc_list, FALSE); 
                      inval_selstructpos(adp,adp->caret.entityID, adp->caret.itemID, adp->caret.itemtype, &rp, from);
                   }
                }
             }
          }
          else if (ommsp->itemtype == OBJ_SEQFEAT) 
          {
             if (ommsp->regiontype == OM_REGION_SEQLOC) {
                sip = SeqLocId ((SeqLocPtr) ommsp->region);
                from = SeqLocStart ((SeqLocPtr) ommsp->region);
                from = SeqCoordToAlignCoord(from, sip, salp, 0, 0);
                to = SeqLocStop ((SeqLocPtr) ommsp->region);
                chklocp =chkloc(sip, to, adp->sqloc_list, &to);
                to = SeqCoordToAlignCoord(to, sip, salp, 0, chklocp);
                inval_selstructloc_forfeat (adp, ommsp->entityID, ommsp->itemID, ommsp->itemtype, 255, &rp, from, to);
             }
          }
          else if (ommsp->itemtype == OBJ_SEQALIGN)
          {
             visible_off (adp->params, ommsp->entityID, ommsp->itemID);
             data_collect_arrange (adp, TRUE);
             inval_panel (wdp->pnl, -1, -1);
          }
          break;

      case OM_MSG_SELECT: 
          if (ommsp->itemtype == OBJ_BIOSEQ) 
          {
             if (ommsp->regiontype == OM_REGION_SEQLOC) {
                slp = (SeqLocPtr)ommsp->region;
                if (is_id_in (SeqLocId(slp), adp->sqloc_list)) {
                   ssptmp = SelStructNew (ommsp->entityID, ommsp->itemID, ommsp->itemtype, SeqLocStart (slp), SeqLocStop (slp), SeqLocId (slp), SeqLocStrand(slp), FALSE);
                   update_select_region (ssptmp, wdp->pnl, adp, TRUE); 
                   SelStructDel (ssptmp);
                }
             } 
             else  {
                ssptmp=is_selectedbyID (ommsp->entityID, ommsp->itemID, ommsp->itemtype);
                if ( ssptmp != NULL ) 
                {
                   if (is_id_in (SeqLocId((SeqLocPtr)ssptmp->region), adp->sqloc_list)) {
                      update_select_region (ssptmp, wdp->pnl, adp, FALSE); 
                   } 
                   else 
                      inval_selstruct (adp, ommsp->entityID, ommsp->itemID, ommsp->itemtype, 255, &rp, adp->margin.left,(Int2)(width *adp->charw));
                }
             }
          }
          if (ommsp->itemtype == OBJ_SEQFEAT)
          {
           if ( adp->showfeat && adp->seqfeat!=NULL )
           {   
             if (ommsp->region == NULL) {
                ommsp->region = checkseqlocfeature_for_editor (ommsp->entityID, ommsp->itemID, adp->seqfeat);
                if (ommsp->region != NULL) {
                   ommsp->regiontype = OM_REGION_SEQLOC;
                   checkselectsequinfeature_for_editor (adp->seqfeat);
                }
             }
             if (ommsp->regiontype == OM_REGION_SEQLOC) 
             {
                sip = SeqLocId ((SeqLocPtr) ommsp->region);
                from = SeqLocStart ((SeqLocPtr) ommsp->region);
                from = SeqCoordToAlignCoord(from, sip, salp, 0, 0);
                to = SeqLocStop ((SeqLocPtr) ommsp->region);
                chklocp =chkloc(sip, to, adp->sqloc_list, &to);
                to = SeqCoordToAlignCoord(to, sip, salp, 0, chklocp);
/*
                if(to < adp->hoffset || from > adp->hoffset+ adp->visibleLength)
                {
                   vsb = GetSlateVScrollBar ((SlatE) wdp->pnl);
                   adp->voffset = (Int2)(from/(adp->visibleWidth * adp->vscrollunit));
                   ResetClip ();
                   data_collect_arrange (adp, TRUE);
                   SetCorrectVBarMax (wdp->pnl, adp->nlines, adp->voffset);
                }
*/
                inval_selstructloc_forfeat (adp, ommsp->entityID, ommsp->itemID, ommsp->itemtype, 255, &rp, from, to);
             } else {
                inval_selstruct (adp, ommsp->entityID, ommsp->itemID, ommsp->itemtype, 255, &rp, adp->margin.left,(Int2)(width *adp->charw));
             }
           }
           else {   
             inval_panel (wdp->pnl, -1, -1);
           }
          }
          else if (ommsp->itemtype == OBJ_SEQALIGN)
          {
             visible_on (adp->params, ommsp->entityID, ommsp->itemID);
             data_collect_arrange (adp, TRUE);
             inval_panel (wdp->pnl, -1, -1);
          }
          break;
      case OM_MSG_DEL:
          break;
      default:
          break;
  }
  RestorePort (temport);
  UseWindow (temport);
  return OM_MSG_RET_OK;
}

static void checkEntityIDForMsg (SeqEditViewFormPtr wdp)
{
  EditAlignDataPtr adp;
  AlignNodePtr    anp;
  ValNodePtr      vnp;
  OMUserDataPtr   omudp;
  Uint2           eID;
  SeqEntryPtr     sep;

  if ( ( adp = GetAlignDataPanel (wdp->pnl) ) == NULL ) 
         return;
  if ( adp->seqnumber == 0 ) 
         return;
  for (vnp=adp->anp_list; vnp!=NULL; vnp=vnp->next) {
   anp = (AlignNodePtr) vnp->data.ptrvalue;
   if (anp!=NULL) {
      sep = GetTopSeqEntryForEntityID (anp->seq_entityID);
      if (sep!=NULL) {
         eID = SeqMgrGetEntityIDForSeqEntry(sep);
         omudp = ObjMgrGetUserData (eID, wdp->procid, OMPROC_EDIT, wdp->userkey);
         if (omudp == NULL) {
            omudp = ObjMgrAddUserData (eID, wdp->procid, OMPROC_EDIT, wdp->userkey);
            if (omudp != NULL) {
               omudp->userdata.ptrvalue = (Pointer) wdp;
               omudp->messagefunc = BioseqEditMsgFunc;
            }
         }
      }
   }
  }
}



/******************************************
***
***  IMPORT FUNCTIONS
***
*******************************************/
static void repopulate_panel (WindoW w, EditAlignDataPtr adp, SeqAlignPtr salp)
{
  WindoW temport;
  SeqEditViewFormPtr wdp;

  if (salp==NULL || adp==NULL)
     return;
  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  EditAlignDataRepopulateFromSeqAlign (wdp->pnl, adp, salp);
  temport = SavePort(w);
  Select (wdp->pnl);
  inval_panel (wdp->pnl, -1 ,-1);
  RestorePort (temport);
}

/**************************
     sep = FastaRead (NULL, adp->mol_type);
     if (sep!=NULL) {
        importslp = SeqEntryToSeqLoc (sep, &nseq, adp->mol_type);
**************************/
static SeqAlignPtr ImportFunc (EditAlignDataPtr adp)
{
  SeqAlignPtr      salp = NULL;
  SeqEntryPtr      sep = NULL;
  ValNodePtr       importslp = NULL;
  SeqLocPtr        slp;
  BioseqPtr        bsp;
  Int2             nseq;
  Boolean          is_prot;

  WatchCursor ();
  importslp = CCReadAnythingLoop (NULL);
  if (importslp != NULL) 
  {
     slp = (SeqLocPtr)importslp->data.ptrvalue;
     bsp = BioseqLockById (SeqLocId (slp));
     if (bsp!=NULL) 
     {
        is_prot = (Boolean) ISA_aa(bsp->mol);
        BioseqUnlock (bsp);
        if (is_prot != ISA_aa(adp->mol_type)) {
           Message (MSG_ERROR, "The imported sequence does not have the same molecule type");
        }
        else {
           salp = AlignToFunc (adp, importslp, FALSE, PRGALIGNDEFAULT);
        }
     }
  }
  ArrowCursor ();
  return salp;
}

extern void ReadAlignView (BioseqPtr target_bsp, Uint2 import_format)
{
  ValNodePtr  vnp=NULL;
  SeqLocPtr   slp=NULL;
  SeqLocPtr   source_slp;
  SeqIdPtr    sip;
  SeqAlignPtr salp;
  BioseqPtr   source_bsp = NULL;
  SeqEntryPtr source_sep = NULL;

  if (target_bsp==NULL)
     return;
  switch (import_format) {
     case SALSA_FASTA:
        source_sep = FastaRead (NULL, Seq_mol_na);
        break;
     case SALSA_ASN1:
        source_sep = AsnReadForSalsa (NULL);
        break;
     default:
        break;
  }
  if (source_sep!=NULL) {
     if (IS_Bioseq_set(source_sep))
        source_sep = FindNucSeqEntry(source_sep);
     if (IS_Bioseq(source_sep))
     {
        sip = SeqIdFindBest(target_bsp->id, 0);
        slp = SeqLocIntNew (0, target_bsp->length - 1, Seq_strand_plus, sip);
        ValNodeAddPointer(&vnp, 0, (Pointer)slp);
        source_bsp = (BioseqPtr)source_sep->data.ptrvalue;
        source_slp = SeqLocIntNew (0, source_bsp->length-1, Seq_strand_plus, source_bsp->id);
        ValNodeAddPointer(&vnp, 0, (Pointer)source_slp);
        salp = SeqLocListToSeqAlign (vnp, PRGALIGNDEFAULT, NULL);
        if (salp != NULL) {
           LaunchAlignViewer (salp);
        }
     }
  }
  return;
}

static void ImportProc (IteM i)
{
  WindoW           w;
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr adp;
  SeqAlignPtr      salp;
  SeqAnnotPtr      sap;

  w = (WindoW) ParentWindow (i);
  if ( ( adp = GetAlignEditData (w) ) == NULL ) return;
  if ( adp->seqnumber == 0 ) return;
  salp = ImportFunc (adp);
  if (salp!=NULL) 
  {
     repopulate_panel (w, adp, salp);
     wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
     if (wdp!=NULL) 
        checkEntityIDForMsg (wdp);
  }
}

static void ExportTextProc (IteM i)
{
  EditAlignDataPtr adp;
  if ( ( adp = GetAlignEditData ((WindoW)ParentWindow (i)) ) == NULL ) return;
  adp->align_format = SALSA_SHWTXT;
  SalsaExportDialog (i);
}
static void ExportFastaProc (IteM i)
{
  EditAlignDataPtr adp;
  if ( ( adp = GetAlignEditData ((WindoW)ParentWindow (i)) ) == NULL ) return;
  adp->align_format = SALSA_FASTA;
  SalsaExportDialog (i);
}
static void ExportFastaGapProc (IteM i)
{
  EditAlignDataPtr adp;
  if ( ( adp = GetAlignEditData ((WindoW)ParentWindow (i)) ) == NULL ) return;
  adp->align_format = SALSA_FASTGAP;
  SalsaExportDialog (i);
}
static void ExportPhyProc (IteM i)
{
  EditAlignDataPtr adp;
  if ( ( adp = GetAlignEditData ((WindoW)ParentWindow (i)) ) == NULL ) return;
  adp->align_format = SALSA_PHYLIP;
  SalsaExportDialog (i);
}
static void ExportSeqAnnotProc (IteM i)
{
  EditAlignDataPtr adp;
  if ( ( adp = GetAlignEditData ((WindoW)ParentWindow (i)) ) == NULL ) return;
  adp->align_format = SALSA_ASN1;
  SalsaExportDialog (i);
}
static void ExportSeqAnnotButtonProc (ButtoN b)
{
  WindoW           wdialog;
  DialogBoxDataPtr dbdp;
  EditAlignDataPtr adp;
 
  wdialog = ParentWindow (b);
  Hide (wdialog);
  Update ();
  dbdp = (DialogBoxDataPtr) GetObjectExtra (wdialog);
  if ((adp = GetAlignEditData (dbdp->w)) != NULL ) {
     if ( adp->seqnumber > 0 ) {
        adp->align_format = SALSA_ASN1;
        ExportTextFunc (b);
     }
  }
  Hide (dbdp->w);
  Remove (wdialog);
  Remove (dbdp->w);
}

/***********************************************************
***
*** CloseWindow 
***     reads the temporary file
***     sends a message to ObjMgr (ObjMgrSendMessage(MSG_UPDATE))
***     frees the data structure
***     deletes the temporary file 
***
************************************************************/
static Boolean sap_replace (SeqAnnotPtr sap, SeqAlignPtr salp, Uint1 choice)
{
  if (sap != NULL) {
     for (; sap!= NULL; sap=sap->next) {
        if (sap->type == choice) {
           SeqAlignFree ((SeqAlignPtr)sap->data);
           sap->data = (Pointer)salp;
           return TRUE;
        }
     }   
  }
  return FALSE;
}

extern void ReplaceSeqAlignInSeqEntry (Uint2 entityID, Uint2 itemID, SeqAlignPtr salp)
{
  SeqEntryPtr      sep,
                   sep1 = NULL;
  SeqEntryPtr      sept = NULL;
  BioseqSetPtr     bssp = NULL;
  BioseqPtr        bsp = NULL;
  SeqAnnotPtr      sap = NULL;

  sep = GetBestTopParentForItemID (entityID, itemID, OBJ_BIOSEQ);
  if (sep != NULL) {
     if (IS_Bioseq(sep)) {
        entityID = ObjMgrGetEntityIDForChoice (sep);
        sep1 = GetTopSeqEntryForEntityID (entityID);
        bsp = (BioseqPtr) sep->data.ptrvalue;
     }   
     else if(IS_Bioseq_set(sep)) {
        sep1 = sep;
     }   
     if (sep1 != NULL) {
        bssp = NULL; bsp = NULL;
        if (IS_Bioseq(sep1)) {
           bsp = (BioseqPtr) sep1->data.ptrvalue;
           sap_replace(bsp->annot, salp, 2);
        }
        else if(IS_Bioseq_set(sep1)) {
           bssp = (BioseqSetPtr)sep1->data.ptrvalue;
           while (bssp!=NULL && bssp->_class == 7) {
              sept = bssp->seq_set;
              bssp = NULL; bsp = NULL;
              if (IS_Bioseq(sept))  {
                 bsp = (BioseqPtr) sept->data.ptrvalue;
                 break;
              }
              else if (IS_Bioseq_set(sept))
                 bssp = (BioseqSetPtr) sept->data.ptrvalue;
           }
           if (bssp!=NULL) {
              sap = bssp->annot;
              if((sap==NULL || salp==NULL) && IS_Bioseq(sep)) {
                 bsp = (BioseqPtr) sep->data.ptrvalue;
                 sap_replace(bsp->annot, salp, 2);
              }
              else 
                 sap_replace(sap, salp, 2);
              if (sap==NULL && IS_Bioseq_set(sep)) {
                 bssp = (BioseqSetPtr) sep->data.ptrvalue;
                 for (sept = bssp->seq_set; sept!=NULL; sept=sept->next) {
                    if (IS_Bioseq(sept)) {
                       bsp = (BioseqPtr) sept->data.ptrvalue;
                       sap_replace(bsp->annot, salp, 2);
                    }
                 }  
              }
           }
           else if (bsp!=NULL) {
              sap_replace(bsp->annot, salp, 2);
           }
        }
     }   
  }     
  return;
}

static void NotReplaceSeqAlignProc (ButtoN b)
{
  WindoW           wdialog;
  DialogBoxDataPtr dbdp;

  wdialog = ParentWindow (b);
  Hide (wdialog);
  Update ();
  dbdp = (DialogBoxDataPtr) GetObjectExtra (wdialog);
  Hide (dbdp->w);
  Remove (wdialog);
  Remove (dbdp->w);
}

static void ReplaceSeqAlignProc (ButtoN b)
{
  WindoW           wdialog;
  DialogBoxDataPtr dbdp;
  EditAlignDataPtr adp;
  SeqAlignPtr      salp;

  wdialog = ParentWindow (b);
  Hide (wdialog);
  Update ();
  dbdp = (DialogBoxDataPtr) GetObjectExtra (wdialog);
  if ( ( adp = GetAlignEditData (dbdp->w) ) != NULL ) {
     salp = (SeqAlignPtr)adp->sap_align->data;
     salp = SeqAlignBoolSegToDenseSeg (salp);
     if (salp!=NULL && adp->master.entityID > 0) {
        ReplaceSeqAlignInSeqEntry (adp->master.entityID, adp->master.itemID, salp);
        ObjMgrSendMsg (OM_MSG_UPDATE, adp->input_entityID, adp->input_itemID, OBJ_SEQALIGN);
     }
  }
  Hide (dbdp->w);
  Remove (wdialog);
  Remove (dbdp->w);
}

static void SaveAlignDlg (WindoW w)
{
  WindoW           wdialog;
  DialogBoxDataPtr dbdp;
  GrouP            g, c;

  wdialog = FixedWindow (-50, -33, -10, -10, "Save Alignment", StdCloseWindowProc);
  dbdp = (DialogBoxDataPtr) MemNew (sizeof (DialogBoxData));
  SetObjectExtra (wdialog, (Pointer) dbdp, StdCleanupExtraProc);
  dbdp->w = w;
  g = HiddenGroup (wdialog, 1, 0, NULL);
  StaticPrompt (g, "Before you exit the window, do you want", 0, dialogTextHeight,systemFont, 'c');
  StaticPrompt (g, "to replace or export the alignment ?", 0, dialogTextHeight,systemFont, 'c');
  c = HiddenGroup (wdialog, 3, 0, NULL);
  PushButton (c, "Replace", ReplaceSeqAlignProc);
  PushButton (c, "Export", ExportSeqAnnotButtonProc);
  PushButton (c, "Dismiss", NotReplaceSeqAlignProc);
  AlignObjects (ALIGN_CENTER, (HANDLE) g, (HANDLE) c, NULL);
  RealizeWindow (wdialog);
  Show (wdialog);
  return;
}

static void CloseWindowProc (PaneL pnl)
{
  WindoW             w;
  EditAlignDataPtr   adp;
  SeqEntryPtr        prevsep;
  SeqEntryPtr        currentsep;

  w = (WindoW)ParentWindow (pnl);
  if ( ( adp = GetAlignEditData (w) ) == NULL ) return;
  if (adp->dirty) {
     if (adp->input_format == OBJ_BIOSEQ) 
     {
        Hide (w);
        Update ();
        prevsep = seqentry_read (adp->tmpfile);
        currentsep = GetTopSeqEntryForEntityID (adp->master.entityID);
        ReplaceSeqEntryWithSeqEntry (currentsep, prevsep, TRUE);
        ObjMgrSendMsg (OM_MSG_UPDATE, adp->master.entityID, adp->master.itemID, adp->master.itemtype);
        Remove (w);
     }
     else if (adp->input_format == OBJ_SEQALIGN && adp->dirty) 
     {
/******
        SaveAlignDlg (w);
******/
        Hide (w);
        Update ();
        Remove (w);
     }
  }
  else {
     Hide (w);
     Update ();
     Remove (w);
  }
}

static void CloseWindowItemProc (IteM i)
{
  PaneL            pnl;
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  CloseWindowProc (pnl);
}

static void CloseWindowButton (ButtoN b)
{
  PaneL            pnl;
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (b)) ) == NULL) return;
  CloseWindowProc (pnl);
}

static void AcceptCloseFunc (WindoW w, EditAlignDataPtr adp)
{
  PaneL              pnl;

  if ( ( pnl= GetPanelFromWindow (w) ) == NULL) return;
  Hide (w);
  Update ();
  if (adp->dirty) {
     if (adp->input_format == OBJ_BIOSEQ) {
        SaveAllFeatProc (pnl);
        ObjMgrSendMsg (OM_MSG_UPDATE, adp->master.entityID, adp->master.itemID, adp->master.itemtype);
     }
  }
  Remove (w);
}

static void AcceptCloseWithoutSaveCDSProc (ButtoN b)
{
  WindoW           wdialog;
  DialogBoxDataPtr dbdp;
  EditAlignDataPtr adp;

  wdialog = ParentWindow (b);
  Hide (wdialog);
  Update ();
  dbdp = (DialogBoxDataPtr) GetObjectExtra (wdialog);
  if ( ( adp = GetAlignEditData (dbdp->w) ) != NULL ) {
     adp->feat = SeqfeatlistFree (adp->feat);
     AcceptCloseFunc (dbdp->w, adp);
  }
  Remove (wdialog);
}

static void SaveCDSConfirmDlg (WindoW w)
{
  WindoW           wdialog;
  DialogBoxDataPtr dbdp;
  GrouP            g, c;

  wdialog = FixedWindow (-50, -33, -10, -10, "Save CDS", StdCloseWindowProc);
  dbdp = (DialogBoxDataPtr) MemNew (sizeof (DialogBoxData));
  SetObjectExtra (wdialog, (Pointer) dbdp, StdCleanupExtraProc);
  dbdp->w = w;
  g = HiddenGroup (wdialog, 1, 0, NULL);
  StaticPrompt (g, "Closing the window will lose unsaved features.", 0, dialogTextHeight,systemFont, 'l');
  c = HiddenGroup (wdialog, 2, 0, NULL);
  PushButton (c, "OK", AcceptCloseWithoutSaveCDSProc);
  PushButton (c, "Cancel", StdCancelButtonProc);
  AlignObjects (ALIGN_CENTER, (HANDLE) g, (HANDLE) c, NULL);
  RealizeWindow (wdialog);
  Show (wdialog);
  return;
}

static void AcceptCloseWindowButton (ButtoN b)
{
  WindoW             w;
  EditAlignDataPtr   adp;

  w = (WindoW)ParentWindow (b);
  if ((adp = GetAlignEditData (w)) == NULL ) return;
  ObjMgrSetDirtyFlag (adp->input_entityID, TRUE);
  if (adp->feat != NULL) {
     SaveCDSConfirmDlg (w);
  }
  else AcceptCloseFunc (w, adp);
}

/*********************************************************
***
***
*********************************************************/
static void UndoProc (IteM i)
{
  PaneL              pnl;
  EditAlignDataPtr   adp;
  SelStructPtr       ssp = NULL;
  SeqEntryPtr        prevsep, currentsep;
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL) return;
  if (adp->tmpfile != NULL && adp->tmpfile[0] != '\0') 
  {
     if (adp->master.entityID > 0) {
        Update ();
        prevsep = seqentry_read (adp->tmpfile);
        currentsep = GetTopSeqEntryForEntityID (adp->master.entityID);
        ReplaceSeqEntryWithSeqEntry (currentsep, prevsep, TRUE);
        ObjMgrSendMsg (OM_MSG_UPDATE, adp->master.entityID, adp->master.itemID, adp->master.itemtype);
     }
  }
  return;
}

/*********************************************************
***  CutProc
***    get the selected segment, call do_cut 
***    Deselect the segment
**********************************************************/
static void CutProc (IteM i)
{
  PaneL              pnl;
  EditAlignDataPtr   adp;
  SelStructPtr       ssp = NULL;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL) return;
  if (adp->input_format == OBJ_BIOSEQ)
  {
     if ( checkOMss_for_itemtype (OBJ_BIOSEQ) == 0 ) return;
     ssp = ObjMgrGetSelected ();
     for (; ssp != NULL; ssp = ssp->next)
        if ( ssp->itemtype == OBJ_BIOSEQ && checkssp_for_editor (ssp) ) 
           break;
     if (ssp != NULL) {
        do_cut (pnl, adp, ssp, TRUE);
     }
  }
  else if (adp->input_format == OBJ_SEQALIGN) {
     Beep ();
  }
  return;
}

/******************************
***  PasteProc
***     
*******************************/
static void PasteProc (IteM i)
{
  PaneL              pnl;
  EditAlignDataPtr   adp;
  SelStructPtr       ssp = NULL;
  BioseqPtr          bsp = NULL;
  ObjMgrDataPtr      omdp = NULL;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) 
     return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) 
     return;
  if (adp->input_format != OBJ_BIOSEQ)
     return;
  if (ClipboardHasString ()) 
  {
     CharPtr     str = ClipboardToString ();
     CharPtr     strtmp;
     size_t      len = StringLen (str);

     strtmp = char_to_insert (str, len, adp->mol_type);
     if (insertchar_atcaret (strtmp, adp)) {
        adp->dirty = TRUE;
        ObjMgrSendMsg (OM_MSG_UPDATE, adp->caret.entityID, adp->caret.itemID, adp->caret.itemtype);
     }
     MemFree (strtmp);
     return;
  }
  if ( adp->caret.regiontype == 0 ) {
        Message (MSG_ERROR, "Click before paste");
        return;
  }
  omdp = ObjMgrGetClipBoard ();
  if (omdp == NULL) {
        Message (MSG_ERROR, "ClipBoard empty");
        return;
  }
  if (omdp->datatype != OBJ_BIOSEQ || omdp->dataptr == NULL) {
        Message (MSG_ERROR, "Error: ClipBoard is not BIOSEQ");
        return;
  }
  bsp = (BioseqPtr) omdp->dataptr;
  if (bsp!=NULL && bsp->length > 0) {
     if (ISA_na (bsp->mol) != ISA_na (adp->mol_type)) {
        Message(MSG_ERROR,"Error: Pasting uncorrect molecule type [%d / %d]", 
                  bsp->mol, adp->mol_type);
     }
     else {
        ssp = ObjMgrGetSelected ();
        if (ssp!=NULL) {
           if (checkssp_for_editor(ssp) && ssp->itemtype==OBJ_BIOSEQ) {
              if (SeqIdForSameBioseq (SeqLocId((SeqLocPtr)(adp->caret.region)), SeqLocId((SeqLocPtr)(ssp->region))) )
                 if (do_cut (pnl, adp, ssp, FALSE)) 
                    adp->dirty = TRUE;
           }
        }
        else ssp = NULL;
        if (do_paste (pnl, adp, bsp->id) ) {
           adp->dirty = TRUE;
           ObjMgrSendMsg (OM_MSG_UPDATE, adp->caret.entityID, adp->caret.itemID, adp->caret.itemtype);
        }
     }
  }
  return;
}

/******************************
*******************************/
static void DeleteProc (IteM i)
{
  PaneL              pnl;
  EditAlignDataPtr   adp;
  SelStructPtr       ssp = NULL;
  Boolean            nocds = TRUE;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if (checkOMss_for_itemtype (OBJ_VIRT) > 0) nocds = FALSE;
  ssp = ObjMgrGetSelected ();
  if (ssp!=NULL) {
     for (; ssp != NULL; ssp = ssp->next) {
       if (checkssp_for_editor (ssp))
       {
        if ( ssp->itemtype == OBJ_BIOSEQ && nocds ) {
            if (adp->input_format == OBJ_BIOSEQ)
            {
               if ( do_cut (pnl, adp, ssp, FALSE) ) {
                  return;
               }
            }
        }
        else if ( ssp->itemtype == OBJ_VIRT ) {
            adp->feat = del_ssp_fromid (adp->feat, 255, ss_to_ses(ssp));
            adp->dirty = TRUE;
            ObjMgrSendMsg (OM_MSG_UPDATE, ssp->entityID, ssp->itemID, ssp->itemtype);
            ObjMgrDeSelect (ssp->entityID, ssp->itemID, ssp->itemtype, ssp->regiontype, ssp->region);
            return;
        }
        else if (ssp->itemtype == OBJ_SEQFEAT ) {
            Message (MSG_ERROR, "Can't delete a feature");
            return;
        }
       }
     }
  }
  return;
}

static void DeleteSeqId (IteM i)
{
  WindoW           temport;
  PaneL              pnl;
  EditAlignDataPtr   adp;
  SeqAlignPtr        salp;
  SelStructPtr       ssp;
  SeqIdPtr           sip;
  SeqLocPtr          slp;
  Int2               j;

  if ((pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) 
     return;
  if ((adp = GetAlignDataPanel(pnl))!= NULL) {
     if ((j = GetNumberObjMgrSelect()) < 1) {
        Message(MSG_ERROR, "Please select sequence(s)");
        return;
     }
     if (j >= adp->seqnumber) {
        Message(MSG_ERROR, "You will delete all the sequences except the last one");
        return;
     }
     temport = SavePort((WindoW)ParentWindow (i));
     Select (pnl);
     ssp = ObjMgrGetSelected();
     for (; ssp != NULL; ssp = ssp->next) 
      if (checkssp_for_editor (ssp) && ssp->itemtype == OBJ_BIOSEQ)
      {
        slp = (SeqLocPtr) ssp->region;
        sip = SeqLocId(slp);
        salp = (SeqAlignPtr)adp->sap_align->data;
        salp = SeqAlignBoolSegToDenseSeg (salp);
        if (sip!=NULL && salp!=NULL) 
        {   
           DeleteSeqIdInSeqAlign (salp, sip);
           adp = EditAlignDataRepopulateFromSeqAlign (pnl, adp, salp);
        }
      }
     inval_panel (pnl, -1 , -1);
     RestorePort (temport);
  }
}

static void ShowAllSeq (IteM i)
{
  WindoW           temport;
  PaneL            pnl;
  EditAlignDataPtr adp;
  ValNodePtr       vnp;
  SeqParamPtr      prm;

  if ((pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL)
     return;
  if ((adp = GetAlignDataPanel(pnl))!= NULL) {
     for (vnp = adp->params; vnp!=NULL; vnp=vnp->next) {
        prm = (SeqParamPtr) vnp->data.ptrvalue;
        prm->visible = TRUE;
     }
  }
  data_collect_arrange (adp, TRUE);
  temport = SavePort (pnl);
  Select (pnl);
  inval_panel (pnl, -1, -1);
  RestorePort (temport);
  return;
}


/**********************************************************
***
***  ChangeCharModeProc:   
***     adp->charmode shows all char 
***     otherwise dots replacing chars identical to master's
***
**********************************************************/
static void ChangeCharModeProc (IteM i)
{
  PaneL            pnl;
  WindoW           temport;
  EditAlignDataPtr adp;
 
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  adp->charmode = GetStatus (i);
  ResetClip ();
  if ( !adp->charmode )
        SetTitle (i, "Show differences");
  else  SetTitle (i, "Show all");
  temport = SavePort((WindoW)ParentWindow (i));
  Select (pnl);
  inval_panel (pnl, -1, -1);
  RestorePort (temport);
  return;
}

/*********************************************************
***    
***  SelectAllProc: 
***    SelectAllSeqEntry + highlight all the sequence Ids 
***      
**********************************************************/
static void SelectAllProc (IteM i)
{
  WindoW           temport;
  PaneL            pnl;
  EditAlignDataPtr adp;
  ValNodePtr       vnp;
  SeqLocPtr        slp;
  AlignNodePtr     anp;
  Uint2            ei, ii;
  Boolean          first = TRUE;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  temport = SavePort((WindoW)ParentWindow (i));
  Select (pnl);
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if ( adp->seqnumber == 0 || adp->anp_list == NULL) return;
  for (vnp =adp->anp_list; vnp !=NULL; vnp =vnp->next)
  {
     anp = (AlignNodePtr) vnp->data.ptrvalue;
     if (anp != NULL)
     {   
           ei = anp->seq_entityID;
           ii = anp->bsp_itemID;
           slp = CollectSeqLocFromAlignNode (anp);
           if (slp != NULL) { 
              if (first) {
                 ObjMgrSelect (ei, ii, OBJ_BIOSEQ, OM_REGION_SEQLOC, slp);
                 first = FALSE;
              }
              else {
                 ObjMgrAlsoSelect (ei, ii, OBJ_BIOSEQ, OM_REGION_SEQLOC, slp);
              }
           }
     }   
  }  
  RestorePort (temport);
  return;
}

/*********************************************************
***
***  SelectMasterProc
***
**********************************************************/
static void SelectMasterProc (IteM i)
{
  WindoW           temport;
  PaneL            pnl;
  EditAlignDataPtr adp;
  SelStructPtr     ssp;
  SeqLocPtr        slp, slptmp;
  RecT             rp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if ( adp->seqnumber == 0 ) return;
  temport = SavePort((WindoW)ParentWindow (i));
  Select (pnl);
  ssp = ObjMgrGetSelected();  
  if ( !checkssp_for_editor (ssp) ) return;
  if ( ssp->itemtype == OBJ_BIOSEQ && (adp->master.itemID != ssp->itemID
     || adp->master.entityID != ssp->entityID )) 
  {
         adp->master.entityID = ssp->entityID;
         adp->master.itemID = ssp->itemID;
         SeqLocFree ((SeqLocPtr) adp->master.region);
         slp = (SeqLocPtr) ssp->region;
         slptmp = SeqLocIntNew (SeqLocStart(slp), SeqLocStop(slp), SeqLocStrand(slp), SeqLocId(slp));
         adp->master.region = (Pointer) slptmp;
         if (adp->charmode) {
            inval_panel (pnl, -1, -1);
         }
         else {
            get_client_rect (pnl, &rp);
            inval_rect (rp.left, rp.top, (Int2)(rp.left+adp->margin.left-1), rp.bottom);
         }
  }
  RestorePort (temport);
}

/**********************************************************
***
*** RefreshAlignData: refreches the window
***
***********************************************************/
static void RefreshAlignDataProc (PaneL pnl)
{
  WindoW           temport;
  EditAlignDataPtr adp;

  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if ( adp->seqnumber == 0 ) return;
  data_collect_arrange (adp, TRUE);
  temport = SavePort (pnl);
  Select (pnl);
  inval_panel (pnl, -1, -1);
  RestorePort (temport);
  return;
}

static void RefreshAlignDataItem (IteM i)
{
  PaneL            pnl;
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  RefreshAlignDataProc (pnl);
  CaptureSlateFocus ((SlatE) pnl);
}

static void RefreshAlignDataButton (ButtoN b)
{
  PaneL            pnl;
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (b)) ) == NULL) return;
  RefreshAlignDataProc (pnl);
  CaptureSlateFocus ((SlatE) pnl);
}

/**********************************************************
***
***  Complement
***
***  SetRf, SetRfFunc:
***
***********************************************************/
static void ComplementProc (EditAlignDataPtr adp, Boolean choice)
{
  SelStructPtr     ssp;
  ValNodePtr       vnp;
  SeqParamPtr      prm;

  if ( checkOMss_for_itemtype (OBJ_BIOSEQ) == 0 ) {
     ssp = &(adp->master);
  }
  else ssp = ObjMgrGetSelected();  
  for (; ssp != NULL; ssp = ssp->next) {
     if (checkssp_for_editor (ssp) && ssp->itemtype == OBJ_BIOSEQ) 
     {
        for (vnp = adp->params; vnp != NULL; vnp = vnp->next) {
           prm = (SeqParamPtr) vnp->data.ptrvalue;
           if ( prm->entityID == ssp->entityID )
              prm->complement = !(prm->complement);
        } 
     }
  }
  return; 
}

static void ComplementItemProc (IteM i)
{
  WindoW           temport;
  PaneL            pnl;
  EditAlignDataPtr adp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if (adp->seqnumber == 0 || ISA_aa(adp->mol_type))
    return;
  adp->drawcomplt = GetStatus (i);
  temport = SavePort((WindoW)ParentWindow (i));
  Select (pnl);
  inval_panel (pnl, -1, -1);
  ComplementProc (adp, adp->drawcomplt);
  data_collect_arrange (adp, TRUE);
  SetCorrectVBarMax (pnl, adp->nlines, adp->voffset);
  RestorePort (temport);
}

static void SetRfFunc (PaneL pnl, Uint2 rf)
{
  WindoW           temport;
  EditAlignDataPtr adp;
  SeqEditViewFormPtr wdp;
  SelStructPtr     ssp;
  ValNodePtr       vnp;
  SeqParamPtr      prm;
  Int2             j;

  wdp = (SeqEditViewFormPtr) GetObjectExtra ((WindoW)ParentWindow (pnl));
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if (adp->seqnumber == 0 || ISA_aa(adp->mol_type))
    return;
  temport = SavePort(pnl);
  Select (pnl);
  inval_panel (pnl, -1, -1);
  if ( checkOMss_for_itemtype (OBJ_BIOSEQ) == 0 ) {
         ssp = &(adp->master);
  }
  else ssp = ObjMgrGetSelected();  
  for (; ssp != NULL; ssp = ssp->next) {
     if (checkssp_for_editor (ssp) && ssp->itemtype == OBJ_BIOSEQ) {
        for (vnp = adp->params; vnp != NULL; vnp = vnp->next) {
           prm = (SeqParamPtr) vnp->data.ptrvalue;
           if ( prm->entityID == ssp->entityID ) {
             if (rf < (Uint2) 6) {
                prm->rf[rf] = TRUE; 
             }
             switch ((int)rf) {
               case 0: 
                       prm->rf[1] = prm->rf[2] = FALSE; 
                       SetStatus (wdp->rfitem [1], FALSE);
                       SetStatus (wdp->rfitem [2], FALSE);
                       SetStatus (wdp->rfitem [6], FALSE);
                       SetStatus (wdp->rfitem [8], FALSE);
                       break;
               case 1: 
                       prm->rf[0] = prm->rf[2] = FALSE; 
                       SetStatus (wdp->rfitem [0], FALSE);
                       SetStatus (wdp->rfitem [2], FALSE);
                       SetStatus (wdp->rfitem [6], FALSE);
                       SetStatus (wdp->rfitem [8], FALSE);
                       break;
               case 2: 
                       prm->rf[0] = prm->rf[1] = FALSE; 
                       SetStatus (wdp->rfitem [0], FALSE);
                       SetStatus (wdp->rfitem [1], FALSE);
                       SetStatus (wdp->rfitem [6], FALSE);
                       SetStatus (wdp->rfitem [8], FALSE);
                       break;
               case 3: 
                       prm->rf[4] = prm->rf[5] = FALSE; 
                       SetStatus (wdp->rfitem [4], FALSE);
                       SetStatus (wdp->rfitem [5], FALSE);
                       SetStatus (wdp->rfitem [7], FALSE);
                       SetStatus (wdp->rfitem [8], FALSE);
                       break;
               case 4: 
                       prm->rf[3] = prm->rf[5] = FALSE; 
                       SetStatus (wdp->rfitem [3], FALSE);
                       SetStatus (wdp->rfitem [5], FALSE);
                       SetStatus (wdp->rfitem [7], FALSE);
                       SetStatus (wdp->rfitem [8], FALSE);
                       break;
               case 5: 
                       prm->rf[3] = prm->rf[4] = FALSE; 
                       SetStatus (wdp->rfitem [3], FALSE);
                       SetStatus (wdp->rfitem [4], FALSE);
                       SetStatus (wdp->rfitem [7], FALSE);
                       SetStatus (wdp->rfitem [8], FALSE);
                       break;
               case 6: if (GetStatus (wdp->rfitem [6]) == FALSE) {
                        prm->rf[0] =FALSE; prm->rf[1] =FALSE; prm->rf[2]=FALSE; 
                       } else {
                        prm->rf[0] =TRUE; prm->rf[1] =TRUE; prm->rf[2] =TRUE; 
                       }
                       SetStatus (wdp->rfitem [0], FALSE);
                       SetStatus (wdp->rfitem [1], FALSE);
                       SetStatus (wdp->rfitem [2], FALSE);
                       SetStatus (wdp->rfitem [8], FALSE);
                       break;
               case 7: if (GetStatus (wdp->rfitem [7]) == FALSE) {
                        prm->rf[3] =FALSE; prm->rf[4] =FALSE; prm->rf[5]=FALSE; 
                       } else {
                        prm->rf[3] = TRUE; prm->rf[4] = TRUE; prm->rf[5] =TRUE; 
                       }
                       SetStatus (wdp->rfitem [3], FALSE);
                       SetStatus (wdp->rfitem [4], FALSE);
                       SetStatus (wdp->rfitem [5], FALSE);
                       SetStatus (wdp->rfitem [8], FALSE);
                       break;
               case 8: 
                       if (GetStatus (wdp->rfitem [8]) == FALSE) {
                        prm->rf[0] =FALSE; prm->rf[1] =FALSE; prm->rf[2]=FALSE; 
                        prm->rf[3] =FALSE; prm->rf[4] =FALSE; prm->rf[5]=FALSE; 
                       } else {
                        prm->rf[0] =TRUE; prm->rf[1] =TRUE; prm->rf[2] =TRUE; 
                        prm->rf[3] =TRUE; prm->rf[4] =TRUE; prm->rf[5] =TRUE; 
                       }
                       for (j=0; j<8; j++) 
                          SetStatus (wdp->rfitem [0], FALSE);
                       break;
               case 9: 
                       prm->rf[0] =FALSE; prm->rf[1] =FALSE; prm->rf[2]=FALSE; 
                       prm->rf[3] =FALSE; prm->rf[4] =FALSE; prm->rf[5]=FALSE; 
                       for (j=0; j<9; j++) 
                          SetStatus (wdp->rfitem [j], FALSE);
                       break;
               default : break;
             } 
           } 
        } 
     }
  }
  data_collect_arrange (adp, TRUE);
  SetCorrectVBarMax (pnl, adp->nlines, adp->voffset);
  RestorePort (temport);
}

static void rf1ItemProc (IteM i) {
   SetRfFunc ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)0);
}
static void rf2ItemProc (IteM i) {
   SetRfFunc ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)1);
}
static void rf3ItemProc (IteM i) {
   SetRfFunc ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)2);
}
static void rf4ItemProc (IteM i) {
   SetRfFunc ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)3);
}
static void rf5ItemProc (IteM i) {
   SetRfFunc ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)4);
}
static void rf6ItemProc (IteM i) {
   SetRfFunc ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)5);
}
static void rf7ItemProc (IteM i) {
   SetRfFunc ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)6);
}
static void rf8ItemProc (IteM i) {
   SetRfFunc ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)7);
}
static void rf9ItemProc (IteM i) {
   SetRfFunc ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)8);
}
static void rf10ItemProc (IteM i) {
   SetRfFunc ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)9);
}

static void changeseqid (PaneL pnl, Uint2 choice)
{
  WindoW           temport;
  EditAlignDataPtr adp;

  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) 
     return;
  if (adp->seqnumber == 0)
    return;
  if (choice < 6) {
     adp->printid = choice;
     adp->size_labels = getsize_seqids (adp->sqloc_list, choice);
     adp->marginwithIds = TRUE;
  }
  if (choice == 6) {
     adp->printid = 0;
     adp->size_labels = 15;
     adp->marginwithIds = FALSE;
  }
  temport = SavePort(pnl);
  Select (pnl);
  inval_panel (pnl, -1, -1);
  RestorePort (temport);
}
static void changeseqid1 (IteM i) {
  changeseqid ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)PRINTID_FASTA_SHORT);
}
static void changeseqid2 (IteM i) {
  changeseqid ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)PRINTID_FASTA_LONG);
}
static void changeseqid3 (IteM i) {
  changeseqid ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)PRINTID_TEXTID_LOCUS);
}
static void changeseqid4 (IteM i) {
  changeseqid ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)PRINTID_TEXTID_ACCESSION);
}
static void changeseqid5 (IteM i) {
  changeseqid ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)PRINTID_REPORT);
}
static void changeseqid6 (IteM i) {
  changeseqid ((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), (Uint2)6);
}

/******************************************************************
***
***  MakemRNAProc , MakemRNArProc
***  MakeCdRgnProc , MakeCdRgnrProc
***
***  SaveFeatureProc
***  SaveFeatureButton
***
*******************************************************************/
static void MakemRNAProc (IteM i) {
  MakeFeatProc((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), SEQFEAT_RNA, Seq_strand_plus);
}

static void MakemRNArProc (IteM i) {
  MakeFeatProc((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), SEQFEAT_RNA, Seq_strand_minus);
}

static void MakeCdRgnProc (IteM i) {
  MakeFeatProc((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), SEQFEAT_CDREGION, Seq_strand_plus);
}

static void MakeCdRgnrProc (IteM i) {
  MakeFeatProc((PaneL)GetPanelFromWindow((WindoW)ParentWindow (i)), SEQFEAT_CDREGION, Seq_strand_minus);
}

static void SaveFeatureProc (IteM i)
{
  PaneL              pnl;
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  SaveFeatProc (pnl);
}

static void SaveFeatureButton (ButtoN b)
{
  PaneL              pnl;
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (b)) ) == NULL) return;
  SaveFeatProc (pnl);
  CaptureSlateFocus ((SlatE) pnl);
}

/******************************************************************
***
***  TranslateProc , TranslateButton
***  UntranslateProc
***
*******************************************************************/
static void TranslateProc (IteM i)
{
  PaneL            pnl;
  EditAlignDataPtr adp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if ( adp->seqnumber == 0 ) return;
  if ( checkOMss_for_itemtype (OBJ_VIRT) != 0 
     || checkOMss_for_itemtype (OBJ_SEQFEAT) != 0 ) { 
     CdRgnToProtProc (pnl, adp);
     if (!adp->display_panel) {
        update_translateitem (pnl, adp->seqfeat, adp->feat);
        update_codonstartbt (pnl, adp->seqfeat, adp->feat);
     }
  }
}

static void TranslateButton (ButtoN b)
{
  PaneL            pnl;
  EditAlignDataPtr adp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (b)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if ( adp->seqnumber == 0 ) return;
  if (GetNumberObjMgrSelect() == 0)
     TranslateAllBioseq (pnl, adp);
  if ( checkOMss_for_itemtype (OBJ_VIRT) != 0 
     || checkOMss_for_itemtype (OBJ_SEQFEAT) != 0 ) {
     CdRgnToProtProc (pnl, adp);
     if (!adp->display_panel) {
        update_translateitem (pnl, adp->seqfeat, adp->feat);
        update_codonstartbt (pnl, adp->seqfeat, adp->feat);
     }
  }
  CaptureSlateFocus ((SlatE) pnl);
}

static void UntranslateProc (IteM i)
{
  PaneL            pnl;
  EditAlignDataPtr adp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if ( adp->seqnumber == 0 ) return;
  if ( checkOMss_for_itemtype (OBJ_VIRT) == 0 
    && checkOMss_for_itemtype (OBJ_SEQFEAT) == 0 ) return;
  UntranslateFunc (pnl, adp);
  if (!adp->display_panel) {
     update_translateitem (pnl, adp->seqfeat, adp->feat);
     update_codonstartbt (pnl, adp->seqfeat, adp->feat);
  }
}

/******************************************************************
***
***  CodonStartProc
***  CodonStartButton , CodonStartItem
***
***   DO NOT WORK YET
*******************************************************************/
static void CodonStartProc (SeqEditViewFormPtr wdp)
{
  PaneL              pnl;
  EditAlignDataPtr   adp;
  SelEdStructPtr     feat;
  Char               str[52];
  RecT               rp;
  Int2               k;

  pnl = wdp->pnl;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  k = checkCDSselect_forprotein (adp->seqfeat, adp->feat, TRUE);
  if (k != 1) 
     return;
  feat = getCDSselect (adp->seqfeat, adp->feat);
  if (feat->codonstart == 1)      feat->codonstart = 2;
  else if (feat->codonstart == 2) feat->codonstart = 3;
  else if (feat->codonstart == 3) feat->codonstart = 1;
  sesp_to_pept (feat, (SeqAlignPtr) adp->sap_align->data, adp->sqloc_list, FALSE);
  get_client_rect (pnl, &rp);
  inval_pep (adp, &rp);
  sprintf (str, "Codon start %d", (int)(feat->codonstart));
  ResetClip ();
  SetTitle (wdp->codonstitem, str);
  CaptureSlateFocus ((SlatE) pnl);
}

static void CodonStartButton (ButtoN b)
{
  SeqEditViewFormPtr wdp;

  wdp = (SeqEditViewFormPtr) GetObjectExtra ((WindoW)ParentWindow (b));
  CodonStartProc (wdp);
}

static void CodonStartItem (IteM i)
{
  SeqEditViewFormPtr wdp;

  wdp = (SeqEditViewFormPtr) GetObjectExtra ((WindoW)ParentWindow (i));
  CodonStartProc (wdp);
}

/***********************************************************
***  EditModeProc
************************************************************/
static void SelectEditMode (PopuP p)
{
  WindoW             w, temport;
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  Int2               val;

  w = (WindoW)ParentWindow (p);
  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  if ( ( adp = GetAlignDataPanel (wdp->pnl) ) == NULL ) return;
  if (adp->input_format == OBJ_SEQALIGN) {
     val = GetValue(p); 
     if (val==1) 
        adp->edit_mode = ALIGN_EDIT;
     else if (val==3) 
        adp->edit_mode = PRETTY_EDIT;
     else 
        adp->edit_mode = 0;
     if (adp->edit_mode == PRETTY_EDIT && adp->copyalign == NULL) {
        adp->copyalign = CreateCopyAlign (adp->anp_list, adp->length);
        ColorIdentityDialog (w);
     }
     if (adp->edit_mode == ALIGN_EDIT) {
        temport = SavePort(w);
        Select (wdp->pnl);
        inval_panel (wdp->pnl, -1, -1);
        RestorePort (temport);
     }
  }
  CaptureSlateFocus ((SlatE) wdp->pnl);
}

/***********************************************************
***  FeatEditModeProc
************************************************************/
static void SelectFeatEditMode (PopuP p)
{
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  Int2               val;

  wdp = (SeqEditViewFormPtr) GetObjectExtra ((WindoW)ParentWindow (p));
  if ( ( adp = GetAlignDataPanel (wdp->pnl) ) == NULL ) return;
  val = GetValue(p); 
  if (val==1) 
     adp->spliteditmode = FALSE;
  else if (val==2) 
     adp->spliteditmode = TRUE;
  CaptureSlateFocus ((SlatE) wdp->pnl);
}

/***********************************************************
***  ShowFeatureItemProc
***  ShowFeatureButtonProc
************************************************************/
static void ShowFeatureItemProc (IteM i)
{
  PaneL              pnl;
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  ShowFeatureProc (pnl, TRUE);
}
 
static void ShowFeatureButtonProc (ButtoN b)
{
  PaneL              pnl;
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (b)) ) == NULL) return;
  ShowFeatureProc (pnl, TRUE);
  CaptureSlateFocus ((SlatE) pnl);
}

/***********************************************************
***  ChangeProtModeProc
************************************************************/
static void ChangeProtModeProc (IteM i)
{
  WindoW           temport;
  PaneL            pnl;
  EditAlignDataPtr adp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if ( adp->seqnumber == 0 ) return;
  adp->prot_mode = ALLPROTAA;
  adp->firstssp = NULL;
  data_collect_arrange (adp, FALSE);
  temport = SavePort((WindoW)ParentWindow (i));
  Select (pnl);
  inval_panel (pnl, -1, -1);
  RestorePort (temport);
}

static void ChangeProtModeProc2 (IteM i)
{
  WindoW           temport;
  PaneL            pnl;
  EditAlignDataPtr adp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if ( adp->seqnumber == 0 ) return;
  adp->prot_mode = MPROTAA;
  temport = SavePort((WindoW)ParentWindow (i));
  Select (pnl);
  inval_panel (pnl, -1, -1);
  RestorePort (temport);
}

static void ChangeProtModeProc3 (IteM i)
{
  WindoW           temport;
  PaneL            pnl;
  EditAlignDataPtr adp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if ( adp->seqnumber == 0 ) return;
  adp->prot_mode = PUTPROT;
  adp->firstssp = NULL;
  data_collect_arrange (adp, FALSE);
  temport = SavePort((WindoW)ParentWindow (i));
  Select (pnl);
  inval_panel (pnl, -1, -1);
  RestorePort (temport);
}


static void NextPatternItem (IteM i)
{
  PaneL            pnl;
  EditAlignDataPtr adp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  adp->cur_pat = ShowNextPattern (adp->match_pat, adp->cur_pat, adp->edit_pos);
}

static void PrecPatternItem (IteM i)
{
  PaneL            pnl;
  EditAlignDataPtr adp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  adp->cur_pat = ShowPrecPattern (adp->match_pat, adp->cur_pat, adp->edit_pos);
}

/*********************************************************
***
**********************************************************/
static void SeqAlignToHist (IteM i)
{
  PaneL       pnl;
  EditAlignDataPtr adp = NULL;
  SeqAlignPtr salp;
  SeqLocPtr    slp;
  BioseqPtr    bsp;
  SeqEntryPtr  sep;
 
  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  sep = GetTopSeqEntryForEntityID (adp->input_entityID);
  salp= (SeqAlignPtr) FindSeqAlignInSeqEntry (sep, OBJ_SEQALIGN);
  slp = SeqLocFromSeqAlign (salp, NULL);
  bsp = BioseqLockById (SeqLocId(slp));
  if (bsp!=NULL) {
     bsp->hist = SeqHistNew ();
     bsp->hist->assembly = salp;
     BioseqUnlock (bsp);
  }
}

/*****************************************************************
***    SetupMenus
***
  CommandItem (m, "Open Sequences", OpenSequenceProc);
  CommandItem (m, "Open SeqAnnot",  OpenSeqAnnotProc); 
  CommandItem (m, "Open Alignment", OpenAlignmentProc);
  CommandItem (m, "Add Alignment", OpenAlignmentProc);

*******************************************************************/
static void EditorEditMenu (MenU m, SeqEditViewFormPtr wdp)
{
  CommandItem (m, "Undo", UndoProc);
  SeparatorItem (m);
  wdp->cutitem   = CommandItem (m, "Cut", CutProc);
  Disable (wdp->cutitem);
  wdp->pasteitem = CommandItem (m, "Paste", PasteProc);
  Enable (wdp->pasteitem);
  wdp->copyitem  = CommandItem (m, "Copy", do_copy);
  Disable (wdp->copyitem);
  wdp->delitem   = CommandItem (m, "Clear", DeleteProc);
  Disable (wdp->delitem);
}
  
static void CommonEditMenu (MenU m)
{
  IteM i;

  CommandItem (m, "Refresh", RefreshAlignDataItem);
  SeparatorItem (m);
  i=CommandItem (m, "Find.../F", FindPatternDialog);
  Enable (i);
  i=CommandItem (m, "Find previous/P", PrecPatternItem);
  Enable (i);
  i=CommandItem (m, "Find next/N", NextPatternItem);
  Enable (i);
}

static void CommonViewMenu (MenU m, SeqEditViewFormPtr wdp, Boolean isa_aa)
{
  IteM  localItem;
  MenU  sub;

  sub = SubMenu (m, "Label");
    CommandItem (sub, "FASTA short", changeseqid1);
    CommandItem (sub, "FASTA long", changeseqid2);
    CommandItem (sub, "Locus", changeseqid3);
    CommandItem (sub, "Accession", changeseqid4);
    CommandItem (sub, "Report", changeseqid5);
    CommandItem (sub, "None", changeseqid6);
  
  if (!isa_aa) {
  localItem = StatusItem (m, "Complement", ComplementItemProc);
  SetStatus ( localItem, FALSE);
  sub = SubMenu (m, "Reading frames");
    wdp->rfitem[0] = StatusItem (sub, "1", rf1ItemProc);
    SetStatus ( wdp->rfitem [0], FALSE);
    wdp->rfitem[1] = StatusItem (sub, "2", rf2ItemProc);
    SetStatus ( wdp->rfitem [1], FALSE);
    wdp->rfitem[2] = StatusItem (sub, "3", rf3ItemProc);
    SetStatus ( wdp->rfitem [2], FALSE);
    wdp->rfitem[6] = StatusItem (sub, "all+", rf7ItemProc);
    SetStatus ( wdp->rfitem [6], FALSE);
    SeparatorItem (sub);
    wdp->rfitem[3] = StatusItem (sub, "4", rf4ItemProc);
    SetStatus ( wdp->rfitem [3], FALSE);
    wdp->rfitem[4] = StatusItem (sub, "5", rf5ItemProc);
    SetStatus ( wdp->rfitem [4], FALSE);
    wdp->rfitem[5] = StatusItem (sub, "6", rf6ItemProc);
    SetStatus ( wdp->rfitem [5], FALSE);
    wdp->rfitem[7] = StatusItem (sub, "all-", rf8ItemProc);
    SetStatus ( wdp->rfitem [7], FALSE);
    SeparatorItem (sub);
    wdp->rfitem[8] = StatusItem (sub, "all", rf9ItemProc);
    SetStatus ( wdp->rfitem [8], FALSE);
    wdp->rfitem[9] = StatusItem (sub, "none", rf10ItemProc);
    SetStatus ( wdp->rfitem [0], FALSE);
  SeparatorItem (m);
  CommandItem (m, "Preferences", DefinePanelDialog);
  sub = SubMenu (m, "Translation style");
    CommandItem (sub, "all", ChangeProtModeProc);
    CommandItem (sub, "M*", ChangeProtModeProc2);
    CommandItem (sub, "orf", ChangeProtModeProc3);
  }
  else { 
     CommandItem (m, "Preferences", DefinePanelDialog);
  }
/*
  SeparatorItem (m);
  VSMAddToMenu (m, VSM_DESKTOP);
*/
}

typedef struct newfeaturedata {
  ObjMgrProcPtr  ompp;
  IteM           item;
  Uint1          molgroup;
  struct newfeaturedata PNTR next;
} NewFeatureData, PNTR NewFeaturePtr;

static CharPtr  editOldSrcDescMsg = "\
You may really want to edit an existing BioSource descriptor instead.\n\
Proceed anyway?";

static void SalsaNewFeatureMenuProc (IteM i)

{
  EditAlignDataPtr  adp;
  MsgAnswer         ans;
  NewFeaturePtr     nfp;
  OMProcControl     ompc;
  ObjMgrProcPtr     ompp;
  PaneL             pnl;
  Int2              retval;

  pnl = (PaneL) GetPanelFromWindow ((WindoW) ParentWindow (i));
  if (pnl == NULL) return;
  adp = GetAlignDataPanel (pnl);
  if (adp == NULL) return;
  nfp = (NewFeaturePtr) GetObjectExtra (i);
  if (nfp == NULL) return;
  ompp = nfp->ompp;
  if (ompp == NULL || ompp->func == NULL) return;
  if (ompp->subinputtype == FEATDEF_BIOSRC) {
    ans = Message (MSG_YN, editOldSrcDescMsg);
    if (ans == ANS_NO) return;
  }
  MemSet ((Pointer) (&ompc), 0, sizeof (OMProcControl));
  ompc.input_entityID = adp->master.entityID;
  ompc.input_itemID = adp->master.itemID;
  ompc.input_itemtype = adp->master.itemtype;
  GatherDataForProc (&ompc, FALSE);
  ompc.proc = ompp;
  retval = (*(ompp->func)) (&ompc);
  if (retval == OM_MSG_RET_ERROR) {
    ErrShow ();
  }
}

static void SalsaNewFeaturesMenu (MenU m, Boolean is_na)

{
  FeatDispGroupPtr  fdgp;
  FeatDefPtr        fdp;
  NewFeaturePtr     first;
  IteM              i;
  Uint1             key;
  CharPtr           label;
  NewFeaturePtr     last;
  NewFeaturePtr     nfp;
  ObjMgrPtr         omp;
  ObjMgrProcPtr     ompp;
  ObjMgrTypePtr     omtp;
  MenU              sub;
  Uint2             subtype;

  if (m == NULL) return;
  omp = ObjMgrGet ();
  if (omp == NULL) return;
  ompp = ObjMgrProcFindNext (omp, OMPROC_EDIT, 0, 0, NULL);
  if (ompp == NULL) return;
  omtp = NULL;
  first = NULL;
  last = NULL;
  while ((omtp = ObjMgrTypeFindNext (omp, omtp)) != NULL) {
    ompp = ObjMgrProcFindNext (omp, OMPROC_EDIT, omtp->datatype, 0, NULL);
    if (ompp != NULL) {
      switch (omtp->datatype) {
        case OBJ_SEQFEAT :
          fdgp = NULL;
          while ((fdgp = DispGroupFindNext (fdgp, &key, &label)) != NULL) {
            if (fdgp->groupkey != 0) {
              sub = SubMenu (m, fdgp->groupname);
              fdp = NULL;
              label = NULL;
              while ((fdp = FeatDefFindNext (fdp, &key, &label,
                     fdgp->groupkey, FALSE)) != NULL) {
                if (key != FEATDEF_BAD) {
                  ompp = NULL;
                  while ((ompp = ObjMgrProcFindNext (omp, OMPROC_EDIT,
				          omtp->datatype, 0, ompp)) != NULL) {
                    if (ompp->subinputtype == fdp->featdef_key &&
                        ompp->subinputtype != FEATDEF_PUB) {
                      i = CommandItem (sub, ompp->proclabel, SalsaNewFeatureMenuProc);
                      nfp = (NewFeaturePtr) MemNew (sizeof (NewFeatureData));
                      if (nfp != NULL) {
                        nfp->ompp = ompp;
                        nfp->item = i;
                        nfp->molgroup = fdp->molgroup;
                      }
                      if (first == NULL) {
                        first = nfp;
                      }
                      if (last != NULL) {
                        last->next = nfp;
                      }
                      last = nfp;
                      SetObjectExtra (i, (Pointer) nfp, StdCleanupExtraProc);
                      if ((is_na && (fdp->molgroup == 2 || fdp->molgroup == 3)) ||
                          ((! is_na) && (fdp->molgroup == 1 || fdp->molgroup == 3))) {
                      } else {
                        Disable (i);
                      }
                    }
                  }
                }
              }
            }
          }
          sub = SubMenu (m, "Remaining Features");
          fdp = NULL;
          label = NULL;
          while ((fdp = FeatDefFindNext (fdp, &key, &label, 0, FALSE)) != NULL) {
            if (key != FEATDEF_BAD) {
              ompp = NULL;
              while ((ompp = ObjMgrProcFindNext (omp, OMPROC_EDIT,
                      omtp->datatype, 0, ompp)) != NULL) {
                subtype = ompp->subinputtype;
                if (subtype == fdp->featdef_key &&
                    subtype != FEATDEF_PUB &&
                    subtype != FEATDEF_Imp_CDS &&
                    subtype != FEATDEF_misc_RNA &&
                    subtype != FEATDEF_precursor_RNA &&
                    subtype != FEATDEF_mat_peptide &&
                    subtype != FEATDEF_sig_peptide &&
                    subtype != FEATDEF_transit_peptide &&
                    subtype != FEATDEF_source &&
                    subtype != FEATDEF_site_ref) {
                  i = CommandItem (sub, ompp->proclabel, SalsaNewFeatureMenuProc);
                  nfp = (NewFeaturePtr) MemNew (sizeof (NewFeatureData));
                  if (nfp != NULL) {
                    nfp->ompp = ompp;
                    nfp->item = i;
                    nfp->molgroup = fdp->molgroup;
                  }
                  if (first == NULL) {
                    first = nfp;
                  }
                  if (last != NULL) {
                    last->next = nfp;
                  }
                  last = nfp;
                  SetObjectExtra (i, (Pointer) nfp, StdCleanupExtraProc);
                  if ((is_na && (fdp->molgroup == 2 || fdp->molgroup == 3)) ||
                      ((! is_na) && (fdp->molgroup == 1 || fdp->molgroup == 3))) {
                  } else {
                    Disable (i);
                  }
                }
              }
            }
          }
          break;
        default :
          break;
      }
    }
  }
}

static void SetupMenus (WindoW w, Boolean isa_aa, Boolean ext_dist_menu, Boolean ext_align_menu, Boolean ext_tree_menu)
{
  SeqEditViewFormPtr wdp;
  MenU               m1, m2, m3;
  MenU               sub;

  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  if (wdp == NULL) return;

  m1 = PulldownMenu (w, "Edit");
  if (!isa_aa)
     EditorEditMenu (m1, wdp);
  CommonEditMenu (m1);
#ifdef SALSA_DEBUG
  if (ext_tree_menu) 
  {
     SeparatorItem (m1);
     CommandItem (m1, "debug", NULL);
     CommandItem (m1, "Show tree", TreeViewItem);
  }
#endif
  SeparatorItem (m1);
  CommandItem (m1, "Dismiss", CloseWindowItemProc);

  m2 = PulldownMenu (w, "View");
  CommonViewMenu (m2, wdp, isa_aa);
  SeparatorItem (m2);
  CommandItem (m2, "Import", ImportProc);
  sub = SubMenu (m2, "Export");
  CommandItem (sub, "Text", ExportTextProc);
  CommandItem (sub, "FASTA", ExportFastaProc);
  
  m3 = PulldownMenu (w, "Features");    
  wdp->showfeatitem = CommandItem (m3, "Show Features", ShowFeatureItemProc);
  if (!isa_aa)
  {
     CommandItem (m3, "Temporary CDS +", MakeCdRgnProc);
     CommandItem (m3, "Temporary CDS -", MakeCdRgnrProc);
     SeparatorItem (m3);
     SalsaNewFeaturesMenu (m3, (! isa_aa));
  }
}

static void SetupMenus2 (WindoW w, Boolean isa_aa, Boolean two_seq, Boolean ext_dist_menu, Boolean ext_tree_menu)
{
  SeqEditViewFormPtr  wdp;
  MenU                m1, m2, m3, m4;
  MenU                sub;

  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  if (wdp == NULL) return;

  m1 = PulldownMenu (w, "Edit");
  CommonEditMenu (m1);
  Disable (wdp->pasteitem);
  if (ext_dist_menu)
  {
     SeparatorItem (m1);
     CommandItem (m1, "debug", NULL);
     CommandItem (m1, "Select", SelectSeqDialog);
     CommandItem (m1, "Hide", DeleteSeqId);
     CommandItem (m1, "Show all", ShowAllSeq);
  }
  SeparatorItem (m1);
  CommandItem (m1, "Dismiss", CloseWindowItemProc);

  m2 = PulldownMenu (w, "View");
  CommonViewMenu (m2, wdp, isa_aa);
  if (!isa_aa)
     CommandItem (m2, "Conservation", ColorIdentityDialogItem);
  SeparatorItem (m2);
  CommandItem (m2, "Import", ImportProc);
  sub = SubMenu (m2, "Export");
  CommandItem (sub, "Text", ExportTextProc);
  CommandItem (sub, "FASTA", ExportFastaProc);
  CommandItem (sub, "FASTA+gaps", ExportFastaGapProc);
  CommandItem (sub, "PHYLIP", ExportPhyProc);
  CommandItem (sub, "SeqAnnot", ExportSeqAnnotProc);

  m3 = PulldownMenu (w, "Features");    
  wdp->showfeatitem = StatusItem (m3, "Show Features", ShowFeatureItemProc);
  if (!isa_aa) {
     CommandItem (m3, "Propagate", PropagateFeatDialog); 
  }
  
  m4 = PulldownMenu (w, "Alignment");
  wdp->selmaster = CommandItem (m4, "Select reference", SelectMasterProc);
  CommandItem (m4, "Select all", SelectAllProc);
  wdp->showdiff = StatusItem (m4, "Show differences", ChangeCharModeProc);
  SeparatorItem (m4);
  CommandItem (m4, "Dot matrix (BLAST)", DotPlotItem);
  CommandItem (m4, "Align report", AlignReportItem);
  if (ext_dist_menu) 
  {
     SeparatorItem (m4);
     CommandItem (m4, "debug", NULL);
     CommandItem (m4, "Select region", SelectRegionDialog);
     CommandItem (m4, "Formula", FormulaDialog);
     CommandItem (m4, "Pw distance", PwDistanceItem);
     CommandItem (m4, "Pw dist per pos", DistPosItem);
     CommandItem (m4, "Gp distance", GpDistanceItem);
     CommandItem (m4, "Sort simil", SortbySimItem);
     CommandItem (m4, "Sort lenght", SortbyLenItem);
     CommandItem (m4, "Quorum", QuorumItem);
     CommandItem (m4, "Ks Ka gp", NonsyToSynItem1);
     CommandItem (m4, "Ks Ka gp wd", NonsyToSynItem2);
     CommandItem (m4, "Ks Ka ref", NonsyToSynItem3);
     CommandItem (m4, "Ks Ka ref wd", NonsyToSynItem4);
     CommandItem (m4, "Ks Ka btw gps", NonsyToSynItem5);
     CommandItem (m4, "Ks Ka btw gps wd", NonsyToSynItem6);
     CommandItem (m4, "SeqAlign2 Hist", SeqAlignToHist);
  }
#ifdef SALSA_DEBUG
  if (ext_tree_menu) 
  {
     SeparatorItem (m4);
     CommandItem (m4, "debug", NULL);
     CommandItem (m4, "Show tree", TreeViewItem);
  }
#endif
}

/***************************************************************
***    CleanupSequenceEditorForm 
***
***      sends a message calling ObjMgrFreeUserData
***      StdCleanupFormProc
******************************************************************/
static void CleanupSequenceEditorForm (GraphiC g, VoidPtr data)

{
  StdCleanupFormProc (g, data);
}

/**********************************************************
***
***  CleanupAlignDataPanel: Callback to free any instance
***     specific memory that may be pointed to in the extra. 
***
***********************************************************/
static void CleanupAlignDataPanel (GraphiC g, VoidPtr data)

{
  EditAlignDataPtr   adp;
  SeqEditViewFormPtr wdp;
  Int4               strlens;
  ValNodePtr  vnp;
  SeqEntryPtr sep;
  Uint2       eID;
  AlignNodePtr anp;
  OMUserDataPtr omudp;

  adp = (EditAlignDataPtr) data;
  if (adp != NULL) 
  {
     wdp = (SeqEditViewFormPtr) adp->wdp;
     if (wdp != NULL) {
      if (wdp->input_entityID > 0) 
      {
        ObjMgrFreeUserData (wdp->input_entityID, wdp->procid, wdp->proctype, wdp->userkey);

        for (vnp=adp->anp_list; vnp!=NULL; vnp=vnp->next) {
           anp = (AlignNodePtr) vnp->data.ptrvalue;
           if (anp!=NULL) {
              sep = GetTopSeqEntryForEntityID (anp->seq_entityID);
              if (sep!=NULL) {
                 eID = SeqMgrGetEntityIDForSeqEntry(sep);
                 omudp = ObjMgrGetUserData (eID, wdp->procid, OMPROC_EDIT, wdp->userkey);
                 if (omudp != NULL) 
                    ObjMgrFreeUserData (eID, wdp->procid, wdp->proctype, wdp->userkey);
              }
           }   
        }   
      }
      if (wdp->data!=NULL) {
         wdp->data = ValNodeFreeType (&(wdp->data), TypeEmpty);
      }
     } 
     strlens = StringLen(adp->tmpfile);
     if (strlens > 0) 
        FileRemove (adp->tmpfile);
     adp = EditAlignDataFree (adp);
  }
}

/*****************************************************************
***
*** SalsaPanelHasResized
*** FindIdsInSalsaPanel
*** SaveSalsaPanel
*** ResetSalsaTextPanel
*** SalsaTextPanel
*** PopulateSalsaPanel
***
***/
#define SIDLAND          1
#define SEQLAND          2
#define BADLAND          3
#define HOLDDNLAND       4
#define HOLDUPLAND       5
/*****************************************************************/

static void ViewForGraph (VieweR view, Uint2 entityID, EditAlignDataPtr adp, Int2 width, ValNodePtr anp_graph)
{
  SegmenT   pic;
  SeqLocPtr slpmaster;
  ValNode   vn;
  Int4      y_pos = 0;
  Int4      scale;
  Int4      max_width;
  Uint1     style = MSM_MPAIR;

  pic = CreatePicture();
  slpmaster = (SeqLocPtr) adp->master.region;
  vn.data.ptrvalue = slpmaster;
  vn.next = NULL;
  max_width = CountMaxSeqLabel(&vn);
  scale = FigureMaxScale (&vn, (Int2)width, max_width);
  if (scale<=0)
     scale = 1;
  max_width += 2;
  max_width *= scale;
  DrawMPAlignment (anp_graph, 0, adp->length-1, slpmaster, entityID, scale, &y_pos, (Uint1)style, FALSE, pic);
  AttachPicture (view, pic, -max_width, INT4_MAX, UPPER_LEFT, scale, 1, NULL);
}

static Boolean anp_has_feature (AlignNodePtr anp)
{
  AlignSegPtr asp;

  if(anp != NULL)
  {
     for(asp = anp->segs; asp != NULL; asp = asp->next) {
        if(asp->cnp != NULL)
           return TRUE;
     }
  }
  return FALSE;
}

static Uint1 getmoltype (SeqIdPtr sip)
{
  BioseqPtr bsp;
  Uint1     moltype;

  if (sip != NULL) {
     bsp = BioseqLockById (sip);
     if (bsp != NULL) {
        moltype = bsp->mol;
        BioseqUnlock (bsp);
        return moltype;
     }
  }
  return Seq_mol_na;
}

static Boolean is_segbioseq (BioseqPtr bsp)
{
  BioseqPtr   bigbsp; 
  SeqEntryPtr sep;

  sep = SeqEntryFind (bsp->id);
  if (sep != NULL) {
     bigbsp = find_big_bioseq (sep);
     if (bigbsp != NULL) 
     {
        if (bigbsp->repr == Seq_repr_seg && bigbsp->length > bsp->length) 
           return TRUE;
     }
  }
  return FALSE;
}

static void getcurrentfeatstyle (EditAlignDataPtr adp)
{
  Int2             oldstyle;
  Uint1            groupNum;
  Int2             j;

  oldstyle = GetMuskCurrentSt ();
  SetMuskCurrentSt (GetMuskStyleName (adp->styleNum));
  for(j =0; j<FEATDEF_ANY; j++) 
  {
     adp->featOrder[j] = (Uint1)GetMuskCParam(j, MSM_FORDER, MSM_NUM);
     groupNum = (Uint1)GetMuskCParam(j, MSM_FGROUP, MSM_NUM);
     adp->groupOrder[j] = (Uint1)GetMuskCParam(MSM_GROUPS, (Int2)groupNum, MSM_NUM);
  }
  SetMuskCurrentSt (GetMuskStyleName (oldstyle));
}

static ValNodePtr SetupOptions (Int2 seqnumber)
{
  ValNodePtr  params = NULL; 
  SeqParamPtr prm;
  Int2        j, k;
 
  if (seqnumber > 0) {
   for (j = 0; j < seqnumber; j++) {
     prm = (SeqParamPtr) MemNew (sizeof(SeqParam));
     prm->entityID = 0;
     prm->itemID = 0;
     prm->visible = TRUE;
     prm->complement = FALSE;
     for (k=0; k<6; k++) prm->rf[k] = FALSE;
     prm->group = 0;
     prm->sip_cons = NULL;
     ValNodeAddPointer (&(params), 0, (Pointer) prm);
   }
  }
  return params; 
}

static void CheckSeqAlignCallback (SeqEntryPtr sep, Pointer mydata, Int4 index, Int2 indent)
{
  BioseqPtr          bsp;
  BioseqSetPtr       bssp;
  SeqAnnotPtr        sap,
                     pre;

  if (sep != NULL && sep->data.ptrvalue) {
     if (IS_Bioseq(sep)) {
        bsp = (BioseqPtr) sep->data.ptrvalue;
        if (bsp!=NULL) {
           pre=NULL;
           sap=bsp->annot;
           while (sap!= NULL)
           {
              if (sap->type == 2) {
                 if (is_dim1seqalign ((SeqAlignPtr) sap->data)) {
                    if (pre==NULL) {
                       bsp->annot=sap->next;
                       sap->next=NULL;
                       SeqAnnotFree (sap);
                       sap=bsp->annot; 
                    }
                    else {
                       pre=sap->next;
                       sap->next=NULL; 
                       SeqAnnotFree (sap);
                       sap=pre->next;
                    }
                 }
                 else {
                    pre=sap;
                    sap=sap->next;
                 }
              }
              else {
                 pre=sap;
                 sap=sap->next;
              }
           }
        }
     }
     else if(IS_Bioseq_set(sep)) {
        bssp = (BioseqSetPtr)sep->data.ptrvalue;
        if (bssp!=NULL) {
           pre=NULL;
           sap=bssp->annot;
           while (sap!= NULL)
           {
              if (sap->type == 2) {
                 if (is_dim1seqalign ((SeqAlignPtr) sap->data)) {
                    if (pre==NULL) {
                       bssp->annot=sap->next;
                       sap->next=NULL;
                       SeqAnnotFree (sap);
                       sap=bssp->annot; 
                    }
                    else {
                       pre=sap->next;
                       sap->next=NULL; 
                       SeqAnnotFree (sap);
                       sap=sap->next;
                    }
                 }
                 else {
                    pre=sap;
                    sap=sap->next;
                 }
              }
              else {
                 pre=sap;
                 sap=sap->next;
              }
           }
        }
     }
  }
}

static void CheckSeqAlignInSeqEntry (BioseqPtr bsp)
{
  SeqEntryPtr      sep,
                   sep_head;
  Uint2            entityID;

  sep = SeqMgrGetSeqEntryForData (bsp);
  entityID = ObjMgrGetEntityIDForChoice (sep);
  sep_head = GetTopSeqEntryForEntityID (entityID);
  SeqEntryExplore (sep_head, NULL, CheckSeqAlignCallback);
  return;
}

static EditAlignDataPtr BioseqToEditAlignData (EditAlignDataPtr adp, BioseqPtr bsp, Uint1 showfeature)
{
  SeqEntryPtr      sep;
  ValNodePtr       vnp;
  SeqLocPtr        slp;
  SeqAlignPtr      salp;
  SeqAnnotPtr      sap;
  AlignNodePtr     anp;

  if (adp == NULL)
      return NULL;
  if (showfeature > 0) {
     adp->showfeat = TRUE;
     getcurrentfeatstyle (adp);
  }  
  adp->edit_mode = SEQ_EDIT;
  adp->input_format = OBJ_BIOSEQ;
  sep = SeqEntryNew ();
  if ( sep == NULL )  {
     MemFree (adp);
     return NULL;
  }
  sep->choice = 1;           
  sep->data.ptrvalue = (Pointer) bsp;
  vnp = SeqEntryToSeqLoc (sep, &(adp->seqnumber), bsp->mol);
  if (vnp == NULL) { 
     MemFree (adp);
     return NULL;
  }
  adp->sqloc_list = vnp;
  adp->bsqloc_list = SeqEntryToSeqLoc (sep, &(adp->seqnumber), bsp->mol);
  adp->size_labels = getsize_seqids (adp->sqloc_list, adp->printid);
  adp->params = SetupOptions  (adp->seqnumber);
  adp->length = MaxLengthSeqLoc (adp->sqloc_list);
  salp = SeqLocToFastaSeqAlign (adp->sqloc_list);
  sap = SeqAnnotForSeqAlign (salp);
  slp = (SeqLocPtr) vnp->data.ptrvalue;
  adp->master.region = (Pointer) SeqLocFromSeqAlign (salp, NULL);
  if ( adp->master.region == NULL ) {
     MemFree (adp);
     return NULL;
  }
  adp->master.regiontype = 1;
  adp->caret.regiontype = 1;
  adp->caret.region = SeqLocIntNew (0, 0, Seq_strand_plus, SeqLocId ((SeqLocPtr)adp->master.region));

  adp->mol_type = bsp->mol;
  if (ISA_aa(adp->mol_type)) {
     adp->colorRefs[COLOR_SELECT] = GetColorRGB(255, 255, 0);
  }
  else {
     adp->colorRefs[COLOR_SELECT] = GetColorRGB(0, 0, 0);
  }
  if (adp->display_panel==0 || adp->display_panel==1 || adp->display_panel==2) 
  {
     adp->anp_list = (ValNodePtr) CollAlignFromSeqAnnot (sap, (SeqLocPtr) adp->master.region, adp->featOrder, adp->groupOrder, COLLECT_MD, FALSE, FALSE, FALSE);
  } 
  else if (adp->display_panel == 3) {
     adp->anp_list = (ValNodePtr) CollAlignFromSeqAnnot (sap, (SeqLocPtr) adp->master.region, adp->featOrder, adp->groupOrder, COLLECT_MP, TRUE, FALSE, FALSE);
  }
  if (adp->display_panel==2) {
     adp->anp_graph = (ValNodePtr) CollAlignFromSeqAnnot (sap, (SeqLocPtr)adp->master.region, adp->featOrder, adp->groupOrder, COLLECT_MP, TRUE, FALSE, FALSE);
  }
  clean_annot_for_anp (&(adp->anp_list));
  if ( adp->anp_list ==NULL ) {
     MemFree (adp);
     return NULL;
  }
  if (showfeature > 0) {
     for (vnp = adp->anp_list; vnp != NULL; vnp = vnp->next) {
        anp = (AlignNodePtr)vnp->data.ptrvalue;
        if (anp!=NULL && anp_has_feature(anp)) {
           slp = CollectSeqLocFromAlignNode (anp);
           adp->seqfeat=CollectFeatureForEditor (slp, adp->seqfeat, anp->seq_entityID, anp->bsp_itemID, adp->featOrder, FALSE);
        }
     }
  }
  adp->gr.left = 0;
  adp->gr.right = adp->length;
  adp->sap_align = SeqAnnotDenseSegToBoolSeg (sap);
  sep->data.ptrvalue = NULL;
  SeqEntryFree (sep);  
  adp = SetupDataBuffer (adp);
  adp = SetupDataPanel  (adp);
  if (adp!=NULL) 
  {
     adp->seg_bioseq = is_segbioseq (bsp);
     CheckSeqAlignInSeqEntry (bsp);
  }
  return adp;  
}

/***************************************************************
*** completebioseq_inseqalign
*** SeqAlignToEditAlignData
*** SetupAlignDataSap
******************************************************************/
static SeqAlignPtr create_list_alignedsegs (SeqAlignPtr salp)
{
  SeqAlignPtr   salp2 = NULL;
  DenseDiagPtr  ddp ,
                ddphead=NULL;
  SeqIdPtr      sip = NULL,
                siphead = NULL;
  BioseqPtr     bsp;
  Int2Ptr       tab;
  Int4Ptr       startp;
  Int4          start,
                from = -1,
                lens,
                lensmax;
  Int4          k;
  Int2          dim = 0;

  if (salp!=NULL)
  {
     if (salp->segtype == 1) 
     {
        ddp = (DenseDiagPtr) salp->segs;
        if (ddp!=NULL)
           sip = ddp->id;
     }
  }
  if (sip != NULL) {
     bsp = BioseqLockById (sip);
     if (bsp!=NULL) {
        lensmax = bsp->length;
        BioseqUnlock (bsp);
        tab =  (Int2Ptr)MemNew ((size_t)((lensmax +4)*sizeof(Int2)));
        for (k=0; k<lensmax; k++)
           tab[k]=1;
        siphead = AddSeqId (&siphead, sip);
        dim = 1;
        for (salp2=salp; salp2!=NULL; salp2=salp2->next)
        {
           ddp = (DenseDiagPtr) salp2->segs;
           for (; ddp!=NULL; ddp=ddp->next)
           {
              startp = ddp->starts;
              lens=(Int4)ddp->len;
              for (k=*startp; k<*startp+lens && k<lensmax; k++)
              {
                 tab[k]++;
              }
              if (from<0)
                 from = *startp;
              siphead = AddSeqId (&siphead, ddp->id->next);
           }
           dim++;
        }
        k=0;
        while (k<lensmax)
        {
           if (tab[k] == dim)
           {
              start = k;
              k++;
              while (k<lensmax && tab[k] == dim)
                 k++;
              lens = k-start; 
              ddp = DenseDiagCreate (dim, siphead, &start, lens, NULL, NULL);
              DenseDiagAdd (&ddphead, ddp);
           } 
           k++;
        }
        MemFree (tab);
     }
  }
  salp2 = NULL;
  if (ddphead != NULL)
  {
     salp2 = SeqAlignNew ();
     salp2->type = 3;
     salp2->segtype = 4;
     salp2->dim = 2; 
     salp2->segs = (Pointer) ddphead;
  }
  return salp2;
}

static SeqAlignPtr completebioseq_inseqalign (SeqAlignPtr salp)
{
  BioseqPtr     bsp;
  ValNodePtr    vnp1 = NULL;
  DenseSegPtr   dsp;
  SeqIdPtr      sip;
  Int4Ptr       start;
  Int4Ptr       starttmp;
  Uint1Ptr      strand;
  Uint1         strd;
  Int4Ptr       lenp;
  Int4          sumlen = 0;
  Int4          from;
  Int2          k, nid;
  SeqLocPtr     slp1, slp2;
 
  if (salp!=NULL) {
     dsp = (DenseSegPtr) salp->segs;
     start = dsp->starts;
     strand= dsp->strands;
     for (sip=dsp->ids, nid=0; sip!=NULL; sip=sip->next, nid++)
     {
          bsp = BioseqLockById (sip);
          if (bsp!= NULL) {
             slp1 = SeqLocIntNew (0, bsp->length-1, Seq_strand_plus, sip);
             BioseqUnlock (bsp);

             from = -1;
             starttmp = start;
             starttmp += nid;
             sumlen = 0;
             for(k=0, lenp=dsp->lens; k<dsp->numseg; k++, lenp++, starttmp+=dsp->dim) {
                if ( *starttmp > -1 ) {
                   if (from == -1) from = *starttmp;
                   sumlen += *lenp;
                }
             }
             if (strand != NULL)
                strd = *strand;
             else strd = Seq_strand_plus;
             slp2 = SeqLocIntNew (from, from + sumlen - 1, strd, sip);
             if (slp1!=NULL && slp2!=NULL)
                if (SeqLocCompare(slp1, slp2) != SLC_A_EQ_B) {
                }
          }
     }
  }
  return salp;
}

static Boolean SetupAlignDataSap (EditAlignDataPtr adp, SeqAlignPtr salp_original, SeqIdPtr mastersip)
{
  SeqAnnotPtr  sap, 
               saptmp;
  SeqAlignPtr  newsalp, 
               salp,
               salp1;

  if (adp == NULL || salp_original == NULL)
     return FALSE;
  if ( salp_original->segtype == 1 ) 
  {
     adp->blocks = create_list_alignedsegs (salp_original);
     salp = DenseDiagToDenseSeg (salp_original, TRUE);
  }
  else
     salp = salp_original;

  if ( salp->segtype == 2 ) 
  {
/*************/
         adp->sap_original = SeqAnnotForSeqAlign (salp);
/************/
         saptmp = SeqAnnotForSeqAlign (salp);
         sap = (SeqAnnotPtr) AsnIoMemCopy (saptmp, (AsnReadFunc) SeqAnnotAsnRead, (AsnWriteFunc) SeqAnnotAsnWrite); 
         saptmp->data=NULL;
         SeqAnnotFree(saptmp);
/*       if (is_dim2seqalign(salp)) || (salp->dim==2 && adp->display_panel==1)) 
*/
         if (is_dim2seqalign (salp))
         {
            salp1 = (SeqAlignPtr) sap->data;
            sap->data=NULL;
            SeqAnnotFree(sap);
/*
            salp1 = SortSeqAlign (&salp1);
*/
            sap = multseqalign_from_pairseqalign (salp1);
            SeqAlignFree (salp1);
         }
         if (sap == NULL) 
            return FALSE;
         newsalp = (SeqAlignPtr) sap->data;
         adp->seqnumber = newsalp->dim;
         adp->sqloc_list = SeqLocListFromSeqAlign (newsalp);
         adp->bsqloc_list = WholeSeqLocListFromSeqAlign (newsalp);
         adp->size_labels = getsize_seqids (adp->sqloc_list, adp->printid);
         adp->params = SetupOptions  (adp->seqnumber);

         adp->master.region=(Pointer)SeqLocFromSeqAlign(newsalp, mastersip);
         if (adp->master.region == NULL ) {
            adp->master.region=(Pointer)SeqLocFromSeqAlign(newsalp,NULL);
         }
         adp->master.regiontype = 1;
         if (adp->display_panel==0 || adp->display_panel==1 || adp->display_panel==2) 
         {
            adp->anp_list = (ValNodePtr) CollAlignFromSeqAnnot (sap, (SeqLocPtr) adp->master.region, adp->featOrder, adp->groupOrder, COLLECT_MD, FALSE, FALSE, FALSE);  } 
         else if (adp->display_panel == 3) 
         {
            adp->anp_list = (ValNodePtr) CollAlignFromSeqAnnot (sap, (SeqLocPtr) adp->master.region, adp->featOrder, adp->groupOrder, COLLECT_MP, TRUE, FALSE, FALSE);
         } 
         if (adp->display_panel==2) {
            adp->anp_graph = (ValNodePtr) CollAlignFromSeqAnnot (sap, (SeqLocPtr)adp->master.region, adp->featOrder, adp->groupOrder, COLLECT_MP, TRUE, FALSE, FALSE);
         } 
         clean_annot_for_anp(&(adp->anp_list));
         if ( adp->anp_list == NULL ) {
            Message (MSG_ERROR,"fail in SetupAlignData [1]");
            adp->seqnumber = 0;
            return FALSE;
         }
         adp->length = GetAlignLengthFromAlignNode ((AlignNodePtr) adp->anp_list->data.ptrvalue);
         adp->gr.left = 0;
         adp->gr.right = adp->length; 
         salp1 = SeqAlignDenseSegToBoolSeg (newsalp);
         adp->sap_align = SeqAnnotForSeqAlign (salp1);
         adp->mol_type = getmoltype (SeqLocId((SeqLocPtr)adp->master.region));
         if (ISA_aa(adp->mol_type)) 
         {
            adp->colorRefs[COLOR_SELECT] = GetColorRGB(255, 255, 0);
         }
         else {
            adp->colorRefs[COLOR_SELECT] = GetColorRGB(0, 0, 0);
         }
         SeqAnnotFree (sap);
  }
  else if (salp->segtype == 3) {
     ; 
  }
  else {
     ;
  }
  return TRUE;
}

static EditAlignDataPtr SeqAlignToEditAlignData (EditAlignDataPtr adp, SeqAlignPtr salp, Int4 start, SeqIdPtr mastersip, Uint1 showfeature)
{
  AlignNodePtr     anp;
  SeqLocPtr        slp;
  SeqIdPtr         sip;
  ValNodePtr       vnp;
  
  if (adp == NULL || salp == NULL) 
     return NULL;
  if (SeqAlignMolType (salp) == 0)
     return NULL;
  salp = check_salp_forlength (salp);
  salp = check_salp_forstrand (salp);
  if (salp!=NULL) 
  {
     if (showfeature > 0) 
     {
        adp->showfeat = TRUE;
        getcurrentfeatstyle (adp);
     }  
     adp->edit_mode = ALIGN_EDIT;
     adp->input_format = OBJ_SEQALIGN;
     if (SetupAlignDataSap (adp, salp, mastersip)) 
     {
        if (showfeature == SEQ_FEAT1 && mastersip != NULL) {
           for (vnp = adp->anp_list; vnp != NULL; vnp = vnp->next) {
              anp = (AlignNodePtr)vnp->data.ptrvalue; 
              if (anp!=NULL) {
                 for (sip=mastersip; sip!=NULL; sip=sip->next) {
                    if (SeqIdForSameBioseq(anp->sip, sip)) {
                       if(anp_has_feature(anp)) {
                          slp = CollectSeqLocFromAlignNode (anp);
                          adp->seqfeat=CollectFeatureForEditor (slp, adp->seqfeat, anp->seq_entityID, anp->bsp_itemID, adp->featOrder, FALSE);
                          break;
                       }
                    }
                 }
              } 
           }  
        }
        else { 
           for (vnp = adp->anp_list; vnp != NULL; vnp = vnp->next) {
              anp = (AlignNodePtr)vnp->data.ptrvalue; 
              if (anp!=NULL) {
                 if (anp_has_feature(anp)) {
                    slp = CollectSeqLocFromAlignNode (anp);
                    adp->seqfeat=CollectFeatureForEditor (slp, adp->seqfeat, anp->seq_entityID, anp->bsp_itemID, adp->featOrder, FALSE);
                 }
              } 
           }  
        }
        adp->caret.regiontype = 1;
        adp->caret.region = SeqLocIntNew (start, start, Seq_strand_plus, SeqLocId((SeqLocPtr)adp->master.region));
        adp = SetupDataBuffer (adp);
        adp = SetupDataPanel  (adp);
        if (adp!=NULL) {
           adp->draw_scale = TRUE;
           adp->draw_bars = TRUE;
           adp->marginwithindex = FALSE;
           adp->marginwithIds = TRUE;
           adp->marginwithgroup = FALSE;
           return adp;
        }
     }
  }
  adp = EditAlignDataFree (adp);
  return NULL;
}

extern void SalsaPanelHasResized (PaneL pnl)
{
  EditAlignDataPtr adp; 
  RecT             rp;
 
  GetPanelExtra (pnl, &adp); 
  if (adp != NULL) { 
     get_client_rect (pnl, &rp);
     do_resize_panel (pnl, adp, (Int2)(rp.right - rp.left), (Int2)(rp.bottom - rp.top), TRUE);
  } 
  SetPanelExtra (pnl, &adp); 
}

extern Boolean FindIdsInSalsaPanel (PaneL pnl, PoinT pt, Uint2 *entityID, Uint2 *itemID, Uint2 *itemtype)
{
  EditAlignDataPtr adp;
  RecT             rp;
  Int4             pos;
  Int2             line;
  Uint2            iID, eID, itype, isubtype;
  Uint1            what;
  SeqAlignPtr      salp;

  *entityID = 0; 
  *itemID = 0; 
  *itemtype = 0; 
  GetPanelExtra (pnl, &adp);
  if (adp != NULL) {
     get_client_rect (pnl, &rp);
     what = locate_point (pt, rp, &iID, &eID, &itype, &isubtype, &pos, &line, adp);
     if (what == SIDLAND || (what == SEQLAND && adp->seqnumber == 1)) {
        if (eID > 0) {
           *entityID = eID;
           *itemID = iID;
           *itemtype = itype;
           return TRUE;
        }
     }
     if (what == SEQLAND) {
        salp = SeqAlignBoolSegToDenseSeg((SeqAlignPtr)adp->sap_align->data);
        *entityID = ObjMgrRegister (OBJ_SEQALIGN, (Pointer) salp);
        *itemID = 1;
        *itemtype = OBJ_SEQALIGN;
        return TRUE;
     }
  }
  return FALSE;  
}

extern void SaveSalsaPanel (PaneL pnl)
{
  Char             namep [PATH_MAX];
  EditAlignDataPtr adp;
  Int4             start, stop;
  FILE             *fout;

  if (!GetOutputFileName (namep, PATH_MAX, NULL))
     return;
  GetPanelExtra (pnl, &adp);
  if (adp != NULL) {
     fout = FileOpen (namep, "w");
     if (fout != NULL) {
        WatchCursor ();
        start = 0;
        stop = adp->length -1;
        ShowAlignmentText (fout, adp, NULL, adp->marginleft, start, stop, FALSE);
        ArrowCursor ();
        FileClose (fout);
     }
  }
}

static void ResetSalsaTextPanel (PaneL pnl) 
{ 
  EditAlignDataPtr adp; 
 
  GetPanelExtra (pnl, &adp); 
  if (adp != NULL) { 
    adp = EditAlignDataFree (adp); 
  } 
  SetPanelExtra (pnl, &adp); 
} 
 
extern PaneL SalsaTextPanel (GrouP g, Int2 wid, Int2 hgt)
{
  PaneL pnl;
  pnl = AutonomousPanel (g, wid, hgt, on_draw, VscrlProc, NULL,
                         sizeof (EditAlignDataPtr), ResetSalsaTextPanel, NULL);
  return pnl;
}

static Pointer sap_empty (SeqAnnotPtr sap, Uint1 type, Pointer PNTR ptr)
{
  SeqAlignPtr      salp = NULL;

  if (sap != NULL) {
     for (; sap!= NULL; sap=sap->next) {
        if (sap->type == type) {
           salp = (SeqAlignPtr) sap->data;
           *ptr = (Pointer) sap;
           break;
        }
     }   
  }
  return salp;
}

typedef struct ccid {
  Uint1      choice;
  SeqIdPtr   sip;
  Pointer    sap;
} CcId, PNTR CcIdPtr;


static void FindSeqAlignCallback (SeqEntryPtr sep, Pointer mydata,
                                          Int4 index, Int2 indent)
{
  BioseqPtr          bsp;
  BioseqSetPtr       bssp;
  SeqAlignPtr        salp, 
                     salptmp;
  DenseSegPtr        dsp;
  CcIdPtr            cip;
  Boolean            found;
  Pointer            this_sap;

  cip = (CcIdPtr)mydata;
  if (sep != NULL && sep->data.ptrvalue && mydata != NULL) {
     if (IS_Bioseq(sep)) {
        bsp = (BioseqPtr) sep->data.ptrvalue;
        if (bsp!=NULL) {
           salp=sap_empty(bsp->annot, 2, &this_sap);
           if (salp!=NULL) {
              found=FALSE;
              salptmp=salp;
              if (cip->sip!=NULL) {
                 while (!found && salptmp!=NULL) 
                 {
                    dsp = salptmp->segs;
                    found = (Boolean)(position_inIdlist(cip->sip, dsp->ids)>0);
                    salptmp=salptmp->next;
                 }
              }
              if (found || cip->sip==NULL) {
                 if (cip->sap==NULL) {
                    if (cip->choice==OBJ_SEQALIGN)
                       cip->sap = (Pointer)salp;
                    else if (cip->choice==OBJ_SEQANNOT)
                       cip->sap = (Pointer) this_sap;
                 }
              }
           }
        }
     }
     else if(IS_Bioseq_set(sep)) {
        bssp = (BioseqSetPtr)sep->data.ptrvalue;
        if (bssp!=NULL) {
           salp=sap_empty(bssp->annot, 2, &this_sap);
           if (salp!=NULL) {
              found=FALSE;
              salptmp=salp;
              if (cip->sip!=NULL) {
                 while (!found && salptmp!=NULL) 
                 {
                    dsp = salptmp->segs;
                    found = (Boolean)(position_inIdlist(cip->sip, dsp->ids)>0);
                    salptmp=salptmp->next;
                 }
              }
              if (found || cip->sip==NULL) {
                 if (cip->sap==NULL) {
                    if (cip->choice==OBJ_SEQALIGN)
                       cip->sap = (Pointer)salp;
                    else if (cip->choice==OBJ_SEQANNOT)
                       cip->sap = (Pointer) this_sap;
                 }
              }
           }
        }
     }
  }
}

extern Pointer FindSeqAlignInSeqEntry (SeqEntryPtr sep, Uint1 choice)
{
  SeqEntryPtr      sep_head;
  BioseqPtr        bsp;
  Uint2            entityID;
  CcId             ci;

  ci.sap = NULL;
  ci.sip = NULL;
  if (IS_Bioseq(sep)) {
     bsp = (BioseqPtr) sep->data.ptrvalue;
     if (bsp!=NULL)
        ci.sip = SeqIdDup (bsp->id);
  }
  ci.choice = choice;
  entityID = ObjMgrGetEntityIDForChoice (sep);
  sep_head = GetTopSeqEntryForEntityID (entityID);
  SeqEntryExplore (sep_head, (Pointer)&ci, FindSeqAlignCallback);
  if (ci.sip != NULL)
     SeqIdFree (ci.sip);
  return ci.sap;
}

extern void PopulateSalsaPanel (PaneL pnl, SeqEntryPtr sep, Boolean all_seq, Uint1 sequence_shown, Uint1 show_feat, Uint1 numbering, FonT font)
{
  EditAlignDataPtr adp = NULL;
  BioseqPtr        bsp = NULL;
  SeqAlignPtr      salp = NULL;
  RecT             rp;
  Int2             height, width;

  if (sep == NULL)
     return;
  get_client_rect (pnl, &rp);
  width = (rp.right -rp.left);
  height = (rp.bottom -rp.top);

  if (all_seq || sequence_shown != SEQ_SHOW1)
     salp = (SeqAlignPtr) FindSeqAlignInSeqEntry (sep, OBJ_SEQALIGN);
 
  if (salp==NULL && IS_Bioseq(sep)) {
     bsp = (BioseqPtr) sep->data.ptrvalue;
     if (bsp != NULL) {
        adp = EditAlignDataInit(NULL,width,height,MARGINLEFT15,10,font, TRUE, 1);
        adp = BioseqToEditAlignData (adp, bsp, show_feat);
     }
  }
  if (salp!=NULL) {
     get_client_rect (pnl, &rp);
     adp =EditAlignDataInit(NULL,width,height, MARGINLEFT15, 10, font, TRUE, 1);
     if (IS_Bioseq(sep)) {
        bsp = (BioseqPtr) sep->data.ptrvalue;
        adp = SeqAlignToEditAlignData (adp, salp, 0, bsp->id, show_feat);
     }
     else {
        adp = SeqAlignToEditAlignData (adp, salp, 0, NULL, show_feat);
     }
  }
  if (adp != NULL)
  {
     adp->font = (Handle)(font);
     adp->draw_emptyline = FALSE;
     SetPanelExtra (pnl, &adp);
     adp->marginwithindex = FALSE;
     adp->marginwithfeatid = TRUE;
     adp->all_sequences = all_seq;
     if (!all_seq) {
        adp->charmode = TRUE;
     }
     if (numbering == SEQ_NUM1) {
        adp->draw_scale = FALSE;
        adp->draw_bars = FALSE;
        adp->marginwithpos = TRUE;
        adp->marginwithgroup = FALSE;
     }
     else if (numbering == SEQ_NUM2) {
        adp->draw_scale = TRUE;
        adp->draw_bars = TRUE;
        adp->marginwithpos = TRUE;
        adp->marginwithgroup = FALSE;
     }
     else {
        adp->draw_scale = FALSE;
        adp->draw_bars = FALSE;
        adp->marginwithpos = FALSE;
        adp->marginwithgroup = FALSE;
     }
     do_resize_panel(pnl,adp, width, height, TRUE);
     checkselectsequinfeature_for_editor (adp->seqfeat);
  }
}


extern EditAlignDataPtr EditAlignDataRepopulateFromSeqAlign (PaneL pnl, EditAlignDataPtr adp, SeqAlignPtr salp)
{
  EditAlignDataPtr adpnew;
  RecT rp;
  Int4 hoffset;
  Int2 voffset;

  get_client_rect (pnl, &rp);
  voffset = adp->voffset;
  hoffset = adp->hoffset;
  adpnew = EditAlignDataReset (adp, (rp.right-rp.left), (rp.bottom-rp.top), salp);
  if (adp == NULL) {
     return adp;
  }
  adp = adpnew;
  adp->voffset = voffset;
  adp->hoffset = hoffset;
  do_resize_window  (pnl, adp, TRUE);
  reset_features (pnl, adp);
  SetCorrectVBarMax (pnl, adp->nlines, adp->voffset);
  adp->dirty = TRUE;
  return adp;
}

extern void ResetSalsaAlignPanel (PaneL pnl, SeqAlignPtr salp, SeqIdPtr mastersip, Int2 marginleft, Boolean showfeat, Boolean all_seq)
{
  EditAlignDataPtr adp = NULL;
  RecT             rp;
 
  get_client_rect (pnl, &rp);
  adp = EditAlignDataInit (NULL, (rp.right -rp.left), (rp.bottom -rp.top), marginleft, 10, NULL, TRUE, 1);
  adp = SeqAlignToEditAlignData (adp, salp, 0, mastersip, (Uint1)(showfeat));
  if (adp != NULL)
  {
     adp->draw_emptyline = FALSE;
     SetPanelExtra (pnl, &adp);
     adp->marginwithfeatid = TRUE;
     adp->all_sequences = all_seq;
     if (!all_seq)
        adp->charmode = TRUE;
     do_resize_panel (pnl, adp, (Int2)(rp.right - rp.left), (Int2)(rp.bottom - rp.top), TRUE);
     checkselectsequinfeature_for_editor (adp->seqfeat);
  }
}

static SeqEditViewFormPtr SeqEditViewFormNew (void)
{
  SeqEditViewFormPtr wdp;

  wdp = (SeqEditViewFormPtr) MemNew (sizeof (SeqEditViewForm));
  wdp->pnl = NULL;
  wdp->data = NULL;
  wdp->graph = NULL;

  wdp->pos = NULL;
  wdp->gototxt = NULL;
  wdp->lookattxt = NULL;
  wdp->gotobt = NULL;
  wdp->lookatbt = NULL;

  wdp->rfitem[0] = NULL;  wdp->rfitem[1] = NULL; wdp->rfitem[2] = NULL;
  wdp->rfitem[3] = NULL;  wdp->rfitem[4] = NULL; wdp->rfitem[5] = NULL;
  wdp->rfitem[6] = NULL;  wdp->rfitem[7] = NULL; wdp->rfitem[8] = NULL;
  wdp->rfitem[9] = NULL; 
  wdp->btngp = NULL;
  wdp->showfeatbt = NULL;
  wdp->showfeatitem = NULL;
  wdp->translatebt = NULL;
  wdp->translateitem = NULL;
  wdp->codonstitem = NULL;
  wdp->savefeatbt = NULL;
  wdp->savefeatitem = NULL;
  wdp->refreshbt = NULL;
  wdp->closebt = NULL;
  wdp->svclosebt = NULL;
  wdp->cutitem = NULL;
  wdp->delitem = NULL; 
  wdp->copyitem = NULL; 
  wdp->pasteitem = NULL;

  wdp->selmaster = NULL;
  wdp->showdiff = NULL;
  wdp->showgraph = NULL;

  wdp->extended_align_menu = FALSE;
  wdp->extended_dist_menu = FALSE;
  wdp->extended_tree_menu = FALSE;

  return wdp;
}   
/********************************************
***    Create new Window 
***
***
***********************************************/
static ForM CC_CreateBioseqViewForm (Int2 left, Int2 top, CharPtr windowname, BioseqPtr bsp, Pointer dummy)
{
  SeqEditViewProcsPtr  svpp;
  WindoW             w;
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  PaneL              pnl;
  GrouP              g;
  RecT               rw,    /* window rect */
                     rp,    /* panel rect  */
                     rb;    /* button rect */
  PopuP              pop;
  Char               str [16];
  SeqEntryPtr        sep = NULL;
  SelStructPtr       ssp = NULL;
  Int2               window_width = 600;
  Int2               wid, 
                     window_hgt = 300,
                     hgt,
                     charwidth, 
                     lineheight;
  Uint1              display_panel = 0;
  Uint1              showfeat = 0;
  FonT               font;
  Boolean            is_prot = FALSE;
  
  if (bsp==NULL)
     return NULL;
  is_prot = (Boolean)(ISA_aa(bsp->mol));
  wdp = SeqEditViewFormNew ();
  if (wdp==NULL)
     return NULL;
  
#ifdef WIN_MAC
  font = ParseFont ("Monaco, 9");
#endif
#ifdef WIN_MSWIN
  font = ParseFont ("Courier, 9");
#endif
#ifdef WIN_MOTIF
  font = ParseFont ("fixed, 12");
#endif
  SelectFont (font);
  charwidth  = CharWidth ('0');
  lineheight = LineHeight ();
  left = -66;
  top = -10;

  w = DocumentWindow (left, top, (Int2)(-10), (Int2)(-10),  windowname, NULL, ResizeAlignDataWindow);
  SetObjectExtra (w, (Pointer) wdp, CleanupSequenceEditorForm);
  wdp->form = (ForM) w;
  wdp->formmessage = SalsaFormMessage;
  svpp = (SeqEditViewProcsPtr) GetAppProperty ("SeqEditDisplayForm");
  if (svpp != NULL) {
      SetActivate (w, svpp->activateForm);
      wdp->appmessage = svpp->handleMessages;
      wdp->extended_align_menu = svpp->extended_align_menu;
      wdp->extended_dist_menu = svpp->extended_dist_menu;
      wdp->extended_tree_menu = svpp->extended_tree_menu;
      showfeat = svpp->showfeat;
      if (svpp->minPixelWidth > 0) {
         window_width = svpp->minPixelWidth;
         if (window_width > 600) window_width = 600;
      }
      if (svpp->minPixelHeight > 0) {
         window_hgt = svpp->minPixelHeight;
         if (window_hgt > 300) window_hgt = 300;
      }
  }
  wid = (Int2)(window_width/charwidth);
  hgt = (Int2)(window_hgt / lineheight);

  if (dummy != NULL) {
     ssp = (SelStructPtr) dummy;    
     if (ssp->regiontype>0)
        display_panel = ssp->regiontype;
  }
  adp = EditAlignDataInit (NULL, wid, hgt, MARGINLEFT25, 10, font, FALSE, display_panel);
  adp = BioseqToEditAlignData (adp, bsp, showfeat);
  if (dummy!=NULL) {
     ssp = (SelStructPtr) dummy;    
     adp->input_entityID = ssp->entityID;;
     adp->input_itemID = ssp->itemID;
     adp->input_itemtype = ssp->itemtype;
  }
  else {
     adp->input_entityID = 0;
     adp->input_itemID = 0;
     adp->input_itemtype = 0;
  }
  adp->wdp = (Pointer) wdp;
  if (svpp == NULL)
     SetupMenus (w, is_prot, FALSE, FALSE, FALSE);
  else 
     SetupMenus (w, is_prot, svpp->extended_dist_menu, svpp->extended_align_menu, svpp->extended_tree_menu);

  if (!is_prot) 
  {
     g = HiddenGroup (w, 5, 0, NULL);                  
     sprintf (str, "%ld", (long) adp->caret_orig);
     wdp->gotobt = PushButton (g, "Go to:", GoToButton);
     wdp->gototxt = DialogText (g, str, (Int2)6, NULL);
     wdp->lookatbt = PushButton (g, "Look at:", LookAtButton);
     wdp->lookattxt = DialogText (g, str, (Int2)6, NULL);
     pop = PopupList (g, TRUE, SelectFeatEditMode);
     PopupItem (pop, "Merge feature mode"); 
     PopupItem (pop, "Split feature mode"); 
     SetValue (pop, 1);
  }
  else {
     g = HiddenGroup (w, 4, 0, NULL);                  
     sprintf (str, "%ld", (long) adp->caret_orig);
     wdp->gotobt = PushButton (g, "Go to:", GoToButton);
     wdp->gototxt = DialogText (g, str, (Int2)6, NULL);
     wdp->lookatbt = PushButton (g, "Look at:", LookAtButton);
     wdp->lookattxt = DialogText (g, str, (Int2)6, NULL);
  }  
  g = HiddenGroup (w, 1, 0, NULL);                  
  wdp->pos = StaticPrompt (g, "", window_width, dialogTextHeight, programFont, 'l');
  
  g = HiddenGroup (w, 1, 0, NULL);                  
  pnl = AutonomousPanel (g, window_width, window_hgt, on_draw, VscrlProc, NULL,
                         sizeof (EditAlignDataPtr), NULL, NULL);
  SetPanelExtra (pnl, &adp);
  SetObjectExtra (pnl, (Pointer) adp, CleanupAlignDataPanel);
  wdp->pnl = pnl;

  if (!is_prot) 
  {
     wdp->btngp = HiddenGroup (w, 6, 0, NULL);
     wdp->showfeatbt= PushButton (wdp->btngp, "Show feat.", ShowFeatureButtonProc);
     wdp->translatebt = PushButton (wdp->btngp, "Translate", TranslateButton);
     wdp->savefeatbt = PushButton (wdp->btngp, "Save CDS", SaveFeatureButton);
     wdp->refreshbt = PushButton (wdp->btngp, "Refresh", RefreshAlignDataButton);
     wdp->svclosebt = PushButton (wdp->btngp, "Accept", AcceptCloseWindowButton);
     wdp->closebt = PushButton (wdp->btngp, "Cancel", CloseWindowButton);
  }

  RealizeWindow (w);

  if (!is_prot) {
     Enable (wdp->showfeatbt);
     Enable (wdp->translatebt);
     Disable (wdp->savefeatbt);
     Enable (wdp->svclosebt);
     Enable (wdp->closebt);
  }

  ObjectRect (w, &rw);
  GetPosition (pnl, &rp);
  adp->xoff = rp.left;
  adp->yoff = rp.top;
  adp->x = (rw.right -rw.left+1) - (rp.right -rp.left+1) - rp.left;
  adp->y = (rw.bottom-rw.top +1) - (rp.bottom-rp.top +1) - rp.top;

  if (!is_prot) {
     GetPosition (wdp->closebt, &rb);
     adp->ybutt = (rw.bottom-rw.top +1) - (rb.bottom-rb.top +1) - rb.top;
  }
  
  SetPanelClick(pnl, on_click, on_drag, on_hold, on_release);
  SetSlateChar ((SlatE) pnl, on_key);
  SetWindowTimer (w, on_time);
  do_resize_window (pnl, adp, TRUE);

  if (adp->showfeat) {
     adp->showfeat = FALSE;
     if (wdp->showfeatbt!=NULL)
        ShowFeatureButtonProc (wdp->showfeatbt);
     if (wdp->showfeatitem!=NULL)
        ShowFeatureItemProc (wdp->showfeatitem);
  }
  if (!adp->display_panel)
     to_update_prompt (pnl, &(adp->caret), (SeqAlignPtr) adp->sap_align->data, adp->sqloc_list, FALSE);
  
  if (!is_prot) {
     if (adp->master.entityID > 0) {
        sep = GetTopSeqEntryForEntityID (adp->master.entityID);
        TmpNam (adp->tmpfile);
        seqentry_write (sep, adp->tmpfile);
     }
  }
  return (ForM) w;
}

/********************************************
***
***    Create new Window from SeqAlign 
***
***********************************************/
static ForM CreateSeqAlignEditForm (Int2 left, Int2 top, CharPtr windowname, SeqAlignPtr salp, Pointer dummy)
{
  SeqEditViewProcsPtr  svpp;
  WindoW             w;
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  PaneL              pnl;
  GrouP              g;
  BaR                vsb;
  RecT               rw,    /* window rect */
                     rp,    /* panel rect  */
                     rb;    /* button rect */
  PopuP              pop, pop2;
  SeqEntryPtr        sep = NULL;
  SelStructPtr       ssp = NULL; 
  Char               str [16];
  Int4               start = 0;
  Int2               window_width = 600,
                     wid,
                     window_hgt = 300,
                     hgt,
                     charwidth, lineheight;
  Uint1              display_panel = 0;
  Uint1              showfeat = 0;
  Uint1              moltype = 0;
  FonT               font;
  Boolean            is_prot = FALSE;
  
  if (salp==NULL)
     return NULL;
  moltype = SeqAlignMolType(salp);
  if (moltype == 0)
     return NULL;
  is_prot = (Boolean)(ISA_aa(moltype));
  wdp = SeqEditViewFormNew ();
  if (wdp==NULL)
     return NULL;

#ifdef WIN_MAC
  font = ParseFont ("Monaco, 9");
#endif
#ifdef WIN_MSWIN
  font = ParseFont ("Courier, 9");
#endif
#ifdef WIN_MOTIF
  font = ParseFont ("fixed, 12");
#endif
  SelectFont (font);
  charwidth  = CharWidth ('0');
  lineheight = LineHeight ();
  left = -66;
  top = -10;

  w = DocumentWindow (left, top, (Int2)(-10), (Int2)(-10),  windowname, NULL, ResizeAlignDataWindow);
  SetObjectExtra (w, (Pointer) wdp, CleanupSequenceEditorForm);
  wdp->form = (ForM) w;
  wdp->formmessage = SalsaFormMessage;
  svpp = (SeqEditViewProcsPtr) GetAppProperty ("SeqEditDisplayForm");
  if (svpp != NULL) {
      SetActivate (w, svpp->activateForm);
      wdp->appmessage = svpp->handleMessages;
      wdp->extended_align_menu = svpp->extended_align_menu;
      wdp->extended_dist_menu = svpp->extended_dist_menu;
      wdp->extended_tree_menu = svpp->extended_tree_menu;
      showfeat = svpp->showfeat;

/*** BECAUSE it crashes if tries to load the features  **/ 
/*** before getting the entityIDs of the sequences     **/
      showfeat = FALSE;
      if (svpp->minPixelWidth > 0) {
         window_width = svpp->minPixelWidth;
         if (window_width > 600) 
            window_width = 600;
      }
      if (svpp->minPixelHeight > 0) {
         window_hgt = svpp->minPixelHeight;
         if (window_hgt > 300) 
            window_hgt = 300;
      }
  }
  wid = (Int2)(window_width/charwidth);
  hgt = (Int2)(window_hgt / lineheight);

  if (dummy != NULL) {
     ssp = (SelStructPtr) dummy;    
     if (ssp->regiontype>0)
        display_panel = ssp->regiontype;
  }
  adp = EditAlignDataInit (NULL, wid, hgt, MARGINLEFT25, 10, font, FALSE, display_panel);
  if (dummy != NULL) {
     ssp = (SelStructPtr) dummy;    
     adp->input_entityID = ssp->entityID;
     adp->input_itemID = ssp->itemID;
     adp->input_itemtype = ssp->itemtype;
     if (ssp->region!=NULL)
        start = (Int4)SeqLocStart((SeqLocPtr)ssp->region);
  }
  else {
     adp->input_entityID = 0;
     adp->input_itemID = 0;
     adp->input_itemtype = 0;
  }
  adp = SeqAlignToEditAlignData (adp, salp, start, NULL, showfeat); 
  if (adp == NULL)
     return NULL;
  if (svpp!=NULL) {
     setup_aacolor (adp, svpp);
  }
  adp->wdp = (Pointer) wdp;
  if (svpp == NULL)
     SetupMenus2 (w, is_prot, (Boolean)(adp->seqnumber==2), FALSE, FALSE);
  else 
     SetupMenus2 (w, is_prot, (Boolean)(adp->seqnumber==2), svpp->extended_dist_menu, svpp->extended_tree_menu);

  if (!is_prot) 
  {
     g = HiddenGroup (w, 5, 0, NULL);                  
     sprintf (str, "%ld", (long) adp->caret_orig);
     wdp->gotobt = PushButton (g, "Go to:", GoToButton);
     wdp->gototxt = DialogText (g, str, (Int2)6, NULL);
     wdp->lookatbt = PushButton (g, "Look at:", LookAtButton);
     wdp->lookattxt = DialogText (g, str, (Int2)6, NULL);
     pop = PopupList (g, TRUE, SelectFeatEditMode);
     PopupItem (pop, "Merge feature mode"); 
     PopupItem (pop, "Split feature mode"); 
     SetValue (pop, 1);
  }
  else {
     g = HiddenGroup (w, 4, 0, NULL);                  
     sprintf (str, "%ld", (long) adp->caret_orig);
     wdp->gotobt = PushButton (g, "Go to:", GoToButton);
     wdp->gototxt = DialogText (g, str, (Int2)6, NULL);
     wdp->lookatbt = PushButton (g, "Look at:", LookAtButton);
     wdp->lookattxt = DialogText (g, str, (Int2)6, NULL);
  }  

  g = HiddenGroup (w, 1, 0, NULL);
  wdp->pos = StaticPrompt (g, "", window_width, dialogTextHeight, programFont, 'l');

  g = HiddenGroup (w, 1, 0, NULL);
  pnl = AutonomousPanel (g, window_width, window_hgt, on_draw, VscrlProc, NULL,
                         sizeof (EditAlignDataPtr), NULL, NULL);
  SetPanelExtra (pnl, &adp);
  SetObjectExtra (pnl, (Pointer) adp, CleanupAlignDataPanel);
  wdp->pnl = pnl;

  if (!is_prot) {
     wdp->btngp = HiddenGroup (w, 5, 0, NULL);
     wdp->showfeatbt= PushButton (wdp->btngp, "Show feat.", ShowFeatureButtonProc);
     wdp->translatebt = PushButton (wdp->btngp, "Translate", TranslateButton);
     wdp->savefeatbt = PushButton (wdp->btngp, "Save CDS", SaveFeatureButton);
     wdp->refreshbt = PushButton (wdp->btngp, "Refresh", RefreshAlignDataButton);
     wdp->closebt = PushButton (wdp->btngp, "Dismiss", CloseWindowButton);
  }
  
  RealizeWindow (w);
  
  if (!is_prot) {
     Enable (wdp->showfeatbt);
     Enable (wdp->translatebt);
     Disable (wdp->savefeatbt);
     Enable (wdp->closebt);
  }
  ObjectRect (w, &rw);
  GetPosition (pnl, &rp);
  adp->xoff = rp.left;
  adp->yoff = rp.top;
  adp->x = (rw.right -rw.left+1) - (rp.right -rp.left+1) - rp.left;
  adp->y = (rw.bottom-rw.top +1) - (rp.bottom-rp.top +1) - rp.top;

  if (!is_prot) {
     GetPosition (wdp->closebt, &rb);
     adp->ybutt = (rw.bottom-rw.top +1) - (rb.bottom-rb.top +1) - rb.top;
  }

  if (dummy != NULL) {
     adp->voffset = hoffset2voffset(adp, adp->anp_list, adp->visibleWidth, 0, adp->length-1, start);
  }
  SetPanelClick(pnl, on_click, on_drag, on_hold, on_release);
  SetSlateChar ((SlatE) pnl, on_key);
  SetWindowTimer (w, on_time);
  do_resize_window (pnl, adp, TRUE);
  if (dummy != NULL) {
     vsb = GetSlateVScrollBar ((SlatE) pnl);
     CorrectBarValue (vsb, (Int2) adp->voffset);
  }

  if (adp->showfeat) {
     adp->showfeat = FALSE;
     if (wdp->showfeatbt!=NULL)
        ShowFeatureButtonProc (wdp->showfeatbt);
     if (wdp->showfeatitem!=NULL)
        ShowFeatureItemProc (wdp->showfeatitem);
  }
  if (!adp->display_panel)
     to_update_prompt (pnl, &(adp->caret), (SeqAlignPtr) adp->sap_align->data, adp->sqloc_list, FALSE);

  if (!is_prot) {
     if (adp->master.entityID > 0) {
        sep = GetTopSeqEntryForEntityID (adp->master.entityID);
        TmpNam (adp->tmpfile);
        seqentry_write (sep, adp->tmpfile);
     }
  }
  return (ForM) w;
}


/********************************************
***
***    Create new Window from SeqAlign 
***
***********************************************/

static void DoReplaceBtn (ButtoN b)
{
  WindoW             w;
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  SeqAlignPtr        salp;
  SeqAlignPtr        salp2;
  CompSegPtr         csp;

  w = (WindoW)ParentWindow (b);
  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  adp = (EditAlignDataPtr) wdp->data; 
  if (adp != NULL) {
     salp = (SeqAlignPtr) adp->sap_align->data;
     if (salp->segtype == 4) {
        csp = (CompSegPtr) salp->segs;
        if (csp->ids != NULL && csp->ids->next!=NULL) {
           salp2 = SeqAlignBoolSegToDenseSeg (salp);
           ReplaceBioseq (csp->ids, csp->ids->next, salp2, IGNORE_GAP_CHOICE, adp->stoptransl);
           SeqAlignFree (salp2);
           ObjMgrSendMsg (OM_MSG_UPDATE, adp->master.entityID, adp->master.itemID, adp->master.itemtype);
           /****************************DELETE ADP ****/
           EditAlignDataFree (adp);
           wdp->data = NULL;
           Hide (w); 
           Update ();
           Remove (w);
        }
     }
  }
  return;
}

static void DoMerge3Btn (ButtoN b)
{
  WindoW             w;
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  SeqAlignPtr        salp;
  CompSegPtr       csp;

  w = (WindoW)ParentWindow (b);
  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  adp = (EditAlignDataPtr) wdp->data; 
  if (adp != NULL) {
     salp = (SeqAlignPtr) adp->sap_align->data;
     if (salp->segtype == 4) {
        csp = (CompSegPtr) salp->segs;
        if (csp->ids != NULL && csp->ids->next!=NULL) {
           Merge3Func (csp->ids, csp->ids->next, salp, adp->spliteditmode);
           ObjMgrSendMsg (OM_MSG_UPDATE, adp->master.entityID, adp->master.itemID, adp->master.itemtype);
           /****************************DELETE ADP ****/
           EditAlignDataFree (adp);
           wdp->data = NULL;
           Hide (w); 
           Update ();
           Remove (w);
        }
     }
  }
  return;
}

static void DoMerge5Btn (ButtoN b)
{
  WindoW             w;
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  SeqAlignPtr        salp;
  CompSegPtr       csp;

  w = (WindoW)ParentWindow (b);
  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  adp = (EditAlignDataPtr) wdp->data; 
  if (adp != NULL) {
     salp = (SeqAlignPtr) adp->sap_align->data;
     if (salp->segtype == 4) {
        csp = (CompSegPtr) salp->segs;
        if (csp->ids != NULL && csp->ids->next!=NULL) {
           Merge5Func (csp->ids, csp->ids->next, salp, adp->spliteditmode);
           ObjMgrSendMsg (OM_MSG_UPDATE, adp->master.entityID, adp->master.itemID, adp->master.itemtype);
           /****************************DELETE ADP ****/
           EditAlignDataFree (adp);
           wdp->data = NULL;
           Hide (w); 
           Update ();
           Remove (w);
        }
     }
  }
  return;
}

static void DoCopyFeatBtn (ButtoN b)
{
  WindoW             w;
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  SeqAlignPtr        salp;
  SeqAlignPtr        salp2;
  CompSegPtr         csp;

  w = (WindoW)ParentWindow (b);
  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  adp = (EditAlignDataPtr) wdp->data; 
  if (adp != NULL) {
     salp = (SeqAlignPtr) adp->sap_align->data;
     if (salp->segtype == 4) {
        csp = (CompSegPtr) salp->segs;
        if (csp->ids != NULL && csp->ids->next!=NULL) {
           salp2 = SeqAlignBoolSegToDenseSeg (salp);
           CopyFeatFunc (csp->ids, csp->ids->next, salp2, TAKE_GAP_CHOICE, adp->stoptransl);
           SeqAlignFree (salp2);
           ObjMgrSendMsg (OM_MSG_UPDATE, adp->master.entityID, adp->master.itemID, adp->master.itemtype);
           /****************************DELETE ADP ****/
           EditAlignDataFree (adp);
           wdp->data = NULL;
           Hide (w); 
           Update ();
           Remove (w);
        }
     }
  }
  return;
}

static void LaunchAlignEdit (ButtoN b)
{
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  SeqAlignPtr        salp;

  wdp = (SeqEditViewFormPtr) GetObjectExtra ((WindoW)ParentWindow (b));
  adp = (EditAlignDataPtr) wdp->data;
  if (adp != NULL)
  {
     salp = (SeqAlignPtr)adp->sap_align->data;
     salp = SeqAlignBoolSegToDenseSeg (salp);
     if (salp != NULL)
        LaunchAlignEditor (salp);
  }
  return;
}

static void CloseThisWindowProc (ButtoN b)
{
  WindoW             w;  
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;

  w = (WindoW)ParentWindow (b);
  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  adp = (EditAlignDataPtr) wdp->data;
  if (adp != NULL) {
     EditAlignDataFree (adp);
     wdp->data = NULL;
  }
  Hide (w);
  Update ();
  Remove (w);
}


/***************************************************************
***  switch_featOrder
***
*****************************************************************/
static void switch_featOrder (EditAlignDataPtr adp, Uint1 choice)
{
  Int2  oldstyle;
  Int2  j;
  Int4  groupNum;

  if (choice > 0) {
     oldstyle = GetMuskCurrentSt ();
     SetMuskCurrentSt (GetMuskStyleName (adp->styleNum));
     for(j =0; j<FEATDEF_ANY; j++)
     {   
        adp->featOrder[j] = (Uint1)GetMuskCParam(j, MSM_FORDER, MSM_NUM);
        groupNum = (Uint1)GetMuskCParam(j, MSM_FGROUP, MSM_NUM);
        adp->groupOrder[j] = (Uint1)GetMuskCParam(MSM_GROUPS, (Int2)groupNum, MSM_NUM);
     } 
     SetMuskCurrentSt (GetMuskStyleName (oldstyle));
  }
  else
     for(j=0; j<FEATDEF_ANY; ++j) adp->featOrder[j] = choice;
}

static ForM CreateSeqAlignViewForm (Int2 left, Int2 top, CharPtr windowname, SeqAlignPtr salp, Pointer dummy)
{
  VieweR             view = NULL;
  SeqEditViewProcsPtr  svpp;
  WindoW             w;
  SeqEditViewFormPtr wdp;
  EditAlignDataPtr   adp;
  GrouP              g, g2;
  SelStructPtr       ssp; 
  Int4               start = 0;
  Int2               wid, hgt;
  SeqIdPtr           sip;
  ValNodePtr         vnp;
  AlignNodePtr       anp;

  Char strid1[16], strid2[16];
  Char str [64];
  Int2 j;

  wdp = (SeqEditViewFormPtr) MemNew (sizeof (SeqEditViewForm));
  w = FixedWindow (left, top, (Int2)(-10), (Int2)(-10),  windowname, NULL /*, ResizeAlignDataWindow */);
  SetObjectExtra (w, (Pointer) wdp, CleanupSequenceEditorForm);
  wdp->form = (ForM) w;
  wdp->formmessage = SalsaFormMessage;
  svpp = (SeqEditViewProcsPtr) GetAppProperty ("SeqEditDisplayForm");
  if (svpp != NULL) {
      SetActivate (w, svpp->activateForm);
      wdp->appmessage = svpp->handleMessages;
  }
  wdp->pnl = NULL;
  wdp->data = NULL;

  adp = EditAlignDataInit (NULL, 100, 12, MARGINLEFT25, 10, NULL, FALSE, 3);
  if (dummy != NULL) {
     ssp = (SelStructPtr) dummy;    
     adp->input_entityID = ssp->entityID;
     adp->input_itemID = ssp->itemID;
     adp->input_itemtype = ssp->itemtype;
     if (ssp->region!=NULL)
        start = (Int4)SeqLocStart((SeqLocPtr)ssp->region);
  }
  else {
     adp->input_entityID = 0;
     adp->input_itemID = 0;
     adp->input_itemtype = 0;
  }
  if (salp != NULL) 
  {
     if (svpp != NULL) 
        adp->showfeat = svpp->showfeat;
     switch_featOrder (adp, 1);
     adp = SeqAlignToEditAlignData (adp, salp, start, NULL, (Uint1)(FALSE)); 
     if (adp == NULL)
        return NULL;
     if (adp->master.entityID == 0 && adp->master.region != NULL) 
     {
        j=0;
        sip = SeqLocId((SeqLocPtr)adp->master.region);
        for (vnp=adp->anp_list; vnp!=NULL; vnp=vnp->next) {
           if (vnp->choice != OBJ_SEQANNOT) {
              anp = (AlignNodePtr) vnp->data.ptrvalue;
              if (SeqIdForSameBioseq(sip, anp->sip))
              {
                 adp->master.entityID = anp->seq_entityID;
                 adp->master.itemID = anp->bsp_itemID;
                 adp->master.itemtype = OBJ_BIOSEQ;
                 adp->caret.entityID = anp->seq_entityID;
                 adp->caret.itemID = anp->bsp_itemID;
                 adp->caret.itemtype = OBJ_BIOSEQ;
              }
              j++;
              if (j==1) 
                 SeqIdWrite (anp->sip, strid1, PRINTID_REPORT, 16);
              else if (j==2)
                 SeqIdWrite (anp->sip, strid2, PRINTID_REPORT, 16);
           }
        }
     }
  }
  adp->wdp = (Pointer) wdp;

             /* Panel for aligned sequences *
             ********************************/
  wid = (Int2)(adp->pnlWidth*adp->charw+adp->margin.right);
  hgt = (Int2)(adp->pnlLine * adp->lineheight + adp->margin.bottom);
  view = CreateViewer(w, wid, hgt, TRUE, TRUE);
  wdp->graph = view;
  wdp->data = (Pointer) adp;

  g = HiddenGroup (w, 5, 0, NULL);                  
  sprintf (str, "From '%s' to '%s':", strid2, strid1);
  StaticPrompt (g, str, (Int2)0, dialogTextHeight, programFont, 'r');
  PushButton (g, "Replace", DoReplaceBtn);
  PushButton (g, "Merge 5p", DoMerge5Btn);
  PushButton (g, "Merge 3p", DoMerge3Btn);
  PushButton (g, "Copy features", DoCopyFeatBtn);

  g2 = HiddenGroup (w, 2, 0, NULL);                  
  PushButton (g2, "Preview alignment", LaunchAlignEdit);
  PushButton (g2, "Cancel", CloseThisWindowProc);

  AlignObjects (ALIGN_CENTER, (HANDLE) view, (HANDLE) g, (HANDLE) g2, NULL);
  RealizeWindow (w);

             /* register Panel & adp in SeqEditViewFormPtr *
             ***************************************/
  ViewForGraph (view, adp->input_entityID, adp, wid, adp->anp_list);
  return (ForM) w;
}

static void setup_aacolor (EditAlignDataPtr adp, SeqEditViewProcsPtr svpp)
{
     adp->col[0].r=svpp->col1r; adp->col[0].b=svpp->col1b; adp->col[0].g=svpp->col1g;
     adp->col[1].r=svpp->col2r; adp->col[1].b=svpp->col2b; adp->col[1].g=svpp->col2g;
     adp->col[2].r=svpp->col3r; adp->col[2].b=svpp->col3b; adp->col[2].g=svpp->col3g;
     setUpLetterColor(adp->colorRefs,(Uint1)('K' - '*'), svpp->col4r, svpp->col4g, svpp->col4b);
     setUpLetterColor(adp->colorRefs,(Uint1)('H' - '*'), svpp->col4r, svpp->col4g, svpp->col4b);
     setUpLetterColor(adp->colorRefs,(Uint1)('R' - '*'), svpp->col4r, svpp->col4g, svpp->col4b);
     setUpLetterColor(adp->colorRefs,(Uint1)('M' - '*'), svpp->col5r, svpp->col5g, svpp->col5b);
     setUpLetterColor(adp->colorRefs,(Uint1)('F' - '*'), svpp->col5r, svpp->col5g, svpp->col5b);
     setUpLetterColor(adp->colorRefs,(Uint1)('L' - '*'), svpp->col5r, svpp->col5g, svpp->col5b);
     setUpLetterColor(adp->colorRefs,(Uint1)('V' - '*'), svpp->col5r, svpp->col5g, svpp->col5b);
     setUpLetterColor(adp->colorRefs,(Uint1)('I' - '*'), svpp->col5r, svpp->col5g, svpp->col5b);
     setUpLetterColor(adp->colorRefs,(Uint1)('A' - '*'), svpp->col5r, svpp->col5g, svpp->col5b);
     setUpLetterColor(adp->colorRefs,(Uint1)('W' - '*'), svpp->col5r, svpp->col5g, svpp->col5b);
     setUpLetterColor(adp->colorRefs,(Uint1)('Y' - '*'), svpp->col5r, svpp->col5g, svpp->col5b);
     setUpLetterColor(adp->colorRefs,(Uint1)('E' - '*'), svpp->col6r, svpp->col6g, svpp->col6b);
     setUpLetterColor(adp->colorRefs,(Uint1)('D' - '*'), svpp->col6r, svpp->col6g, svpp->col6b);
     setUpLetterColor(adp->colorRefs,(Uint1)('S' - '*'), svpp->col6r, svpp->col6g, svpp->col6b);
     setUpLetterColor(adp->colorRefs,(Uint1)('Q' - '*'), svpp->col6r, svpp->col6g, svpp->col6b);
     setUpLetterColor(adp->colorRefs,(Uint1)('T' - '*'), svpp->col6r, svpp->col6g, svpp->col6b);
     setUpLetterColor(adp->colorRefs,(Uint1)('N' - '*'), svpp->col6r, svpp->col6g, svpp->col6b);
     setUpLetterColor(adp->colorRefs,(Uint1)('P' - '*'), svpp->col7r, svpp->col7g, svpp->col7b);
     setUpLetterColor(adp->colorRefs,(Uint1)('G' - '*'), svpp->col7r, svpp->col7g, svpp->col7b);
}

/************************************************
***
***   SeqEditFunc : to launch BIOSEQ editor
***
************************************************/
extern Int2 LIBCALLBACK SeqEditFunc (Pointer data)
{
  WindoW              w = NULL;
  BioseqPtr           bsp = NULL;
  SeqEditViewFormPtr  wdp = NULL;
  OMProcControlPtr    ompcp;
  OMUserDataPtr       omudp;
  Char                str [64];
  SelStruct ss;

  BioseqFetchInit (TRUE);

  ompcp = (OMProcControlPtr) data;
  if (ompcp == NULL || ompcp->proc == NULL) 
     return OM_MSG_RET_ERROR;
  switch (ompcp->input_itemtype) {
    case OBJ_BIOSEQ :
      bsp = (BioseqPtr) ompcp->input_data;
      break;
   case 0 :
      return OM_MSG_RET_ERROR;
      break;
    default :
      return OM_MSG_RET_ERROR;
      break;
  }
  if (bsp == NULL) {
     return OM_MSG_RET_ERROR;
  }  
  ss.entityID = ompcp->input_entityID;
  ss.itemID = ompcp->input_itemID;
  ss.itemtype = ompcp->input_itemtype;
  ss.regiontype =0;
  ss.region = NULL;

  SeqIdWrite (bsp->id, str, PRINTID_REPORT, 64);
  w = (WindoW) CC_CreateBioseqViewForm (-50, -33, str, bsp, &ss);
  if (w != NULL) {
     wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
     if (wdp != NULL) {
        wdp->input_entityID = ompcp->input_entityID;
        wdp->input_itemID = ompcp->input_itemID;
        wdp->input_itemtype = ompcp->input_itemtype;
        wdp->this_itemtype = OBJ_BIOSEQ;
        wdp->this_subtype = bsp->repr;
        wdp->procid = ompcp->proc->procid;
        wdp->proctype = ompcp->proc->proctype;
        wdp->userkey = OMGetNextUserKey ();
        omudp = ObjMgrAddUserData (ompcp->input_entityID, ompcp->proc->procid, OMPROC_EDIT, wdp->userkey);
        if (omudp != NULL) {
          omudp->userdata.ptrvalue = (Pointer) wdp;
          omudp->messagefunc = BioseqEditMsgFunc;
        }
        checkEntityIDForMsg (wdp);
     }
     Show (w);
     Update ();
     CaptureSlateFocus ((SlatE) wdp->pnl);
     Select (w);
     return OM_MSG_RET_DONE;
  }
  return OM_MSG_RET_ERROR;
}

/************************************************
***
***   AlgEditFunc : to launch SEQALIGN editor
***
************************************************/
extern Int2 LIBCALLBACK AlgEditFunc (Pointer data)
{
  WindoW              w = NULL;
  SeqAlignPtr         salp = NULL;
  SeqEditViewFormPtr  wdp = NULL;
  OMProcControlPtr    ompcp;
  OMUserDataPtr       omudp;
  Char                str [64];
  SelStruct           ss;
  SeqAnnotPtr         sap, sap2;

  BioseqFetchInit (TRUE);

  ompcp = (OMProcControlPtr) data;
  if (ompcp == NULL || ompcp->proc == NULL) {
     Message (MSG_ERROR, "Data NULL [1]");
     return OM_MSG_RET_ERROR;
  }
  switch (ompcp->input_itemtype) {
    case OBJ_SEQALIGN :
      salp = (SeqAlignPtr) ompcp->input_data;
      break;
   case 0 :
      return OM_MSG_RET_ERROR;
      break;
    default :
      return OM_MSG_RET_ERROR;
      break;
  }
  if (salp == NULL) {
     Message (MSG_ERROR, "Data NULL [2]");
     return OM_MSG_RET_ERROR;
  }
/********** TRICK in case there is a list of SeqAlign *********/
  if (salp->next!=NULL) {
     sap = SeqAnnotForSeqAlign (salp);
     sap2 = (SeqAnnotPtr) AsnIoMemCopy ((Pointer) sap, (AsnReadFunc) SeqAnnotAsnRead, (AsnWriteFunc) SeqAnnotAsnWrite);
     salp = (SeqAlignPtr) sap2->data;
     salp->next = NULL;
     sap->data=NULL;
     SeqAnnotFree (sap);
     sap2->data=NULL;
     SeqAnnotFree (sap2);
  }
/********** TRICK *********/

  ss.entityID = ompcp->input_entityID;
  ss.itemID = ompcp->input_itemID;
  ss.itemtype = ompcp->input_itemtype;
  ss.regiontype =0;
  ss.region = NULL;
  StringCpy (str, "Pairwise Alignment");
  w = (WindoW) CreateSeqAlignEditForm (-50, -33, str, salp, &ss);
  if (w != NULL) {
     wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
     if (wdp != NULL) {
        wdp->input_entityID = ompcp->input_entityID;
        wdp->input_itemID = ompcp->input_itemID;
        wdp->input_itemtype = ompcp->input_itemtype;
        wdp->this_itemtype = OBJ_SEQALIGN;
        wdp->this_subtype = 0;
        wdp->procid = ompcp->proc->procid;
        wdp->proctype = ompcp->proc->proctype;
        wdp->userkey = OMGetNextUserKey ();
        omudp = ObjMgrAddUserData (ompcp->input_entityID, ompcp->proc->procid, OMPROC_EDIT, wdp->userkey);
        if (omudp != NULL) {
           omudp->userdata.ptrvalue = (Pointer) wdp;
           omudp->messagefunc = BioseqEditMsgFunc;
        }
        checkEntityIDForMsg (wdp);
     }
     Show (w);
     Update ();
     CaptureSlateFocus ((SlatE) wdp->pnl);
     Select (w);
     return OM_MSG_RET_DONE;
  }
  return OM_MSG_RET_ERROR;
}

/************************************************
***
***   AnnotAlgEditFunc : to launch SEQANNOT editor
***
************************************************/
extern Int2 LIBCALLBACK AnnotAlgEditFunc (Pointer data)
{
  WindoW              w;
  SeqAnnotPtr         sap;
  SeqAlignPtr         salp = NULL;
  SeqEditViewFormPtr  wdp = NULL;
  OMProcControlPtr    ompcp;
  OMUserDataPtr       omudp;
  Char                str [64];
  SelStruct           ss;

  BioseqFetchInit (TRUE);

  ompcp = (OMProcControlPtr) data;
  if (ompcp == NULL || ompcp->proc == NULL) {
     Message (MSG_ERROR, "Data NULL [1]");
     return OM_MSG_RET_ERROR;
  }
  switch (ompcp->input_itemtype) {
    case OBJ_SEQANNOT :
      sap = (SeqAnnotPtr) ompcp->input_data;
      break;
   case 0 :
      return OM_MSG_RET_ERROR;
      break;
    default :
      return OM_MSG_RET_ERROR;
      break;
  }
  if (sap == NULL) {
     Message (MSG_ERROR, "Data NULL [2]");
     return OM_MSG_RET_ERROR;
  }
  if (sap->data==NULL || sap->type!=2) {
     return OM_MSG_RET_ERROR;
  }
  salp = (SeqAlignPtr)sap->data;
/****************************************/
{
  Uint1 moltype;
  Boolean is_prot; 
  moltype = SeqAlignMolType(salp);
  is_prot = (Boolean)(ISA_aa(moltype));
  if (is_prot)
     salp = DenseSegToDenseDiag (salp);
}
/*******************************************/
  if (salp == NULL)
     return OM_MSG_RET_ERROR;
  ss.entityID = ompcp->input_entityID;
  ss.itemID = ompcp->input_itemID;
  ss.itemtype = ompcp->input_itemtype;
  ss.regiontype =0;
  ss.region = NULL;
  StringCpy (str, "Multiple Alignment");
  w = (WindoW) CreateSeqAlignEditForm (-50, -33, str, salp, &ss);
  if (w != NULL) 
  {
     wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
     if (wdp != NULL) 
     {
        wdp->input_entityID = ompcp->input_entityID;
        wdp->input_itemID = ompcp->input_itemID;
        wdp->input_itemtype = ompcp->input_itemtype;
        wdp->this_itemtype = OBJ_SEQALIGN;
        wdp->this_subtype = 0;
        wdp->procid = ompcp->proc->procid;
        wdp->proctype = ompcp->proc->proctype;
        wdp->userkey = OMGetNextUserKey ();
        omudp = ObjMgrAddUserData (ompcp->input_entityID, ompcp->proc->procid, OMPROC_EDIT, wdp->userkey);
        if (omudp != NULL) 
        {
           omudp->userdata.ptrvalue = (Pointer) wdp;
           omudp->messagefunc = BioseqEditMsgFunc;
        }
        checkEntityIDForMsg (wdp);
     }
     Show (w);
     Update ();
     CaptureSlateFocus ((SlatE) wdp->pnl);
     Select (w);
     return OM_MSG_RET_DONE;
  }
  return OM_MSG_RET_ERROR;
}

extern Int2 LIBCALLBACK AlgViewFunc (Pointer data)
{
  WindoW              w;
  SeqAlignPtr         salp = NULL;
  SeqEditViewFormPtr  wdp = NULL;
  OMProcControlPtr    ompcp;
  OMUserDataPtr       omudp;
  Char                str [64];
  SelStruct           ss;

  BioseqFetchInit (TRUE);

  ompcp = (OMProcControlPtr) data;
  if (ompcp == NULL || ompcp->proc == NULL) {
     Message (MSG_ERROR, "Data NULL [1]");
     return OM_MSG_RET_ERROR;
  }
  switch (ompcp->input_itemtype) {
    case OBJ_SEQALIGN :
      salp = (SeqAlignPtr) ompcp->input_data;
      break;
   case 0 :
      return OM_MSG_RET_ERROR;
      break;
    default :
      return OM_MSG_RET_ERROR;
      break;
  }
  if (salp == NULL) {
     Message (MSG_ERROR, "Data NULL [2]");
     return OM_MSG_RET_ERROR;
  }
  ss.entityID = ompcp->input_entityID;
  ss.itemID = ompcp->input_itemID;
  ss.itemtype = ompcp->input_itemtype;
  ss.regiontype =0;
  ss.region = NULL;
  StringCpy (str, "alignment");
  w = (WindoW) CreateSeqAlignViewForm (-50, -33, str, salp, &ss);
  if (w == NULL) {
     return OM_MSG_RET_ERROR;
  }
  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  if (wdp != NULL) {
     wdp->input_entityID = ompcp->input_entityID;
     wdp->input_itemID = ompcp->input_itemID;
     wdp->input_itemtype = ompcp->input_itemtype;
     wdp->this_itemtype = OBJ_SEQALIGN;
     wdp->this_subtype = 0;
     wdp->procid = ompcp->proc->procid;
     wdp->proctype = ompcp->proc->proctype;
     wdp->userkey = OMGetNextUserKey ();
     omudp = ObjMgrAddUserData (ompcp->input_entityID, ompcp->proc->procid, OMPROC_EDIT, wdp->userkey);
     if (omudp != NULL) {
        omudp->userdata.ptrvalue = (Pointer) wdp;
        omudp->messagefunc = BioseqEditMsgFunc;
     }
     checkEntityIDForMsg (wdp);
  }
  Show (w);
  Update ();
  Select (w);
  return OM_MSG_RET_DONE;
}

extern Int2 LIBCALLBACK SeqEditViewFunc (Pointer data)
{
  WindoW              w;
  BioseqPtr           bsp = NULL;
  SeqEditViewFormPtr  wdp = NULL;
  OMProcControlPtr    ompcp;
  OMUserDataPtr       omudp;
  Char                str [64];
  SelStruct ss;

  BioseqFetchInit (TRUE);

  ompcp = (OMProcControlPtr) data;
  if (ompcp == NULL || ompcp->proc == NULL) 
     return OM_MSG_RET_ERROR;
  switch (ompcp->input_itemtype) {
    case OBJ_BIOSEQ :
      bsp = (BioseqPtr) ompcp->input_data;
      break;
   case 0 :
      return OM_MSG_RET_ERROR;
      break;
    default :
      return OM_MSG_RET_ERROR;
      break;
  }
  if (bsp == NULL) {
     return OM_MSG_RET_ERROR;
  }  
  ss.entityID = ompcp->input_entityID;
  ss.itemID = ompcp->input_itemID;
  ss.itemtype = ompcp->input_itemtype;
  ss.regiontype =2;
  ss.region = NULL;

  SeqIdWrite (bsp->id, str, PRINTID_REPORT, 64);
  w = (WindoW) CC_CreateBioseqViewForm (-50, -33, str, bsp, &ss);
  if (w != NULL) {
     wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
     if (wdp != NULL) {
        wdp->input_entityID = ompcp->input_entityID;
        wdp->input_itemID = ompcp->input_itemID;
        wdp->input_itemtype = ompcp->input_itemtype;
        wdp->this_itemtype = OBJ_BIOSEQ;
        wdp->this_subtype = bsp->repr;
        wdp->procid = ompcp->proc->procid;
        wdp->proctype = ompcp->proc->proctype;
        wdp->userkey = OMGetNextUserKey ();
        omudp = ObjMgrAddUserData (ompcp->input_entityID, ompcp->proc->procid, OMPROC_EDIT, wdp->userkey);
        if (omudp != NULL) {
          omudp->userdata.ptrvalue = (Pointer) wdp;
          omudp->messagefunc = BioseqEditMsgFunc;
        }
        checkEntityIDForMsg (wdp);
     }
     Show (w);
     Update ();
     CaptureSlateFocus ((SlatE) wdp->pnl);
     Select (w);
     return OM_MSG_RET_DONE;
  }
  return OM_MSG_RET_ERROR;
}
extern Int2 LIBCALLBACK AlgEditViewFunc (Pointer data)
{
  WindoW              w;
  SeqAlignPtr         salp = NULL;
  SeqEditViewFormPtr  wdp = NULL;
  OMProcControlPtr    ompcp;
  OMUserDataPtr       omudp;
  Char                str [64];
  SelStruct           ss;
  SeqAnnotPtr         sap, sap2;

  BioseqFetchInit (TRUE);

  ompcp = (OMProcControlPtr) data;
  if (ompcp == NULL || ompcp->proc == NULL) {
     Message (MSG_ERROR, "Data NULL [1]");
     return OM_MSG_RET_ERROR;
  }
  switch (ompcp->input_itemtype) {
    case OBJ_SEQALIGN :
      salp = (SeqAlignPtr) ompcp->input_data;
      break;
   case 0 :
      return OM_MSG_RET_ERROR;
      break;
    default :
      return OM_MSG_RET_ERROR;
      break;
  }
  if (salp == NULL) {
     Message (MSG_ERROR, "Data NULL [2]");
     return OM_MSG_RET_ERROR;
  }
/********** TRICK *********/
  sap = SeqAnnotForSeqAlign (salp);
  sap2 = (SeqAnnotPtr) AsnIoMemCopy ((Pointer) sap, (AsnReadFunc) SeqAnnotAsnRead, (AsnWriteFunc) SeqAnnotAsnWrite);
  salp = (SeqAlignPtr) sap2->data;
  salp->next = NULL;
  sap->data=NULL;
  SeqAnnotFree (sap);
  sap2->data=NULL;
  SeqAnnotFree (sap2);

  ss.entityID = ompcp->input_entityID;
  ss.itemID = ompcp->input_itemID;
  ss.itemtype = ompcp->input_itemtype;
  ss.regiontype =2;
  ss.region = NULL;
  StringCpy (str, "alignment");
  w = (WindoW) CreateSeqAlignEditForm (-50, -33, str, salp, &ss);
  if (w == NULL) {
     return OM_MSG_RET_ERROR;
  }
  wdp = (SeqEditViewFormPtr) GetObjectExtra (w);
  if (wdp != NULL) {
     wdp->input_entityID = ompcp->input_entityID;
     wdp->input_itemID = ompcp->input_itemID;
     wdp->input_itemtype = ompcp->input_itemtype;
     wdp->this_itemtype = OBJ_SEQALIGN;
     wdp->this_subtype = 0;
     wdp->procid = ompcp->proc->procid;
     wdp->proctype = ompcp->proc->proctype;
     wdp->userkey = OMGetNextUserKey ();
     omudp = ObjMgrAddUserData (ompcp->input_entityID, ompcp->proc->procid, OMPROC_EDIT, wdp->userkey);
     if (omudp != NULL) {
        omudp->userdata.ptrvalue = (Pointer) wdp;
        omudp->messagefunc = BioseqEditMsgFunc;
     }
     checkEntityIDForMsg (wdp);
  }
  Show (w);
  Update ();
  CaptureSlateFocus ((SlatE) wdp->pnl);
  Select (w);
  return OM_MSG_RET_DONE;
}

/***********************************************************
***
*** Launch Editors
***     
***
************************************************************/
extern void LaunchAnnotAlignEditor (SeqAnnotPtr sap)
{
  Uint2            entityID,
                   itemID;
  Int2             handled;
  Uint2            options;

  if (sap != NULL) {
     entityID = ObjMgrRegister (OBJ_SEQANNOT, (Pointer) sap);
     options = ObjMgrGetOptions(entityID);
     options |= OM_OPT_FREE_IF_NO_VIEW;
     ObjMgrSetOptions(options, entityID);
     itemID = GetItemIDGivenPointer (entityID, OBJ_SEQANNOT, (Pointer) sap);
     handled = GatherProcLaunch (OMPROC_EDIT, FALSE, entityID, itemID, OBJ_SEQANNOT, 0, 0, OBJ_SEQANNOT, 0);
  }
}

extern void LaunchAlignEditor (SeqAlignPtr salp)
{
  Uint2            entityID,
                   itemID;
  Int2             handled;
  Uint2            options;

  if (salp != NULL) {
     entityID = ObjMgrRegister (OBJ_SEQALIGN, (Pointer) salp);
     options = ObjMgrGetOptions(entityID);
     options |= OM_OPT_FREE_IF_NO_VIEW;
     ObjMgrSetOptions(options, entityID);
     itemID = GetItemIDGivenPointer (entityID, OBJ_SEQALIGN, (Pointer) salp);
     handled = GatherProcLaunch (OMPROC_EDIT, FALSE, entityID, itemID, OBJ_SEQALIGN, 0, 0, OBJ_SEQALIGN, 0);
  }
}

extern void LaunchAlignViewer (SeqAlignPtr salp)
{
  Uint2            entityID,
                   itemID;
  Int2             handled;
  Uint2            options;

  if (salp != NULL) {
     entityID = ObjMgrRegister (OBJ_SEQALIGN, (Pointer) salp);
     options = ObjMgrGetOptions(entityID);
     options |= OM_OPT_FREE_IF_NO_VIEW;
     ObjMgrSetOptions(options, entityID);
     itemID = GetItemIDGivenPointer (entityID, OBJ_SEQALIGN, (Pointer) salp);
     handled = GatherProcLaunch (OMPROC_VIEW, FALSE, entityID, itemID, OBJ_SEQALIGN, 0, 0, OBJ_SEQALIGN, 0);
  }
}

extern Int2 LIBCALLBACK LaunchAlignEditorFromDesktop (Pointer data)
{
  OMProcControlPtr ompcp;
  SeqEntryPtr      sep; 
  BioseqSetPtr     bssp;
  BioseqPtr        bsp;
  SeqAnnotPtr      sap;
  SeqAlignPtr      salp;

  ompcp = (OMProcControlPtr) data;
  sep=ReadLocalAlignment(SALSAA_GCG, NULL);

  sap = NULL;
  if (IS_Bioseq (sep)) {
     bsp = (BioseqPtr) sep->data.ptrvalue;
     sap = bsp->annot;
  } else if (IS_Bioseq_set (sep)) {
     bssp = (BioseqSetPtr) sep->data.ptrvalue;
     sap = bssp->annot;
  }
  if (sap->data != NULL) {
     salp = (SeqAlignPtr)sap->data;
     LaunchAlignEditor (salp);
  }
  return OM_MSG_RET_OK;
}

extern Int2 LIBCALLBACK LaunchAlignEditorFromDesktop2 (Pointer data)
{
  OMProcControlPtr ompcp;
  SeqEntryPtr      sep; 
  BioseqSetPtr     bssp;
  BioseqPtr        bsp;
  SeqAnnotPtr      sap;
  SeqAlignPtr      salp;

  ompcp = (OMProcControlPtr) data;
  sep=ReadLocalAlignment(SALSAA_PHYLIP, NULL);

  sap = NULL;
  if (IS_Bioseq (sep)) {
     bsp = (BioseqPtr) sep->data.ptrvalue;
     sap = bsp->annot;
  } else if (IS_Bioseq_set (sep)) {
     bssp = (BioseqSetPtr) sep->data.ptrvalue;
     sap = bssp->annot;
  }
  if (sap->data != NULL) {
     salp = (SeqAlignPtr)sap->data;
     LaunchAlignEditor (salp);
  }
  return OM_MSG_RET_OK;
}

extern Int2 LIBCALLBACK LaunchAlignEditorFromDesktop3 (Pointer data)
{
  OMProcControlPtr ompcp;
  SeqEntryPtr      sep; 
  BioseqSetPtr     bssp;
  BioseqPtr        bsp;
  SeqAnnotPtr      sap;
  SeqAlignPtr      salp;

  ompcp = (OMProcControlPtr) data;
  sep=ReadLocalAlignment(SALSAA_FASTGAP, NULL);

  sap = NULL;
  if (IS_Bioseq (sep)) {
     bsp = (BioseqPtr) sep->data.ptrvalue;
     sap = bsp->annot;
  } else if (IS_Bioseq_set (sep)) {
     bssp = (BioseqSetPtr) sep->data.ptrvalue;
     sap = bssp->annot;
  }
  if (sap->data != NULL) {
     salp = (SeqAlignPtr)sap->data;
     LaunchAlignEditor (salp);
  }
  return OM_MSG_RET_OK;
}

extern Int2 LIBCALLBACK LaunchAlignEditorFromDesktop4 (Pointer data)
{
  OMProcControlPtr ompcp;
  SeqEntryPtr      sep; 
  SeqAlignPtr      salp;

  ompcp = (OMProcControlPtr) data;
  sep=ReadLocalAlignment(SALSAA_FASTA, NULL);
  salp= seqentrytoseqalign2 (sep);
  if (salp != NULL) {
     LaunchAlignEditor (salp);
  }
  return OM_MSG_RET_OK;
}


/**********************************************
static void LoadHistProc (IteM i)
{
  WindoW             w;
  EditAlignDataPtr   adp;
  BioseqPtr          bsp;
  Uint2         entityID;
  Int2          handled;
 
  w = (WindoW)ParentWindow (i);
  if ( ( adp = GetAlignEditData (w) ) == NULL ) return;
  if ( adp->seqnumber < 1) 
         return;
  bsp = BioseqLockById (SeqLocId((SeqLocPtr)adp->master.region));
  if (bsp != NULL) {
     if(bsp->hist != NULL)
        if (bsp->hist->assembly != NULL) {	
           LaunchAlignEditor (bsp->hist->assembly);
        }
  }
  else
     Message (MSG_ERROR, "fail in LoadAlignment [1]");
  BioseqUnlock (bsp);
  return;
}
/*************************************************
***
*************************************************/

static void showentry_fromSeqId (SeqIdPtr sip)

{
  BioseqPtr  bsp;
  Uint2      entityID;
  Int2       handled;
  Uint2      itemID;

  if (sip == NULL) return;
  bsp = BioseqLockById (sip);
  if (bsp == NULL) return;
  entityID = BioseqFindEntity (sip, &itemID);
  if (entityID == 0) return;
  WatchCursor ();
  handled = GatherProcLaunch (OMPROC_VIEW, FALSE, entityID, itemID, OBJ_BIOSEQ, 0, 0, OBJ_BIOSEQ, 0);
  ArrowCursor ();
  if (handled != OM_MSG_RET_DONE || handled == OM_MSG_RET_NOPROC) {
    Message (MSG_FATAL, "Unable to launch viewer.");
  } else {
    ObjMgrSetOptions (OM_OPT_FREE_IF_NO_VIEW, entityID);
  }
  BioseqUnlockById (sip);

}


static void ShowSequenceProc (IteM i)
{
  SelStructPtr     ssp;

  ssp = ObjMgrGetSelected ();  
  for (; ssp != NULL; ssp = ssp->next) 
         if ( checkssp_for_editor (ssp) )
            showentry_fromSeqId (SeqLocId((SeqLocPtr) ssp->region));
  return;
}
/**********************************************/
/**********************************************
static void DisplayFeatureProc (IteM i)
{
  WindoW           temport;
  PaneL            pnl;
  EditAlignDataPtr adp;

  if ( ( pnl= GetPanelFromWindow ((WindoW)ParentWindow (i)) ) == NULL) return;
  if ( ( adp = GetAlignDataPanel (pnl) ) == NULL ) return;
  if ( adp->seqnumber == 0 ) return;
  adp->displaytype = !(adp->displaytype);
  data_collect_arrange (adp, TRUE);
  temport = SavePort((WindoW)ParentWindow (i));
  Select (pnl);
  inval_panel (pnl, -1, -1);
  RestorePort (temport);
  return;
}

**********************************************/
/**********************************************
**********************************************/
