#ifndef _POSITION_H_
#define _POSITION_H_

#include "atom.h"

class Part;
class ostream;
class Tuplet;

/**
 * The class Position is a very important class. It contains all convenience methods to convert the simple linear time (the ticks, stored as an long int) into
 * the three dimensional position presentation (bars, beats and ticks within the beat) an vice versa.
 **/
class Position : public Atoma
{
 private:
  long totalTicks;
  int modulo(int m) const { return totalTicks%m; }
  int Meter(int,Part*) const;
 public:
  /**
   * Default Constructor
   **/
  Position();

  /**
   * Constructs a position with the initial value p
   **/
  Position(long p);

  /**
   * Constructs a position with the initial value bar.beat.tick
   **/
  Position(int bar, int beat, int tick);

  /**
   * Constructs a position with the format "bar.beat.tick" given as a char*
   **/
  Position(char * str, int);

  /**
   * Copy Constructor
   **/
  Position(const Position&);

  /**
   * Returns the linear time, measured in ticks
   **/
  long ticks() const { return totalTicks; }

  /**
   * depricated
   **/
  long negative();

  /**
   * Returns the bar value of the position
   **/
  int bar() const;

  /**
   * Returns the beat value of the position
   **/
  int beat() const;

  /**
   * Returns the tick value of the position
   **/
  int tick() const;

  /**
   * Returns true if this position and other are located within the same bar
   **/
  bool sameBar(Position & other) const;

  /**
   * Returns true if this position and other are located within the same bar and the same beat
   **/
  bool sameBeat(Position & other) const;

  /**
   * Returns true if this position is located within the same bar or greater than the other
   **/
  bool sameBarOrGreater(Position & other) const;

  /**
   * This method presents the position in the bar.beat.tick format.
   * It sets the first three parameters bar, beat and tick according to the parameters master (specifying the mastertrack's part),
   * meter0 and meter1 (specifying the denominator and the numerator of the global meter).
   * Usually this method is not needed, since the position obtains the master and meter information from the global song instance.
   * Alternatively try the gBBT method with three parameters.
   **/
  void gBBT(int & bar, int & beat, int & tick, Part * master, int & meter0, int & meter1, bool returnMeter=false) const;

  /**
   * This method presents the position in the bar.beat.tick format.
   * It sets the first three parameters bar, beat and tick according to the songs mastertrack and global meter values
   **/
  void gBBT(int & bar, int & beat, int & tick) const;

  /**
   * This method moves the position to the beginning of the next bar
   **/
  void nextBar();

  /**
   * This method moves the position to the beginning of the next beat
   **/
  void nextBeat();

  /**
   * This method moves the position to the beginning of the previous bar
   **/
  void prevBar();

  /**
   * Rounds off the position to the grid specified by the res parameter (resoloution).
   * For Tuplets it is required to provide the Tuplet instance in order to obtain sensible values.
   **/
  void snap(int res, Tuplet * tuplet = 0);

  /**
   * Assignment Operator
   **/
  Position & operator=(const Position&);

  /**
   * Assignment Operator
   **/
  Position & operator=(long);

  /**
   * Sets the position by the format bar.beat.tick, while specifying the mastertrack's part and optionally the global meter.
   * The snap parameter rounds off the position to the specified value.
   **/
  Position & set(int bar, int beat, int tick, Part * master, int m0=0, int m1=0, int snap=0);

  /**
   * Increases the position by the specified value
   **/
  Position & operator+=(Position);

  /**
   * Decreases the position by the specified value
   **/
  Position & operator-=(Position);

  /**
   * Multiplies the position by the specified value
   **/
  double operator*(double);

  /**
   * Adds two positions
   **/
  friend Position operator+(Position,Position);

  /**
   * Subtracts a positions from another position
   **/
  friend Position operator-(Position,Position);

  /**
   * Adds two positions
   **/
  friend long operator+(Position,long);

  /**
   * Subtracts a positions from another position
   **/
  friend long operator-(Position,long);

  /**
   * Returns true if two positions are equal
   **/
  friend bool operator==(Position,Position);

  /**
   * Returns true if two positions are equal
   **/
  friend bool operator==(Position,long);

  /**
   * Returns true if two positions are not equal
   **/
  friend bool operator!=(Position,Position);

  /**
   * Returns true if two positions are not equal
   **/
  friend bool operator!=(Position,long);

  /**
   * Compares two positions
   **/
  friend bool operator<(Position,Position);

  /**
   * Compares two positions
   **/
  friend bool operator<=(Position,Position);

  /**
   * Compares two positions
   **/
  friend bool operator>(Position,Position);

  /**
   * Compares two positions
   **/
  friend bool operator>=(Position,Position);

  /**
   * Increases the position by one tick
   **/
  friend Position & operator++(Position&,int);

  /**
   * Sends the position p (in the bar.beat.tick format) to the stream s
   **/
  friend ostream & operator<<(ostream & s, const Position & p);

  /**
   * Implementation of the print method
  **/
  virtual ostream & print(int,ostream&) const;

  /**
   * Implementation of the flush method
  **/
  virtual void flush(const char*) const;

  /**
   * Implementation of the copy method
  **/
  virtual Element * copy() const;

  /**
   * Always returns false, since positions never are events
   **/
  virtual bool isEvent() const { return false; }


};


#endif
