// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// event.cpp
//
// History:
//  16-12-2000 - created
////////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "event.h"
#include "extwindow.h"
#include "globals.h"

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Event
// DESCRIPTION: Create a new event.
// PARAMETERS : type     - event type
//              trigger  - event trigger type
//              td1..td3 - trigger data
//              d1..d3   - event data
//              title    - message title string index
//              msg      - message string index
//              player   - player the event refers to
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Event::Event( unsigned char type, unsigned char trigger,
         short td1, short td2, short td3, short d1, short d2, short d3,
         short title, short msg, Player *player ) {
  e_type = type;
  e_trigger = trigger;
  e_tdata[0] = td1;
  e_tdata[1] = td2;
  e_tdata[2] = td3;
  e_data[0] = d1;
  e_data[1] = d2;
  e_data[2] = d3;
  e_title = title;
  e_message = msg;
  e_player = player;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Event
// DESCRIPTION: Load an event from a file.
// PARAMETERS : file    - file descriptor
//              players - pointer to the array of players
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Event::Event( SDL_RWops *file, Player **players ) {
  int i;
  unsigned char pid;

  SDL_RWread( file, &e_type, sizeof(unsigned char), 1 );
  SDL_RWread( file, &e_trigger, sizeof(unsigned char), 1 );
  for ( i = 0; i < 3; i++ ) e_tdata[i] = SDL_ReadLE16( file );
  for ( i = 0; i < 3; i++ ) e_data[i] = SDL_ReadLE16( file );
  e_title = SDL_ReadLE16( file );
  e_message = SDL_ReadLE16( file );

  SDL_RWread( file, &pid, sizeof(unsigned char), 1 );
  e_player = players[pid];
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Save
// DESCRIPTION: Save the event data to a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Event::Save( SDL_RWops *file ) const {
  int i;
  unsigned char pid = e_player->ID();

  SDL_RWwrite( file, &e_type, sizeof(unsigned char), 1 );
  SDL_RWwrite( file, &e_trigger, sizeof(unsigned char), 1 );

  for ( i = 0; i < 3; i++ ) SDL_WriteLE16( file, e_tdata[i] );
  for ( i = 0; i < 3; i++ ) SDL_WriteLE16( file, e_data[i] );
  SDL_WriteLE16( file, e_title );
  SDL_WriteLE16( file, e_message );

  SDL_RWwrite( file, &pid, sizeof(unsigned char), 1 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Check
// DESCRIPTION: Check whether the event can be executed. The trigger
//              conditions must be met.
// PARAMETERS : -
// RETURNS    : 0 if the event cannot be activated, 1 if it can, and 2
//              if it can be activated and requires a message window to
//              pop up
//
// HISTORY
//   14-05-2001 - getting the opponent's number of units was broken and
//                sometimes returned random values, often resulting in
//                a first-round win
//   19-05-2001 - also trigger ETRIGGER_UNIT_DESTROYED if the unit was
//                captured by another player
//              - add ETRIGGER_HAVE_UNIT
//              - add turn check to ETRIGGER_HAVE_BUILDING
////////////////////////////////////////////////////////////////////////

short Event::Check( void ) {
  short rc = 0;

  switch ( e_trigger ) {
  case ETRIGGER_TURN:
    if ( Gam->Turn() >= e_tdata[0] ) rc = 1;
    break;
  case ETRIGGER_UNIT_DESTROYED:
    if ( e_tdata[0] == -1 ) {                              // destroy all enemy units to score
      Player *opp = Gam->GetPlayer( e_player->ID()^1 );
      if ( opp->Units(0) == 0 ) rc = 1;
    } else {                                                // trigger if
      Unit *u = Gam->GetUnitByID( e_tdata[0] );             //  * unit not found
      if ( !u || !u->IsAlive() ||                           //  * unit found, but already dead
         (u->Owner() && (u->Owner()->ID() != e_tdata[1])) ) //  * owner != original owner (captured)
        rc = 1;
    }
    break;
  case ETRIGGER_HAVE_BUILDING:
    if ( (e_tdata[2] == -1) || (e_tdata[2] == Gam->Turn()) ) {
      Building *b = Gam->GetBuildingByID( e_tdata[0] );
      if ( b && b->Owner() && (b->Owner()->ID() == e_tdata[1]) ) rc = 1;
    }
    break;
  case ETRIGGER_HAVE_UNIT:
    if ( (e_tdata[2] == -1) || (e_tdata[2] == Gam->Turn()) ) {
      Unit *u = Gam->GetUnitByID( e_tdata[0] );
      if ( u && u->Owner() && (u->Owner()->ID() == e_tdata[1]) ) rc = 1;
    }
    break;
  }

  if ( rc && (e_player->Type() == HUMAN) && (e_player->Mode() != MODE_OFF) &&
       (e_player == Gam->CurrentPlayer()) && (e_message != -1) ) rc = 2;
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Execute
// DESCRIPTION: Execute this event. What this means depends on the event
//              type.
// PARAMETERS : msgwin  - pointer to a message window to show messages
//                        (may be NULL for events with a return code of
//                        1 from Event::Check())
//              strings - pointer to the array of message pointers for
//                        the current level
// RETURNS    : If the event usually had triggered a message given out
//              to a player, but that player was not reachable at
//              execution time (i.e. it was not his turn) a new event
//              of the type EVENT_MESSAGE will be created with its
//              trigger set to next turn. This event will be returned
//              and must be put in the event queue.
//
// HISTORY
//   19-05-2001 - added EVENT_CREATE_UNIT
//   20-09-2001 - added EVENT_VICTORY
////////////////////////////////////////////////////////////////////////

Event *Event::Execute( MessageWindow *msgwin, char **strings ) {
  const char *tstr;
  if ( e_title == -1 ) tstr = e_player->Name();
  else tstr = strings[e_title];

  bool show_msg = true;

  switch ( e_type ) {
  case EVENT_MESSAGE:
    break;
  case EVENT_CREATE_UNIT: {
    Building *b = Gam->GetBuildingByID( e_data[1] );
    if ( b && (b->Owner() == e_player) )
      Gam->CreateUnit( Gam->GetUnitType( e_data[0] ),
             b->Position().x, b->Position().y, e_player );
    else show_msg = false;
    break; }
  case EVENT_PRODUCTION: {
    Building *b = Gam->GetBuildingByID( e_data[0] );

    if ( e_data[2] == 0 ) b->SetCrystalProduction( (unsigned char)e_data[1] );
    else if ( b->Owner() == e_player ) b->SetUnitProduction( 1 << e_data[1] );
    else show_msg = false;

    break; }
  case EVENT_SCORE:
    e_player->Score( e_data[0] );
    e_player->Success( (char)e_data[1] );
    break;
  case EVENT_VICTORY:
    Gam->ShowDebriefing( e_data[0], true );
    break;
  }

  if ( (e_message == -1) || (e_player->Type() != HUMAN) ) show_msg = false;
  if ( show_msg ) {
    if ( (Gam->CurrentPlayer() == e_player) && (e_player->Mode() != MODE_OFF) )
      DisplayMessage( msgwin, tstr, strings[e_message] );
    else return new Event( EVENT_MESSAGE, ETRIGGER_TURN, Gam->Turn(),
                           0, 0, 0, 0, 0, e_title, e_message, e_player );
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::DisplayMessage
// DESCRIPTION: Pop up a MessageWindow with a message.
// PARAMETERS : title - title string
//              body  - body text
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Event::DisplayMessage( MessageWindow *win, const char *title,
                            const char *body ) const {
  win->SetTitle( title );
  win->textscroll->SetText( body );
  win->Draw();
  win->Show();
}

