// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// player.cpp
//
// History:
//  08-12-2000 - created
//  09-03-2001 - added p_briefing to Player class
//             - removed Create() and Load(), added Player() constructor
////////////////////////////////////////////////////////////////////////

#include <string.h>
#include "SDL_endian.h"

#include "player.h"
#include "misc.h"

////////////////////////////////////////////////////////////////////////
// NAME       : Player::Player
// DESCRIPTION: Load player information from a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : -
//
// HISTORY
//   30-04-2001 - load new player password
////////////////////////////////////////////////////////////////////////

Player::Player( SDL_RWops *file ) {
  SDL_RWread( file, &p_id, sizeof(unsigned char), 1 );
  SDL_RWread( file, &p_type, sizeof(unsigned char), 1 );
  SDL_RWread( file, &p_level, sizeof(unsigned char), 1 );
  SDL_RWread( file, &p_success, sizeof(unsigned char), 1 );
  SDL_RWread( file, &p_briefing, sizeof(char), 1 );
  p_score = SDL_ReadLE16( file );
  SDL_RWread( file, &p_name, sizeof(char), 16 );
  SDL_RWread( file, &p_password, sizeof(char), 8 );
  crypt( p_password );

  p_mode = MODE_OFF;
  p_units = 0;
  if ( p_id == PLAYER_ONE ) {
    p_col_light = Color( P1_COLOR_LIGHT );
    p_col_dark = Color( P1_COLOR_SHADOW );
  } else {
    p_col_light = Color( P2_COLOR_LIGHT );
    p_col_dark = Color( P2_COLOR_SHADOW );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Player::Save
// DESCRIPTION: Save player information to a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, non-zero on error
//
// HISTORY
//   30-04-2001 - save player password
////////////////////////////////////////////////////////////////////////

int Player::Save( SDL_RWops *file ) const {
  char crypted[8];
  strcpy( crypted, p_password );
  crypt( crypted );

  SDL_RWwrite( file, &p_id, sizeof(unsigned char), 1 );
  SDL_RWwrite( file, &p_type, sizeof(unsigned char), 1 );
  SDL_RWwrite( file, &p_level, sizeof(unsigned char), 1 );
  SDL_RWwrite( file, &p_success, sizeof(unsigned char), 1 );
  SDL_RWwrite( file, &p_briefing, sizeof(char), 1 );
  SDL_WriteLE16( file, p_score );
  SDL_RWwrite( file, &p_name, sizeof(char), 16 );
  SDL_RWwrite( file, &crypted, sizeof(char), 8 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Player::Password
// DESCRIPTION: Get the player's password.
// PARAMETERS : -
// RETURNS    : pointer to password, or NULL if no password set
//
// HISTORY
////////////////////////////////////////////////////////////////////////

const char *Player::Password( void ) const {
  if ( p_password[0] ) return p_password;
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Player::SetPassword
// DESCRIPTION: Set the player's password.
// PARAMETERS : password - new player password; the function does not
//                         check whether it fits into the password
//                         buffer; may be NULL to erase the player's
//                         current password
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Player::SetPassword( const char *password ) {
  if ( password == NULL ) p_password[0] = '\0';
  else strcpy( p_password, password );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Player::Units
// DESCRIPTION: Increase or decrease the player's unit count.
// PARAMETERS : delta - change in number of units
// RETURNS    : number of units after change
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned short Player::Units( short delta ) {
  // in replay mode no changes are made
  if ( p_mode != MODE_REPLAY ) p_units += delta;
  return p_units;
}

