// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// rect.cpp
//
// History:
//  23-03-2000 - created
////////////////////////////////////////////////////////////////////////

#include "rect.h"

////////////////////////////////////////////////////////////////////////
// NAME       : Rect::Align
// DESCRIPTION: This function ensures that the given Rect is placed
//              inside the boundaries of its parent Rect. If the child
//              is larger than the parent, this will not work correctly,
//              of course.
// PARAMETERS : parent - the parent Rect
// RETURNS    : -
//
// HISTORY
//   03-06-2001 - fix for parents with left/top != 0
////////////////////////////////////////////////////////////////////////

void Rect::Align( const Rect &parent ) {
  if ( x < parent.x ) x = parent.x;
  else if ( x + w >= parent.x + parent.w ) x = parent.x + parent.w - w;

  if ( y < parent.y ) y = parent.y;
  else if ( y + h >= parent.y + parent.h ) y = parent.y + parent.h - h;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Rect::Center
// DESCRIPTION: Center the Rect in another rectangle.
// PARAMETERS : anchor - the rectangle to center the Rect in
// RETURNS    : -
//
// HISTORY
//   03-06-2001 - fix for anchors with left/top != 0
////////////////////////////////////////////////////////////////////////

void Rect::Center( const Rect &anchor ) {
  x = anchor.LeftEdge() + (anchor.Width() - w) / 2;
  y = anchor.TopEdge() + (anchor.Height() - h) / 2;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Rect::Clip
// DESCRIPTION: Cut away the parts of the Rect which are outside the
//              boundaries of the cliping rectangle.
// PARAMETERS : clip - clipping rectangle
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Rect::Clip( const Rect &clip ) {
  int delta;

  // horizontal clip
  if ( (x + w) < clip.x ) {
    x = clip.x;
    w = 0;
  } else if ( x > (clip.x + clip.w) ) {
    x = clip.x + clip.w;
    w = 0;
  } else if ( x < clip.x ) {
    delta = clip.x - x;
    x += delta;
    w -= delta;
  } else if ( (x + w) > (clip.x + clip.w) ) {
    delta = (x + w) - (clip.x + clip.w);
    w -= delta;
  }

  // vertical clip
  if ( (y + h) < clip.y ) {
    y = clip.y;
    h = 0;
  } else if ( y > (clip.y + clip.h) ) {
    y = clip.y + clip.h;
    h = 0;
  } else if ( y < clip.y ) {
    delta = clip.y - y;
    y += delta;
    h -= delta;
  } else if ( (y + h) > (clip.y + clip.h) ) {
    delta = (y + h) - (clip.y + clip.h);
    h -= delta;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Rect::ClipBlit
// DESCRIPTION: Do clipping to the Rect and at the same time adjust the
//              boundaries of a source Rect. This function is used for
//              clipping before blitting.
// PARAMETERS : src  - source rectangle
//              clip - clipping rectangle
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Rect::ClipBlit( Rect &src, const Rect &clip ) {
  int delta;

  if ( x < clip.x) {
    delta = clip.x - x;
    x += delta;
    src.x += delta;

    if ( delta >= w ) {
      w = 0;
      src.w = 0;
    } else {
      w -= delta;
      src.w -= delta;
    }
  }

  if ( (x + w) > (clip.x + clip.w) ) {
    delta = (x + w) - (clip.x + clip.w);

    if ( delta >= w ) {
      w = 0;
      src.w = 0;
    } else {
      w -= delta;
      src.w -= delta;
    }
  }

  if ( y < clip.y ) {
    delta = clip.y - y;
    y += delta;
    src.y += delta;

    if ( delta >= h ) {
      h = 0;
      src.h = 0;
    } else {
      h -= delta;
      src.h -= delta;
    }
  }

  if ( (y + h) > (clip.y + clip.h) ) {
    delta = (y + h) - (clip.y + clip.h);

    if ( delta >= h ) {
      h = 0;
      src.h = 0;
    } else {
      h -= delta;
      src.h -= delta;
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Rect::Contains
// DESCRIPTION: Check whether some coordinates lie inside the boundaries
//              of the Rect.
// PARAMETERS : x - horizontal coordinate
//              y - vertical coordinate
// RETURNS    : true if coordinates are contained within the Rect,
//              false otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Rect::Contains( short x, short y ) const {
  return( (x >= this->x) && (x < (this->x + w))
       && (y >= this->y) && (y < (this->y + h)) );
}

