/*
 * EveryBuddy 
 *
 * Copyright (C) 1999, Torrey Searle <tsearle@uci.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * service.h
 * Header file for service bits.
 */

#ifndef __SERVICE_H__
#define __SERVICE_H__
 
#include "account.h"
#include "chat_room.h"
#include "input_list.h"
#include <gtk/gtk.h>
#include <stdio.h>

#ifdef ENABLE_ZEPHYR
#  define NUM_SERVICES 7
#else
#  define NUM_SERVICES 6
#endif

#define GET_SERVICE(x) eb_services[x->service_id]
#define RUN_SERVICE(x) GET_SERVICE(x).sc

enum {
	AIM_SERVICE_ID = 0x00,
	ICQ_SERVICE_ID = 0x01,
	YAHOO_SERVICE_ID = 0x02,
	MSN_SERVICE_ID = 0x03,
	JABBER_SERVICE_ID = 0x04,
#ifdef ENABLE_ZEPHYR
  ZEPHYR_SERVICE_ID = 0x05,
  IRC_SERVICE_ID = 0x06
#else
  IRC_SERVICE_ID = 0x05
#endif
};


enum {
	AIM_PROTOCOL_ID = 0x00,
	ICQ_PROTOCOL_ID = 0x01,
	YAHOO_PROTOCOL_ID = 0x02,
	MSN_PROTOCOL_ID = 0x03,
	JABBER_PROTOCOL_ID = 0x04,
#ifdef ENABLE_ZEPHYR
  ZEPHYR_PROTOCOL_ID = 0x05,
  IRC_PROTOCOL_ID = 0x06
#else
  IRC_PROTOCOL_ID = 0x05
#endif
};


struct service_callbacks {
	/*callback to determine if remote account is online*/
	gboolean (*query_connected) (eb_account *account);
	
	/*callback that will establish connection to server, and set up
	  any timers and listeners necessary */
	void (*login) (eb_local_account *account);
	
	/*callback to disable any timers and disconnect from server*/
	void (*logout) (eb_local_account *account);
	
	/* send message from one account to another */
	void (*send_im) (eb_local_account *account_from,
			 eb_account *account_to,
			 gchar *message);
	
	/* reads local account information from a file */
	eb_local_account * (*read_local_account_config) (GList * values);

	/* gets the configuration necessary to be written to a file */
	GList * (*write_local_config)( eb_local_account * account );
	
	/* reads contact information */
	eb_account * (*read_account_config) (GList * config, struct contact *contact);
	
	/*Returns the list of all possible states that the user can be in*/
	GList * (*get_states)();
	
	/*Retruns the index to the current state that the service is int*/
	gint (*get_current_state)(eb_local_account * account);
	
	/*Sets the online status for the given account*/
	void (*set_current_state)(eb_local_account * account, gint state);
	
	/*Informs the service of an account that it needs to manage*/
	void (*add_user)(eb_account * account);
	
	/*Notifies the service that it doesn't need to track an account*/
	void (*del_user)(eb_account * account);
	
	/*Informs the service of an account that it needs to manage*/
	gboolean (*is_suitable)(eb_local_account *local, eb_account *remote);

	/*Creates a new account*/
	eb_account*(*new_account)( gchar * account );
	
	/*This returns the string representing the status,
	  this will get used on the Contact List, if statuses
	  are not available, this should return the empty string */

	gchar*(*get_status_string)(eb_account * account);
	
	/*This returns the string representing the status,
	  this will get used on the Contact List, if statuses
	  are not available, this should return the empty string */

	void (*get_status_pixmap)(eb_account * account, GdkPixmap **pm, GdkBitmap **bm);
	
	/*set the idle time (set this to null if N/A)*/
	
	void (*set_idle)(eb_local_account * account, gint idle );

	/* set an away message */
	
	void (*set_away)(eb_local_account * account, gchar * message );

	/*send a message to a chat room*/

	void (*send_chat_room_message)(eb_chat_room * room, gchar * message);

	/*these are used to join and leave a chat room */

	void (*join_chat_room)(eb_chat_room * room);
	void (*leave_chat_room)(eb_chat_room * room);

	/*this it to create a new chat room*/

	eb_chat_room * (*make_chat_room)(gchar * name, eb_local_account * account);

	/*this is to invite somebody into the chat room*/

	void (*send_invite)( eb_local_account * account, eb_chat_room * room,
				                         char * user, char * message);

	void (*accept_invite)( eb_local_account * account, void * invitation );

	void (*decline_invite)( eb_local_account * account, void * inviatation );

	/*this is to send a file */
	
	void (*send_file)( eb_local_account * from, eb_account * to, char * file );

	/* gets called for each account a contact has when a chat windows is closed
	 * can be null if you don't need it
	 */

	void (*terminate_chat)(eb_account * account );

        /* called to request a users information */	
        void(*get_info)(eb_local_account * account_from, eb_account * account_to);			


	/* this is used so that the 'plugin' can add it's own proprietary
	 * config featurs into the prefs window
	 */

        input_list * (*get_prefs)();			

	/* 
	 * processes any service specific preferences
	 */

	void (*read_prefs_config) (GList * values);

	/*
	 * get the service specific preferences so they can be written
	 * to a file
	 */

	GList * (*write_prefs_config)();

};

/*for every server you have the following: the name of the service,
  the server, port, the type of protocall used by the service, and a pointer
  to the service specific callbacks. 
  */
struct service {
	gchar *name;
	gint protocol_id;
	gboolean offline_messaging;
	gboolean group_chat;
	gboolean file_transfer;
	struct service_callbacks *sc;
};


gint get_service_id( gchar * servicename );
gchar* get_service_name( gint serviceid );

void services_init();

extern struct service eb_services[];

void add_idle_check();
void serv_touch_idle();
	
void rename_nick_log(char *oldnick, char *newnick);

GList * get_service_list();


#endif /*__SERVICE_H__*/
