/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "ferite.h"
#include "triton.h"

FeriteModule  *__ferite_root_module = NULL;
FeriteModule  *__ferite_current_module = NULL;
char         **__ferite_module_list = NULL;
int            __ferite_num_modules;
int            __ferite_modules_availible = 0;

extern int __ferite_compile_error;

FeriteModule *__ferite_create_module( char *name, char *filename )
{
   FeriteModule *ptr;
   FE_ENTER_FUNCTION;
   ptr = fmalloc( sizeof(FeriteModule) );
   ptr->name = fstrdup( name );
   ptr->filename = fstrdup( filename );
   ptr->handle = NULL;
   ptr->module_init = NULL;
   ptr->module_deinit = NULL;
   ptr->next = NULL;
   FE_LEAVE_FUNCTION( ptr );
}

void __ferite_init_module_list()
{
   int i;

   FE_ENTER_FUNCTION;
   __ferite_modules_availible++;
   /* initialise */
   FUD(( "MODULE LOADER: initialising module loader... " ));
   /* if( lt_dlinit() != 0 ) */
   if( triton_init() != 0 )
   {
      printf( "Module error.\n");
   }
   else
   {
      __ferite_module_list = __ferite_list_modules( &__ferite_num_modules );

      FUD(( "MODULE LOADER: Found %d module(s)\n", __ferite_num_modules ));
      for( i = 0; i < __ferite_num_modules; i++ )
		FUD(( "MODULE LOADER:  `-> \"%s\"\n", __ferite_module_list[i] ));

      __ferite_root_module = __ferite_create_module( "__ferite_root_module", "" );
      __ferite_current_module = __ferite_root_module;
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_destroy_module_list( FeriteModule *mod )
{
   FE_ENTER_FUNCTION;
   if( mod )
   {
      if( mod->next != NULL )
         __ferite_destroy_module_list( mod->next );

      FUD(( "MODULE LOADER: deleting module \"%s\"\n", mod->name ));

      ffree( mod->name );
      ffree( mod->filename );
      mod->module_init = NULL;

      if( mod->module_deinit != NULL )
         (mod->module_deinit)();

      mod->module_deinit = NULL;
      ffree( mod );
      mod = NULL;
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_deinit_module_list()
{
   int i;

   FE_ENTER_FUNCTION;

   if( __ferite_modules_availible )
   {
      if( __ferite_num_modules > 0 )
      {
         FUD(( "MODULE LOADER: destroying module file list\n" ));
         for( i = 0; i < __ferite_num_modules; i++ )
         {
            ffree( __ferite_module_list[i] );
         }
         ffree( __ferite_module_list );
      }
      FUD(( "MODULE LOADER: closing all opened modules\n" ));
      __ferite_destroy_module_list( __ferite_root_module );
      __ferite_root_module = NULL;

      FUD(( "MODULE LOADER: closing down module loader\n" ));
      __ferite_modules_availible = 0;
   }
   else
      ffree( __ferite_module_list );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_load_module( FeriteScript *script, FeriteNamespace *ns, char *name )
{
   FeriteModule *ptr = NULL;
   int i = 0, offset = 0;
   char buf[1024];

   FE_ENTER_FUNCTION;

   if( __ferite_modules_availible )
   {
      __ferite_lowercase( name );
      FUD(( "MODULE LOADER: Trying to locate module \"%s\"\n", name ));
      for( ptr = __ferite_root_module; ptr != NULL; ptr = ptr->next )
      {
         FUD(("MODULE LOADER: Checking \"%s\" and \"%s\"\n", name, ptr->name ));
         if( strcmp( name, ptr->name ) == 0 )
         {
            FUD(( "MODULE LOADER: Module \"%s\" is allready loaded\n", name ));
            FUD(( "MODULE LOADER: Calling init on module\n" ));
            (ptr->module_init)( script, ns );
            FE_LEAVE_FUNCTION( NOWT );
         }
      }
      FUD(( "MODULE LOADER: Module \"%s\" is not already loaded, will attempt to locate file, and load\n", name ));
      if( __ferite_num_modules > 0 )
      {
         for( i = 0; i < __ferite_num_modules; i++ )
         {
            offset = 0;
            /* get the module name (eg /blah/blim/console) */
            for( offset = strlen( __ferite_module_list[i] ) - 1; offset >= 0; offset-- )
            {
               if( __ferite_module_list[i][offset] == DIRDLIM )
               {
                  offset++;
                  break;
               }
            }
            FUD(( "MODULE LOADER: Comparing module name %s and %s\n", name, __ferite_module_list[i]+offset ));
            if( strcmp( __ferite_module_list[i]+offset, name ) == 0 )
            {
               __ferite_current_module->next = __ferite_create_module( name, __ferite_module_list[i] );
               FUD(( "MODULE LOADER: Loading module \"%s\" from file \"%s\"\n", name, __ferite_current_module->filename ));
               __ferite_current_module = __ferite_current_module->next;
               __ferite_current_module->handle = triton_openext( __ferite_current_module->filename );
			   /* __ferite_current_module->handle = lt_dlopenext( __ferite_current_module->filename );*/
			   if( __ferite_current_module->handle != NULL )
			   {
				  sprintf( buf, "%s_init\0", name );
				  /* __ferite_current_module->module_init = (void (*)(FeriteScript*,FeriteNamespace*))lt_dlsym( __ferite_current_module->handle, buf ); */
				  __ferite_current_module->module_init = (void (*)(FeriteScript*,FeriteNamespace*))triton_getsym( __ferite_current_module->handle, buf );
				  sprintf( buf, "%s_deinit\0", name );
				  /* __ferite_current_module->module_deinit = (void (*)())lt_dlsym( __ferite_current_module->handle, buf ); */
				  __ferite_current_module->module_deinit = (void (*)())triton_getsym( __ferite_current_module->handle, buf );
				  FUD(( "MODULE LOADER: Calling %s::module_init()\n", name ));
				  (__ferite_current_module->module_init)( script, ns );
			   }
			   else
				 printf( "Error: %s : %s\n", triton_error(),  __ferite_module_list[i] );
			   FE_LEAVE_FUNCTION( NOWT );
			}
		 }
      }
      __ferite_compile_error = 1;
   }
   FE_LEAVE_FUNCTION( NOWT );
}

char **__ferite_list_modules( int *num_ret )
{
   char **list = NULL, **tmp_file_list, s[4096], *home, **retv;
   int num = 0, i, pi = 0;

   FE_ENTER_FUNCTION;
   *num_ret = 0;
   /* get the user's home dir */
   home = __ferite_user_home_dir( getuid() );
   sprintf(s, "%s/.ferite/modules", home);
   FUD(( "MODULE LOADER: Searching \"%s\" for modules\n", s ));
   /* list the dir contents of their loader dir */
   tmp_file_list = __ferite_file_dir( s, &num );
   /* if theres files */
   if (num > 0)
   {
      /* make a list of them */
      *num_ret += num;
	  list = fmalloc(sizeof(char *) * *num_ret);
	  for (i = 0; i < num; i++)
	  {
		 sprintf(s, "%s/.ferite/modules/%s", home, tmp_file_list[i]);
		 list[i] = fstrdup(s);
	  }
      pi = i;
      __ferite_file_dir_free(tmp_file_list, num);
   }
   else
   {
      if( tmp_file_list )
		ffree( tmp_file_list );
   }
   ffree(home);
   /* same for system loader path */
   sprintf( s, MODULE_DIR );
   FUD(( "MODULE LOADER: Searching \"%s\" for modules\n", s ));
   tmp_file_list = __ferite_file_dir( s, &num );
   if (num > 0)
   {
      *num_ret += num;
      list = frealloc(list, sizeof(char *) * *num_ret);
      for (i = 0; i < num; i++)
      {
		 for (i = 0; i < num; i++)
		 {
			sprintf(s, MODULE_DIR "/%s", tmp_file_list[i]);
			list[pi + i] = fstrdup(s);
		 }
      }
      __ferite_file_dir_free(tmp_file_list, num);
   }
   else
   {
      if( tmp_file_list )
		ffree( tmp_file_list );
   }

   /* List currently contains *everything in there* we need to weed out
    * the .so, .la, .a versions of the same loader or whatever else.
    * lt_dlopen can take an extension-less name and do the Right Thing
    * with it, so that's what we'll give it. */
   retv = __ferite_trim_module_list(list, num_ret);

   FE_LEAVE_FUNCTION( retv );
}

char **__ferite_trim_module_list( char **list, int *num )
{
   int i, n, size = 0;
   char **ret = NULL;

   FE_ENTER_FUNCTION;

   n = *num;
   if( list == NULL )
   {
      FE_LEAVE_FUNCTION( NULL );
   }
   if( n < 1 )
   {
      FE_LEAVE_FUNCTION( list );
   }

   for (i = 0; i < n; i++)
   {
      char *ext;

      if (!list[i])
		continue;
      ext = strrchr(list[i], '.');
      if (ext)
      {
		 *ext = '\0';
	 /* Don't add the same loader multiple times... */
		 if (!__ferite_item_in_list(ret, size, list[i]))
		 {
			ret = frealloc(ret, sizeof(char *) * (size + 1));
			ret[size++] = fstrdup(list[i]);
		 }
      }
      if( list[i] )
		ffree( list[i] );
   }
   if( list != NULL )
     ffree(list);

   *num = size;

   FE_LEAVE_FUNCTION( ret );
}
