package Lire::Plugin;

use strict;

use Carp;

=pod

=head1 NAME

Lire::Plugin - Base interface of all Lire plugins.

=head1 SYNOPSIS

  use base qw/ Lire::Plugin /;

=head1 DESCRIPTION

This package defines the base interface implemented by all plugins.

=head1 DEFAULT CONSTRUCTOR

The plugin class has new() constructor which blesses an hash
reference.

=cut

sub new {
    return bless {}, shift;
}

=pod

=head2 name()

Returns the name of the plugin.

=cut

sub name {
    croak "name() not implemented by ", ref $_[0] || $_[0];
}

=pod

=head2 title()

Returns a more human friendly name for the plugin.

=cut

sub title {
    croak "title() not implemented by ", ref $_[0] || $_[0];
}

=pod

=head2 description()

Returns a DocBook XML based documentation of the plugin.

=cut

sub description {
    croak "description() not implemented by ", ref $_[0] || $_[0];
}

=pod

=head2 type

Returns the type of plugin. There are three plugin types in Lire:

=over

=item dlf_converter

These plugins are used to converter log files to DLF. Plugin will
also need to implement the Lire::DlfConverter interface.

=item output_format

These plugins are used to format XML reports. Plugin of this type
should implement the Lire::OutputFormat interface.

=item dlf_analyser

These plugins are used to generate other DLF records from existing
Dlf records. They should implement the Lire::Analaser interface.

=item chart_type

These plugins are used to generate charts.

=cut

sub type {
    croak "type() not implemented by ", ref $_[0] || $_[0];
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::PluginManager(3pm), Lire::DlfConverter(3pm),
Lire::OutputFormat(3pm), Lire::DlfAnalyser(3pm), Lire::DlfChartType(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: Plugin.pm,v 1.5 2006/07/23 13:16:29 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut

