/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include <stdlib.h>
#include <string>
#include "include/support.h"
#include "include/variable.h"
#include "include/environment.h"

#define MAKE_TWIN(v) LWValue *twin = (FALSE == (v)->isSpecial()) ? v : (v)->get(); \
					 const gboolean needRelease = (TRUE == v->isSpecial());

#define FREE_TWIN if (needRelease == TRUE) delete twin;

#define MAKE_TWIN2(v) LWValue *twin2 = (FALSE == (v)->isSpecial()) ? v : (v)->get(); \
					  const gboolean needRelease2 = (TRUE == (v)->isSpecial());

#define FREE_TWINS  if (needRelease == TRUE) delete twin; \
					if (needRelease2 == TRUE) delete twin2;

static int pixmapToInt (LWPixmap *pixmap)
{
	g_return_val_if_fail (pixmap != NULL, 0);

	gchar *s = pixmap->getName ();
	
	if (!strcmp(s, "true"))
		return 1;
	
	if (!strcmp (s, "false"))
		return 0;
	
	return atoi (s);
}

static int _isInt (LWPixmap *pixmap)
{
	g_return_val_if_fail (pixmap != NULL, 0);
	
	gchar *s = pixmap->getName ();
	return (s[0] >= '0' && s[0] <= '9') ? TRUE : FALSE;
}

static LWPixmap *intToPixmap (gint n)
{
	g_return_val_if_fail (NULL != LWEnvironment::getPixmapSet(), NULL);
	g_return_val_if_fail (n >= 0, NULL);
	
	gchar *s = g_strdup_printf ("%i", n);
	LWPixmap *pixmap = LWEnvironment::getPixmapSet()->getPixmap(s);
	g_free (s);
	return pixmap;
}

static LWPixmap *getMinusSign ()
{
	g_return_val_if_fail (NULL != LWEnvironment::getPixmapSet(), NULL);
	return LWEnvironment::getPixmapSet()->getPixmap("-");
}

void LWValue::convertToList ()
{
	g_return_if_fail (TRUE == canBeIndexed());
	
	if (is_list == FALSE) {
		list = g_slist_append (NULL, new LWValue (pixmap));
		is_list = TRUE;
	}
}

	
/*! \brief Konstruktor pustej wartości

	Tworzy pustą wartość
*/
LWValue::LWValue () : is_list(TRUE), list(NULL)
{
}

/*! \brief Konstruktor wartości z ikoną

	Tworzy wartość zawierającą pojedynczą ikonę
*/
LWValue::LWValue (LWPixmap *a_pixmap): is_list(FALSE), pixmap(a_pixmap)
{
}

/*! \brief Konstruktor kopiujący

	Konstruktor kopiujący wartości

	\param value Wartość na podstawie której ma powstać duplikat
*/
LWValue::LWValue (const LWValue *value): is_list(TRUE), list(NULL)
{
	set ( (LWValue *) value);
}

/*! \brief Konstruktor wartości na podstawie liczby

	Tworzy wartość zawierającą listę ikon będącą odpowiednikiem liczby
	\param ivalue Liczba 
*/
LWValue::LWValue (gint ivalue) : is_list(TRUE), list(NULL)
{
	gboolean hasminus = (ivalue < 0) ? TRUE : FALSE;
	
	if (hasminus == TRUE)
		ivalue = - ivalue;
	
	do {
		LWPixmap *pixmap = intToPixmap(ivalue % 10);
		list = g_slist_prepend (list, (gpointer) new LWValue(pixmap));
		ivalue /= 10;
	} while (ivalue > 0);
	
	if (hasminus == TRUE) {
		LWPixmap *pixmap = getMinusSign ();
		list = g_slist_prepend (list, (gpointer) new LWValue(pixmap));
	}
}

/*! \brief Destruktor

	Niszczy wartość
*/	
LWValue::~LWValue ()
{
	clear();
}

/*! \brief Czyszczenie wartości

	Czyści wartość niszcząc przy tym wszystkie ikony wchodzące w jej skład
*/	
void LWValue::clear ()
{
	if (TRUE == isSpecial()) return;

	if (is_list == TRUE) {
		
		for (GSList *l=list; l != NULL; l = l->next) {
			LWValue *value = (LWValue *) l->data;
			delete value;
		}
		
		g_slist_free (list);
	}
	
	is_list = TRUE;
	list = NULL;
}

/*! \brief Dołączenie ikony

	Dołącza ikonę do końca listy przechywowanej w wartości
	\param pixmap Ikona
*/
void LWValue::append (LWPixmap *pixmap)
{
	g_return_if_fail (FALSE == isSpecial());
	
	if (TRUE == canBeIndexed ())
		convertToList();
	
	if (is_list == TRUE)
		list = g_slist_append (list, 
			(TRUE == isVariable()) ? 
				new LWVariable(pixmap) : new LWValue(pixmap));
}

/*! \brief Łączenie wartości

	Dołącza listę przechywaną w value na koniec listy przechywowanej w wartości
	\param value Warość dołączana
*/
void LWValue::concat (LWValue *value)
{
	g_return_if_fail (value != NULL);
	MAKE_TWIN (value)
	
	if (TRUE == canBeIndexed ())
		convertToList();
	
	if (is_list == FALSE) 
		return;

	if (twin->is_list == FALSE)
		list = g_slist_append (list, (TRUE == isVariable()) ?
			new LWVariable(twin->pixmap) : new LWValue(twin->pixmap));
	else	
	for (GSList *l= twin->list; l != NULL; l = l->next) {
		LWValue *v = (LWValue *) l->data;
		
		if (TRUE == isVariable())
			list = (v->is_list == TRUE) ? 
				g_slist_append (list, new LWValue (v->pixmap)) :
					g_slist_append (list, new LWValue (v));
		else
			list = (v->is_list == TRUE) ? 
				g_slist_append (list, new LWVariable (v->pixmap)) :
					g_slist_append (list, new LWVariable (v));
	}
		
	FREE_TWIN
}

/*! \brief Interpretacja jako liczba

	Interpretuje wartość jako liczbę
	\return Liczba
*/
gint LWValue::toInt ()
{
	MAKE_TWIN(this)
	
	if (twin->is_list == FALSE)
		return (twin->pixmap != NULL) ? pixmapToInt (twin->pixmap) : 0;
			
	gboolean isminus = FALSE;
	gint ivalue = 0;
	GSList *l;
	
	for (l = twin->list; l != NULL; l = l->next) {
		LWValue *v = (LWValue *) l->data;
		if (v->is_list == TRUE) return 0;
		if (v->pixmap == NULL) continue;
		if (getMinusSign () == v->pixmap) isminus = TRUE; 
		else if (pixmapToInt( (LWPixmap *) v->pixmap) > 0) break;
		}
		
	for (; l != NULL; l = l->next) {
		LWValue *v = (LWValue *) l->data;
		if (v->is_list == TRUE || v->pixmap == NULL || FALSE == _isInt(v->pixmap)) break;
		ivalue *= 10; 
		ivalue += pixmapToInt( (LWPixmap *) v->pixmap);
	}

	FREE_TWIN
	
	return (isminus == FALSE) ? ivalue : -ivalue;
}

/*! \brief Czy wartość indeksowana

	Czy wartość może być indeksowana,
	zwraca prawdę. Metoda może być przeciążona
	w implementacji specjalnych wartości 
	\return Czy wartość może być indeksowana
*/
gboolean LWValue::canBeIndexed ()
{
	return TRUE;
}

/*! \brief Pobierz indeks

	Pobiera wartość znajdującą sie pod indeksem
	\param n Indeks do pobrania (liczony od 1)
	\return Pobrany indeks
*/
LWValue *LWValue::getIndex (guint n)
{
	g_return_val_if_fail (n > 0, new LWValue());
	g_return_val_if_fail (TRUE == canBeIndexed(), new LWValue());
	
	if (is_list == FALSE) 
		return (n == 1) ? new LWValue (pixmap) : new LWValue (); //this);
		
	LWValue *value = (LWValue *) g_slist_nth_data (list, n-1);
	return (value != NULL) ? new LWValue (value) : new LWValue();
}

void LWValue::deleteIndex (guint n)
{
	g_return_if_fail (n > 0);
	g_return_if_fail (TRUE == canBeIndexed());
	g_return_if_fail (FALSE == isSpecial());
		
	if (is_list == FALSE) {
		convertToList ();
	}
		
	GSList *l = g_slist_nth (list, n-1);
	
	if (l != NULL) {
		if (l->data)
			delete (LWValue *) l->data;
		
		list = g_slist_delete_link (list, l);
	}
}

/*! \brief Pobiera listę ikon

	Pobiera listę ikon znajdującą się w wartości
	\return Pobrana lista
*/
GSList *LWValue::getListPixmap ()
{
	MAKE_TWIN(this)
	
	if (twin->is_list == FALSE) {
		return g_slist_append (NULL, twin->pixmap);
	}
	
	GSList *list_pixmap=NULL;
	
	for (GSList *l = twin->list; l != NULL; l = l->next) {
		LWValue *v = (LWValue *) l->data;
		if (v->is_list == TRUE) break;
		list_pixmap = g_slist_append (list_pixmap, v->pixmap);
	}
	
	FREE_TWIN
	
	return list_pixmap;	
}


/*! \brief Pobranie piksmapy

	Pobranie piksmapy zawartej w wartości
	Wartość nie może być jest listą, lecz pojedynczą piksamapą,
	ewentualnie wartość może przechowywać listę jednoelementową
	\return Pobrana piksamapa
*/
LWPixmap *LWValue::getPixmap ()
{
LWPixmap *p=NULL;
	
	MAKE_TWIN(this)
	
	if (FALSE == twin->is_list) {
		p = twin->pixmap;
	} else
	if (twin->list != NULL)
		if (twin->list->next == NULL) {
			LWValue *v = (LWValue *) twin->list->data;
			if (FALSE == v->is_list)
				p = v->pixmap;
		}

	FREE_TWIN
	return p;
}

/*! \brief Set the value 

	Set the value basing on another value
	\param value The value that should be used for assigment
*/
void LWValue::set (LWValue *value)
{
	g_return_if_fail (value != NULL);
	
	if (this == value) return;
	
	MAKE_TWIN(value)
	
	if (FALSE == twin->is_list) {
		pixmap = twin->pixmap;
		is_list = FALSE;
		FREE_TWIN
		return;
	}

	if (twin->list != NULL)
		if (twin->list->next == NULL) {
			LWValue *v = (LWValue *) twin->list->data;
			if (FALSE == v->is_list) {
				pixmap = v->pixmap;
				is_list = FALSE;
				FREE_TWIN
				return;
			}
		}
				
	clear();
	
	for (GSList *l = twin->list; l != NULL; l = l->next) {
		LWValue *v = (LWValue *) l->data;
		list = g_slist_append (list, 
			(gpointer) ((TRUE == isVariable()) ?
				new LWVariable(v) : new LWValue (v)));
	}
	
	FREE_TWIN
}

/*! \brief Get the copy of the value

	It gets the copy of the value
	\return Copy of a value
*/
LWValue *LWValue::get()
{
	return new LWValue(this);
}

/*! \brief Counting elements in a list
	
	Counts elements in a list, if value is a pixmap it returns 1.
	\return Number of elements in a list
*/
guint LWValue::count ()
{
	MAKE_TWIN(this)
	
	guint count = (twin->is_list == TRUE) ? g_slist_length (twin->list) : 1;
	
	FREE_TWIN
	
	return count;
}

/*! \brief Checking if value is a integer
	
	Checks if value is a integer (string with only digits and optional minus sign at first position).
	\return Result of of test.
*/
gboolean LWValue::isInt ()
{
	MAKE_TWIN(this)
	
	if (twin->is_list == FALSE) {
		gboolean r = (twin->pixmap != NULL)	? _isInt (twin->pixmap) : FALSE;
		FREE_TWIN
		return r;
	}
			
	GSList *l = twin->list;
		
	if (l == NULL) {
		FREE_TWIN
		return FALSE;
	}
				
	LWValue *v = (LWValue *) l->data;
	
	if (v->is_list == FALSE)	
		if (getMinusSign () == v->pixmap)
			l = l->next;
		
	for (; l != NULL; l = l->next) {
		v = (LWValue *) l->data;
		
		if (v->is_list == TRUE || v->pixmap == NULL || _isInt((LWPixmap *) v->pixmap) == FALSE) {
			FREE_TWIN	
			return FALSE;
		}
	}

	FREE_TWIN
	
	return TRUE;
}

/*! \brief Checking if the value fits in the integer range
	
	Checks if the value can be converted to "gint" without an overflow.
	\return True if can fit in integer range
*/
gboolean LWValue::checkRange ()
{
	MAKE_TWIN(this)
	
	if (twin->is_list == FALSE) {
		FREE_TWIN 
		return TRUE;
	}

	GSList *l = twin->list;

	LWValue *v = (LWValue *) l->data;
	
	if (v->is_list == TRUE) {
		FREE_TWIN 
		return TRUE;
	}

	guint limit;

	if (getMinusSign () == v->pixmap) {
		gint min = G_MININT;
		limit = (guint) -min;
		l = l->next;
	} else {
		limit = (guint) G_MAXINT;
	}
	
	for (; l != NULL && limit > 0; l = l->next, limit /= 10) {
		v = (LWValue *) l->data;
				
		if (v->is_list == TRUE || v->pixmap == NULL || _isInt((LWPixmap *) v->pixmap) == FALSE) {
			FREE_TWIN	
			return TRUE;
		}
			
		if (limit < 10 && limit < (guint) pixmapToInt((LWPixmap *) v->pixmap)) {
			FREE_TWIN	
			return FALSE;
		}
	}

	FREE_TWIN

	return (limit > 0) ? TRUE : FALSE;
}

/*! \brief Comparation of the values
	
	The smart way of comparing of the values. It can be used to numerical or
	lexigraphical sorting depends on the contet of values.
	It can be used for comparing single values or whole arrays.

	\param value The compared value
	\return -1, 0 or 1 depends on the result of comparation.
*/
gint LWValue::compare (LWValue *value)
{
	if (this == value) return 0;
	
	MAKE_TWIN(this)
	MAKE_TWIN2(value)
	
	if (twin->isInt() == TRUE && twin2->isInt() == TRUE) {
		gint i1 = twin->toInt();
		gint i2 = twin2->toInt();
		FREE_TWINS
		return (i1 == i2) ? 0 : ((i1 < i2) ? -1 : 1);
	}
	
	if (FALSE == twin->is_list && FALSE == twin2->is_list) {
		gchar *s1 = (twin->pixmap != NULL) ? twin->pixmap->getName() : (gchar *) " ";
		gchar *s2 = (twin2->pixmap != NULL) ? twin2->pixmap->getName() : (gchar *) " ";
		gint r = strcmp (s1, s2);
		FREE_TWINS
		return r;
	}
	
	if (TRUE == twin->is_list && FALSE == twin2->is_list) {
		gint r;
		
		if (twin->list != NULL) {
			LWValue *v = twin->getIndex(1);
			r = v->compare(twin2);
			if (r == 0) r = 1;
			if (!v->isVariable()) delete v;
		} else 
			r = -1;
			
		FREE_TWINS
		return r;
	}
	
	if (FALSE == twin->is_list && TRUE == twin2->is_list) {
		gint r;
	
		if (twin2->list != NULL) {
			LWValue *v = twin2->getIndex(1);
			r = twin->compare(v);
			if (r == 0) r = -1;
			if (!v->isVariable()) delete v;
		} else r = 1;
					
		FREE_TWINS
		return r;
	}
		
	GSList *l = twin->list;
	GSList *l2 = twin2->list;	

	for (; l != NULL && l2 != NULL; l = l->next, l2 = l2->next) {
		LWValue *t1 = (LWValue *) l->data;
		LWValue *t2 = (LWValue *) l2->data;
		
		guint r = t1->compare (t2);
		
		if (r == 0) continue;
			
		FREE_TWINS
		
		return r;
	}
	
	FREE_TWINS
	
	return (l == NULL && l2 == NULL) ? 0 : ((l != NULL) ? 1 : -1);
}

/*! \brief Is it a variable

	Is it a variable ?
	\return Not overwritten method returns always FALSE
*/
gboolean LWValue::isVariable ()
{
	return FALSE;
}

/*! \brief Is it a special value
	
	Is it a special value?
	Method overwritten in implementation of special values (f.i world variable)
	The "special" means that the content is not stored directly in a value,
	but the value class is only some kind of interface that behaves like an
	value, but the real content is stored somewhere else. For special value 
	at least "set" and "get" methods should work correctly, other methods
	internally should use copy generated by "get" method, 
	which is no longer special value.

	\return Not overwritten method returns always FALSE
*/
gboolean LWValue::isSpecial ()
{
	return FALSE;
}
