/*
 *  Merlin's Clock Applet
 *  - A GNOME panel applet that displays the time or date.
 *  Copyright (C) 1999 Merlin Hughes
 *  - merlin@merlin.org
 *  - http://nitric.com/freeware/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <time.h>
/* #include <config.h> */
#include <gnome.h>
#include <gdk/gdkx.h>

#include <applet-widget.h>

#include "merlin-clock.h"
#include "session.h"
#include "properties.h"

static void merlin_clock_set_timeout(MerlinClockData *mc);

int
main (int argc, char ** argv)
{
  const gchar *goad_id;
  GtkWidget *applet;

  /* Initialize i18n */
  bindtextdomain (PACKAGE, GNOMELOCALEDIR);
  textdomain (PACKAGE);

  applet_widget_init ("merlin-clock_applet", VERSION, argc, argv, NULL, 0, NULL);
  applet_factory_new ("merlin-clock_applet", NULL,
		     (AppletFactoryActivator) applet_start_new_applet);

  goad_id = goad_server_activation_id ();
  if (! goad_id)
    return 1;

  /* Create the merlin-clock applet widget */
  applet = make_new_merlin_clock_applet (goad_id);

  /* Run... */
  applet_widget_gtk_main ();

  return 0;
} /* main */

/*
 * This function, merlin_clock_update, gets the new time or date and updates the
 * pixmap.
 *
 */
gint
merlin_clock_update (gpointer data)
{
  MerlinClockData * mc = data;

  /* The time and date */
  time_t now_;
  struct tm *now;
  char text[9];
  gint strl, strr, strw, stra, strd;

  if (!mc->setup)
    return 0;

  time (&now_);
  now = localtime (&now_);

  if (mc->toggle && (now_ >= mc->toggle_t + MERLIN_CLOCK_TOGGLE_PERIOD))
    mc->toggle = FALSE;
  
  if (mc->datemode ^ mc->toggle) {
    if (mc->extended)
      sprintf (text, "%.2d/%.2d/%.2d", now->tm_year % 100, 1 + now->tm_mon, now->tm_mday);
    else
      sprintf (text, "%.2d/%.2d", 1 + now->tm_mon, now->tm_mday);
  } else {
    if (mc->extended)
      sprintf (text, "%.2d:%.2d:%.2d", now->tm_hour, now->tm_min, now->tm_sec);
    else
      sprintf (text, "%.2d:%.2d", now->tm_hour, now->tm_min);
  }

  /* Clear the graph pixmap */
  gdk_gc_set_foreground (mc->gc, & (mc->background));
  gdk_draw_rectangle (mc->pixmap, mc->gc, TRUE, 0, 0, mc->breadth, mc->depth);

  /* Draw the time/date */
  gdk_gc_set_foreground (mc->gc, & (mc->foreground));
  gdk_string_extents (mc->font, text, &strl, &strr, &strw, &stra, &strd);
  gdk_draw_string (mc->pixmap, mc->font, mc->gc, (mc->breadth - strw) / 2, stra, text);

  /* Verticalize it */
  if (mc->vertical) {
    GdkImage *grab;
    int i, j;

    grab = gdk_image_get (mc->pixmap, 0, 0, mc->breadth, mc->depth);
    for (i = 0; i < mc->breadth; ++ i) {
      for (j = 0; j < mc->depth; ++ j) {
        gint pixel;

        pixel = gdk_image_get_pixel (grab, i, j);
        gdk_image_put_pixel (mc->image, mc->depth - 1 - j, i, pixel);
      }
    }

    gdk_image_destroy (grab);
  }
  
  /* Update the display. */
  merlin_clock_expose_handler (mc->area, NULL, mc);

  mc->force_update = FALSE;

  merlin_clock_set_timeout (mc);

  return TRUE;
} /* merlin_clock_update */


/*
 * This function, merlin_clock_expose, is called whenever a portion of the
 * applet window has been exposed and needs to be redrawn.  In this
 * function, we just blit the appropriate portion of the pixmap onto the window.
 *
 */
gint
merlin_clock_expose_handler (GtkWidget * ignored, GdkEventExpose * expose,
			gpointer data)
{
  MerlinClockData * mc = data;

  if (!mc->setup)
    return FALSE;

  if (mc->vertical) {
    gdk_draw_image (mc->area->window, mc->area->style->fg_gc[GTK_WIDGET_STATE (mc->area)],
                    mc->image, 0, 0, 0, 0, mc->depth, mc->breadth);
  } else {
    gdk_draw_pixmap (mc->area->window, mc->area->style->fg_gc[GTK_WIDGET_STATE (mc->area)],
                     mc->pixmap, 0, 0, 0, 0, mc->breadth, mc->depth);
  }
  
  return FALSE; 
} /* merlin_clock_expose_handler */

/* This handler gets called whenever the panel changes orientations.
   When the applet is set up, we get an initial call, too. */
gint
merlin_clock_orient_handler (GtkWidget * w, PanelOrientType o, gpointer data)
{
  MerlinClockData * mc = data;
  gboolean vertical = (o == ORIENT_UP) || (o == ORIENT_DOWN);

  if (vertical != mc->vertical) {
    mc->vertical = vertical;
    
    merlin_clock_set_size (mc);
  }

  return FALSE;
} /* merlin_clock_orient_handler */

gint
merlin_clock_configure_handler (GtkWidget *widget, GdkEventConfigure *event,
			   gpointer data)
{
  MerlinClockData * mc = data;
  
  mc->force_update = TRUE;
  merlin_clock_update ( (gpointer) mc);

  return TRUE;
}  /* merlin_clock_configure_handler */

static void
merlin_clock_toggle (MerlinClockData * mc)
{
  mc->toggle = !mc->toggle;
  time (&(mc->toggle_t));

  merlin_clock_update (mc);
} /* merlin_clock_toggle */

static gint
merlin_clock_button_press_handler (GtkWidget * w, GdkEventButton * ev,
			      gpointer data)
{
  MerlinClockData * mc = data;

  merlin_clock_toggle (mc);

  return TRUE;
} /* merlin_clock_button_press_handler */

GtkWidget *
applet_start_new_applet (const gchar *goad_id, const char **params,
			 int nparams)
{
  return make_new_merlin_clock_applet (goad_id);
} /* applet_start_new_applet */

/* This is the function that actually creates the display widgets */
GtkWidget *
make_new_merlin_clock_applet (const gchar *goad_id)
{
  MerlinClockData * mc;
  gchar * param = "merlin-clock_applet";

  mc = g_new0 (MerlinClockData, 1);

  mc->applet = applet_widget_new (goad_id);

  if (mc->applet == NULL)
    g_error (_("Can't create applet!\n"));

  mc->setup = FALSE;
  mc->force_update = TRUE;

  /*
   * Load all the saved session parameters (or the defaults if none
   * exist).
   */
  if ( (APPLET_WIDGET (mc->applet)->privcfgpath) &&
       * (APPLET_WIDGET (mc->applet)->privcfgpath))
    merlin_clock_session_load (APPLET_WIDGET (mc->applet)->privcfgpath, mc);
  else
    merlin_clock_session_defaults (mc);

  /*
   * area is the drawing area into which the little picture of
   * the merlin-clock gets drawn.
   */
  mc->area = gtk_drawing_area_new ();
  gtk_widget_set_usize (mc->area, 1, 1); /* right now I don't know my orientation */

  /* Set up the mode-changing callback */
  gtk_signal_connect (GTK_OBJECT (mc->applet), "button_press_event",
		      (GtkSignalFunc) merlin_clock_button_press_handler, mc);
  gtk_widget_set_events (GTK_WIDGET (mc->applet), GDK_BUTTON_PRESS_MASK);


  /* Set up the event callbacks for the area. */
  gtk_signal_connect (GTK_OBJECT (mc->area), "expose_event",
		      (GtkSignalFunc)merlin_clock_expose_handler, mc);
  gtk_widget_set_events (mc->area, GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK);

  /* This will let us know when the panel changes orientation */
  gtk_signal_connect (GTK_OBJECT (mc->applet), "change_orient",
		      GTK_SIGNAL_FUNC (merlin_clock_orient_handler),
		      mc);

  applet_widget_add (APPLET_WIDGET (mc->applet), mc->area);

  gtk_signal_connect (GTK_OBJECT (mc->applet), "save_session",
		      GTK_SIGNAL_FUNC (merlin_clock_session_save),
		      mc);

  applet_widget_register_stock_callback (APPLET_WIDGET (mc->applet),
					 "about",
					 GNOME_STOCK_MENU_ABOUT,
					 _("About..."),
					 about_cb,
					 mc);

  applet_widget_register_stock_callback (APPLET_WIDGET (mc->applet),
					 "properties",
					 GNOME_STOCK_MENU_PROP,
					 ("Properties..."),
					 merlin_clock_properties_window,
					 mc);

  gtk_widget_show_all (mc->applet);

  /* Allocate the colors... */
  merlin_clock_create_gc (mc);
  merlin_clock_setup_colors (mc);

  /* Nothing is drawn until this is set. */
  mc->setup = TRUE;

  /* Size things according to the saved settings. */
  merlin_clock_set_size (mc);

  mc->force_update = TRUE;

  /* Will schedule a timeout automatically */
  merlin_clock_update (mc);

  return mc->applet;
} /* make_new_merlin_clock_applet */

static void merlin_clock_set_timeout (MerlinClockData *mc) { 
  gint when;
  
  if (mc->extended && !(mc->datemode ^ mc->toggle)) { /* seconds showing */
    when = 1000;
  } else if (mc->toggle) {  /* toggled */
    time_t now_, then_;
    
    time (&now_);
    then_ = mc->toggle_t + MERLIN_CLOCK_TOGGLE_PERIOD;
    
    when = (then_ > now_) ? (then_ - now_) * 1000 + 10 : 100;
  } else { /* days or minutes showing */
    time_t now_;
    struct tm *now;
    gboolean minutes = !(mc->datemode ^ mc->toggle);
    
    time (&now_);
    now = localtime (&now_);
    if (!now->tm_sec && (minutes || (!now->tm_min && !now->tm_hour))) { /* period 0 */
      when = minutes ? (60 * 1000) : (24 * 60 * 60 * 1000);
    } else {
      struct tm then = *now;
      time_t then_;

      if (minutes) {
        then.tm_sec = 0;
        ++ then.tm_min;
      } else {
        then.tm_sec = then.tm_min = then.tm_hour = 0;
        ++ then.tm_mday;
      }
      then_ = mktime (&then);

      /* I add 10ms to address toggle conditions that can
       * arise if things keep happening on the minute boundary */
      when = (then_ > now_) ? (then_ - now_) * 1000 + 10 : 100;
    }
  }

  if (when != mc->timeout_t) {
    if (mc->timeout_t)
      gtk_timeout_remove (mc->timeout);
    mc->timeout_t = when;
    mc->timeout = gtk_timeout_add (when, (GtkFunction) merlin_clock_update, mc);
  }
}

void
destroy_about (GtkWidget *w, gpointer data)
{
  MerlinClockData *mc = data;
} /* destroy_about */

void
about_cb (AppletWidget *widget, gpointer data)
{
  MerlinClockData *mc = data;
  char *authors[2];
  
  authors[0] = "Merlin Hughes <merlin@merlin.org>";
  authors[1] = NULL;

  mc->about_box =
    gnome_about_new (_("Merlin's Clock Applet"), VERSION,
		     _("Copyright (C) 1999 Merlin Hughes"),
		     (const char **) authors,
	     _("This applet displays the time or date in a compact manner.  "
               "Click on it to briefly toggle display mode.\n"
               "This applet comes with ABSOLUTELY NO WARRANTY.  "
               "See the LICENSE file for details.\n"
               "This is free software, and you are welcome to redistribute it "
               "under certain conditions.  "
               "See the LICENSE file for details.\n"
               "http://nitric.com/freeware/"),
		     NULL);

  gtk_signal_connect (GTK_OBJECT (mc->about_box), "destroy",
		      GTK_SIGNAL_FUNC (destroy_about), mc);

  gtk_widget_show (mc->about_box);
} /* about_cb */

void
merlin_clock_set_size (MerlinClockData * mc)
{
  if (mc->vertical)
    gtk_widget_set_usize (mc->area, mc->depth, mc->breadth);
  else
    gtk_widget_set_usize (mc->area, mc->breadth, mc->depth);

  /*
   * If pixmaps have already been allocated, then free them here
   * before creating new ones.  */
  if (mc->pixmap != NULL)
    gdk_pixmap_unref (mc->pixmap);

  mc->pixmap = gdk_pixmap_new (mc->area->window, mc->breadth, mc->depth,
                               gtk_widget_get_visual (mc->area)->depth);

  if (mc->image != NULL)
    gdk_image_destroy (mc->image);

  if (mc->vertical) {
    mc->image = gdk_image_new (GDK_IMAGE_SHARED, gtk_widget_get_visual (mc->area), mc->depth, mc->breadth);
  } else {
    mc->image = NULL;
  }
  
  mc->force_update = TRUE;

  merlin_clock_update ( (gpointer) mc);

} /* merlin_clock_set_size */

void
merlin_clock_create_gc (MerlinClockData * mc)
{
  mc->gc = gdk_gc_new (mc->area->window);
  gdk_gc_copy (mc->gc, mc->area->style->white_gc);

  gdk_gc_set_function (mc->gc, GDK_COPY);
} /* merlin_clock_create_gc */

void 
merlin_clock_setup_colors (MerlinClockData * mc)
{
  GdkColormap *colormap;
  gint strl, strr, strw, stra, strd;

  /*
   * FIXME: We should use gdk_color_change if we've already set up the
   * colors. And should I be freeings fonts? Who knows...
   */

  colormap = gtk_widget_get_colormap (mc->area);

  gdk_color_parse (mc->foreground_s, & (mc->foreground));
  gdk_color_alloc (colormap, & (mc->foreground));

  gdk_color_parse (mc->background_s, & (mc->background));
  gdk_color_alloc (colormap, & (mc->background));

  mc->font = gdk_font_load (mc->font_s);
  gdk_string_extents (mc->font, "0123456789:/", &strl, &strr, &strw, &stra, &strd);

  mc->depth = stra + strd;
} /* merlin_clock_setup_colors */
