//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: minstrument.cpp,v 1.2 2002/02/07 08:14:48 muse Exp $
//
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include "minstrument.h"
#include "midiport.h"
#include "progname.h"
#include <qpopupmenu.h>
#include "globals.h"
#include "progname.h"
#include "xml.h"
#include "event.h"

#include <stdio.h>

MidiInstrumentList midiInstruments;
MidiInstrument* genericMidiInstrument;

//---------------------------------------------------------
//   initMidiInstruments
//---------------------------------------------------------

void initMidiInstruments()
      {
      genericMidiInstrument = new MidiInstrument("generic midi");
      midiInstruments.push_back(genericMidiInstrument);
      midiInstruments.push_back(new MidiGMInstrument("generic GM"));
      midiInstruments.push_back(new MidiGSInstrument("generic GS"));
      midiInstruments.push_back(new MidiXGInstrument("generic XG"));
      midiInstruments.push_back(new MidiP50mInstrument("Yamaha P50m"));
      midiInstruments.push_back(new MidiNS5RInstrument("Korg NS5R"));
      }

//---------------------------------------------------------
//   registerMidiInstrument
//---------------------------------------------------------

MidiInstrument* registerMidiInstrument(const QString& name)
      {
      for (iMidiInstrument i = midiInstruments.begin();
         i != midiInstruments.end(); ++i) {
            if ((*i)->iname() == name)
                  return *i;
            }
      return genericMidiInstrument;
      }

//---------------------------------------------------------
//   removeMidiInstrument
//---------------------------------------------------------

void removeMidiInstrument(const QString& name)
      {
      for (iMidiInstrument i = midiInstruments.begin();
         i != midiInstruments.end(); ++i) {
            printf("remove Instrument %s\n", name.latin1());
            midiInstruments.erase(i);
            return;
            }
      }

      EventList* _midiInit;
      EventList* _midiState;

//---------------------------------------------------------
//   MidiInstrument
//---------------------------------------------------------

MidiInstrument::MidiInstrument()
      {
      _midiInit = new EventList();
      _midiState = new EventList();
      }

//---------------------------------------------------------
//   MidiInstrument
//---------------------------------------------------------

MidiInstrument::MidiInstrument(const QString& txt)
      {
      _name = txt;
      _midiInit = new EventList();
      _midiState = new EventList();
      }

//---------------------------------------------------------
//   MidiInstrument
//---------------------------------------------------------

MidiInstrument::~MidiInstrument()
      {
      delete _midiInit;
      delete _midiState;
      }

//---------------------------------------------------------
//   getPatchName
//---------------------------------------------------------

const char* MidiInstrument::getPatchName(int, int, int, int, MType)
      {
      return "???";
      }

//---------------------------------------------------------
//   reset
//---------------------------------------------------------

void MidiGMInstrument::reset(int port, MType)
      {
      midiPorts[port].gmOn();
      }

//---------------------------------------------------------
//   getPatchName
//---------------------------------------------------------

const char* MidiGMInstrument::getPatchName(int channel,
  int, int, int prog, MType)
      {
      return getPatch(channel, 0, 0, prog, MT_GM);
      }

void MidiGMInstrument::populatePatchPopup(QPopupMenu* menu, int, MType)
      {
      menu->clear();
      int mask = 1;
      for (int i = 0; i < PROG_GROUPS; ++i) {
            QPopupMenu* pm = new QPopupMenu(menu);
            pm->setFont(font0);
            const MidiPatch* mp = patchGroups[i];
            int k = 0;
            while (mp->name) {
                  if (mp->typ & mask) {
                        int id = 0xffff00 + mp->prog;
                        pm->insertItem(mp->name, id);
                        }
                  ++k;
                  ++mp;
                  }
            menu->insertItem(gmGroups[i], pm);
            }
      }

//---------------------------------------------------------
//   reset
//---------------------------------------------------------

void MidiGSInstrument::reset(int port, MType type)
      {
      if (type == MT_GM)
            midiPorts[port].gmOn();
      else
            midiPorts[port].gsOn();
      }

//---------------------------------------------------------
//   getPatchName
//---------------------------------------------------------

const char* MidiGSInstrument::getPatchName(int channel,
  int hbank, int lbank, int prog, MType)
      {
      return getPatch(channel, hbank, lbank, prog, MT_GS);
      }

void MidiGSInstrument::populatePatchPopup(QPopupMenu* menu, int, MType songType)
      {
      menu->clear();
      int mask = 0;
      switch(songType) {
            case MT_GS: mask = 2; break;
            case MT_GM:
            case MT_XG:
            case MT_UNKNOWN:  mask = 1; break;
            }
      for (int i = 0; i < PROG_GROUPS; ++i) {
            QPopupMenu* pm = new QPopupMenu(menu);
            pm->setFont(font0);
            const MidiPatch* mp = patchGroups[i];
            int k = 0;
            while (mp->name) {
                  if (mp->typ & mask) {
                        int id = (mp->hbank<<16) + (mp->lbank<<8) + mp->prog;
                        pm->insertItem(mp->name, id);
                        }
                  ++k;
                  ++mp;
                  }
            menu->insertItem(gmGroups[i], pm);
            }
      }

//---------------------------------------------------------
//   reset
//---------------------------------------------------------

void MidiXGInstrument::reset(int port, MType)
      {
      midiPorts[port].xgOn();
      }

//---------------------------------------------------------
//   getPatchName
//---------------------------------------------------------

const char* MidiXGInstrument::getPatchName(int channel,
  int hbank, int lbank, int prog, MType)
      {
      return getPatch(channel, hbank, lbank, prog, MT_XG);
      }

void MidiXGInstrument::populatePatchPopup(QPopupMenu* menu, int, MType songType)
      {
      menu->clear();
      int mask = 0;
      switch(songType) {
            case MT_XG: mask = 4; break;
            case MT_GS:
            case MT_GM:
            case MT_UNKNOWN:  mask = 1; break;
            }
      for (int i = 0; i < PROG_GROUPS; ++i) {
            QPopupMenu* pm = new QPopupMenu(menu);
            pm->setFont(font0);
            const MidiPatch* mp = patchGroups[i];
            int k = 0;
            while (mp->name) {
                  if (mp->typ & mask) {
                        int id = (mp->hbank<<16) + (mp->lbank<<8) + mp->prog;
                        pm->insertItem(mp->name, id);
                        }
                  ++k;
                  ++mp;
                  }
            menu->insertItem(gmGroups[i], pm);
            }
      }

ProgItem MidiP50mInstrument::progList[29] = {
      { "bright piano s+s", 0xffff00 },
      { "bright piano s",   0xffff01 },
      { "bright piano m+s", 0xffff02 },
      { "bright piano m",   0xffff03 },
      { "dark piano",       0xffff04 },
      { "grand piano s+s",  0xffff05 },
      { "grand piano s",    0xffff06 },
      { "grand piano m+s",  0xffff07 },
      { "grand piano m",    0xffff08 },
      { "dance",            0xffff09 },
      { "honkytonk",        0xffff0a },
      { "hybrid piano",     0xffff0b },
      { "cp80, mono",       0xffff0c },
      { "cp80, mono s",     0xffff0d },
      { "cp80+chorus",      0xffff0e },
      { "dx ep",            0xffff0f },
      { "dx pad",           0xffff10 },
      { "dx ep+chorus",     0xffff11 },
      { "roads",            0xffff12 },
      { "roads+chorus",     0xffff13 },
      { "soft roads",       0xffff14 },
      { "hard roads",       0xffff15 },
      { "dyno",             0xffff16 },
      { "resonant dyno",    0xffff17 },
      { "dyno+chorus",      0xffff18 },
      { "wurli",            0xffff19 },
      { "clavi",            0xffff1a },
      { "clavi+wah",        0xffff1b },
      { "no sound",         0xffff1c },
      };

//---------------------------------------------------------
//   reset
//---------------------------------------------------------

void MidiP50mInstrument::reset(int port, MType)
      {
      midiPorts[port].gmOn();
      }

//---------------------------------------------------------
//   getPatchName
//---------------------------------------------------------

const char* MidiP50mInstrument::getPatchName(int, int, int, int prog, MType)
      {
      int nprogs = sizeof(progList)/sizeof(*progList);
      if (prog > nprogs-1)
            return "???";
      return progList[prog].name;
      }

void MidiP50mInstrument::populatePatchPopup(QPopupMenu* menu, int, MType)
      {
      menu->clear();
      int nprogs = sizeof(progList)/sizeof(*progList);
      for (int i = 0; i < nprogs; ++i)
            menu->insertItem(progList[i].name, progList[i].id);
      }

//---------------------------------------------------------
//   reset
//    the Korg NS5R is GM/GS/XG compatible
//---------------------------------------------------------

void MidiNS5RInstrument::reset(int port, MType _mode)
      {
      mode = _mode;
      switch (mode) {
            case MT_UNKNOWN:
            case MT_XG:
                  midiPorts[port].xgOn();
                  break;
            case MT_GS:
                  midiPorts[port].gsOn();
                  break;
            case MT_GM:
                  midiPorts[port].gmOn();
                  break;
            }
      }

//---------------------------------------------------------
//   MidiNS5R::getPatchName
//---------------------------------------------------------

const char* MidiNS5RInstrument::getPatchName(int channel,
  int hbank, int lbank, int prog, MType mode)
      {
      return getPatch(channel, hbank, lbank, prog, mode);
      }

//---------------------------------------------------------
//   MidiNS5R::populatePatchPopup
//---------------------------------------------------------

void MidiNS5RInstrument::populatePatchPopup(QPopupMenu* menu, int, MType songType)
      {
      menu->clear();
      int mask = 0;
      switch(songType) {
            case MT_XG: mask = 4; break;
            case MT_GS: mask = 2; break;
            case MT_GM:
            case MT_UNKNOWN:  mask = 1; break;
            }
      for (int i = 0; i < PROG_GROUPS; ++i) {
            QPopupMenu* pm = new QPopupMenu(menu);
            pm->setFont(font0);
            const MidiPatch* mp = patchGroups[i];
            int k = 0;
            while (mp->name) {
                  if (mp->typ & mask) {
                        int id = (mp->hbank<<16) + (mp->lbank<<8) + mp->prog;
                        pm->insertItem(mp->name, id);
                        }
                  ++k;
                  ++mp;
                  }
            menu->insertItem(gmGroups[i], pm);
            }
      }

//---------------------------------------------------------
//   readMidiState
//---------------------------------------------------------

void MidiInstrument::readMidiState(Xml& xml)
      {
      for (;;) {
            Xml::Token token = xml.parse();
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (tag == "event") {
                              Event* e = new MidiEvent();
                              e->read(xml);
                              _midiState->add(e);
                              }
                        else
                              xml.unknown("readMidiState");
                        break;
                  case Xml::TagEnd:
                        if (tag == "midistate")
                              return;
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   readMidiInit
//---------------------------------------------------------

void MidiInstrument::readMidiInit(Xml& xml)
      {
      for (;;) {
            Xml::Token token = xml.parse();
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (tag == "event") {
                              Event* e = new MidiEvent();
                              e->read(xml);
                              _midiInit->add(e);
                              }
                        else
                              xml.unknown("readMidiInit");
                        break;
                  case Xml::TagEnd:
                        if (tag == "midiinit")
                              return;
                  default:
                        break;
                  }
            }
      }

