'encoding UTF-8  Do not remove or change this line!
'*************************************************************************
'*
'*  OpenOffice.org - a multi-platform office productivity suite
'*
'*  $RCSfile: filetools.inc,v $
'*
'*  $Revision: 1.45 $
'*
'*  last change: $Author: jsk $ $Date: 2006/07/25 07:51:32 $
'*
'*  The Contents of this file are made available subject to
'*  the terms of GNU Lesser General Public License Version 2.1.
'*
'*
'*    GNU Lesser General Public License Version 2.1
'*    =============================================
'*    Copyright 2005 by Sun Microsystems, Inc.
'*    901 San Antonio Road, Palo Alto, CA 94303, USA
'*
'*    This library is free software; you can redistribute it and/or
'*    modify it under the terms of the GNU Lesser General Public
'*    License version 2.1, as published by the Free Software Foundation.
'*
'*    This library is distributed in the hope that it will be useful,
'*    but WITHOUT ANY WARRANTY; without even the implied warranty of
'*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
'*    Lesser General Public License for more details.
'*
'*    You should have received a copy of the GNU Lesser General Public
'*    License along with this library; if not, write to the Free Software
'*    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
'*    MA  02111-1307  USA
'*
'/************************************************************************
'*
'*  owner : joerg.skottke@sun.com
'*
'*  short description : Functions that work with files like deleting, loading...
'*
'*******************************************************************************
'**
' #1 hDeleteFile           ' delete a file 
' #1 hLoadFile             ' a more fault-tolerant version of hDateiOeffnen
' #1 hLoadRecentDocument   ' Load the first file in the picklist
' #1 hSaveFile             ' a function to write the file in the current format
' #1 hWaitWhileLoading     ' replaces IsItLoaded 
' #1 hHandleActivesOnLoad  ' handles possible messageboxes/dialogs when loading
' #1 hHandleInitialDialogs ' handles dialogs that can be triggered by templates
' #1 hGetSuffixFromFilename' get the suffix (incl. dot) from a filename
' #1 hCancelFilterDialog   ' cancel the filterdialog (filterdetection failure)
' #1 hFileHasThreeCharSuffix ' see if the dot is at pos 4 from the end of string
' #1 hGetFileNameFromProperties ' Get the filename from the document properties
' #1 hGetWorkPath          ' get the default work path for office files
' #1 hGetWorkFile          ' as hGetWorkPath + append the filename, convertpath
' #1 hGetInputPath         ' Build the path to a ref-file below .../input/...
' #1 hHandleSecurityWarning ' close the security warning if it exists
' #1 hGetFileSizeAsLong    ' Find the size of a file
' #1 hCreateDirectory      ' Create a directory (if it does not exist)
' #1 hGetLocalizedFileName ' prpend a filename with a numeric language code
'**
'\******************************************************************************

Option Explicit

private const CBUILDID = "680" 

function hDeleteFile( cFile as string ) as boolean

    const CFN = "hDeleteFile::"

    '///<h3>Delete a file</h3>
    '///<i>In many cases it is a good idea to use this function outside the testcase</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Filename (string)</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus (boolean)</li>
    '///<ul>
    '///+<li>TRUE: File was deleted/does not exist</li>
    '///+<li>FALSE: File could not be deleted/any other error</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    

    ' this function tries to delete a file and does some very basic error-
    ' handling. Returns 'true' on success, only error while deleting returns
    ' 'false', if the file does not exist, it is considered to be successfully
    ' deleted.
    ' i introduced this function due to a number of cases where deleting files
    ' actually failed because of weird code or situations where the user lacks
    ' accessrights to files are not handled at all.

    '///+<li>Check that the file exists</li>
    if ( dir( cFile ) <> "" ) then

        '///+<li>Use kill to delete</li>
        try

            kill( cFile )

            '///+<li>Verify that the file does not exist anymore</li>
            if ( dir( cFile ) = "" ) then
                printlog( CFN & "File successfully deleted: " & cFile )
                hDeleteFile() = true
            else
                warnlog( CFN & "File was not deleted: " & cFile )
                hDeleteFile() = false
            endif

        catch

            '///+<li>in very rare cases 'kill' fails and can be handled this way</li>
            qaerrorlog( CFN & "Deleting file failed: " & cFile )
            hDeleteFile() = false

        endcatch

    else

        '///+<li>write some text to the log if the file does not exist</li>
        printlog( CFN & "Nothing to do" )
        hDeleteFile() = true

    endif
    
    '///</ul>

end function

'*******************************************************************************

function hLoadFile( cFile as string ) as boolean

    '///<h3>Load a file via FileOpen</h3>
    '///<i>Replaces: hDateiOeffnen</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Filename (string)</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus (boolean)</li>
    '///<ul>
    '///+<li>TRUE: File was loaded</li>
    '///+<li>FALSE: File was not loaded or loading takes too long (timeout)</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
                   
    const CFN = "hLoadFile::"
    
    if ( cFile = "" ) then
        warnlog( CFN & "Empty string passed as filename" )
        hLoadFile() = false
        exit function
    endif
    

    dim bFileExists as boolean
    dim bDialogsAreClosed as boolean
    dim bFileIsOpen as boolean
    dim iTry as integer
    dim cMsg as string
    
    
    printlog( CFN & cFile )
    hLoadFile() = false


    '///+<li>look if the requested file exists</li>
    bFileExists = hFileExists( cFile )

    '///+<li>if the workfile does not exist, return false and end the function</li>
    if ( not bFileExists ) then
        warnlog( CFN & "Requested file could not be found. Aborting" )
        exit function
    endif


    '///+<li>Open the FileOpen dialog</li>
    FileOpen

    kontext "Active"
    if ( active.exists() ) then
	cMsg = active.getText()
        cMsg = hRemoveLineBreaks( cMsg )
        warnlog( CFN & "Unexpected messagebox: " & cMsg )
        active.ok()
    endif

    '///+<li>Enter the filename</li>
    Kontext "OeffnenDlg"
    Dateiname.SetText( cFile )

    '///+<li>Click &quot;Open&quot;</li>
    if ( not Oeffnen.isEnabled() ) then
        warnlog( CFN & "The open-button is disabled, aborting" )

        kontext "OeffnenDlg"
        OeffnenDlg.cancel()
        exit function
    endif

    Oeffnen.click()
    
    '///+<li>Verify that the document is opened</li>
    ' look if the document has been loaded by now, otherwise return false
    ' this will only be executed if all dialogs have been closed by
    ' hHandleActivesOnLoad.
    bFileIsOpen = hWaitWhileLoading()
    if ( not bFileIsOpen ) then
        warnlog( CFN & "File is not open. Aborting." )
    else
        sleep( 2 )
        printlog( CFN & "Open Documents: " & getDocumentCount )
    endif

    ' if everything went well this far, we exit the function successfully
    hLoadFile() = true
    
    '///</ul>    

end function

'*******************************************************************************

function hLoadRecentDocument( bSecWarn as boolean, _
                             iSecLevel as integer, _
                                iTries as integer ) as boolean

    '///<h3>Load the topmost document from the picklist</h3>
    '///<i>Starting point: Plain document (recommended)</i><br>
    '///<i>NOTE: Non-optional parameter iSecLevel is yet unused</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Allow security warning (boolean). valid options are:</li>
    '///<ul>
    '///+<li>TRUE: Yes</li>
    '///+<li>FALSE: No</li>
    '///</ul>
    '///+<li>Security Level (integer). Valid options are:</li>
    '///<ul>
    '///+<li>0: Low</li>
    '///+<li>1: Medium</li>
    '///+<li>2: High</li>
    '///+<li>3: Very high</li>
    '///</ul>
    '///+<li>Retries (integer)</li>
    '///<ul>
    '///+<li>Number of expected dialogs/warnings/messageboxes during load</li>
    '///</ul>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus (boolean)</li>
    '///<ul>
    '///+<li>TRUE: File was successfully loaded</li>
    '///+<li>FALSE: File not loaded/Loading too slow/getDocumentCount returned 0</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
    const CFN = "hLoadRecentDocument::"
    
    if ( ( iSecLevel < 0 ) or ( iSecLevel > 3 ) ) then
        warnlog( CFN & "Invalid security level passed to function:" & iSecLevel )
        hLoadRecentDocument() = false
        exit function
    endif
    
    if ( ( iTries < 0 ) or ( iTries > 3 ) ) then
        warnlog( CFN & "Unrealistic number of retries passed to function" & _
                 iTries )
        hLoadRecentDocument() = false
        exit function
    endif            
    
    dim brc as boolean
    
    '///+<li>Init Menu usage</li>
    hUseMenu()
    
    '///+<li>Select "File"</li>
    hMenuSelectNr( 1 )
    wait( 100 )
    
    '///+<li>Select "Recent Documents"</li>
    hMenuSelectNr( 3 )
    wait( 100 )
    
    '///+<li>Select "1"</li>
    hMenuSelectNr( 1 )
    
    '///+<li>Wait for the document to load</li>
    brc = hWaitWhileLoading()
    
    '///+<li>Handle Actives on load (like "Update external links" etc.)</li>
    brc = hHandleActivesOnLoad( bSecWarn , iTries )
    
    '///+<li>Handle special dialogs for some documents</li>
    brc = hHandleInitialDialogs()
    
    '///</ul>
    
end function

'*******************************************************************************

function hSaveFile( sFile as string, _
                 sBuildId as string, _
               bOverwrite as boolean, _
              bAutoSuffix as boolean ) as boolean
              
    const CFN = "hSaveFile::"   
    
    '///<h3>Save a file with options</h3>
    '///<i>Replaces: hSpeichernUnterMitFilterKill</i><br>
    '///<i>NOTE: Special behavior to suit framework needs, use original 
    '///+ function if possible</i><br>
    '///<i>Please refer to the inline documentation for more details</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Filename (string)</li>
    '///+<li>Build-ID (string). Valid options are:</li>
    '///<ul>
    '///+<li>&quot;Current&quot;: Uses default extension/filter</li>
    '///+<li>&quot;680&quot;: Use 680 default format</li>
    '///+<li>&quot;645&quot;: Use 645 default format (alien xml-filter)</li>
    '///+<li>&quot;641&quot;: Use 641 default format (alien xml-filter)</li>
    '///+<li>&quot;569&quot;: Use 569 default format (alien bin-filter)</li>
    '///</ul>
    '///+<li>Delete file in advance (boolean). Valid options are:</li>
    '///<ul>
    '///+<li>TRUE: Yes</li>
    '///+<li>FALSE: No</li>
    '///</ul>
    '///+<li>Use automatic filename-extension (boolean)</li>
    '///<ul>
    '///+<li>TRUE: Append suffix</li>
    '///+<li>FALSE: Omit suffix</li>
    '///</ul>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus</li>
    '///<ul>
    '///+<li>TRUE: Saving succeeded/document is still open</li>
    '///+<li>FALSE: On any other condition</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
       
    ' NOTE: If we specify bOverwrite = false and the file exists, we issue a 
    '       warning but do not handle the case here.
    '       Be aware that this function does not handle all gApplication,
    '       filtername and suffix combinations correctly.
    '       If you do not use autoextension ensure that gApplication and
    '       sBuildId are valid or you will get hard to debug warnings

    dim sFilterName as string
    
    dim brc as boolean
    dim iDocCount as integer
    dim bFileExists as boolean
        bFileExists = false

        
    ' optimism:
    hSaveFile() = true
    
    '///+<li>Count the open documents, Saving should not change the number</li>
    iDocCount = getDocumentCount()

    '///+<li>Check if the file exists</li> 
    if ( app.dir( sFile ) <> "" ) then
        bFileExists = true
    endif
    
    '///+<li>Check whether we may overwrite the file or not</li>
    if ( ( not bOverwrite ) and bFileExists ) then
        warnlog( CFN & "Will not overwrite existing file" )
        hSaveFile() = false
        exit function
    endif
        
    '///+<li>If we are allowed to overwrite, delete the file</li>
    ' NOTE: hDeleteFile tries to delete the file, no matter if it exists or not.
    '       The function only returns false if an existing file could not be 
    '       deleted (e.g. due to accessrights)
    '       The function does not verify that the file-save dialog handles
    '       overwriting correctly
    if ( bOverwrite ) then
        brc = hDeleteFile( sFile )
        if ( not brc ) then
            warnlog( CFN & "Delete File failed" )
        endif
    endif

    '///+<li>Open FileSaveAs</li>
    printlog( CFN & sFile )
    FileSaveAs
    
    '///+<li>Verify that the FileSaveAs dialog is indeed open</li>
    kontext "SpeichernDlg"
    if ( not SpeichernDlg.exists() ) then
        warnlog( CFN & "FileSaveAs failed: Dialog is not open" ) 
        hSaveFile() = false()
        exit function
    endif
    
    '///+<li>Enable/disable autoextension, set filename and save</li>
    if ( bAutoSuffix ) then
        AutomatischeDateinamenserweiterung.check()
    else
        AutomatischeDateinamenserweiterung.uncheck()
    endif
    
    '///+<li>Use a filter for saving - if provided</li>
    if ( sBuildId <> "" ) then
        sFilterName = hGetFilter( sBuildId )
        DateiTyp.select( sFilterName )
        printlog( CFN & "Using Filter: " & sFilterName )
    else
        printlog( CFN & "Filter has not been set, using default" )
    endif

    '///+<li>Set the filename</li>
    Dateiname.SetText( sFile )
    
    '///+<li>Save the file by clicking on "save"</li>
    Speichern.Click()
    
    '///+<li>Handle Alien-Warning, if it comes up</li>
    kontext "AlienWarning"
    if ( AlienWarning.exists() ) then
        printlog( CFN & "Closed Alien Warning" )
        AlienWarning.ok()
    endif
    
    '///+<li>Check if saving was successful</li>
    brc = IsItSaved
    if ( not brc ) then
        warnlog( CFN & "IsItSaved claims to have failed" )
        hSaveFile() = false
        exit function
    endif
    
    '///+<li>Verify we still have an unchanged number of documents</li>
    brc = hCheckDocCount( iDocCount , false )
    if ( not brc ) then
        hSaveFile() = false()
    endif
    
    '///</ul>
    
end function

'*******************************************************************************

function hHandleActivesOnLoad( iTries as integer , iAdd as integer ) as boolean

    '///<h3>Handle any dialogs that might pop up when loading a file</h3>
    '///<i>Beware: This function gives only limited control over the dialogs closed,
    '///+ it just closes anything it can.</i><br>
    '///<i>Please refer to the inline documentation for further details</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Number of dialogs to be closed (integer)</li>
    '///<ul>
    '///+<li>Max 3 allowed</li>
    '///</ul>
    '///+<li>Additional dialogs (integer).</li>
    '///<ul>
    '///+<li>Max 2 allowed</li>
    '///</ul>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorcondition (boolean)</li>
    '///<ul>
    '///+<li>TRUE: The expected number of dialogs were closed</li>
    '///+<li>FASLE: On any other condition</li>
    '///</ul>
     '///</ol>
    '///<u>Description</u>:
    '///<ul>
    '///+<li>OK to create a new document based on a template</li>
    '///+<li>YES to update links</li>
    '///+<li>YES to execute macros</li>
    '///</ul>
    
    const CFN = "hHandleActivesOnLoad::"
    
    if ( ( iTries < 0 ) or ( iTries > 3 ) ) then
        warnlog( CFN & "Unrealistic number of dialogs passed to function" & _
                 iTries )
        hHandleActivesOnLoad() = false
        exit function
    endif               

    if ( ( iAdd < 0 ) or ( iAdd > 2 ) ) then
        warnlog( CFN & "Unrealistic number of retries passed to function" & _
                 iAdd )
        hHandleActivesOnLoad() = false
        exit function
    endif               
    

    dim iTry as integer
    dim iActives as integer
    dim cMessage as string
    dim bSkip as boolean


    ' not good but apparently required - else hDestroyDocument will handle the
    ' dialogs and warn about them.
    sleep( 2 )

    ' think positive: preset the returnvalue with "true"
    hHandleActivesOnLoad() = true
    
    for iTry = 1 to iTries + iAdd

        ' handle two possible dialogs:
        ' 1. The warning that the user should create a new document based
        '    on this Sample/Template (the document is writeprotected)
        ' 2. The question to update links to the internet. Note: If you
        '    use proxies, they must be set correctly otherwise the result
        '    is undefined.

        ' Although this works quite well, there is a flaw that we will not
        ' get information about an "active" dialog that cannot be closed by
        ' .yes() or .ok(). If this happens, we probably run into an error 
        kontext "Active"
        if ( active.exists() ) then
           
            cMessage = active.getText()
            cMessage = hRemoveLineBreaks( cMessage )
            bSkip = false

            wait( 100 )
            

            ' ok to create a new document
            try
                active.ok()
                iActives = iActives + 1
                printlog( CFN & "MSG (" & iActives & "): " & cMessage )
                printlog( CFN & "MSG (" & iActives & "): closed with OK" )
                bSkip = true
            catch
                ' do nothing, probably wrong dialog
            endcatch

            if ( not bSkip ) then
                ' no to update links
                try
                    active.no()
                    iActives = iActives + 1
                    printlog( CFN & "MSG (" & iActives & "): " & cMessage )
                    printlog( CFN & "MSG (" & iActives & "): closed with NO" )
                catch
                    ' do nothing, probably wrong dialog
                endcatch
            endif

        endif

    next iTry

    ' now see how many dialogs were allowed and how many have been closed
    ' this does not change the return value of the function
    if ( iActives > iTries ) then
        printlog( CFN & "The test closed more dialogs than expected" )
        hHandleActivesOnLoad() = false
    endif
       
end function       

'*******************************************************************************

function hHandleInitialDialogs() as integer

    const CFN = "hHandleInitialDialogs::"

    '///<h3>Handle dialogs while opening samples/templates etc.</h3>
    '///<i>The returnvalue is of limited use</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Nothing</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Id of the dialog (integer)</li>
    '///<ul>
    '///+<li>0: No dialog was found</li>
    '///+<li>1: UseOfThisTemplate</li>
    '///+<li>2: StarOfficeCalendar1</li>
    '///+<li>3: NewsletterLayout</li>
    '///+<li>4: PortfolioCurrency</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
    
    ' NOTE: The ID of the dialog will be returned but is of limited use
    '       Function returns 0 if no dialogs have been encountered
    
    dim incident as integer
        incident = 0

    '///+<li>Test for UseOfThisTemplate-dialog</li>
    Kontext "UseOfThisTemplate"
    if ( UseOfThisTemplate.Exists() ) then
        printlog( CFN & "Closing UseOfThisTemplate-dialog" )
        incident = 1
        CancelButton.Click()
    endif
      
    '///+<li>Test for StarOfficeCalendar1-dialog</li>
    Kontext "StarOfficeCalendar1"
    if ( StarOfficeCalendar1.Exists() ) then
        printlog( CFN & "Closing StarOfficeCalendarl-dialog" )
        incident = 2
        Create.Click()
    endif
      
    '///+<li>Test for NewsletterLayout-dialog</li>
    Kontext "NewsletterLayout"
    if ( NewsletterLayout.Exists() ) then
        printlog( CFN & "Closing NewsletterLayout-dialog" )
        incident = 3
        OkButton.Click()
    endif
      
    '///+<li>Test for PortfolioCurrency-dialog</li>
    Kontext "PortfolioCurrency"
    if ( PortfolioCurrency.Exists() ) then
        printlog( CFN & "Closing PortfolioCurrency-dialog" )
        incident = 4
        OKButton.Click()
    endif
      
    hHandleInitialDialogs() = incident
    
    '///</ul>

end function         

'*******************************************************************************

function hWaitWhileLoading() as boolean

    '///<h3>Wait for a document to load</h3>
    '///<i>Extends: IsItLoaded with 10 retries</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Nothing</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus (boolean)</li>
    '///<ul>
    '///+<li>TRUE: Document was loaded</li>
    '///+<li>FALSE: Document not loaded</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
    ' IsItLoaded wastes at least 5 seconds on each call which is bad
    ' for the extras test which run at least 20 minutes longer than 
    ' necessary. The function works the same way as IsItLoaded but
    ' eliminates a lot of sleep-cycles.
    
    dim iNextProbe as integer
    dim brc as boolean
    dim iTime as integer

    const CFN = "hWaitWhileLoading::"
    const I_MAX_WAIT = 100
    const I_PROBE_INTERVAL = 100

    '///+<li>Cycle for at most ten times to check the IsDocLoading-slot</li>
    for iNextProbe = 1 to I_MAX_WAIT
    
        brc = false
        wait( I_PROBE_INTERVAL )

        '///+<li>while the file is loaded the slot will not be dispatched -> catch</li>
        try
            brc = IsDocLoading
            hWaitWhileLoading() = true
            iTime = iNextProbe * I_PROBE_INTERVAL
            printlog( CFN & "File loaded in less than " & iTime & " ms" )
        catch
        endcatch

        '///+<li>"exit for" within try...catch does not always work</li>
        if ( brc ) then
            exit for
        endif

    next iNextProbe

    '///+<li>Warn if the document did not respond in at most 10 seconds</li>
    if ( not brc ) then
        printlog( CFN & "Did not get feedback within 10 seconds" )
        hWaitWhileLoading() = true
    endif
    '///</ul>

end function

'*******************************************************************************

function hFileHasThreeCharSuffix( sFile as string ) as boolean

    '///<h3>Test whether a standard three-char suffix exists in a filename</h3>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Filename with optional suffix (string)</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus (boolean)</li>
    '///<ul>
    '///+<li>TRUE: A suffix was found</li>
    '///+<li>FALSE: A suffix could not be identified</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    

    const CFN = "hFileHasThreeCharSuffix::"

    dim iLen as integer
        iLen = len ( sFile )
        
    '///+<li>Check that the filename is long enough to have a standard suffix</li>
    if ( iLen < 4 ) then
        printlog( CFN & "Filename too short, len = " & iLen )
        hFileHasThreeCharSuffix() = false
        exit function
    endif

    '///+<li>If the fourth character from right is a dot, return true</li>
    if ( mid( sFile , iLen - 4 , 1 ) = "." ) then
        hFileHasThreeCharSuffix() = true
    else
        hFileHasThreeCharSuffix() = false
    endif
    
    '///</ul>
    
end function

'*******************************************************************************

function hGetSuffixFromFilename( sFile as string ) as string

    '///<h3>Retrieve the suffix from a filename</h3>
    '///<i>Suffix must be of type &quot;.xxx&quot;</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Filename (string)</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Suffix (string)</li>
    '///<ul>
    '///+<li>&quot;&quot; if file has no suffix</li>
    '///+<li>The suffix &quot;.XXX&quot;</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
    const CFN = "hGetSuffixFromFilename::"
    
    '///+<li>Determine the length of the filename</li>
    dim iLen as integer
    dim cSuffix as string
    
    iLen = len( sFile )
    
    '///+<li>Verify that the length is at least 5 chars (a letter, dot, suffix)</li>
    if ( iLen < 5 ) then
        warnlog( CFN & "Filename is too short to hold a suffix" )
        hGetSuffixFromFilename() = ""
        exit function
    endif
    
    '///+<li>Verify that the 4th character from right is a dot</li>
    if ( not ( mid( sFile , iLen - 4 , 1 ) = "." ) ) then
        warnlog( CFN & "The separator-dot is not at the expected position" )
        hGetSuffixFromFilename() = ""
        exit function
    endif
    
    '///+<li>Isolate the dot plus three-letter suffix</li>
    cSuffix = right( sFile , 4 )

    '///+<li>Return the suffix</li>
    printlog( CFN & "Returning: " & cSuffix ) 
    hGetSuffixFromFilename() = cSuffix
    
    '///</ul>
    
end function

'*******************************************************************************

function hCancelFilterDialog() as boolean

    '///<h3>Handle filter selection dialog</h3>
    '///<i>This dialog might come up if the documenttype could not be determined.
    '///+ This is most likely a bug since we should handle any type of document
    '///+ correctly</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Nothing</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus</li>
    '///<ul>
    '///+<li>TRUE: If the dialog was found and closed</li>
    '///+<li>FALSE: If no filterdialog exists</li>
    '///</ul>
    '///+<li>Variable</li>
    '///<ul>
    '///+<li>Variable description</li>
    '///+<li>Variable description</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
                              
    const CFN = "hCancelFilterDialog::"
    
    '///+<li>Check whether the filter dialog is open or not, close it</li>
    kontext "FilterAuswahl"
    if ( FilterAuswahl.exists() ) then
        warnlog( CFN & "Found. Closing with CANCEL" )
        hCancelFilterDialog() = true
        FilterAuswahl.cancel()
    else
        printlog( CFN & "No FilterDialog." )
        hCancelFilterDialog() = false
    endif
    
    '///</ul>
    
end function

'*******************************************************************************

function hGetFileNameFromProperties() as string

    '///<h3>Retrieve the filename from document-properties</h3>
    '///<i>This is one of the few places where the suffix is actually displayed</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Nothing</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Filename without path(string)</li>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
    ' Note: As soon as a document has been saved the filename in the document-
    '       properties dialog will be updated. Additionally it will have the
    '       proper suffix attached if the file was saved using autoextension.
    '       The function tries up to 5 times to get the filename including the 
    '       suffix. The document-properties are closed and reopened between attempts.
    '       Usually it works on the first try, only on extremely slow machines
    '       we might run into retries (seen on pII/300 with little RAM)
    

    const CFN = "hGetFileNameFromProperties::"
    dim sFile as string
    dim iRetries as integer
    
    
    for iRetries = 1 to 5

        '///+<li>Open File->Properties</li>
        FileProperties
    
        '///+<li>Select Document tab</li>
        kontext "TabDokument"
        if ( TabDokument.exists() ) then
        
            '///+<li>Get name of document from entryfield</li>
            sFile = DokumentName.getText()
            printlog( CFN & "Name is: " & sFile )
            TabDokument.cancel()
        else
            warnlog( CFN & "Document Properties Dialog is not open" )
            sFile = ""
        endif
        
        '///+<li>Check that the name has a suffix</li>
        if ( instr( sFile , "." ) > 0 ) then
            exit for
        else
            qaerrorlog( CFN & "Properties dialog not updated with filename yet" )
            wait( 100 )
        endif
        
        if ( iRetries = 3 ) then
            warnlog( CFN & "Updating the properties page takes too long" )
        endif 
        
    next iRetries
    
    '///+<li>Return the filename including suffix</li>
    hGetFileNameFromProperties() = sFile
    
    '///</ul>
    
end function

'******************************************************************************

function hGetWorkPath() as string

    '///<h3>Retrieve the users work directory</h3>
    '///<i>Uses: Call to UNO service</i><br>
    '///<i>Errorhandling: Fixes &quot;$(user)&quot;-type path (uses fallback)</i><br>
    '///<i>Errorhandling: Handles broken UNO connection (uses fallback)</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Nothing</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Path to local workdir (string)</li>
    '///<ul>
    '///+<li>Includes trailing slash/backslash</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>

    dim sPath as string
    dim xViewRoot as object
    dim oOfficeConnect as object
    dim oOfficeConfig as object
    dim aPropertyValue( 1 ) as new com.sun.star.beans.PropertyValue

    const CONNECT = "com.sun.star.configuration.ConfigurationProvider"
    const PROPVAL = "/org.openoffice.Office.Common/Path/Current"
    const CONFIG1 = "com.sun.star.configuration.ConfigurationUpdateAccess"
    const CFN     = "hGetWorkPath::"

    '///+<li>Create an UNO service and ask it for the location of &quot;Work&quot;</li>
    try
        aPropertyValue( 0 ).Name = "nodepath"

        oOfficeConnect = hGetUnoService( true )
        oOfficeConfig  = oOfficeConnect.createInstance( CONNECT )
        aPropertyValue( 0 ).Value = PROPVAL
        xViewRoot = oOfficeConfig.createInstanceWithArguments( CONFIG1 , aPropertyValue() )

        sPath = convertFromURL( xViewRoot.getByName( "Work" ) ) 

        xViewRoot.dispose()
    catch
        '///+<li>Handle broken UNO connection</li>
        printlog( CFN & "Could not access service, connection broken?" )
        sPath = "$(user)"
    endcatch
                
    '///+<li>Apply fallback in case of broken connection or invalid path</li>
    if ( instr( sPath , "$(user)" ) > 0 ) then
        printlog( CFN & "sPath has $(user)-type string, using fallback" )
        sPath = gOfficePath & "user/work"
    endif

    '///+<li>Add trailing pathseparator, do convertpath</li>
    sPath = sPath & "/"
    sPath = convertpath( sPath ) 
    
    '///+<li>Print info to the log and return the path</li>
    printlog( CFN & sPath )
    hGetWorkPath() = sPath 
    '///</ul>

end function

'*******************************************************************************

function hGetInputPath( byval sBasePath as string, byval sFile as string ) as string

    '///<h3>Create an input path for reference files</h3>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>BasePath (string) without trailing slash/backslash</li>
    '///<ul>
    '///+<li>Format: gTesttoolPath &amp; &quot;[project]\input\[dir]&quot;</li>
    '///</ul>
    '///+<li>Filename (string)</li>
    '///<ul>
    '///+<li>The name of the file below BasePath to be used</li>
    '///</ul>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Full path to file: BasePath + Build-Id + ProductName + sFile</li>
    '///<ul>
    '///+<li>Build-ID (e.g. &quot;680&quot;, from environment)</li>
    '///+<li>ProducName (e.g &quot;OpenOffice.org&quot;, from environment)</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
    const CFN = "hGetInputPath::"
    dim sPath as string
    dim iAttr as integer
    dim cLastChar as string
    dim iLen as integer
    
    sPath = convertpath( sBasePath )
    
    '///+<li>None of the two parameters may be empty</li>
    if ( sPath = "" or sFile = "" ) then
        warnlog( CFN & "Invalid parameter: Empty string passed to function" )
        hGetInputPath() = ""
        exit function
    endif

    '///+<li>Remove trailing slash or backslash</li>
    iLen = len( sPath )
    cLastChar = right( sPath , 1 )
    if ( cLastChar = gPathSigne ) then
        sPath = mid( sPath , 1 , iLen - 1 )
        printlog( CFN & "Truncated path: Trailing Pathseparator not allowed" )
    endif

    '///+<li>At least Basepath should exist</li>
    if ( dir ( sPath ) = "" ) then
        warnlog( CFN & "BasePath does not exist: " & sPath )
        hGetInputPath() = ""
        exit function
    endif    
    
    '///+<li>We must be certain, that basepath is a directory</li>
    iAttr = getAttr( sPath )
    if ( iAttr < 16 or iAttr > 17 ) then
        warnlog( CFN & "Invalid Path: Not a directory: " & sPath )
        hGetInputPath() = ""
        exit function
    endif
     
    '///+<li>Now we concatenate all the information we have to a fully qualified path</li>
    sPath = sPath & "\" & CBUILDID & "\" & gProductName & "\" & sFile
    sPath = convertpath( sPath )
    
    '///+<li>...and return it (without further verification)</li>
    printlog( CFN & sPath )
    hGetInputPath() = sPath
    
    '///</ul>
    
end function

'*******************************************************************************

function hHandleSecurityWarning() as boolean

    const CFN = "hHandleSecurityWarning::"

    '///<h3>Handle the Macro Security warning if it exists.</h3>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Nothing</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Whether dialog was closed or not</li>
    '///<ul>
    '///+<li>TRUE: Macro Security Warning was closed</li>
    '///+<li>FALSE: If the dialog was not found</li>
    '///</ul>
    '///+<li>Variable</li>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>    
    
    dim bFound as boolean
        bFound = false
        
    dim cMessage as string

    '///+<li>See if the dialog is present</li>    
    kontext "SecurityWarning"
    if ( SecurityWarning.exists() ) then
        bFound = true
    endif

    '///+<li>If it exists, close it and return TRUE</li>
    if ( bFound ) then
        SecurityWarning.ok()
        printlog( CFN & "Closed macro security warning" )
        hHandleSecurityWarning() = true
    '///+<li>If the warning is missing, return FALSE</li>
    else
        printlog( CFN & "Security warning is missing" )
        hHandleSecurityWarning() = false
    endif
    '///</ul>
    
end function    

'*******************************************************************************

function hGetFileSizeAsLong( cFileName as string ) as long

    '///<h3>Get the size (long) of a file</h3>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Filename (string)</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Size of file (long)</li>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>    
    
    const CFN = "hGetFileSizeAsLong::"

   ' This function returns the size of a specified file. If it is a directory,
   ' the size will default to -1 (A filesize can never be <0 - hopefully)

   dim iAttrib as integer

   '///+<li>Verify that the given file exists</li>
   if ( app.dir( cFileName ) <> "" ) then

      try
          iAttrib = getattr( cFileName )
      
          '///+<li>Return the size for a file or -1 for a directory</li>
          if ( iAttrib <> 16 and iAttrib <> 17 ) then
             hGetFileSizeAsLong() = filelen( cFileName )
          else
             hGetFileSizeAsLong() = -1
          endif
      catch
          qaerrorlog( CFN & "File was not handled correctly: " & cFileName )
          hGetFileSizeAsLong() = 0
      endcatch

   else

      '///+<li>Return a size of 0 if the file was not found</li>
      hGetFileSizeAsLong() = 0
      warnlog( CFN & "File does not exist: " & cFileName )
                
   endif
   
   '///</ul>

end function

'*******************************************************************************

function hCreateDirectory( cDir as string ) as integer

    '///<h3>Create a directory (if it does not exist)</h3>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Name of the directory without trailing slash/backslash</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus (integer)</li>
    '///<ul>
    '///+<li>0: Directory exists/has been created</li>
    '///+<li>1: Failed to create directory</li>
    '///+<li>2: Object exists but is no directory</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>

    const CFN = "hCreateDirectory::"
    dim iStatus as integer
        
    '///+<li>See if anything exists a the designated location</li>
    '///+<li>Create the directory</li>
    if ( dir( cDir ) = "" ) then
        'printlog( CFN & "Dir does not exist, trying to create" )
        mkdir( cDir )
    endif
    
    '///+<li>Verify that the directory exists</li>
    if ( dir( cDir ) <> "" ) then
        if ( dir( cDir , 16 ) <> "" ) then
            printlog( CFN & "Directory exists/has been created:" & cDir )
            iStatus = 0
        else
            warnlog( CFN & "Object exists but is not a directory: " & cDir  )
            iStatus = 2
        endif
    else
        warnlog( CFN & "Directory could not be created: " & cDir )
        iStatus = 1
    endif
    
    hCreateDirectory() = iStatus
    '///</ul>
    
end function

'*******************************************************************************

function hGetWorkFile( cFileName as string ) as string

    '///<h3>Returns a fully qualified filename to a workfile</h3>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Filename without path (string)</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Filename including path to user work directory (string)</li>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
    '///+<li>Concatenate workpath and filename, convertpath</li>
    hGetWorkFile() = convertpath( hGetWorkPath() & cFileName )
    '///</ul>

end function

'*******************************************************************************

function hGetLocalizedFileName( cFileName as string ) as string

    '///<h3>Function to prepend numeric language code to a filename</h3>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Filename (string)</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Filename with leading numeric language code (string)</li>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    '///+<li>Build a string of type <code>Number + _ + Filename</code>, convertpath</li>
    hGetLocalizedFileName() = convertpath( iSprache & "_" & cFileName )
    '///</ul>

end function

'*******************************************************************************

function hTestFile( cFile as string ) as boolean

    '///<h3>Test whether a file exists or not</h3>
    '///<i></i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Filename (string)</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus (boolean)</li>
    '///<ul>
    '///+<li>TRUE: File exists</li>
    '///+<li>FALSE: File does not exist</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    const CFN = "hTestFile::"
    
    '///+<li>Stat the file</li>
    if ( dir( cFile ) <> "" ) then
        hTestFile = true
        printlog( CFN & "File exists" )
    else
        htestFile = false
        printlog( CFN & "File does not exist" )
    endif
    '///</ul>
    
end function
    
