/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1999, 2000, 2001 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/parted.h>
#include <string.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#define BUFFER_SIZE	4096		/* in sectors */

static PedFileSystemType*	fs_types = NULL;

void
ped_file_system_type_register (PedFileSystemType* fs_type)
{
	PED_ASSERT (fs_type != NULL, return);
	PED_ASSERT (fs_type->ops != NULL, return);
	PED_ASSERT (fs_type->name != NULL, return);
	
	/* pretend that "next" isn't part of the struct :-) */
	((struct _PedFileSystemType*) fs_type)->next = fs_types;
	fs_types = (struct _PedFileSystemType*) fs_type;
}

void
ped_file_system_type_unregister (PedFileSystemType* fs_type)
{
	PedFileSystemType*	walk;
	PedFileSystemType*	last = NULL;

	PED_ASSERT (fs_type != NULL, return);

	for (walk = fs_types; walk != fs_type; walk = walk->next) {
		if (!walk) return;
		last = walk;
	}

	if (last) {
		((struct _PedFileSystemType*) last)->next = fs_type->next;
	} else {
		fs_types = fs_types->next;
	}
}

PedFileSystemType*
ped_file_system_type_get (const char* name)
{
	PedFileSystemType*	walk;

	PED_ASSERT (name != NULL, return NULL);

	for (walk = fs_types; walk != NULL; walk = walk->next) {
		if (!strcasecmp (walk->name, name))
			break;
	}
	return walk;
}

PedFileSystemType*
ped_file_system_type_get_next (const PedFileSystemType* fs_type)
{
	if (fs_type)
		return fs_type->next;
	else
		return fs_types;
}

static int
_test_open (PedFileSystemType* fs_type, const PedGeometry* geom)
{
	PedFileSystem*		fs;

	ped_exception_fetch_all ();
	fs = fs_type->ops->open (geom);
	if (fs)
		ped_file_system_close (fs);
	else
		ped_exception_catch ();
	ped_exception_leave_all ();
	return fs != NULL;
}

static PedFileSystemType*
_probe_with_open (const PedGeometry* geom, int detected_count,
		  PedFileSystemType* detected[])
{
	int			i;
	PedFileSystemType*	open_detected = NULL;

	for (i=0; i<detected_count; i++) {
		if (detected [i]->ops->open && !_test_open (detected [i], geom))
			continue;

		if (open_detected)
			return NULL;
		else
			open_detected = detected [i];
	}
	return open_detected;
}

PedFileSystemType*
ped_file_system_probe (const PedGeometry* geom)
{
	PedFileSystemType*	detected[32];
	int			detected_count = 0;
	PedFileSystemType*	walk = NULL;

	PED_ASSERT (geom != NULL, return NULL);

	ped_exception_fetch_all ();
	while ( (walk = ped_file_system_type_get_next (walk)) ) {
		if (walk->ops->probe (geom))
			detected [detected_count++] = walk;
		else
			ped_exception_catch ();
	}
	ped_exception_leave_all ();

	if (!detected_count)
		return NULL;
	if (detected_count == 1)
		return detected [0];
	return _probe_with_open (geom, detected_count, detected);
}

int
ped_file_system_clobber (PedGeometry* geom)
{
	PedFileSystemType*	fs_type = NULL;

	PED_ASSERT (geom != NULL, return 0);

	ped_exception_fetch_all ();
	while ((fs_type = ped_file_system_type_get_next (fs_type))) {
		int		probbed;
		PED_ASSERT (fs_type->ops->clobber != NULL, return 0);
		PED_ASSERT (fs_type->ops->probe != NULL, return 0);

		probbed = fs_type->ops->probe (geom);
		if (!probbed) {
			ped_exception_catch ();
			continue;
		}

		if (!fs_type->ops->clobber (geom))
			goto error;
	}
	ped_exception_leave_all ();
	return 1;

error:
	ped_exception_leave_all ();
	return 0;
}

static int
ped_file_system_clobber_exclude (const PedGeometry* geom,
				 const PedGeometry* exclude)
{
	PedGeometry*    clobber_geom;
	int             status;

	if (ped_geometry_test_sector_inside (exclude, geom->start))
		return 1;

	clobber_geom = ped_geometry_duplicate (geom);
	if (ped_geometry_test_overlap (clobber_geom, exclude))
		ped_geometry_set_end (clobber_geom, exclude->start - 1);

	status = ped_file_system_clobber (clobber_geom);
	ped_geometry_destroy (clobber_geom);
	return status;
}

PedFileSystem*
ped_file_system_open (PedGeometry* geom)
{
	PedFileSystemType*	type;

	PED_ASSERT (geom != NULL, return NULL);

	type = ped_file_system_probe (geom);
	if (!type) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
				     _("Could not detect file system."));
		return NULL;
	}
	if (!type->ops->open) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				     PED_EXCEPTION_CANCEL,
				     _("Support for opening %s file systems "
				       "is not implemented yet."),
				     type->name);
		return NULL;
	}
	return type->ops->open (geom);
}

PedFileSystem*
ped_file_system_create (PedGeometry* geom, const PedFileSystemType* type)
{
	PED_ASSERT (geom != NULL, return NULL);
	PED_ASSERT (type != NULL, return NULL);

	if (!type->ops->create) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				     PED_EXCEPTION_CANCEL,
				     _("Support for creating %s file systems "
				       "is not implemented yet."),
				     type->name);
		return NULL;
	}

	if (!ped_file_system_clobber (geom))
		return NULL;

	return type->ops->create (geom);
}

int
ped_file_system_close (PedFileSystem* fs)
{
	PED_ASSERT (fs != NULL, return 0);
	return fs->type->ops->close (fs);
}

int
ped_file_system_check (PedFileSystem* fs)
{
	PED_ASSERT (fs != NULL, return 0);

	if (!fs->type->ops->check) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				     PED_EXCEPTION_CANCEL,
				     _("Support for checking %s file systems "
				       "is not implemented yet."),
				     fs->type->name);
		return 0;
	}
	return fs->type->ops->check (fs);
}

static int
_raw_copy (const PedGeometry* src, PedGeometry* dest)
{
	char*		buf;
	PedSector	pos;

	PED_ASSERT (src != NULL, goto error);
	PED_ASSERT (dest != NULL, goto error);
	PED_ASSERT (src->length <= dest->length, goto error);

	buf = ped_malloc (BUFFER_SIZE * 512);		/* FIXME */
	if (!buf)
		goto error;

	for (pos = 0; pos + BUFFER_SIZE < src->length; pos += BUFFER_SIZE) {
		if (!ped_geometry_read (src, buf, pos, BUFFER_SIZE))
			goto error_free_buf;
		if (!ped_geometry_write (dest, buf, pos, BUFFER_SIZE))
			goto error_free_buf;
	}

	if (pos < src->length) {
		if (!ped_geometry_read (src, buf, pos, src->length - pos))
			goto error_free_buf;
		if (!ped_geometry_write (dest, buf, pos, src->length - pos))
			goto error_free_buf;
	}

	ped_free (buf);
	return 1;

error_free_buf:
	ped_free (buf);
error:
	return 0;
}

static int
_raw_copy_and_resize (const PedFileSystem* fs, PedGeometry* geom)
{
	PedFileSystem*		new_fs;

	if (!_raw_copy (fs->geom, geom))
		goto error;

	new_fs = ped_file_system_open (geom);
	if (!new_fs)
		goto error;
	if (!ped_file_system_resize (new_fs, geom))
		goto error_close_new_fs;
	ped_file_system_close (new_fs);
	return 1;

error_close_new_fs:
	ped_file_system_close (new_fs);
error:
	return 0;
}

int
ped_file_system_copy (const PedFileSystem* fs, PedGeometry* geom)
{
	PED_ASSERT (fs != NULL, return 0);
	PED_ASSERT (geom != NULL, return 0);

	if (ped_geometry_test_overlap (fs->geom, geom)) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Can't copy onto an overlapping partition."));
		return 0;
	}

	if (!ped_file_system_clobber_exclude (geom, fs->geom))
		return 0;

	if (!fs->type->ops->copy) {
		if (fs->type->ops->resize) {
			if (fs->geom->length <= geom->length)
				return _raw_copy_and_resize (
						fs,
					       	(PedGeometry*) geom);
				
			ped_exception_throw (
				PED_EXCEPTION_NO_FEATURE,
				PED_EXCEPTION_CANCEL,
				_("Direct support for copying file systems is "
				  "not yet implemented for %s.  However, "
				  "support for resizing implemented.  "
				  "Therefore, the file system can be copied if "
				  "the new partition is at least as big as the "
				  "old one.  So, either shrink the partition "
				  "you are trying to copy, or copy to a bigger "
				  "partition."),
				fs->type->name);
			return 0;
		} else {
			ped_exception_throw (
				PED_EXCEPTION_NO_FEATURE,
				PED_EXCEPTION_CANCEL,
				_("Support for copying %s file systems is not "
				  "implemented yet."),
				fs->type->name);
			return 0;
		}
	}
	return fs->type->ops->copy (fs, geom);
}

int
ped_file_system_resize (PedFileSystem* fs, const PedGeometry* geom)
{
	PED_ASSERT (fs != NULL, return 0);
	PED_ASSERT (geom != NULL, return 0);

	if (!fs->type->ops->resize) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				     PED_EXCEPTION_CANCEL,
				     _("Support for resizing %s file systems "
				       "is not implemented yet."),
				     fs->type->name);
		return 0;
	}
	if (!ped_file_system_clobber_exclude (geom, fs->geom))
		return 0;
	return fs->type->ops->resize (fs, geom);
}

PedConstraint*
ped_file_system_get_resize_constraint (const PedFileSystem* fs)
{
	PED_ASSERT (fs != NULL, return 0);

	if (!fs->type->ops->get_resize_constraint) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				     PED_EXCEPTION_CANCEL,
				     _("No get_resize_constraint for %s!"),
				     fs->type->name);
		return 0;
	}
	return fs->type->ops->get_resize_constraint (fs);
}

int
ped_file_system_set_system (const PedFileSystem* fs, PedPartition* part,
			    const PedDiskType* disk_type)
{
	PED_ASSERT (fs != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);
	PED_ASSERT (disk_type != NULL, return 0);
	PED_ASSERT (fs->type->ops->set_system != NULL, return 0);

	if (!fs->type->ops->set_system (fs, part, disk_type)) {
		return ped_exception_throw (
			PED_EXCEPTION_NO_FEATURE,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("The %s file system code doesn't support %s disk "
			  "labels."),
			fs->type->name, disk_type->name)
				== PED_EXCEPTION_IGNORE;
	}

	return 1;
}

