/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *             Author(s): Holger Macht <hmacht@suse.de>                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include <getopt.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <liblazy.h>
#include <limits.h>

#include "powerlib.h"
#include "config.h"

#include <string>
#include <iostream>

#define MAX_OPT_LEN 20
#define MAX_PARAM_LENGTH 50

#define DBUS_HAL_INTERFACE		"org.freedesktop.Hal"
#define DBUS_HAL_CPUFREQ_INTERFACE	"org.freedesktop.Hal.Device.CPUFreq"
#define DBUS_HAL_SYSTEM_POWER_INTERFACE	"org.freedesktop.Hal.Device.SystemPowerManagement"
#define HAL_UDI_COMPUTER		"/org/freedesktop/Hal/devices/computer"
#define HAL_DEVICE_LAPTOP_PANEL		"org.freedesktop.Hal.Device.LaptopPanel"

using namespace std;

static int option = -1;
static int verbose = 0;
static char buffer[MAX_PARAM_LENGTH + 1] = "";

// toDo: set modes to specification ...
static int BATT_INFO = 0;
static int THROT_INFO = 0;
static int SET_THROT_PERC = 0;
static int AC_INFO = 0;
static int APM_ACPI = 0;
static int CPUFREQ_INFO = 0;
static int EXT_BATT_INFO = 0;
static int THERMAL_INFO = 0;
static int FAN_INFO = 0;

static int PERFORMANCE_SPEED = 0;
static int POWERSAVE_SPEED = 0;
static int DYNAMIC_SPEED = 0;
static int DYNAMIC_PERFORMANCE = 0;
static int GET_PERFORMANCE = 0;
static int SET_GOVERNOR = 0;
static int GET_GOVERNOR = 0;

static int SUSPEND_TO_DISK = 0;
static int SUSPEND_TO_RAM = 0;
static int STANDBY = 0;
static int VERSION_INFO = 0;

static int LIST_SCHEMES = 0;
static int SET_ACTIVE_SCHEME = 0;
static int GET_SCHEME_DESCRIPTION = 0;

static int GET_BRIGHTNESS = 0;
static int GET_BRIGHTNESS_LEVELS = 0;
static int SET_BRIGHTNESS = 0;

static int CPU_ENABLE = 0;
static int CPU_DISABLE = 0;

void general_error()
{
	fprintf(stderr, "An error occured. Make sure that the powersave daemon is\n"
		"running and use -v to increase verbose level and have a look into syslog.\n");
}

void hal_error()
{
	fprintf(stderr, "An error occured. Cannot connect to HAL daemon. Make sure it is\n"
		"running and use -v to increase verbose level and have a look into syslog.\n");
}

void privilege_error(const char *privilege)
{
	fprintf(stderr, "User is not allowed for %s according to PolicyKit.\n", privilege);
}

void no_connect_no_rights_error()
{
	fprintf(stderr, "Could not connect to powersaved because of missing permissions.\n"
		"Are you priviledged to connect to the powersaved?\n");
}

void no_connect_no_daemon_error()
{
	fprintf(stderr, "Could not connect to powersaved. Is the powersaved running?\n");
}

void no_cpufreq_error()
{
	fprintf(stderr, "Speedstepping is not supported.\n");
}

void usage()
{
	fprintf(stderr,
		"Usage: powersave {-f|-l|-A} {-u|-U|-m} -[crbBsSaTFVtxKL] [-p percent] [-v level] [-e scheme_name] -k [brightness level]\n"
		"                 [-h|--help] -i [device class] -I [device class] -J [device class] -j [device number]\n\n"
		" Suspend/Standby:\n"
		"   -U, --suspend-to-disk                  set machine into suspend-to-disk (ACPI S4/APM suspend)\n"
		"   -u, --suspend-to-ram                   set machine into suspend-to-ram  (ACPI S3/APM suspend)\n"
		"   -m, --standby                          set machine into standby         (ACPI S1/APM standby)\n\n"
		" CPUFreq modes:\n"
		"   -f, --performance-speed                set cpufreq to performance mode\n"
		"   -l, --powersave-speed                  set cpufreq to powersave mode\n"
		"   -A, --dynamic-speed                    set cpufreq to dynamic mode\n"
		"   -g, --set-dynamic-performance          set cpufreq dynamic performance\n"
		"   -G, --get-dynamic-performance          get cpufreq dynamic performance\n"
		"   -o, --set-governor                     set cpufreq governor\n"
		"   -O, --get-governor                     get cpufreq governor\n"
		"   -c, --cpufreq-state-info               print out the current cpufreq policy\n\n"
		" Throttling:\n"
		"   -p <x>, --set-throttling-percent <x>   throttles processors of machine by X percent\n"
		"   -t, --throttling-info                  prints out current throttling state\n\n"
		" Schemes:\n"
		"   -x, --list-schemes                     show all available schemes\n"
		"   -X, --get-scheme-description           show the description of scheme\n"
		"   -e <x>, --set-active-scheme <x>        switch currently active scheme\n\n"
		" Brightness:\n"
		"   -k, --set-brightness <x>               set current brightness level\n"
		"   -K, --get-brightness                   get current brightness level\n"
		"   -L, --get-brightness-levels            get number of brightness levels\n\n"
		" CPU hotplugging:\n"
		"   -E --cpu-enable <x>                   enable a specific CPU starting from 0\n"
		"   -D --cpu-disable <x>                  disable a specific CPU starting from 0 (CPU 0 can't be disabled)\n\n"
		" Print information:\n"
		"   -b, --battery-info                     general battery info\n"
		"   -B, --detailed-battery-info            info of each battery\n"
		"   -S, --apm-acpi                         prints out whether APM or ACPI is supported\n"
		"   -a, --ac-status-info                   power supply info (AC/Battery)\n"
		"   -T, --get-thermal-info                 thermal Info\n"
		"   -F, --get-fan-info                     fan Info\n"
		"   -v <x>, --verbose <x>                  Verbose level (see manpage for values)\n"
		"   -V, --version                          current compiled version\n\n");
	exit(EXIT_FAILURE);
}

void checkArgs()
{
	int info =
	    BATT_INFO + THROT_INFO + SET_THROT_PERC + AC_INFO + APM_ACPI + CPUFREQ_INFO +
	    EXT_BATT_INFO + THERMAL_INFO + FAN_INFO + GET_BRIGHTNESS + GET_BRIGHTNESS_LEVELS;
	int cpufreq = PERFORMANCE_SPEED + POWERSAVE_SPEED + DYNAMIC_SPEED + DYNAMIC_PERFORMANCE;
	int sleep = SUSPEND_TO_DISK + SUSPEND_TO_RAM;

	if (sleep > 1 || cpufreq > 1) {
		usage();
		exit(EXIT_FAILURE);
	}
	if (sleep == 1 && (info > 0 || cpufreq > 0)) {
		usage();
		exit(EXIT_FAILURE);
	}
	return;
}

void getArgs(int argc, char **argv)
{

	int option_index = 0;
	char *endptr[MAX_PARAM_LENGTH + 1];

	struct option opts[] = {
		{"performance-speed", 0, 0, 'f'},
		{"powersave-speed", 0, 0, 'l'},
		{"dynamic-speed", 0, 0, 'A'},
		{"set-dynamic-performance", 1, 0, 'g'},
		{"get-dynamic-performance", 0, 0, 'G'},
		{"set-governor", 1, 0, 'o'},
		{"get-governor", 0, 0, 'O'},
		{"cpufreq-state-info", 0, 0, 'c'},
		{"apm-acpi", 0, 0, 'S'},
		{"throttling-info", 0, 0, 't'},
		{"set-throttling-percent", 1, 0, 'p'},
		{"suspend-to-disk", 0, 0, 'U'},
		{"suspend-to-ram", 0, 0, 'u'},
		{"standby", 0, 0, 'm'},
		{"get-thermal-info", 0, 0, 'T'},
		{"get-fan-info", 0, 0, 'F'},
		{"battery-info", 0, 0, 'b'},
		{"detailed-battery-info", 0, 0, 'B'},
		{"ac-status-info", 0, 0, 'a'},
		{"list-schemes", 0, 0, 'x'},
		{"set-active-scheme", 1, 0, 'e'},
		{"get-scheme-description", 1, 0, 'X'},
		{"set-brightness", 1, 0, 'k'},
		{"get-brightness", 0, 0, 'K'},
		{"get-brightness-levels", 0, 0, 'L'},
		{"cpu-enable", 1, 0, 'E'},
		{"cpu-disable", 1, 0, 'D'},
		{"verbose", 0, 0, 'v'},
		{"version", 0, 0, 'V'},
		{"help", 0, 0, 'h'},
		{NULL, 0, 0, 0},
	};
	while (1) {
		int i = getopt_long(argc, argv, "v:p:e:cbBSaTFVxuUmhtlfAg:Go:Ok:KLX:D:E:", opts, &option_index);
		if (argc <= 1) {
			usage();
			exit(0);
			break;
		}
		if (i == -1) {
			break;
		}
		if (optarg && strlen(optarg) >= MAX_PARAM_LENGTH) {
			fprintf(stderr, "Too long parameter\n");
			exit(EXIT_FAILURE);
		}
		switch (i) {
		case 0:
			/*
			   if (optarg && strlen(optarg) >= MAX_PARAM_LENGTH){
			   fprintf(stderr, "Too long parameter\n");
			   exit(EXIT_FAILURE);
			   }
			 */
			break;

		case 'c':
			CPUFREQ_INFO = 1;
			break;
		case 'b':
			BATT_INFO = 1;
			break;
		case 'B':
			EXT_BATT_INFO = 1;
			break;
		case 'S':
			APM_ACPI = 1;
			break;
		case 'u':
			SUSPEND_TO_RAM = 1;
			break;
		case 'm':
			STANDBY = 1;
			break;
		case 't':
			THROT_INFO = 1;
			break;
		case 'f':
			PERFORMANCE_SPEED = 1;
			break;
		case 'l':
			POWERSAVE_SPEED = 1;
			break;
		case 'A':
			DYNAMIC_SPEED = 1;
			break;
		case 'g':
			DYNAMIC_PERFORMANCE = 1;
			option = strtol(optarg, endptr, 10);
			break;
		case 'G':
			GET_PERFORMANCE = 1;
			break;
		case 'o':
			SET_GOVERNOR = 1;
			strcpy(buffer, optarg);
			break;
		case 'O':
			GET_GOVERNOR = 1;
			break;
		case 'U':
			SUSPEND_TO_DISK = 1;
			break;
		case 'a':
			AC_INFO = 1;
			break;
		case 'T':
			THERMAL_INFO = 1;
			break;
		case 'e':
			SET_ACTIVE_SCHEME = 1;
			strcpy(buffer, optarg);
			break;
		case 'x':
			LIST_SCHEMES = 1;
			break;
		case 'X':
			GET_SCHEME_DESCRIPTION = 1;
			strcpy(buffer, optarg);
			break;
		case 'F':
			FAN_INFO = 1;
			break;
		case 'p':
			SET_THROT_PERC = 1;
			option = strtol(optarg, endptr, 10);
			if (**endptr != '\0' || option < 0 || option > 100) {
				printf("Wrong throttling in percent value,"
				       " must be between 0-100: %s\n", strerror(errno));
				exit(EXIT_FAILURE);
			}
			break;
		case 'k':
			SET_BRIGHTNESS = 1;
			if (*optarg == 'u')
				option = -1;	// up
			else if (*optarg == 'd')
				option = -2;	// down
			else
				option = strtol(optarg, endptr, 10);
			break;
		case 'K':
			GET_BRIGHTNESS = 1;
			break;
		case 'L':
			GET_BRIGHTNESS_LEVELS = 1;
			break;
		case 'E':
			CPU_ENABLE = 1;
			option = strtol(optarg, endptr, 10);
			break;
		case 'D':
			CPU_DISABLE = 1;
			option = strtol(optarg, endptr, 10);
			break;
		case 'v':
			verbose = strtol(optarg, endptr, 10);
			if (**endptr != '\0' || verbose < 0 || verbose > 36) {
				printf("Wrong verbose(-v, debug) paramater: %s\n", optarg);
				exit(EXIT_FAILURE);
			} else {
				setDebugLevel(verbose);
				//fprintf(stderr, "Debug Level set to %d\n", verbose);
			}
			break;
		case 'V':
			VERSION_INFO = 1;
			break;
		case 'h':
			usage();
			exit(0);
			break;
		}
	}

}

/* mainly copied from liblazy_dbus_send_method_call implementation. Just
 * added var_args list and INT_MAX as timeout */
static int send_sleep_request(const char *method, int first_arg_type, ...)
{
	DBusError	dbus_error;
	DBusConnection	*dbus_connection	= NULL;
	DBusMessage	*message		= NULL;
	DBusMessage	*reply			= NULL;
	int		ret			= 0;
	va_list		var_args;

	dbus_error_init(&dbus_error);

	dbus_connection = dbus_bus_get(DBUS_BUS_SYSTEM, &dbus_error);
	if (dbus_connection == NULL || dbus_error_is_set(&dbus_error)) {
		fprintf(stderr, "Connection to dbus not ready, skipping method call %s: %s\n",
			method, dbus_error.message);
		ret = -1;
		goto Free_Error;
	}

	message = dbus_message_new_method_call(DBUS_HAL_INTERFACE, HAL_UDI_COMPUTER,
					       DBUS_HAL_SYSTEM_POWER_INTERFACE, method);
	va_start(var_args, first_arg_type);
	dbus_message_append_args_valist(message, first_arg_type, var_args);
	va_end(var_args);

	reply = dbus_connection_send_with_reply_and_block(dbus_connection,
							  message, INT_MAX,
							  &dbus_error);
	if (dbus_error_is_set(&dbus_error)) {
		/* ignore timeout errors because it's possible that we are suspending > 6 hours */
		if (!dbus_error_has_name(&dbus_error, "org.freedesktop.DBus.Error.NoReply")) {
			fprintf(stderr, "Received error reply: %s\n", dbus_error.message);
			ret = -1;
		}
	}
	
	dbus_message_unref(message);
	dbus_connection_unref(dbus_connection);
Free_Error:
	dbus_error_free(&dbus_error);
	return ret;
}

int get_bat_info(int ext)
{
	BatteryGeneral bg;
	int ac_state = 0;

	// test of general battery information:
	if (getBatteriesInfo(&bg) < 0) {
		fprintf(stderr, "Could not get battery information.");
		return -1;
	} else {
		ac_state = getACAdapterStatus();
		if (bg.remaining_percent >= 0) {
			fprintf(stdout, "All primary batteries:\n");

			fprintf(stdout, "\tRemaining percent: %d\n", bg.remaining_percent);

			if (ac_state == AC_ONLINE && bg.remaining_minutes > 0) {
				fprintf(stdout, "\t%d minutes until fully charged\n", bg.remaining_minutes);
			} else if (ac_state == AC_OFFLINE && bg.remaining_minutes > 0) {
				fprintf(stdout, "\tRemaining minutes: %d\n", bg.remaining_minutes);
			}

			printf("\tCharging state: ");
		        if (bg.charging_state & CHARG_STATE_CHARGING) {
                                fprintf(stdout, "charging");
                        }
                        if (bg.charging_state & CHARG_STATE_DISCHARGING) {
                                fprintf(stdout, "discharging");
			}

			fprintf(stdout, "\n");
		}		

		printf("AC state: ");
		if (ac_state == AC_ONLINE)
			fprintf(stdout, "online\n");
		else 
			fprintf(stdout, "offline\n");
	}

	// Test of detailed battery information
	if (ext) {
		char **batteries = NULL;
		int error = liblazy_hal_find_device_by_capability("battery", &batteries);
		
		if (error) {
			pDebug(DBG_INFO, "Could not get list of batteries in system");
			liblazy_free_strlist(batteries);
			return 0;
		}

		/* if there are no batteries */
		if (batteries == NULL && *batteries == NULL) {
			liblazy_free_strlist(batteries);
			fprintf(stderr, "No batteries found\n");
			return -1;
		}
		
		for (int i = 0; batteries[i] != NULL; ++i) {
			char *type;
			error = liblazy_hal_get_property_string(batteries[i], "battery.type", &type);
			printf("Battery %d:\n", i + 1);
			printf("\tType: %s\n", type);
			liblazy_free_string(type);
			int present;
			error = liblazy_hal_get_property_bool(batteries[i], "battery.present", &present);
			
			if (!present) {
				printf("\tNot present\n");
				continue;
			}

			int remaining_percent;
			liblazy_hal_get_property_int(batteries[i],
						     "battery.charge_level.percentage",
						     &remaining_percent);

			fprintf(stdout, "\tRemaining percent: %d\n", remaining_percent);

			int remaining_time;
			liblazy_hal_get_property_int(batteries[i], "battery.remaining_time", &remaining_time);

			if (remaining_time >= 0 ) {
				fprintf(stdout, "\tRemaining minutes: %d\n", remaining_time / 60 );
			}
			
			int charging, discharging;
			liblazy_hal_get_property_bool(batteries[i],
						      "battery.rechargeable.is_charging", &charging);
			liblazy_hal_get_property_bool(batteries[i],
						      "battery.rechargeable.is_discharging",
						      &discharging);
			printf("\tCharging state: ");
			if (charging)
				fprintf(stdout, "charging\n");
			else if (discharging)
				fprintf(stdout, "discharging\n");
			else 
				fprintf(stdout, "neither charging nor discharging\n");
		}
	}
	return 1;
}

string get_brightness_udi()
{
	char **panels;
	int error = liblazy_hal_find_device_by_capability("laptop_panel", &panels);

	if (error) {
		hal_error();
		return string();
	}

	if (panels == NULL || panels[0] == NULL) {
		pDebug(DBG_INFO, "No Laptop Panels found\n");
		return string();
	}
	string udi = panels[0];
	liblazy_free_strlist(panels);
	return udi;
}

int get_brightness_levels()
{
	string udi = get_brightness_udi();
	if (udi.empty())
		return -1;

	char prop[] = "laptop_panel.num_levels";
	int num_levels;
	if (!liblazy_hal_get_property_int((char*)udi.c_str(), prop, &num_levels))
		return num_levels;
	else
		return -1;
}

int get_brightness()
{
	int brightness = -1;
	string udi = get_brightness_udi();
	DBusMessage *reply;
	int error;

	if (udi.empty()) {
		return -1;
	}

	error = liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
						     udi.c_str(),
						     HAL_DEVICE_LAPTOP_PANEL,
						     "GetBrightness",
						     &reply,
						     DBUS_TYPE_INVALID);
	if (error)
		pDebug(DBG_INFO, "Could not get current brightness from HAL");

	liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_INT32, &brightness, 0);

	return brightness;

}

bool set_brightness(int level)
{
	int error;
	string udi = get_brightness_udi();
	if (udi.empty())
		return false;

	int num_levels = get_brightness_levels();

	if (level > num_levels - 1)
		level = num_levels - 1;
	else if (level < 0)
		level = 0;


	error = liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
						     udi.c_str(),
						     HAL_DEVICE_LAPTOP_PANEL,
						     "SetBrightness",
						     NULL,
						     DBUS_TYPE_INT32,
						     &level,
						     DBUS_TYPE_INVALID);
	if (error) {
		pDebug(DBG_INFO, "Could not set brightness");
		return false;
	}

	return true;
}

 /** @brief main function
  *
  * errors:
  *     - -1 for parameter specific error
  *     - -2 if user has no rights to connect to daemon
  *     - -3 if socket conn could not be established (no daemon...)
 */
 int main(int argc, char **argv)
 {
	 int ret = 0;
	 int states, current, err_code;
	 unsigned int min, max;
	 ThermalDev *thermal_dev;
	 min = max = 0;
	 getArgs(argc, argv);
	 checkArgs();
	 int y, x = 0;

	 DBusMessage *reply;
	 DBusError error;
	 dbus_error_init(&error);

	 string result;
	 char *dummy;

	 if (SUSPEND_TO_RAM) {
		 int supported;
		 if (liblazy_hal_get_property_bool(HAL_UDI_COMPUTER,
						   "power_management.can_suspend", &supported)
			 || supported != 1) {
			 printf("Suspend to ram is not supported.\n");
			 ret = -1;
		 } else {
			 int wake_up = 0;
			 if (liblazy_hal_is_caller_privileged("org.freedesktop.hal.power-management.suspend") != 1)
				 privilege_error("org.freedesktop.hal.power-management.suspend");
			 else if (send_sleep_request("Suspend",
						     DBUS_TYPE_INT32,
						     &wake_up,
						     DBUS_TYPE_INVALID)) {
				 fprintf(stderr, "Could not send suspend to ram request to HAL\n");
				 ret = -1;
			 } else
				 printf("Suspend to ram request sent.\n");
		 }
	 }
	 if (STANDBY) {
		 int supported;
		 if (liblazy_hal_get_property_bool(HAL_UDI_COMPUTER,
						    "power_management.can_standby", &supported)
			 || supported != 1) {
			 printf("Standby is not supported.\n");
			 ret = -1;
		 } else {
			 if (liblazy_hal_is_caller_privileged("org.freedesktop.hal.power-management.standby") != 1)
				 privilege_error("org.freedesktop.hal.power-management.standby");
			 else if (send_sleep_request("Standby", DBUS_TYPE_INVALID)) {
				 fprintf(stderr, "Could not send standby request to HAL\n");
				 ret = -1;
			 } else
				 printf("Standby request sent.\n");

		 }
	 } else if (SUSPEND_TO_DISK) {
		 int supported;
		 if (liblazy_hal_get_property_bool(HAL_UDI_COMPUTER,
					       "power_management.can_hibernate", &supported)
			 || supported != 1) {
			 printf("Suspend to disk is not supported.\n");
			 ret = -1;
		 } else {
			 if (liblazy_hal_is_caller_privileged("org.freedesktop.hal.power-management.hibernate") != 1)
				 privilege_error("org.freedesktop.hal.power-management.hibernate");

			 else if (send_sleep_request("Hibernate", DBUS_TYPE_INVALID)) {
				 fprintf(stderr, "Could not send suspend to disk request to HAL\n");
				 ret = -1;
			 } else
				 printf("Suspend to disk request sent.\n");
		 }
	 }
	 if (PERFORMANCE_SPEED) {
		 const char *governor = "performance";
		 if (liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
							  HAL_UDI_COMPUTER,
							  DBUS_HAL_CPUFREQ_INTERFACE,
							  "SetCPUFreqGovernor",
							  &reply,
							  DBUS_TYPE_STRING,
							  &governor,
							  DBUS_TYPE_INVALID)) {
			 fprintf(stderr, "Could not set CPUFreq policy to performance.\n");
			 ret = -1;
		 } else
			 printf("Policy set to performance\n");
	 } else if (POWERSAVE_SPEED) {
		 const char *governor = "powersave";
		 if (liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
							  HAL_UDI_COMPUTER,
							  DBUS_HAL_CPUFREQ_INTERFACE,
							  "SetCPUFreqGovernor",
							  &reply,
							  DBUS_TYPE_STRING,
							  &governor,
							  DBUS_TYPE_INVALID)) {
			 fprintf(stderr, "Could not set CPUFreq policy to powersave.\n");
			 ret = -1;
		 } else
			 printf("Policy set to powersave\n");
	 } else if (DYNAMIC_SPEED) {
		 const char *governor = "ondemand";
		 if (liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
							  HAL_UDI_COMPUTER,
							  DBUS_HAL_CPUFREQ_INTERFACE,
							  "SetCPUFreqGovernor",
							  &reply,
							  DBUS_TYPE_STRING,
							  &governor,
							  DBUS_TYPE_INVALID)) {
			 const char *gov2 = "userspace";
			 if (liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
								  HAL_UDI_COMPUTER,
								  DBUS_HAL_CPUFREQ_INTERFACE,
								  "SetCPUFreqGovernor",
								  &reply,
								  DBUS_TYPE_STRING,
								  &gov2,
								  DBUS_TYPE_INVALID)) {
				 fprintf(stderr, "Could not set CPUFreq policy to dynamic.\n");
				 ret = -1;
			 }
		 } else
			 printf("Policy set to dynamic\n");
	 } else if (DYNAMIC_PERFORMANCE) {
		 if (liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
							  HAL_UDI_COMPUTER,
							  DBUS_HAL_CPUFREQ_INTERFACE,
							  "SetCPUFreqPerformance",
							  &reply,
							  DBUS_TYPE_INT32,
							  &option,
							  DBUS_TYPE_INVALID)) {
			 fprintf(stderr, "Could not set performance.\n");
			 ret = -1;
		 } else
			 printf("Performance set to %d\n", option);
	 } else if (GET_PERFORMANCE) {
		 int perf = -1;
		 if (!liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
							   HAL_UDI_COMPUTER,
							   DBUS_HAL_CPUFREQ_INTERFACE,
							   "GetCPUFreqPerformance",
							   &reply,
							   DBUS_TYPE_INVALID)
		     && !liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_INT32, &perf, 0)) {
			 printf("Current performance: %d\n", perf);
		 } else {
			 fprintf(stderr, "Could not get performance for current governor.\n");
			 ret = -1;
		 }
	 } else if (SET_GOVERNOR) {
		 char *gov = buffer;
		 if (liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
							  HAL_UDI_COMPUTER,
							  DBUS_HAL_CPUFREQ_INTERFACE,
							  "SetCPUFreqGovernor",
							  &reply,
							  DBUS_TYPE_STRING,
							  &gov,
							  DBUS_TYPE_INVALID)) {
			 fprintf(stderr, "Could not set CPUFreq governor to %s.\n", buffer);
			 ret = -1;
		 } else
			 printf("CPUFreq governor set to %s\n", gov);
	 } else if (GET_GOVERNOR) {
		 char *gov;
		 if (!liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
							   HAL_UDI_COMPUTER,
							   DBUS_HAL_CPUFREQ_INTERFACE,
							   "GetCPUFreqGovernor",
							   &reply,
							   DBUS_TYPE_INVALID)
		     && !liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_STRING, &gov, 0)) {
			 printf("Current CPUFreq governor: %s\n", gov);
		 } else {
			 fprintf(stderr, "Could not get current CPUFreq governor.\n");
			 ret = -1;
		 }
	 }
	 if (CPUFREQ_INFO) {
		 char *governor = NULL;

		 if (!liblazy_dbus_system_send_method_call(DBUS_HAL_INTERFACE,
							   HAL_UDI_COMPUTER,
							   DBUS_HAL_CPUFREQ_INTERFACE,
							   "GetCPUFreqGovernor",
							   &reply,
							   DBUS_TYPE_INVALID)
		     && !liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_STRING, &governor, 0)) {
			 if (governor != NULL)
				 if (!strcmp(governor, "ondemand")
					|| !strcmp(governor, "userspace")
					|| !strcmp(governor, "conservative"))
					 cout << "DYNAMIC" << endl;
				 else if (!strcmp(governor, "powersave"))
					 cout << "POWERSAVE" << endl;
				 else if (!strcmp(governor, "performance"))
					 cout << "PERFORMANCE" << endl;
				 else
					 cout << "Unknown CPUFreq policy" << endl;

		 } else {
			 fprintf(stderr, "Could not get current CPUFreq policy.\n");
			 ret = -1;
		 }
	 }
	 if (SET_ACTIVE_SCHEME) {
		 if (strlen(buffer) != 0) {
			 dummy = buffer;

			 if (!strcmp(buffer, "AdvancedPowersave")) {
				 int a;
				 fprintf(stdout, "About to activate AdvancedPowersave scheme. This scheme "
					 "is highly experimental. Really activate it? (y/n) ");

				 a = getchar();

				 if (tolower(a) != 'y') {
					 fprintf(stdout, "Scheme AdvancedPowersave not set\n");
					 exit(EXIT_SUCCESS);
				 }

			 }

			 err_code = liblazy_dbus_system_send_method_call(PS_DBUS_SERVICE,
									 PS_DBUS_PATH,
									 PS_DBUS_INTERFACE,
									 "SchemesSet",
									 &reply,
									 DBUS_TYPE_STRING,
									 &dummy,
									 DBUS_TYPE_INVALID);

			 liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_UINT16, &err_code, 0);

			 switch (err_code) {
			 case REPLY_SUCCESS:
				 printf("Scheme %s successfully activated\n", buffer);
				 break;
			 case REPLY_ALREADY_SET:
				 printf("Scheme %s already active\n", buffer);
				 break;
			 case REPLY_INVALID_PARAM:
				 printf("Scheme %s does not exist\n", buffer);
				 ret = -1;
				 break;
			 case REPLY_NO_RIGHTS:
				 no_connect_no_rights_error();
				 ret = -2;
				 break;
			 default:
				 printf("Error.\n");
				 general_error();
			 }
		 }
	 }
	 if (LIST_SCHEMES) {
		 char *_schemes;
		 int active;
		 int battery;
		 int ac_power;

		 err_code = liblazy_dbus_system_send_method_call(PS_DBUS_SERVICE,
								 PS_DBUS_PATH,
								 PS_DBUS_INTERFACE,
								 "SchemesGet",
								 &reply,
								 DBUS_TYPE_INVALID);
		 liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_UINT16, &err_code, 0);

		 if (err_code == REPLY_SUCCESS) {
			 if (!liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_INT32, &active, 0)
			     && !liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_INT32, &battery, 1)
			     && !liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_INT32, &ac_power, 2))
				if (err_code < 0) {
					fprintf(stderr, "Could not connect with daemon\n");
				} else {	
					for (int x = 0;
					     !liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_STRING,
										 &_schemes, x)
						     && dummy != NULL; x++) {
						printf("%s%s%s%s\n",
						       _schemes,
						       ((active == x) ? "\tactive" : ""),
						       ((battery == x) ? "\tbattery_default_scheme" : ""),
						       ((ac_power == x) ? "\tAC_default_scheme" : ""));
					}
				} else {
					fprintf(stderr, "Could not assign schemes to current/battery/ac_power\n");
					ret = -1;
				}
		} else if (err_code == REPLY_NO_RIGHTS) {
			no_connect_no_rights_error();
			ret = -2;
		} else {
			general_error();
			ret = -1;
		}
	 }
	 if (GET_SCHEME_DESCRIPTION) {
		 dummy = buffer;
		 char *description;

		 err_code = liblazy_dbus_system_send_method_call(PS_DBUS_SERVICE,
								 PS_DBUS_PATH,
								 PS_DBUS_INTERFACE,
								 "SchemesGetDescription",
								 &reply,
								 DBUS_TYPE_STRING,
								 &dummy,
								 DBUS_TYPE_INVALID);

		 liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_UINT16, &err_code, 0);

		 switch (err_code) {
		 case REPLY_INVALID_PARAM:
			 printf("Error: Scheme %s does not exist\n", buffer);
			 ret = -1;
			 break;
		 case REPLY_SUCCESS:
			 if (!liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_STRING, &description, 0)) {
				 printf("%s\n", description);
			 }
			 else {
				 general_error();
			 }
			 break;
		 default:
			 general_error();
                 }
	 }
	 if (APM_ACPI) {
		 switch (checkACPI()) {
		 case APM:
			 fprintf(stdout, "APM\n");
			 ret = 2;
			 break;
		 case ACPI:
			 fprintf(stdout, "ACPI\n");
			 ret = 1;
			 break;
		 default:
			 fprintf(stdout, "No ACPI/APM support \n");
			 ret = -1;
		 }
	 }
	 if (BATT_INFO) {
		 if (get_bat_info(0) < 0) {
			 fprintf(stderr, "Could not read out battery info properly\n");
			 ret = -2;
		 }
	 }
	 if (EXT_BATT_INFO) {
		 if (get_bat_info(1) < 0) {
			 fprintf(stderr, "Could not read out battery info properly\n");
			 ret = -2;
		 }
	 }
	 if (AC_INFO) {
		 states = getACAdapterStatus();
		 if (states == AC_ONLINE) {
			 fprintf(stdout, "ONLINE\n");
			 ret = 2;
		 } else if (states == AC_OFFLINE) {
			 fprintf(stdout, "OFFLINE\n");
			 ret = 1;
		 } else {
			 fprintf(stderr, "AC state is unknown, assume ONLINE\n");
			 ret = -1;
		 }
	 }
	 if (THROT_INFO) {
		 for (x = 0; getThrottlingInfoCPU(x, &states, &current) > 0; x++) {
			 fprintf(stdout, "CPU: %d\nThrottling states: %d\nCurrent state: %d\n", x, states, current);
		 }
		 if (x == 0)
			 fprintf(stderr, "Throttling not supported \n");
	 }
	 if (SET_THROT_PERC) {
		 if (getThrottlingInfo(&states, &current) < 0) {
			 fprintf(stderr, "Throttling not supported\n");
			 ret = -1;
		 } else {
			 for (x = 0; getThrottlingInfoCPU(x, &states, &current) > 0; x++) {
				 fprintf(stdout, "CPU %d:\nThrottling states : %d\nCurrent state: %d\n", x, states,
					 current);
				 if (setThrottlingPercentCPU(x, option) < 0) {
					 fprintf(stderr, "Cannot set throttling state to %d percent for cpu %d\n", option,
						 x);
					 ret = -1;
					 break;
				 }
			 }
			 usleep(100);
			 fprintf(stdout, "Changed to:\n");
			 for (y = 0; getThrottlingInfoCPU(y, &states, &current) > 0; y++) {
				 fprintf(stdout, "CPU %d:\nThrottling states : %d\nCurrent state: %d\n", y, states,
					 current);
			 }
		 }
	 }
	 if (THERMAL_INFO) {
		 err_code = getThermalZonesNum();
		 if (err_code < 0) {
			 fprintf(stderr, "No thermal devices available: %d\n", err_code);
			 ret = -1;
		 }
		 thermal_dev = (ThermalDev *) malloc(sizeof(ThermalDev));
		 for (x = 0; x < err_code && !(getThermalZone(x, thermal_dev) < 0); x++) {
			 fprintf(stdout, "Thermal Device no. %d:\n", x);
			 if (thermal_dev->temperature > 0)
				 fprintf(stdout, "Temperature: %d\n", thermal_dev->temperature);
			 if (thermal_dev->state > 0) {
				 fprintf(stdout, "State: ");
				 switch (thermal_dev->state) {
				 case HOT:
					 fprintf(stdout, "HOT\n");
					 break;
				 case CRITICAL:
					 fprintf(stdout, "CRITICAL\n");
					 break;
				 case PASSIVE:
					 fprintf(stdout, "PASSIVE\n");
					 break;
				 case ACTIVE:
					 fprintf(stdout, "ACTIVE\n");
					 break;
				 default:
					 fprintf(stdout, "UNDEFINED\n");
				 }
			 }
			 if (thermal_dev->critical > 0)
				 fprintf(stdout, "Critical: %d\n", thermal_dev->critical);
			 if (thermal_dev->hot > 0)
				 fprintf(stdout, "Hot: %d\n", thermal_dev->hot);
			 if (thermal_dev->passive > 0)
				 fprintf(stdout, "Passive: %d\n", thermal_dev->passive);
			 for (y = 0; y < MAX_THERMAL_ACTIVE; y++) {
				 if (thermal_dev->active[y] > 0)
					 fprintf(stdout, "Active %d: %d\n", y, thermal_dev->active[y]);
			 }
		 }
		 free(thermal_dev);
	 }
	 if (FAN_INFO) {
		 err_code = getFanNum();
		 if (err_code <= 0) {
			 fprintf(stderr, "No fan devices available.\n");
			 ret = -1;
		 } else {
			 //for (x = 0; x < err_code && !((current = getFanStatus(x)) < 0); x++) //stops at first fan /w error
			 for (x = 0; x < err_code; x++) {
				 if (!((current = getFanStatus(x)) < 0))
					 printf("Fan no %d: %s\n", x, current ? "off" : "on");
				 else
					 printf("Fan no %d: unknown\n",x);
			 }
		 }
	 }
	 if (SET_BRIGHTNESS) {
		 int brightness = get_brightness();
		 if (brightness < 0)
			 exit(EXIT_FAILURE);
		 int levels = get_brightness_levels();

		 switch (option) {
		 case -1: {
			 brightness = get_brightness();
			 if (brightness >= levels -1) {
				 printf("Brightness already at max\n");
				 brightness = -1;
			 } else 
				 brightness++;
			 break;
		 }
		 case -2:
			 brightness = get_brightness();
			 if (brightness <= 0) {
				 printf("Brightness already at min\n");
				 brightness = -1;
			 } else 
				 brightness--;
			 break;
		 default:
			 brightness = option;
			 break;
		 }

		 if (brightness < 0 || brightness > levels - 1) {
			 fprintf(stderr, "Brightness must be between 0 and %d\n", levels - 1);
			 ret = -1;
		 } else {
			 if (!set_brightness(brightness)) {
				 fprintf(stderr, "Setting of brightness not supported.\n");
				 ret = -1;
			 } else
				 fprintf(stderr, "Brightness level set to %d.\n", brightness);
		 }
	 }
	 if (GET_BRIGHTNESS) {
		 int level = get_brightness();
		 if (level < 0) {
			 fprintf(stderr, "Brightness not supported.\n");
			 ret = -1;
		 } else {
			 printf("Current brightness level: %d.\n", level);
		 }
	}
	if (GET_BRIGHTNESS_LEVELS) {
		int l = get_brightness_levels();
		if (l < 0) {
			fprintf(stderr, "Brightness not supported.\n");
			ret = -1;
		} else {
			printf("Available brightness levels: %d.\n", l);
		}
	}
	if (CPU_ENABLE) {
		const char *method = "CpufreqCPUEnable";

		err_code = liblazy_dbus_system_send_method_call(PS_DBUS_SERVICE,
								PS_DBUS_PATH,
								 PS_DBUS_INTERFACE,
								 method,
								 &reply,
								 DBUS_TYPE_INT32,
								 &option,
								 DBUS_TYPE_INVALID);

		liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_UINT16, &err_code, 0);

		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			fprintf(stderr, "CPU hotplugging not supported for this CPU.\n");
			ret = -1;
			break;
		case REPLY_ALREADY_SET:
			fprintf(stderr, "CPU already enabled.\n");
			//noerror
			break;
		case REPLY_SUCCESS:
			fprintf(stderr, "CPU %d enabled.\n", option);
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		default:
			general_error();
			ret = -1;
		}
	}
	if (CPU_DISABLE) {
		const char *method = "CpufreqCPUDisable";

		err_code = liblazy_dbus_system_send_method_call(PS_DBUS_SERVICE,
								PS_DBUS_PATH,
								PS_DBUS_INTERFACE,
								method,
								&reply,
								DBUS_TYPE_INT32,
								&option,
								DBUS_TYPE_INVALID);

		liblazy_dbus_message_get_basic_arg(reply, DBUS_TYPE_UINT16, &err_code, 0);
		printf("err_code: %d\n", err_code);
		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			fprintf(stderr, "CPU hotplugging not supported for this CPU.\n");
			ret = -1;
			break;
		case REPLY_ALREADY_SET:
			fprintf(stderr, "CPU already disabled.\n");
			//noerror
			break;
		case REPLY_SUCCESS:
                        fprintf(stderr, "CPU %d disabled.\n", option);
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		default:
			general_error();
                       ret = -1;
		}
	}
	if (VERSION_INFO) {
#ifdef VERSION
		printf("Powersave package version: %s\n", VERSION);
#else
		printf("Version has not been defined by VERSION at compile time.\n");
#endif
	}
	if (ret == -1)
		exit(EXIT_FAILURE);
	else
		exit(ret);
}
