#ifndef _RHEO_CHARACTERISTIC_H
#define _RHEO_CHARACTERISTIC_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/riesz.h"

namespace rheolef { 

// -------------------------------------------------------------------
// characteristic_on_quadrature: store the localization in mesh of the 
// quadrature point tacking: y = x + d(x) : that belongs in K as hat_y
// -------------------------------------------------------------------
template <class T, class M>
struct characteristic_on_quadrature_rep {
// allocators:
  characteristic_on_quadrature_rep (quadrature_option_type qopt1 = quadrature_option_type(quadrature_option_type::max_family,0))
  : _qopt         (qopt1),
    _quad         (),
    _piola_on_quad(),
    _ie2dis_ix    (),
    _hat_y        (),
    _yq           ()
  {}

  characteristic_on_quadrature_rep (const characteristic_on_quadrature_rep<T,M>& x)
  : _qopt         (x._qopt),
#ifdef TO_CLEAN
    _quad         (x._quad),
    _piola_on_quad(x._piola_on_quad),
    _ie2dis_ix    (x._ie2dis_ix),
    _hat_y        (x._hat_y),
    _yq           (x._yq)
#endif // TO_CLEAN
    _quad         (),
    _piola_on_quad(),
    _ie2dis_ix    (),
    _hat_y        (),
    _yq           ()
  {
    error_macro ("physical copy of characteristic_on_quadrature_rep may not happend"); 
  }

public:
// data:
  quadrature_option_type        _qopt;
  quadrature<T>                 _quad;
  basis_on_pointset<T>          _piola_on_quad;
  array<index_set,M>            _ie2dis_ix;
  array<point_basic<T>,M>       _hat_y;
  array<point_basic<T>,M>       _yq;
};
template <class T, class M>
class characteristic_on_quadrature : public smart_pointer<characteristic_on_quadrature_rep<T,M> > {
public:
// typedefs:
  typedef characteristic_on_quadrature_rep<T,M> rep;
  typedef smart_pointer<rep>                    base;
// allocator:
  characteristic_on_quadrature
    (quadrature_option_type qopt = quadrature_option_type(quadrature_option_type::max_family,0))
  : base (new_macro(rep(qopt)))
  {}
// data:
};
// -------------------------------------------------------------------
// characteristic_rep: store a list of characteristic_on_quadrature(s)
// together with the displacement field dh
// -------------------------------------------------------------------
template<class T, class M>
class characteristic_rep {
public:
  typedef std::map<std::string,characteristic_on_quadrature<T,M> > map_type;

// allocator:

  characteristic_rep(const field_basic<T,M>& dh)
  : _dh(dh),
    _coq_map()
  {}

// accesor:

  const field_basic<T,M>& get_displacement() const { return _dh; }

  const characteristic_on_quadrature<T,M>&
  get_pre_computed (
    const space_basic<T,M>&       Xh,
    const field_basic<T,M>&       dh,
    const quadrature_option_type& qopt) const;

// data:
protected:
  field_basic<T,M>  _dh;
  mutable map_type  _coq_map;
};

/*Class:characteristic
NAME: @code{characteristic} - the Lagrange-Galerkin method implemented
@cindex  Lagrange-Galerkin method
@cindex  method of characteristic
@cindex  quadrature formulae
@findex  riesz
@clindex space
@clindex geo

SYNOPSYS:
 The class characteristic implements the Lagrange-Galerkin method:
 It is the extension of the method of characteristic from the finite
 difference to the finite element context.
EXAMPLE:
@noindent
 The following code compute the Riesz representer (see @ref{riesz algorithm}),
 denoted by @code{lh} of u(x)=uh(x+dh(x)) where @code{ah} is the
 deformation vector field. The deformation field @code{dh=-dt*uh} 
 in Lagrange-Galerkin methods, where @code{ah} is the advection field
 and @code{dt} a time step.
 @example
  geo omega;
  field dh = ...;
  field uh = ...;
  characteristic X (dh);
  field lh = riesz(Xh, compose(uh, X));
 @end example
 The Riesz representer is the @code{lh} vector field defined by:
 @example
	lh(i) = integrate uh(x+dh(x)) phi_i(x) dx
 @end example
 where phi_i is the i-th basis function in the space Xh
 and the integral is evaluated by using a quadrature formulae.
 By default the quadrature formule is Gauss-Lobatto with
 the order equal to the polynomial order of Xh
 (order 1: trapeze, order 2: simpson, etc).
 Recall that this choice of quadrature formulae guaranties inconditional
 stability at any polynomial order.
 Alternative quadrature formulae or order can be used by using the 
 additional quadrature option argument to the @code{riesz} function.
End: */

//<characteristic:
template<class T, class M = rheo_default_memory_model>
class characteristic_basic : public smart_pointer<characteristic_rep<T,M> > {
public:
  typedef characteristic_rep<T,M> rep;
  typedef smart_pointer<rep>      base;

// allocator:

  characteristic_basic(const field_basic<T,M>& dh);

// accesors:

  const field_basic<T,M>& get_displacement() const;

  const characteristic_on_quadrature<T,M>&
  get_pre_computed (
    const space_basic<T,M>&       Xh,
    const field_basic<T,M>&       dh,
    const quadrature_option_type& qopt) const;

};
typedef characteristic_basic<Float> characteristic;
//>characteristic:

template<class T, class M>
inline
characteristic_basic<T,M>::characteristic_basic (const field_basic<T,M>& dh)
 : base (new_macro(rep(dh)))
{
}
template<class T, class M>
inline
const field_basic<T,M>&
characteristic_basic<T,M>::get_displacement() const
{
  return base::data().get_displacement();
}
template<class T, class M>
inline
const characteristic_on_quadrature<T,M>&
characteristic_basic<T,M>::get_pre_computed (
    const space_basic<T,M>&       Xh,
    const field_basic<T,M>&       dh,
    const quadrature_option_type& qopt) const
{
  return base::data().get_pre_computed (Xh,dh,qopt);
}
// ===========================================================================
// the temporary class returned by compose (phi_h,X)
// ===========================================================================
template<class T, class M>
class field_o_characteristic {
public:

// allocator:

  field_o_characteristic(const field_basic<T,M>& uh, const characteristic_basic<T,M>& X)
    : _uh(uh), _X(X) {}

// accesor:

  const field_basic<T,M>&           get_field() const { return _uh; }
  const characteristic_basic<T,M>&  get_characteristic()  const { return _X; }

// data:
protected:
  field_basic<T,M>          _uh;
  characteristic_basic<T,M> _X;
};
// ===========================================================================
// compose (uh,X) returns a temporary:
// ===========================================================================
//<characteristic:
template<class T, class M>
inline
field_o_characteristic<T,M>
compose (const field_basic<T,M>& uh, characteristic_basic<T,M>& X)
//>characteristic:
{
  return field_o_characteristic<T,M>(uh, X);
}
// ===========================================================================
// riesz(Vh,compose (phi_h,X)) returns a field:
// => specialisation of the riesz function:
// ===========================================================================
//<characteristic:
template <class T, class M>
field_basic<T,M>
riesz (
    const space_basic<T,M>&             Xh,
    const field_o_characteristic<T,M>&  f,
    quadrature_option_type              qopt
       = quadrature_option_type(quadrature_option_type::max_family,0));
//>characteristic:

}// namespace rheolef
#endif // _RHEO_CHARACTERISTIC_H
