/* get_default_type.c - contains implementation of functions to get a default
 *                  type to use for a role.  Used by the newrole program.
 */

#include "get_default_type.h"

/*
 * get_default_type - given a role of length len, sets def_type to a
 *		      the name of the default type to use for that role
 *		      Returns 1 if a type was found, 0 otherwise.
 */
int get_default_type (const char* role, const int len, char** type)
{
  FILE* fdeftype=NULL;			 /* default type file */
    
  if (type==NULL || role==NULL)
    return 0;

  /* Open file */
  if (!(fdeftype = fopen (_DEFTYPE_PATH, "r"))) {
    return -1;
  }
  else if (!find_default_type (fdeftype, role, len, type)) {
    /* Obtain default type - set to NULL if not found */
    *type=NULL;
    return -1;
  }

  if ( fdeftype!=NULL )
    fclose (fdeftype);

  /* Success */
  return 0;
}


/* find_default_type - given a pointer to the default type file,
 *		       finds the line of the file that corresponds
 *		       to the role.  That line without any 
 *		       leading blanks is set to be the default security
 *                     context.
 *                     Returns 1 if a line matching the role is found, 0
 *		       otherwise
 */
int find_default_type (FILE* fdeftype, const char* role, const int len, 
                      char** type)
{
    char buf[250];    /* contains the line read from the file       */
    int i = 0;        /* index into buf                             */
    int found = 0;    /* true if the line corresponding to the role
                         has been found                             */

    /* Make sure to start at the beginning of the file */
    rewind (fdeftype);

    /* Search line by line until the role is found or eof is reached */
    while ((!found) && (!feof (fdeftype)))
    {
        fgets (buf, 250, fdeftype);
        i = 0;

        /* Make sure there are no leading spaces in the buffer */
        while (buf[i] == ' ')
            i++;
    
        if ((strncmp (role, &buf[i], len) == 0) && (buf[i+len] == ':'))
            found = 1;  /* This line goes with the role */
    }

    if (found)  /* A line matching the role has been found */
    {
        /* malloc space for the type */
        (*type) = (char*) malloc (sizeof(char) * (strlen(buf)-i-len));
        if ((*type) == NULL)
        {
            fprintf (stderr, "Unable to malloc\n");
            return 0;
        }
	
        /* copy the line into the space pointed to by type */
        strcpy ((*type), &buf[i]+len+1);
        if ((*type)[strlen(*type)-1] == '\n')
            (*type)[strlen(*type)-1] = '\0';
    }

    return found;
}
