/* compute_av.c
 *
 * Used to compute access vectors based on a SID pair for the permissions
 * in a particular class.  
 * Input parameters are: ssid (source SID), tsid (target SID),
 *                       and tclass (target object class)
 * Output - prints the permissions that are allowed 
 */

/*
  Orig:    13 June 2000
  Revised: 25 July 2000
*/


#include <stdio.h>
#include <ss.h>
#include <errno.h>

#include <linux/flask/av_permissions.h>
#include <linux/flask/class_to_string.h>
#include <linux/flask/common_perm_to_string.h>
#include <linux/flask/av_inherit.h>
#include <linux/flask/av_perm_to_string.h>

struct av_idx {
	int c_idx;
	int p_idx;
	struct av_idx *nxt;
};

char *calloc();
char *myname;
FILE *errfp;

#define new_av_idx(x) {  \
                         (x)->nxt = (struct av_idx *) calloc(1, sizeof(struct av_idx));  \
                         (x) = (x)->nxt;   \
                         (x)->c_idx = -1;  \
                         (x)->p_idx = -1;  \
                      }

#define new_classname(x) calloc(1, strlen(x)+1)

static void Usage(FILE *, char *);
static void my_print_access_vector(FILE *, char **);
static char **get_av(security_class_t, access_vector_t);


char **compute_av(security_id_t, security_id_t, char *);



int main(int argc, char **argv)
{
	char **access_vector;

	if (argc != 4)
		Usage(stderr, argv[0]);

	myname = argv[0];

#ifdef ERRFP
	errfp = stderr;
#else
	errfp = (FILE *) NULL;
#endif

	access_vector =
	    compute_av((security_id_t) atoi(argv[1]),
		       (security_id_t) atoi(argv[2]), argv[3]);

	my_print_access_vector(stdout, access_vector);

	exit(access_vector ? 0 : 1);
}


char **compute_av(security_id_t ssid, security_id_t tsid, char *class)
{
	struct security_query sq;
	struct security_response sr;
	int rv, verbose, atoi();

	sq.tclass = string_to_security_class(class);
	if (!sq.tclass)
		return ((char **) NULL);

	sq.ssid = ssid;
	sq.tsid = tsid;
	sq.requested = 0;

	rv = security_compute_av(&sq, &sr);
	if (rv) {
		if (errfp) {
			(void) fprintf(errfp,
				       "%s: security_compute_av(): %s\n",
				       myname, strerror(errno));
			(void) fflush(errfp);
		}
		return ((char **) NULL);
	}

	return (get_av(sq.tclass, sr.allowed));
}


static char **get_av(security_class_t tclass, access_vector_t av)
{
	char **common_pts = 0;
	access_vector_t common_base = 0;
	int k, j, perm, nv;
	static struct av_idx av_idxs;
	struct av_idx *avp, *avp_trail;
	char **access_vector;
	char *access;


	if (av == 0) {		/* should not happen */
		if (errfp) {
			(void) fprintf(errfp,
				       "%s: get_av(tclass, NULL) == NULL\n",
				       myname);
			(void) fflush(errfp);
		}
		return ((char **) NULL);
	}

	for (k = 0; k < AV_INHERIT_SIZE; k++) {
		if (av_inherit[k].tclass == tclass) {
			common_pts = av_inherit[k].common_pts;
			common_base = av_inherit[k].common_base;
			break;
		}
	}

	av_idxs.c_idx = -1;
	av_idxs.p_idx = -1;
	av_idxs.nxt = (struct av_idx *) NULL;
	avp = &av_idxs;

	perm = 1;
	k = nv = 0;

	while (perm < common_base) {
		if (perm & av) {
			avp->c_idx = k;
			nv++;
			new_av_idx(avp);
		}
		k++;
		perm <<= 1;
	}

	while (k < sizeof(access_vector_t) * 8) {
		if (perm & av) {
			for (j = 0; j < AV_PERM_TO_STRING_SIZE; j++) {
				if ((av_perm_to_string[j].tclass == tclass)
				    && (av_perm_to_string[j].value ==
					perm))
					break;
			}
			if (j < AV_PERM_TO_STRING_SIZE) {
				avp->p_idx = j;
				nv++;
				new_av_idx(avp);
			}
		}
		k++;
		perm <<= 1;
	}

	avp->nxt = (struct av_idx *) NULL;

	avp = &av_idxs;
	access_vector = (char **) calloc(1, 1 + (nv * sizeof(char *)));
	access = (char *) NULL;

	for (k = 0; k < nv; k++) {
		if (avp->c_idx >= 0)
			access = common_pts[avp->c_idx];
		else if (avp->p_idx >= 0)
			access = av_perm_to_string[avp->p_idx].name;

		if (access) {
			access_vector[k] = calloc(1, 1 + strlen(access));
			(void) strcpy(access_vector[k], access);
			access = (char *) NULL;
		}

		avp = avp->nxt;
	}

	return (access_vector);
}

#ifdef NL
static char nl = '\n';
#else
static char nl = '\0';
#endif

static void my_print_access_vector(FILE * fp, char **av)
{
	int k;

	if (!fp)
		return;

	if (!av) {
		(void) fprintf(fp, "null%c", nl);
		(void) fflush(fp);
		return;
	}

	if (!av[0])
		return;

	for (k = 0; av[k]; k++)
		(void) fprintf(fp, "%s ", av[k]);

	(void) fprintf(fp, "%c", nl);
	(void) fflush(fp);

	return;
}


static void Usage(FILE * fp, char *argv0)
{
	(void) fprintf(fp,
		       "\nUsage:  %s src_sid target_sid target_class\n",
		       argv0);
	(void) fflush(fp);
	exit(1);
}
