/*
 * udp_server
 * 
 * Create a server socket on a specified port, wait for messages,
 * and echo them to the client.
 *
 * Permit testing of socket_secure, connect_secure, getsockname_secure,  
 * getpeername_secure, recvfrom_secure and sendto_secure.
 * 
 * Calls socket_secure if '-s socket_context' is specified.
 * Calls connect_secure if '-p peer_context ... peer_address peer_port' is specified.
 *
 * Usage:  udp_server [-s socket_context] [-p peer_context ] local_port [peer_address peer_port]
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <arpa/inet.h>
#include <unistd.h>
#include <getopt.h>
#include <stdio.h>

#include <socket_secure.h>
#include <ss.h>

void usage(char *progname) 
{
	printf("usage:  %s [-s socket_context] [-p peer_context ] local_port [peer_address peer_port]\n", progname);
	exit(1);
}

int main(int argc, char **argv)
{
	struct sockaddr_in sin;
	char buf[1024], context[255], context2[255];
	int s, sinlen, ret, buflen, c, contextlen;
	security_id_t so_sid = 0, peer_sid = 0, msg_sid = 0, tmp_sid = 0; 

	while ( (c = getopt(argc, argv, "s:p:")) != EOF) {
		switch (c) {
		case 's':
		case 'p':
			ret = security_context_to_sid(optarg,strlen(optarg)+1,&tmp_sid);
			if (ret) {
				perror(optarg);
				exit(ret);
			}
			if (c == 's')
				so_sid = tmp_sid;
			else if (c == 'p')
				peer_sid = tmp_sid;
			break;
		default:
			usage(argv[0]);
		}
	}

	if (optind != (argc - 3)) {
		if (peer_sid) 
			usage(argv[0]);
		else if (optind != (argc - 1))
			usage(argv[0]);
	}

	/* Create the server socket */
	if (so_sid) 
		s = socket_secure(AF_INET, SOCK_DGRAM, 0, so_sid);
	else
		s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s == -1) {
		perror("socket");
		exit(1);
	}

	/* Obtain the security context of the server socket */
	sin.sin_family = AF_INET;
	sinlen = sizeof(struct sockaddr_in);
	ret = getsockname_secure(s, &sin, &sinlen, &tmp_sid);
	if (ret) {
		perror("getsockname_secure");
		exit(1);
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(tmp_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}
	
	printf("Created socket with context %s\n", context);

	/* Bind to the specified port */
	sin.sin_family = AF_INET;
	sin.sin_port = atoi(argv[optind++]);
	sin.sin_port = htons(sin.sin_port);
	sin.sin_addr.s_addr = INADDR_ANY;
	sinlen = sizeof(struct sockaddr_in);
	ret = bind(s, (struct sockaddr *)&sin, sinlen);
	if (ret == -1) {
		perror("bind");
		exit(1);
	}

	if (optind < argc) {
		/* Set the peer */
		sin.sin_addr.s_addr = inet_addr(argv[optind++]);
		sin.sin_port = atoi(argv[optind++]);
		sin.sin_port = htons(sin.sin_port);
		if (peer_sid) 
			ret = connect_secure(s, (struct sockaddr *)&sin, sinlen, peer_sid);
		else
			ret = connect(s, (struct sockaddr *)&sin, sinlen);
	}
	
	while (1) {
		/* Receive a message from a client and obtain the peer socket  context and the message context */
		buf[0] = 0;
		ret = recvfrom_secure(s, buf, sizeof(buf), 0, (struct sockaddr*)&sin, &sinlen,
				      &peer_sid, &msg_sid);
		if (ret == -1) {
			perror("recvfrom");
			exit(1);
		}

		contextlen = sizeof(context);
		ret = security_sid_to_context(peer_sid, context, &contextlen);
		if (ret) {
			perror("security_sid_to_context");
			exit(1);
		}

		contextlen = sizeof(context2);
		ret = security_sid_to_context(msg_sid, context2, &contextlen);
		if (ret) {
			perror("security_sid_to_context");
			exit(1);
		}

		printf("Received %s from peer (%s,%d) with socket context %s, message context %s\n", 
		       buf, inet_ntoa(sin.sin_addr), ntohs(sin.sin_port), context, context2);

		/* Echo the message back to the client */ 
		ret = sendto_secure(s, buf, sizeof(buf), 0, (struct sockaddr*)&sin, sinlen,
				    peer_sid, msg_sid);
		if (ret == -1) {
			perror("sendto");
			exit(1);
		}
	}
}
