/*
 * flmon - Flnetlink monitor utility.
 *
 * Copyright (c) 2001-2002 James Morris <jmorris@intercode.com.au>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <getopt.h>
#include <libgen.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
          
#include "libflnetlink.h"
#include "libflutil.h"

#define BUFSIZE		8192		/* Needs to be page-sized for netlink dumps. */
#define SHOW_USAGE	1

static unsigned char verbose = 0;
static unsigned char *appname = NULL;
static struct fln_handle *h = NULL;

static struct option longopts[] = {
	{ "help",     0, 0, 'h' },
	{ "verbose",  0, 0, 'v'},
	{ 0 }
};

static void cleanup(void)
{
	if (h) {
		fln_destroy_handle(h);
		h = NULL;
	}

	if (appname) {
		free(appname);
		appname = NULL;
	}
}

static void warn(char *msg, ...)
{
	va_list args;
	
	va_start(args, msg);
	vfprintf(stderr, msg, args);
	va_end(args);
	
	fputc('\n', stderr);
}

static void usage(void)
{
	warn(
"\nUsage: %s [options]\n\n"
"\n"
"Options:\n"
"  --help, -h                          Display help (this).\n"
"  --verbose, -v                       Act verbosely.\n"
"\n"
		, appname);

}

static void die(int show_usage, char *msg, ...)
{
	va_list args;
	
	fputs("Fatal: ", stderr);
	
	va_start(args, msg);
	vfprintf(stderr, msg, args);
	va_end(args);
	
	if (h)
		fprintf(stderr, ": %s", fln_errstr());

	if (errno)
		fprintf(stderr, ": %s", strerror(errno));
		
	fputc('\n', stderr);
	
	if (show_usage)
		usage();
		
	exit(1);
}

static void parse_commandline(int argc, char **argv)
{
	int i;
	
	appname = strdup(basename(argv[0]));
	if (appname == NULL)
		die(0, "strdup");

	while ((i = getopt_long(argc, argv, "hv", longopts, NULL)) != -1) {
	
		switch (i) {
		
			case 'h':
				usage();
				exit(0);
			
			case 'v':
				verbose = 1;
				break;
			
			case ':':
			case '?':
			default:
				die(SHOW_USAGE, "Invalid option: '%c'", i);
		}
	}
}

static void print_loginfo(const unsigned char *buf)
{
	struct timeval tv;
	
	if (gettimeofday(&tv, NULL))
		die(0, "gettimeofday");
	
	printf("%lu.%lu %u %u: ", tv.tv_sec, tv.tv_usec,
	       fln_message_pid(buf), fln_message_seq(buf));
}

int main(int argc, char **argv)
{
	int status;
	unsigned char buf[BUFSIZE];

	if (atexit(cleanup)) {
		warn("Error: Unable to register exit handler.");
		exit(1);
	}

	parse_commandline(argc, argv);
	
	h = fln_create_handle(0, FLN_G_ALL);
	if (!h)
		die(0, "Unable to create handle");

	while (1) {
		int type;
		
		status = fln_read(h, buf, BUFSIZE, 0);
		if (status < 0)
			die(0, "fln_read");
		
		type = fln_message_type(buf);
		
		if (verbose)
			print_loginfo(buf);
		
		switch (type) {
		
		case FLMSG_PERIM_ADD:
			fputs("perimeter add ", stdout);
			print_perimtab_entry(fln_get_msg(buf));
			break;
		
		case FLMSG_PERIM_DEL:
			fputs("perimeter del ", stdout);
			print_perimtab_entry(fln_get_msg(buf));
			break;
			
		case FLMSG_PERIM_FLUSH:
			fputs("perimeter flush\n", stdout);
			break;
		
		case FLMSG_CACHE_ADD:
			fputs("cache add ", stdout);
			print_cache_entry(fln_get_msg(buf));
			break;

		case FLMSG_CACHE_DEL:
			fputs("cache del ", stdout);
			print_cache_entry(fln_get_msg(buf));
			break;

		case FLMSG_CACHE_FLUSH:
			fputs("cache flush\n", stdout);
			break;
			
		case FLMSG_CACHE_MAP_REQ:
			fputs("map request ", stdout);
			print_mapreq(fln_get_msg(buf));
			break;
		
		case FLMSG_QUEUE_FLUSH:
			fputs("queue flush\n", stdout);
			break;

		default:
			warn("Received unknown Netlink message type: %d\n", type);
			break;
		}
	}
	
	return 0;
}
