/***************************************************************************
                             th-job-progress-dialog.c
                             ----------------------
    begin                : Wed Nov 10 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "th-job-progress-dialog.h"
#include "th-job.h"
#include "th-power-manager-agent.h"
#include "th-utils.h"

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <time.h>
#include <string.h>

enum
{
	PROP_DELETE_PARTIAL_FILE_ON_CANCEL = 1
};

enum
{
	TH_RESPONSE_TOGGLE_PAUSE = 9876
};

struct _ThJobProgressDialogPrivate
{
	ThJob        *job;
	GTimer       *job_timer;       /* reset when starting to encode       */
	GTimer       *fps_timer;       /* reset every X secs to calculate fps */

	GtkWidget    *preview_img;
	GtkWidget    *pbar_current;
	GtkWidget    *pbar_total;
	
	GtkWidget    *fn_label;
	GtkWidget    *fps_label;
	GtkWidget    *written_label;
	GtkWidget    *elapsed_label;
	GtkWidget    *timeleft_label;
	
	GtkWidget    *close_button;
	GtkWidget    *pause_button;
	GtkWidget    *cancel_button;
	GtkWidget    *pause_label;

	gint          paused_txt_pts; /* size of the 'paused' text in points */

	GdkPixbuf    *empty_pixbuf;
	
	gulong        response_id;

	gdouble       weight_total, weight_done, weight_curjob;
	
	gboolean      out_of_disk_space;
	gboolean      skip_preview_image;

	/* prefs */
	gboolean      delete_partial_file_on_cancel;

	ThPowerManagerAgent *power_manager_agent;
};

static void             job_progress_dialog_class_init    (ThJobProgressDialogClass *klass);

static void             job_progress_dialog_instance_init (ThJobProgressDialog *cp);

static void             job_progress_dialog_finalize      (GObject *object);

static gboolean         job_progress_dialog_run_job       (ThJobProgressDialog *jpd, ThJob *job, gboolean *p_cancelled);

static void             job_progress_dialog_response      (ThJobProgressDialog *jpd, gint ret, gpointer foo);

static gboolean         job_progress_poll_status          (ThJobProgressDialog *jpd);

static void             job_progress_dialog_set_job_progress_bar (ThJobProgressDialog *jpd, 
                                                                  guint                n, 
                                                                  guint                num_jobs, 
                                                                  gdouble              cur_percent);


/* variables */

static GObjectClass    *jpd_parent_class;          /* NULL */


/***************************************************************************
 *
 *   job_progress_dialog_get_property
 *
 ***************************************************************************/

static void
job_progress_dialog_get_property (GObject      *object,
                                  guint         param_id,
                                  GValue       *value,
                                  GParamSpec   *pspec)
{
	ThJobProgressDialog *jpd = (ThJobProgressDialog*) object;

	g_return_if_fail (TH_IS_JOB_PROGRESS_DIALOG (object));

	switch (param_id)
	{
		case PROP_DELETE_PARTIAL_FILE_ON_CANCEL:
		{
			g_value_set_boolean (value, jpd->priv->delete_partial_file_on_cancel);
		}
		break;
			
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
	}
}


/***************************************************************************
 *
 *   job_progress_dialog_set_property
 *
 ***************************************************************************/

static void
job_progress_dialog_set_property (GObject      *object,
                                  guint         param_id,
                                  const GValue *value,
                                  GParamSpec   *pspec)
{
	ThJobProgressDialog *jpd = (ThJobProgressDialog*) object;

	g_return_if_fail (TH_IS_JOB_PROGRESS_DIALOG (object));

	switch (param_id)
	{
		case PROP_DELETE_PARTIAL_FILE_ON_CANCEL:
		{
			jpd->priv->delete_partial_file_on_cancel = g_value_get_boolean (value);
		}
		break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
	}
}

/***************************************************************************
 *
 *   job_progress_dialog_class_init
 *
 ***************************************************************************/

static void
job_progress_dialog_class_init (ThJobProgressDialogClass *klass)
{
	GObjectClass  *object_class; 

	object_class = G_OBJECT_CLASS (klass);
	
	jpd_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = job_progress_dialog_finalize;
	object_class->set_property = job_progress_dialog_set_property;
	object_class->get_property = job_progress_dialog_get_property;

	g_object_class_install_property (object_class, PROP_DELETE_PARTIAL_FILE_ON_CANCEL, 
	                                 g_param_spec_boolean ("delete-partial-file-on-cancel", 
	                                                       "delete-partial-file-on-cancel", 
	                                                       "delete-partial-file-on-cancel", 
	                                                       TRUE, G_PARAM_READWRITE));
}

/***************************************************************************
 *
 *   job_progress_dialog_window_state_cb
 *
 ***************************************************************************/

static gboolean
job_progress_dialog_window_state_cb (ThJobProgressDialog *jpd, 
                                     GdkEventWindowState *ws_ev, 
                                     gpointer             foobar)
{
	const GdkWindowState mask = GDK_WINDOW_STATE_WITHDRAWN | GDK_WINDOW_STATE_ICONIFIED;
	
	if ((ws_ev->changed_mask & mask))
	{
		if (jpd->priv->skip_preview_image != ((ws_ev->new_window_state & mask) != 0))
		{
			jpd->priv->skip_preview_image = ((ws_ev->new_window_state & mask) != 0);

			/* If dialog is made visible again, update immediately */
			if (jpd->priv->skip_preview_image == FALSE && jpd->priv->job != NULL)
				job_progress_poll_status (jpd);
		}
	}
	
	return FALSE; /* propagate event further */
}

/***************************************************************************
 *
 *   job_progress_dialog_destroy_cb
 *
 ***************************************************************************/

static void
job_progress_dialog_destroy_cb (ThJobProgressDialog *jpd, gboolean *p_destroyed)
{
	job_progress_dialog_response (jpd, TH_RESPONSE_DESTROYED, NULL);
	*p_destroyed = TRUE;
}

/***************************************************************************
 *
 *   job_progress_dialog_instance_init
 *
 ***************************************************************************/

static void
job_progress_dialog_instance_init (ThJobProgressDialog *jpd)
{
	GtkWidget *appwin;

	jpd->priv = g_new0 (ThJobProgressDialogPrivate, 1);

	jpd->priv->delete_partial_file_on_cancel = TRUE;

	jpd->priv->job_timer = g_timer_new ();
	jpd->priv->fps_timer = g_timer_new ();

	jpd->priv->power_manager_agent = th_power_manager_agent_new ();

	jpd->priv->response_id = g_signal_connect (jpd, "response", 
	                            G_CALLBACK (job_progress_dialog_response), 
	                            NULL);

	appwin = gtk_widget_get_toplevel (GTK_WIDGET (jpd));

	g_signal_connect_swapped (appwin, "window-state-event",
	                          G_CALLBACK (job_progress_dialog_window_state_cb),
	                          jpd);
}


/***************************************************************************
 *
 *   job_progress_dialog_finalize
 *
 ***************************************************************************/

static void
job_progress_dialog_finalize (GObject *object)
{
	ThJobProgressDialog *jpd;

	jpd = (ThJobProgressDialog*) object;

	th_log ("ThJobProgressDialog: finalize\n");

	g_object_unref (jpd->priv->empty_pixbuf);
	g_object_unref (jpd->priv->power_manager_agent);

	g_timer_destroy (jpd->priv->job_timer);
	g_timer_destroy (jpd->priv->fps_timer);

	memset (jpd->priv, 0xab, sizeof (ThJobProgressDialogPrivate));
	g_free (jpd->priv);
	jpd->priv = NULL;

	/* chain up */
	jpd_parent_class->finalize (object);
}


/***************************************************************************
 *
 *   th_job_progress_dialog_get_type
 *
 ***************************************************************************/

GType
th_job_progress_dialog_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThJobProgressDialogClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) job_progress_dialog_class_init,
			NULL, NULL,
			sizeof (ThJobProgressDialog),
			0,
			(GInstanceInitFunc) job_progress_dialog_instance_init
		};

		type = g_type_register_static (TH_TYPE_FAKE_DIALOG, "ThJobProgressDialog", &info, 0);
	}

	return type;
}


/***************************************************************************
 *
 *   th_job_progress_dialog_new
 *
 ***************************************************************************/

static void
job_progress_dialog_set_empty_image (ThJobProgressDialog *jpd, guint width, guint height)
{
	if (jpd->priv->empty_pixbuf)
		g_object_unref (jpd->priv->empty_pixbuf);
	
	jpd->priv->empty_pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, width, height);
	
	gdk_pixbuf_fill (jpd->priv->empty_pixbuf, 0x0000FF00); /* blue */
	
	gtk_image_set_from_pixbuf (GTK_IMAGE (jpd->priv->preview_img), jpd->priv->empty_pixbuf);
}

/***************************************************************************
 *
 *   th_job_progress_dialog_new
 *
 ***************************************************************************/

GtkWidget *
th_job_progress_dialog_new (void)
{
	ThJobProgressDialog *jpd;
	GtkWidget         *glade_window = NULL;
	GtkWidget         *toplevel_vbox;

	jpd = (ThJobProgressDialog *) g_object_new (TH_TYPE_JOB_PROGRESS_DIALOG, NULL);
	
	if (!th_utils_ui_load_interface ("th-job-progress-dialog.glade", 
	                                 FALSE,
	                                 "th-job-progress-dialog", &glade_window,
	                                 "th-toplevel-vbox",       &toplevel_vbox,
	                                 "th-preview-image",       &jpd->priv->preview_img,
	                                 "th-current-progressbar", &jpd->priv->pbar_current,
	                                 "th-all-progressbar",     &jpd->priv->pbar_total,
	                                 "th-filename-label",      &jpd->priv->fn_label,
	                                 "th-written-label",       &jpd->priv->written_label,
	                                 "th-timeleft-label",      &jpd->priv->timeleft_label,
	                                 "th-elapsed-label",       &jpd->priv->elapsed_label,
	                                 "th-fps-label",           &jpd->priv->fps_label,
                                   NULL))
	{
		g_warning ("th_utils_ui_load_interface (\"th-ui-job-progress-dialog.glade\") failed.\n");
		if (glade_window)
			gtk_widget_destroy (glade_window);
		gtk_widget_destroy (GTK_WIDGET (jpd));
		return NULL;
	}
	
	g_object_ref (toplevel_vbox);
	gtk_container_remove (GTK_CONTAINER (glade_window), toplevel_vbox);
	gtk_container_add (GTK_CONTAINER (TH_FAKE_DIALOG (jpd)->vbox), toplevel_vbox);
	g_object_unref (toplevel_vbox);

	jpd->priv->cancel_button = th_fake_dialog_add_button (TH_FAKE_DIALOG (jpd),
	                                                      GTK_STOCK_CANCEL,
	                                                      GTK_RESPONSE_REJECT);

	/* makes cancel button stick to the left; let's see
	 *  how long it takes until some HIGian complains ;) */
	gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (TH_FAKE_DIALOG (jpd)->action_area),
	                                    jpd->priv->cancel_button,
	                                    TRUE);

	jpd->priv->pause_button = th_fake_dialog_add_button (TH_FAKE_DIALOG (jpd),
	                                                     GTK_STOCK_MEDIA_PAUSE,
	                                                     TH_RESPONSE_TOGGLE_PAUSE);
	{
		GtkWidget *hbox, *img, *align;
		align = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
		hbox = gtk_hbox_new (FALSE, 0);
		img = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PAUSE, GTK_ICON_SIZE_BUTTON);
		jpd->priv->pause_label = gtk_label_new_with_mnemonic (_("_Pause"));
		gtk_box_pack_start (GTK_BOX (hbox), img, FALSE, FALSE, 2);
		gtk_box_pack_start (GTK_BOX (hbox), jpd->priv->pause_label, FALSE, FALSE, 2);
		gtk_widget_destroy (GTK_BIN (jpd->priv->pause_button)->child);
		gtk_container_add (GTK_CONTAINER (align), hbox);
		gtk_container_add (GTK_CONTAINER (jpd->priv->pause_button), align);
		gtk_widget_show_all (align);
	}

	th_fake_dialog_set_default_response (TH_FAKE_DIALOG (jpd), TH_RESPONSE_TOGGLE_PAUSE); 

	jpd->priv->close_button = th_fake_dialog_add_button (TH_FAKE_DIALOG (jpd),
	                                                     GTK_STOCK_CLOSE,
	                                                     GTK_RESPONSE_ACCEPT);

	gtk_widget_hide (jpd->priv->close_button);
	gtk_widget_hide (jpd->priv->pause_button);

	job_progress_dialog_set_empty_image (jpd, 384, 288);

	return GTK_WIDGET (jpd);
}

/***************************************************************************
 *
 *   job_progress_dialog_response
 *
 *   This can only be the cancel button or the 'close window' button
 *
 ***************************************************************************/

static void
job_progress_dialog_response (ThJobProgressDialog *jpd, gint ret, gpointer foo)
{
	if (ret == TH_RESPONSE_TOGGLE_PAUSE)
	{
		gboolean paused;
	
		g_object_get (jpd->priv->job, "pipeline-paused", &paused, NULL);
		g_object_set (jpd->priv->job, "pipeline-paused", !paused, NULL);

		/* button label will be set in notify::pipeline-paused callback */	
		return;
	}

	if (ret == TH_RESPONSE_DESTROYED)
		goto cancel_job;

	/* will be 'Cancel' or 'Close' */

	if (GTK_WIDGET_VISIBLE (jpd->priv->cancel_button) 
	 && g_timer_elapsed (jpd->priv->job_timer, NULL) >= 30.0)
	{
		GtkWidget *dlg, *appwin;

		appwin = gtk_widget_get_toplevel (GTK_WIDGET (jpd));

		dlg = gtk_message_dialog_new_with_markup (GTK_WINDOW (appwin),
		                                          GTK_DIALOG_MODAL,
		                                          GTK_MESSAGE_QUESTION,
		                                          GTK_BUTTONS_YES_NO,
		                                          _("Do you really want to quit?"));

		gtk_window_set_title (GTK_WINDOW (dlg), _("Do you really want to quit?"));

		ret = gtk_dialog_run (GTK_DIALOG (dlg));
	
		gtk_widget_destroy (dlg);

		if (ret == GTK_RESPONSE_REJECT || ret == GTK_RESPONSE_NO)
			return;
	}

cancel_job:

	th_job_cancel (jpd->priv->job, jpd->priv->delete_partial_file_on_cancel);
}

/***************************************************************************
 *
 *   job_secs_to_str
 *
 ***************************************************************************/

static gchar *
job_secs_to_str (guint secs)
{
	gchar *s;
	 
	s = g_strdup_printf ("%u:%02u:%02u", 
	                     secs / 3600, 
	                     (secs % 3600) / 60, 
	                     secs % 60);

	return s;
}

/***************************************************************************
 *
 *   job_get_human_time_interval_str
 *
 ***************************************************************************/

static const gchar *
job_get_human_time_interval_str (gint secs)
{
	static gchar  s[128];
	 
	if (secs <= 0)
		*s = 0x00;
	else if (secs < 60)
		g_snprintf (s, sizeof (s), _("%u seconds"), secs);
	else if (secs < 3600)
		g_snprintf (s, sizeof (s), _("%u minutes"), secs/60);
	else 
		g_snprintf (s, sizeof (s), _("%u:%02u hours"), secs/3600, (secs % 3600) / 60);

	return (const gchar*) s;
}

/***************************************************************************
 *
 *   job_progress_calc_fps
 *
 ***************************************************************************/

static void
job_progress_calc_fps (ThJobProgressDialog *jpd, const gchar *num_frames_prop_name, GTimer *timer)
{
	gdouble  elapsed, fps;
	guint    num_frames;

	g_return_if_fail (jpd->priv->job != NULL);
	
	elapsed = g_timer_elapsed (timer, NULL);

	g_object_get (jpd->priv->job, num_frames_prop_name, &num_frames, NULL);

	fps = (gdouble) num_frames / elapsed;
	
	if (fps >= 1.0)
		th_label_set_text (jpd->priv->fps_label, _("%.1f frames per second"), (gfloat) fps);
	else
		th_label_set_text (jpd->priv->fps_label, _("%.2f frames per second"), (gfloat) fps);
}

/***************************************************************************
 *
 *   job_progress_calc_written
 *
 ***************************************************************************/

static void
job_progress_calc_written (ThJobProgressDialog *jpd, guint pos_msecs)
{
	gdouble  total_estimate, curpos, length;
	gchar   *now_str, *total_str;
	guint    bytes_written, target_quality, title_len_secs;

	g_return_if_fail (jpd->priv->job != NULL);

	g_object_get (jpd->priv->job, 
	              "title-length", &title_len_secs, 
	              "bytes-written", &bytes_written, 
	              "target-quality", &target_quality,
	              NULL);
	
	curpos = pos_msecs / 1000.0;
	length = title_len_secs * 1.0;
	total_estimate = (bytes_written / curpos) * length;
	
	now_str = th_utils_get_human_size_str ((guint64) bytes_written);

	/* Show total written when job is done */
	if (pos_msecs == G_MAXUINT)
	{
		th_label_set_text (jpd->priv->written_label, "%s", now_str);
		g_free (now_str);
		return;
	}
	
	/* note: secs_left might be negative, as length is not exact */
	if (bytes_written < (100 * 1024) 
	 || total_estimate < bytes_written
	 || (curpos / length) < 0.03)
	{
		th_label_set_text (jpd->priv->written_label, "%s", now_str);
		g_free (now_str);
		return;
	}
	
	total_str = th_utils_get_human_size_str ((guint64) total_estimate);
	
	th_label_set_text (jpd->priv->written_label, _("%s of ca. %s"), now_str, total_str);
	
	g_free (now_str);
	g_free (total_str);
}

/***************************************************************************
 *
 *   job_progress_calc_time_left
 *
 ***************************************************************************/

static void
job_progress_calc_time_left (ThJobProgressDialog *jpd, guint pos_msecs)
{
	const gchar *cstr;
	gdouble      elapsed, curpos, length, secs_left;
	time_t       eta;
	guint        title_len_secs;
	gchar        timestr[64];

	elapsed = g_timer_elapsed (jpd->priv->job_timer, NULL);
	
	cstr = job_get_human_time_interval_str ((gint) elapsed);
	gtk_label_set_text (GTK_LABEL (jpd->priv->elapsed_label), cstr); 
	
	if (elapsed < 5.0 || jpd->priv->job == NULL || pos_msecs == G_MAXUINT)
		return;
	
	g_object_get (jpd->priv->job, "title-length", &title_len_secs, NULL);
	
	curpos = pos_msecs / 1000.0;
	length = title_len_secs * 1.0;
	secs_left = (elapsed / curpos) * (length - curpos);

	/* note: secs_left might be negative, as length is not exact */
	cstr = job_get_human_time_interval_str ((gint) secs_left);
	eta = time (NULL) + (time_t) MAX (15, secs_left);
	strftime (timestr, sizeof (timestr), "%R", localtime (&eta));
	th_label_set_text (jpd->priv->timeleft_label, "%s [%sh]", cstr, timestr); 
}

/***************************************************************************
 *
 *   job_progress_dialog_set_preview_image
 *
 ***************************************************************************/

/* how much bigger the shaded area in the picture is than the text */
#define TXT_BORDER 15

static void
job_progress_dialog_set_preview_image (ThJobProgressDialog *jpd, 
                                       GdkPixbuf           *pixbuf, 
                                       gboolean             paused)
{
	PangoRectangle  ink_rect, logical_rect;
	PangoLayout    *layout = NULL;
	const gchar    *msg;
	GdkPixbuf      *pixbuf_shaded;
	GdkPixmap      *pixmap;
	gint            width, height, txt_x, txt_y, pts;
	
	if (!paused)
	{
		gtk_image_set_from_pixbuf (GTK_IMAGE (jpd->priv->preview_img), pixbuf);
		return;
	}

	if (jpd->priv->out_of_disk_space)
		msg = _("Paused\n(Out of Disk Space)");
	else
		msg = _("Paused");

	width = gdk_pixbuf_get_width (pixbuf);
	height = gdk_pixbuf_get_height (pixbuf);
	
	/* Find text size that makes text occupy 
	 *  ca. 3/5 of the pixbuf width. */
	if (jpd->priv->paused_txt_pts > 0)
		pts = jpd->priv->paused_txt_pts + 8;
	else
		pts = 128;

	do
	{
		gchar *markup;

		pts = pts - 8;
		
		if (layout)
			g_object_unref (layout);

		layout = gtk_widget_create_pango_layout (GTK_WIDGET (jpd), NULL);
		markup = g_markup_printf_escaped ("<span font_desc='%u' color='white'><b>%s</b></span>", pts, msg);
	
		pango_layout_set_markup (layout, markup, -1);
		g_free (markup);
		
		pango_layout_set_alignment (layout, PANGO_ALIGN_CENTER);
	
		pango_layout_get_pixel_extents (layout, &ink_rect, &logical_rect);
	
		txt_x = (width - ink_rect.width) / 2;
		txt_y = (height - ink_rect.height) / 2;
	} 
	while ((ink_rect.width * 1.0) > (width * 0.60));
	
	if (jpd->priv->paused_txt_pts < 0)
		jpd->priv->paused_txt_pts = pts;

	pixmap = gdk_pixmap_new (GDK_DRAWABLE (GTK_WIDGET (jpd)->window), width, height, -1);
	
	pixbuf_shaded = th_utils_get_shaded_pixbuf (pixbuf, 
	                                            txt_x - TXT_BORDER, txt_y - TXT_BORDER,
	                                            ink_rect.width + 2*TXT_BORDER, ink_rect.height + 2*TXT_BORDER,
	                                            -48);

	gdk_draw_pixbuf (GDK_DRAWABLE (pixmap), NULL, pixbuf_shaded,
	                 0, 0, 0, 0, width, height, 
	                 GDK_RGB_DITHER_NORMAL, 0, 0);

	gdk_draw_layout (GDK_DRAWABLE (pixmap), 
	                 GTK_WIDGET (jpd->priv->preview_img)->style->text_gc[GTK_STATE_NORMAL],
	                 txt_x - ((logical_rect.width - ink_rect.width)/2), 
	                 txt_y - ((logical_rect.height - ink_rect.height)/2), 
	                 layout);

	gtk_image_set_from_pixmap (GTK_IMAGE (jpd->priv->preview_img), pixmap, NULL);
	
	g_object_unref (pixbuf_shaded);
	g_object_unref (pixmap);
	g_object_unref (layout);
}

/***************************************************************************
 *
 *   job_progress_dialog_set_current_progress
 *
 ***************************************************************************/

static void
job_progress_dialog_set_current_progress (ThJobProgressDialog *jpd, guint pos_secs, guint len_secs, gdouble fraction)
{
	gchar  *spos, *slen, *s;
	
	spos = job_secs_to_str (pos_secs);
	slen = job_secs_to_str (len_secs);

	if (g_str_has_prefix (spos, "0:0") && g_str_has_prefix (slen, "0:0"))
		s = g_strdup_printf ("%s / %s", spos + 3, slen + 3);
	else if (g_str_has_prefix (spos, "0:") && g_str_has_prefix (slen, "0:"))
		s = g_strdup_printf ("%s / %s", spos + 2, slen + 2);
	else
		s = g_strdup_printf ("%s / %s", spos, slen);

	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (jpd->priv->pbar_current), fraction);
	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (jpd->priv->pbar_current), s);

	g_free (spos);
	g_free (slen);
	g_free (s);
}

/***************************************************************************
 *
 *   job_progress_poll_status
 *
 *   Called every 1-2 seconds to update the preview snapshot and the labels
 *
 ***************************************************************************/

static gboolean
job_progress_poll_status (ThJobProgressDialog *jpd)
{
	GdkPixbuf   *pixbuf;
	gboolean     paused;
	gdouble      fraction;
	guint        pos_msecs, title_len_secs;
	
	if (jpd->priv->skip_preview_image)
		return TRUE; /* call again later*/

	g_return_val_if_fail (jpd->priv->job != NULL, TRUE);

	g_object_get (jpd->priv->job, 
	              "current-frame", &pixbuf, 
	              "title-length", &title_len_secs,
	              "pipeline-paused", &paused,
	              NULL);

	if (pixbuf == NULL)
		return TRUE;

	job_progress_dialog_set_preview_image (jpd, pixbuf, paused);
	
	/* set fatal mask only after those stupid state change warnings from gst launch */
	/* g_log_set_always_fatal (G_LOG_LEVEL_ERROR | G_LOG_LEVEL_CRITICAL | G_LOG_LEVEL_WARNING); */
	
	pos_msecs = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (pixbuf), "timestamp"));
	
	fraction = MAX (0.0, MIN ((gdouble) pos_msecs / (title_len_secs * 1000.0), 1.0));

	job_progress_dialog_set_current_progress (jpd, pos_msecs/1000, title_len_secs, fraction);
	job_progress_dialog_set_job_progress_bar (jpd, 0, 0, fraction);

	g_object_unref (pixbuf);

	job_progress_calc_time_left (jpd, pos_msecs);
	job_progress_calc_written (jpd, pos_msecs);

	job_progress_calc_fps (jpd, "num-frames-encoded", jpd->priv->fps_timer);
	g_object_set (jpd->priv->job, "num-frames-encoded", 0, NULL);
	g_timer_reset (jpd->priv->fps_timer);

	return TRUE; /* call us again */
}

/***************************************************************************
 *
 *   jpd_pipeline_paused_toggled_cb
 *
 ***************************************************************************/

static void
jpd_pipeline_paused_toggled_cb (ThJobProgressDialog *jpd, GParamSpec *pspec, ThJob *job)
{
	gboolean paused;
	
	g_object_get (jpd->priv->job, "pipeline-paused", &paused, NULL);
	
	if (paused)
	{
		gtk_label_set_text_with_mnemonic (GTK_LABEL (jpd->priv->pause_label), _("_Unpause"));
		g_timer_stop (jpd->priv->job_timer);
	}
	else
	{
		gtk_label_set_text_with_mnemonic (GTK_LABEL (jpd->priv->pause_label), _("_Pause"));
		g_timer_continue (jpd->priv->job_timer);
		jpd->priv->out_of_disk_space = FALSE;
		jpd->priv->paused_txt_pts = -1;
	}
}

/***************************************************************************
 *
 *   jpd_job_out_of_disk_space
 *
 ***************************************************************************/

static gboolean
jpd_job_out_of_disk_space_dlg (ThJobProgressDialog *jpd)
{
	GtkWidget *dlg, *appwin;
	GString   *errmsg;
	gchar     *fn, *dir;

	jpd->priv->out_of_disk_space = TRUE;
	jpd->priv->paused_txt_pts = -1;

	errmsg = g_string_new (NULL);
	
	g_object_get (jpd->priv->job, "output-fn", &fn, NULL);
	dir = g_path_get_dirname (fn);

	g_string_append_printf (errmsg, "\n<b>%s</b>\n\n", _("Not enough free disk space"));
	g_string_append_printf (errmsg, _("You are about to run out of free disk space in\n"
	                        "\n    %s\n\n"
	                        "Please make some space there before continuing.\n"), dir);
	
	appwin = gtk_widget_get_toplevel (GTK_WIDGET (jpd));
	dlg = gtk_message_dialog_new_with_markup (GTK_WINDOW (appwin), 
	                                          GTK_DIALOG_MODAL,
	                                          GTK_MESSAGE_WARNING,
	                                          GTK_BUTTONS_CLOSE,
	                                          errmsg->str);
	
	(void) gtk_dialog_run (GTK_DIALOG (dlg));

	gtk_widget_destroy (dlg);

	g_string_free (errmsg, TRUE);
	g_free (dir);
	g_free (fn);
	
	return FALSE; /* only run once */
}

/***************************************************************************
 *
 *   jpd_job_out_of_disk_space
 *
 ***************************************************************************/

static void
jpd_job_out_of_disk_space (ThJobProgressDialog *jpd, ThJob *job)
{
	g_idle_add ((GSourceFunc) jpd_job_out_of_disk_space_dlg, jpd);
}

/***************************************************************************
 *
 *   job_progress_dialog_run_job
 *
 ***************************************************************************/

static gboolean
job_progress_dialog_run_job (ThJobProgressDialog *jpd, ThJob *job, gboolean *p_cancelled)
{
	ThPictureSize  picsize;
	gboolean  ret, destroyed;
	GError   *err = NULL;
	gulong    sigid1, sigid2, destroyid;
	guint     title_num, title_len_secs, sec_id, width, height;
	gchar    *fn;
	
	g_return_val_if_fail (TH_IS_JOB (job), TRUE);

	destroyid = g_signal_connect (jpd, "destroy",
	                              G_CALLBACK (job_progress_dialog_destroy_cb),
	                              &destroyed);

	gtk_label_set_text (GTK_LABEL (jpd->priv->fps_label), "");
	gtk_label_set_text (GTK_LABEL (jpd->priv->written_label), "");
	gtk_label_set_text (GTK_LABEL (jpd->priv->timeleft_label), "");
	gtk_label_set_text (GTK_LABEL (jpd->priv->elapsed_label), "");

	jpd->priv->job = job;
	jpd->priv->paused_txt_pts = -1;

	jpd->priv->weight_curjob = 0.0;
	
	gtk_label_set_text_with_mnemonic (GTK_LABEL (jpd->priv->pause_label), _("_Pause"));

	/* job might set pipeline to paused by itself 
	 *  (e.g. if we run out of disk space */
	sigid1 = g_signal_connect_swapped (jpd->priv->job, 
	                                   "notify::pipeline-paused", 
	                                   G_CALLBACK (jpd_pipeline_paused_toggled_cb),
	                                   jpd);
	
	sigid2 = g_signal_connect_swapped (jpd->priv->job, 
	                                   "out-of-disk-space", 
	                                   G_CALLBACK (jpd_job_out_of_disk_space),
	                                   jpd);
	
	jpd->priv->out_of_disk_space = FALSE;
	
	g_object_get (job, 
	              "title-num", &title_num, 
	              "output-fn", &fn, 
	              "weight", &jpd->priv->weight_curjob,
	              "picture-size", &picsize,
	              "title-length", &title_len_secs,
	              NULL);

	gtk_label_set_text (GTK_LABEL (jpd->priv->fn_label), fn);
	g_free (fn);

	th_job_get_effective_picture_size (job, picsize, &width, &height);
	
	job_progress_dialog_set_empty_image (jpd, width, height);
	
	sec_id = g_timeout_add (3000, (GSourceFunc) job_progress_poll_status, jpd);

	g_timer_reset (jpd->priv->job_timer);
	g_timer_reset (jpd->priv->fps_timer);

	destroyed = FALSE;

	ret = th_job_run (job, &err);

	g_source_remove (sec_id);

	if (destroyed == TRUE)
	{
		*p_cancelled = TRUE;
		return FALSE;
	}

	jpd->priv->weight_done += jpd->priv->weight_curjob;
	
	g_signal_handler_disconnect (jpd, destroyid);
	g_signal_handler_disconnect (jpd->priv->job, sigid1);
	g_signal_handler_disconnect (jpd->priv->job, sigid2);
	
	job_progress_calc_written (jpd, G_MAXUINT);
	job_progress_dialog_set_current_progress (jpd, title_len_secs, title_len_secs, 1.0);
	job_progress_calc_fps (jpd, "total-frames-encoded", jpd->priv->job_timer);
	gtk_label_set_text (GTK_LABEL (jpd->priv->timeleft_label), "");
	job_progress_calc_time_left (jpd, G_MAXUINT);

	jpd->priv->job = NULL;

	if (err)
	{
		GtkWidget *dlg, *appwin;
		
		appwin = gtk_widget_get_toplevel (GTK_WIDGET (jpd));
		g_assert (GTK_IS_WINDOW (appwin));
		dlg = gtk_message_dialog_new (GTK_WINDOW (appwin),
		                              GTK_DIALOG_MODAL,
		                              GTK_MESSAGE_ERROR,
		                              GTK_BUTTONS_OK,
		                              _("An error occured:\n\n  %s\n"),
		                              (err) ? err->message : _("unknown error"));
		
		(void) gtk_dialog_run (GTK_DIALOG (dlg));
		
		gtk_widget_destroy (dlg);
		
		if (err)
			g_error_free (err);
		
		return FALSE;
	}
	
	if (ret == FALSE)
		*p_cancelled = TRUE;

	return ret;
}

/***************************************************************************
 *
 *   job_progress_dialog_set_job_progress_bar
 *
 ***************************************************************************/

static void
job_progress_dialog_set_job_progress_bar (ThJobProgressDialog *jpd, guint n, guint num_jobs, gdouble cur_percent)
{
	gdouble w;

	w = (jpd->priv->weight_done + (jpd->priv->weight_curjob * cur_percent)) / (jpd->priv->weight_total + 0.0000001);

	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (jpd->priv->pbar_total), MAX (0.0, MIN (w, 1.0)));
	
	if (num_jobs > 0)
	{
		gchar *s = g_strdup_printf (_("%u of %u"), n, num_jobs);
		gtk_progress_bar_set_text (GTK_PROGRESS_BAR (jpd->priv->pbar_total), s);
		g_free (s);
	}
}

/***************************************************************************
 *
 *   th_job_progress_dialog_run_jobs
 *
 ***************************************************************************/

gboolean
th_job_progress_dialog_run_jobs (ThJobProgressDialog *jpd, GList *jobs)
{
	gboolean  got_cancelled = FALSE;
	guint     num_jobs, n = 0;
	GList    *l;
	
	g_return_val_if_fail (TH_IS_JOB_PROGRESS_DIALOG (jpd), FALSE);
	g_return_val_if_fail (jobs != NULL, FALSE);

	g_object_set (jpd->priv->power_manager_agent,
	              "inhibit-power-manager", TRUE,
	              NULL);

	gtk_widget_hide (jpd->priv->close_button);
	gtk_widget_show (jpd->priv->pause_button);
	gtk_widget_show (jpd->priv->cancel_button);
	
	num_jobs = g_list_length (jobs);
	
	jpd->priv->weight_total = 0.0;
	jpd->priv->weight_done = 0.0;
	
	for (l = jobs;  l;  l = l->next)
	{
		gdouble w = 0.0;
		g_object_get (l->data, "weight", &w, NULL);
		jpd->priv->weight_total += w;
	}

	for (l = jobs;  l;  l = l->next)
	{
		ThJob *job = TH_JOB (l->data);
		
		job_progress_dialog_set_job_progress_bar (jpd, n, num_jobs, 0.0);

		if (!job_progress_dialog_run_job (jpd, job, &got_cancelled)  &&  got_cancelled)
		{
			/* the dialog object has been destroyed now */
			return FALSE;
		}
		
		++n;
	}
	
	job_progress_dialog_set_job_progress_bar (jpd, n, num_jobs, 1.0);

	/* All done, let's show Close button, and
	 *  wait for that to be clicked before
	 *  we close the dialog */
	gtk_widget_show (jpd->priv->close_button);
	gtk_widget_hide (jpd->priv->cancel_button);
	gtk_widget_hide (jpd->priv->pause_button);

	g_signal_handler_block (jpd, jpd->priv->response_id);

	(void) th_fake_dialog_run (TH_FAKE_DIALOG (jpd));

	g_signal_handler_unblock (jpd, jpd->priv->response_id);
	
	gtk_widget_hide (GTK_WIDGET (jpd));

	g_object_set (jpd->priv->power_manager_agent,
	              "inhibit-power-manager", FALSE,
	              NULL);

	return TRUE;
}
