------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ U T I L                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--          Copyright (C) 1992-2026, Free Software Foundation, Inc.         --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT; see file COPYING3.  If not, go to --
-- http://www.gnu.org/licenses for a complete copy of the license.          --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

--  Package containing utility procedures used throughout the semantics

with Aspects;        use Aspects;
with Atree;          use Atree;
with Einfo.Entities; use Einfo.Entities;
with Exp_Tss;        use Exp_Tss;
with Namet;          use Namet;
with Opt;            use Opt;
with Snames;         use Snames;
with Types;          use Types;
with Uintp;          use Uintp;
with Urealp;         use Urealp;

package Sem_Util is

   function Abstract_Interface_List (Typ : Entity_Id) return List_Id;
   --  The list of interfaces implemented by Typ. Empty if there are none,
   --  including the cases where there can't be any because e.g. the type is
   --  not tagged.

   function Acquire_Warning_Match_String (Str_Lit : Node_Id) return String;
   --  Used by pragma Warnings (Off, string), and Warn_As_Error (string) to get
   --  the given string argument, adding leading and trailing asterisks if they
   --  are not already present.  Str_Lit is the static value of the pragma
   --  argument.

   procedure Add_Access_Type_To_Process (E : Entity_Id; A : Entity_Id);
   --  Add A to the list of access types to process when expanding the
   --  freeze node of E.

   procedure Add_Global_Declaration (Decl : Node_Id);
   --  This procedure adds a declaration Decl at the library level, to be
   --  elaborated before any other code in the unit. It is used for example
   --  for the entity that marks whether a unit has been elaborated. The
   --  declaration is added to the Declarations list of the Aux_Decls_Node
   --  for the current unit. The declared entity is added to current scope,
   --  so the caller should push a new scope as required before the call.

   procedure Add_Local_Declaration
     (Decl : Node_Id;
      N    : Node_Id;
      Scop : Entity_Id);
   --  This procedure adds a declaration Decl to the innermost declarative
   --  part that covers N, whose associated scope is Scop if Scop is present,
   --  and before N if N is in this declarative part. The declared entity is
   --  added to the scope associated with the declarative part.

   function Add_Suffix (E : Entity_Id; Suffix : Character) return Name_Id;
   --  Returns the name of E adding Suffix

   function Address_Integer_Convert_OK (T1, T2 : Entity_Id) return Boolean;
   --  Given two types, returns True if we are in Allow_Integer_Address mode
   --  and one of the types is (a descendant of) System.Address (and this type
   --  is private), and the other type is any integer type.

   function Address_Value (N : Node_Id) return Node_Id;
   --  Return the underlying value of the expression N of an address clause

   function Addressable (V : Uint) return Boolean;
   function Addressable (V : Int)  return Boolean;
   pragma Inline (Addressable);
   --  Returns True if the value of V is the word size or an addressable factor
   --  or multiple of the word size (typically 8, 16, 32, 64 or 128).

   procedure Aggregate_Constraint_Checks
     (Exp       : Node_Id;
      Check_Typ : Entity_Id);
   --  Checks expression Exp against subtype Check_Typ. If Exp is an aggregate
   --  and Check_Typ a constrained record type with discriminants, we generate
   --  the appropriate discriminant checks. If Exp is an array aggregate then
   --  emit the appropriate length checks. If Exp is a scalar type, or a string
   --  literal, Exp is changed into Check_Typ'(Exp) to ensure that range checks
   --  are performed at run time. Also used for expressions in the argument of
   --  'Update, which shares some of the features of an aggregate.

   function Alignment_In_Bits (E : Entity_Id) return Uint;
   --  If the alignment of the type or object E is currently known to the
   --  compiler, then this function returns the alignment value in bits.
   --  Otherwise Uint_0 is returned, indicating that the alignment of the
   --  entity is not yet known to the compiler.

   function All_Composite_Constraints_Static (Constr : Node_Id) return Boolean;
   --  Used to implement pragma Restrictions (No_Dynamic_Sized_Objects).
   --  Given a constraint or subtree of a constraint on a composite
   --  subtype/object, returns True if there are no nonstatic constraints,
   --  which might cause objects to be created with dynamic size.
   --  Called for subtype declarations (including implicit ones created for
   --  subtype indications in object declarations, as well as discriminated
   --  record aggregate cases). For record aggregates, only records containing
   --  discriminant-dependent arrays matter, because the discriminants must be
   --  static when governing a variant part. Access discriminants are
   --  irrelevant. Also called for array aggregates, but only named notation,
   --  because those are the only dynamic cases.

   procedure Append_Entity_Name (Buf : in out Bounded_String; E : Entity_Id);
   --  Construct a user-readable expanded name for E, for printing in messages,
   --  such as run-time errors for unhandled exceptions. Names created for
   --  internal use are not included. The name is appended to Buf.

   procedure Append_Inherited_Subprogram (S : Entity_Id);
   --  If the parent of the operation is declared in the visible part of
   --  the current scope, the inherited operation is visible even though the
   --  derived type that inherits the operation may be completed in the private
   --  part of the current package.

   procedure Apply_Compile_Time_Constraint_Error
     (N            : Node_Id;
      Msg          : String;
      Reason       : RT_Exception_Code;
      Ent          : Entity_Id  := Empty;
      Typ          : Entity_Id  := Empty;
      Loc          : Source_Ptr := No_Location;
      Warn         : Boolean    := False;
      Emit_Message : Boolean    := True);
   --  N is a subexpression that will raise Constraint_Error when evaluated
   --  at run time. Msg is a message that explains the reason for raising the
   --  exception. The last character is ? if the message is always a warning,
   --  even in Ada 95, and is not a ? if the message represents an illegality
   --  (because of violation of static expression rules) in Ada 95 (but not
   --  in Ada 83). Typically this routine posts all messages at the Sloc of
   --  node N. However, if Loc /= No_Location, Loc is the Sloc used to output
   --  the message. After posting the appropriate message, this routine
   --  replaces the expression with an appropriate N_Raise_Constraint_Error
   --  node using the given Reason code. This node is then marked as being
   --  static if the original node is static, but sets the flag
   --  Raises_Constraint_Error, preventing further evaluation. The error
   --  message may contain a } or & insertion character. This normally
   --  references Etype (N), unless the Ent argument is given explicitly, in
   --  which case it is used instead. The type of the raise node that is built
   --  is normally Etype (N), but if the Typ parameter is present, this is used
   --  instead. Warn is normally False. If it is True then the message is
   --  treated as a warning even though it does not end with a ? (this is used
   --  when the caller wants to parameterize whether an error or warning is
   --  given), or when the message should be treated as a warning even when
   --  SPARK_Mode is On (which otherwise would force an error).
   --  If Emit_Message is False, then do not emit any message.

   function Async_Readers_Enabled (Id : Entity_Id) return Boolean;
   --  Id should be the entity of a state abstraction, an object, or a type.
   --  Returns True iff Id is subject to external property Async_Readers.

   function Async_Writers_Enabled (Id : Entity_Id) return Boolean;
   --  Id should be the entity of a state abstraction, an object, or a type.
   --  Returns True iff Id is subject to external property Async_Writers.

   function Available_Full_View_Of_Component (T : Entity_Id) return Boolean;
   --  If at the point of declaration an array type has a private or limited
   --  component, several array operations are not available on the type, and
   --  the array type is flagged accordingly. If in the immediate scope of
   --  the array type the component becomes non-private or non-limited, these
   --  operations become available. This can happen if the scopes of both types
   --  are open, and the scope of the array is not outside the scope of the
   --  component.

   procedure Bad_Aspect
     (N    : Node_Id;
      Nam  : Name_Id;
      Warn : Boolean := False);
   --  Called when node N is expected to contain a valid aspect name, and
   --  Nam is found instead. If Warn is set True this is a warning, else this
   --  is an error.

   procedure Bad_Attribute
     (N    : Node_Id;
      Nam  : Name_Id;
      Warn : Boolean := False);
   --  Called when node N is expected to contain a valid attribute name, and
   --  Nam is found instead. If Warn is set True this is a warning, else this
   --  is an error.

   procedure Bad_Predicated_Subtype_Use
     (Msg            : String;
      N              : Node_Id;
      Typ            : Entity_Id;
      Suggest_Static : Boolean := False);
   --  This is called when Typ, a predicated subtype, is used in a context
   --  which does not allow the use of a predicated subtype. Msg is passed to
   --  Error_Msg_FE to output an appropriate message using N as the location,
   --  and Typ as the entity. The caller must set up any insertions other than
   --  the & for the type itself. Note that if Typ is a generic actual type,
   --  then the message will be output as a warning, and a raise Program_Error
   --  is inserted using Insert_Action with node N as the insertion point. Node
   --  N also supplies the source location for construction of the raise node.
   --  If Typ does not have any predicates, the call has no effect. Set flag
   --  Suggest_Static when the context warrants an advice on how to avoid the
   --  use error.

   function Bad_Unordered_Enumeration_Reference
     (N : Node_Id;
      T : Entity_Id) return Boolean;
   --  Node N contains a potentially dubious reference to type T, either an
   --  explicit comparison, or an explicit range. This function returns True
   --  if the type T is an enumeration type for which No pragma Order has been
   --  given, and the reference N is not in the same extended source unit as
   --  the declaration of T.

   function Begin_Keyword_Location (N : Node_Id) return Source_Ptr;
   --  Given block statement, entry body, package body, subprogram body, or
   --  task body N, return the closest source location to the "begin" keyword.

   function Build_Actual_Subtype
     (T : Entity_Id;
      N : Node_Or_Entity_Id) return Node_Id;
   --  Build an anonymous subtype for an entity or expression, using the
   --  bounds of the entity or the discriminants of the enclosing record.
   --  T is the type for which the actual subtype is required, and N is either
   --  a defining identifier, or any subexpression.

   function Build_Actual_Subtype_Of_Component
     (T : Entity_Id;
      N : Node_Id) return Node_Id;
   --  Determine whether a selected component has a type that depends on
   --  discriminants, and build actual subtype for it if so.

   --  Handling of inherited primitives whose ancestors have class-wide
   --  pre/postconditions.

   --  If a primitive operation of a parent type has a class-wide pre/post-
   --  condition that includes calls to other primitives, and that operation
   --  is inherited by a descendant type that also overrides some of these
   --  other primitives, the condition that applies to the inherited
   --  operation has a modified condition in which the overridden primitives
   --  have been replaced by the primitives of the descendent type. A call
   --  to the inherited operation cannot be simply a call to the parent
   --  operation (with an appropriate conversion) as is the case for other
   --  inherited operations, but must appear with a