/***************************** LICENSE START ***********************************

 Copyright 2017 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#ifndef LOCATIONVIEW_H
#define LOCATIONVIEW_H

#include <QGraphicsItem>
#include <QGraphicsPixmapItem>
#include <QGraphicsView>
#include <QGraphicsScene>
#include <QGraphicsSvgItem>

class QGraphicsSvgItem;
class QSvgRenderer;
class LocationView;

class MapLayerItem : public QGraphicsItem
{
public:
    enum
    {
        Type = UserType + 1
    };
    MapLayerItem();
    QRectF boundingRect() const;
    void paint(QPainter* painter, const QStyleOptionGraphicsItem*, QWidget*);
    int type() const { return Type; }

protected:
    QPixmap pix_;
    QRectF bRect_;
    QRectF srcRect_;
    QSvgRenderer* svg_;
};

class MapMarkerItem : public QGraphicsItem
{
public:
    enum
    {
        Type = UserType + 2
    };
    enum Symbol
    {
        CircleSymbol,
        CrossSymbol
    };

    MapMarkerItem(int id);

    QRectF boundingRect() const;
    void paint(QPainter* painter, const QStyleOptionGraphicsItem*, QWidget*);
    void setCurrent(bool);
    int type() const { return Type; }
    int id() const { return id_; }
    static void setSymbol(Symbol sym, float size);

protected:
    int id_;
    bool current_;
    static Symbol symbol_;
    static float symbolSize_;
    static QRectF bRect_;
    static QBrush brush_;
    static QBrush selectBrush_;
    static QPen pen_;
    static QPen crossPen_;
    static QPen selectPen_;
    static QPen crossSelectPen_;
};

class LocationView : public QGraphicsView
{
    Q_OBJECT

public:
    LocationView(QWidget* parent = 0);
    void addPoints(const std::vector<double>& lat,
                   const std::vector<double>& lon);
    void setCurrentPoint(int);
    void clearPoints();
    int minZoomLevel() const { return minZoomLevel_; }
    int maxZoomLevel() const { return maxZoomLevel_; }
    int zoomLevel() const { return zoomLevel_; }
    void setZoomLevel(int);
    float currentScale() const;

public Q_SLOTS:
    void showFullMap();
    void zoomToData();
    void zoomIn();
    void zoomOut();

Q_SIGNALS:
    void currentChanged(int);
    void zoomLevelChanged(int);

protected:
    void keyPressEvent(QKeyEvent* event);
    void wheelEvent(QWheelEvent* e);
    void mousePressEvent(QMouseEvent* event);
    void mouseMoveEvent(QMouseEvent* event);
    void mouseReleaseEvent(QMouseEvent* event);
    void zoom(int level);
    void pan(QPointF delta);
    float scaleFromLevel(int level) const;
    int zoomLevelFromScale(float) const;
    void adjustMarkers(int);

    QGraphicsScene* scene_;
    MapLayerItem* layer_;
    MapLayerItem* svg_;

    std::vector<MapMarkerItem*> markers_;
    int minZoomLevel_;
    int maxZoomLevel_;
    int zoomLevel_;
    float zoomDelta_;
    QRectF dataRect_;
    MapMarkerItem* selectedItem_;
    QPoint dragPos_;
    bool inDrag_;
};

#endif  // LOCATIONVIEW_H
