/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package zbxflag

import (
	"testing"

	"github.com/google/go-cmp/cmp"
)

func TestFlag_Usage(t *testing.T) {
	t.Parallel()

	type fields struct {
		Name        string
		Shorthand   string
		Description string
	}

	tests := []struct {
		name   string
		fields fields
		want   string
	}{
		{
			"+valid",
			fields{"config", "c", "config file"},
			"  -c --config                             config file",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			f := &Flag{
				Name:        tt.fields.Name,
				Shorthand:   tt.fields.Shorthand,
				Description: tt.fields.Description,
			}

			got := f.Usage()

			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("Flag.Usage() = %s", diff)
			}
		})
	}
}

func TestFlags_Usage(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name  string
		flags Flags
		want  string
	}{
		{
			"+valid",
			[]Flager{
				&BoolFlag{
					Flag: Flag{
						Name:        "version",
						Shorthand:   "V",
						Description: "print version number",
					},
				},
				&StringFlag{
					Flag: Flag{
						Name:        "config",
						Shorthand:   "c",
						Description: "config file",
					},
				},
			},
			"  -V --version                            print version number\n" +
				"  -c --config                             config file",
		},
		{"-empty", []Flager{}, ""},
		{"-nil", nil, ""},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got := tt.flags.Usage()
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("Flags.Usage() = %s", diff)
			}
		})
	}
}
