// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package secret

import (
	"context"
	"testing"

	"github.com/drone/drone/core"
)

var noContext = context.Background()

func TestStatic(t *testing.T) {
	secrets := []*core.Secret{
		{Name: "docker_username"},
		{Name: "docker_password"},
	}
	args := &core.SecretArgs{
		Name:  "docker_password",
		Build: &core.Build{Event: core.EventPush},
	}
	service := Static(secrets)
	secret, err := service.Find(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}
	if secret != secrets[1] {
		t.Errorf("expect docker_password")
	}
}

func TestStaticNotFound(t *testing.T) {
	secrets := []*core.Secret{
		{Name: "docker_username"},
		{Name: "docker_password"},
	}
	args := &core.SecretArgs{
		Name:  "slack_token",
		Build: &core.Build{Event: core.EventPush},
	}
	service := Static(secrets)
	secret, err := service.Find(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}
	if secret != nil {
		t.Errorf("Expect secret not found")
	}
}

func TestStaticPullRequestDisabled(t *testing.T) {
	secrets := []*core.Secret{
		{Name: "docker_username"},
		{Name: "docker_password", PullRequest: false},
	}
	args := &core.SecretArgs{
		Name:  "docker_password",
		Build: &core.Build{Event: core.EventPullRequest},
	}
	service := Static(secrets)
	secret, err := service.Find(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}
	if secret != nil {
		t.Errorf("Expect secret not found")
	}
}

func TestStaticPullRequestEnabled(t *testing.T) {
	secrets := []*core.Secret{
		{Name: "docker_username"},
		{Name: "docker_password", PullRequest: true},
	}
	args := &core.SecretArgs{
		Name:  "docker_password",
		Build: &core.Build{Event: core.EventPullRequest},
	}
	service := Static(secrets)
	secret, err := service.Find(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}
	if err != nil {
		t.Error(err)
		return
	}
	if secret != secrets[1] {
		t.Errorf("expect docker_username")
	}
}
