// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package repo

import (
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/go-scm/scm"

	"github.com/google/go-cmp/cmp"
)

func TestConvertRepository(t *testing.T) {
	from := &scm.Repository{
		ID:        "42",
		Namespace: "octocat",
		Name:      "hello-world",
		Branch:    "master",
		Private:   true,
		Clone:     "https://github.com/octocat/hello-world.git",
		CloneSSH:  "git@github.com:octocat/hello-world.git",
		Link:      "https://github.com/octocat/hello-world",
	}
	want := &core.Repository{
		UID:        "42",
		Namespace:  "octocat",
		Name:       "hello-world",
		Slug:       "octocat/hello-world",
		HTTPURL:    "https://github.com/octocat/hello-world.git",
		SSHURL:     "git@github.com:octocat/hello-world.git",
		Link:       "https://github.com/octocat/hello-world",
		Private:    true,
		Branch:     "master",
		Visibility: core.VisibilityPrivate,
	}
	got := convertRepository(from, "", false)
	if diff := cmp.Diff(want, got); len(diff) != 0 {
		t.Errorf(diff)
	}
}

func TestConvertVisibility(t *testing.T) {
	tests := []struct {
		r *scm.Repository
		v string
	}{
		{
			r: &scm.Repository{Private: false},
			v: core.VisibilityPublic,
		},
		{
			r: &scm.Repository{Private: true},
			v: core.VisibilityPrivate,
		},
		{
			r: &scm.Repository{Private: true, Visibility: scm.VisibilityInternal},
			v: core.VisibilityInternal,
		},
	}

	for i, test := range tests {
		if got, want := convertVisibility(test.r, ""), test.v; got != want {
			t.Errorf("Want visibility %s, got %s for index %d", got, want, i)
		}
	}
}

func TestDefinedVisibility(t *testing.T) {
	from := &scm.Repository{
		ID:        "42",
		Namespace: "octocat",
		Name:      "hello-world",
		Branch:    "master",
		Private:   false,
		Clone:     "https://github.com/octocat/hello-world.git",
		CloneSSH:  "git@github.com:octocat/hello-world.git",
		Link:      "https://github.com/octocat/hello-world",
	}
	want := &core.Repository{
		UID:        "42",
		Namespace:  "octocat",
		Name:       "hello-world",
		Slug:       "octocat/hello-world",
		HTTPURL:    "https://github.com/octocat/hello-world.git",
		SSHURL:     "git@github.com:octocat/hello-world.git",
		Link:       "https://github.com/octocat/hello-world",
		Private:    false,
		Branch:     "master",
		Visibility: core.VisibilityInternal,
	}
	got := convertRepository(from, "internal", false)
	if diff := cmp.Diff(want, got); len(diff) != 0 {
		t.Errorf(diff)
	}
}
