package ntlmssp

import (
	"errors"
	"io"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestChallengeMessageUnmarshal(t *testing.T) {
	tables := []struct {
		got  []byte
		want challengeMessage
		err  error
	}{
		{
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x37, 0x82, 0x8a, 0x82,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x00, 0x24, 0x00, 0x44, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x02, 0x00, 0x0c, 0x00,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x01, 0x00, 0x0c, 0x00,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			challengeMessage{
				challengeMessageFields: challengeMessageFields{
					messageHeader: newMessageHeader(ntLmChallenge),
					TargetNameFields: payload{
						Len:    12,
						MaxLen: 12,
						Offset: 56,
					},
					NegotiateFlags: 0x828a8237,
					ServerChallenge: [8]byte{
						0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
					},
					TargetInfoFields: payload{
						Len:    36,
						MaxLen: 36,
						Offset: 68,
					},
				},
				TargetName: "Server",
				TargetInfo: targetInfo{
					map[avID][]uint8{
						msvAvNbComputerName: {
							0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
							0x65, 0x00, 0x72, 0x00,
						},
						msvAvNbDomainName: {
							0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
							0x69, 0x00, 0x6e, 0x00,
						},
					},
					[]avID{
						msvAvNbDomainName,
						msvAvNbComputerName,
					},
				},
				Version: &Version{
					ProductMajorVersion: 6,
					ProductMinorVersion: 0,
					ProductBuild:        6000,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
			},
			nil,
		},
		{
			// 47 bytes to force a short challengeMessageFields read
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x37, 0x82, 0x8a, 0x82,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x00, 0x24, 0x00, 0x44, 0x00, 0x00,
			},
			challengeMessage{},
			io.ErrUnexpectedEOF,
		},
		{
			// 9th byte changed to force invalid challenge message
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x03, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x37, 0x82, 0x8a, 0x82,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x00, 0x24, 0x00, 0x44, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x02, 0x00, 0x0c, 0x00,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x01, 0x00, 0x0c, 0x00,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			challengeMessage{},
			errInvalidChallengeMessage,
		},
		{
			// 67 bytes to force a short TargetName read
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x37, 0x82, 0x8a, 0x82,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x00, 0x24, 0x00, 0x44, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72,
			},
			challengeMessage{},
			errors.New("expected 12 bytes, only read 11"),
		},
		{
			// 103 bytes to force a short TargetInfo read
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x37, 0x82, 0x8a, 0x82,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x00, 0x24, 0x00, 0x44, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x02, 0x00, 0x0c, 0x00,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x01, 0x00, 0x0c, 0x00,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00,
			},
			challengeMessage{},
			errors.New("expected 36 bytes, only read 35"),
		},
		{
			// 103 bytes with a malformed TargetInfo block
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x37, 0x82, 0x8a, 0x82,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x23, 0x00, 0x23, 0x00, 0x44, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x02, 0x00, 0x0c, 0x00,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x01, 0x00, 0x0c, 0x00,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00,
			},
			challengeMessage{},
			io.ErrUnexpectedEOF,
		},
		{
			// 55 bytes to force a short Version read
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x37, 0x82, 0x8a, 0x82,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00,
			},
			challengeMessage{},
			io.ErrUnexpectedEOF,
		},
	}

	for _, table := range tables {
		m := challengeMessage{}
		err := m.Unmarshal(table.got)
		assert.Equal(t, table.err, err)
		if err == nil {
			assert.Equal(t, table.want, m)
		}
	}
}

func TestChallengeMessageMarshal(t *testing.T) {
	tables := []struct {
		got  challengeMessage
		want []byte
		err  error
	}{
		{
			challengeMessage{
				challengeMessageFields: challengeMessageFields{
					messageHeader: newMessageHeader(ntLmChallenge),
					TargetNameFields: payload{
						Len:    12,
						MaxLen: 12,
						Offset: 56,
					},
					NegotiateFlags: 0x828a8237,
					ServerChallenge: [8]byte{
						0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
					},
					TargetInfoFields: payload{
						Len:    36,
						MaxLen: 36,
						Offset: 68,
					},
				},
				TargetName: "Server",
				TargetInfo: targetInfo{
					map[avID][]uint8{
						msvAvNbComputerName: {
							0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
							0x65, 0x00, 0x72, 0x00,
						},
						msvAvNbDomainName: {
							0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
							0x69, 0x00, 0x6e, 0x00,
						},
					},
					[]avID{
						msvAvNbComputerName,
						msvAvNbDomainName,
					},
				},
				Version: &Version{
					ProductMajorVersion: 6,
					ProductMinorVersion: 0,
					ProductBuild:        6000,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x37, 0x82, 0x8a, 0x82,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x00, 0x24, 0x00, 0x44, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x01, 0x00, 0x0c, 0x00,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x02, 0x00, 0x0c, 0x00,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			nil,
		},
		{
			challengeMessage{
				challengeMessageFields: challengeMessageFields{
					messageHeader: newMessageHeader(ntLmChallenge),
					TargetNameFields: payload{
						Len:    12,
						MaxLen: 12,
						Offset: 56,
					},
					NegotiateFlags: 0x828a8236,
					ServerChallenge: [8]byte{
						0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
					},
					TargetInfoFields: payload{
						Len:    36,
						MaxLen: 36,
						Offset: 68,
					},
				},
				TargetName: "Server",
				TargetInfo: targetInfo{
					map[avID][]uint8{
						msvAvNbComputerName: {
							0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
							0x65, 0x00, 0x72, 0x00,
						},
						msvAvNbDomainName: {
							0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
							0x69, 0x00, 0x6e, 0x00,
						},
					},
					[]avID{
						msvAvNbComputerName,
						msvAvNbDomainName,
					},
				},
				Version: &Version{
					ProductMajorVersion: 6,
					ProductMinorVersion: 0,
					ProductBuild:        6000,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x06, 0x00, 0x06, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x36, 0x82, 0x8a, 0x82,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x00, 0x24, 0x00, 0x3e, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x01, 0x00,
				0x0c, 0x00, 0x53, 0x00, 0x65, 0x00, 0x72, 0x00,
				0x76, 0x00, 0x65, 0x00, 0x72, 0x00, 0x02, 0x00,
				0x0c, 0x00, 0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00,
				0x61, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x00, 0x00,
				0x00, 0x00,
			},
			nil,
		},
	}

	for _, table := range tables {
		b, err := table.got.Marshal()
		assert.Equal(t, table.err, err)
		if err == nil {
			assert.Equal(t, table.want, b)
		}
	}
}
