// Licensed to Elasticsearch B.V. under one or more contributor
// license agreements. See the NOTICE file distributed with
// this work for additional information regarding copyright
// ownership. Elasticsearch B.V. licenses this file to you under
// the Apache License, Version 2.0 (the "License"); you may
// not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

//go:build linux && amd64
// +build linux,amd64

package rule_test

import (
	"encoding/binary"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
	"gopkg.in/yaml.v2"

	"github.com/elastic/go-libaudit/v2/rule"
	"github.com/elastic/go-libaudit/v2/rule/flags"
	"github.com/elastic/go-libaudit/v2/sys"
)

var tempDir = "/tmp/audit-test"

type GoldenData struct {
	Rules []TestCase `yaml:"rules"`
}

type TestCase struct {
	Flags string `yaml:"flags"`
	Bytes string `yaml:"bytes"`
}

// TestBuild compares the WireFormat (binary data) generated by this package
// against known golden data generated on Linux with auditctl.
func TestBuildGolden(t *testing.T) {
	if sys.GetEndian() != binary.LittleEndian {
		t.Skip("golden test data is for little endian, but test machine is big endian")
	}

	goldenFiles, err := filepath.Glob("testdata/*.rules.golden.yml")
	if err != nil {
		t.Fatal(err)
	}

	for _, goldenFile := range goldenFiles {
		testRulesFromGoldenFile(t, goldenFile)
	}
}

func testRulesFromGoldenFile(t *testing.T, goldenFile string) {
	t.Run(filepath.Base(goldenFile), func(t *testing.T) {
		testdata, err := ioutil.ReadFile(goldenFile)
		if err != nil {
			t.Fatal(err)
		}

		var tests GoldenData
		if err := yaml.Unmarshal(testdata, &tests); err != nil {
			t.Fatal(err)
		}

		for i, test := range tests.Rules {
			t.Run(fmt.Sprintf("rule %d", i), func(t *testing.T) {
				defer os.RemoveAll(tempDir)

				if testing.Verbose() {
					t.Log("rule:", test.Flags)
				}

				r, err := flags.Parse(test.Flags)
				if err != nil {
					t.Fatal("rule:", test.Flags, "error:", err)
				}

				if v, ok := r.(*rule.FileWatchRule); ok {
					mkdirTempPaths(t, v.Path)
				}

				actualBytes, err := rule.Build(r)
				if err != nil {
					t.Fatal("rule:", test.Flags, "error:", err)
				}

				// fmt.Println(hex.Dump([]byte(actualBytes)))
				assert.EqualValues(t, []byte(test.Bytes), []byte(actualBytes), "rule: %v", test.Flags)
			})
		}
	})
}

// mkdirThenCleanup create a directory on the system if the rule requires a
// directory to be present.
func mkdirTempPaths(t testing.TB, path string) {
	if !strings.HasPrefix(path, tempDir) {
		t.Fatalf("path is not inside the test temp dir (%v): %v", tempDir, path)
	}

	if strings.HasSuffix(path, "/") {
		if err := os.MkdirAll(path, 0o700); err != nil {
			t.Fatal(err)
		}
	} else {
		// Touch a file.
		dir := filepath.Dir(path)
		if err := os.MkdirAll(dir, 0o700); err != nil {
			t.Fatal(err)
		}
		if err := ioutil.WriteFile(path, nil, 0o600); err != nil {
			t.Fatal(err)
		}
	}
}
