#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <sys/types.h>

#include <lufs/proto.h>
#include <lufs/fs.h>

#include "list.h"
#include "vtree.h"


struct vtree*
lu_vtree_create(struct lufs_fattr *root_fattr){
    struct vtree *vt;

    TRACE("creating vtree...");

    if(!(vt = malloc(sizeof(struct vtree))))
	return NULL;

    memset(vt, 0, sizeof(struct vtree));

    INIT_LIST_HEAD(&vt->root.children);
    memcpy(&vt->root.fattr, root_fattr, sizeof(struct lufs_fattr));
    
    vt->root.vtree = vt;
    vt->root.name = "/";

    vt->root.stamp = time(NULL);
    
    return vt;
}

void
lu_vtree_delete(struct ventry *ve){
    struct list_head *p, *tmp;

    list_for_each_safe(p, tmp, &ve->children)
	lu_vtree_delete(list_entry(p, struct ventry, list));

    TRACE("deleting %s", ve->name);

    ve->vtree->entries--;
    list_del(&ve->list);

    free(ve->name);
    if(ve->link)
	free(ve->link);

    free(ve);
}

void
lu_vtree_destroy(struct vtree *vt){
    struct list_head *p, *tmp;

    TRACE("deleting tree");

    list_for_each_safe(p, tmp, &vt->root.children)
	lu_vtree_delete(list_entry(p, struct ventry, list));

    free(vt);
}

struct ventry*
lu_vtree_search(struct ventry *ve, char *name){
    struct list_head *p;    
    char *sep;

    TRACE("searching for %s in %s", name, ve->name);
    
    do{
	if((sep = strchr(name, '/'))){
	    *sep = 0;
	}

	TRACE("scanning for %s in %s", name, ve->name);

	list_for_each(p, &ve->children){
	    if(!strcmp(name, list_entry(p, struct ventry, list)->name)){
		TRACE("portion found: %s", list_entry(p, struct ventry, list)->name);
		ve = list_entry(p, struct ventry, list);
		break;
	    }
	}

	if(strcmp(name, ve->name)){
	    TRACE("portion not found");
	    return NULL;
	}

	if(sep)
	    *sep = '/';

	name = sep + 1;

    }while(sep);

    TRACE("entry found");

    return ve;
}

struct ventry*
lu_vtree_find(struct vtree *vt, char *entry){

    TRACE("entry: %s", entry);

    if(*entry != '/'){
	TRACE("entry is not an absolute path");
	return NULL;
    }

    if(!strcmp(entry, "/"))
	return &vt->root;
    
    return lu_vtree_search(&vt->root, entry + 1);
}

int
lu_vtree_add(struct vtree *vt, char *dir, char *name, char *link, struct lufs_fattr *fattr, void *private){
    struct ventry *ve, *new;

    TRACE("add %s to %s", name, dir);

    if(!(ve = lu_vtree_find(vt, dir)))
	return -1;

    if(!(new = lu_vtree_search(ve, name))){
	TRACE("allocating new entry");

	if(!(new = malloc(sizeof(struct ventry))))
	    return -1;

	memset(new, 0, sizeof(struct ventry));
	
	INIT_LIST_HEAD(&new->children);
	new->vtree = vt;

	vt->entries++;
	list_add_tail(&new->list, &ve->children);
    }else{
	TRACE("emtry already in tree");

	free(new->name);
	if(new->link)
	    free(new->link);
    }

    if(!(new->name = malloc(strlen(name) + 1)))
	goto fail_entry;

    if(link && !(new->link = malloc(strlen(link) + 1)))
	goto fail_name;

    strcpy(new->name, name);
    if(link)
	strcpy(new->link, link);
    
    memcpy(&new->fattr, fattr, sizeof(struct lufs_fattr));

    new->private = private;
    new->stamp = time(NULL);

    return 0;

  fail_name:
    free(new->name);
  fail_entry:
    vt->entries--;
    list_del(&new->list);
    free(new);
    return -1;
}

int
lu_vtree_lookup(struct vtree *vt, char *file, struct lufs_fattr *fattr, char *link, int buflen, void **private){
    struct ventry *ve;

    TRACE("looking up %s", file);

    if(*file != '/'){
	TRACE("we need an absolute path here...");
	return -1;
    }

    if(!strcmp(file, "/"))
	ve = &vt->root;
    else 
	if(!(ve = lu_vtree_search(&vt->root, file + 1)))
	    return -1;
    
    TRACE("file found");

    memcpy(fattr, &ve->fattr, sizeof(struct lufs_fattr));
    if(link){
	if(ve->link){
	    if(snprintf(link, buflen, "%s", ve->link) >= buflen){
		WARN("link too long!");
		link[buflen - 1] = 0;
	    }
	}else{
	    link[0] = 0;
	}
    }

    if(private)
	*private = ve->private;
    
    return 0;
}

int
lu_vtree_readdir(struct vtree *vt, char *dir, int offset, char *buf, int buflen){
    struct ventry *ve, *e;
    struct list_head *p;
    unsigned slen, off = 0, len = 0;

    TRACE("reading directory %s", dir);

    if(*dir != '/'){
	TRACE("we need an absolute path here...");
	return -1;
    }

    if(!strcmp(dir, "/"))
	ve = &vt->root;
    else 
	if(!(ve = lu_vtree_search(&vt->root, dir + 1)))
	    return -1;

    TRACE("directory found");

    buf[0] = 0;

    list_for_each(p, &ve->children){
	if(off >= offset){
	    e = list_entry(p, struct ventry, list);
	    slen = strlen(e->name);

	    if(len + slen + 2 >= buflen){
		TRACE("buffer filled up");
		break;
	    }

	    strcat(buf, e->name);
	    strcat(buf, "\n");

	    len += slen + 1;
	}

	off++;
    }

    return len;
}

