/*
 *  Print Dialog for Canon LIPS/PS/LIPSLX/UFR2/CAPT Printer.
 *  Copyright CANON INC. 2017
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "options.h"

#ifdef DRIVER_TYPE_UFR2
#define CNGPLP_OPTION_PATH						"/etc/cngplp2/options/"
#elif  DRIVER_TYPE_BLANCA
#define CNGPLP_OPTION_PATH						"/etc/cngplp2l/options/"
#elif  DRIVER_TYPE_LIPS
#define CNGPLP_OPTION_PATH						"/etc/cngplpl/options/"
#else
#define CNGPLP_OPTION_PATH						"/etc/cngplp/options/"
#endif

#define CNGPLP_OPTION_FILE_TYPE_CONF_READ		1
#define CNGPLP_OPTION_FILE_TYPE_CONF_WRITE		2
#define CNGPLP_OPTION_FILE_CONF					"options.conf"
#define CNGPLP_OPTION_FILE_CONF_NEW				"options.conf.new"

#define	SEPARATOR								","
#define	SPACE									" "
#define	DOUBLE_QUART							"\""
#define	CNGPLP_OPTION_KEY_CNGSADDITIONALOPTIONS	"*CNGSAdditionalOptions:"
#define	CNGPLP_OPTION_CNGSADDITIONALOPTIONS_USECIECOLOR	"-dUseCIEColor"
#define CNGPLP_OPTION_KEY_CNCMCORRECTGRAY	"*CNCMCorrectGray:"
#define CNGPLP_OPTION_CNCMCORRECTGRAY_TRUE	"TRUE"
#define	REAR_ARGUMENT							"-"
#define	LINE_BREAK								"\n"
#define CNGPLP_OPTION_KEY_SPACE_LEN				1

#define	LINE_BUF_SIZE	1024
#define	BUFSIZE			256

#define IS_BLANK(c)		(c == ' '  || c == '\t')
#define IS_RETURN(c)	(c == '\r' || c == '\n')

static int exist_file(const char *file_name);
static char* make_file_path(int type, char *filename);
static int file_write(int fd, char *line, int line_size);
static char* get_option_value(char* pKey);
static int GetValue(const char *p_line_buf, char *p_value_buf, int value_len);

static
int exist_file(const char *file_name)
{
	int ret = 0;

	if(file_name != NULL){
		struct stat info;

		ret = stat(file_name, &info);
	}
	return (ret == 0) ? 0 : 1;
}

static
char* make_file_path(int type, char *filename)
{
	char path[256];
	char *resultPath = NULL;

	memset(path, 0, sizeof(char)*256);
	strncpy(path, CNGPLP_OPTION_PATH, 255);
	strncat(path, "/", 255 - strlen(path));
	strncat(path, filename, 255 - strlen(path));

	switch(type){
	case CNGPLP_OPTION_FILE_TYPE_CONF_READ:
		if(exist_file(path) == 0) {
			resultPath =  strdup(path);
		}
		break;
	case CNGPLP_OPTION_FILE_TYPE_CONF_WRITE:
		resultPath =  strdup(path);
		break;
	default:
		break;
	}

	return resultPath;
}

static
int file_write(int fd, char *line, int line_size)
{
	int w_bytes = 0;
	int offset = 0;

	fsync(fd);
	do {
		w_bytes = write(fd, line + offset, line_size - offset);
		if(w_bytes + offset == line_size) {
			break;
		}
		offset += w_bytes;
	} while(w_bytes + offset < line_size);
	sync();

	return 0;
}

static
int GetValue(const char *p_line_buf, char *p_value_buf, int value_len)
{
	int val_pos = 0;
	char *pDoubleQuotation = NULL;

	pDoubleQuotation = strchr(p_line_buf, '"');
	if( pDoubleQuotation != NULL ) {
		int pos = 0;
		pos = (pDoubleQuotation - p_line_buf);
		pos++;

		pDoubleQuotation = strchr(&p_line_buf[pos], '"');
		if( pDoubleQuotation != NULL ) {
			while( !IS_RETURN(p_line_buf[pos]) && (p_line_buf[pos] != '"') && (val_pos < value_len - 1))
				p_value_buf[val_pos++] = p_line_buf[pos++];

			p_value_buf[val_pos++] = 0;
		}
	}

	return val_pos;
}

static
char* get_option_value(char* pKey)
{
	char	*pValue = NULL;

	if (pKey != NULL) {
		char	*p_option_file_name = make_file_path(CNGPLP_OPTION_FILE_TYPE_CONF_READ, CNGPLP_OPTION_FILE_CONF);

		if (p_option_file_name != NULL) {
			FILE	*fp = NULL;
			fp = fopen(p_option_file_name, "r");
			if (fp != NULL) {
				char	line_buf[LINE_BUF_SIZE];

				while (fgets(line_buf, sizeof(line_buf), fp)) {
					char	*pColon = NULL;
					pColon = strchr(line_buf, ':');
					if(pColon != NULL) {
				 		pColon++;
						if (strncmp(line_buf, pKey, (pColon-line_buf)) == 0) {
							pValue = (char*)calloc((size_t)BUFSIZE, sizeof(char));
							if (pValue != NULL) {
								(void)GetValue((line_buf + (int)strlen(pKey)), pValue, BUFSIZE);
							}
							break;
						}
					}
				}
				fclose(fp);
			}
			free(p_option_file_name);
		}
	}
	return pValue;
}

void SaveCngplpOption(const CNGPLP_OPTIONS *options)
{
	if (options != NULL) {
		char *file_org = NULL;

		file_org = make_file_path(CNGPLP_OPTION_FILE_TYPE_CONF_WRITE, CNGPLP_OPTION_FILE_CONF);
		if(file_org != NULL) {
			char *file_new = NULL;

			file_new = make_file_path(CNGPLP_OPTION_FILE_TYPE_CONF_WRITE, CNGPLP_OPTION_FILE_CONF_NEW);
			if(file_new != NULL) {
				int fd_new = 0;

				fd_new = open(file_new, O_RDWR | O_CREAT | O_EXCL, 0666);

				if(fd_new >= 0) {
					char line_buf[LINE_BUF_SIZE] = {0};

					strncat(line_buf, CNGPLP_OPTION_KEY_CNGSADDITIONALOPTIONS, strlen(CNGPLP_OPTION_KEY_CNGSADDITIONALOPTIONS) + 1);
					strncat(line_buf, SPACE, strlen(SPACE) + 1);
					strncat(line_buf, DOUBLE_QUART, strlen(DOUBLE_QUART) + 1);

					if (options->use_cie_color == kOPTION_USE_CIE_COLOR_ON) {
						strncat(line_buf, CNGPLP_OPTION_CNGSADDITIONALOPTIONS_USECIECOLOR, strlen(CNGPLP_OPTION_CNGSADDITIONALOPTIONS_USECIECOLOR) + 1);
					}

					strncat(line_buf, DOUBLE_QUART, strlen(DOUBLE_QUART) + 1);
					strncat(line_buf, LINE_BREAK, strlen(LINE_BREAK));
					file_write(fd_new, line_buf, strlen(line_buf));

					memset(line_buf, 0x00, sizeof(line_buf));

					strncat(line_buf, CNGPLP_OPTION_KEY_CNCMCORRECTGRAY, strlen(CNGPLP_OPTION_KEY_CNCMCORRECTGRAY) + 1);
					strncat(line_buf, SPACE, strlen(SPACE) + 1);
					strncat(line_buf, DOUBLE_QUART, strlen(DOUBLE_QUART) + 1);

					if (options->correct_gray == kOPTION_CORRECT_GRAY_ON) {
						strncat(line_buf, CNGPLP_OPTION_CNCMCORRECTGRAY_TRUE, strlen(CNGPLP_OPTION_CNCMCORRECTGRAY_TRUE) + 1);
					}

					strncat(line_buf, DOUBLE_QUART, strlen(DOUBLE_QUART) + 1);
					strncat(line_buf, LINE_BREAK, strlen(LINE_BREAK));
					file_write(fd_new, line_buf, strlen(line_buf));

					close(fd_new);
				}

				unlink(file_org);
				rename(file_new, file_org);
				free(file_new);
			}
			free(file_org);
		}
	}

	return;
}

CNGPLP_OPTIONS* LoadCngplpOption()
{
	CNGPLP_OPTIONS *loadedoptions = NULL;

	loadedoptions = (CNGPLP_OPTIONS *)malloc(sizeof(CNGPLP_OPTIONS));
	if (loadedoptions != NULL)
	{
		CNGPLP_OPTIONS defaultoptioins = kCNGPLP_OPTIONS_INITIALIZER;
		char *optionvalue = NULL;

		*loadedoptions = defaultoptioins;
		optionvalue = get_option_value( CNGPLP_OPTION_KEY_CNGSADDITIONALOPTIONS );

		if (optionvalue != NULL) {
			if ( strcmp(optionvalue, CNGPLP_OPTION_CNGSADDITIONALOPTIONS_USECIECOLOR) == 0 ) {
				loadedoptions->use_cie_color = kOPTION_USE_CIE_COLOR_ON;
			}
			free(optionvalue);
			optionvalue = NULL;
		}

		optionvalue = get_option_value( CNGPLP_OPTION_KEY_CNCMCORRECTGRAY );

		if (optionvalue != NULL) {
			if ( strcmp(optionvalue, CNGPLP_OPTION_CNCMCORRECTGRAY_TRUE) == 0 ) {
				loadedoptions->correct_gray = kOPTION_CORRECT_GRAY_ON;
			}
			free(optionvalue);
			optionvalue = NULL;
		}
	}

	return loadedoptions;
}
