from uuid import uuid4

import boto3
import pytest
import requests
from botocore.exceptions import ClientError

from moto import mock_aws, settings

# See our Development Tips on writing tests for hints on how to write good tests:
# http://docs.getmoto.org/en/latest/docs/contributing/development_tips/tests.html

valid_resiliency_policy = {
    "Software": {"rpoInSecs": 1, "rtoInSecs": 1},
    "Hardware": {"rpoInSecs": 2, "rtoInSecs": 2},
    "AZ": {"rpoInSecs": 3, "rtoInSecs": 3},
}


@mock_aws
def test_create_app():
    client = boto3.client("resiliencehub", region_name="us-east-2")
    app = client.create_app(name="myapp")["app"]
    assert app["assessmentSchedule"] == "Disabled"
    assert app["complianceStatus"] == "NotAssessed"
    assert app["creationTime"]
    assert app["name"] == "myapp"
    assert app["status"] == "Active"
    assert app["tags"] == {}

    app2 = client.describe_app(appArn=app["appArn"])["app"]
    assert app == app2


@mock_aws
def test_create_app_advanced():
    event_subs = [
        {
            "eventType": "ScheduledAssessmentFailure",
            "name": "some event",
            "snsTopicArn": "some sns arn",
        }
    ]
    perm_model = {
        "crossAccountRoleArns": ["arn1", "arn2"],
        "invokerRoleName": "irn",
        "type": "Rolebased",
    }

    client = boto3.client("resiliencehub", region_name="us-east-2")
    app = client.create_app(
        name="myapp",
        assessmentSchedule="Daily",
        description="some desc",
        eventSubscriptions=event_subs,
        permissionModel=perm_model,
        policyArn="some policy arn",
    )["app"]
    assert app["assessmentSchedule"] == "Daily"
    assert app["description"] == "some desc"
    assert app["eventSubscriptions"] == event_subs
    assert app["permissionModel"] == perm_model
    assert app["policyArn"] == "some policy arn"


@mock_aws
def test_describe_unknown_app():
    client = boto3.client("resiliencehub", region_name="us-east-1")
    app_arn = f"arn:aws:resiliencehub:us-east-1:486285699788:app/{str(uuid4())}"
    with pytest.raises(ClientError) as exc:
        client.describe_app(appArn=app_arn)
    err = exc.value.response["Error"]
    assert err["Code"] == "ResourceNotFoundException"
    assert err["Message"] == f"App not found for appArn {app_arn}"


@mock_aws
def test_create_resilience_policy():
    client = boto3.client("resiliencehub", region_name="us-east-1")

    policy = client.create_resiliency_policy(
        policy=valid_resiliency_policy, policyName="polname", tier="NonCritical"
    )["policy"]
    assert policy["creationTime"]
    assert policy["policy"] == valid_resiliency_policy
    assert policy["policyName"] == "polname"
    assert policy["tags"] == {}
    assert policy["tier"] == "NonCritical"

    policy2 = client.describe_resiliency_policy(policyArn=policy["policyArn"])["policy"]
    assert policy == policy2


@mock_aws
def test_create_resilience_policy_advanced():
    client = boto3.client("resiliencehub", region_name="us-east-1")

    policy = client.create_resiliency_policy(
        dataLocationConstraint="AnyLocation",
        policy=valid_resiliency_policy,
        policyName="polname",
        policyDescription="test policy",
        tier="NonCritical",
    )["policy"]
    assert policy["dataLocationConstraint"] == "AnyLocation"
    assert policy["policyDescription"] == "test policy"


@mock_aws
def test_create_resilience_policy_missing_types():
    client = boto3.client("resiliencehub", region_name="us-east-1")

    with pytest.raises(ClientError) as exc:
        client.create_resiliency_policy(
            policy={"Software": {"rpoInSecs": 1, "rtoInSecs": 1}},
            policyName="polname",
            tier="NonCritical",
        )
    err = exc.value.response["Error"]
    assert err["Code"] == "ValidationException"
    assert err["Message"] == "FailureType HARDWARE does not exist"

    with pytest.raises(ClientError) as exc:
        client.create_resiliency_policy(
            policy={
                "Software": {"rpoInSecs": 1, "rtoInSecs": 1},
                "Hardware": {"rpoInSecs": 2, "rtoInSecs": 2},
            },
            policyName="polname",
            tier="NonCritical",
        )
    err = exc.value.response["Error"]
    assert err["Code"] == "ValidationException"
    assert err["Message"] == "FailureType AZ does not exist"

    with pytest.raises(ClientError) as exc:
        client.create_resiliency_policy(
            policy={"Hardware": {"rpoInSecs": 1, "rtoInSecs": 1}},
            policyName="polname",
            tier="NonCritical",
        )
    err = exc.value.response["Error"]
    assert err["Message"] == "FailureType SOFTWARE does not exist"


@mock_aws
def test_create_resilience_policy_with_unknown_policy_type():
    client = boto3.client("resiliencehub", region_name="us-east-1")
    with pytest.raises(ClientError) as exc:
        client.create_resiliency_policy(
            policy={
                "st": {"rpoInSecs": 1, "rtoInSecs": 1},
            },
            policyName="polname",
            tier="NonCritical",
        )
    err = exc.value.response["Error"]
    assert err["Code"] == "ValidationException"
    assert (
        "Member must satisfy enum value set: [Software, Hardware, Region, AZ]"
        in err["Message"]
    )


@mock_aws
def test_list_apps():
    client = boto3.client("resiliencehub", region_name="ap-southeast-1")
    assert client.list_apps()["appSummaries"] == []

    for i in range(5):
        arn = client.create_app(name=f"app_{i}")["app"]["appArn"]

    app_2 = client.list_apps(name="app_2")["appSummaries"][0]
    assert app_2["name"] == "app_2"

    app_4 = client.list_apps(appArn=arn)["appSummaries"][0]
    assert app_4["name"] == "app_4"

    all_apps = client.list_apps()["appSummaries"]
    assert len(all_apps) == 5
    assert [a["name"] for a in all_apps] == [
        "app_0",
        "app_1",
        "app_2",
        "app_3",
        "app_4",
    ]

    all_apps = client.list_apps(reverseOrder=True)["appSummaries"]
    assert len(all_apps) == 5
    assert [a["name"] for a in all_apps] == [
        "app_4",
        "app_3",
        "app_2",
        "app_1",
        "app_0",
    ]

    page1 = client.list_apps(maxResults=2)
    assert len(page1["appSummaries"]) == 2

    page2 = client.list_apps(maxResults=2, nextToken=page1["nextToken"])
    assert len(page2["appSummaries"]) == 2

    full_page = client.list_apps(nextToken=page1["nextToken"])
    assert len(full_page["appSummaries"]) == 3


@mock_aws
def test_list_app_assessments():
    client = boto3.client("resiliencehub", region_name="ap-southeast-1")
    assert client.list_app_assessments()["assessmentSummaries"] == []
    assert client.list_app_assessments(appArn="asdf")["assessmentSummaries"] == []

    # Setup
    base_url = (
        settings.test_server_mode_endpoint()
        if settings.TEST_SERVER_MODE
        else "http://motoapi.amazonaws.com"
    )
    # Configure results for the first two requests
    summary1 = {"appArn": "app_arn1", "appVersion": "some version"}
    summary2 = {"appArn": "app_arn2", "assessmentName": "assessment 2"}
    results = {
        "results": [[summary1, summary2], [summary2]],
        "region": "ap-southeast-1",
    }
    resp = requests.post(
        f"{base_url}/moto-api/static/resilience-hub-assessments/response",
        json=results,
    )
    assert resp.status_code == 201

    # First response
    assert client.list_app_assessments()["assessmentSummaries"] == [summary1, summary2]

    # Stays the same on subsequent invocations
    assert client.list_app_assessments()["assessmentSummaries"] == [summary1, summary2]

    # Second response with different args
    assert client.list_app_assessments(appArn="asdf")["assessmentSummaries"] == [
        summary2
    ]

    # Third response returns an empty response, because nothing else is configured
    assert client.list_app_assessments(appArn="diff")["assessmentSummaries"] == []

    # First response in different region also returns nothing, because we only configured this in ap-southeast
    us_client = boto3.client("resiliencehub", region_name="us-east-1")
    assert us_client.list_app_assessments()["assessmentSummaries"] == []


@mock_aws
def test_list_resiliency_policies():
    client = boto3.client("resiliencehub", region_name="ap-southeast-1")
    assert client.list_resiliency_policies()["resiliencyPolicies"] == []

    for i in range(5):
        client.create_resiliency_policy(
            policy=valid_resiliency_policy, policyName=f"policy_{i}", tier="NonCritical"
        )["policy"]

    assert len(client.list_resiliency_policies()["resiliencyPolicies"]) == 5

    policy2 = client.list_resiliency_policies(policyName="policy_2")[
        "resiliencyPolicies"
    ][0]
    assert policy2["policyName"] == "policy_2"

    page1 = client.list_resiliency_policies(maxResults=2)
    assert len(page1["resiliencyPolicies"]) == 2

    page2 = client.list_resiliency_policies(maxResults=2, nextToken=page1["nextToken"])
    assert len(page2["resiliencyPolicies"]) == 2
    assert page2["nextToken"]

    page_full = client.list_resiliency_policies(nextToken=page1["nextToken"])
    assert len(page_full["resiliencyPolicies"]) == 3
    assert "nextToken" not in page_full


@mock_aws
def test_describe_unknown_resiliency_policy():
    client = boto3.client("resiliencehub", region_name="eu-west-1")
    with pytest.raises(ClientError) as exc:
        client.describe_resiliency_policy(policyArn="unknownarn")
    err = exc.value.response["Error"]
    assert err["Code"] == "ResourceNotFoundException"
    assert err["Message"] == "ResiliencyPolicy unknownarn not found"
