//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file PhysicsList.cc
/// \brief Implementation of the PhysicsList class

#include "PhysicsList.hh"

#include "G4SystemOfUnits.hh"
#include "G4UnitsTable.hh"

#include "PhysicsListMessenger.hh"

// Particles
#include "G4Gamma.hh"
#include "G4Electron.hh"
#include "G4Positron.hh"
#include "G4Proton.hh"
#include "G4BaryonConstructor.hh"
#include "G4BosonConstructor.hh"
#include "G4DNAGenericIonsManager.hh"
#include "G4IonConstructor.hh"
#include "G4LeptonConstructor.hh"
#include "G4MesonConstructor.hh"
#include "G4ShortLivedConstructor.hh"

// Physics packages (builders contained in the Geant4 source code)
// Electromagnetic
#include "G4EmDNAPhysics.hh"
#include "G4EmDNAPhysics_option1.hh"
#include "G4EmDNAPhysics_option2.hh"
#include "G4EmDNAPhysics_option3.hh"
#include "G4EmDNAPhysics_option4.hh"
#include "G4EmDNAPhysics_option5.hh"
#include "G4EmDNAPhysics_option6.hh"
#include "G4EmDNAPhysics_option7.hh"
#include "G4EmDNAPhysics_option8.hh"
#include "G4EmLivermorePhysics.hh"
#include "G4EmPenelopePhysics.hh"
#include "G4EmStandardPhysics_option4.hh"



#include <memory>

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

PhysicsList::PhysicsList() : G4VModularPhysicsList()
{
  fMessenger = std::make_unique<PhysicsListMessenger>(this);
  SetVerboseLevel(1);
  fPhysicsList = std::make_unique<G4EmDNAPhysics_option2>();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

PhysicsList::~PhysicsList() = default;

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void PhysicsList::ConstructParticle()
{
  // This method is invoked when the Geant4 application starts
  // (do not mix with run initialization).

  // (Taken from G4DecayPhysics)
  G4BosonConstructor pBosonConstructor;
  pBosonConstructor.ConstructParticle();

  G4LeptonConstructor pLeptonConstructor;
  pLeptonConstructor.ConstructParticle();

  G4MesonConstructor pMesonConstructor;
  pMesonConstructor.ConstructParticle();

  G4BaryonConstructor pBaryonConstructor;
  pBaryonConstructor.ConstructParticle();

  G4IonConstructor pIonConstructor;
  pIonConstructor.ConstructParticle();

  G4ShortLivedConstructor pShortLivedConstructor;
  pShortLivedConstructor.ConstructParticle();

  G4DNAGenericIonsManager* genericIonsManager;
  genericIonsManager = G4DNAGenericIonsManager::Instance();
  genericIonsManager->GetIon("hydrogen");
  genericIonsManager->GetIon("deuteron");
  genericIonsManager->GetIon("triton");
  genericIonsManager->GetIon("helium");
  genericIonsManager->GetIon("alpha");
  genericIonsManager->GetIon("alpha+");
  genericIonsManager->GetIon("alpha++");
  genericIonsManager->GetIon("carbon");
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void PhysicsList::ConstructProcess()
{
  // Transportation
  AddTransportation();

  // Physics list
  fPhysicsList->ConstructProcess();

  // Set cuts for particles
  SetCuts();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void PhysicsList::AddPhysicsList(const G4String& name)
{
  if (verboseLevel > -1) {
    G4cout << "PhysicsList::AddPhysicsList: <" << name << ">" << G4endl;
  }

  if (name == fName) return;

  if (name == "dna") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmDNAPhysics>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }
  else if (name == "dna_opt1") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmDNAPhysics_option1>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else if (name == "dna_opt2") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmDNAPhysics_option2>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else if (name == "dna_opt3") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmDNAPhysics_option3>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else if (name == "dna_opt4") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmDNAPhysics_option4>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else if (name == "dna_opt5") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmDNAPhysics_option5>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else if (name == "dna_opt6") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmDNAPhysics_option6>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else if (name == "dna_opt7") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmDNAPhysics_option7>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else if (name == "dna_opt8") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmDNAPhysics_option8>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else if (name == "liv") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmLivermorePhysics>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else if (name == "penelope") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmPenelopePhysics>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else if (name == "em_standard_opt4") {
    fName = name;
    fPhysicsList = std::make_unique<G4EmStandardPhysics_option4>();
    G4cout << fPhysicsList->GetPhysicsName()
           << " physics package has been activated." << G4endl;
  }

  else {
    G4cout << "PhysicsList::AddPhysicsList: \"" << name << "\" is not defined!"
           << G4endl;
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......