/*
    razertool - Tool for controlling Razer Copperhead(TM) mice
    Copyright (C) 2006  Christopher Lais

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
*/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <usb.h>

#include "razertool.h"
#include "razerlib.h"

static razer_t *razer = NULL;
static int profile = 0;
static int switch_profile = 0;
static razer_profile_t profiles[RAZER_MAX_PROFILES];
static int profile_loaded[RAZER_MAX_PROFILES];
static int profile_changed[RAZER_MAX_PROFILES];

static void do_get_profile(void)
{
	int status;

	status = razer_read_profile_number(razer, &profile, RAZERLIB_DEFAULT_TIMEOUT);
	if (status < 0) {
		fprintf(stderr, "Unable to retreive profile number: %s\n", razer_strerror(-status));
		exit(-1);
	}

	fprintf(stderr, "Using profile %d\n", profile);
}

static void do_load_profile(int load_profile)
{
	int status;

	status = razer_set_profile(razer, load_profile, RAZERLIB_DEFAULT_TIMEOUT);
	if (status < 0) {
		fprintf(stderr, "Unable to set profile %d for retrieval: %s\n", load_profile, razer_strerror(-status));
		exit(-1);
	}

	status = razer_read_profile(razer, &profiles[load_profile-1], RAZERLIB_DEFAULT_TIMEOUT);
	if (status < 0) {
		fprintf(stderr, "Unable to retreive profile %d: %s\n", load_profile, razer_strerror(-status));
		exit(-1);
	}

	profile_loaded[load_profile-1] = 1;
	profile_changed[load_profile-1] = 0;
}

static void do_flush_profile(int flush_profile)
{
	int status;

	if (!profile_loaded[flush_profile-1])
		do_load_profile(flush_profile);

	status = razer_write_profile(razer, &profiles[flush_profile-1], RAZERLIB_DEFAULT_TIMEOUT);
	if (status < 0) {
		fprintf(stderr, "Unable to write profile %d: %s\n", flush_profile, razer_strerror(-status));
		exit(-1);
	}

	do_load_profile(flush_profile);
}

static void do_set_profile(const char *arg)
{
	profile = atoi(arg);
	if (profile < 1 || profile > razer->profiles) {
		fprintf(stderr, "Invalid profile: %s\n", arg);
		exit(-1);
	}
}

static void do_set_dpi(const char *arg)
{
	int dpi = atoi(arg);

	switch (dpi) {
	  case 400: dpi = RAZER_DPI_400; break;
	  case 800: dpi = RAZER_DPI_800; break;
	  case 1600: dpi = RAZER_DPI_1600; break;
	  case 2000: dpi = RAZER_DPI_2000; break;
	  default:
		fprintf(stderr, "Invalid dpi: %s\n", arg);
		exit(-1);
	}

	if (!profile)
		do_get_profile();

	if (!profile_loaded[profile-1])
		do_load_profile(profile);

	if (profiles[profile-1].dpi != dpi) {
		profiles[profile-1].dpi = dpi;
		profile_changed[profile-1] = 1;
	}
}

static void do_set_hz(const char *arg)
{
	int hz = atoi(arg);

	switch (hz) {
	  case 125: hz = RAZER_HZ_125; break;
	  case 500: hz = RAZER_HZ_500; break;
	  case 1000: hz = RAZER_HZ_1000; break;
	  default:
		fprintf(stderr, "Invalid report rate: %s\n", arg);
		exit(-1);
	}

	if (!profile)
		do_get_profile();

	if (!profile_loaded[profile-1])
		do_load_profile(profile);

	if (profiles[profile-1].hz != hz) {
		profiles[profile-1].hz = hz;
		profile_changed[profile-1] = 1;
	}
}

static void do_parse_key(const char *str, int *macro_len, razer_macro_char_t *macro)
{
	int key;
	key = razer_translate_key(str);
	if (key <= 0) {
		fprintf(stderr, "Invalid key: %s\n", str);
		exit(-1);
	}
	macro[0].key = key;
	*macro_len = 1;
}

static void do_parse_macro(const char *str, int *macro_len, razer_macro_char_t *macro, int macro_max_len)
{
	char tmp[64];

	*macro_len = 0;

	while (*str) {
		int key_len = strcspn(str, ", ;-");
		int key;
		char *flag;
		int how;

		if (key_len >= sizeof(tmp)) {
			fprintf(stderr, "Macro key specifier too big at: %s\n", str);
			exit(-1);
		}

		strncpy(tmp, str, key_len);
		tmp[key_len] = '\0';
		str += key_len;
		if (*str) ++str;

		key_len = strcspn(tmp, ":");
		flag = tmp + key_len;
		if (*flag) { *flag = 0; ++flag; }

		if (!*flag && !strcasecmp(tmp, "DELAY")) {
			if (!*macro_len) {
				fprintf(stderr, "Can't delay before first macro key\n");
				exit(-1);
			}

			macro[*macro_len - 1].how |= 0x80;
			continue;
		}

		if (*macro_len >= macro_max_len) {
			fprintf(stderr, "Macro too long\n");
			exit(-1);
		}

		key = razer_translate_key(tmp);
		if (key <= 0) {
			fprintf(stderr, "Invalid macro key: %s\n", tmp);
			exit(-1);
		}

		if (!*flag || !strcasecmp(flag, "down")) {
			how = RAZER_MACRO_KEY_DOWN;
		} else if (!strcasecmp(flag, "up")) {
			how = RAZER_MACRO_KEY_UP;
		} else {
			fprintf(stderr, "Expected up/down: %s\n", flag);
			exit(-1);
		}

		macro[*macro_len].key = key;
		macro[*macro_len].how = how;

		(*macro_len)++;

		if (!*flag) {
			if (*macro_len >= macro_max_len) {
				fprintf(stderr, "Macro too long\n");
				exit(-1);
			}

			macro[*macro_len].key = key;
			macro[*macro_len].how = RAZER_MACRO_KEY_UP;

			(*macro_len)++;
		}
	}
}

static void do_set_button(int button, const char *arg)
{
	int action = razer_action_int(arg);
	int i;
	int macro_len;
	razer_macro_char_t macro[RAZER_MACRO_MAX_LEN];

	macro_len = 0;
	memset(macro, 0, sizeof(macro));

	if (action < 0) {
		if (!strncasecmp(arg, "KEY:", 4)) {
			do_parse_key(arg+4, &macro_len, macro);
			action = RAZER_ACTION_KEY;
		} else if (!strncasecmp(arg, "MACRO:", 6)) {
			do_parse_macro(arg+6, &macro_len, macro, RAZER_MACRO_MAX_LEN);
			action = RAZER_ACTION_MACRO;
		} else {
			fprintf(stderr, "Invalid action: %s\n", arg);
			exit(-1);
		}
	} else if (action == RAZER_ACTION_KEY) {
		fprintf(stderr, "KEY action requires a key\n");
		exit(-1);
	} else if (action == RAZER_ACTION_MACRO) {
		fprintf(stderr, "MACRO action requires a macro\n");
		exit(-1);
	}

	if (!profile)
		do_get_profile();

	if (!profile_loaded[profile-1])
		do_load_profile(profile);

	for (i = 0; i < razer->buttons; i++) {
		razer_button_t *b = &profiles[profile-1].button[i];

		if (b->number != button)
			continue;

		if (b->macro_max_len < macro_len) {
			fprintf(stderr, "Macro too long\n");
			exit(-1);
		}

		if (b->action != action || b->macro_len != macro_len || memcmp(b->macro, macro, sizeof(*b->macro)*macro_len)) {
			profiles[profile-1].button[i].action = action;
			profiles[profile-1].button[i].macro_len = macro_len;
			memcpy(b->macro, macro, sizeof(*b->macro)*RAZER_MACRO_MAX_LEN);
			profile_changed[profile-1] = 1;
		}

		return;
	}

	fprintf(stderr, "Button %d not found\n", button);
	exit(-1);
}

static void do_switch_profile(void)
{
	if (!profile)
		do_get_profile();

	switch_profile = profile;
}

static void do_help(const char *path)
{
	printf(
		"razertool " VERSION ", Copyright (C) 2006  Christopher Lais\n"
		"razertool comes with ABSOLUTELY NO WARRANTY; refer to the\n"
		"GNU GPL version 2 or later for details.\n"
		"\n"
		"%s [options]\n"
		"        -p PROFILE      select profile to modify/display\n"
		"\n"
		"        -1...-7 ACTION  select button action\n"
		"        -d DPI          select dpi (400/800/1600/2000)\n"
		"        -r HZ           select report rate (125/500/1000)\n"
		"\n"
		"        -s              switch to profile\n"
		"\n"
		"        -h              display this help\n"
		"        -i              display information about device\n"
		"        -P              display profile\n",
		path
	);
}

static void do_info(void)
{
	int status;
	int version;

	status = razer_read_firmware_version(razer, &version, RAZERLIB_DEFAULT_TIMEOUT);
	if (status < 0) {
		fprintf(stderr, "Unable to read firmware version: %s\n", razer_strerror(-status));
		exit(-1);
	}

	printf("Firmware version: %d.%02d\n", version / 100, version % 100);
}

static void do_show_profile(void)
{
	int i;
	razer_profile_t *prof;

	if (!profile)
		do_get_profile();

	if (profile_changed[profile-1])
		do_flush_profile(profile);

	if (!profile_loaded[profile-1])
		do_load_profile(profile);

	prof = &profiles[profile-1];

	printf("Profile %d:\n", prof->number);
	printf("\tDPI: %ddpi\n", razer_dpi_int(prof->dpi));
	printf("\tRate: %dHz\n", razer_hz_int(prof->hz));
	for (i = 0; i < razer->buttons; i++) {
		printf("\tButton %d: %s [0x%02x]",
			prof->button[i].number,
			razer_action_string(prof->button[i].action),
			prof->button[i].action
		);
		if (prof->button[i].action == RAZER_ACTION_MACRO || prof->button[i].action == RAZER_ACTION_KEY) {
			int j;
			char key[64], *slash;

			for (j = 0; j < prof->button[i].macro_len; j++) {
				int k = prof->button[i].macro[j].key, f = prof->button[i].macro[j].how;

				razer_untranslate_key(k, key, sizeof(key));

				/* only display the first */
				slash = strchr(key, '/');
				if (slash) { *slash = '\0'; }

				printf(" %s", key);

				if ((f & ~RAZER_MACRO_KEY_DELAY_AFTER) == RAZER_MACRO_KEY_DOWN)
					printf(":down");
				else if ((f & ~RAZER_MACRO_KEY_DELAY_AFTER) == RAZER_MACRO_KEY_UP)
					printf(":up");
				else
					printf(":%02x", f & ~RAZER_MACRO_KEY_DELAY_AFTER);

				if (f & RAZER_MACRO_KEY_DELAY_AFTER)
					printf(" DELAY");
			}
		}
		printf("\n");
	}
}

int main(int argc, char *argv[])
{
	struct usb_device *dev;
	int i, ch;

	usb_init();

	memset(profile_loaded, 0, sizeof(profile_loaded));
	memset(profile_changed, 0, sizeof(profile_changed));

	/* TODO: Open by device, etc */
	dev = razer_find(NULL);
	if (!dev) {
		fprintf(stderr, "Unable to find Razer Copperhead Laser Mouse\n");
		return -1;
	}

	razer = razer_open(dev);
	if (!razer) {
		perror("razer_open()");
		return -1;
	}

	if (argc == 1) {
		do_help(argv[0]);
		exit(-1);
	}

	while ((ch = getopt(argc, argv, "1:2:3:4:5:6:7:d:hip:Pr:s")) != -1) {
		switch (ch) {
		  case '1': case '2': case '3': case '4': case '5': case '6':
		  case '7': do_set_button(ch - '0', optarg); break;

		  case 'd': do_set_dpi(optarg); break;
		  case 'h': do_help(argv[0]); break;
		  case 'i': do_info(); break;
		  case 'p': do_set_profile(optarg); break;
		  case 'P': do_show_profile(); break;
		  case 'r': do_set_hz(optarg); break;
		  case 's': do_switch_profile(); break;
		  default: do_help(argv[0]); return -1; break;
		}
	}

	if (optind < argc) {
		fprintf(stderr, "%s: invalid argument(s) --", argv[0]);
		for (i = optind; i < argc; i++)
			fprintf(stderr, " %s", argv[i]);
		fprintf(stderr, "\n");
		do_help(argv[0]);
		return -1;
	}

	for (i = 1; i <= razer->profiles; i++) {
		if (profile_changed[i-1])
			do_flush_profile(i);
	}

	if (switch_profile) {
		int status = razer_select_profile(razer, switch_profile, RAZERLIB_DEFAULT_TIMEOUT);
		if (status < 0) {
			fprintf(stderr, "Unable to switch profiles: %s\n", razer_strerror(-status));
		}
	}

	razer_close(razer);

	return 0;
}
