# (c) Copyright 2009-2015. CodeWeavers, Inc.

import os
import cxproduct
import distversion

import cxutils
import cxobjc
import cxlog
import bottlequery

# for localization
from cxutils import cxgettext as _


class BottleManagement(cxobjc.Proxy):
    pass


def _bottle_tool():
    return os.path.join(cxutils.CX_ROOT, "bin", "cxbottle")


@cxobjc.method(BottleManagement, 'createBottle_fromTemplate_')
def create_bottle(bottlename, template="win10_64", env=None, appid=None):
    """Creates the specified bottle and returns a (success, err_msg) tuple.
    """
    args = [_bottle_tool(), "--bottle", bottlename,
            "--create", "--template", template, "--install"]
    if appid is not None:
        s = "EnvironmentVariables:CX_BOTTLE_CREATOR_APPID=%s" % appid
        args.append("--param")
        args.append(s)
    retcode, _out, err = cxutils.run(args, env=env, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


@cxobjc.method(BottleManagement, 'duplicateBottle_fromBottle_')
def copy_bottle(bottlename, bottleToCopy):
    """Copies the specified bottle and returns a (success, err_msg) tuple.
    """
    args = [_bottle_tool(), "--bottle", bottlename,
            "--copy", bottleToCopy, "--install"]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


def publish_bottle(bottlename, bottleToCopy):
    """Publishes the specified bottle and returns a (success, err_msg) tuple.
    """
    if cxproduct.is_root_install():
        cxsu = os.path.join(cxutils.CX_ROOT, "bin", "cxsu")
        cxsu_args = [cxsu, '--ignore-home']
    else:
        cxsu_args = []

    prefix = bottlequery.get_prefix_for_bottle(bottleToCopy)
    args = cxsu_args + [_bottle_tool(), "--bottle", bottlename,
                        "--copy", prefix, "--install", "--scope", "managed"]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


@cxobjc.method(BottleManagement, 'restoreBottle_fromArchive_')
def restore_bottle(bottlename, archiveFile):
    """Restores the specified archive to the specified bottle name and
    returns a (success, err_msg) tuple.
    """
    args = [_bottle_tool(), "--bottle", bottlename,
            "--restore", archiveFile, "--install"]
    env = None
    if distversion.IS_MACOSX and 'LANG' not in os.environ and \
            'LC_CTYPE' not in os.environ and 'LC_ALL' not in os.environ:
        env = os.environ.copy()
        env['LC_CTYPE'] = 'UTF-8'
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, env=env, logprefix=bottlename)
    return (retcode == 0), err


def rename_bottle(bottlename, newBottleName):
    """Renames the specified bottle and returns a (success, err_msg) tuple."""
    oldPrefix = bottlequery.get_prefix_for_bottle(bottlename)
    newPrefix = os.path.join(os.path.dirname(oldPrefix), newBottleName)
    wasDefault = bottlequery.get_default_bottle() == bottlename

    if os.path.exists(newPrefix):
        err = "Cannot rename; that bottle already exists."
        return (False, err)

    args = [_bottle_tool(), "--bottle", bottlename,
            "--removeall"]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    if retcode != 0:
        errstr = "Failed to uninstall the old bottle name: %s" % err
        return (False, errstr)

    os.rename(oldPrefix, newPrefix)

    args = [_bottle_tool(), "--bottle", newBottleName,
            "--new-uuid", "--install"]
    if wasDefault:
        args.append("--default")
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    if retcode != 0:
        errstr = "Failed to install the new bottle name: %s" % err
        return (False, errstr)

    return (True, "")


def delete_bottle(bottlename, isManaged=False):
    """Deletes the specified bottle and returns a (success, err_msg) tuple.
    """

    if isManaged:
        if cxproduct.is_root_install():
            cxsu = os.path.join(cxutils.CX_ROOT, "bin", "cxsu")
            cxsu_args = [cxsu, '--ignore-home']
        else:
            cxsu_args = []
        args = cxsu_args + [_bottle_tool(), "--bottle", bottlename,
                            "--removeall", "--delete", "--force"]
        retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)

        if retcode != 0:
            cxlog.log("Failed to delete %s." % cxlog.to_str(bottlename))

        # Delete the stub, if it's there.
        args = [_bottle_tool(), "--bottle", bottlename,
                "--removeall", "--delete", "--force", "--scope", "private"]
        retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)

        if retcode != 0:
            cxlog.log("Failed to delete the stub for %s." % cxlog.to_str(bottlename))

        return (retcode == 0), err

    args = [_bottle_tool(), "--bottle", bottlename,
            "--removeall", "--delete", "--force", "--scope", "private"]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)

    if retcode != 0:
        cxlog.log("Failed to delete %s." % cxlog.to_str(bottlename))

    return (retcode == 0), err


def set_default_bottle(bottlename, inState):
    """Makes or unmakes the bottle the default bottle and returns a
    (success, err_msg) tuple.
    """
    if inState:
        cmd = "--default"
    else:
        cmd = "--undefault"
    args = [_bottle_tool(), "--bottle", bottlename, cmd]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


def package_bottle(bottlename, packagetype, release, productid, packager, outdir):
    args = [_bottle_tool(), "--%s" % packagetype, "--release", release, "--bottle", bottlename,
            "--productid", productid, "--packager", packager]
    retcode, _out, err = cxutils.run(args, cwd=outdir, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


@cxobjc.method(BottleManagement, 'archiveBottle_toPath_')
def archive_bottle(bottlename, inArchivePath):
    """Archives the bottle to the specified file and returns a
    (success, err_msg) tuple.
    """
    args = [_bottle_tool(), "--bottle", bottlename,
            "--tar", inArchivePath]
    env = None
    if distversion.IS_MACOSX and 'LANG' not in os.environ and \
            'LC_CTYPE' not in os.environ and 'LC_ALL' not in os.environ:
        env = os.environ.copy()
        env['LC_CTYPE'] = 'UTF-8'
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, env=env, logprefix=bottlename)
    return (retcode == 0), err


_TEMPLATE_PROPERTIES = {
    'win98': (0, _('Windows 98 (deprecated)')),
    'win2000': (1, _('Windows 2000 (deprecated)')),
    'winxp': (2, _('Windows XP 32-bit (deprecated)')),
    'winxp_64': (3, _('Windows XP')),
    'winvista': (4, _('Windows Vista 32-bit (deprecated)')),
    'winvista_64': (5, _('Windows Vista')),
    'win7': (6, _('Windows 7 32-bit (deprecated)')),
    'win7_64': (7, _('Windows 7')),
    'win8': (8, _('Windows 8 32-bit (deprecated)')),
    'win8_64': (9, _('Windows 8')),
    'win10': (10, _('Windows 10 32-bit (deprecated)')),
    'win10_64': (11, _('Windows 10')),
    'win11_64': (12, _('Windows 11')),
}


def _get_template_key(template):
    prop = _TEMPLATE_PROPERTIES.get(template)
    if prop is None:
        return (99, 'Unknown', template)
    return (prop[0], prop[1], template)


@cxobjc.method(BottleManagement, 'displayNameForTemplate_')
def get_template_name(template):
    if template not in _TEMPLATE_PROPERTIES:
        return 'Unknown'
    prop = _TEMPLATE_PROPERTIES[template]
    return prop[1]


@cxobjc.method(BottleManagement, 'templates')
def template_list():
    template_names = []
    template_dir = os.path.join(cxutils.CX_ROOT, "share", "crossover", "bottle_templates")
    for dentry in os.listdir(template_dir):
        if not dentry.endswith("_64") and \
           not cxproduct.get_config_boolean("OfficeSetup", "Enable32BitBottles", False):
            continue
        if os.path.exists(os.path.join(template_dir, dentry, "setup")):
            template_names.append(dentry)
    return sorted(template_names, key=_get_template_key)


def is_running(bottlename):

    args = [os.path.join(cxutils.CX_ROOT, "bin", "wine"),
            "--bottle", bottlename, "--no-update",
            "--ux-app", "wineserver", "-k0"]
    retcode = cxutils.system(args)
    return retcode == 0


def quit_bottle(bottlename):
    """Return True if shutdown succeeded, False if failed or canceled."""

    if not is_running(bottlename):
        return True

    args = [os.path.join(cxutils.CX_ROOT, "bin", "wine"),
            "--bottle", bottlename, "--no-update",
            "--wl-app", "wineboot.exe", "--", "--end-session",
            "--shutdown", "--force", "--kill"]
    retcode, _out, _err = cxutils.run(args, stdout=cxutils.GRAB, logprefix=bottlename)
    return retcode == 0


def get_up_to_date(bottlename, scope="private"):
    args = [_bottle_tool(), "--bottle", bottlename, "--status", "--scope", scope]
    retcode, out, _err = cxutils.run(args, stdout=cxutils.GRAB, logprefix=bottlename)
    if retcode == 0 and "Status=uptodate" in out:
        return True
    return False


def kill_bottle(bottlename):
    if not is_running(bottlename):
        return True

    args = [os.path.join(cxutils.CX_ROOT, "bin", "wine"),
            "--bottle", bottlename, "--no-update",
            "--ux-app", "wineserver", "-k"]
    retcode = cxutils.system(args)
    return retcode == 0
