/**************************************************************************
 *
 * Copyright 2007 VMware, Inc.
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL VMWARE AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 **************************************************************************/


#include "main/accum.h"
#include "main/context.h"
#include "main/debug_output.h"
#include "main/framebuffer.h"
#include "main/glthread.h"
#include "main/shaderobj.h"
#include "main/state.h"
#include "main/version.h"
#include "main/hash.h"
#include "program/prog_cache.h"
#include "vbo/vbo.h"
#include "glapi/glapi/glapi.h"
#include "st_manager.h"
#include "st_context.h"
#include "st_debug.h"
#include "st_cb_bitmap.h"
#include "st_cb_clear.h"
#include "st_cb_drawpixels.h"
#include "st_cb_drawtex.h"
#include "st_cb_eglimage.h"
#include "st_cb_feedback.h"
#include "st_cb_flush.h"
#include "st_atom.h"
#include "st_draw.h"
#include "st_extensions.h"
#include "st_gen_mipmap.h"
#include "st_pbo.h"
#include "st_program.h"
#include "st_sampler_view.h"
#include "st_shader_cache.h"
#include "st_texcompress_compute.h"
#include "st_texture.h"
#include "st_util.h"
#include "pipe/p_context.h"
#include "util/u_cpu_detect.h"
#include "util/u_inlines.h"
#include "util/u_upload_mgr.h"
#include "util/u_vbuf.h"
#include "util/u_memory.h"
#include "util/hash_table.h"
#include "util/thread_sched.h"
#include "util/u_threaded_context.h"
#include "cso_cache/cso_context.h"
#include "compiler/glsl/glsl_parser_extras.h"
#include "nir.h"

void
st_invalidate_buffers(struct st_context *st)
{
   ST_SET_STATE4(st->ctx->NewDriverState, ST_NEW_BLEND, ST_NEW_DSA,
                 ST_NEW_FB_STATE, ST_NEW_SAMPLE_STATE);
   ST_SET_STATE4(st->ctx->NewDriverState, ST_NEW_SAMPLE_SHADING,
                 ST_NEW_FS_STATE, ST_NEW_POLY_STIPPLE, ST_NEW_VIEWPORT);
   ST_SET_STATE3(st->ctx->NewDriverState, ST_NEW_RASTERIZER,
                 ST_NEW_SCISSOR, ST_NEW_WINDOW_RECTANGLES);
}


static inline bool
st_vp_uses_current_values(const struct gl_context *ctx)
{
   /* mesh shader pipeline */
   if (!ctx->VertexProgram._Current)
      return false;

   const uint64_t inputs = ctx->VertexProgram._Current->info.inputs_read;

   return ~_mesa_get_enabled_vertex_arrays(ctx) & inputs;
}


void
st_invalidate_state(struct gl_context *ctx)
{
   GLbitfield new_state = ctx->NewState;
   struct st_context *st = st_context(ctx);

   if (new_state & _NEW_BUFFERS) {
      st_invalidate_buffers(st);
   } else {
      /* These set a subset of flags set by _NEW_BUFFERS, so we only have to
       * check them when _NEW_BUFFERS isn't set.
       */
      if (new_state & _NEW_FOG)
         ST_SET_STATE(ctx->NewDriverState, ST_NEW_FS_STATE);
   }

   if (new_state & (_NEW_LIGHT_STATE |
                    _NEW_POINT))
      ST_SET_STATE(ctx->NewDriverState, ST_NEW_RASTERIZER);

   if ((new_state & _NEW_LIGHT_STATE) &&
       (st->lower_flatshade || st->lower_two_sided_color))
      ST_SET_STATE(ctx->NewDriverState, ST_NEW_FS_STATE);

   if (new_state & _NEW_PROJECTION &&
       st_user_clip_planes_enabled(ctx))
      ST_SET_STATE(ctx->NewDriverState, ST_NEW_CLIP_STATE);

   if (new_state & _NEW_PIXEL)
      ST_SET_STATE(ctx->NewDriverState, ST_NEW_PIXEL_TRANSFER);

   if (new_state & _NEW_CURRENT_ATTRIB && st_vp_uses_current_values(ctx)) {
      ST_SET_STATE(ctx->NewDriverState, ST_NEW_VERTEX_ARRAYS);
      /* glColor3f -> glColor4f changes the vertex format. */
      ctx->Array.NewVertexElements = true;
   }

   /* Update the vertex shader if ctx->Light._ClampVertexColor was changed. */
   if (st->clamp_vert_color_in_shader && (new_state & _NEW_LIGHT_STATE)) {
      ST_SET_STATE(ctx->NewDriverState, ST_NEW_VS_STATE);
      if (_mesa_is_desktop_gl_compat(st->ctx) && ctx->Version >= 32) {
         ST_SET_STATE2(ctx->NewDriverState, ST_NEW_GS_STATE, ST_NEW_TES_STATE);
      }
   }

   /* Update the vertex shader if ctx->Point was changed. */
   if (st->lower_point_size && new_state & _NEW_POINT) {
      if (ctx->GeometryProgram._Current)
         ST_SET_STATE2(ctx->NewDriverState, ST_NEW_GS_STATE, ST_NEW_GS_CONSTANTS);
      else if (ctx->TessEvalProgram._Current)
         ST_SET_STATE2(ctx->NewDriverState, ST_NEW_TES_STATE, ST_NEW_TES_CONSTANTS);
      else
         ST_SET_STATE2(ctx->NewDriverState, ST_NEW_VS_STATE, ST_NEW_VS_CONSTANTS);
   }

   if (new_state & _NEW_TEXTURE_OBJECT) {
      st_state_bitset states = {0};
      ST_SET_SHADER_STATES(ctx->NewDriverState, SAMPLER_VIEWS);
      ST_SET_SHADER_STATES(ctx->NewDriverState, SAMPLERS);
      ST_SET_SHADER_STATES(ctx->NewDriverState, IMAGES);
      BITSET_AND(states, states, st->active_states);
      ST_SET_STATES(ctx->NewDriverState, states);

      if (ctx->FragmentProgram._Current) {
         struct gl_program *fp = ctx->FragmentProgram._Current;

         if (fp->ExternalSamplersUsed || fp->ati_fs ||
            (!fp->shader_program && fp->ShadowSamplers))
            ST_SET_STATE(ctx->NewDriverState, ST_NEW_FS_STATE);
      }
   }
}


/*
 * In some circumstances (such as running google-chrome) the state
 * tracker may try to delete a resource view from a context different
 * than when it was created.  We don't want to do that.
 *
 * In that situation, st_texture_release_all_sampler_views() calls this
 * function to transfer the sampler view reference to this context (expected
 * to be the context which created the view.)
 */
void
st_save_zombie_sampler_view(struct st_context *st,
                            struct pipe_sampler_view *view)
{
   struct st_zombie_sampler_view_node *entry;

   assert(view->context == st->pipe);

   entry = MALLOC_STRUCT(st_zombie_sampler_view_node);
   if (!entry)
      return;

   entry->view = view;

   /* We need a mutex since this function may be called from one thread
    * while free_zombie_resource_views() is called from another.
    */
   simple_mtx_lock(&st->zombie_sampler_views.mutex);
   list_addtail(&entry->node, &st->zombie_sampler_views.list.node);
   simple_mtx_unlock(&st->zombie_sampler_views.mutex);
}


/*
 * Since OpenGL shaders may be shared among contexts, we can wind up
 * with variants of a shader created with different contexts.
 * When we go to destroy a gallium shader, we want to free it with the
 * same context that it was created with, unless the driver reports
 * pipe_caps.shareable_shaders = TRUE.
 */
void
st_save_zombie_shader(struct st_context *st,
                      mesa_shader_stage type,
                      struct pipe_shader_state *shader)
{
   struct st_zombie_shader_node *entry;

   /* we shouldn't be here if the driver supports shareable shaders */
   assert(!st->has_shareable_shaders);

   entry = MALLOC_STRUCT(st_zombie_shader_node);
   if (!entry)
      return;

   entry->shader = shader;
   entry->type = type;

   /* We need a mutex since this function may be called from one thread
    * while free_zombie_shaders() is called from another.
    */
   simple_mtx_lock(&st->zombie_shaders.mutex);
   list_addtail(&entry->node, &st->zombie_shaders.list.node);
   simple_mtx_unlock(&st->zombie_shaders.mutex);
}


/*
 * Free any zombie sampler views that may be attached to this context.
 */
static void
free_zombie_sampler_views(struct st_context *st)
{
   struct st_zombie_sampler_view_node *entry, *next;

   if (list_is_empty(&st->zombie_sampler_views.list.node)) {
      return;
   }

   simple_mtx_lock(&st->zombie_sampler_views.mutex);

   LIST_FOR_EACH_ENTRY_SAFE(entry, next,
                            &st->zombie_sampler_views.list.node, node) {
      list_del(&entry->node);  // remove this entry from the list

      assert(entry->view->context == st->pipe);
      st->pipe->sampler_view_release(st->pipe, entry->view);

      free(entry);
   }

   assert(list_is_empty(&st->zombie_sampler_views.list.node));

   simple_mtx_unlock(&st->zombie_sampler_views.mutex);
}


/*
 * Free any zombie shaders that may be attached to this context.
 */
static void
free_zombie_shaders(struct st_context *st)
{
   struct st_zombie_shader_node *entry, *next;

   if (list_is_empty(&st->zombie_shaders.list.node)) {
      return;
   }

   simple_mtx_lock(&st->zombie_shaders.mutex);

   LIST_FOR_EACH_ENTRY_SAFE(entry, next,
                            &st->zombie_shaders.list.node, node) {
      list_del(&entry->node);  // remove this entry from the list

      switch (entry->type) {
      case MESA_SHADER_VERTEX:
         ST_SET_STATE(st->ctx->NewDriverState, ST_NEW_VS_STATE);
         st->pipe->delete_vs_state(st->pipe, entry->shader);
         break;
      case MESA_SHADER_FRAGMENT:
         ST_SET_STATE(st->ctx->NewDriverState, ST_NEW_FS_STATE);
         st->pipe->delete_fs_state(st->pipe, entry->shader);
         break;
      case MESA_SHADER_GEOMETRY:
         ST_SET_STATE(st->ctx->NewDriverState, ST_NEW_GS_STATE);
         st->pipe->delete_gs_state(st->pipe, entry->shader);
         break;
      case MESA_SHADER_TESS_CTRL:
         ST_SET_STATE(st->ctx->NewDriverState, ST_NEW_TCS_STATE);
         st->pipe->delete_tcs_state(st->pipe, entry->shader);
         break;
      case MESA_SHADER_TESS_EVAL:
         ST_SET_STATE(st->ctx->NewDriverState, ST_NEW_TES_STATE);
         st->pipe->delete_tes_state(st->pipe, entry->shader);
         break;
      case MESA_SHADER_COMPUTE:
         ST_SET_STATE(st->ctx->NewDriverState, ST_NEW_CS_STATE);
         st->pipe->delete_compute_state(st->pipe, entry->shader);
         break;
      case MESA_SHADER_TASK:
         ST_SET_STATE(st->ctx->NewDriverState, ST_NEW_TS_STATE);
         st->pipe->delete_ts_state(st->pipe, entry->shader);
         break;
      case MESA_SHADER_MESH:
         ST_SET_STATE(st->ctx->NewDriverState, ST_NEW_MS_STATE);
         st->pipe->delete_ms_state(st->pipe, entry->shader);
         break;
      default:
         UNREACHABLE("invalid shader type in free_zombie_shaders()");
      }
      free(entry);
   }

   assert(list_is_empty(&st->zombie_shaders.list.node));

   simple_mtx_unlock(&st->zombie_shaders.mutex);
}


/*
 * This function is called periodically to free any zombie objects
 * which are attached to this context.
 */
void
st_context_free_zombie_objects(struct st_context *st)
{
   free_zombie_sampler_views(st);
   free_zombie_shaders(st);
}


static void
st_destroy_context_priv(struct st_context *st, bool destroy_pipe)
{
   st_destroy_draw(st);
   st_destroy_clear(st);
   st_destroy_bitmap(st);
   st_destroy_drawpix(st);
   st_destroy_drawtex(st);
   st_destroy_pbo_helpers(st);

   if (_mesa_has_compute_shaders(st->ctx) && st->transcode_astc)
      st_destroy_texcompress_compute(st);

   st_destroy_bound_texture_handles(st);
   st_destroy_bound_image_handles(st);

   /* free glReadPixels cache data */
   st_invalidate_readpix_cache(st);
   util_throttle_deinit(st->screen, &st->throttle);

   cso_destroy_context(st->cso_context);

   if (st->pipe && destroy_pipe)
      st->pipe->destroy(st->pipe);

   st->ctx->st = NULL;
   FREE(st);
}


static void
st_init_driver_flags(struct st_context *st)
{
   struct gl_driver_flags *f = &st->ctx->DriverFlags;

   /* Shader resources */
   if (st->has_hw_atomics)
      ST_SET_STATE2(f->NewAtomicBuffer, ST_NEW_HW_ATOMICS, ST_NEW_CS_ATOMICS);
   else
      ST_SET_SHADER_STATES(f->NewAtomicBuffer, ATOMICS);

   ST_SET_STATE(f->NewShaderConstants[MESA_SHADER_VERTEX], ST_NEW_VS_CONSTANTS);
   ST_SET_STATE(f->NewShaderConstants[MESA_SHADER_TESS_CTRL], ST_NEW_TCS_CONSTANTS);
   ST_SET_STATE(f->NewShaderConstants[MESA_SHADER_TESS_EVAL], ST_NEW_TES_CONSTANTS);
   ST_SET_STATE(f->NewShaderConstants[MESA_SHADER_GEOMETRY], ST_NEW_GS_CONSTANTS);
   ST_SET_STATE(f->NewShaderConstants[MESA_SHADER_FRAGMENT], ST_NEW_FS_CONSTANTS);
   ST_SET_STATE(f->NewShaderConstants[MESA_SHADER_COMPUTE], ST_NEW_CS_CONSTANTS);

   if (st->lower_alpha_test)
      ST_SET_STATE2(f->NewAlphaTest, ST_NEW_FS_STATE, ST_NEW_FS_CONSTANTS);
   else
      ST_SET_STATE(f->NewAlphaTest, ST_NEW_DSA);

   ST_SET_STATE4(f->NewMultisampleEnable, ST_NEW_BLEND, ST_NEW_RASTERIZER,
                 ST_NEW_SAMPLE_STATE, ST_NEW_SAMPLE_SHADING);
   ST_SET_STATE(f->NewSampleShading, ST_NEW_SAMPLE_SHADING);

   /* This depends on what the gallium driver wants. */
   if (st->force_persample_in_shader) {
      ST_SET_STATE(f->NewMultisampleEnable, ST_NEW_FS_STATE);
      ST_SET_STATE(f->NewSampleShading, ST_NEW_FS_STATE);
   } else {
      ST_SET_STATE(f->NewSampleShading, ST_NEW_RASTERIZER);
   }

   if (st->clamp_frag_color_in_shader) {
      ST_SET_STATE(f->NewFragClamp, ST_NEW_FS_STATE);
   } else {
      ST_SET_STATE(f->NewFragClamp, ST_NEW_RASTERIZER);
   }

   ST_SET_STATE(f->NewClipPlaneEnable, ST_NEW_RASTERIZER);
   if (st->lower_ucp) {
      ST_SET_STATE3(f->NewClipPlaneEnable, ST_NEW_VS_STATE, ST_NEW_GS_STATE,
                    ST_NEW_TES_STATE);
   }

   if (st->emulate_gl_clamp) {
      ST_SET_SHADER_STATES(f->NewSamplersWithClamp, SAMPLERS);
      ST_SET_SHADER_STATES(f->NewSamplersWithClamp, STATE);
   }

   if (!st->has_hw_atomics && st->ctx->Const.ShaderStorageBufferOffsetAlignment > 4)
      ST_SET_SHADER_STATES(f->NewAtomicBuffer, CONSTANTS);
}

static bool
st_have_perfquery(struct st_context *ctx)
{
   struct pipe_context *pipe = ctx->pipe;

   return pipe->init_intel_perf_query_info && pipe->get_intel_perf_query_info &&
          pipe->get_intel_perf_query_counter_info &&
          pipe->new_intel_perf_query_obj && pipe->begin_intel_perf_query &&
          pipe->end_intel_perf_query && pipe->delete_intel_perf_query &&
          pipe->wait_intel_perf_query && pipe->is_intel_perf_query_ready &&
          pipe->get_intel_perf_query_data;
}

static struct st_context *
st_create_context_priv(struct gl_context *ctx, struct pipe_context *pipe,
                       const struct st_config_options *options)
{
   struct pipe_screen *screen = pipe->screen;
   struct st_context *st = CALLOC_STRUCT( st_context);

   st->options = *options;

   ctx->st_opts = &st->options;
   ctx->st = st;

   st->ctx = ctx;
   st->screen = screen;
   st->pipe = pipe;

   st->can_bind_const_buffer_as_vertex =
      screen->caps.can_bind_const_buffer_as_vertex;

   /* st/mesa always uploads zero-stride vertex attribs, and other user
    * vertex buffers are only possible with a compatibility profile.
    * So tell the u_vbuf module that user VBOs are not possible with the Core
    * profile, so that u_vbuf is bypassed completely if there is nothing else
    * to do.
    */
   unsigned cso_flags;
   switch (ctx->API) {
   case API_OPENGL_CORE:
      cso_flags = CSO_NO_USER_VERTEX_BUFFERS;
      break;
   case API_OPENGLES:
   case API_OPENGLES2:
      cso_flags = CSO_NO_64B_VERTEX_BUFFERS;
      break;
   default:
      cso_flags = 0;
      break;
   }
   st->is_threaded_context = pipe->draw_vbo == tc_draw_vbo;

   st->cso_context = cso_create_context(pipe, cso_flags);
   ctx->cso_context = st->cso_context;

#define ST_STATE(FLAG, st_update) st->update_functions[FLAG] = st_update;
#include "st_atom_list.h"
#undef ST_STATE

   st_init_clear(st);
   {
      enum pipe_texture_transfer_mode val = screen->caps.texture_transfer_modes;
      st->prefer_blit_based_texture_transfer = (val & PIPE_TEXTURE_TRANSFER_BLIT) != 0;
      st->allow_compute_based_texture_transfer = (val & PIPE_TEXTURE_TRANSFER_COMPUTE) != 0;
   }
   st_init_pbo_helpers(st);

   /* Choose texture target for glDrawPixels, glBitmap, renderbuffers */
   if (screen->caps.npot_textures)
      st->internal_target = PIPE_TEXTURE_2D;
   else
      st->internal_target = PIPE_TEXTURE_RECT;

   /* Setup vertex element info for 'struct st_util_vertex'.
    */
   {
      STATIC_ASSERT(sizeof(struct st_util_vertex) == 9 * sizeof(float));

      memset(&st->util_velems, 0, sizeof(st->util_velems));
      st->util_velems.velems[0].src_offset = 0;
      st->util_velems.velems[0].vertex_buffer_index = 0;
      st->util_velems.velems[0].src_format = PIPE_FORMAT_R32G32B32_FLOAT;
      st->util_velems.velems[0].src_stride = sizeof(struct st_util_vertex);
      st->util_velems.velems[1].src_offset = 3 * sizeof(float);
      st->util_velems.velems[1].vertex_buffer_index = 0;
      st->util_velems.velems[1].src_format = PIPE_FORMAT_R32G32B32A32_FLOAT;
      st->util_velems.velems[1].src_stride = sizeof(struct st_util_vertex);
      st->util_velems.velems[2].src_offset = 7 * sizeof(float);
      st->util_velems.velems[2].vertex_buffer_index = 0;
      st->util_velems.velems[2].src_format = PIPE_FORMAT_R32G32_FLOAT;
      st->util_velems.velems[2].src_stride = sizeof(struct st_util_vertex);
   }

   ctx->Const.PackedDriverUniformStorage =
      screen->caps.packed_uniforms;

   ctx->Const.BitmapUsesRed =
      screen->is_format_supported(screen, PIPE_FORMAT_R8_UNORM,
                                  PIPE_TEXTURE_2D, 0, 0,
                                  PIPE_BIND_SAMPLER_VIEW);

   ctx->Const.QueryCounterBits.Timestamp =
      screen->caps.query_timestamp_bits;

   st->has_stencil_export =
      screen->caps.shader_stencil_export;
   st->has_etc1 = screen->is_format_supported(screen, PIPE_FORMAT_ETC1_RGB8,
                                              PIPE_TEXTURE_2D, 0, 0,
                                              PIPE_BIND_SAMPLER_VIEW);
   st->has_etc2 = screen->is_format_supported(screen, PIPE_FORMAT_ETC2_RGB8,
                                              PIPE_TEXTURE_2D, 0, 0,
                                              PIPE_BIND_SAMPLER_VIEW);
   st->transcode_etc = options->transcode_etc &&
                       screen->is_format_supported(screen, PIPE_FORMAT_DXT1_SRGBA,
                                                   PIPE_TEXTURE_2D, 0, 0,
                                                   PIPE_BIND_SAMPLER_VIEW);
   st->transcode_astc = options->transcode_astc &&
                        screen->is_format_supported(screen, PIPE_FORMAT_DXT5_SRGBA,
                                                    PIPE_TEXTURE_2D, 0, 0,
                                                    PIPE_BIND_SAMPLER_VIEW) &&
                        screen->is_format_supported(screen, PIPE_FORMAT_DXT5_RGBA,
                                                    PIPE_TEXTURE_2D, 0, 0,
                                                    PIPE_BIND_SAMPLER_VIEW);
   st->has_astc_2d_ldr =
      screen->is_format_supported(screen, PIPE_FORMAT_ASTC_4x4_SRGB,
                                  PIPE_TEXTURE_2D, 0, 0, PIPE_BIND_SAMPLER_VIEW);
   st->has_astc_5x5_ldr =
      screen->is_format_supported(screen, PIPE_FORMAT_ASTC_5x5_SRGB,
                                  PIPE_TEXTURE_2D, 0, 0, PIPE_BIND_SAMPLER_VIEW);
   st->astc_void_extents_need_denorm_flush =
      screen->caps.astc_void_extents_need_denorm_flush;

   st->has_s3tc = screen->is_format_supported(screen, PIPE_FORMAT_DXT5_RGBA,
                                              PIPE_TEXTURE_2D, 0, 0,
                                              PIPE_BIND_SAMPLER_VIEW);
   st->has_rgtc = screen->is_format_supported(screen, PIPE_FORMAT_RGTC2_UNORM,
                                              PIPE_TEXTURE_2D, 0, 0,
                                              PIPE_BIND_SAMPLER_VIEW);
   st->has_latc = screen->is_format_supported(screen, PIPE_FORMAT_LATC2_UNORM,
                                              PIPE_TEXTURE_2D, 0, 0,
                                              PIPE_BIND_SAMPLER_VIEW);
   st->has_bptc = screen->is_format_supported(screen, PIPE_FORMAT_BPTC_SRGBA,
                                              PIPE_TEXTURE_2D, 0, 0,
                                              PIPE_BIND_SAMPLER_VIEW);
   st->force_persample_in_shader =
      screen->caps.sample_shading &&
      !screen->caps.force_persample_interp;
   st->has_shareable_shaders = screen->caps.shareable_shaders;
   st->needs_texcoord_semantic =
      screen->caps.tgsi_texcoord;
   st->apply_texture_swizzle_to_border_color =
      !!(screen->caps.texture_border_color_quirk &
         (PIPE_QUIRK_TEXTURE_BORDER_COLOR_SWIZZLE_NV50 |
          PIPE_QUIRK_TEXTURE_BORDER_COLOR_SWIZZLE_R600));
   st->use_format_with_border_color =
      !!(screen->caps.texture_border_color_quirk &
         PIPE_QUIRK_TEXTURE_BORDER_COLOR_SWIZZLE_FREEDRENO);
   st->alpha_border_color_is_not_w =
      !!(screen->caps.texture_border_color_quirk &
         PIPE_QUIRK_TEXTURE_BORDER_COLOR_SWIZZLE_ALPHA_NOT_W);
   st->emulate_gl_clamp =
      !screen->caps.gl_clamp;
   st->has_time_elapsed =
      screen->caps.query_time_elapsed;
   ctx->Const.GLSLHasHalfFloatPacking =
      screen->caps.shader_pack_half_float;
   st->has_multi_draw_indirect =
      screen->caps.multi_draw_indirect;
   st->has_indirect_partial_stride =
      screen->caps.multi_draw_indirect_partial_stride;
   st->has_occlusion_query =
      screen->caps.occlusion_query;
   st->has_single_pipe_stat =
      screen->caps.query_pipeline_statistics_single;
   st->has_pipeline_stat =
      screen->caps.query_pipeline_statistics;
   st->has_indep_blend_enable =
      screen->caps.indep_blend_enable;
   st->has_indep_blend_func =
      screen->caps.indep_blend_func;
   st->can_dither =
      screen->caps.dithering;
   st->lower_flatshade =
      !screen->caps.flatshade;
   st->lower_alpha_test =
      !screen->caps.alpha_test;
   switch (screen->caps.point_size_fixed) {
   case PIPE_POINT_SIZE_LOWER_ALWAYS:
      st->lower_point_size = true;
      st->add_point_size = true;
      break;
   case PIPE_POINT_SIZE_LOWER_USER_ONLY:
      st->lower_point_size = true;
      break;
   default: break;
   }
   st->lower_two_sided_color =
      !screen->caps.two_sided_color;
   st->lower_ucp =
      !screen->caps.clip_planes;
   st->prefer_real_buffer_in_constbuf0 =
      screen->caps.prefer_real_buffer_in_constbuf0;
   st->has_conditional_render =
      screen->caps.conditional_render;
   st->lower_rect_tex =
      !screen->caps.texrect;
   st->allow_st_finalize_nir_twice =
      screen->caps.call_finalize_nir_in_linker;

   st->has_hw_atomics =
      screen->shader_caps[MESA_SHADER_FRAGMENT].max_hw_atomic_counters
      ? true : false;

   st->validate_all_dirty_states =
      screen->caps.validate_all_dirty_states
      ? true : false;
   st->can_null_texture =
      screen->caps.null_textures
      ? true : false;

   util_throttle_init(&st->throttle,
                      screen->caps.max_texture_upload_memory_budget);

   /* GL limits and extensions */
   st_init_limits(screen, &ctx->Const, &ctx->Extensions, ctx->API);
   st_init_extensions(screen, &ctx->Const,
                      &ctx->Extensions, &st->options, ctx->API);

   if (st_have_perfquery(st)) {
      ctx->Extensions.INTEL_performance_query = GL_TRUE;
   }

   /* Enable shader-based fallbacks for ARB_color_buffer_float if needed. */
   if (screen->caps.vertex_color_unclamped) {
      if (!screen->caps.vertex_color_clamped) {
         st->clamp_vert_color_in_shader = GL_TRUE;
      }

      if (!screen->caps.fragment_color_clamped) {
         st->clamp_frag_color_in_shader = GL_TRUE;
      }

      /* For drivers which cannot do color clamping, it's better to just
       * disable ARB_color_buffer_float in the core profile, because
       * the clamping is deprecated there anyway. */
      if (_mesa_is_desktop_gl_core(ctx) &&
          (st->clamp_frag_color_in_shader || st->clamp_vert_color_in_shader)) {
         st->clamp_vert_color_in_shader = GL_FALSE;
         st->clamp_frag_color_in_shader = GL_FALSE;
         ctx->Extensions.ARB_color_buffer_float = GL_FALSE;
      }
   }

   /* called after _mesa_create_context/_mesa_init_point, fix default user
    * settable max point size up
    */
   ctx->Point.MaxSize = MAX2(ctx->Const.MaxPointSize,
                             ctx->Const.MaxPointSizeAA);

   ctx->Const.NoClippingOnCopyTex = screen->caps.no_clip_on_copy_tex;

   ctx->Const.ForceFloat32TexNearest =
      !screen->caps.texture_float_linear;

   ctx->Const.VSPositionAlwaysInvariant = options->vs_position_always_invariant;

   ctx->Const.TESPositionAlwaysPrecise = options->vs_position_always_precise;

   /* Set which shader types can be compiled at link time. */
   st->shader_has_one_variant[MESA_SHADER_VERTEX] =
         st->has_shareable_shaders &&
         !st->clamp_vert_color_in_shader &&
         !st->lower_point_size &&
         !st->lower_ucp;

   st->shader_has_one_variant[MESA_SHADER_FRAGMENT] =
         st->has_shareable_shaders &&
         !st->lower_flatshade &&
         !st->lower_alpha_test &&
         !st->clamp_frag_color_in_shader &&
         !st->force_persample_in_shader &&
         !st->lower_two_sided_color;

   st->shader_has_one_variant[MESA_SHADER_TESS_CTRL] = st->has_shareable_shaders;
   st->shader_has_one_variant[MESA_SHADER_TESS_EVAL] =
         st->has_shareable_shaders &&
         !st->clamp_vert_color_in_shader &&
         !st->lower_point_size &&
         !st->lower_ucp;

   st->shader_has_one_variant[MESA_SHADER_GEOMETRY] =
         st->has_shareable_shaders &&
         !st->clamp_vert_color_in_shader &&
         !st->lower_point_size &&
         !st->lower_ucp;
   st->shader_has_one_variant[MESA_SHADER_COMPUTE] = st->has_shareable_shaders;

   if (!st->pipe->set_context_param || !util_thread_scheduler_enabled())
      st->thread_scheduler_disabled = true;

   st->bitmap.cache.empty = true;

   _mesa_override_extensions(ctx);
   _mesa_compute_version(ctx);

   if (ctx->Version == 0 ||
       !_mesa_initialize_dispatch_tables(ctx)) {
      /* This can happen when a core profile was requested, but the driver
       * does not support some features of GL 3.1 or later.
       */
      st_destroy_context_priv(st, false);
      return NULL;
   }

   if (_mesa_has_compute_shaders(ctx) &&
       st->transcode_astc && !st_init_texcompress_compute(st)) {
      /* Transcoding ASTC to DXT5 using compute shaders can provide a
       * significant performance benefit over the CPU path. It isn't strictly
       * necessary to fail if we can't use the compute shader path, but it's
       * very convenient to do so. This should be rare.
       */
      st_destroy_context_priv(st, false);
      return NULL;
   }

   /* This must be done after extensions are initialized to enable persistent
    * mappings immediately.
    */
   _vbo_CreateContext(ctx);

   st_init_driver_flags(st);
   st_init_update_array(st);

   /* Initialize context's winsys buffers list */
   list_inithead(&st->winsys_buffers);

   list_inithead(&st->zombie_sampler_views.list.node);
   simple_mtx_init(&st->zombie_sampler_views.mutex, mtx_plain);
   list_inithead(&st->zombie_shaders.list.node);
   simple_mtx_init(&st->zombie_shaders.mutex, mtx_plain);

   st->release_resources = UTIL_DYNARRAY_INIT;

   ctx->Const.DriverSupportedPrimMask = screen->caps.supported_prim_modes |
                                        /* patches is always supported */
                                        BITFIELD_BIT(MESA_PRIM_PATCHES);
   _mesa_set_active_states(ctx);

   return st;
}

void
st_set_background_context(struct gl_context *ctx,
                          struct util_queue_monitoring *queue_info)
{
   struct st_context *st = ctx->st;
   struct pipe_frontend_screen *fscreen = st->frontend_screen;

   assert(fscreen->set_background_context);
   fscreen->set_background_context(st, queue_info);
}

static void
st_init_driver_functions(struct pipe_screen *screen,
                         struct dd_function_table *functions,
                         bool has_egl_image_validate)
{
   st_init_draw_functions(screen, functions);

   functions->NewProgram = _mesa_new_program;
   st_init_flush_functions(screen, functions);

   /* GL_ARB_get_program_binary */
   functions->ShaderCacheSerializeDriverBlob =  st_serialise_nir_program;
   functions->ProgramBinarySerializeDriverBlob =
      st_serialise_nir_program_binary;
   functions->ProgramBinaryDeserializeDriverBlob =
      st_deserialise_nir_program;
}


struct st_context *
st_create_context(gl_api api, struct pipe_context *pipe,
                  const struct gl_config *visual,
                  struct st_context *share,
                  const struct st_config_options *options,
                  bool no_error, bool has_egl_image_validate)
{
   struct gl_context *ctx;
   struct gl_context *shareCtx = share ? share->ctx : NULL;
   struct dd_function_table funcs;
   struct st_context *st;

   memset(&funcs, 0, sizeof(funcs));
   st_init_driver_functions(pipe->screen, &funcs, has_egl_image_validate);

   /* gl_context must be 16-byte aligned due to the alignment on GLmatrix. */
   ctx = align_malloc(sizeof(struct gl_context), 16);
   if (!ctx)
      return NULL;
   memset(ctx, 0, sizeof(*ctx));

   ctx->pipe = pipe;
   ctx->screen = pipe->screen;

   if (!_mesa_initialize_context(ctx, api, no_error, visual, shareCtx, &funcs,
                                 options)) {
      align_free(ctx);
      return NULL;
   }

   st_debug_init();

   if (pipe->screen->get_disk_shader_cache)
      ctx->Cache = pipe->screen->get_disk_shader_cache(pipe->screen);

   if (pipe->screen->caps.invalidate_buffer)
      ctx->has_invalidate_buffer = true;

   if (pipe->screen->caps.string_marker)
      ctx->has_string_marker = true;

   st = st_create_context_priv(ctx, pipe, options);
   if (!st) {
      _mesa_free_context_data(ctx, true);
      align_free(ctx);
   }

   return st;
}


/**
 * When we destroy a context, we must examine all texture objects to
 * find/release any sampler views created by that context.
 *
 * This callback is called per-texture object.  It releases all the
 * texture's sampler views which belong to the context.
 */
static void
destroy_tex_sampler_cb(void *data, void *userData)
{
   struct gl_texture_object *texObj = (struct gl_texture_object *) data;
   struct st_context *st = (struct st_context *) userData;

   st_texture_release_context_sampler_view(st, texObj);
}

static void
destroy_framebuffer_attachment_sampler_cb(void *data, void *userData)
{
   struct gl_framebuffer* glfb = (struct gl_framebuffer*) data;
   struct st_context *st = (struct st_context *) userData;

    for (unsigned i = 0; i < BUFFER_COUNT; i++) {
      struct gl_renderbuffer_attachment *att = &glfb->Attachment[i];
      if (att->Texture) {
        st_texture_release_context_sampler_view(st, att->Texture);
      }
   }
}

void
st_prune_releasebufs(struct st_context *st)
{
   struct pipe_resource **pres = st->release_resources.data;
   unsigned count = util_dynarray_num_elements(&st->release_resources, struct pipe_resource*);
   for (unsigned j = 0; j < count; j++)
      pipe_resource_release(st->pipe, pres[j]);
   util_dynarray_clear(&st->release_resources);
   st->release_counter = 0;
}

void
st_add_releasebuf(struct st_context *st, struct pipe_resource *releasebuf)
{
   if (!releasebuf)
      return;
   if (st->release_counter != st->work_counter)
      st_prune_releasebufs(st);
   util_dynarray_append(&st->release_resources, releasebuf);
   st->release_counter = st->work_counter;
}

void
st_destroy_context(struct st_context *st)
{
   struct gl_context *ctx = st->ctx;
   struct gl_framebuffer *stfb, *next;
   struct gl_framebuffer *save_drawbuffer;
   struct gl_framebuffer *save_readbuffer;

   /* Save the current context and draw/read buffers*/
   GET_CURRENT_CONTEXT(save_ctx);
   if (save_ctx) {
      save_drawbuffer = save_ctx->WinSysDrawBuffer;
      save_readbuffer = save_ctx->WinSysReadBuffer;
      /* Calling _mesa_glthread_finish is required to avoid having an
       * unmarshalling thread execute calls while the current thread
       * flushes the context.
       */
      _mesa_glthread_finish(save_ctx);
   } else {
      save_drawbuffer = save_readbuffer = NULL;
   }

   /*
    * We need to bind the context we're deleting so that
    * _mesa_reference_texobj_() uses this context when deleting textures.
    * Similarly for framebuffer objects, etc.
    */
   _mesa_make_current(ctx, NULL, NULL);

   /* This must be called first so that glthread has a chance to finish */
   _mesa_glthread_destroy(ctx);

   _mesa_HashWalk(&ctx->Shared->TexObjects, destroy_tex_sampler_cb, st);

   /* For the fallback textures, free any sampler views belonging to this
    * context.
    */
   for (unsigned i = 0; i < NUM_TEXTURE_TARGETS; i++) {
      for (unsigned j = 0; j < ARRAY_SIZE(ctx->Shared->FallbackTex[0]); j++) {
         struct gl_texture_object *stObj =
            ctx->Shared->FallbackTex[i][j];
         if (stObj) {
            st_texture_release_context_sampler_view(st, stObj);
         }
      }
   }

   st_prune_releasebufs(st);
   util_dynarray_fini(&st->release_resources);

   st_release_program(st, &st->fp);
   st_release_program(st, &st->gp);
   st_release_program(st, &st->vp);
   st_release_program(st, &st->tcp);
   st_release_program(st, &st->tep);
   st_release_program(st, &st->cp);
   st_release_program(st, &st->tp);
   st_release_program(st, &st->mp);

   if (st->hw_select_shaders) {
      hash_table_foreach(st->hw_select_shaders, entry)
         st->pipe->delete_gs_state(st->pipe, entry->data);
      _mesa_hash_table_destroy(st->hw_select_shaders, NULL);
   }

   /* release framebuffer in the winsys buffers list */
   LIST_FOR_EACH_ENTRY_SAFE_REV(stfb, next, &st->winsys_buffers, head) {
      _mesa_reference_framebuffer(&stfb, NULL);
   }

   _mesa_HashWalk(&ctx->Shared->FrameBuffers, destroy_framebuffer_attachment_sampler_cb, st);

   st->pipe->sampler_view_release(st->pipe, st->pixel_xfer.pixelmap_sampler_view);
   pipe_resource_reference(&st->pixel_xfer.pixelmap_texture, NULL);

   _vbo_DestroyContext(ctx);

   st_destroy_program_variants(st);

   /* Do not release debug_output yet because it might be in use by other threads.
    * These threads will be terminated by _mesa_free_context_data and
    * st_destroy_context_priv.
    */
   _mesa_free_context_data(ctx, false);

   st_context_free_zombie_objects(st);

   simple_mtx_destroy(&st->zombie_sampler_views.mutex);
   simple_mtx_destroy(&st->zombie_shaders.mutex);

   /* This will free the st_context too, so 'st' must not be accessed
    * afterwards. */
   st_destroy_context_priv(st, true);
   st = NULL;

   _mesa_destroy_debug_output(ctx);

   align_free(ctx);

   if (save_ctx == ctx) {
      /* unbind the context we just deleted */
      _mesa_make_current(NULL, NULL, NULL);
   } else {
      /* Restore the current context and draw/read buffers (may be NULL) */
      _mesa_make_current(save_ctx, save_drawbuffer, save_readbuffer);
   }
}
