// SPDX-License-Identifier: GPL-2.0
/*
 * Sophgo SG2042 SoC pinctrl driver.
 *
 * Copyright (C) 2024 Inochi Amaoto <inochiama@outlook.com>
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of.h>

#include <linux/pinctrl/pinctrl.h>
#include <linux/pinctrl/pinmux.h>

#include <dt-bindings/pinctrl/pinctrl-sg2044.h>

#include "pinctrl-sg2042.h"

static int sg2044_get_pull_up(const struct sophgo_pin *sp, const u32 *psmap)
{
	return 19500;
}

static int sg2044_get_pull_down(const struct sophgo_pin *sp, const u32 *psmap)
{
	return 23200;
}

static const u32 sg2044_oc_map[] = {
	3200,  6400,  9600,  12700,
	15900, 19100, 22200, 25300,
	29500, 32700, 35900, 39000,
	42000, 45200, 48300, 51400
};

static int sg2044_get_oc_map(const struct sophgo_pin *sp, const u32 *psmap,
			     const u32 **map)
{
	*map = sg2044_oc_map;
	return ARRAY_SIZE(sg2044_oc_map);
}

static const struct sophgo_vddio_cfg_ops sg2044_vddio_cfg_ops = {
	.get_pull_up		= sg2044_get_pull_up,
	.get_pull_down		= sg2044_get_pull_down,
	.get_oc_map		= sg2044_get_oc_map,
};

static const struct pinctrl_pin_desc sg2044_pins[] = {
	PINCTRL_PIN(PIN_IIC0_SMBSUS_IN,		"iic0_smbsus_in"),
	PINCTRL_PIN(PIN_IIC0_SMBSUS_OUT,	"iic0_smbsus_out"),
	PINCTRL_PIN(PIN_IIC0_SMBALERT,		"iic0_smbalert"),
	PINCTRL_PIN(PIN_IIC1_SMBSUS_IN,		"iic1_smbsus_in"),
	PINCTRL_PIN(PIN_IIC1_SMBSUS_OUT,	"iic1_smbsus_out"),
	PINCTRL_PIN(PIN_IIC1_SMBALERT,		"iic1_smbalert"),
	PINCTRL_PIN(PIN_IIC2_SMBSUS_IN,		"iic2_smbsus_in"),
	PINCTRL_PIN(PIN_IIC2_SMBSUS_OUT,	"iic2_smbsus_out"),
	PINCTRL_PIN(PIN_IIC2_SMBALERT,		"iic2_smbalert"),
	PINCTRL_PIN(PIN_IIC3_SMBSUS_IN,		"iic3_smbsus_in"),
	PINCTRL_PIN(PIN_IIC3_SMBSUS_OUT,	"iic3_smbsus_out"),
	PINCTRL_PIN(PIN_IIC3_SMBALERT,		"iic3_smbalert"),
	PINCTRL_PIN(PIN_PCIE0_L0_RESET,		"pcie0_l0_reset"),
	PINCTRL_PIN(PIN_PCIE0_L1_RESET,		"pcie0_l1_reset"),
	PINCTRL_PIN(PIN_PCIE0_L0_WAKEUP,	"pcie0_l0_wakeup"),
	PINCTRL_PIN(PIN_PCIE0_L1_WAKEUP,	"pcie0_l1_wakeup"),
	PINCTRL_PIN(PIN_PCIE0_L0_CLKREQ_IN,	"pcie0_l0_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE0_L1_CLKREQ_IN,	"pcie0_l1_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE1_L0_RESET,		"pcie1_l0_reset"),
	PINCTRL_PIN(PIN_PCIE1_L1_RESET,		"pcie1_l1_reset"),
	PINCTRL_PIN(PIN_PCIE1_L0_WAKEUP,	"pcie1_l0_wakeup"),
	PINCTRL_PIN(PIN_PCIE1_L1_WAKEUP,	"pcie1_l1_wakeup"),
	PINCTRL_PIN(PIN_PCIE1_L0_CLKREQ_IN,	"pcie1_l0_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE1_L1_CLKREQ_IN,	"pcie1_l1_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE2_L0_RESET,		"pcie2_l0_reset"),
	PINCTRL_PIN(PIN_PCIE2_L1_RESET,		"pcie2_l1_reset"),
	PINCTRL_PIN(PIN_PCIE2_L0_WAKEUP,	"pcie2_l0_wakeup"),
	PINCTRL_PIN(PIN_PCIE2_L1_WAKEUP,	"pcie2_l1_wakeup"),
	PINCTRL_PIN(PIN_PCIE2_L0_CLKREQ_IN,	"pcie2_l0_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE2_L1_CLKREQ_IN,	"pcie2_l1_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE3_L0_RESET,		"pcie3_l0_reset"),
	PINCTRL_PIN(PIN_PCIE3_L1_RESET,		"pcie3_l1_reset"),
	PINCTRL_PIN(PIN_PCIE3_L0_WAKEUP,	"pcie3_l0_wakeup"),
	PINCTRL_PIN(PIN_PCIE3_L1_WAKEUP,	"pcie3_l1_wakeup"),
	PINCTRL_PIN(PIN_PCIE3_L0_CLKREQ_IN,	"pcie3_l0_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE3_L1_CLKREQ_IN,	"pcie3_l1_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE4_L0_RESET,		"pcie4_l0_reset"),
	PINCTRL_PIN(PIN_PCIE4_L1_RESET,		"pcie4_l1_reset"),
	PINCTRL_PIN(PIN_PCIE4_L0_WAKEUP,	"pcie4_l0_wakeup"),
	PINCTRL_PIN(PIN_PCIE4_L1_WAKEUP,	"pcie4_l1_wakeup"),
	PINCTRL_PIN(PIN_PCIE4_L0_CLKREQ_IN,	"pcie4_l0_clkreq_in"),
	PINCTRL_PIN(PIN_PCIE4_L1_CLKREQ_IN,	"pcie4_l1_clkreq_in"),
	PINCTRL_PIN(PIN_SPIF0_CLK_SEL1,		"spif0_clk_sel1"),
	PINCTRL_PIN(PIN_SPIF0_CLK_SEL0,		"spif0_clk_sel0"),
	PINCTRL_PIN(PIN_SPIF0_WP,		"spif0_wp"),
	PINCTRL_PIN(PIN_SPIF0_HOLD,		"spif0_hold"),
	PINCTRL_PIN(PIN_SPIF0_SDI,		"spif0_sdi"),
	PINCTRL_PIN(PIN_SPIF0_CS,		"spif0_cs"),
	PINCTRL_PIN(PIN_SPIF0_SCK,		"spif0_sck"),
	PINCTRL_PIN(PIN_SPIF0_SDO,		"spif0_sdo"),
	PINCTRL_PIN(PIN_SPIF1_CLK_SEL1,		"spif1_clk_sel1"),
	PINCTRL_PIN(PIN_SPIF1_CLK_SEL0,		"spif1_clk_sel0"),
	PINCTRL_PIN(PIN_SPIF1_WP,		"spif1_wp"),
	PINCTRL_PIN(PIN_SPIF1_HOLD,		"spif1_hold"),
	PINCTRL_PIN(PIN_SPIF1_SDI,		"spif1_sdi"),
	PINCTRL_PIN(PIN_SPIF1_CS,		"spif1_cs"),
	PINCTRL_PIN(PIN_SPIF1_SCK,		"spif1_sck"),
	PINCTRL_PIN(PIN_SPIF1_SDO,		"spif1_sdo"),
	PINCTRL_PIN(PIN_EMMC_WP,		"emmc_wp"),
	PINCTRL_PIN(PIN_EMMC_CD,		"emmc_cd"),
	PINCTRL_PIN(PIN_EMMC_RST,		"emmc_rst"),
	PINCTRL_PIN(PIN_EMMC_PWR_EN,		"emmc_pwr_en"),
	PINCTRL_PIN(PIN_SDIO_CD,		"sdio_cd"),
	PINCTRL_PIN(PIN_SDIO_WP,		"sdio_wp"),
	PINCTRL_PIN(PIN_SDIO_RST,		"sdio_rst"),
	PINCTRL_PIN(PIN_SDIO_PWR_EN,		"sdio_pwr_en"),
	PINCTRL_PIN(PIN_RGMII0_TXD0,		"rgmii0_txd0"),
	PINCTRL_PIN(PIN_RGMII0_TXD1,		"rgmii0_txd1"),
	PINCTRL_PIN(PIN_RGMII0_TXD2,		"rgmii0_txd2"),
	PINCTRL_PIN(PIN_RGMII0_TXD3,		"rgmii0_txd3"),
	PINCTRL_PIN(PIN_RGMII0_TXCTRL,		"rgmii0_txctrl"),
	PINCTRL_PIN(PIN_RGMII0_RXD0,		"rgmii0_rxd0"),
	PINCTRL_PIN(PIN_RGMII0_RXD1,		"rgmii0_rxd1"),
	PINCTRL_PIN(PIN_RGMII0_RXD2,		"rgmii0_rxd2"),
	PINCTRL_PIN(PIN_RGMII0_RXD3,		"rgmii0_rxd3"),
	PINCTRL_PIN(PIN_RGMII0_RXCTRL,		"rgmii0_rxctrl"),
	PINCTRL_PIN(PIN_RGMII0_TXC,		"rgmii0_txc"),
	PINCTRL_PIN(PIN_RGMII0_RXC,		"rgmii0_rxc"),
	PINCTRL_PIN(PIN_RGMII0_REFCLKO,		"rgmii0_refclko"),
	PINCTRL_PIN(PIN_RGMII0_IRQ,		"rgmii0_irq"),
	PINCTRL_PIN(PIN_RGMII0_MDC,		"rgmii0_mdc"),
	PINCTRL_PIN(PIN_RGMII0_MDIO,		"rgmii0_mdio"),
	PINCTRL_PIN(PIN_PWM0,			"pwm0"),
	PINCTRL_PIN(PIN_PWM1,			"pwm1"),
	PINCTRL_PIN(PIN_PWM2,			"pwm2"),
	PINCTRL_PIN(PIN_PWM3,			"pwm3"),
	PINCTRL_PIN(PIN_FAN0,			"fan0"),
	PINCTRL_PIN(PIN_FAN1,			"fan1"),
	PINCTRL_PIN(PIN_FAN2,			"fan2"),
	PINCTRL_PIN(PIN_FAN3,			"fan3"),
	PINCTRL_PIN(PIN_IIC0_SDA,		"iic0_sda"),
	PINCTRL_PIN(PIN_IIC0_SCL,		"iic0_scl"),
	PINCTRL_PIN(PIN_IIC1_SDA,		"iic1_sda"),
	PINCTRL_PIN(PIN_IIC1_SCL,		"iic1_scl"),
	PINCTRL_PIN(PIN_IIC2_SDA,		"iic2_sda"),
	PINCTRL_PIN(PIN_IIC2_SCL,		"iic2_scl"),
	PINCTRL_PIN(PIN_IIC3_SDA,		"iic3_sda"),
	PINCTRL_PIN(PIN_IIC3_SCL,		"iic3_scl"),
	PINCTRL_PIN(PIN_UART0_TX,		"uart0_tx"),
	PINCTRL_PIN(PIN_UART0_RX,		"uart0_rx"),
	PINCTRL_PIN(PIN_UART0_RTS,		"uart0_rts"),
	PINCTRL_PIN(PIN_UART0_CTS,		"uart0_cts"),
	PINCTRL_PIN(PIN_UART1_TX,		"uart1_tx"),
	PINCTRL_PIN(PIN_UART1_RX,		"uart1_rx"),
	PINCTRL_PIN(PIN_UART1_RTS,		"uart1_rts"),
	PINCTRL_PIN(PIN_UART1_CTS,		"uart1_cts"),
	PINCTRL_PIN(PIN_UART2_TX,		"uart2_tx"),
	PINCTRL_PIN(PIN_UART2_RX,		"uart2_rx"),
	PINCTRL_PIN(PIN_UART2_RTS,		"uart2_rts"),
	PINCTRL_PIN(PIN_UART2_CTS,		"uart2_cts"),
	PINCTRL_PIN(PIN_UART3_TX,		"uart3_tx"),
	PINCTRL_PIN(PIN_UART3_RX,		"uart3_rx"),
	PINCTRL_PIN(PIN_UART3_RTS,		"uart3_rts"),
	PINCTRL_PIN(PIN_UART3_CTS,		"uart3_cts"),
	PINCTRL_PIN(PIN_SPI0_CS0,		"spi0_cs0"),
	PINCTRL_PIN(PIN_SPI0_CS1,		"spi0_cs1"),
	PINCTRL_PIN(PIN_SPI0_SDI,		"spi0_sdi"),
	PINCTRL_PIN(PIN_SPI0_SDO,		"spi0_sdo"),
	PINCTRL_PIN(PIN_SPI0_SCK,		"spi0_sck"),
	PINCTRL_PIN(PIN_SPI1_CS0,		"spi1_cs0"),
	PINCTRL_PIN(PIN_SPI1_CS1,		"spi1_cs1"),
	PINCTRL_PIN(PIN_SPI1_SDI,		"spi1_sdi"),
	PINCTRL_PIN(PIN_SPI1_SDO,		"spi1_sdo"),
	PINCTRL_PIN(PIN_SPI1_SCK,		"spi1_sck"),
	PINCTRL_PIN(PIN_JTAG0_TDO,		"jtag0_tdo"),
	PINCTRL_PIN(PIN_JTAG0_TCK,		"jtag0_tck"),
	PINCTRL_PIN(PIN_JTAG0_TDI,		"jtag0_tdi"),
	PINCTRL_PIN(PIN_JTAG0_TMS,		"jtag0_tms"),
	PINCTRL_PIN(PIN_JTAG0_TRST,		"jtag0_trst"),
	PINCTRL_PIN(PIN_JTAG0_SRST,		"jtag0_srst"),
	PINCTRL_PIN(PIN_JTAG1_TDO,		"jtag1_tdo"),
	PINCTRL_PIN(PIN_JTAG1_TCK,		"jtag1_tck"),
	PINCTRL_PIN(PIN_JTAG1_TDI,		"jtag1_tdi"),
	PINCTRL_PIN(PIN_JTAG1_TMS,		"jtag1_tms"),
	PINCTRL_PIN(PIN_JTAG1_TRST,		"jtag1_trst"),
	PINCTRL_PIN(PIN_JTAG1_SRST,		"jtag1_srst"),
	PINCTRL_PIN(PIN_JTAG2_TDO,		"jtag2_tdo"),
	PINCTRL_PIN(PIN_JTAG2_TCK,		"jtag2_tck"),
	PINCTRL_PIN(PIN_JTAG2_TDI,		"jtag2_tdi"),
	PINCTRL_PIN(PIN_JTAG2_TMS,		"jtag2_tms"),
	PINCTRL_PIN(PIN_JTAG2_TRST,		"jtag2_trst"),
	PINCTRL_PIN(PIN_JTAG2_SRST,		"jtag2_srst"),
	PINCTRL_PIN(PIN_JTAG3_TDO,		"jtag3_tdo"),
	PINCTRL_PIN(PIN_JTAG3_TCK,		"jtag3_tck"),
	PINCTRL_PIN(PIN_JTAG3_TDI,		"jtag3_tdi"),
	PINCTRL_PIN(PIN_JTAG3_TMS,		"jtag3_tms"),
	PINCTRL_PIN(PIN_JTAG3_TRST,		"jtag3_trst"),
	PINCTRL_PIN(PIN_JTAG3_SRST,		"jtag3_srst"),
	PINCTRL_PIN(PIN_GPIO0,			"gpio0"),
	PINCTRL_PIN(PIN_GPIO1,			"gpio1"),
	PINCTRL_PIN(PIN_GPIO2,			"gpio2"),
	PINCTRL_PIN(PIN_GPIO3,			"gpio3"),
	PINCTRL_PIN(PIN_GPIO4,			"gpio4"),
	PINCTRL_PIN(PIN_GPIO5,			"gpio5"),
	PINCTRL_PIN(PIN_GPIO6,			"gpio6"),
	PINCTRL_PIN(PIN_GPIO7,			"gpio7"),
	PINCTRL_PIN(PIN_GPIO8,			"gpio8"),
	PINCTRL_PIN(PIN_GPIO9,			"gpio9"),
	PINCTRL_PIN(PIN_GPIO10,			"gpio10"),
	PINCTRL_PIN(PIN_GPIO11,			"gpio11"),
	PINCTRL_PIN(PIN_GPIO12,			"gpio12"),
	PINCTRL_PIN(PIN_GPIO13,			"gpio13"),
	PINCTRL_PIN(PIN_GPIO14,			"gpio14"),
	PINCTRL_PIN(PIN_GPIO15,			"gpio15"),
	PINCTRL_PIN(PIN_GPIO16,			"gpio16"),
	PINCTRL_PIN(PIN_GPIO17,			"gpio17"),
	PINCTRL_PIN(PIN_GPIO18,			"gpio18"),
	PINCTRL_PIN(PIN_GPIO19,			"gpio19"),
	PINCTRL_PIN(PIN_GPIO20,			"gpio20"),
	PINCTRL_PIN(PIN_GPIO21,			"gpio21"),
	PINCTRL_PIN(PIN_GPIO22,			"gpio22"),
	PINCTRL_PIN(PIN_GPIO23,			"gpio23"),
	PINCTRL_PIN(PIN_GPIO24,			"gpio24"),
	PINCTRL_PIN(PIN_GPIO25,			"gpio25"),
	PINCTRL_PIN(PIN_GPIO26,			"gpio26"),
	PINCTRL_PIN(PIN_GPIO27,			"gpio27"),
	PINCTRL_PIN(PIN_GPIO28,			"gpio28"),
	PINCTRL_PIN(PIN_GPIO29,			"gpio29"),
	PINCTRL_PIN(PIN_GPIO30,			"gpio30"),
	PINCTRL_PIN(PIN_GPIO31,			"gpio31"),
	PINCTRL_PIN(PIN_MODE_SEL0,		"mode_sel0"),
	PINCTRL_PIN(PIN_MODE_SEL1,		"mode_sel1"),
	PINCTRL_PIN(PIN_MODE_SEL2,		"mode_sel2"),
	PINCTRL_PIN(PIN_BOOT_SEL0,		"boot_sel0"),
	PINCTRL_PIN(PIN_BOOT_SEL1,		"boot_sel1"),
	PINCTRL_PIN(PIN_BOOT_SEL2,		"boot_sel2"),
	PINCTRL_PIN(PIN_BOOT_SEL3,		"boot_sel3"),
	PINCTRL_PIN(PIN_BOOT_SEL4,		"boot_sel4"),
	PINCTRL_PIN(PIN_BOOT_SEL5,		"boot_sel5"),
	PINCTRL_PIN(PIN_BOOT_SEL6,		"boot_sel6"),
	PINCTRL_PIN(PIN_BOOT_SEL7,		"boot_sel7"),
	PINCTRL_PIN(PIN_MULTI_SCKT,		"multi_sckt"),
	PINCTRL_PIN(PIN_SCKT_ID0,		"sckt_id0"),
	PINCTRL_PIN(PIN_SCKT_ID1,		"sckt_id1"),
	PINCTRL_PIN(PIN_PLL_CLK_IN_MAIN,	"pll_clk_in_main"),
	PINCTRL_PIN(PIN_PLL_CLK_IN_DDR_0,	"pll_clk_in_ddr_0"),
	PINCTRL_PIN(PIN_PLL_CLK_IN_DDR_1,	"pll_clk_in_ddr_1"),
	PINCTRL_PIN(PIN_PLL_CLK_IN_DDR_2,	"pll_clk_in_ddr_2"),
	PINCTRL_PIN(PIN_PLL_CLK_IN_DDR_3,	"pll_clk_in_ddr_3"),
	PINCTRL_PIN(PIN_XTAL_32K,		"xtal_32k"),
	PINCTRL_PIN(PIN_SYS_RST,		"sys_rst"),
	PINCTRL_PIN(PIN_PWR_BUTTON,		"pwr_button"),
	PINCTRL_PIN(PIN_TEST_EN,		"test_en"),
	PINCTRL_PIN(PIN_TEST_MODE_MBIST,	"test_mode_mbist"),
	PINCTRL_PIN(PIN_TEST_MODE_SCAN,		"test_mode_scan"),
	PINCTRL_PIN(PIN_TEST_MODE_BSD,		"test_mode_bsd"),
	PINCTRL_PIN(PIN_BISR_BYP,		"bisr_byp"),
};

static const struct sg2042_pin sg2044_pin_data[ARRAY_SIZE(sg2044_pins)] = {
	SG2042_GENERAL_PIN(PIN_IIC0_SMBSUS_IN, 0x000,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC0_SMBSUS_OUT, 0x000,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC0_SMBALERT, 0x004,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC1_SMBSUS_IN, 0x004,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC1_SMBSUS_OUT, 0x008,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC1_SMBALERT, 0x008,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC2_SMBSUS_IN, 0x00c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC2_SMBSUS_OUT, 0x00c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC2_SMBALERT, 0x010,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC3_SMBSUS_IN, 0x010,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC3_SMBSUS_OUT, 0x014,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC3_SMBALERT, 0x014,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE0_L0_RESET, 0x018,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE0_L1_RESET, 0x018,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE0_L0_WAKEUP, 0x01c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE0_L1_WAKEUP, 0x01c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE0_L0_CLKREQ_IN, 0x020,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE0_L1_CLKREQ_IN, 0x020,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE1_L0_RESET, 0x024,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE1_L1_RESET, 0x024,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE1_L0_WAKEUP, 0x028,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE1_L1_WAKEUP, 0x028,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE1_L0_CLKREQ_IN, 0x02c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE1_L1_CLKREQ_IN, 0x02c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE2_L0_RESET, 0x030,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE2_L1_RESET, 0x030,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE2_L0_WAKEUP, 0x034,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE2_L1_WAKEUP, 0x034,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE2_L0_CLKREQ_IN, 0x038,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE2_L1_CLKREQ_IN, 0x038,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE3_L0_RESET, 0x03c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE3_L1_RESET, 0x03c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE3_L0_WAKEUP, 0x040,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE3_L1_WAKEUP, 0x040,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE3_L0_CLKREQ_IN, 0x044,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE3_L1_CLKREQ_IN, 0x044,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE4_L0_RESET, 0x048,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE4_L1_RESET, 0x048,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE4_L0_WAKEUP, 0x04c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE4_L1_WAKEUP, 0x04c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PCIE4_L0_CLKREQ_IN, 0x050,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PCIE4_L1_CLKREQ_IN, 0x050,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPIF0_CLK_SEL1, 0x054,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPIF0_CLK_SEL0, 0x054,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPIF0_WP, 0x058,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPIF0_HOLD, 0x058,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPIF0_SDI, 0x05c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPIF0_CS, 0x05c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPIF0_SCK, 0x060,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPIF0_SDO, 0x060,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPIF1_CLK_SEL1, 0x064,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPIF1_CLK_SEL0, 0x064,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPIF1_WP, 0x068,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPIF1_HOLD, 0x068,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPIF1_SDI, 0x06c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPIF1_CS, 0x06c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPIF1_SCK, 0x070,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPIF1_SDO, 0x070,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_EMMC_WP, 0x074,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_EMMC_CD, 0x074,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_EMMC_RST, 0x078,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_EMMC_PWR_EN, 0x078,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SDIO_CD, 0x07c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SDIO_WP, 0x07c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SDIO_RST, 0x080,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SDIO_PWR_EN, 0x080,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXD0, 0x084,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXD1, 0x084,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXD2, 0x088,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXD3, 0x088,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXCTRL, 0x08c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXD0, 0x08c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXD1, 0x090,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXD2, 0x090,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXD3, 0x094,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXCTRL, 0x094,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_TXC, 0x098,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_RXC, 0x098,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_REFCLKO, 0x09c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_IRQ, 0x09c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_RGMII0_MDC, 0x0a0,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_RGMII0_MDIO, 0x0a0,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PWM0, 0x0a4,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PWM1, 0x0a4,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_PWM2, 0x0a8,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_PWM3, 0x0a8,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_FAN0, 0x0ac,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_FAN1, 0x0ac,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_FAN2, 0x0b0,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_FAN3, 0x0b0,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC0_SDA, 0x0b4,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC0_SCL, 0x0b4,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC1_SDA, 0x0b8,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC1_SCL, 0x0b8,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC2_SDA, 0x0bc,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC2_SCL, 0x0bc,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_IIC3_SDA, 0x0c0,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_IIC3_SCL, 0x0c0,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_UART0_TX, 0x0c4,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_UART0_RX, 0x0c4,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_UART0_RTS, 0x0c8,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_UART0_CTS, 0x0c8,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_UART1_TX, 0x0cc,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_UART1_RX, 0x0cc,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_UART1_RTS, 0x0d0,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_UART1_CTS, 0x0d0,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_UART2_TX, 0x0d4,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_UART2_RX, 0x0d4,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_UART2_RTS, 0x0d8,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_UART2_CTS, 0x0d8,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_UART3_TX, 0x0dc,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_UART3_RX, 0x0dc,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_UART3_RTS, 0x0e0,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_UART3_CTS, 0x0e0,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPI0_CS0, 0x0e4,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPI0_CS1, 0x0e4,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPI0_SDI, 0x0e8,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPI0_SDO, 0x0e8,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPI0_SCK, 0x0ec,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPI1_CS0, 0x0ec,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPI1_CS1, 0x0f0,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPI1_SDI, 0x0f0,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_SPI1_SDO, 0x0f4,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_SPI1_SCK, 0x0f4,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG0_TDO, 0x0f8,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG0_TCK, 0x0f8,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG0_TDI, 0x0fc,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG0_TMS, 0x0fc,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG0_TRST, 0x100,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG0_SRST, 0x100,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG1_TDO, 0x104,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG1_TCK, 0x104,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG1_TDI, 0x108,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG1_TMS, 0x108,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG1_TRST, 0x10c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG1_SRST, 0x10c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG2_TDO, 0x110,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG2_TCK, 0x110,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG2_TDI, 0x114,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG2_TMS, 0x114,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG2_TRST, 0x118,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG2_SRST, 0x118,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG3_TDO, 0x11c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG3_TCK, 0x11c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG3_TDI, 0x120,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG3_TMS, 0x120,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_JTAG3_TRST, 0x124,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_JTAG3_SRST, 0x124,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO0, 0x128,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO1, 0x128,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO2, 0x12c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO3, 0x12c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO4, 0x130,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO5, 0x130,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO6, 0x134,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO7, 0x134,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO8, 0x138,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO9, 0x138,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO10, 0x13c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO11, 0x13c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO12, 0x140,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO13, 0x140,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO14, 0x144,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO15, 0x144,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO16, 0x148,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO17, 0x148,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO18, 0x14c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO19, 0x14c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO20, 0x150,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO21, 0x150,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO22, 0x154,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO23, 0x154,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO24, 0x158,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO25, 0x158,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO26, 0x15c,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO27, 0x15c,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO28, 0x160,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO29, 0x160,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_GPIO30, 0x164,
			   PIN_FLAG_DEFAULT),
	SG2042_GENERAL_PIN(PIN_GPIO31, 0x164,
			   PIN_FLAG_WRITE_HIGH),
	SG2042_GENERAL_PIN(PIN_MODE_SEL0, 0x168,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_MODE_SEL1, 0x168,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_MODE_SEL2, 0x16c,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL0, 0x16c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL1, 0x170,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL2, 0x170,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL3, 0x174,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL4, 0x174,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL5, 0x178,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL6, 0x178,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BOOT_SEL7, 0x17c,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_MULTI_SCKT, 0x17c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_SCKT_ID0, 0x180,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_SCKT_ID1, 0x180,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_PLL_CLK_IN_MAIN, 0x184,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_PLL_CLK_IN_DDR_0, 0x184,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_PLL_CLK_IN_DDR_1, 0x188,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_PLL_CLK_IN_DDR_2, 0x188,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_PLL_CLK_IN_DDR_3, 0x18c,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_XTAL_32K, 0x18c,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_SYS_RST, 0x190,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_PWR_BUTTON, 0x190,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_TEST_EN, 0x194,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_TEST_MODE_MBIST, 0x194,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_TEST_MODE_SCAN, 0x198,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_TEST_MODE_BSD, 0x198,
			   PIN_FLAG_WRITE_HIGH | PIN_FLAG_NO_PINMUX |
			   PIN_FLAG_NO_OEX_EN),
	SG2042_GENERAL_PIN(PIN_BISR_BYP, 0x19c,
			   PIN_FLAG_NO_PINMUX | PIN_FLAG_NO_OEX_EN),
};

static const struct sophgo_pinctrl_data sg2044_pindata = {
	.pins		= sg2044_pins,
	.pindata	= sg2044_pin_data,
	.vddio_ops	= &sg2044_vddio_cfg_ops,
	.cfg_ops	= &sg2042_cfg_ops,
	.pctl_ops	= &sg2042_pctrl_ops,
	.pmx_ops	= &sg2042_pmx_ops,
	.pconf_ops	= &sg2042_pconf_ops,
	.npins		= ARRAY_SIZE(sg2044_pins),
	.pinsize	= sizeof(struct sg2042_pin),
};

static const struct of_device_id sg2044_pinctrl_ids[] = {
	{ .compatible = "sophgo,sg2044-pinctrl", .data = &sg2044_pindata },
	{ }
};
MODULE_DEVICE_TABLE(of, sg2044_pinctrl_ids);

static struct platform_driver sg2044_pinctrl_driver = {
	.probe	= sophgo_pinctrl_probe,
	.driver	= {
		.name			= "sg2044-pinctrl",
		.suppress_bind_attrs	= true,
		.of_match_table		= sg2044_pinctrl_ids,
	},
};
module_platform_driver(sg2044_pinctrl_driver);

MODULE_DESCRIPTION("Pinctrl driver for the SG2002 series SoC");
MODULE_LICENSE("GPL");
