/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella
 *<Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/


// ./tests/peptidoms/catch2-only-peptidoms [peptidoms] -s


#include <catch2/catch_test_macros.hpp>
#include <catch2/catch_approx.hpp>
#include <catch2/matchers/catch_matchers_vector.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include <QString>
#include "../common.h"
#include "tests/tests-config.h"
#include <pappsomspp/core/amino_acid/aastringcodemassmatching.h>
#include <pappsomspp/core/fasta/fastareader.h>
#include <pappsomspp/core/fasta/fastafileindexer.h>
#include <pappsomspp/core/massspectrum/massspectrum.h>
#include <pappsomspp/core/processing/filters/filterresample.h>
#include <pappsomspp/core/processing/filters/filterpass.h>
#include <pappsomspp/core/processing/filters/filterchargedeconvolution.h>
#include <pappsomspp/core/processing/specpeptidoms/correctiontree.h>
#include <pappsomspp/core/processing/specpeptidoms/locationsaver.h>
#include <pappsomspp/core/processing/specpeptidoms/semiglobalalignment.h>
#include <pappsomspp/core/processing/specpeptidoms/spomsspectrum.h>
#include <pappsomspp/core/processing/specpeptidoms/types.h>
#include <pappsomspp/core/protein/protein.h>

class FastaSeq : public pappso::FastaHandlerInterface
{
  public:
  const QString &
  getDescription() const
  {
    return description;
  };
  const QString &
  getSequence() const
  {
    return sequence;
  };
  void
  setSequence(const QString &description_in, const QString &sequence_in) override
  {
    std::cout << std::endl << "begin description=" << description_in.toStdString();
    std::cout << std::endl << "sequence=" << sequence_in.toStdString();
    std::cout << std::endl << "end" << std::endl;
    description = description_in;
    sequence    = sequence_in;
  };

  private:
  QString description;
  QString sequence;
};

TEST_CASE("test for peptidoms alignment.", "[test_diff_java]")
{
  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  SECTION(
    "..:: BASIC sequence alignment scan_8838 spectrum easy met on fasta file KGIGSGKVLKSGPQ "
    "::..",
    "[peptidoms]")
  {
    AaCode aa_code;
    aa_code.addAaModification('C', AaModification::getInstance("MOD:00397"));

    pappso::specpeptidoms::SpOMSProtein protein(
      "protein_id", "TVEQIYQKGIGSGKVLKSGPQVNHFVPEA", aa_code);

    pappso::PrecisionPtr precision_ptr = pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::QualifiedMassSpectrum spectrum_simple =
      readQualifiedMassSpectrumMgf(QString(CMAKE_SOURCE_DIR).append("/tests/data/scans/8838.mgf"));
    pappso::FilterGreatestY(60).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));

    pappso::specpeptidoms::SpOMSSpectrum experimental_spectrum(
      spectrum_simple, precision_ptr, aa_code);

    std::vector<double> mass_list(experimental_spectrum.getMassList());
    REQUIRE(mass_list.size() == 58);
    const std::vector<double> sp_ref(
      {19.017,   147.076,  186.123,  227.102,  242.112,  244.128,  254.150,  271.214,  284.124,
       299.207,  301.149,  356.229,  371.155,  388.185,  429.247,  443.263,  500.285,  516.279,
       542.331,  599.352,  614.356,  628.378,  629.362,  644.368,  647.372,  670.394,  686.914,
       712.439,  727.448,  728.434,  741.949,  742.450,  757.454,  770.960,  814.476,  827.503,
       840.528,  841.530,  856.521,  913.552,  927.571,  968.622,  1000.575, 1055.656, 1057.604,
       1058.607, 1072.682, 1112.685, 1114.694, 1129.705, 1170.683, 1185.695, 1201.726, 1209.731,
       1213.690, 1226.758, 1227.709, 1355.800});
    REQUIRE_THAT(mass_list, Catch::Matchers::Approx(sp_ref).margin(0.001));
    std::vector<specpeptidoms::AaPosition> aa_pos(
      experimental_spectrum.getAaPositions(aa_code.getAaCode('G')));
    REQUIRE(aa_pos.size() == 12);
    REQUIRE(aa_pos.at(0).l_peak == 0);
    REQUIRE(aa_pos.at(0).r_peak == 2);
    REQUIRE(aa_pos.at(0).l_support == false);
    REQUIRE(aa_pos.at(1).l_peak == 3);
    REQUIRE(aa_pos.at(1).r_peak == 8);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 5);
    REQUIRE(aa_pos.at(2).r_peak == 10);
    REQUIRE(aa_pos.at(2).l_support == true);
    REQUIRE(aa_pos.at(3).l_peak == 9);
    REQUIRE(aa_pos.at(3).r_peak == 11);
    REQUIRE(aa_pos.at(3).l_support == true);
    REQUIRE(aa_pos.at(4).l_peak == 15);
    REQUIRE(aa_pos.at(4).r_peak == 16);
    REQUIRE(aa_pos.at(4).l_support == true);
    REQUIRE(aa_pos.at(5).l_peak == 18);
    REQUIRE(aa_pos.at(5).r_peak == 19);
    REQUIRE(aa_pos.at(5).l_support == true);
    REQUIRE(aa_pos.at(6).l_peak == 32);
    REQUIRE(aa_pos.at(6).r_peak == 34);
    REQUIRE(aa_pos.at(6).l_support == true);
    REQUIRE(aa_pos.at(7).l_peak == 38);
    REQUIRE(aa_pos.at(7).r_peak == 39);
    REQUIRE(aa_pos.at(7).l_support == true);
    REQUIRE(aa_pos.at(8).l_peak == 42);
    REQUIRE(aa_pos.at(8).r_peak == 44);
    REQUIRE(aa_pos.at(8).l_support == true);
    REQUIRE(aa_pos.at(9).l_peak == 43);
    REQUIRE(aa_pos.at(9).r_peak == 47);
    REQUIRE(aa_pos.at(9).l_support == true);
    REQUIRE(aa_pos.at(10).l_peak == 46);
    REQUIRE(aa_pos.at(10).r_peak == 49);
    REQUIRE(aa_pos.at(10).l_support == true);
    REQUIRE(aa_pos.at(11).l_peak == 50);
    REQUIRE(aa_pos.at(11).r_peak == 56);
    REQUIRE(aa_pos.at(11).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('A'));
    REQUIRE(aa_pos.size() == 4);
    REQUIRE(aa_pos.at(0).l_peak == 14);
    REQUIRE(aa_pos.at(0).r_peak == 16);
    REQUIRE(aa_pos.at(0).l_support == true);
    REQUIRE(aa_pos.at(1).l_peak == 19);
    REQUIRE(aa_pos.at(1).r_peak == 25);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 25);
    REQUIRE(aa_pos.at(2).r_peak == 32);
    REQUIRE(aa_pos.at(2).l_support == false);
    REQUIRE(aa_pos.at(3).l_peak == 38);
    REQUIRE(aa_pos.at(3).r_peak == 40);
    REQUIRE(aa_pos.at(3).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('S'));
    REQUIRE(aa_pos.size() == 12);
    REQUIRE(aa_pos.at(0).l_peak == 1);
    REQUIRE(aa_pos.at(0).r_peak == 4);
    REQUIRE(aa_pos.at(0).l_support == false);
    REQUIRE(aa_pos.at(1).l_peak == 8);
    REQUIRE(aa_pos.at(1).r_peak == 12);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 10);
    REQUIRE(aa_pos.at(2).r_peak == 13);
    REQUIRE(aa_pos.at(2).l_support == true);
    REQUIRE(aa_pos.at(3).l_peak == 11);
    REQUIRE(aa_pos.at(3).r_peak == 15);
    REQUIRE(aa_pos.at(3).l_support == true);
    REQUIRE(aa_pos.at(4).l_peak == 14);
    REQUIRE(aa_pos.at(4).r_peak == 17);
    REQUIRE(aa_pos.at(4).l_support == true);
    REQUIRE(aa_pos.at(5).l_peak == 18);
    REQUIRE(aa_pos.at(5).r_peak == 22);
    REQUIRE(aa_pos.at(5).l_support == true);
    REQUIRE(aa_pos.at(6).l_peak == 28);
    REQUIRE(aa_pos.at(6).r_peak == 34);
    REQUIRE(aa_pos.at(6).l_support == true);
    REQUIRE(aa_pos.at(7).l_peak == 36);
    REQUIRE(aa_pos.at(7).r_peak == 40);
    REQUIRE(aa_pos.at(7).l_support == true);
    REQUIRE(aa_pos.at(8).l_peak == 39);
    REQUIRE(aa_pos.at(8).r_peak == 42);
    REQUIRE(aa_pos.at(8).l_support == true);
    REQUIRE(aa_pos.at(9).l_peak == 41);
    REQUIRE(aa_pos.at(9).r_peak == 43);
    REQUIRE(aa_pos.at(9).l_support == true);
    REQUIRE(aa_pos.at(10).l_peak == 41);
    REQUIRE(aa_pos.at(10).r_peak == 46);
    REQUIRE(aa_pos.at(10).l_support == false);
    REQUIRE(aa_pos.at(11).l_peak == 48);
    REQUIRE(aa_pos.at(11).r_peak == 52);
    REQUIRE(aa_pos.at(11).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('P'));
    REQUIRE(aa_pos.size() == 4);
    REQUIRE(aa_pos.at(0).l_peak == 0);
    REQUIRE(aa_pos.at(0).r_peak == 3);
    REQUIRE(aa_pos.at(0).l_support == false);
    REQUIRE(aa_pos.at(1).l_peak == 1);
    REQUIRE(aa_pos.at(1).r_peak == 5);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 47);
    REQUIRE(aa_pos.at(2).r_peak == 53);
    REQUIRE(aa_pos.at(2).l_support == true);
    REQUIRE(aa_pos.at(3).l_peak == 49);
    REQUIRE(aa_pos.at(3).r_peak == 55);
    REQUIRE(aa_pos.at(3).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('V'));
    REQUIRE(aa_pos.size() == 12);
    REQUIRE(aa_pos.at(0).l_peak == 1);
    REQUIRE(aa_pos.at(0).r_peak == 6);
    REQUIRE(aa_pos.at(0).l_support == false);
    REQUIRE(aa_pos.at(1).l_peak == 15);
    REQUIRE(aa_pos.at(1).r_peak == 18);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 16);
    REQUIRE(aa_pos.at(2).r_peak == 19);
    REQUIRE(aa_pos.at(2).l_support == true);
    REQUIRE(aa_pos.at(3).l_peak == 16);
    REQUIRE(aa_pos.at(3).r_peak == 20);
    REQUIRE(aa_pos.at(3).l_support == false);
    REQUIRE(aa_pos.at(4).l_peak == 17);
    REQUIRE(aa_pos.at(4).r_peak == 21);
    REQUIRE(aa_pos.at(4).l_support == false);
    REQUIRE(aa_pos.at(5).l_peak == 21);
    REQUIRE(aa_pos.at(5).r_peak == 28);
    REQUIRE(aa_pos.at(5).l_support == true);
    REQUIRE(aa_pos.at(6).l_peak == 22);
    REQUIRE(aa_pos.at(6).r_peak == 29);
    REQUIRE(aa_pos.at(6).l_support == true);
    REQUIRE(aa_pos.at(7).l_peak == 29);
    REQUIRE(aa_pos.at(7).r_peak == 35);
    REQUIRE(aa_pos.at(7).l_support == true);
    REQUIRE(aa_pos.at(8).l_peak == 31);
    REQUIRE(aa_pos.at(8).r_peak == 37);
    REQUIRE(aa_pos.at(8).l_support == true);
    REQUIRE(aa_pos.at(9).l_peak == 32);
    REQUIRE(aa_pos.at(9).r_peak == 38);
    REQUIRE(aa_pos.at(9).l_support == true);
    REQUIRE(aa_pos.at(10).l_peak == 34);
    REQUIRE(aa_pos.at(10).r_peak == 39);
    REQUIRE(aa_pos.at(10).l_support == true);
    REQUIRE(aa_pos.at(11).l_peak == 46);
    REQUIRE(aa_pos.at(11).r_peak == 52);
    REQUIRE(aa_pos.at(11).l_support == false);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('T'));
    REQUIRE(aa_pos.size() == 2);
    REQUIRE(aa_pos.at(0).l_peak == 18);
    REQUIRE(aa_pos.at(0).r_peak == 26);
    REQUIRE(aa_pos.at(0).l_support == false);
    REQUIRE(aa_pos.at(1).l_peak == 24);
    REQUIRE(aa_pos.at(1).r_peak == 33);
    REQUIRE(aa_pos.at(1).l_support == false);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('I'));
    REQUIRE(aa_pos.size() == 12);
    REQUIRE(aa_pos.at(0).l_peak == 2);
    REQUIRE(aa_pos.at(0).r_peak == 9);
    REQUIRE(aa_pos.at(0).l_support == true);
    REQUIRE(aa_pos.at(1).l_peak == 14);
    REQUIRE(aa_pos.at(1).r_peak == 18);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 16);
    REQUIRE(aa_pos.at(2).r_peak == 21);
    REQUIRE(aa_pos.at(2).l_support == false);
    REQUIRE(aa_pos.at(3).l_peak == 17);
    REQUIRE(aa_pos.at(3).r_peak == 22);
    REQUIRE(aa_pos.at(3).l_support == true);
    REQUIRE(aa_pos.at(4).l_peak == 19);
    REQUIRE(aa_pos.at(4).r_peak == 27);
    REQUIRE(aa_pos.at(4).l_support == true);
    REQUIRE(aa_pos.at(5).l_peak == 20);
    REQUIRE(aa_pos.at(5).r_peak == 28);
    REQUIRE(aa_pos.at(5).l_support == true);
    REQUIRE(aa_pos.at(6).l_peak == 22);
    REQUIRE(aa_pos.at(6).r_peak == 31);
    REQUIRE(aa_pos.at(6).l_support == true);
    REQUIRE(aa_pos.at(7).l_peak == 23);
    REQUIRE(aa_pos.at(7).r_peak == 32);
    REQUIRE(aa_pos.at(7).l_support == true);
    REQUIRE(aa_pos.at(8).l_peak == 28);
    REQUIRE(aa_pos.at(8).r_peak == 36);
    REQUIRE(aa_pos.at(8).l_support == true);
    REQUIRE(aa_pos.at(9).l_peak == 29);
    REQUIRE(aa_pos.at(9).r_peak == 37);
    REQUIRE(aa_pos.at(9).l_support == true);
    REQUIRE(aa_pos.at(10).l_peak == 34);
    REQUIRE(aa_pos.at(10).r_peak == 40);
    REQUIRE(aa_pos.at(10).l_support == true);
    REQUIRE(aa_pos.at(11).l_peak == 44);
    REQUIRE(aa_pos.at(11).r_peak == 50);
    REQUIRE(aa_pos.at(11).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('N'));
    REQUIRE(aa_pos.size() == 0);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('D'));
    REQUIRE(aa_pos.size() == 4);
    REQUIRE(aa_pos.at(0).l_peak == 0);
    REQUIRE(aa_pos.at(0).r_peak == 5);
    REQUIRE(aa_pos.at(0).l_support == false);
    REQUIRE(aa_pos.at(1).l_peak == 2);
    REQUIRE(aa_pos.at(1).r_peak == 10);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 43);
    REQUIRE(aa_pos.at(2).r_peak == 50);
    REQUIRE(aa_pos.at(2).l_support == true);
    REQUIRE(aa_pos.at(3).l_peak == 47);
    REQUIRE(aa_pos.at(3).r_peak == 56);
    REQUIRE(aa_pos.at(3).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('Q'));
    REQUIRE(aa_pos.size() == 3);
    REQUIRE(aa_pos.at(0).l_peak == 0);
    REQUIRE(aa_pos.at(0).r_peak == 1);
    REQUIRE(aa_pos.at(0).l_support == true);
    REQUIRE(aa_pos.at(1).l_peak == 18);
    REQUIRE(aa_pos.at(1).r_peak == 25);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 25);
    REQUIRE(aa_pos.at(2).r_peak == 34);
    REQUIRE(aa_pos.at(2).l_support == false);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('K'));
    REQUIRE(aa_pos.size() == 16);
    REQUIRE(aa_pos.at(0).l_peak == 0);
    REQUIRE(aa_pos.at(0).r_peak == 7);
    REQUIRE(aa_pos.at(0).l_support == false);
    REQUIRE(aa_pos.at(1).l_peak == 1);
    REQUIRE(aa_pos.at(1).r_peak == 9);
    REQUIRE(aa_pos.at(1).l_support == false);
    REQUIRE(aa_pos.at(2).l_peak == 10);
    REQUIRE(aa_pos.at(2).r_peak == 14);
    REQUIRE(aa_pos.at(2).l_support == true);
    REQUIRE(aa_pos.at(3).l_peak == 13);
    REQUIRE(aa_pos.at(3).r_peak == 17);
    REQUIRE(aa_pos.at(3).l_support == true);
    REQUIRE(aa_pos.at(4).l_peak == 16);
    REQUIRE(aa_pos.at(4).r_peak == 21);
    REQUIRE(aa_pos.at(4).l_support == true);
    REQUIRE(aa_pos.at(5).l_peak == 17);
    REQUIRE(aa_pos.at(5).r_peak == 23);
    REQUIRE(aa_pos.at(5).l_support == true);
    REQUIRE(aa_pos.at(6).l_peak == 19);
    REQUIRE(aa_pos.at(6).r_peak == 28);
    REQUIRE(aa_pos.at(6).l_support == true);
    REQUIRE(aa_pos.at(7).l_peak == 20);
    REQUIRE(aa_pos.at(7).r_peak == 31);
    REQUIRE(aa_pos.at(7).l_support == true);
    REQUIRE(aa_pos.at(8).l_peak == 22);
    REQUIRE(aa_pos.at(8).r_peak == 32);
    REQUIRE(aa_pos.at(8).l_support == true);
    REQUIRE(aa_pos.at(9).l_peak == 27);
    REQUIRE(aa_pos.at(9).r_peak == 36);
    REQUIRE(aa_pos.at(9).l_support == true);
    REQUIRE(aa_pos.at(10).l_peak == 29);
    REQUIRE(aa_pos.at(10).r_peak == 38);
    REQUIRE(aa_pos.at(10).l_support == true);
    REQUIRE(aa_pos.at(11).l_peak == 36);
    REQUIRE(aa_pos.at(11).r_peak == 41);
    REQUIRE(aa_pos.at(11).l_support == true);
    REQUIRE(aa_pos.at(12).l_peak == 40);
    REQUIRE(aa_pos.at(12).r_peak == 43);
    REQUIRE(aa_pos.at(12).l_support == true);
    REQUIRE(aa_pos.at(13).l_peak == 44);
    REQUIRE(aa_pos.at(13).r_peak == 51);
    REQUIRE(aa_pos.at(13).l_support == true);
    REQUIRE(aa_pos.at(14).l_peak == 46);
    REQUIRE(aa_pos.at(14).r_peak == 54);
    REQUIRE(aa_pos.at(14).l_support == false);
    REQUIRE(aa_pos.at(15).l_peak == 56);
    REQUIRE(aa_pos.at(15).r_peak == 57);
    REQUIRE(aa_pos.at(15).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('E'));
    REQUIRE(aa_pos.size() == 5);
    REQUIRE(aa_pos.at(0).l_peak == 1);
    REQUIRE(aa_pos.at(0).r_peak == 8);
    REQUIRE(aa_pos.at(0).l_support == false);
    REQUIRE(aa_pos.at(1).l_peak == 4);
    REQUIRE(aa_pos.at(1).r_peak == 12);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 41);
    REQUIRE(aa_pos.at(2).r_peak == 48);
    REQUIRE(aa_pos.at(2).l_support == false);
    REQUIRE(aa_pos.at(3).l_peak == 46);
    REQUIRE(aa_pos.at(3).r_peak == 52);
    REQUIRE(aa_pos.at(3).l_support == true);
    REQUIRE(aa_pos.at(4).l_peak == 55);
    REQUIRE(aa_pos.at(4).r_peak == 57);
    REQUIRE(aa_pos.at(4).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('M'));
    REQUIRE(aa_pos.size() == 2);
    REQUIRE(aa_pos.at(0).l_peak == 8);
    REQUIRE(aa_pos.at(0).r_peak == 14);
    REQUIRE(aa_pos.at(0).l_support == false);
    REQUIRE(aa_pos.at(1).l_peak == 40);
    REQUIRE(aa_pos.at(1).r_peak == 45);
    REQUIRE(aa_pos.at(1).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('H'));
    REQUIRE(aa_pos.size() == 2);
    REQUIRE(aa_pos.at(0).l_peak == 1);
    REQUIRE(aa_pos.at(0).r_peak == 8);
    REQUIRE(aa_pos.at(0).l_support == true);
    REQUIRE(aa_pos.at(1).l_peak == 46);
    REQUIRE(aa_pos.at(1).r_peak == 53);
    REQUIRE(aa_pos.at(1).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('F'));
    REQUIRE(aa_pos.size() == 2);
    REQUIRE(aa_pos.at(0).l_peak == 16);
    REQUIRE(aa_pos.at(0).r_peak == 24);
    REQUIRE(aa_pos.at(0).l_support == true);
    REQUIRE(aa_pos.at(1).l_peak == 26);
    REQUIRE(aa_pos.at(1).r_peak == 38);
    REQUIRE(aa_pos.at(1).l_support == false);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('R'));
    REQUIRE(aa_pos.size() == 6);
    REQUIRE(aa_pos.at(0).l_peak == 0);
    REQUIRE(aa_pos.at(0).r_peak == 9);
    REQUIRE(aa_pos.at(0).l_support == false);
    REQUIRE(aa_pos.at(1).l_peak == 15);
    REQUIRE(aa_pos.at(1).r_peak == 19);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 18);
    REQUIRE(aa_pos.at(2).r_peak == 30);
    REQUIRE(aa_pos.at(2).l_support == false);
    REQUIRE(aa_pos.at(3).l_peak == 20);
    REQUIRE(aa_pos.at(3).r_peak == 33);
    REQUIRE(aa_pos.at(3).l_support == false);
    REQUIRE(aa_pos.at(4).l_peak == 32);
    REQUIRE(aa_pos.at(4).r_peak == 39);
    REQUIRE(aa_pos.at(4).l_support == true);
    REQUIRE(aa_pos.at(5).l_peak == 44);
    REQUIRE(aa_pos.at(5).r_peak == 54);
    REQUIRE(aa_pos.at(5).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('C'));
    REQUIRE(aa_pos.size() == 2);
    REQUIRE(aa_pos.at(0).l_peak == 11);
    REQUIRE(aa_pos.at(0).r_peak == 17);
    REQUIRE(aa_pos.at(0).l_support == true);
    REQUIRE(aa_pos.at(1).l_peak == 36);
    REQUIRE(aa_pos.at(1).r_peak == 42);
    REQUIRE(aa_pos.at(1).l_support == true);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('Y'));
    REQUIRE(aa_pos.size() == 0);
    aa_pos = experimental_spectrum.getAaPositions(aa_code.getAaCode('W'));
    REQUIRE(aa_pos.size() == 3);
    REQUIRE(aa_pos.at(0).l_peak == 15);
    REQUIRE(aa_pos.at(0).r_peak == 22);
    REQUIRE(aa_pos.at(0).l_support == true);
    REQUIRE(aa_pos.at(1).l_peak == 21);
    REQUIRE(aa_pos.at(1).r_peak == 34);
    REQUIRE(aa_pos.at(1).l_support == true);
    REQUIRE(aa_pos.at(2).l_peak == 34);
    REQUIRE(aa_pos.at(2).r_peak == 42);
    REQUIRE(aa_pos.at(2).l_support == true);

    pappso::specpeptidoms::ScoreValues score_values;

    pappso::specpeptidoms::SemiGlobalAlignment semi_global_alignment(
      score_values, precision_ptr, aa_code);

    semi_global_alignment.initFastAlign(experimental_spectrum);

    std::vector<pappso::specpeptidoms::KeyCell> key_cells(experimental_spectrum.size());

    std::vector<std::vector<pappso::specpeptidoms::KeyCell>> key_cells_ref(
      {{{0, 0, 0, 0},   {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{1, 0, 0, 0},   {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {1, -8, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{2, 0, 0, 0},   {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 1},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {1, -8, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 2},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{3, 0, 0, 0},   {0, -19, 0, 0}, {0, -19, 0, 0}, {3, -8, 0, 3},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 1},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {1, -8, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 2},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{4, 0, 0, 0},   {0, -19, 0, 0}, {0, -19, 0, 0}, {3, -8, 0, 3},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {4, -8, 0, 4},  {0, -19, 0, 0}, {2, -8, 0, 1},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {4, -8, 0, 5},  {4, -6, 0, 6},  {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {1, -8, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 2},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {4, -8, 0, 7},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{5, 0, 0, 0},   {0, -19, 0, 0}, {0, -19, 0, 0}, {3, -8, 0, 3},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {4, -8, 0, 4},  {0, -19, 0, 0}, {2, -8, 0, 1},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {4, -8, 0, 5},  {4, -6, 0, 6},  {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {1, -8, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {5, -8, 0, 8},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 2},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {4, -8, 0, 7},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{6, 0, 0, 0},   {0, -19, 0, 0}, {0, -19, 0, 0}, {3, -8, 0, 3},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {4, -8, 0, 4},  {0, -19, 0, 0}, {2, -8, 0, 1},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {4, -8, 0, 5},  {4, -6, 0, 6},  {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {1, -8, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {5, -8, 0, 8},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 2},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {4, -8, 0, 7},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{7, 0, 0, 0},   {0, -19, 0, 0}, {0, -19, 0, 0}, {3, -8, 0, 3},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {4, -8, 0, 4},  {0, -19, 0, 0}, {2, -8, 0, 1},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {4, -8, 0, 5},  {4, -6, 0, 6},  {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {1, -8, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {5, -8, 0, 8},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 2},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {4, -8, 0, 7},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{8, 0, 0, 0},   {0, -19, 0, 0}, {0, -19, 0, 0}, {3, -8, 0, 3},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {8, -8, 1, 9},  {0, -19, 0, 0}, {2, -8, 0, 1},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {8, -8, 0, 10}, {8, -6, 0, 11}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {1, -8, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {5, -8, 0, 8},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 2},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {8, -8, 0, 12}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{9, 0, 0, 0},   {9, 7, 4, 13},  {0, -19, 0, 0}, {3, -8, 0, 3},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {8, -8, 1, 9},  {0, -19, 0, 0}, {2, -8, 0, 1},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {8, -8, 0, 10}, {8, -6, 0, 11}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {9, -8, 0, 14}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {1, -8, 0, 0},  {0, -19, 0, 0}, {9, -8, 0, 15},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {5, -8, 0, 8},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 2},  {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {8, -8, 0, 12}, {0, -19, 0, 0}, {0, -19, 0, 0},
        {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{10, 0, 0, 0},   {9, 7, 4, 13},  {0, -19, 0, 0}, {10, -8, 4, 16}, {0, -19, 0, 0},
        {10, 17, 4, 13}, {8, -8, 1, 9},  {0, -19, 0, 0}, {2, -8, 0, 1},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},  {0, -19, 0, 0},
        {8, -8, 0, 10},  {8, -6, 0, 11}, {0, -19, 0, 0}, {0, -19, 0, 0},  {0, -19, 0, 0},
        {9, -8, 0, 14},  {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {1, -8, 0, 0},  {0, -19, 0, 0},  {9, -8, 0, 15},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0}, {5, -8, 0, 8},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0}, {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0}, {2, -8, 0, 2},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {8, -8, 0, 12}, {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{11, 0, 0, 0},   {9, 7, 4, 13},  {11, -8, 5, 17}, {10, -8, 4, 16}, {0, -19, 0, 0},
        {10, 17, 4, 13}, {8, -8, 1, 9},  {0, -19, 0, 0},  {11, -1, 4, 16}, {0, -19, 0, 0},
        {11, 27, 4, 13}, {0, -19, 0, 0}, {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {8, -8, 0, 10},  {8, -6, 0, 11}, {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {9, -8, 0, 14},  {0, -19, 0, 0}, {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {1, -8, 0, 0},   {0, -19, 0, 0},  {9, -8, 0, 15},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},  {5, -8, 0, 8},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0},  {2, -8, 0, 2},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {8, -8, 0, 12},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0}, {0, -19, 0, 0}},
       {{12, 0, 0, 0},   {9, 7, 4, 13},   {11, -8, 5, 17}, {10, -8, 4, 16}, {12, -8, 5, 18},
        {10, 17, 4, 13}, {8, -8, 1, 9},   {0, -19, 0, 0},  {11, -1, 4, 16}, {0, -19, 0, 0},
        {11, 27, 4, 13}, {0, -19, 0, 0},  {12, 6, 4, 16},  {12, 37, 4, 13}, {0, -19, 0, 0},
        {12, 21, 4, 13}, {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {8, -8, 0, 10},  {8, -6, 0, 11},  {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {9, -8, 0, 14},  {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {1, -8, 0, 0},   {0, -19, 0, 0},  {9, -8, 0, 15},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},  {5, -8, 0, 8},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {12, 21, 4, 13}, {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {12, 19, 4, 13}, {0, -19, 0, 0},  {2, -8, 0, 2},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {8, -8, 0, 12},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0}},
       {{13, 0, 0, 0},   {9, 7, 4, 13},   {11, -8, 5, 17}, {10, -8, 4, 16}, {12, -8, 5, 18},
        {10, 17, 4, 13}, {8, -8, 1, 9},   {13, -8, 6, 19}, {11, -1, 4, 16}, {13, -6, 6, 20},
        {11, 27, 4, 13}, {0, -19, 0, 0},  {12, 6, 4, 16},  {12, 37, 4, 13}, {0, -19, 0, 0},
        {12, 21, 4, 13}, {0, -19, 0, 0},  {13, 47, 4, 13}, {0, -19, 0, 0},  {0, -19, 0, 0},
        {8, -8, 0, 10},  {8, -6, 0, 11},  {0, -19, 0, 0},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},  {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {13, 29, 4, 13}, {0, -19, 0, 0},  {9, -8, 0, 15},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},  {5, -8, 0, 8},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {12, 21, 4, 13}, {13, 31, 4, 13}, {0, -19, 0, 0},
        {0, -19, 0, 0},  {12, 19, 4, 13}, {0, -19, 0, 0},  {2, -8, 0, 2},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {8, -8, 0, 12},  {0, -19, 0, 0},  {13, 29, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0}},
       {{14, 0, 0, 0},   {9, 7, 4, 13},   {11, -8, 5, 17}, {10, -8, 4, 16}, {12, -8, 5, 18},
        {10, 17, 4, 13}, {8, -8, 1, 9},   {13, -8, 6, 19}, {11, -1, 4, 16}, {13, -6, 6, 20},
        {11, 27, 4, 13}, {0, -19, 0, 0},  {12, 6, 4, 16},  {12, 37, 4, 13}, {0, -19, 0, 0},
        {12, 21, 4, 13}, {0, -19, 0, 0},  {13, 47, 4, 13}, {14, 25, 4, 13}, {0, -19, 0, 0},
        {8, -8, 0, 10},  {14, 27, 4, 13}, {14, 57, 4, 13}, {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},  {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {14, 39, 4, 13}, {0, -19, 0, 0},  {9, -8, 0, 15},
        {0, -19, 0, 0},  {14, 41, 4, 13}, {0, -19, 0, 0},  {5, -8, 0, 8},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {12, 21, 4, 13}, {13, 31, 4, 13}, {0, -19, 0, 0},
        {0, -19, 0, 0},  {12, 19, 4, 13}, {0, -19, 0, 0},  {2, -8, 0, 2},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {8, -8, 0, 12},  {0, -19, 0, 0},  {13, 29, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0}},
       {{15, 0, 0, 0},   {9, 7, 4, 13},   {11, -8, 5, 17}, {10, -8, 4, 16}, {12, -8, 5, 18},
        {10, 17, 4, 13}, {15, -8, 8, 21}, {13, -8, 6, 19}, {11, -1, 4, 16}, {13, -6, 6, 20},
        {11, 27, 4, 13}, {0, -19, 0, 0},  {12, 6, 4, 16},  {12, 37, 4, 13}, {0, -19, 0, 0},
        {12, 21, 4, 13}, {0, -19, 0, 0},  {13, 47, 4, 13}, {14, 25, 4, 13}, {0, -19, 0, 0},
        {15, 21, 4, 13}, {15, 37, 4, 13}, {14, 57, 4, 13}, {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},  {0, -19, 0, 0},  {0, -19, 0, 0},  {15, 37, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {14, 39, 4, 13}, {0, -19, 0, 0},  {9, -8, 0, 15},
        {0, -19, 0, 0},  {14, 41, 4, 13}, {15, 49, 4, 13}, {15, 49, 4, 13}, {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {12, 21, 4, 13}, {13, 31, 4, 13}, {0, -19, 0, 0},
        {0, -19, 0, 0},  {12, 19, 4, 13}, {0, -19, 0, 0},  {2, -8, 0, 2},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {15, 49, 4, 13}, {0, -19, 0, 0},  {13, 29, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0}},
       {{16, 0, 0, 0},   {9, 7, 4, 13},   {11, -8, 5, 17}, {10, -8, 4, 16}, {12, -8, 5, 18},
        {10, 17, 4, 13}, {15, -8, 8, 21}, {16, -8, 9, 22}, {11, -1, 4, 16}, {16, -6, 9, 23},
        {11, 27, 4, 13}, {0, -19, 0, 0},  {12, 6, 4, 16},  {12, 37, 4, 13}, {0, -19, 0, 0},
        {12, 21, 4, 13}, {0, -19, 0, 0},  {13, 47, 4, 13}, {14, 25, 4, 13}, {0, -19, 0, 0},
        {15, 21, 4, 13}, {15, 37, 4, 13}, {14, 57, 4, 13}, {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},  {0, -19, 0, 0},  {0, -19, 0, 0},  {15, 37, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},  {16, 31, 4, 13}, {14, 39, 4, 13}, {0, -19, 0, 0},  {9, -8, 0, 15},
        {0, -19, 0, 0},  {14, 41, 4, 13}, {15, 49, 4, 13}, {16, 74, 4, 13}, {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {12, 21, 4, 13}, {13, 31, 4, 13}, {0, -19, 0, 0},
        {0, -19, 0, 0},  {12, 19, 4, 13}, {0, -19, 0, 0},  {2, -8, 0, 2},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {15, 49, 4, 13}, {0, -19, 0, 0},  {16, 59, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0}},
       {{17, 0, 0, 0},   {9, 7, 4, 13},   {17, -8, 11, 24}, {10, -8, 4, 16}, {12, -8, 5, 18},
        {10, 17, 4, 13}, {15, -8, 8, 21}, {16, -8, 9, 22},  {11, -1, 4, 16}, {16, -6, 9, 23},
        {11, 27, 4, 13}, {17, 4, 9, 23},  {12, 6, 4, 16},   {12, 37, 4, 13}, {0, -19, 0, 0},
        {12, 21, 4, 13}, {0, -19, 0, 0},  {13, 47, 4, 13},  {14, 25, 4, 13}, {0, -19, 0, 0},
        {15, 21, 4, 13}, {15, 37, 4, 13}, {14, 57, 4, 13},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},  {0, -19, 0, 0},  {0, -19, 0, 0},   {15, 37, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},  {16, 31, 4, 13}, {14, 39, 4, 13},  {0, -19, 0, 0},  {17, 55, 4, 13},
        {0, -19, 0, 0},  {14, 41, 4, 13}, {15, 49, 4, 13},  {16, 74, 4, 13}, {17, 84, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {12, 21, 4, 13},  {13, 31, 4, 13}, {0, -19, 0, 0},
        {0, -19, 0, 0},  {12, 19, 4, 13}, {17, 68, 4, 13},  {2, -8, 0, 2},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {15, 49, 4, 13},  {0, -19, 0, 0},  {16, 59, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0}},
       {{18, 0, 0, 0},   {9, 7, 4, 13},   {17, -8, 11, 24}, {10, -8, 4, 16}, {18, -8, 11, 25},
        {10, 17, 4, 13}, {15, -8, 8, 21}, {16, -8, 9, 22},  {11, -1, 4, 16}, {16, -6, 9, 23},
        {11, 27, 4, 13}, {17, 4, 9, 23},  {18, -8, 9, 26},  {12, 37, 4, 13}, {0, -19, 0, 0},
        {18, 14, 9, 23}, {0, -19, 0, 0},  {13, 47, 4, 13},  {14, 25, 4, 13}, {0, -19, 0, 0},
        {15, 21, 4, 13}, {15, 37, 4, 13}, {14, 57, 4, 13},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},  {0, -19, 0, 0},  {0, -19, 0, 0},   {15, 37, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},  {16, 31, 4, 13}, {14, 39, 4, 13},  {0, -19, 0, 0},  {17, 55, 4, 13},
        {0, -19, 0, 0},  {14, 41, 4, 13}, {15, 49, 4, 13},  {16, 74, 4, 13}, {17, 84, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {18, 94, 4, 13},  {13, 31, 4, 13}, {0, -19, 0, 0},
        {0, -19, 0, 0},  {18, 76, 4, 13}, {17, 68, 4, 13},  {2, -8, 0, 2},   {0, -19, 0, 0},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {15, 49, 4, 13},  {0, -19, 0, 0},  {16, 59, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0}},
       {{19, 0, 0, 0},   {9, 7, 4, 13},   {19, -8, 13, 27}, {10, -8, 4, 16}, {18, -8, 11, 25},
        {10, 17, 4, 13}, {15, -8, 8, 21}, {16, -8, 9, 22},  {11, -1, 4, 16}, {16, -6, 9, 23},
        {11, 27, 4, 13}, {17, 4, 9, 23},  {18, -8, 9, 26},  {12, 37, 4, 13}, {0, -19, 0, 0},
        {18, 14, 9, 23}, {19, 21, 9, 23}, {13, 47, 4, 13},  {14, 25, 4, 13}, {0, -19, 0, 0},
        {15, 21, 4, 13}, {15, 37, 4, 13}, {14, 57, 4, 13},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},  {0, -19, 0, 0},  {0, -19, 0, 0},   {15, 37, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},  {16, 31, 4, 13}, {14, 39, 4, 13},  {0, -19, 0, 0},  {17, 55, 4, 13},
        {0, -19, 0, 0},  {14, 41, 4, 13}, {15, 49, 4, 13},  {16, 74, 4, 13}, {17, 84, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {18, 94, 4, 13},  {13, 31, 4, 13}, {19, 104, 4, 13},
        {0, -19, 0, 0},  {18, 76, 4, 13}, {19, 88, 4, 13},  {2, -8, 0, 2},   {19, 86, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {15, 49, 4, 13},  {0, -19, 0, 0},  {16, 59, 4, 13},
        {0, -19, 0, 0},  {0, -19, 0, 0},  {0, -19, 0, 0}},
       {{20, 0, 0, 0},    {9, 7, 4, 13},   {19, -8, 13, 27}, {10, -8, 4, 16}, {18, -8, 11, 25},
        {10, 17, 4, 13},  {15, -8, 8, 21}, {16, -8, 9, 22},  {11, -1, 4, 16}, {20, 2, 13, 27},
        {11, 27, 4, 13},  {17, 4, 9, 23},  {18, -8, 9, 26},  {12, 37, 4, 13}, {0, -19, 0, 0},
        {18, 14, 9, 23},  {19, 21, 9, 23}, {13, 47, 4, 13},  {14, 25, 4, 13}, {0, -19, 0, 0},
        {15, 21, 4, 13},  {15, 37, 4, 13}, {14, 57, 4, 13},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},   {0, -19, 0, 0},  {20, 15, 4, 13},  {15, 37, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},   {16, 31, 4, 13}, {14, 39, 4, 13},  {0, -19, 0, 0},  {17, 55, 4, 13},
        {0, -19, 0, 0},   {14, 41, 4, 13}, {15, 49, 4, 13},  {16, 74, 4, 13}, {17, 84, 4, 13},
        {20, 45, 4, 13},  {0, -19, 0, 0},  {18, 94, 4, 13},  {13, 31, 4, 13}, {19, 104, 4, 13},
        {0, -19, 0, 0},   {18, 76, 4, 13}, {19, 88, 4, 13},  {2, -8, 0, 2},   {19, 86, 4, 13},
        {20, 111, 4, 13}, {0, -19, 0, 0},  {15, 49, 4, 13},  {0, -19, 0, 0},  {16, 59, 4, 13},
        {0, -19, 0, 0},   {0, -19, 0, 0},  {0, -19, 0, 0}},
       {{21, 0, 0, 0},    {9, 7, 4, 13},    {21, -8, 15, 28}, {10, -8, 4, 16}, {18, -8, 11, 25},
        {10, 17, 4, 13},  {15, -8, 8, 21},  {16, -8, 9, 22},  {11, -1, 4, 16}, {20, 2, 13, 27},
        {11, 27, 4, 13},  {21, 12, 13, 27}, {18, -8, 9, 26},  {12, 37, 4, 13}, {0, -19, 0, 0},
        {18, 14, 9, 23},  {19, 21, 9, 23},  {13, 47, 4, 13},  {14, 25, 4, 13}, {21, 11, 4, 13},
        {15, 21, 4, 13},  {15, 37, 4, 13},  {14, 57, 4, 13},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},   {0, -19, 0, 0},   {20, 15, 4, 13},  {15, 37, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},   {16, 31, 4, 13},  {14, 39, 4, 13},  {0, -19, 0, 0},  {17, 55, 4, 13},
        {0, -19, 0, 0},   {14, 41, 4, 13},  {15, 49, 4, 13},  {16, 74, 4, 13}, {17, 84, 4, 13},
        {20, 45, 4, 13},  {0, -19, 0, 0},   {18, 94, 4, 13},  {13, 31, 4, 13}, {19, 104, 4, 13},
        {0, -19, 0, 0},   {18, 76, 4, 13},  {19, 88, 4, 13},  {2, -8, 0, 2},   {19, 86, 4, 13},
        {20, 111, 4, 13}, {0, -19, 0, 0},   {15, 49, 4, 13},  {0, -19, 0, 0},  {16, 59, 4, 13},
        {0, -19, 0, 0},   {21, 121, 4, 13}, {0, -19, 0, 0}},
       {{22, 0, 0, 0},    {9, 7, 4, 13},    {21, -8, 15, 28}, {10, -8, 4, 16}, {18, -8, 11, 25},
        {10, 17, 4, 13},  {15, -8, 8, 21},  {22, -8, 15, 29}, {11, -1, 4, 16}, {20, 2, 13, 27},
        {11, 27, 4, 13},  {21, 12, 13, 27}, {18, -8, 9, 26},  {12, 37, 4, 13}, {22, -6, 13, 30},
        {18, 14, 9, 23},  {19, 21, 9, 23},  {13, 47, 4, 13},  {14, 25, 4, 13}, {21, 11, 4, 13},
        {15, 21, 4, 13},  {15, 37, 4, 13},  {14, 57, 4, 13},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},   {0, -19, 0, 0},   {20, 15, 4, 13},  {22, 21, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},   {16, 31, 4, 13},  {14, 39, 4, 13},  {0, -19, 0, 0},  {17, 55, 4, 13},
        {0, -19, 0, 0},   {14, 41, 4, 13},  {15, 49, 4, 13},  {16, 74, 4, 13}, {17, 84, 4, 13},
        {20, 45, 4, 13},  {0, -19, 0, 0},   {18, 94, 4, 13},  {13, 31, 4, 13}, {19, 104, 4, 13},
        {0, -19, 0, 0},   {18, 76, 4, 13},  {19, 88, 4, 13},  {2, -8, 0, 2},   {19, 86, 4, 13},
        {20, 111, 4, 13}, {22, 88, 4, 13},  {15, 49, 4, 13},  {0, -19, 0, 0},  {22, 88, 4, 13},
        {0, -19, 0, 0},   {21, 121, 4, 13}, {22, 128, 4, 13}},
       {{23, 0, 0, 0},    {23, 7, 18, 31},  {21, -8, 15, 28}, {10, -8, 4, 16}, {18, -8, 11, 25},
        {10, 17, 4, 13},  {15, -8, 8, 21},  {22, -8, 15, 29}, {11, -1, 4, 16}, {20, 2, 13, 27},
        {11, 27, 4, 13},  {21, 12, 13, 27}, {18, -8, 9, 26},  {12, 37, 4, 13}, {22, -6, 13, 30},
        {18, 14, 9, 23},  {19, 21, 9, 23},  {13, 47, 4, 13},  {14, 25, 4, 13}, {21, 11, 4, 13},
        {15, 21, 4, 13},  {15, 37, 4, 13},  {14, 57, 4, 13},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},   {0, -19, 0, 0},   {20, 15, 4, 13},  {22, 21, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},   {16, 31, 4, 13},  {14, 39, 4, 13},  {0, -19, 0, 0},  {17, 55, 4, 13},
        {0, -19, 0, 0},   {14, 41, 4, 13},  {15, 49, 4, 13},  {16, 74, 4, 13}, {17, 84, 4, 13},
        {20, 45, 4, 13},  {0, -19, 0, 0},   {18, 94, 4, 13},  {13, 31, 4, 13}, {19, 104, 4, 13},
        {0, -19, 0, 0},   {18, 76, 4, 13},  {19, 88, 4, 13},  {2, -8, 0, 2},   {19, 86, 4, 13},
        {20, 111, 4, 13}, {22, 88, 4, 13},  {15, 49, 4, 13},  {0, -19, 0, 0},  {22, 88, 4, 13},
        {0, -19, 0, 0},   {21, 121, 4, 13}, {22, 128, 4, 13}},
       {{24, 0, 0, 0},    {23, 7, 18, 31},  {21, -8, 15, 28}, {10, -8, 4, 16}, {18, -8, 11, 25},
        {10, 17, 4, 13},  {15, -8, 8, 21},  {22, -8, 15, 29}, {11, -1, 4, 16}, {20, 2, 13, 27},
        {11, 27, 4, 13},  {21, 12, 13, 27}, {18, -8, 9, 26},  {12, 37, 4, 13}, {22, -6, 13, 30},
        {18, 14, 9, 23},  {19, 21, 9, 23},  {13, 47, 4, 13},  {14, 25, 4, 13}, {21, 11, 4, 13},
        {15, 21, 4, 13},  {15, 37, 4, 13},  {14, 57, 4, 13},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},   {0, -19, 0, 0},   {20, 15, 4, 13},  {22, 21, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},   {16, 31, 4, 13},  {14, 39, 4, 13},  {0, -19, 0, 0},  {17, 55, 4, 13},
        {0, -19, 0, 0},   {14, 41, 4, 13},  {15, 49, 4, 13},  {16, 74, 4, 13}, {17, 84, 4, 13},
        {20, 45, 4, 13},  {0, -19, 0, 0},   {18, 94, 4, 13},  {13, 31, 4, 13}, {19, 104, 4, 13},
        {0, -19, 0, 0},   {18, 76, 4, 13},  {19, 88, 4, 13},  {2, -8, 0, 2},   {19, 86, 4, 13},
        {20, 111, 4, 13}, {22, 88, 4, 13},  {15, 49, 4, 13},  {0, -19, 0, 0},  {22, 88, 4, 13},
        {0, -19, 0, 0},   {21, 121, 4, 13}, {22, 128, 4, 13}},
       {{25, 0, 0, 0},    {23, 7, 18, 31},  {21, -8, 15, 28}, {10, -8, 4, 16}, {18, -8, 11, 25},
        {10, 17, 4, 13},  {15, -8, 8, 21},  {22, -8, 15, 29}, {11, -1, 4, 16}, {20, 2, 13, 27},
        {11, 27, 4, 13},  {21, 12, 13, 27}, {18, -8, 9, 26},  {12, 37, 4, 13}, {22, -6, 13, 30},
        {18, 14, 9, 23},  {19, 21, 9, 23},  {13, 47, 4, 13},  {14, 25, 4, 13}, {21, 11, 4, 13},
        {15, 21, 4, 13},  {15, 37, 4, 13},  {14, 57, 4, 13},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {9, -8, 0, 14},   {0, -19, 0, 0},   {20, 15, 4, 13},  {22, 21, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},   {16, 31, 4, 13},  {14, 39, 4, 13},  {0, -19, 0, 0},  {17, 55, 4, 13},
        {0, -19, 0, 0},   {14, 41, 4, 13},  {15, 49, 4, 13},  {16, 74, 4, 13}, {17, 84, 4, 13},
        {20, 45, 4, 13},  {0, -19, 0, 0},   {18, 94, 4, 13},  {13, 31, 4, 13}, {19, 104, 4, 13},
        {0, -19, 0, 0},   {18, 76, 4, 13},  {19, 88, 4, 13},  {2, -8, 0, 2},   {19, 86, 4, 13},
        {20, 111, 4, 13}, {22, 88, 4, 13},  {15, 49, 4, 13},  {0, -19, 0, 0},  {22, 88, 4, 13},
        {0, -19, 0, 0},   {21, 121, 4, 13}, {22, 128, 4, 13}},
       {{26, 0, 0, 0},    {26, 7, 21, 32},  {21, -8, 15, 28}, {10, -8, 4, 16}, {18, -8, 11, 25},
        {10, 17, 4, 13},  {15, -8, 8, 21},  {22, -8, 15, 29}, {11, -1, 4, 16}, {20, 2, 13, 27},
        {11, 27, 4, 13},  {21, 12, 13, 27}, {18, -8, 9, 26},  {12, 37, 4, 13}, {22, -6, 13, 30},
        {18, 14, 9, 23},  {19, 21, 9, 23},  {13, 47, 4, 13},  {14, 25, 4, 13}, {21, 11, 4, 13},
        {15, 21, 4, 13},  {15, 37, 4, 13},  {14, 57, 4, 13},  {13, 29, 4, 13}, {0, -19, 0, 0},
        {26, -8, 12, 33}, {0, -19, 0, 0},   {20, 15, 4, 13},  {22, 21, 4, 13}, {15, 67, 4, 13},
        {0, -19, 0, 0},   {16, 31, 4, 13},  {14, 39, 4, 13},  {0, -19, 0, 0},  {17, 55, 4, 13},
        {0, -19, 0, 0},   {14, 41, 4, 13},  {15, 49, 4, 13},  {16, 74, 4, 13}, {17, 84, 4, 13},
        {20, 45, 4, 13},  {0, -19, 0, 0},   {18, 94, 4, 13},  {13, 31, 4, 13}, {19, 104, 4, 13},
        {0, -19, 0, 0},   {18, 76, 4, 13},  {19, 88, 4, 13},  {2, -8, 0, 2},   {19, 86, 4, 13},
        {20, 111, 4, 13}, {22, 88, 4, 13},  {15, 49, 4, 13},  {0, -19, 0, 0},  {22, 88, 4, 13},
        {0, -19, 0, 0},   {21, 121, 4, 13}, {22, 128, 4, 13}},
       {{27, 0, 0, 0},    {26, 7, 21, 32},  {21, -8, 15, 28}, {10, -8, 4, 16},  {18, -8, 11, 25},
        {10, 17, 4, 13},  {15, -8, 8, 21},  {22, -8, 15, 29}, {27, -1, 21, 32}, {20, 2, 13, 27},
        {11, 27, 4, 13},  {21, 12, 13, 27}, {18, -8, 9, 26},  {12, 37, 4, 13},  {22, -6, 13, 30},
        {18, 14, 9, 23},  {19, 21, 9, 23},  {13, 47, 4, 13},  {14, 25, 4, 13},  {21, 11, 4, 13},
        {15, 21, 4, 13},  {15, 37, 4, 13},  {14, 57, 4, 13},  {13, 29, 4, 13},  {0, -19, 0, 0},
        {26, -8, 12, 33}, {0, -19, 0, 0},   {20, 15, 4, 13},  {22, 21, 4, 13},  {15, 67, 4, 13},
        {0, -19, 0, 0},   {16, 31, 4, 13},  {14, 39, 4, 13},  {0, -19, 0, 0},   {17, 55, 4, 13},
        {0, -19, 0, 0},   {14, 41, 4, 13},  {15, 49, 4, 13},  {16, 74, 4, 13},  {17, 84, 4, 13},
        {20, 45, 4, 13},  {0, -19, 0, 0},   {18, 94, 4, 13},  {13, 31, 4, 13},  {19, 104, 4, 13},
        {0, -19, 0, 0},   {18, 76, 4, 13},  {19, 88, 4, 13},  {27, 40, 4, 13},  {19, 86, 4, 13},
        {20, 111, 4, 13}, {22, 88, 4, 13},  {15, 49, 4, 13},  {0, -19, 0, 0},   {22, 88, 4, 13},
        {0, -19, 0, 0},   {21, 121, 4, 13}, {22, 128, 4, 13}},
       {{28, 0, 0, 0},    {26, 7, 21, 32},  {21, -8, 15, 28}, {10, -8, 4, 16},  {18, -8, 11, 25},
        {10, 17, 4, 13},  {28, -5, 21, 32}, {22, -8, 15, 29}, {27, -1, 21, 32}, {20, 2, 13, 27},
        {11, 27, 4, 13},  {21, 12, 13, 27}, {18, -8, 9, 26},  {12, 37, 4, 13},  {22, -6, 13, 30},
        {18, 14, 9, 23},  {19, 21, 9, 23},  {13, 47, 4, 13},  {14, 25, 4, 13},  {21, 11, 4, 13},
        {28, -5, 21, 32}, {28, -3, 21, 32}, {14, 57, 4, 13},  {13, 29, 4, 13},  {0, -19, 0, 0},
        {26, -8, 12, 33}, {0, -19, 0, 0},   {20, 15, 4, 13},  {22, 21, 4, 13},  {15, 67, 4, 13},
        {0, -19, 0, 0},   {16, 31, 4, 13},  {14, 39, 4, 13},  {0, -19, 0, 0},   {17, 55, 4, 13},
        {0, -19, 0, 0},   {14, 41, 4, 13},  {15, 49, 4, 13},  {16, 74, 4, 13},  {17, 84, 4, 13},
        {20, 45, 4, 13},  {0, -19, 0, 0},   {18, 94, 4, 13},  {13, 31, 4, 13},  {19, 104, 4, 13},
        {0, -19, 0, 0},   {18, 76, 4, 13},  {19, 88, 4, 13},  {27, 40, 4, 13},  {19, 86, 4, 13},
        {20, 111, 4, 13}, {22, 88, 4, 13},  {28, 64, 4, 13},  {0, -19, 0, 0},   {22, 88, 4, 13},
        {0, -19, 0, 0},   {21, 121, 4, 13}, {22, 128, 4, 13}}});

    // TODO Ajouter structure stockant les différents états des cases clés
    // Vérifier état après chaque étape.
    key_cells =
      semi_global_alignment.oneAlignStep(protein,
                                         1,
                                         experimental_spectrum.getAaPositions(protein.at(0).code),
                                         experimental_spectrum,
                                         true,
                                         &protein);

    std::vector<pappso::specpeptidoms::Location> locations =
      semi_global_alignment.getLocationSaver().getLocations();
    REQUIRE(locations.size() == 1);
    REQUIRE(locations.at(0).score == 17);
    REQUIRE(locations.at(0).beginning == 4);
    REQUIRE(locations.at(0).length == 32);

    for(auto loc : locations)
      {
        semi_global_alignment.preciseAlign(
          experimental_spectrum, &protein, loc.beginning, loc.length);
        pappso::specpeptidoms::Alignment best_alignment = semi_global_alignment.getBestAlignment();

        REQUIRE(best_alignment.m_peptideModel.toInterpretation().toStdString() == "KGIGSGKVLKSGPQ");
        REQUIRE(best_alignment.SPC == 25);
        REQUIRE_THAT(best_alignment.peaks,
                     Catch::Matchers::Approx(std::vector<std::size_t>(
                       {0, 4, 12, 21, 27, 37, 47, 55, 66, 69, 76, 82, 94, 103, 106})));
        REQUIRE_THAT(best_alignment.shifts,
                     Catch::Matchers::Approx(std::vector<double>({})).margin(0.001));
        REQUIRE(best_alignment.score == 131);
        REQUIRE(best_alignment.beginning == 7);
        REQUIRE(best_alignment.end == 20);
        REQUIRE(best_alignment.begin_shift == 0.0);
        REQUIRE(best_alignment.end_shift == 0);
        REQUIRE((best_alignment.begin_shift > 0 || best_alignment.end_shift > 0 ||
                 best_alignment.shifts.size() > 0) == false);

        REQUIRE(best_alignment.getNonAlignedMass() == Catch::Approx(0));
        REQUIRE_THAT(best_alignment.m_peptideModel.getNonAlignedMass(),
                     Catch::Matchers::WithinAbs(0, 0.02));
      }
  }

  SECTION(
    "..:: BASIC sequence alignment scan_13970 spectrum easy met on fasta file "
    "RDYVQ[K][-128,10]AQTKEQ "
    "::..",
    "[peptidoms]")
  {
    AaCode aa_code;
    aa_code.addAaModification('C', AaModification::getInstance("MOD:00397"));

    pappso::specpeptidoms::SpOMSProtein protein(
      "protein_id", "QDTKALHQIKVGRDYVQKAQTKEQPHVARALT", aa_code);

    pappso::PrecisionPtr precision_ptr = pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::QualifiedMassSpectrum spectrum_simple =
      readQualifiedMassSpectrumMgf(QString(CMAKE_SOURCE_DIR).append("/tests/data/scans/13970.mgf"));
    pappso::FilterChargeDeconvolution(precision_ptr)
      .filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterResampleKeepGreater(150).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterGreatestY(120).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));

    pappso::specpeptidoms::SpOMSSpectrum experimental_spectrum(
      spectrum_simple, precision_ptr, aa_code);


    pappso::specpeptidoms::ScoreValues score_values;

    pappso::specpeptidoms::SemiGlobalAlignment semi_global_alignment(
      score_values, precision_ptr, aa_code);

    semi_global_alignment.fastAlign(experimental_spectrum, &protein);
    std::vector<pappso::specpeptidoms::Location> locations =
      semi_global_alignment.getLocationSaver().getLocations();
    REQUIRE(locations.size() == 1);
    REQUIRE(locations.at(0).score == 51);
    REQUIRE(locations.at(0).beginning == 4);
    REQUIRE(locations.at(0).length == 28);

    for(auto loc : locations)
      {
        semi_global_alignment.preciseAlign(
          experimental_spectrum, &protein, loc.beginning, loc.length);
        pappso::specpeptidoms::Alignment best_alignment = semi_global_alignment.getBestAlignment();

        REQUIRE(best_alignment.m_peptideModel.toInterpretation().toStdString() ==
                "RDYVQ[K][-128.095]AQTKEQ");
        REQUIRE(best_alignment.SPC == 16);
        REQUIRE_THAT(best_alignment.peaks,
                     Catch::Matchers::Approx(
                       std::vector<std::size_t>({0, 1, 14, 22, 30, 42, 54, 69, 75, 87, 101, 107})));
        REQUIRE_THAT(best_alignment.shifts,
                     Catch::Matchers::Approx(std::vector<double>({})).margin(0.001));
        REQUIRE(best_alignment.score == 75);
        REQUIRE(best_alignment.beginning == 12);
        REQUIRE(best_alignment.end == 23);
        REQUIRE(best_alignment.begin_shift == 0.0);
        REQUIRE(best_alignment.end_shift == 0);
        REQUIRE((best_alignment.begin_shift > 0 || best_alignment.end_shift > 0 ||
                 best_alignment.shifts.size() > 0) == false);

        REQUIRE(best_alignment.getNonAlignedMass() == Catch::Approx(0));
        REQUIRE_THAT(best_alignment.m_peptideModel.getNonAlignedMass(),
                     Catch::Matchers::WithinAbs(0, 0.02));
      }
  }

  SECTION(
    "..:: BASIC sequence alignment scan_24160 spectrum easy met on fasta file ENV[2,08]YGLEENAQSRQ "
    "::..",
    "[peptidoms]")
  {
    AaCode aa_code;
    aa_code.addAaModification('C', AaModification::getInstance("MOD:00397"));

    pappso::specpeptidoms::SpOMSProtein protein(
      "protein_id", "EVNTRRAQENVYGLEENAQSRQDQHFVKIQ", aa_code);

    pappso::PrecisionPtr precision_ptr = pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::QualifiedMassSpectrum spectrum_simple =
      readQualifiedMassSpectrumMgf(QString(CMAKE_SOURCE_DIR).append("/tests/data/scans/24160.mgf"));
    pappso::FilterGreatestY(60).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));

    pappso::specpeptidoms::SpOMSSpectrum experimental_spectrum(
      spectrum_simple, precision_ptr, aa_code);


    pappso::specpeptidoms::ScoreValues score_values;

    pappso::specpeptidoms::SemiGlobalAlignment semi_global_alignment(
      score_values, precision_ptr, aa_code);

    semi_global_alignment.fastAlign(experimental_spectrum, &protein);
    std::vector<pappso::specpeptidoms::Location> locations =
      semi_global_alignment.getLocationSaver().getLocations();
    REQUIRE(locations.size() == 1);
    REQUIRE(locations.at(0).score == 20);
    REQUIRE(locations.at(0).beginning == 0);
    REQUIRE(locations.at(0).length == 33);

    for(auto loc : locations)
      {
        semi_global_alignment.preciseAlign(
          experimental_spectrum, &protein, loc.beginning, loc.length);
        pappso::specpeptidoms::Alignment best_alignment = semi_global_alignment.getBestAlignment();

        REQUIRE(best_alignment.m_peptideModel.toInterpretation().toStdString() ==
                "ENV[2.084]YGLEENAQSRQ");
        // REQUIRE(best_alignment.SPC == 16);
        // REQUIRE_THAT(best_alignment.peaks,
        //              Catch::Matchers::Approx(std::vector<std::size_t>(
        //                {0, 1, 14, 22, 30, 42, 54, 69, 75, 87, 101, 107})));
        // REQUIRE_THAT(best_alignment.shifts,
        //              Catch::Matchers::Approx(std::vector<double>({})).margin(0.001));
        // REQUIRE(best_alignment.score == 75);
        // REQUIRE(best_alignment.beginning == 12);
        // REQUIRE(best_alignment.end == 24);
        // REQUIRE(best_alignment.begin_shift == 0.0);
        // REQUIRE(best_alignment.end_shift == 0);
        // REQUIRE((best_alignment.begin_shift > 0 || best_alignment.end_shift > 0 ||
        //          best_alignment.shifts.size() > 0) == false);

        // REQUIRE(best_alignment.getNonAlignedMass() == Catch::Approx(0));
        // REQUIRE_THAT(best_alignment.m_peptideModel.getNonAlignedMass(),
        //              Catch::Matchers::WithinAbs(0, 0.02));
        // qDebug() << "SPC" << best_alignment.SPC;
        // qDebug() << best_alignment.peaks;
        // qDebug() << best_alignment.shifts;
        // qDebug() << best_alignment.score;
        // qDebug() << best_alignment.beginning;
        // qDebug() << best_alignment.end;
        // qDebug() << best_alignment.begin_shift;
        // qDebug() << best_alignment.end_shift;
        // qDebug() << best_alignment.getNonAlignedMass();
        // std::cin.get();
      }
  }

  SECTION(
    "..:: BASIC sequence alignment scan_25976 spectrum easy met on fasta file [244,07]EEESDKAALLQ "
    "::..",
    "[peptidoms]")
  {
    AaCode aa_code;
    aa_code.addAaModification('C', AaModification::getInstance("MOD:00397"));

    pappso::specpeptidoms::SpOMSProtein protein(
      "protein_id", "TAAESAEQDEVEEESDKAALLQQVLGLAQG", aa_code);

    pappso::PrecisionPtr precision_ptr = pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::QualifiedMassSpectrum spectrum_simple =
      readQualifiedMassSpectrumMgf(QString(CMAKE_SOURCE_DIR).append("/tests/data/scans/25976.mgf"));
    pappso::FilterChargeDeconvolution(precision_ptr)
      .filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterResampleKeepGreater(150).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterGreatestY(120).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));

    pappso::specpeptidoms::SpOMSSpectrum experimental_spectrum(
      spectrum_simple, precision_ptr, aa_code);


    pappso::specpeptidoms::ScoreValues score_values;

    pappso::specpeptidoms::SemiGlobalAlignment semi_global_alignment(
      score_values, precision_ptr, aa_code);

    semi_global_alignment.fastAlign(experimental_spectrum, &protein);
    std::vector<pappso::specpeptidoms::Location> locations =
      semi_global_alignment.getLocationSaver().getLocations();
    REQUIRE(locations.size() == 2);
    REQUIRE(locations.at(0).score == 21);
    REQUIRE(locations.at(0).beginning == 0);
    REQUIRE(locations.at(0).length == 31);

    for(auto loc : locations)
      {
        semi_global_alignment.preciseAlign(
          experimental_spectrum, &protein, loc.beginning, loc.length);
        pappso::specpeptidoms::Alignment best_alignment = semi_global_alignment.getBestAlignment();

        REQUIRE(best_alignment.m_peptideModel.toInterpretation().toStdString() ==
                "[244.069]EEESDKAALLQ");
        REQUIRE(best_alignment.SPC == 21);
        REQUIRE_THAT(best_alignment.peaks,
                     Catch::Matchers::Approx(
                       std::vector<std::size_t>({0, 1, 7, 19, 28, 41, 62, 77, 90, 109, 128, 147})));
        REQUIRE_THAT(best_alignment.shifts,
                     Catch::Matchers::Approx(std::vector<double>({})).margin(0.001));
        REQUIRE(best_alignment.score == 104);
        REQUIRE(best_alignment.beginning == 11);
        REQUIRE(best_alignment.end == 21);
        REQUIRE(best_alignment.begin_shift == 0.0);
        REQUIRE(best_alignment.end_shift == Catch::Approx(244.069));
        REQUIRE((best_alignment.begin_shift > 0 || best_alignment.shifts.size() > 0) == false);

        REQUIRE(best_alignment.getNonAlignedMass() == Catch::Approx(244.069));
        REQUIRE_THAT(best_alignment.m_peptideModel.getNonAlignedMass(),
                     Catch::Matchers::WithinAbs(0, 0.02));
      }
  }

  SECTION(
    "..:: BASIC sequence alignment scan_52119 spectrum easy met on fasta file "
    "[V][A]EEIQDEVDELLQKEQ "
    "::..",
    "[peptidoms]")
  {
    AaCode aa_code;
    aa_code.addAaModification('C', AaModification::getInstance("MOD:00397"));

    pappso::specpeptidoms::SpOMSProtein protein(
      "protein_id", "AGQVAEEIQDEVDELLQKEQAIATKKII", aa_code);

    pappso::PrecisionPtr precision_ptr = pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::QualifiedMassSpectrum spectrum_simple =
      readQualifiedMassSpectrumMgf(QString(CMAKE_SOURCE_DIR).append("/tests/data/scans/52119.mgf"));
    pappso::FilterChargeDeconvolution(precision_ptr)
      .filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterResampleKeepGreater(150).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterGreatestY(120).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));

    pappso::specpeptidoms::SpOMSSpectrum experimental_spectrum(
      spectrum_simple, precision_ptr, aa_code);


    pappso::specpeptidoms::ScoreValues score_values;

    pappso::specpeptidoms::SemiGlobalAlignment semi_global_alignment(
      score_values, precision_ptr, aa_code);

    semi_global_alignment.fastAlign(experimental_spectrum, &protein);
    std::vector<pappso::specpeptidoms::Location> locations =
      semi_global_alignment.getLocationSaver().getLocations();
    REQUIRE(locations.size() == 1);
    REQUIRE(locations.at(0).score == 57);
    REQUIRE(locations.at(0).beginning == 4);
    REQUIRE(locations.at(0).length == 36);

    for(auto loc : locations)
      {
        semi_global_alignment.preciseAlign(
          experimental_spectrum, &protein, loc.beginning, loc.length);
        pappso::specpeptidoms::Alignment best_alignment = semi_global_alignment.getBestAlignment();

        REQUIRE(best_alignment.m_peptideModel.toInterpretation().toStdString() ==
                "[V][A]EEIQDEVDELLQKEQ");
        REQUIRE(best_alignment.SPC == 26);
        REQUIRE_THAT(best_alignment.peaks,
                     Catch::Matchers::Approx(std::vector<std::size_t>(
                       {0, 1, 5, 17, 31, 45, 57, 71, 80, 90, 103, 112, 128, 137, 154, 167})));
        REQUIRE_THAT(best_alignment.shifts,
                     Catch::Matchers::Approx(std::vector<double>({})).margin(0.001));
        REQUIRE(best_alignment.score == 138);
        REQUIRE(best_alignment.beginning == 3);
        REQUIRE(best_alignment.end == 19);
        REQUIRE(best_alignment.begin_shift == 0.0);
        REQUIRE(best_alignment.end_shift == 0);
        REQUIRE((best_alignment.begin_shift > 0 || best_alignment.end_shift > 0 ||
                 best_alignment.shifts.size() > 0) == false);

        REQUIRE(best_alignment.getNonAlignedMass() == Catch::Approx(0));
        REQUIRE_THAT(best_alignment.m_peptideModel.getNonAlignedMass(),
                     Catch::Matchers::WithinAbs(0, 0.02));
      }
  }

  SECTION(
    "..:: BASIC sequence alignment scan_54257 spectrum easy met on fasta file "
    "[426,21]IQDEVDELLQKEQ "
    "::..",
    "[peptidoms]")
  {
    AaCode aa_code;
    aa_code.addAaModification('C', AaModification::getInstance("MOD:00397"));

    pappso::specpeptidoms::SpOMSProtein protein(
      "protein_id", "AGQVAEEIQDEVDELLQKEQAIATKKII", aa_code);

    pappso::PrecisionPtr precision_ptr = pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::QualifiedMassSpectrum spectrum_simple =
      readQualifiedMassSpectrumMgf(QString(CMAKE_SOURCE_DIR).append("/tests/data/scans/54257.mgf"));
    pappso::FilterChargeDeconvolution(precision_ptr)
      .filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterResampleKeepGreater(150).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterGreatestY(120).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));

    pappso::specpeptidoms::SpOMSSpectrum experimental_spectrum(
      spectrum_simple, precision_ptr, aa_code);


    pappso::specpeptidoms::ScoreValues score_values;

    pappso::specpeptidoms::SemiGlobalAlignment semi_global_alignment(
      score_values, precision_ptr, aa_code);

    semi_global_alignment.fastAlign(experimental_spectrum, &protein);
    std::vector<pappso::specpeptidoms::Location> locations =
      semi_global_alignment.getLocationSaver().getLocations();
    REQUIRE(locations.size() == 1);
    REQUIRE(locations.at(0).score == 47);
    REQUIRE(locations.at(0).beginning == 2);
    REQUIRE(locations.at(0).length == 33);

    for(auto loc : locations)
      {
        semi_global_alignment.preciseAlign(
          experimental_spectrum, &protein, loc.beginning, loc.length);
        pappso::specpeptidoms::Alignment best_alignment = semi_global_alignment.getBestAlignment();

        REQUIRE(best_alignment.m_peptideModel.toInterpretation().toStdString() ==
                "[426.209]IQDEVDELLQKEQ");
        REQUIRE(best_alignment.SPC == 13);
        REQUIRE_THAT(best_alignment.peaks,
                     Catch::Matchers::Approx(
                       std::vector<std::size_t>({0, 8, 12, 16, 20, 22, 25, 27, 29, 33, 37, 41})));
        REQUIRE_THAT(best_alignment.shifts,
                     Catch::Matchers::Approx(std::vector<double>({})).margin(0.001));
        REQUIRE(best_alignment.score == 80);
        REQUIRE(best_alignment.beginning == 7);
        REQUIRE(best_alignment.end == 19);
        REQUIRE(best_alignment.begin_shift == 0.0);
        REQUIRE(best_alignment.end_shift == Catch::Approx(426.209));
        REQUIRE((best_alignment.begin_shift > 0 || best_alignment.shifts.size() > 0) == false);

        REQUIRE(best_alignment.getNonAlignedMass() == Catch::Approx(426.209));
        REQUIRE_THAT(best_alignment.m_peptideModel.getNonAlignedMass(),
                     Catch::Matchers::WithinAbs(0, 0.02));
      }
  }

  SECTION(
    "..:: BASIC sequence alignment scan_57729 spectrum easy met on fasta file [271,12]VYLAAVNRLYQ "
    "::..",
    "[peptidoms]")
  {
    AaCode aa_code;
    aa_code.addAaModification('C', AaModification::getInstance("MOD:00397"));

    pappso::specpeptidoms::SpOMSProtein protein(
      "protein_id", "LWADHGVQAGTVYLAAVNRLYQQALPMGAL", aa_code);

    pappso::PrecisionPtr precision_ptr = pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::QualifiedMassSpectrum spectrum_simple =
      readQualifiedMassSpectrumMgf(QString(CMAKE_SOURCE_DIR).append("/tests/data/scans/57729.mgf"));
    pappso::FilterChargeDeconvolution(precision_ptr)
      .filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterResampleKeepGreater(150).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterGreatestY(120).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));

    pappso::specpeptidoms::SpOMSSpectrum experimental_spectrum(
      spectrum_simple, precision_ptr, aa_code);


    pappso::specpeptidoms::ScoreValues score_values;

    pappso::specpeptidoms::SemiGlobalAlignment semi_global_alignment(
      score_values, precision_ptr, aa_code);

    semi_global_alignment.fastAlign(experimental_spectrum, &protein);
    std::vector<pappso::specpeptidoms::Location> locations =
      semi_global_alignment.getLocationSaver().getLocations();
    REQUIRE(locations.size() == 2);
    REQUIRE(locations.at(0).score == 76);
    REQUIRE(locations.at(0).beginning == 4);
    REQUIRE(locations.at(0).length == 26);

    for(auto loc : locations)
      {
        semi_global_alignment.preciseAlign(
          experimental_spectrum, &protein, loc.beginning, loc.length);
        pappso::specpeptidoms::Alignment best_alignment = semi_global_alignment.getBestAlignment();

        REQUIRE(best_alignment.m_peptideModel.toInterpretation().toStdString() ==
                "[271.125]VYLAAVNRLYQ");
        REQUIRE(best_alignment.SPC == 17);
        REQUIRE_THAT(
          best_alignment.peaks,
          Catch::Matchers::Approx(std::vector<std::size_t>({0, 1, 6, 12, 20, 21, 24, 26, 27, 32})));
        REQUIRE_THAT(best_alignment.shifts,
                     Catch::Matchers::Approx(std::vector<double>({})).margin(0.001));
        REQUIRE(best_alignment.score == 76);
        REQUIRE(best_alignment.beginning == 11);
        REQUIRE(best_alignment.end == 21);
        REQUIRE(best_alignment.begin_shift == 0.0);
        REQUIRE(best_alignment.end_shift == Catch::Approx(271.125));
        REQUIRE((best_alignment.begin_shift > 0 || best_alignment.shifts.size() > 0) == false);

        REQUIRE(best_alignment.getNonAlignedMass() == Catch::Approx(271.125));
        REQUIRE_THAT(best_alignment.m_peptideModel.getNonAlignedMass(),
                     Catch::Matchers::WithinAbs(0, 0.02));
      }
  }
}
