//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.tooldef.typechecker;

import java.net.MalformedURLException;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.escet.common.app.framework.PlatformUtils;
import org.eclipse.escet.common.java.TextPosition;
import org.eclipse.escet.common.java.UncachedUrlClassLoader.OpenUrlException;
import org.eclipse.escet.common.typechecker.SemanticException;
import org.eclipse.escet.tooldef.common.ClassLoaderObtainer;
import org.osgi.framework.Bundle;

/** Class loader obtainer for ToolDef imports. */
public class ImportClassLoaderObtainer extends ClassLoaderObtainer {
    /** The position information to use to report problems. */
    private final TextPosition pos;

    /** The type checker context. */
    private final CheckerContext ctxt;

    /** The text describing what is being imported. Must be either {@code "Java code"} or {@code "ToolDef script"}. */
    private final String useText;

    /**
     * Constructor for the {@link ImportClassLoaderObtainer} class.
     *
     * @param pos The position information to use to report problems.
     * @param ctxt The type checker context.
     * @param useText The text describing what is being imported. Must be either {@code "Java code"} or
     *     {@code "ToolDef script"}.
     */
    public ImportClassLoaderObtainer(TextPosition pos, CheckerContext ctxt, String useText) {
        this.pos = pos;
        this.ctxt = ctxt;
        this.useText = useText;
    }

    @Override
    protected RuntimeException createNotPluginProjException(String name) {
        ctxt.addProblem(Message.IMPORT_PLUGIN_NOT_PLUGIN, pos, useText, name);
        return new SemanticException();
    }

    @Override
    protected RuntimeException createPluginClassicFormatException(String name) {
        ctxt.addProblem(Message.IMPORT_PLUGIN_CLASSIC_FORMAT, pos, useText, name);
        return new SemanticException();
    }

    @Override
    protected RuntimeException createComputeClassPathException(String name, CoreException ex) {
        String exMsg = ex.getMessage();
        if (exMsg == null) {
            exMsg = "no additional details available.";
        }
        ctxt.addProblem(Message.IMPORT_PLUGIN_NO_JAVA_NATURE, pos, useText, name, exMsg);
        return new SemanticException();
    }

    @Override
    protected RuntimeException createMalformedUrlException(String name, MalformedURLException ex) {
        String exMsg = ex.getMessage();
        if (exMsg == null) {
            exMsg = "no additional details available.";
        }
        ctxt.addProblem(Message.IMPORT_PLUGIN_MALFORMED_URL, pos, useText, name, exMsg);
        return new SemanticException();
    }

    @Override
    protected RuntimeException createOpenUrlException(String name, OpenUrlException ex) {
        String exMsg = ex.getMessage();
        if (exMsg == null) {
            exMsg = "no additional details available.";
        }
        ctxt.addProblem(Message.IMPORT_PLUGIN_OPEN_URL, pos, useText, name, ex.url.toString(), exMsg);
        return new SemanticException();
    }

    @Override
    protected RuntimeException createNotFoundException(String name) {
        ctxt.addProblem(Message.IMPORT_PLUGIN_NOT_FOUND, pos, name);
        return new SemanticException();
    }

    @Override
    protected RuntimeException createWrongStateException(String name, Bundle bundle, int state) {
        ctxt.addProblem(Message.IMPORT_PLUGIN_STATE, pos, useText, name, PlatformUtils.getStateName(bundle));
        return new SemanticException();
    }

    @Override
    protected RuntimeException createNoClassLoaderException(String name, Bundle bundle) {
        ctxt.addProblem(Message.IMPORT_PLUGIN_ADAPT, pos, useText, name);
        return new SemanticException();
    }
}
