/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd, node*/
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) { //$NON-NLS-0$
        define('orion/Deferred',factory);
    } else if (typeof exports === "object") { //$NON-NLS-0$
        module.exports = factory();
    } else {
        root.orion = root.orion || {};
        root.orion.Deferred = factory();
    }
}(this, function() {
    var queue = [],
        running = false;

    function run() {
        var fn;
        while ((fn = queue.shift())) {
            fn();
        }
        running = false;
    }

	var runAsync = (function() {
		if (typeof process !== "undefined" && typeof process.nextTick === "function") {
			var nextTick = process.nextTick;
    		return function() {
    			nextTick(run);
    		};
		} else if (typeof MutationObserver === "function") {
			var div = document.createElement("div");
			var observer = new MutationObserver(run);
			observer.observe(div, {
            	attributes: true
        	});
        	return function() {
        		div.setAttribute("class", "_tick");
        	};
		}
		return function() {
			setTimeout(run, 0);
		};
	})();

    function enqueue(fn) {
        queue.push(fn);
        if (!running) {
            running = true;
            runAsync();
        }
    }

    function noReturn(fn) {
        return function(result) {
            fn(result);
        };
    }
    
    function settleDeferred(fn, result, deferred) {
    	try {
    		var listenerResult = fn(result);
    		var listenerThen = listenerResult && (typeof listenerResult === "object" || typeof listenerResult === "function") && listenerResult.then;
    		if (typeof listenerThen === "function") {
    			if (listenerResult === deferred.promise) {
    				deferred.reject(new TypeError());
    			} else {
    				var listenerResultCancel = listenerResult.cancel;
    				if (typeof listenerResultCancel === "function") {
    					deferred._parentCancel = listenerResultCancel.bind(listenerResult);
    				} else {
    					delete deferred._parentCancel;
    				}
    				listenerThen.call(listenerResult, noReturn(deferred.resolve), noReturn(deferred.reject), noReturn(deferred.progress));
    			}
    		} else {
    			deferred.resolve(listenerResult);
    		}
    	} catch (e) {
    		deferred.reject(e);
    	}
    }


    /**
     * @name orion.Promise
     * @class Interface representing an eventual value.
     * @description Promise is an interface that represents an eventual value returned from the single completion of an operation.
     *
     * <p>For a concrete class that implements Promise and provides additional API, see {@link orion.Deferred}.</p>
     * @see orion.Deferred
     * @see orion.Deferred#promise
     */
    /**
     * @name then
     * @function
     * @memberOf orion.Promise.prototype
     * @description Adds handlers to be called on fulfillment or progress of this promise.
     * @param {Function} [onResolve] Called when this promise is resolved.
     * @param {Function} [onReject] Called when this promise is rejected.
     * @param {Function} [onProgress] May be called to report progress events on this promise.
     * @returns {orion.Promise} A new promise that is fulfilled when the given <code>onResolve</code> or <code>onReject</code>
     * callback is finished. The callback's return value gives the fulfillment value of the returned promise.
     */
    /**
     * Cancels this promise.
     * @name cancel
     * @function
     * @memberOf orion.Promise.prototype
     * @param {Object} reason The reason for canceling this promise.
     * @param {Boolean} [strict]
     */

    /**
     * @name orion.Deferred
     * @borrows orion.Promise#then as #then
     * @borrows orion.Promise#cancel as #cancel
     * @class Provides abstraction over asynchronous operations.
     * @description Deferred provides abstraction over asynchronous operations.
     *
     * <p>Because Deferred implements the {@link orion.Promise} interface, a Deferred may be used anywhere a Promise is called for.
     * However, in most such cases it is recommended to use the Deferred's {@link #promise} field instead, which exposes a 
     * simplified, minimally <a href="https://github.com/promises-aplus/promises-spec">Promises/A+</a>-compliant interface to callers.</p>
     */
    function Deferred() {
        var result, state, listeners = [],
            _this = this;

        function notify() {
            var listener;
            while ((listener = listeners.shift())) {
                var deferred = listener.deferred;
                var methodName = state === "fulfilled" ? "resolve" : "reject"; //$NON-NLS-0$ //$NON-NLS-1$ //$NON-NLS-2$
                var fn = listener[methodName];
                if (typeof fn === "function") { //$NON-NLS-0$
                	settleDeferred(fn, result, deferred);
                } else {
                    deferred[methodName](result);
                }
            }
        }

        function _reject(error) {
            delete _this._parentCancel;
            state = "rejected";
            result = error;
            if (listeners.length) {
                enqueue(notify);
            }
        }

        function _resolve(value) {
            function once(fn) {
                return function(result) {
                    if (!state || state === "assumed") {
                          fn(result);
                    }
                };
            }
            delete _this._parentCancel;
            try {
                var valueThen = value && (typeof value === "object" || typeof value === "function") && value.then;
                if (typeof valueThen === "function") {
                    if (value === _this) {
                        _reject(new TypeError());
                    } else {
                        state = "assumed";
                        var valueCancel = value && value.cancel;
                        if (typeof valueCancel !== "function") {
                            var deferred = new Deferred();
                            value = deferred.promise;
                            try {
                                valueThen(deferred.resolve, deferred.reject, deferred.progress);
                            } catch (thenError) {
                                deferred.reject(thenError);
                            }
                            valueCancel = value.cancel;
                            valueThen = value.then;
                        }
                        result = value;
                        valueThen.call(value, once(_resolve), once(_reject));
                        _this._parentCancel = valueCancel.bind(value);
                    }
                } else {
                    state = "fulfilled";
                    result = value;
                    if (listeners.length) {
                        enqueue(notify);
                    }
                }
            } catch (error) {
                once(_reject)(error);
            }
        }

        function cancel() {
            var parentCancel = _this._parentCancel;
            if (parentCancel) {
                delete _this._parentCancel;
                parentCancel();
            } else if (!state) {
                var cancelError = new Error("Cancel");
                cancelError.name = "Cancel";
                _reject(cancelError);
            }
        }


        /**
         * Resolves this Deferred.
         * @name resolve
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} value
         * @returns {orion.Promise}
         */
        this.resolve = function(value) {
            if (!state) {
                _resolve(value);
            }
            return _this;
        };

        /**
         * Rejects this Deferred.
         * @name reject
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} error
         * @param {Boolean} [strict]
         * @returns {orion.Promise}
         */
        this.reject = function(error) {
            if (!state) {
                _reject(error);
            }
            return _this;
        };

        /**
         * Notifies listeners of progress on this Deferred.
         * @name progress
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} update The progress update.
         * @returns {orion.Promise}
         */
        this.progress = function(update) {
            if (!state) {
                listeners.forEach(function(listener) {
                    if (listener.progress) {
                        try {
                            listener.progress(update);
                        } catch (ignore) {
                            // ignore
                        }
                    }
                });
            }
            return _this.promise;
        };

        this.cancel = function() {
            if (_this._parentCancel) {
                setTimeout(cancel, 0);
            } else {
                cancel();
            }
            return _this;
        };

        // Note: "then" ALWAYS returns before having onResolve or onReject called as per http://promises-aplus.github.com/promises-spec/
        this.then = function(onFulfill, onReject, onProgress) {
        	var deferred = new Deferred();
            deferred._parentCancel = _this.promise.cancel;
            listeners.push({
                resolve: onFulfill,
                reject: onReject,
                progress: onProgress,
                deferred: deferred
            });
            if (state === "fulfilled" || state === "rejected") {
                enqueue(notify);
            }
            return deferred.promise;
        };

        /**
         * The promise exposed by this Deferred.
         * @name promise
         * @field
         * @memberOf orion.Deferred.prototype
         * @type orion.Promise
         */
        this.promise = {
            then: _this.then,
            cancel: _this.cancel
        };
    }

    /**
     * Returns a promise that represents the outcome of all the input promises.
     * <p>When <code>all</code> is called with a single parameter, the returned promise has <dfn>eager</dfn> semantics,
     * meaning that if any input promise rejects, the returned promise immediately rejects, without waiting for the rest of the
     * input promises to fulfill.</p>
     *
     * To obtain <dfn>lazy</dfn> semantics (meaning the returned promise waits for every input promise to fulfill), pass the
     * optional parameter <code>optOnError</code>.
     * @name all
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {orion.Promise[]} promises The input promises.
     * @param {Function} [optOnError] Handles a rejected input promise. <code>optOnError</code> is invoked for every rejected
     * input promise, and is passed the reason the input promise was rejected. <p><code>optOnError</code> can return a value, which
     * allows it to act as a transformer: the return value serves as the final fulfillment value of the rejected promise in the 
     * results array generated by <code>all</code>.
     * @returns {orion.Promise} A new promise. The returned promise is generally fulfilled to an <code>Array</code> whose elements
     * give the fulfillment values of the input promises. <p>However, if an input promise rejects and eager semantics is used, the 
     * returned promise will instead be fulfilled to a single error value.</p>
     */
    Deferred.all = function(promises, optOnError) {
        var count = promises.length,
            result = [],
            rejected = false,
            deferred = new Deferred();

        deferred.then(undefined, function() {
            rejected = true;
            promises.forEach(function(promise) {
                if (promise.cancel) {
                    promise.cancel();
                }
            });
        });

        function onResolve(i, value) {
            if (!rejected) {
                result[i] = value;
                if (--count === 0) {
                    deferred.resolve(result);
                }
            }
        }

        function onReject(i, error) {
            if (!rejected) {
                if (optOnError) {
                    try {
                        onResolve(i, optOnError(error));
                        return;
                    } catch (e) {
                        error = e;
                    }
                }
                deferred.reject(error);
            }
        }

        if (count === 0) {
            deferred.resolve(result);
        } else {
            promises.forEach(function(promise, i) {
                promise.then(onResolve.bind(undefined, i), onReject.bind(undefined, i));
            });
        }
        return deferred.promise;
    };

    /**
     * Applies callbacks to a promise or to a regular object.
     * @name when
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {Object|orion.Promise} value Either a {@link orion.Promise}, or a normal value.
     * @param {Function} onResolve Called when the <code>value</code> promise is resolved. If <code>value</code> is not a promise,
     * this function is called immediately.
     * @param {Function} onReject Called when the <code>value</code> promise is rejected. If <code>value</code> is not a promise, 
     * this function is never called.
     * @param {Function} onProgress Called when the <code>value</code> promise provides a progress update. If <code>value</code> is
     * not a promise, this function is never called.
     * @returns {orion.Promise} A new promise.
     */
    Deferred.when = function(value, onResolve, onReject, onProgress) {
        var promise, deferred;
        if (value && typeof value.then === "function") { //$NON-NLS-0$
            promise = value;
        } else {
            deferred = new Deferred();
            deferred.resolve(value);
            promise = deferred.promise;
        }
        return promise.then(onResolve, onReject, onProgress);
    };

    return Deferred;
}));
/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/EventTarget',[],function() {
	/**
	 * Creates an Event Target
	 *
	 * @name orion.EventTarget
	 * @class Base for creating an Orion event target
	 */
	function EventTarget() {
		this._namedListeners = {};
	}

	EventTarget.prototype = /** @lends orion.EventTarget.prototype */
	{
		/**
		 * Dispatches a named event along with an arbitrary set of arguments. Any arguments after <code>eventName</code>
		 * will be passed to the event listener(s).
		 * @param {Object} event The event to dispatch. The event object MUST have a type field
		 * @returns {boolean} false if the event has been canceled and any associated default action should not be performed
		 * listeners (if any) have resolved.
		 */
		dispatchEvent: function(event) {
			if (!event.type) {
				throw new Error("unspecified type");
			}
			var listeners = this._namedListeners[event.type];
			if (listeners) {
				listeners.forEach(function(listener) {
					try {
						if (typeof listener === "function") {
							listener(event);
						} else {
							listener.handleEvent(event);
						}
					} catch (e) {
						if (typeof console !== 'undefined') {
							console.log(e); // for now, probably should dispatch an ("error", e)
						}
					}			
				});
			}
			return !event.defaultPrevented;
		},

		/**
		 * Adds an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		addEventListener: function(eventName, listener) {
			if (typeof listener === "function" || listener.handleEvent) {
				this._namedListeners[eventName] = this._namedListeners[eventName] || [];
				this._namedListeners[eventName].push(listener);
			}
		},

		/**
		 * Removes an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		removeEventListener: function(eventName, listener) {
			var listeners = this._namedListeners[eventName];
			if (listeners) {
				for (var i = 0; i < listeners.length; i++) {
					if (listeners[i] === listener) {
						if (listeners.length === 1) {
							delete this._namedListeners[eventName];
						} else {
							listeners.splice(i, 1);
						}
						break;
					}
				}
			}
		}
	};
	EventTarget.prototype.constructor = EventTarget;
	
	EventTarget.attach = function(obj) {
		var eventTarget = new EventTarget();
		obj.dispatchEvent = eventTarget.dispatchEvent.bind(eventTarget);
		obj.addEventListener = eventTarget.addEventListener.bind(eventTarget);
		obj.removeEventListener = eventTarget.removeEventListener.bind(eventTarget);
	};
	
	return EventTarget;
});
/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd, node*/
/* eslint-disable missing-nls */
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) {
        define('orion/plugin',["orion/Deferred", "orion/EventTarget"], factory);
    } else if (typeof exports === "object") {
        module.exports = factory(require("orion/Deferred"), require("orion/EventTarget"));
    } else {
        root.orion = root.orion || {};
        root.orion.PluginProvider = factory(root.orion.Deferred, root.orion.EventTarget);
    }
}(this, function(Deferred, EventTarget) {

    function _equal(obj1, obj2) {
        var keys1 = Object.keys(obj1);
        var keys2 = Object.keys(obj2);
        if (keys1.length !== keys2.length) {
            return false;
        }
        keys1.sort();
        keys2.sort();
        for (var i = 0, len = keys1.length; i < len; i++) {
            var key = keys1[i];
            if (key !== keys2[i]) {
                return false;
            }
            var value1 = obj1[key],
                value2 = obj2[key];
            if (value1 === value2) {
                continue;
            }
            if (JSON.stringify(value1) !== JSON.stringify(value2)) {
                return false;
            }
        }
        return true;
    }

    function ObjectReference(objectId, methods) {
        this.__objectId = objectId;
        this.__methods = methods;
    }
    
    function PluginProvider(headers, serviceRegistry) {
        var _headers = headers;
        var _connected = false;

        var _currentMessageId = 0;
        var _currentObjectId = 0;
        var _currentServiceId = 0;

        var _requestReferences = {};
        var _responseReferences = {};
        var _objectReferences = {};
        var _serviceReferences = {};
        
        var _services;
        var _remoteServices = {};
        var _registry = serviceRegistry;
        var _connectCallback;
        
        var _ports = [];
        var _shared = false;
        
        var _target = null;
        if (typeof(window) === "undefined") {
            if (self.postMessage) {
                _target = self;
            } else {
                _shared = true;
            }
        } else if (window !== window.parent) {
            _target = window.parent;
        } else if (window.opener !== null) {
            _target = window.opener;
        }        

        function _publish(message, target) {
            target = target || _target;
            if (target) {
                if (typeof(ArrayBuffer) === "undefined") {
                    message = JSON.stringify(message);
                }
                if (target === self || _shared) {
                    target.postMessage(message);
                } else {
                    target.postMessage(message, "*");
                }
            }
        }
        var _notify = _publish;
        var _errHandler = function(evt){
        	_publish({method: "error", error: _serializeError(evt.error)});
        };
        addEventListener("error", _errHandler);
        
        var lastHeartbeat;
        var startTime = Date.now();
        function log(state) {
            if (typeof(localStorage) !== "undefined" && localStorage.pluginLogging) {
            	console.log(state + "(" + (Date.now() - startTime) + "ms)=" + self.location);
        	}
        }
        function heartbeat() {
            var time = Date.now();
            // This timeout depends on the handshake timeout of the plugin registry. Update both accordingly.
            if (lastHeartbeat  && time - lastHeartbeat < 4000) return;
            lastHeartbeat = time;
            _publish({
                method: "loading"
            });
            log("heartbeat");
        }
        heartbeat();

        if (_shared) {
            self.addEventListener("connect", function(evt) {
                var port = evt.ports[0];
                _ports.push(port);
                if (_connected) {
                    var message = {
                        method: "plugin",
                        params: [_getPluginData()]
                    };
                    _publish(message, port);
                } else {
                    heartbeat();
                }
                port.addEventListener("message",  function(evt) {
                	_handleMessage(evt, port);
                });
                port.start();
            });
        }

        function _getPluginData() {
            var services = [];
            // we filter out the service implementation from the data
            Object.keys(_serviceReferences).forEach(function(serviceId) {
                var serviceReference = _serviceReferences[serviceId];
                services.push({
                    serviceId: serviceId,
                    names: serviceReference.names,
                    methods: serviceReference.methods,
                    properties: serviceReference.properties
                });
            });
            return {
            	updateRegistry: !!_registry,
                headers: _headers || {},
                services: services
            };
        }

        function _jsonXMLHttpRequestReplacer(name, value) {
            if (value && value instanceof XMLHttpRequest) {
                var status, statusText;
                try {
                    status = value.status;
                    statusText = value.statusText;
                } catch (e) {
                    // https://bugs.webkit.org/show_bug.cgi?id=45994
                    status = 0;
                    statusText = ""; //$NON-NLS-0
                }
                return {
                    status: status || 0,
                    statusText: statusText
                };
            }
            return value;
        }

        function _serializeError(error) {
            var result = error ? JSON.parse(JSON.stringify(error, _jsonXMLHttpRequestReplacer)) : error; // sanitizing Error object
            if (error instanceof Error) {
                result.__isError = true;
                result.message = result.message || error.message;
                result.name = result.name || error.name;
            }
            return result;
        }

        function _request(message, target) {
            target = target || _target;
            if (!target) {
                return new Deferred().reject(new Error("plugin not connected"));
            }

            message.id = String(_currentMessageId++);
            var d = new Deferred();
            _responseReferences[message.id] = d;
            d.then(null, function(error) {
                if (_connected && error instanceof Error && error.name === "Cancel") {
                    _notify({
                        requestId: message.id,
                        method: "cancel",
                        params: error.message ? [error.message] : []
                    }, target);
                }
            });

            var toStr = Object.prototype.toString;
            message.params.forEach(function(param, i) {
                if (toStr.call(param) === "[object Object]" && !(param instanceof ObjectReference)) {
                    var candidate, methods;
                    for (candidate in param) {
                        if (toStr.call(param[candidate]) === "[object Function]") {
                            methods = methods || [];
                            methods.push(candidate);
                        }
                    }
                    if (methods) {
                        var objectId = _currentObjectId++;
                        _objectReferences[objectId] = param;
                        var removeReference = function() {
                            delete _objectReferences[objectId];
                        };
                        d.then(removeReference, removeReference);
                        message.params[i] = new ObjectReference(objectId, methods);
                    }
                }
            });
            _notify(message, target);
            return d.promise;
        }

        function _throwError(messageId, error, target) {
            if (messageId || messageId === 0) {
                _notify({
                    id: messageId,
                    result: null,
                    error: error
                }, target);
            } else {
                console.log(error);
            }
        }

        function _callMethod(messageId, implementation, method, params, target) {
            params.forEach(function(param, i) {
                if (param && typeof param.__objectId !== "undefined") {
                    var obj = {};
                    param.__methods.forEach(function(method) {
                        obj[method] = function() {
                            return _request({
                                objectId: param.__objectId,
                                method: method,
                                params: Array.prototype.slice.call(arguments)
                            }, target);
                        };
                    });
                    params[i] = obj;
                }
            });
            var response = typeof messageId === "undefined" ? null : {
                id: messageId,
                result: null,
                error: null
            };
            try {
                var promiseOrResult = method.apply(implementation, params);
                if (!response) {
                    return;
                }

                if (promiseOrResult && typeof promiseOrResult.then === "function") {
                    _requestReferences[messageId] = promiseOrResult;
                    promiseOrResult.then(function(result) {
                        delete _requestReferences[messageId];
                        response.result = result;
                        _notify(response, target);
                    }, function(error) {
                        if (_requestReferences[messageId]) {
                            delete _requestReferences[messageId];
                            response.error = _serializeError(error);
                            _notify(response, target);
                        }
                    }, function() {
                        _notify({
                            responseId: messageId,
                            method: "progress",
                            params: Array.prototype.slice.call(arguments)
                        }, target);
                    });
                } else {
                    response.result = promiseOrResult;
                    _notify(response, target);
                }
            } catch (error) {
                if (response) {
                    response.error = _serializeError(error);
                    _notify(response, target);
                }
            }
        }

        function _handleMessage(evnt, target) {
            if (!_shared && evnt.source !== _target && typeof window !== "undefined") {
                return;
            }
            var data = evnt.data;
            var message = (typeof data !== "string" ? data : JSON.parse(data));
            try {
                if (message.method) { // request
                    var method = message.method,
                        params = message.params || [];
                    if ("serviceId" in message) {
                        var service = _serviceReferences[message.serviceId];
                        if (!service) {
                            _throwError(message.id, "service not found", target);
                        } else {
	                        service = service.implementation;
	                        if (method in service) {
	                            _callMethod(message.id, service, service[method], params, target);
	                        } else {
	                            _throwError(message.id, "method not found", target);
	                        }
                    	}
                    } else if ("objectId" in message) {
                        var object = _objectReferences[message.objectId];
                        if (!object) {
                            _throwError(message.id, "object not found", target);
                        } else if (method in object) {
                            _callMethod(message.id, object, object[method], params, target);
                        } else {
                            _throwError(message.id, "method not found", target);
                        }
                    } else if ("requestId" in message) {
                        var request = _requestReferences[message.requestId];
                        if (request && method === "cancel" && request.cancel) {
                            request.cancel.apply(request, params);
                        }
                    } else if ("responseId" in message) {
                        var response = _responseReferences[message.responseId];
                        if (response && method === "progress" && response.progress) {
                            response.progress.apply(response, params);
                        }
                    } else {
                        if ("plugin" === message.method) { //$NON-NLS-0$
                            var manifest = message.params[0];
                            _update({
                                services: manifest.services
                            });
                        } else {
                            throw new Error("Bad method: " + message.method);
                        }
                    }
                } else if (message.id) {
                    var deferred = _responseReferences[String(message.id)];
                    if (deferred) {
	                    delete _responseReferences[String(message.id)];
	                    if (message.error) {
	                        deferred.reject(message.error);
	                    } else {
	                        deferred.resolve(message.result);
	                    }
                    }
                }
            } catch (e) {
                console.log("Plugin._messageHandler " + e);
            }
        }        
        
        function _createServiceProxy(service) {
            var serviceProxy = {};
            if (service.methods) {
                service.methods.forEach(function(method) {
                    serviceProxy[method] = function() {
                        var message = {
                            serviceId: service.serviceId,
                            method: method,
                            params: Array.prototype.slice.call(arguments)
                        };
                        return _request(message);
                    };
                });

                if (serviceProxy.addEventListener && serviceProxy.removeEventListener && EventTarget) {
                    var eventTarget = new EventTarget();
                    var objectId = _currentObjectId++;
                    _objectReferences[objectId] = {
                        handleEvent: eventTarget.dispatchEvent.bind(eventTarget)
                    };
                    var listenerReference = new ObjectReference(objectId, ["handleEvent"]);

                    var _addEventListener = serviceProxy.addEventListener;
                    serviceProxy.addEventListener = function(type, listener) {
                        if (!eventTarget._namedListeners[type]) {
                            _addEventListener(type, listenerReference);
                        }
                        eventTarget.addEventListener(type, listener);
                    };
                    var _removeEventListener = serviceProxy.removeEventListener;
                    serviceProxy.removeEventListener = function(type, listener) {
                        eventTarget.removeEventListener(type, listener);
                        if (!eventTarget._namedListeners[type]) {
                            _removeEventListener(type, listenerReference);
                        }
                    };
                }
            }
            return serviceProxy;
        }

        function _createServiceProperties(service) {
            var properties = JSON.parse(JSON.stringify(service.properties));
            var objectClass = service.names || service.type || [];
            if (!Array.isArray(objectClass)) {
                objectClass = [objectClass];
            }
            properties.objectClass = objectClass;
            return properties;
        }

        function _registerService(service) {
        	if (!_registry) return;
            var serviceProxy = _createServiceProxy(service);
            var properties = _createServiceProperties(service);
            var registration = _registry.registerService(service.names || service.type, serviceProxy, properties);
            _remoteServices[service.serviceId] = {
                registration: registration,
                proxy: serviceProxy
            };
        }

        function _update(input) {
            var oldServices = _services || [];
            _services = input.services || [];

            if (!_equal(_services, oldServices)) {
	            var serviceIds = [];
				_services.forEach(function(service) {
					var serviceId = service.serviceId;
	                serviceIds.push(serviceId);
	                var remoteService = _remoteServices[serviceId];
	                if (remoteService) {
	                    if (_equal(service.methods, Object.keys(remoteService.proxy))) {
	                        var properties = _createServiceProperties(service);
	                        var reference = remoteService.registration.getReference();
	                        var currentProperties = {};
	                        reference.getPropertyKeys().forEach(function(_name) {
	                            currentProperties[_name] = reference.getProperty(_name);
	                        });
	                        if (!_equal(properties, currentProperties)) {
	                            remoteService.registration.setProperties(properties);
	                        }
	                        return;
	                    }
	                    remoteService.registration.unregister();
	                    delete _remoteServices[serviceId];
	                }
	                _registerService(service);
	            });
	            Object.keys(_remoteServices).forEach(function(serviceId) {
	                if (serviceIds.indexOf(serviceId) === -1) {
	                    _remoteServices[serviceId].registration.unregister();
	                    delete _remoteServices[serviceId];
	                }
	            });
           }
           
           if (_connectCallback) {
               _connectCallback();
               _connectCallback = null;
           }
        }

        this.updateHeaders = function(headers) {
            if (_connected) {
                throw new Error("Cannot update headers. Plugin Provider is connected");
            }
            _headers = headers;
        };

        this.registerService = function(names, implementation, properties) {
            if (_connected) {
                throw new Error("Cannot register service. Plugin Provider is connected");
            }

            if (typeof names === "string") {
                names = [names];
            } else if (!Array.isArray(names)) {
                names = [];
            }

            var method = null;
            var methods = [];
            for (method in implementation) {
                if (typeof implementation[method] === 'function') {
                    methods.push(method);
                }
            }
            _serviceReferences[_currentServiceId++] = {
                names: names,
                methods: methods,
                implementation: implementation,
                properties: properties || {},
                listeners: {}
            };
            heartbeat();
        };
        this.registerServiceProvider = this.registerService;

        this.connect = function(callback, errback) {
            if (_connected) {
                if (callback) {
                    callback();
                }
                return;
            }
            removeEventListener("error", _errHandler);
            var message = {
                method: "plugin",
                params: [_getPluginData()]
            };
            if (!_shared) {
                if (!_target) {
                    if (errback) {
                        errback("No valid plugin target");
                    }
                    return;
                }           
                addEventListener("message", _handleMessage, false);
                _publish(message);
            }
            if (typeof(window) !== "undefined") {
            	var head = document.getElementsByTagName("head")[0] || document.documentElement;
            	var title = head.getElementsByTagName("title")[0];
            	if (!title) {
	            	title = document.createElement("title");
	            	title.textContent = _headers ? _headers.name : '';
	            	head.appendChild(title);
	        	}
        	}

            _ports.forEach(function(port) {
                _publish(message, port);
            });
            _connected = true;
            if (_registry) {
            	_connectCallback = callback;
            } else {
	            if (callback) {
	                callback();
	            }
            }
        };

        this.disconnect = function() {
            if (_connected) {
                removeEventListener("message", _handleMessage);
                _ports.forEach(function(port) {
                    port.close();
                });
                _ports = null;
                _target = null;
                _connected = false;
            }
            // Note: re-connecting is not currently supported
        };            
    }
    
    return PluginProvider;
}));

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/arduino/arduinoPlugin',['orion/plugin'], function(PluginProvider) {

	function connect() {
		var headers = {
			name: "Orion Arduino Tool Support",
			version: "1.0",
			description: "This plugin provides Arduino tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-arduino", // TODO could not find a commonly-used value for this
					"extends": "text/x-csrc",
					name: "Arduino",
					extension: ["ino", "pde"]
				}
			] 
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/lib/syntax", [], function() {
	return {
		id: "orion.lib",
		grammars: [{
			id: "orion.lib",
			repository: {
				brace_open: {
					match: "{",
					name: "punctuation.section.block.begin"
				},
				brace_close: {
					match: "}",
					name: "punctuation.section.block.end"
				},
				bracket_open: {
					match: "\\[",
					name: "punctuation.section.bracket.begin"
				},
				bracket_close: {
					match: "\\]",
					name: "punctuation.section.bracket.end"
				},
				parenthesis_open: {
					match: "\\(",
					name: "punctuation.section.parens.begin"
				},
				parenthesis_close: {
					match: "\\)",
					name: "punctuation.section.parens.end"
				},
				operator: {
					match: "(\\+|-|!|=|>|<|&|(\\|\\|))+",
					name: "punctuation.operator"
				},
				doc_block: {
					begin: "/\\*\\*",
					end: "\\*/",
					name: "comment.block.documentation",
					beginCaptures: {
						0: {name: "comment.block.documentation.start"}
					},
					endCaptures: {
						0: {name: "comment.block.documentation.end"}
					},
					patterns: [
						{
							match: "@(?:(?!\\*/)\\S)*",
							name: "meta.documentation.annotation"
						}, {
							match: "<[^\\s>]*>",
							name: "meta.documentation.tag"
						}, {
							match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
							name: "meta.annotation.task.todo",
							captures: {
								2: {name: "keyword.other.documentation.task"},
								4: {name: "comment.block"}
							}
						}
					]
				},
				number_decimal: {
					match: "\\b-?(?:\\.\\d+|\\d+\\.?\\d*)(?:[eE][+-]?\\d+)?\\b",
					name: "constant.numeric.number"
				},
				number_hex: {
					match: "\\b0[xX][0-9A-Fa-f]+\\b",
					name: "constant.numeric.hex"
				},
				string_doubleQuote: {
					match: '"(?:\\\\.|[^"])*"?',
					name: "string.quoted.double"
				},
				string_singleQuote: {
					match: "'(?:\\\\.|[^'])*'?",
					name: "string.quoted.single"
				},
				todo_comment_singleLine: {
					match: "(\\b)(TODO)(\\b)(.*)",
					name: "meta.annotation.task.todo",
					captures: {
						2: {name: "keyword.other.documentation.task"},
						4: {name: "comment.line"}
					}
				}
			}
		}, {
			id: "orion.c-like",
			repository: {
				comment_singleLine: {
					match: {match: "(//).*", literal: "//"},
					name: "comment.line.double-slash",
					captures: {
						1: {name: "comment.line.double-slash.start"}
					},
					patterns: [
						{
							include: "orion.lib#todo_comment_singleLine"
						}
					]
				},
				comment_block: {
					begin: {match: "/\\*", literal: "/*"},
					end: {match: "\\*/", literal: "*/"}, 
					name: "comment.block",
					beginCaptures: {
						0: {name: "comment.block.start"}
					},
					endCaptures: {
						0: {name: "comment.block.end"}
					},
					patterns: [
						{
							match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
							name: "meta.annotation.task.todo",
							captures: {
								2: {name: "keyword.other.documentation.task"},
								4: {name: "comment.block"}
							}
						}
					]
				}
			}
		}],
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/application_x-sh/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"alias",
		"bg", "bind", "builtin",
		"caller", "command", "cd", "coproc",
		"declare", "dirs", "disown",
		"echo", "enable", "eval", "exec", "export",
		"fc", "fg",
		"getopts",
		"hash", "help", "history",
		"jobs",
		"kill",
		"let", "local", "logout",
		"mapfile",
		"popd", "printf", "pushd", "pwd",
		"readarray", "readonly", "read",
		"set", "shift", "shopt", "source", "suspend",
		"test", "times", "trap", "typeset", "type",
		"ulimit", "umask", "unalias", "unset",
		"wait"
	];
	var controlKeywords = [
		"break",
		"case", "continue",
		"done, do", 
		"elif", "else", "esac", "exit",
		"fi", "for",
		"if", "in",
		"return", 
		"select",
		"then",
		"until", "while"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.bash",
		contentTypes: ["application/x-sh"],
		patterns: [
			{include: "#string_doubleQuote"},
			{include: "#string_singleQuote"},
			{include: "#comment"},
			{include: "#variable"},
			{include: "#keywordsControl"},
			{include: "#keywordsOperator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{include: "#operator"}
		],
		repository: {
			comment: {
				match: {match: "#.*", literal: "#"},
				name: "comment.line.number-sign.bash",
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			keywordsControl: {
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.bash"
			},
			keywordsOperator: {
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.bash"
			},
			operator: {
				match: "(\\+|-|!|~|\\*|\\/|%|=|>|<|&|\\^|\\||\\?|:|,|\\[|\\]|\\(|\\)|\\{|\\})+",
				name: "punctuation.operator.bash"
			},
			string_doubleQuote: {
				match: '\\$?"(?:\\\\.|[^"])*"?',
				name: "string.quoted.double.bash"
			},
			string_singleQuote: {
				match: "\\$?'[^']'",
				name: 'string.quoted.single.bash'
			},
			variable: {
				match: "\\$\\w+\\b",
				name: "variable.other.bash"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/bash/bashPlugin',['orion/plugin', 'orion/editor/stylers/application_x-sh/syntax'], function(PluginProvider, mBash) {

	function connect() {
		var headers = {
			name: "Orion Bash Script Tool Support",
			version: "1.0",
			description: "This plugin provides Bash script tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "application/x-sh",
					"extends": "text/plain",
					name: "Bash",
					extension: ["sh", "bash"]
				}
			] 
		});
		mBash.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-csrc/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"auto",
		"break",
		"case", "char", "const", "continue",
		"default", "double", "do",
		"else", "enum", "extern",
		"float", "for",
		"goto",
		"if", "inline", "int",
		"long",
		"register", "return",
		"short", "signed", "sizeof", "static", "struct", "switch",
		"typedef",
		"union", "unsigned",
		"void", "volatile",
		"while",
		"_Bool", "_Complex", "_Imaginary"
	];

	var directives = [
		"define", "elif", "else", "endif", "error", "ifdef",
		"ifndef", "if", "include", "line", "pragma", "undef"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.c",
		contentTypes: ["text/x-csrc", "text/x-c"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "#directive"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.c"
			}
		],
		repository: {
			directive: {
				match: "#\\s*(?:" + directives.join("|") + ")\\b[^$]*",
				name: "meta.preprocessor.c"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(directives)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/c/cPlugin',['orion/plugin', 'orion/editor/stylers/text_x-csrc/syntax'], function(PluginProvider, mC) {

	function connect() {
		var headers = {
			name: "Orion C Tool Support",
			version: "1.0",
			description: "This plugin provides C tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-csrc",
					"extends": "text/plain",
					name: "C",
					extension: ["c", "h"]
				}
			] 
		});
		mC.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/application_javascript/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"async", "await",
		"class", "const",
		"debugger", "delete",
		"enum", "export", "extends",
		"function",
		"implements", "import", "in", "instanceof", "interface",
		"let",
		"new",
		"of",
		"package", "private", "protected", "public",
		"static", "super",
		"typeof",
		"var", "void",
		"with"
	];
	var controlKeywords = [
		"break",
		"case", "catch", "continue",
		"default", "do",
		"else",
		"finally", "for",
		"if",
		"return",
		"switch",
		"throw", "try",
		"while",
		"yield"
	];
	var languageVariables = ["this"];
	var constants = [
		"false", "null", "true", "undefined"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.js",
		contentTypes: ["application/javascript"],
		patterns: [
			{include: "#string_multiline_singleQuote"},
			{include: "#string_multiline_doubleQuote"},
			{include: "#templateLiteral"},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "#regex"},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "#jsFunctionDef"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{include: "#keywordsOperator"},
			{include: "#keywordsControl"},
			{include: "#languageConstant"},
			{include: "#languageVariable"}
		],
		repository: {
			jsFunctionDef: {
				/*
				 * http://stackoverflow.com/questions/2008279/validate-a-javascript-function-name/2008444#2008444
				 * was referenced in the composition of the "begin" pattern below.
				 */
				begin: "(function)(\\s+[_$a-zA-Z\\xA0-\\uFFFF][_$a-zA-Z0-9\\xA0-\\uFFFF]*)?\\s*\\(",
				end: "\\)",
				captures: {
					1: {name: "keyword.operator.js"},
					2: {name: "entity.name.function.js"}
				},
				patterns: [
					{include: "orion.c-like#comment_singleLine"},
					{include: "orion.c-like#comment_block"},
					{
						match: "[^\\s,]+",
						name: "variable.parameter.js"
					}
				]
			},
			keywordsControl: {
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.js"
			},
			keywordsOperator: {
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.js"
			},
			languageConstant: {
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.js"
			},
			languageVariable: {
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.js"
			},
			string_multiline_singleQuote: {
				begin: "'(?:\\\\.|[^\\\\'])*\\\\$",
				end: "^(?:$|(?:\\\\.|[^\\\\'])*('|[^\\\\]$))",
				name: "string.quoted.single.js"
			},
			string_multiline_doubleQuote: {
				begin: '"(?:\\\\.|[^\\\\"])*\\\\$',
				end: '^(?:$|(?:\\\\.|[^\\\\"])*("|[^\\\\]$))',
				name: "string.quoted.double.js"
			},
			regex: {
				match: "/(?![\\s\\*])(?:\\\\.|[^/])+/(?:[gim]{0,3})",
				name: "string.regexp.js"
			},
			templateLiteral: {
				begin: "`",
				end: "`",
				name: "string.quoted.backtick.js",
				patterns: [
					{
						begin: "\\$\\{",
						end: "\\}",
						name: "string.interpolated.js",
						patterns: [
							{include: "#string_multiline_singleQuote"},
							{include: "#string_multiline_doubleQuote"},
							{include: "#templateLiteral"},
							{include: "orion.lib#string_doubleQuote"},
							{include: "orion.lib#string_singleQuote"},
							{include: "orion.c-like#comment_singleLine"},
							{include: "#regex"},
							{include: "orion.lib#doc_block"},
							{include: "orion.c-like#comment_block"},
							{include: "#jsFunctionDef"},
							{include: "orion.lib#brace_open"},
							{include: "orion.lib#brace_close"},
							{include: "orion.lib#bracket_open"},
							{include: "orion.lib#bracket_close"},
							{include: "orion.lib#parenthesis_open"},
							{include: "orion.lib#parenthesis_close"},
							{include: "orion.lib#operator"},
							{include: "orion.lib#number_decimal"},
							{include: "orion.lib#number_hex"},
							{include: "#keywordsOperator"},
							{include: "#keywordsControl"},
							{include: "#languageConstant"},
							{include: "#languageVariable"}
						]
					}
				]
			}
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(languageVariables).concat(constants)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/application_vnd.coffeescript/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/application_javascript/syntax"], function(mLib, mJS) {
	var controlKeywords = [
		"break",
		"case", "catch", "continue",
		"default", "do",
		"else",
		"finally", "for",
		"if",
		"loop",
		"return",
		"switch",
		"then", "throw", "try",
		"unless", "until",
		"when", "while",
		"yield"
	];
	var keywords = [
		"and", "arguments",
		"by",
		"class", "const",
		"debugger", "delete",
		"enum", "eval", "export", "extends",
		"function",
		"implements", "import", "instanceof", "interface", "in", "isnt", "is",
		"let",
		"native", "new", "not", "no",
		"off", "of", "on", "or",
		"package", "private", "protected", "public",
		"static", "super",
		"typeof",
		"var", "void",
		"with",
		"yes",
		"__bind", "__extends", "__hasProp", "__indexOf", "__slice",
	];
	var constants = [
		"false", "null", "true", "undefined"
	];
	var languageVariables = ["this"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push({
		id: "orion.coffeescript",
		contentTypes: ["application/vnd.coffeescript"],
		patterns: [
			{include: "#comment_block"},
			{include: "#embedded_js"},
			{include: "#string_singleline"},
			{include: "#string_multiline"},
			{include: "#heregex"},
			{include: "#regex"},
			{include: "orion.lib#string_singleQuote"},
			{include: "#comment_singleLine"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{include: "#reserved_words"},
			{include: "#keyword_control"},
			{include: "#keyword_operator"},
			{include: "#language_variable"},
			{include: "#constant"},
			{include: "#punctuation"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
		],
		repository: {
			comment_block: {
				begin: {match: "###(?!#)", literal: "###"},
				end: {match: "###", literal: "###"},
				name: "comment.block.coffeescript"
			},
			comment_singleLine: {
				match: {match: "#[^$]*", literal: "#"},
				name: "comment.line.coffeescript"
			},
			constant: {
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.coffeescript"
			},
			embedded_js: {
				begin: "`",
				end: "`",
				patterns: [
					{include: "orion.js"}
				]
			},
			heregex: {
				begin: "\\/{3}",
				end: "\\/{3}(?:[gim]{0,3})",
				name: "string.heregexp.coffeescript",
				patterns: [{
					match: "\\s#[^$]*",
					name: "comment.block.coffeescript"
				}]
			},
			keyword_control: {
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.coffeescript"
			},
			keyword_operator: {
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.coffeescript"
			},
			language_variable: {
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.coffeescript"
			},
			punctuation: {
				/* the following match is based on a regular expression from http://coffeescript.org/ */
				match: "[-=]?>|<|[-+*\\/%<>&|^!?=]?=|>>>=?|([-+*/%:])\\1|([&|<>])\\2=?|\\?\\.?|\\.{2,3}|%|@",
				name: "punctuation.operator.coffeescript"
			},
			regex: {
				match: "/(?![\\s\\*])(?:\\\\.|[^/])+/(?:[gim]{0,3})",
				name: "string.regexp.coffeescript"
			},
			string_singleline: {
				match: '"(?:\\\\.|[^"])*"',
				name: "string.quoted.double.coffeescript",
				patterns: [
					{
						begin: "#{",
						end: "}",
						name: "string.interpolated.coffeescript",
						patterns: [
							{include: "#comment_block"},
							{include: "#embedded_js"},
							{include: "#comment_singleLine"},
							{include: "#heregex"},
							{include: "#regex"},
							{include: "orion.lib#string_singleQuote"},
							{include: "orion.lib#number_decimal"},
							{include: "#reserved_words"},
							{include: "#keyword_control"},
							{include: "#keyword_operator"},
							{include: "#language_variable"},
							{include: "#constant"},
							{include: "#punctuation"},
							{include: "orion.lib#brace_open"},
							{include: "orion.lib#brace_close"},
							{include: "orion.lib#bracket_open"},
							{include: "orion.lib#bracket_close"},
							{include: "orion.lib#parenthesis_open"},
							{include: "orion.lib#parenthesis_close"},
						]
					}
				]
			},
			string_multiline: {
				begin: '("("")?)',
				end: '\\1',
				name: "string.quoted.multiline.coffeescript",
				patterns: [
					{
						begin: "#{",
						end: "}",
						name: "string.interpolated.coffeescript",
						patterns: [
							{include: "#comment_block"},
							{include: "#embedded_js"},
							{include: "#comment_singleLine"},
							{include: "#heregex"},
							{include: "#regex"},
							{include: "orion.lib#string_singleQuote"},
							{include: "orion.lib#number_decimal"},
							{include: "#reserved_words"},
							{include: "#keyword_control"},
							{include: "#keyword_operator"},
							{include: "#language_variable"},
							{include: "#constant"},
							{include: "#punctuation"},
							{include: "orion.lib#brace_open"},
							{include: "orion.lib#brace_close"},
							{include: "orion.lib#bracket_open"},
							{include: "orion.lib#bracket_close"},
							{include: "orion.lib#parenthesis_open"},
							{include: "orion.lib#parenthesis_close"}
						]
					}
				]
			},
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: controlKeywords.concat(constants)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/coffeescript/coffeescriptPlugin',['orion/plugin', 'orion/editor/stylers/application_vnd.coffeescript/syntax'], function(PluginProvider, mCoffeescript) {

	function connect() {
		var headers = {
			name: "Orion CoffeeScript Tool Support",
			version: "1.0",
			description: "This plugin provides CoffeeScript tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "application/vnd.coffeescript",
					"extends": "text/plain",
					name: "CoffeeScript",
					extension: ["coffee"]
				}
			] 
		});
		mCoffeescript.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-c__src/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"alignas", "alignof", "asm", "and_eq", "and", "auto",
		"bitand", "bitor", "bool", "break",
		"case", "catch", "char16_t", "char32_t", "char", "class",
		"compl", "constexpr", "const_cast", "const", "continue",
		"decltype", "default", "delete", "double", "do", "dynamic_cast",
		"else", "enum", "explicit", "export", "extern",
		"false", "float", "for", "friend",
		"goto",
		"if", "inline", "int",
		"long",
		"mutable",
		"namespace", "new", "noexcept", "not_eq", "not", "nullptr",
		"operator", "or_eq", "or",
		"private", "protected", "public",
		"register", "reinterpret_cast", "return",
		"short", "signed", "sizeof", "static_assert",
		"static_cast", "static", "struct", "switch",
		"template", "this", "thread_local", "throw", "true",
		"try", "typedef", "typeid", "typename",
		"union", "unsigned", "using",
		"virtual", "void", "volatile",
		"wchar_t", "while",
		"xor_eq", "xor",
		"_Bool", "_Complex", "_Imaginary"
	];

	var directives = [
		"define", "elif", "else", "endif", "error", "ifdef",
		"ifndef", "if", "include", "line", "pragma", "undef"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.cpp",
		contentTypes: ["text/x-c++src", "text/x-c++"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "#directive"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.cpp"
			}
		],
		repository: {
			directive: {
				match: "#\\s*(?:" + directives.join("|") + ")\\b[^$]*",
				name: "meta.preprocessor.cpp"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/cpp/cppPlugin',['orion/plugin', 'orion/editor/stylers/text_x-c__src/syntax'], function(PluginProvider, mCPP) {

	function connect() {
		var headers = {
			name: "Orion C++ Tool Support",
			version: "1.0",
			description: "This plugin provides C++ tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-c++src",
					"extends": "text/plain",
					name: "C++",
					extension: ["cpp", "hpp"]
				}
			] 
		});
		mCPP.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-csharp/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"abstract", "as",
		"base", "bool", "break", "byte", "by",
		"case", "catch", "char", "checked", "class", "const", "continue",
		"decimal", "default", "delegate", "descending", "double", "do",
		"else", "enum", "event", "explicit", "extern",
		"false", "finally", "fixed", "float", "foreach", "for", "from",
		"goto", "group",
		"if", "implicit", "int", "interface", "internal", "into", "in", "is",
		"lock", "long",
		"namespace", "new", "null",
		"object", "operator", "orderby", "out", "override",
		"params", "private", "protected", "public",
		"readonly", "ref", "return",
		"sbyte", "sealed", "select", "short", "sizeof", "stackalloc", "static", "string", "struct", "switch", 
		"this", "throw", "true", "try", "typeof",
		"uint", "ulong", "unchecked", "unsafe", "ushort", "using",
		"var", "virtual", "volatile", "void",
		"while", "where",
		"yield"
	];

	var preprocessorDirectives = [
		"define",
		"elif", "else", "endif", "endregion", "error",
		"if",
		"line",
		"pragma checksum", "pragma warning", "pragma",
		"region",
		"undef",
		"warning"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.csharp",
		contentTypes: ["text/x-csharp"],
		patterns: [
			{
				match: "^\\s*#(?:" + preprocessorDirectives.join("|") + ")\\b[^$]*",
				name: "meta.preprocessor.csharp"
			},
			{include: "#string_verbatim"},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "#doc_line"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.csharp"
			}
		],
		repository: {
			doc_block: {
				begin: "/\\*\\*",
				end: "\\*/",
				name: "comment.block.documentation.csharp",
				patterns: [
					{
						match: "<[^\\s>]*>",
						name: "meta.documentation.tag"
					}, {
						match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.block"}
						}
					}
				]
			},
			doc_line: {
				match: "// /.*",
				name: "comment.line.documentation.csharp",
				patterns: [
					{
						match: "<[^\\s>]*>",
						name: "meta.documentation.tag"
					}, {
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			string_verbatim: {
				begin: '@"',
				end: '^(?:""|[^"])*"(?!")',
				name: "string.quoted.verbatim.csharp",
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
/* eslint-disable missing-nls */
define("orion/editor/stylers/application_xml/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.xml",
		contentTypes: ["application/xml", "application/xhtml+xml"],
		patterns: [
			{include: "#comment"},
			{include: "#doctype"},
			{include: "#xmlDeclaration"},
			{include: "#tag"},
			{include: "#ampersandEscape"}
		],
		repository: {
			ampersandEscape: {
				match: "&lt;|&gt;|&amp;",
				name: "constant.character"
			},
			comment: {
				begin: {match: "<!--", literal: "<!--"},
				end: {match: "-->", literal: "-->"},
				name: "comment.block.xml",
				beginCaptures: {
					0: {name: "comment.block.start.xml"}
				},
				endCaptures: {
					0: {name: "comment.block.end.xml"}
				},
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!-->).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.line"}
						}
					}
				]
			},
			doctype: {
				begin: "<!(?:doctype|DOCTYPE)",
				end: ">",
				name: "meta.tag.doctype.xml",
				captures: {
					0: {name: "meta.tag.doctype.xml"}
				},
				patterns: [
					{include: "#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"}
				]
			},
			tag: {
				// https://www.w3.org/TR/2006/REC-xml11-20060816/#sec-common-syn
				begin: "</?[A-Za-z:_][A-Za-z0-9:_\\-.]*",
				end: "/?>",
				captures: {
					0: {name: "meta.tag.xml"}
				},
				patterns: [
					{include: "#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"}
				]	
			},
			xmlDeclaration: {
				begin: "<\\?xml",
				end: "\\?>",
				captures: {
					0: {name: "meta.tag.declaration.xml"}
				},
				patterns: [
					{include: "#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"}
				],
				name: "meta.tag.declaration.xml"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
/*eslint-disable missing-nls */
define("orion/editor/stylers/text_css/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	// This list generated by modifying cssContentAssist.js getPropertyProposals() to dump a list of all property proposals to the console
	// The list should be reversed so that long names come before short ones ('flex-flow' before 'flex') so the entire word is marked as a keyword
	var keywords = [
		"zoom", "z-index", "writing-mode", "word-wrap", "word-spacing", 
		"word-break", "will-change", "width", "widows", "white-space-collapse", "white-space", "volume", "voice-volume", "voice-stress", "voice-rate", 
		"voice-pitch-range", "voice-pitch", "voice-family", "voice-duration", "voice-balance", "visibility", "vertical-align", "user-select", 
		"user-modify", "unicode-bidi", "transition-timing-function", "transition-property", "transition-duration", "transition-delay", "transition", 
		"transform-style", "transform-origin", "transform", "touch-action", "-ms-touch-action", "top", "text-wrap", "text-transform", "text-shadow", 
		"text-rendering", "text-overflow", "text-outline", "text-justify", "text-indent", "text-height", "text-emphasis", "text-decoration-style", 
		"text-decoration-line", "text-decoration-color", "text-decoration", "text-anchor", "text-align-last", "text-align", "target-position", 
		"target-new", "target-name", "target", "tab-size", "table-layout", "stroke-width", "stroke-opacity", "stroke-miterlimit", "stroke-linejoin", 
		"stroke-linecap", "stroke-dashoffset", "stroke-dasharray", "stroke", "string-set", "stress", "stop-opacity", "stop-color", "src", "speech-rate", 
		"speak-punctuation", "speak-numeral", "speak-header", "speak", "size", "shape-rendering", "ruby-span", "ruby-position", "ruby-overhang", 
		"ruby-align", "rotation-point", "rotation", "right", "richness", "rest-before", "rest-after", "rest", "resize", "rendering-intent", "quotes", 
		"punctuation-trim", "presentation-level", "position", "pointer-events", "play-during", "pitch-range", "pitch", "phonemes", "perspective-origin", 
		"perspective", "pause-before", "pause-after", "pause", "page-policy", "page-break-inside", "page-break-before", "page-break-after", "page", 
		"padding-top", "padding-right", "padding-left", "padding-bottom", "padding", "overflow-y", "overflow-x", "overflow-wrap", "overflow-style", 
		"overflow", "outline-width", "outline-style", "outline-offset", "outline-color", "outline", "orphans", "-webkit-order", "order", "opacity", 
		"object-position", "object-fit", "nav-up", "nav-right", "nav-left", "nav-index", "nav-down", "move-to", "min-width", "min-height", 
		"max-width", "max-height", "mask", "marquee-style", "marquee-speed", "marquee-play-count", "marquee-direction", "marks", "marker-start", 
		"marker-mid", "marker-end", "marker", "mark-before", "mark-after", "mark", "margin-top", "margin-right", "margin-left", "margin-bottom", 
		"margin", "list-style-type", "list-style-position", "list-style-image", "list-style", "line-stacking-strategy", "line-stacking-shift", 
		"line-stacking-ruby", "line-stacking", "line-break", "line-height", "letter-spacing", "left", "kerning", "-webkit-justify-content", "justify-content", 
		"inline-box-align", "ime-mode", "image-resolution", "image-rendering", "image-orientation", "icon", "hyphens", "hyphenate-resource", 
		"hyphenate-lines", "hyphenate-character", "hyphenate-before", "hyphenate-after", "height", "hanging-punctuation", "grid-template-rows", 
		"grid-template-columns", "grid-template-areas", "grid-template", "grid-row-sizing", "grid-row-span", "grid-row-end", "grid-row-start", "grid-row-align", 
		"grid-rows", "grid-row", "grid-layer", "grid-flow", "grid-column-span", "grid-column-end", "grid-column-start", "grid-column-sizing", 
		"grid-column-align", "grid-columns", "grid-column", "grid-cell-stacking", "grid-auto-rows", "grid-auto-position", "grid-auto-flow", 
		"grid-auto-columns", "grid-area", "grid", "glyph-orientation-vertical", "glyph-orientation-horizontal", "font-weight", "font-variant-position", 
		"font-variant-numeric", "font-variant-ligatures", "font-variant-east-asian", "font-variant-caps", "font-variant-alternates", "font-variant", 
		"font-style", "font-stretch", "font-size-adjust", "font-size", "font-kerning", "font-feature-settings", "font-family", "font", "flood-opacity", 
		"flood-color", "float-offset", "float", "-ms-flex-wrap", "-ms-flex-pack", "-ms-flex-order", "-ms-flex-direction", "-ms-flex-align", "-ms-flex", 
		"-webkit-flex-wrap", "-webkit-flex-shrink", "-webkit-flex-grow", "-webkit-flex-flow", "-webkit-flex-direction", "-webkit-flex-basis", "-webkit-flex", 
		"flex-wrap", "flex-shrink", "flex-grow", "flex-flow", "flex-direction", "flex-basis", "flex", "fit-position", "fit", "filter", "fill-rule", 
		"fill-opacity", "fill", "enable-background", "empty-cells", "elevation", "drop-initial-value", "drop-initial-size", "drop-initial-before-align", 
		"drop-initial-before-adjust", "drop-initial-after-align", "drop-initial-after-adjust", "dominant-baseline", "display", "direction", "cursor", 
		"cue-before", "cue-after", "cue", "crop", "counter-reset", "counter-increment", "content", "columns", "column-width", "column-span", 
		"column-rule-width", "column-rule-style", "column-rule-color", "column-rule", "column-gap", "column-fill", "column-count", "color-rendering", 
		"color-profile", "color-interpolation-filters", "color-interpolation", "color", "clip-rule", "clip-path", "-webkit-clip-path", "clip", "clear", 
		"caption-side", "break-inside", "break-before", "break-after", "box-sizing", "box-shadow", "box-decoration-break", "-webkit-box-pack", 
		"-webkit-box-orient", "-webkit-box-ordinal-group", "-webkit-box-lines", "-webkit-box-flex-group", "-webkit-box-flex", "-webkit-box-direction", 
		"-webkit-box-decoration-break", "-webkit-box-align", "-o-box-decoration-break", "-moz-box-pack", "-moz-box-orient", "-moz-box-ordinal-group", "-moz-box-lines", 
		"-moz-box-flex-group", "-moz-box-flex", "-moz-box-direction", "-moz-box-decoration-break", "-moz-box-align", "bottom", "border-width", 
		"border-top-width", "border-top-style", "border-top-right-radius", "border-top-left-radius", "border-top-color", "border-top", "border-style", 
		"border-spacing", "border-right-width", "border-right-style", "border-right-color", "border-right", "border-radius", "border-left-width", 
		"border-left-style", "border-left-color", "border-left", "border-image-width", "border-image-source", "border-image-slice", "border-image-repeat", 
		"border-image-outset", "border-image", "border-color", "border-collapse", "border-bottom-width", "border-bottom-style", "border-bottom-right-radius", 
		"border-bottom-left-radius", "border-bottom-color", "border-bottom", "border", "bookmark-target", "bookmark-state", "bookmark-level", "bookmark-label", 
		"bleed", "binding", "behavior", "baseline-shift", "background-size", "background-repeat", "background-position", "background-origin", 
		"background-image", "background-color", "background-clip", "background-attachment", "background", "backface-visibility", "azimuth", 
		"-o-appearance", "-webkit-appearance", "-ms-appearance", "-moz-appearance", "appearance", "-o-animation-play-state", "-o-animation-name", 
		"-o-animation-iteration-count", "-o-animation-duration", "-o-animation-direction", "-o-animation-delay", "-webkit-animation-play-state", 
		"-webkit-animation-name", "-webkit-animation-iteration-count", "-webkit-animation-fill-mode", "-webkit-animation-duration", "-webkit-animation-direction", 
		"-webkit-animation-delay", "-ms-animation-play-state", "-ms-animation-name", "-ms-animation-iteration-count", "-ms-animation-duration", 
		"-ms-animation-direction", "-ms-animation-delay", "-moz-animation-play-state", "-moz-animation-name", "-moz-animation-iteration-count", 
		"-moz-animation-duration", "-moz-animation-direction", "-moz-animation-delay", "animation-timing-function", "animation-play-state", "animation-name", 
		"animation-iteration-count", "animation-fill-mode", "animation-duration", "animation-direction", "animation-delay", "animation", "alignment-baseline", 
		"alignment-adjust", "-webkit-align-self", "-webkit-align-content", "-webkit-align-items", "all", "align-self", "align-content", "align-items"
	];
	var colors = [
		"AliceBlue", "AntiqueWhite", "Aquamarine", "Aqua", "Azure",
		"Beige", "Bisque", "Black", "BlanchedAlmond", "BlueViolet", "Blue", "Brown", "BurlyWood",
		"CadetBlue", "Chartreuse", "Chocolate", "Coral", "CornflowerBlue", "Cornsilk", "Crimson", "Cyan",
		"DarkBlue", "DarkCyan", "DarkGoldenRod", "DarkGray", "DarkGrey", "DarkGreen", "DarkKhaki", "DarkMagenta", "DarkOliveGreen",
		"DarkOrange", "DarkOrchid", "DarkRed", "DarkSalmon", "DarkSeaGreen", "DarkSlateBlue", "DarkSlateGray", "DarkSlateGrey",
		"DarkTurquoise", "DarkViolet", "DeepPink", "DeepSkyBlue", "DimGray", "DimGrey", "DodgerBlue",
		"FireBrick", "FloralWhite", "ForestGreen", "Fuchsia",
		"Gainsboro", "GhostWhite", "Gold", "GoldenRod", "Gray", "Grey", "GreenYellow", "Green",
		"HoneyDew", "HotPink",
		"IndianRed", "Indigo", "Ivory",
		"Khaki",
		"LavenderBlush", "Lavender", "LawnGreen", "LemonChiffon", "LightBlue", "LightCoral", "LightCyan", "LightGoldenRodYellow",
		"LightGray", "LightGrey", "LightGreen", "LightPink", "LightSalmon", "LightSeaGreen", "LightSkyBlue", "LightSlateGray",
		"LightSlateGrey", "LightSteelBlue", "LightYellow", "LimeGreen", "Lime", "Linen",
		"Magenta", "Maroon", "MediumAquaMarine", "MediumBlue", "MediumOrchid", "MediumPurple", "MediumSeaGreen", "MediumSlateBlue",
		"MediumSpringGreen", "MediumTurquoise", "MediumVioletRed", "MidnightBlue", "MintCream", "MistyRose", "Moccasin",
		"NavajoWhite", "Navy",
		"OldLace", "OliveDrab", "Olive", "OrangeRed", "Orange", "Orchid",
		"PaleGoldenRod", "PaleGreen", "PaleTurquoise", "PaleVioletRed", "PapayaWhip", "PeachPuff", "Peru", "Pink", "Plum", "PowderBlue", "Purple",
		"RebeccaPurple", "Red", "RosyBrown", "RoyalBlue",
		"SaddleBrown", "Salmon", "SandyBrown", "SeaGreen", "SeaShell", "Sienna", "Silver", "SkyBlue", "SlateBlue", "SlateGray", "SlateGrey", "Snow", "SpringGreen", "SteelBlue",
		"Tan", "Teal", "Thistle", "Tomato", "Turquoise",
		"Violet",
		"Wheat", "WhiteSmoke", "White",
		"YellowGreen", "Yellow"
	];
	var directives = ["charset", "document", "font-face", "import", "keyframes", "media", "namespace", "page", "supports"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.css",
		contentTypes: ["text/css"],
		patterns: [
			{include: "#string_single_multiline"},
			{include: "#string_double_multiline"},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#number_decimal"},
			{include: "#number_hex"},
			{include: "#numeric_value"},
			{include: "#color"},
			{include: "#keyword"},
			{include: "#directive"}
		],
		repository: {
			color: {
				match: "(?i)\\b(?:" + colors.join("|") + ")\\b",
				name: "constant.other.color.css"
			},
			directive: {
				match: "(^|\\s)(@("  + directives.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.directive.css"}
				}
			},
			keyword: {
				match: "(?:-webkit-|-moz-|-ms-|-o-|\\b)(?:" + keywords.join("|") + ")\\b",
				name: "support.type.propertyName.css"
			},
			number_hex: {
				match: "#[0-9A-Fa-f]+\\b",
				name: "constant.numeric.hex.css"
			},
			numeric_value: {
				match: "(?i)\\b-?(?:\\.\\d+|\\d+\\.?\\d*)(?:%|em|ex|ch|rem|vw|vh|vmin|vmax|in|cm|mm|pt|pc|px|deg|grad|rad|turn|s|ms|Hz|kHz|dpi|dpcm|dppx)?\\b",
				name: "constant.numeric.value.css"
			},
			string_double_multiline: {
				begin: '"(?:\\\\.|[^\\\\"])*\\\\$',
				end: '^(?:$|(?:\\\\.|[^\\\\"])*("|[^\\\\]$))',
				name: "string.quoted.double.css"
			},
			string_single_multiline: {
				begin: "'(?:\\\\.|[^\\\\'])*\\\\$",
				end: "^(?:$|(?:\\\\.|[^\\\\'])*('|[^\\\\]$))",
				name: "string.quoted.single.css"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_html/syntax", ["orion/editor/stylers/application_javascript/syntax", "orion/editor/stylers/text_css/syntax", "orion/editor/stylers/application_xml/syntax"],
	/* eslint-disable missing-nls */
	function(mJS, mCSS, mXML) {
	var attributes = [
		"accept-charset", "accept", "accesskey", "action", "align", "alt",  
		"async", "autocomplete", "autoplay", "autosave",
		"bgcolor", "border", "buffered", 
		"challenge", "charset", "checked", "cite", "class", "codebase", "code", "color",
		"colspan", "cols", "contenteditable", "content", "contextmenu", "controls", "coords",
		"data-[A-Za-z_:][\\w.:-]*", "data", "datetime", "default", "defer", "dirname", "dir",
		"disabled", "download", "draggable", "dropzone",
		"enctype",
		"formaction", "form", "for", 
		"headers", "height", "hidden", "high", "hreflang", "href", "http-equiv",
		"icon", "id", "ismap", "itemprop",
		"keytype", "kind", 
		"label", "language", "lang", "list", "loop", "low",  
		"manifest", "maxlength", "max", "media", "method", "min", "multiple",
		"name", "novalidate", 
		"open", "optimum", 
		"pattern", "ping", "placeholder", "poster", "preload", "pubdate",  
		"radiogroup", "readonly", "rel", "required", "reversed", "rowspan", "rows",
		"sandbox", "scoped", "scope", "seamless", "selected", "shape", "sizes", "size", "span", "spellcheck",
		"srcdoc", "srclang","srcset", "src", "start", "step", "style", "summary",
		"tabindex", "target", "title", "type",
		"usemap",
		"value",
		"width", "wrap" 
	];

	var ariaAttributes = [
		"activedescendant", "atomic", "autocomplete", 
		"busy", 
		"checked", "controls", 
		"describedby", "disabled", "dropeffect", 
		"expanded", 
		"flowto", 
		"grabbed", 
		"haspopup", "hidden", 
		"invalid", 
		"labelledby", "label", "level", "live", 
		"multiline", "multiselectable", 
		"orientation", "owns", 
		"posinset", "pressed", 
		"readonly", "relevant", "required", 
		"selected", "setsize", "sort", 
		"valuemax", "valuemin", "valuenow", "valuetext"
	];

	var grammars = [];
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push.apply(grammars, mCSS.grammars);
	grammars.push.apply(grammars, mXML.grammars);
	grammars.push({
		id: "orion.html",
		contentTypes: ["text/html"],
		patterns: [
			{
				begin: "(?i)(<style)([^>]*)(>)",
				end: "(?i)(</style>)",
				captures: {
					1: {name: "meta.tag.html"},
					3: {name: "meta.tag.html"}
				},
				contentName: "source.css.embedded.html",
				patterns: [
					{include: "orion.css"}
				]
			}, {
				begin: "(?i)<script\\s*>|<script\\s.*?(?:language\\s*=\\s*(['\"])javascript\\1|type\\s*=\\s*(['\"])(?:text|application)/(?:javascript|ecmascript)\\2).*?>",
				end: "(?i)</script>",
				captures: {
					0: {name: "meta.tag.html"}
				},
				contentName: "source.js.embedded.html",
				patterns: [
					{include: "orion.js"}
				]
			}, {
				begin: "</?[A-Za-z0-9]+",
				end: "/?>",
				captures: {
					0: {name: "meta.tag.html"}
				},
				patterns: [
					{include: "orion.xml#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
					{include: "#attribute"}
				]
			},
			{include: "orion.xml#comment"},
			{include: "orion.xml#doctype"},
			{include: "orion.xml#ampersandEscape"}
		],
		repository: {
			attribute:{
				match: "\\b(?:" + attributes.join("|") + "|role|aria-(" + ariaAttributes.join("|") + "))\\b",  
				name: "meta.tag.attribute.html"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: [],
		attributes: attributes
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-cshtml/syntax", [
	"orion/editor/stylers/application_xml/syntax",
	"orion/editor/stylers/text_html/syntax",
	"orion/editor/stylers/text_x-csharp/syntax"], function(mXML, mHTML, mCSharp) {

	var grammars = [];
	grammars.push.apply(grammars, mXML.grammars);
	grammars.push.apply(grammars, mHTML.grammars);
	grammars.push.apply(grammars, mCSharp.grammars);
	grammars.push({
		id: "orion.cshtml",
		contentTypes: ["text/x-cshtml"],
		patterns: [
			{include: "#comment"},
			{include: "#codeBlock"},
			{include: "#expression"},
			{include: "#reference"},
			{include: "orion.html"},
		],
		repository: {
			comment: {
				begin: {match: "@\\*", literal: "@*"},
				end: {match: "\\*@", literal: "*@"},
				name: "comment.block.cshtml",
			},
			codeBlock: {
				begin: "(^\\s*)(@)(?=([^{]*){)",
				end: "}",
				captures: {
					2: {name: "entity.name.declaration.csharp"}
				},
				contentName: "source.csharp.embedded.cshtml",
				patterns: [
				    {include: "orion.xml#tag"},
				    {include: "#reference"},
					{include: "orion.csharp"},
				]
			},
			expression: {
				match: "^\\s*@[^{]*$",
				name: "source.csharp.embedded.cshtml",
				patterns: [
				    {include: "#reference"},
					{include: "orion.csharp"},
				]
			},
			reference: {
				match: "@",
				name: "entity.name.declaration.csharp"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/csharp/csharpPlugin',['orion/plugin', 'orion/editor/stylers/text_x-csharp/syntax', 'orion/editor/stylers/text_x-cshtml/syntax'], function(PluginProvider, mCSharp, mCSHtml) {

	function connect() {
		var headers = {
			name: "Orion C# Tool Support",
			version: "1.0",
			description: "This plugin provides C# tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-csharp",
					"extends": "text/plain",
					name: "C#",
					extension: ["cs"]
				}, {id: "text/x-cshtml",
					"extends": "text/plain",
					name: "cshtml",
					extension: ["cshtml"]
				}
			] 
		});
		mCSharp.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
		mCSHtml.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-dockerfile/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"ADD", "ARG",
		"CMD", "COPY",
		"ENTRYPOINT", "ENV", "EXPOSE",
		"FROM",
		"HEALTHCHECK",
		"LABEL",
		"MAINTAINER",
		"ONBUILD",
		"RUN",
		"SHELL",
		"STOPSIGNAL",
		"USER",
		"VOLUME",
		"WORKDIR"
	];

	var id = "orion.dockerfile";
	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: id,
		contentTypes: ["text/x-dockerfile"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "#numberSignComment"},
			{
				match: "\\b-?[0-9]+(\\.[0-9]+)?\\b",
				name: "constant.numeric.dockerfile"
			}, {
				match: "(?i)^\\s*(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.dockerfile"
			}
		],
		repository: {
			numberSignComment: {
				begin: {match: "^\\s*#", literal: "#"},
				end: {match: "$", literal: ""},
				name: "comment.line.number-sign.dockerfile",
				patterns: [
					{include: "orion.lib#todo_comment_singleLine"}
				]
			}
		}
	});

	return {
		id: id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/objects',[], function() {
	function mixin(target/*, source..*/) {
		var hasOwnProperty = Object.prototype.hasOwnProperty;
		for (var j = 1, len = arguments.length; j < len; j++) {
			var source = arguments[j];
			for (var key in source) {
				if (hasOwnProperty.call(source, key)) {
					target[key] = source[key];
				}
			}
		}
		return target;
	}

	/**
	 * @name orion.objects
	 * @class Object-oriented helpers.
	 */
	return {
		/**
		 * Creates a shallow clone of the given <code>object</code>.
		 * @name orion.objects.clone
		 * @function
		 * @static
		 * @param {Object|Array} object The object to clone. Must be a "normal" Object or Array. Other built-ins,
		 * host objects, primitives, etc, will not work.
		 * @returns {Object|Array} A clone of <code>object</code>.
		 */
		clone: function(object) {
			if (Array.isArray(object)) {
				return Array.prototype.slice.call(object);
			}
			var clone = Object.create(Object.getPrototypeOf(object));
			mixin(clone, object);
			return clone;
		},
		/**
		 * Mixes all <code>source</code>'s own enumerable properties into <code>target</code>. Multiple source objects
		 * can be passed as varargs.
		 * @name orion.objects.mixin
		 * @function
		 * @static
		 * @param {Object} target
		 * @param {Object} source
		 */
		mixin: mixin,
		/**
		 * Wraps an object into an Array if necessary.
		 * @name orion.objects.toArray
		 * @function
		 * @static
		 * @param {Object} obj An object.
		 * @returns {Array} Returns <code>obj</code> unchanged, if <code>obj</code> is an Array. Otherwise returns a 1-element Array
		 * whose sole element is <code>obj</code>.
		 */
		toArray: function(o) {
			return Array.isArray(o) ? o : [o];
		}
	};
});
/**
 * @license RequireJS i18n 2.0.6 Copyright (c) 2010-2012, The Dojo Foundation All Rights Reserved.
 * Available via the MIT or new BSD license.
 * see: http://github.com/requirejs/i18n for details
 */
/*jslint regexp: true */
/*global require: false, navigator: false, define: false */

/**
 * This plugin handles i18n! prefixed modules. It does the following:
 *
 * 1) A regular module can have a dependency on an i18n bundle, but the regular
 * module does not want to specify what locale to load. So it just specifies
 * the top-level bundle, like 'i18n!nls/colors'.
 *
 * This plugin will load the i18n bundle at nls/colors, see that it is a root/master
 * bundle since it does not have a locale in its name. It will then try to find
 * the best match locale available in that master bundle, then request all the
 * locale pieces for that best match locale. For instance, if the locale is 'en-us',
 * then the plugin will ask for the 'en-us', 'en' and 'root' bundles to be loaded
 * (but only if they are specified on the master bundle).
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/colors bundle to be that mixed in locale.
 *
 * 2) A regular module specifies a specific locale to load. For instance,
 * i18n!nls/fr-fr/colors. In this case, the plugin needs to load the master bundle
 * first, at nls/colors, then figure out what the best match locale is for fr-fr,
 * since maybe only fr or just root is defined for that locale. Once that best
 * fit is found, all of its locale pieces need to have their bundles loaded.
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/fr-fr/colors bundle to be that mixed in locale.
 */
(function () {
    'use strict';

    //regexp for reconstructing the master bundle name from parts of the regexp match
    //nlsRegExp.exec('foo/bar/baz/nls/en-ca/foo') gives:
    //['foo/bar/baz/nls/en-ca/foo', 'foo/bar/baz/nls/', '/', '/', 'en-ca', 'foo']
    //nlsRegExp.exec('foo/bar/baz/nls/foo') gives:
    //['foo/bar/baz/nls/foo', 'foo/bar/baz/nls/', '/', '/', 'foo', '']
    //so, if match[5] is blank, it means this is the top bundle definition.
    var nlsRegExp = /(^.*(^|\/)nls(\/|$))([^\/]*)\/?([^\/]*)/;

    //Helper function to avoid repeating code. Lots of arguments in the
    //desire to stay functional and support RequireJS contexts without having
    //to know about the RequireJS contexts.
    function addPart(locale, master, needed, toLoad, prefix, suffix) {
        if (master[locale]) {
            needed.push(locale);
            if (master[locale] === true || master[locale] === 1) {
                toLoad.push(prefix + locale + '/' + suffix);
            }
        }
    }

    function addIfExists(req, locale, toLoad, prefix, suffix) {
        var fullName = prefix + locale + '/' + suffix;
        if (require._fileExists(req.toUrl(fullName + '.js'))) {
            toLoad.push(fullName);
        }
    }

    /**
     * Simple function to mix in properties from source into target,
     * but only if target does not already have a property of the same name.
     * This is not robust in IE for transferring methods that match
     * Object.prototype names, but the uses of mixin here seem unlikely to
     * trigger a problem related to that.
     */
    function mixin(target, source, force) {
        var prop;
        for (prop in source) {
            if (source.hasOwnProperty(prop) && (!target.hasOwnProperty(prop) || force)) {
                target[prop] = source[prop];
            } else if (typeof source[prop] === 'object') {
                if (!target[prop] && source[prop]) {
                    target[prop] = {};
                }
                mixin(target[prop], source[prop], force);
            }
        }
    }

    define('i18n',['module'], function (module) {
        var masterConfig = module.config ? module.config() : {};
        masterConfig = masterConfig || {};

        return {
            version: '2.0.6',
            /**
             * Called when a dependency needs to be loaded.
             */
            load: function (name, req, onLoad, config) {
                config = config || {};

                if (config.locale) {
                    masterConfig.locale = config.locale;
                }

                var masterName,
                    match = nlsRegExp.exec(name),
                    prefix = match[1],
                    locale = match[4],
                    suffix = match[5],
                    parts = locale.split('-'),
                    toLoad = [],
                    value = {},
                    i, part, current = '';

                //If match[5] is blank, it means this is the top bundle definition,
                //so it does not have to be handled. Locale-specific requests
                //will have a match[4] value but no match[5]
                if (match[5]) {
                    //locale-specific bundle
                    prefix = match[1];
                    masterName = prefix + suffix;
                } else {
                    //Top-level bundle.
                    masterName = name;
                    suffix = match[4];
                    locale = masterConfig.locale;
                    if (!locale) {
                        locale = masterConfig.locale =
                            typeof navigator === 'undefined' ? 'root' :
                            ((navigator.languages && navigator.languages[0]) ||
                             navigator.language ||
                             navigator.userLanguage || 'root').toLowerCase();
                    }
                    parts = locale.split('-');
                }

                if (config.isBuild) {
                    //Check for existence of all locale possible files and
                    //require them if exist.
                    toLoad.push(masterName);
                    addIfExists(req, 'root', toLoad, prefix, suffix);
                    for (i = 0; i < parts.length; i++) {
                        part = parts[i];
                        current += (current ? '-' : '') + part;
                        addIfExists(req, current, toLoad, prefix, suffix);
                    }

                    if(config.locales) {
                    	var j, k; 
                    	for (j = 0; j < config.locales.length; j++) {
                    		locale = config.locales[j];
                    		parts = locale.split("-");
                    		current = "";
	                    	for (k = 0; k < parts.length; k++) {
		                        part = parts[k];
		                        current += (current ? "-" : "") + part;
		                        addIfExists(req, current, toLoad, prefix, suffix);
	                    	}
                    	}
                    }

                    req(toLoad, function () {
                        onLoad();
                    });
                } else {
                    //First, fetch the master bundle, it knows what locales are available.
                    req([masterName], function (master) {
                        //Figure out the best fit
                        var needed = [],
                            part;

                        //Always allow for root, then do the rest of the locale parts.
                        addPart('root', master, needed, toLoad, prefix, suffix);
                        for (i = 0; i < parts.length; i++) {
                            part = parts[i];
                            current += (current ? '-' : '') + part;
                            addPart(current, master, needed, toLoad, prefix, suffix);
                        }

                        //Load all the parts missing.
                        req(toLoad, function () {
                            var i, partBundle, part;
                            for (i = needed.length - 1; i > -1 && needed[i]; i--) {
                                part = needed[i];
                                partBundle = master[part];
                                if (partBundle === true || partBundle === 1) {
                                    partBundle = req(prefix + part + '/' + suffix);
                                }
                                mixin(value, partBundle);
                            }

                            //All done, notify the loader.
                            onLoad(value);
                        });
                    });
                }
            }
        };
    });
}());

/*******************************************************************************
 * @license
 * Copyright (c) 2017 Remy Suen and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     Remy Suen - initial API and implementation
 ******************************************************************************/
/* eslint-env amd */
define('plugins/languages/docker/nls/messages',{
	root: true
});

/*******************************************************************************
 * @license
 * Copyright (c) 2017 Remy Suen and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     Remy Suen - initial API and implementation
 ******************************************************************************/
/* eslint-env amd */
define('plugins/languages/docker/nls/root/messages',{
	"pluginName": "Orion Dockerfile Editor",
	"pluginDescription": "This plug-in provides Dockerfile editing support for Orion, like keyword completion and syntax highlighting.",
	"dockerContentAssist": "Dockerfile Content Assist",
	"dockerContentHover": "Docker Content Hover",

	"hoverAdd": "Copy files, folders, or remote URLs from `source` to the `dest` path in the image's filesystem.\n\n",
	"hoverArg": "Define a variable with an optional default value that users can override at build-time when using `docker build`.\n\n",
	"hoverCmd": "Provide defaults for an executing container. If an executable is not specified, then `ENTRYPOINT` must be specified as well. There can only be one `CMD` instruction in a `Dockerfile`.\n\n",
	"hoverCopy": "Copy files or folders from `source` to the `dest` path in the image's filesystem.\n\n",
	"hoverEntrypoint": "Configures the container to be run as an executable.\n\n",
	"hoverEnv": "Set the environment variable `key` to the value `value`.\n\n",
	"hoverExpose": "Define the network `port`s that this container will listen on at runtime.\n\n",
	"hoverFrom": "Set the `baseImage` to use for subsequent instructions. `FROM` must be the first instruction in a `Dockerfile`.\n\n",
	"hoverHealthcheck": "Define how Docker should test the container to check that it is still working. Alternatively, disable the base image's `HEALTHCHECK` instruction. There can only be one `HEALTHCHECK` instruction in a `Dockerfile`.\n\nSince Docker 1.12\n\n",
	"hoverLabel": "Adds metadata to an image.\n\nSince Docker 1.6\n\n",
	"hoverMaintainer": "Set the _Author_ field of the generated images. This instruction has been deprecated in favor of `LABEL`.\n\n",
	"hoverOnbuild": "Add a _trigger_ instruction to the image that will be executed when the image is used as a base image for another build.\n\n",
	"hoverRun": "Execute any commands on top of the current image as a new layer and commit the results.\n\n",
	"hoverShell": "Override the default shell used for the _shell_ form of commands.\n\nSince Docker 1.12\n\n",
	"hoverStopsignal": "Set the system call signal to use to send to the container to exit. Signals can be valid unsigned numbers or a signal name in the `SIGNAME` format such as `SIGKILL`.\n\nSince Docker 1.12\n\n",
	"hoverUser": "Set the user name or UID to use when running the image in addition to any subsequent `CMD`, `ENTRYPOINT`, or `RUN` instructions that follow it in the `Dockerfile`.\n\n",
	"hoverVolume": "Create a mount point with the specifid name and mark it as holding externally mounted volumes from the native host or from other containers.\n\n",
	"hoverWorkdir": "Set the working directory for any subsequent `ADD`, `COPY`, `CMD`, `ENTRYPOINT`, or `RUN` instructions that follow it in the `Dockerfile`.\n\n",
	"hoverOnlineDocumentationFooter": "\n\n[Online documentation](${0})",

	"hoverEscape": "Sets the character to use to escape characters and newlines in this Dockerfile. If unspecified, the default escape character is `\\`.\n\n",

 	"proposalArgNameOnly": "Define a variable that users can set at build-time when using `docker build`.\n\n",
	"proposalArgDefaultValue": "Define a variable with the given default value that users can override at build-time when using `docker build`.\n\n",
	"proposalHealthcheckExec": "Define how Docker should test the container to check that it is still working. There can only be one `HEALTHCHECK` instruction in a `Dockerfile`.\n\nSince Docker 1.12\n\n",
	"proposalHealthcheckNone": "Disable the `HEALTHCHECK` instruction inherited from the base image if one exists. There can only be one `HEALTHCHECK` instruction in a `Dockerfile`.\n\nSince Docker 1.12"
});


/*******************************************************************************
 * @license
 * Copyright (c) 2012, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors: IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global requirejs*/
define('orion/i18nUtil',[], function() {
	/**
	 * Performs string substitution. Can be invoked in 2 ways:
	 *
	 * i) vargs giving numbered substition values:
	 *   formatMessage("${0} is ${1}", "foo", "bar")  // "foo is bar"
	 *
	 * ii) a map giving the substitutions:
	 *   formatMessage("${thing} is ${1}", {1: "bar", thing: "foo"})  // "foo is bar"
	 */
	function formatMessage(msg) {
		var pattern = /\$\{([^\}]+)\}/g, args = arguments;
		if (args.length === 2 && args[1] && typeof args[1] === "object") {
			return msg.replace(pattern, function(str, key) {
				return args[1][key];
			});
		}
		return msg.replace(pattern, function(str, index) {
			return args[(index << 0) + 1];
		});
	}
	return {
		formatMessage: formatMessage
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2017 Remy Suen and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     Remy Suen - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('plugins/languages/docker/dockerAssist',[
	'orion/objects',
	'i18n!plugins/languages/docker/nls/messages',
	'orion/i18nUtil'
], function(Objects, dockerMessages, i18nUtil) {

	/**
	 * @description Creates a new DockerContentAssist object
	 * @constructor
	 * @public
	 * @param {Object} keywords an array of Dockerfile keywords
	 * @param {Object} markdowns a dictionary of markdown documentation hovers
	 */
	function DockerContentAssist(keywords, markdowns) {
		this.keywords = keywords;
		this.markdowns = markdowns;
	}

	Objects.mixin(DockerContentAssist.prototype, {
		computeProposals: function (buffer, offset, context) {
			var firstCommentIdx = -1;
			var escapeCharacter = "\\";
			directiveCheck: for (var i = 0; i < buffer.length; i++) {
				switch (buffer.charAt(i)) {
					case '#':
						firstCommentIdx = i;
						// in the first comment of the file, look for directives
						var directive = "";
						var capture = false;
						escapeCheck: for (var j = i + 1; j < buffer.length; j++) {
							var char = buffer.charAt(j);
							switch (char) {
								case ' ':
								case '\t':
									// ignore whitespace if directive is well-formed or hasn't been found yet
									if (directive !== "escape" && directive !== "") {
										break escapeCheck;
									}
									continue;
								case '=':
									if (directive === "escape") {
										// '=' found and the directive that has been declared is the escape directive,
										// record its value so we know what the escape character of this Dockerfile is
										capture = true;
									} else {
										// unknown directive found, stop searching
										break escapeCheck;
									}
									break;
								default:
									if (capture) {
										// the escape directive should be a single character and followed by whitespace,
										// it should also either be a backslash or a backtick
										if ((j + 1 === buffer.length || isWhitespace(buffer.charAt(j + 1)))
												&& (char === '\\' || char === '`')) {
											escapeCharacter = char;
										}
										break escapeCheck;
									}
									directive = directive + char.toLowerCase();
									break;
							}
						}
						break directiveCheck;
					case ' ':
					case '\t':
						// ignore whitespace
						continue;
					case '\r':
					case '\n':
						break directiveCheck;
					default:
						// not a comment then not a directive
						break directiveCheck;
				}
			}

			// start from the offset and walk back
			commentCheck: for (i = offset - 1; i >= 0; i--) {
				switch (buffer.charAt(i)) {
					case '#':
						if (i === firstCommentIdx) {
							// we're in the first comment, might need to suggest
							// the escape directive as a proposal
							var directivePrefix = buffer.substring(i + 1, offset).trimLeft().toLowerCase();
							if ("escape".indexOf(directivePrefix) === 0) {
								return [ createEscape(context.prefix, offset - context.prefix.length, this.markdowns["escape"]) ];
							}
						}
						// in a comment, no proposals to suggest
						return [];
					case ' ':
					case '\t':
						// ignore whitespace
						continue;
					case '\r':
					case '\n':
						// walked back to the beginning of this line, not in a comment
						break commentCheck;
				}
			}

			// get every line in the file
			var split = buffer.trim().split("\n");
			var fromOnly = split.some(function(line) {
				var trimmed = line.trim();
				// check if it's a comment or an empty line
				return trimmed.length !== 0 && trimmed.charAt(0) !== '#';
			});
			if (!fromOnly) {
				// if we only have empty lines and comments, only suggest FROM
				return [ createFROM(context.prefix, offset, this.markdowns["FROM"]) ];
			}
			
			if (context.prefix === "") {
				context.prefix = calculateTruePrefix(buffer, offset, escapeCharacter);
			}

			var previousWord = "";
			var whitespace = false;
			var lineStart = 0;
			lineCheck: for (i = offset - 1; i >= 0; i--) {
				char = buffer.charAt(i);
				switch (char) {
					case '\n':
						if (buffer.charAt(i - 1) === escapeCharacter) {
							i--;
							continue;
						} else if (buffer.charAt(i - 1) === '\r' && buffer.charAt(i - 2) === escapeCharacter) {
							i = i - 2;
							continue;
						}

						if (previousWord !== "" && previousWord !== "ONBUILD") {
							// keyword content assist only allowed after an ONBUILD
							return [];
						}
						lineStart = i + 1;
						break lineCheck;
					case ' ':
					case '\t':
						if (whitespace) {
							if (previousWord !== "" && previousWord !== "ONBUILD") {
								// keyword content assist only allowed after an ONBUILD
								return [];
							}
						} else {
							whitespace = true;
						}
						break;
					default:
						if (whitespace) {
							previousWord = char.toUpperCase() + previousWord;
						}
						break;
				}
			}

			if (previousWord !== "" && previousWord !== "ONBUILD") {
				// only suggest proposals if at the front or after an ONBUILD
				return [];
			}

			var proposals = [];
			if (context.prefix === "") {
				createProposals(proposals, this.keywords, previousWord, "", offset, this.markdowns);
				return proposals;
			}

			var suggestions = [];
			var uppercasePrefix = context.prefix.toUpperCase();
			for (i = 0; i < this.keywords.length; i++) {
				if (this.keywords[i] === uppercasePrefix) {
					// prefix is a keyword already, nothing to suggest
					return [];
				} else if (this.keywords[i].indexOf(uppercasePrefix) === 0) {
					suggestions.push(this.keywords[i]);
				}
			}

			if (suggestions.length === 0) {
				// prefix doesn't match any keywords, nothing to suggest
				return [];
			}

			if (lineStart + context.line.indexOf(context.prefix) + context.prefix.length === offset) {
				createProposals(proposals, suggestions, previousWord, context.prefix, offset, this.markdowns);
				return proposals;
			}
			return [];
		}
	});

	function createProposals(proposals, keywords, previousWord, prefix, offset, markdowns) {
		for (var i = 0; i < keywords.length; i++) {
			switch (keywords[i]) {
				case "ARG":
					proposals.push(createARG_NameOnly(prefix, offset - prefix.length));
					proposals.push(createARG_DefaultValue(prefix, offset - prefix.length));
					break;
				case "HEALTHCHECK":
					proposals.push(createHEALTHCHECK_CMD(prefix, offset - prefix.length));
					proposals.push(createHEALTHCHECK_NONE(prefix, offset - prefix.length));
					break;
				default:
					proposals.push(createSingleProposals(keywords[i], prefix, offset, markdowns));
					break;
			}
		}

		if (previousWord === "ONBUILD") {
			// can't have FROM, MAINTAINER, or ONBUILD follow an ONBUILD
			for (i = 0; i < proposals.length; i++) {
				switch (proposals[i].name) {
					case "FROM":
					case "MAINTAINER":
					case "ONBUILD":
						proposals.splice(i, 1);
						i--;
						break;
				}
			}
		}
	}

	function isWhitespace(char) {
		return char === ' ' || char === '\t' || char === '\r' || char === '\n';
	}

	/**
	 * Walks back in the text buffer to calculate the true prefix of the
	 * current text caret offset. Orion's provided prefix does not include
	 * symbols but we do want to consider that a prefix in Dockerfiles.
	 * 
	 * @param buffer the content of the opened file
	 * @param offset the current text caret's offset
	 * @param escapeCharacter the escape character defined in this Dockerfile
	 */
	function calculateTruePrefix(buffer, offset, escapeCharacter) {
		var char = buffer.charAt(offset - 1);
		switch (char) {
			case '\n':
				var escapedPrefix = "";
				for (var i = offset - 1; i >= 0; i--) {
					if (buffer.charAt(i) === '\n') {
						if (buffer.charAt(i - 1) === escapeCharacter) {
							i--;
						} else if (buffer.charAt(i - 1) === '\r' && buffer.charAt(i - 2) === escapeCharacter) {
							i = i -2;
						} else {
							break;
						}
					} else if (buffer.charAt(i) === ' ' || buffer.charAt(i) === '\t') {
						break;
					} else {
						escapedPrefix = buffer.charAt(i).toUpperCase() + escapedPrefix;
					}
				}
				
				if (escapedPrefix !== "") {
					return escapedPrefix;
				}
				break;
			case '\r':
			case ' ':
			case '\t':
				break;
			default:
				var truePrefix = char;
				prefixCheck: for (i = offset - 2; i >= 0; i--) {
					char = buffer.charAt(i);
					switch (char) {
						case '\r':
						case '\n':
						case ' ':
						case '\t':
							break prefixCheck;
						default:
							for (i = offset - 2; i >= 0; i--) {
								truePrefix = char + truePrefix;
							}
							break;
					}
				}
				return truePrefix;
		}
		return "";
	}

	function createSingleProposals(keyword, prefix, offset, markdowns) {
		switch (keyword) {
			case "ADD":
				return createADD(prefix, offset - prefix.length, markdowns[keyword]);
			case "CMD":
				return createCMD(prefix, offset - prefix.length, markdowns[keyword]);
			case "COPY":
				return createCOPY(prefix, offset - prefix.length, markdowns[keyword]);
			case "ENTRYPOINT":
				return createENTRYPOINT(prefix, offset - prefix.length, markdowns[keyword]);
			case "ENV":
				return createENV(prefix, offset - prefix.length, markdowns[keyword]);
			case "EXPOSE":
				return createEXPOSE(prefix, offset - prefix.length, markdowns[keyword]);
			case "FROM":
				return createFROM(prefix, offset - prefix.length, markdowns[keyword]);
			case "LABEL":
				return createLABEL(prefix, offset - prefix.length, markdowns[keyword]);
			case "MAINTAINER":
				return createMAINTAINER(prefix, offset - prefix.length, markdowns[keyword]);
			case "ONBUILD":
				return createONBUILD(prefix, offset - prefix.length, markdowns[keyword]);
			case "RUN":
				return createRUN(prefix, offset - prefix.length, markdowns[keyword]);
			case "SHELL":
				return createSHELL(prefix, offset - prefix.length, markdowns[keyword]);
			case "STOPSIGNAL":
				return createSTOPSIGNAL(prefix, offset - prefix.length, markdowns[keyword]);
			case "WORKDIR":
				return createWORKDIR(prefix, offset - prefix.length, markdowns[keyword]);
			case "VOLUME":
				return createVOLUME(prefix, offset - prefix.length, markdowns[keyword]);
			case "USER":
				return createUSER(prefix, offset - prefix.length, markdowns[keyword]);
		}
	}

	function createADD(prefix, offset, markdown) {
		return {
			name: "ADD",
			description: " source dest",
			proposal: "ADD source dest",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'source'
				{
					offset: offset + 4,
					length: 6
				},
				// linked mode for 'dest'
				{
					offset: offset + 11,
					length: 4
				}
			],
			escapePosition: offset + 15,
			hover: markdown
		};
	}

	function createARG_NameOnly(prefix, offset) {
		return {
			name: "ARG",
			description: " name",
			proposal: "ARG name",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'name'
				{
					offset: offset + 4,
					length: 4
				}
			],
			escapePosition: offset + 8,
			hover: {
				type: "markdown",
				content: dockerMessages["proposalArgNameOnly"] +
					"```\n" +
					"ARG userName\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#arg")
			}
		};
	}

	function createARG_DefaultValue(prefix, offset) {
		return {
			name: "ARG",
			description: " name=defaultValue",
			proposal: "ARG name=defaultValue",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'name'
				{
					offset: offset + 4,
					length: 4
				},
				// linked mode for 'defaultValue'
				{
					offset: offset + 9,
					length: 12
				}
			],
			escapePosition: offset + 21,
			hover: {
				type: "markdown",
				content: dockerMessages["proposalArgDefaultValue"] +
					"```\n" +
					"ARG testOutputDir=test\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#arg")
			}
		};
	}

	function createCMD(prefix, offset, markdown) {
		return {
			name: "CMD",
			description: " [ \"executable\" ]",
			proposal: "CMD [ \"executable\" ]",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'executable'
				{
					offset: offset + 7,
					length: 10
				}
			],
			escapePosition: offset + 20,
			hover: markdown
		};
	}

	function createCOPY(prefix, offset, markdown) {
		return {
			name: "COPY",
			description: " source dest",
			proposal: "COPY source dest",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'source'
				{
					offset: offset + 5,
					length: 6
				},
				// linked mode for 'dest'
				{
					offset: offset + 12,
					length: 4
				}
			],
			escapePosition: offset + 16,
			hover: markdown
		};
	}

	function createENTRYPOINT(prefix, offset, markdown) {
		return {
			name: "ENTRYPOINT",
			description: " [ \"executable\" ]",
			proposal: "ENTRYPOINT [ \"executable\" ]",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'executable'
				{
					offset: offset + 14,
					length: 10
				}
			],
			escapePosition: offset + 27,
			hover: markdown
		};
	}

	function createENV(prefix, offset, markdown) {
		return {
			name: "ENV",
			description: " key=value",
			proposal: "ENV key=value",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'key'
				{
					offset: offset + 4,
					length: 3
				},
				// linked mode for 'value'
				{
					offset: offset + 8,
					length: 5
				}
			],
			escapePosition: offset + 13,
			hover: markdown
		};
	}

	function createEXPOSE(prefix, offset, markdown) {
		return {
			name: "EXPOSE",
			description: " port",
			proposal: "EXPOSE port",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'port'
				{
					offset: offset + 7,
					length: 4
				}
			],
			escapePosition: offset + 11,
			hover: markdown
		};
	}

	function createFROM(prefix, offset, markdown) {
		return {
			name: "FROM",
			description: " baseImage",
			proposal: "FROM baseImage",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'baseImage'
				{
					offset: offset + 5,
					length: 9
				}
			],
			escapePosition: offset + 14,
			hover: markdown
		};
	}

	function createHEALTHCHECK_CMD(prefix, offset) {
		return {
			name: "HEALTHCHECK",
			description: " --interval=30s --timeout=30s --retries=3 CMD [ \"executable\" ]",
			proposal: "HEALTHCHECK --interval=30s --timeout=30s --retries=3 CMD [ \"executable\" ]",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'interval'
				{
					offset: offset + 23,
					length: 3
				},
				// linked mode for 'timeout'
				{
					offset: offset + 37,
					length: 3
				},
				// linked mode for 'retries'
				{
					offset: offset + 51,
					length: 1
				},
				// linked mode for 'executable'
				{
					offset: offset + 60,
					length: 10
				}
			],
			escapePosition: offset + 73,
			hover: {
				type: "markdown",
				content: dockerMessages["proposalHealthcheckExec"] +
				"```\n" +
				"HEALTHCHECK --interval=10m --timeout=5s \\\n" +
				"    CMD curl -f http://localhost/ || exit 1\n" +
				"```" + 
				i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#healthcheck")
			}
		};
	}

	function createHEALTHCHECK_NONE(prefix, offset) {
		return {
			name: "HEALTHCHECK",
			description: " NONE",
			proposal: "HEALTHCHECK NONE",
			prefix: prefix,
			positions: [],
			overwrite: true,
			escapePosition: offset + 16,
			hover: {
				type: "markdown",
				content: dockerMessages["proposalHealthcheckNone"] +
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#healthcheck")
			}
		};
	}

	function createLABEL(prefix, offset, markdown) {
		return {
			name: "LABEL",
			description: " key=\"value\"",
			proposal: "LABEL key=\"value\"",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'key'
				{
					offset: offset + 6,
					length: 3
				},
				// linked mode for 'value'
				{
					offset: offset + 11,
					length: 5
				}
			],
			escapePosition: offset + 17,
			hover: markdown
		};
	}

	function createMAINTAINER(prefix, offset, markdown) {
		return {
			name: "MAINTAINER",
			description: " name",
			proposal: "MAINTAINER name",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'name'
				{
					offset: offset + 11,
					length: 4
				}
			],
			escapePosition: offset + 15,
			hover: markdown
		};
	}

	function createONBUILD(prefix, offset, markdown) {
		return {
			name: "ONBUILD",
			description: " INSTRUCTION",
			proposal: "ONBUILD INSTRUCTION",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'INSTRUCTION'
				{
					offset: offset + 8,
					length: 11
				}
			],
			escapePosition: offset + 19,
			hover: markdown
		};
	}

	function createRUN(prefix, offset, markdown) {
		return {
			name: "RUN",
			description: " command",
			proposal: "RUN command",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'command'
				{
					offset: offset + 4,
					length: 7
				}
			],
			escapePosition: offset + 11,
			hover: markdown
		};
	}

	function createSHELL(prefix, offset, markdown) {
		return {
			name: "SHELL",
			description: " [ \"executable\" ]",
			proposal: "SHELL [ \"executable\" ]",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'executable'
				{
					offset: offset + 9,
					length: 10
				}
			],
			escapePosition: offset + 22,
			hover: markdown
		};
	}

	function createSTOPSIGNAL(prefix, offset, markdown) {
		return {
			name: "STOPSIGNAL",
			description: " signal",
			proposal: "STOPSIGNAL signal",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'signal'
				{
					offset: offset + 11,
					length: 6
				}
			],
			escapePosition: offset + 17,
			hover: markdown
		};
	}

	function createUSER(prefix, offset, markdown) {
		return {
			name: "USER",
			description: " daemon",
			proposal: "USER daemon",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'daemon'
				{
					offset: offset + 5,
					length: 6
				}
			],
			escapePosition: offset + 11,
			hover: markdown
		};
	}

	function createVOLUME(prefix, offset, markdown) {
		return {
			name: "VOLUME",
			description: " [ \"/data\" ]",
			proposal: "VOLUME [ \"/data\" ]",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for '/data'
				{
					offset: offset + 10,
					length: 5
				}
			],
			escapePosition: offset + 18,
			hover: markdown
		};
	}

	function createWORKDIR(prefix, offset, markdown) {
		return {
			name: "WORKDIR",
			description: " /path",
			proposal: "WORKDIR /path",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'path'
				{
					offset: offset + 8,
					length: 5
				}
			],
			escapePosition: offset + 13,
			hover: markdown
		};
	}

	function createEscape(prefix, offset, markdown) {
		return {
			name: "escape",
			description: "=`",
			proposal: "escape=`",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for '`'
				{
					offset: offset + 7,
					length: 1
				}
			],
			escapePosition: offset + 8,
			hover: markdown
		};
	}

	return {
		DockerContentAssist : DockerContentAssist
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('plugins/languages/docker/dockerPlugin',[
	'orion/plugin',
	'orion/editor/stylers/text_x-dockerfile/syntax',
	'plugins/languages/docker/dockerAssist',
	'i18n!plugins/languages/docker/nls/messages',
	'orion/i18nUtil'
], function(PluginProvider, mDockerfile, DockerAssist, dockerMessages, i18nUtil) {

	function connect() {
		var headers = {
			name: dockerMessages['pluginName'],
			version: "1.0",
			description: dockerMessages['pluginDescription'],
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		var markdowns = {
			ADD: {
				type: "markdown",
				content: dockerMessages["hoverAdd"] +
					"```\n" +
					"ADD hello.txt /absolute/path\n" +
					"ADD hello.txt relative/to/workdir\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#add")
			},

			ARG: {
				type: "markdown",
				content: dockerMessages["hoverArg"] +
					"```\n" +
					"ARG userName\n" +
					"ARG testOutputDir=test\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#arg")
			},

			CMD: {
				type: "markdown",
				content:  dockerMessages["hoverCmd"] +
					"`CMD [ \"/bin/ls\", \"-l\" ]`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#cmd")
			},

			COPY: {
				type: "markdown",
				content:  dockerMessages["hoverCopy"] +
					"```\n" +
					"COPY hello.txt /absolute/path\n" +
					"COPY hello.txt relative/to/workdir\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#copy")
			},

			ENTRYPOINT: {
				type: "markdown",
				content:  dockerMessages["hoverEntrypoint"] +
					"`ENTRYPOINT [ \"/opt/app/run.sh\", \"--port\", \"8080\" ]`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#entrypoint")
			},

			ENV: {
				type: "markdown",
				content:  dockerMessages["hoverEnv"] +
					"`ENV buildTag=1.0`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#env")
			},

			EXPOSE: {
				type: "markdown",
				content:  dockerMessages["hoverExpose"] +
					"```\n" +
					"EXPOSE 8080\n" +
					"EXPOSE 80 443 22\n" +
					"EXPOSE 7000-8000\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#expose")
			},

			FROM: {
				type: "markdown",
				content:  dockerMessages["hoverFrom"] +
					"```\n" +
					"FROM baseImage\n" +
					"FROM baseImage:tag\n" +
					"FROM baseImage@digest\n" + 
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#from")
			},

			HEALTHCHECK: {
				type: "markdown",
				content:  dockerMessages["hoverHealthcheck"] +
				"```\n" +
				"HEALTHCHECK --interval=10m --timeout=5s \\\n" +
				"    CMD curl -f http://localhost/ || exit 1\n" +
				"HEALTHCHECK NONE\n" +
				"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#healthcheck")
			},

			LABEL: {
				type: "markdown",
				content:  dockerMessages["hoverLabel"] +
					"`LABEL version=\"1.0\"`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#label")
			},

			MAINTAINER: {
				type: "markdown",
				content: dockerMessages["hoverMaintainer"] +
					"`MAINTAINER name`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#maintainer")
			},

			ONBUILD: {
				type: "markdown",
				content: dockerMessages["hoverOnbuild"] +
					"```\n" +
					"ONBUILD ADD . /opt/app/src/extensions\n" +
					"ONBUILD RUN /usr/local/bin/build.sh /opt/app" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#cmd")
			},

			RUN: {
				type: "markdown",
				content: dockerMessages["hoverRun"] +
					"`RUN apt-get update && apt-get install -y curl`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#run")
			},

			SHELL: {
				type: "markdown",
				content: dockerMessages["hoverShell"] +
					"`SHELL [ \"powershell\", \"-command\" ]`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#shell")
			},

			STOPSIGNAL: {
				type: "markdown",
				content: dockerMessages["hoverStopsignal"] +
					"`STOPSIGNAL 9`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#stopsignal")
			},

			USER: {
				type: "markdown",
				content: dockerMessages["hoverUser"] +
					"`USER daemon`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#user")
			},

			VOLUME: {
				type: "markdown",
				content: dockerMessages["hoverVolume"] +
					"`VOLUME [ \"/var/db\" ]`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#volume")
			},

			WORKDIR: {
				type: "markdown",
				content: dockerMessages["hoverWorkdir"] +
					"```\n" +
					"WORKDIR /path/to/workdir\n" +
					"WORKDIR relative/path\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#workdir")
			},

			escape: {
				type: "markdown",
				content: dockerMessages["hoverEscape"] +
					"```\n" + 
					"# escape=`\n" +
					"```" +
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#escape")
			}
		};

		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-dockerfile",
					"extends": "text/plain",
					name: "dockerfile",
					extension: ["dockerfile"],
					filenamePattern: "^dockerfile"
				}
			] 
		});
		
		pluginProvider.registerServiceProvider("orion.edit.contentAssist", new DockerAssist.DockerContentAssist(mDockerfile.keywords, markdowns), {
			name: dockerMessages['dockerContentAssist'],
			contentType:  ["text/x-dockerfile" ]
		});

		pluginProvider.registerServiceProvider("orion.edit.hover", {
			computeHoverInfo: function (editorContext, context) {
				if (context.proposal && context.proposal.hover) {
					return context.proposal.hover;
				}

				var textLength;
				var content;
				return editorContext.getText()
				.then(function(text) {
					content = text;
					textLength = text.length;
					return editorContext.getLineAtOffset(context.offset);
				})
				.then(function(line) {
					return editorContext.getLineStart(line);
				}).then(function(lineStart) {
					for (var i = lineStart; i < textLength; i++) {
						if (content.charAt(i) === '#') {
							// might be hovering over a directive
							var directive = "";
							var directiveOffset = -1;
							var stop = false;
							for (var j = i + 1; j < textLength; j++) {
								if (content.charAt(j) === '=') {
									if (directiveOffset === -1) {
										// record the end offset for the directive if not already recorded
										directiveOffset = j;
									}
									break;
								} else if (content.charAt(j) === ' ' || content.charAt(j)  === '\t'
										|| content.charAt(j) === '\r' || content.charAt(j)  === '\n') {
									if (directive !== "" && !stop) {
										// directive has been captured, stop and record the ending offset
										directiveOffset = j;
										stop = true;
									}
									continue;
								}

								if (stop) {
									// a whitespace was encountered and we should stop capturing but
									// another character was found, so this is not a directive
									return null;
								} else {
									// capture the directive
									directive = directive + content.charAt(j);
								}
							}
							// check to make sure the user is hovering over the directive itself
							if (i <= context.offset && context.offset <= j) {
								return markdowns[directive.toLowerCase()];
							}
							return null;
						}
						// skip initial whitespace at the beginning of the line
						if (content.charAt(i) !== ' ' && content.charAt(i) !== '\t') {
							for (var j = i + 1; j < textLength; j++) {
								// find the end of the word
								if (content.charAt(j) === ' ' || content.charAt(j)  === '\t'
										|| content.charAt(j) === '\r' || content.charAt(j)  === '\n') {
									if (i <= context.offset && context.offset <= j) {
										return markdowns[content.substring(i, j).toUpperCase()];
									}
									return null;
								}
							}
							return markdowns[content.substring(i, j).toUpperCase()];
						}
					}
					return null;
				});
			}
		}, {
			name: dockerMessages['dockerContentHover'],
			contentType: [ "text/x-dockerfile" ]
		});

		mDockerfile.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-erlang/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"xor",
		"when",
		"try",
		"rem", "receive",
		"query",
		"orelse", "or", "of",
		"not",
		"let",
		"if",
		"fun",
		"end",
		"div",
		"cond", "catch", "case",
		"bxor", "bsr", "bsl", "bor", "bnot", "begin", "band",
		"andalso", "and", "after"
	];

	// For Preprocessors, Records and so on specified with hyphens
	var hyphenKeywords = [
		"vsn",
		"undef",
		"type",
		"spec",
		"record",
		"on_load", "opaque",
		"module",
		"include_lib", "include", "import", "ifndef", "ifdef",
		"file",
		"export_type", "export", "endif", "else",
		"define",
		"callback", "compile",
		"behaviour"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.erlang",
		contentTypes: ["text/x-erlang"],
		patterns: [
			{include: "#comment"},
			{include: "#stringSingleLine"},
			{include: "#stringMultiLine"},
			{include: "#method"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.erlang"
			},
			{
				match: "^\\s*-(?:" + hyphenKeywords.join("|") + ")\\b",
				name: "keyword.operator.erlang"
			}
		],
		repository: {
			comment: {
				match: {match: "%.*", literal: "%"},
				name: "comment.line.erlang",
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			method: {
				match: "(^|\\s)[a-zA-Z0-9_.]+(?=\\(|\\s\\()",
			    name: "entity.name.function.erlang"
			},
			stringMultiLine: {
				begin: '"(?:\\\\.|[^\\\\"])*$',
				end: '^(?:\\\\.|[^\\\\"])*"',
				name: "string.quoted.double.erlang"
			},
			stringSingleLine: {
				match: '"(?:\\\\.|[^\\\\"])*"',
				name: "string.quoted.double.erlang"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/erlang/erlangPlugin',['orion/plugin', 'orion/editor/stylers/text_x-erlang/syntax'], function(PluginProvider, mErlang) {

	function connect() {
		var headers = {
			name: "Orion Erlang Tool Support",
			version: "1.0",
			description: "This plugin provides Erlang tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-erlang",
					"extends": "text/plain",
					name: "Erlang",
					extension: ["erl", "hrl"]
				}
			] 
		});
		mErlang.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global escape*/
define('orion/form',[], function() {
	function x_www_form_urlencode(value) {
		return encodeURIComponent(value).replace(/[!'()*]/g, escape).replace('%20', '+'); //$NON-NLS-0$ //$NON-NLS-1$
	}

	/**
	 * @name orion.form
	 * @class Utilities for handling encoding.
	 */
	return /** @lends orion.form */ {
		/**
		 * Encodes an object of form fields and values into an <code>application/x-www-form-urlencoded</code> string.
		 * @static
		 * @param {Object} data The form data to encode.
		 * @returns {String} The <code>x-www-form-urlencoded</code> string.
		 */
		encodeFormData: function(data) {
			data = data || {};
			var paramNames = Object.keys(data);
			var buf = [];
			for (var i=0; i < paramNames.length; i++) {
				var param = paramNames[i], value = data[param];
				buf.push(x_www_form_urlencode(param) + '=' + x_www_form_urlencode(value)); //$NON-NLS-0$
			}
			return buf.join('&'); //$NON-NLS-0$
		},
		/**
		 * Encodes a string into an <a href="http://tools.ietf.org/html/rfc5023#section-9.7.1">RFC 5023</a>-compliant
		 * <tt>Slug</tt> header.
		 * @static
		 * @param {String} s The string to encode.
		 * @returns {String} The encoded <tt>Slug</tt>.
		 */
		encodeSlug: function(s) {
			return s.replace(/([^\u0020-\u007e]|%)+/g, encodeURIComponent);
		}
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10/.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-go/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"break",
		"case", "const", "continue",
		"default", "defer",
		"else",
		"fallthrough", "false", "for", "func",
		"goto", "go",
		"if", "import",
		"nil",
		"package",
		"range", "return",
		"select", "switch",
		"true", "type",
		"var"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.go",
		contentTypes: ["text/x-go"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{
				begin: "`",
				end: "`",
				name: "string.quoted.raw.go",
			},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.c-like#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.go"
			},
			{
				match: "\\b(?:len|cap|new|make|append|close|copy|delete|complex|real|imag|panic|recover)\\b",
				name: "support.function.go"
			},
			{
				match: "\\b(?:bool|chan|uint8|uint16|uint32|uint64|int8|int16|int32|int64|float32|float64|complex64|complex128|byte|map|rune|uint|interface|int|uintptr|string|struct|error)\\b",
				name: "support.function.type"
			}
		]
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser,amd*/
define('plugins/languages/go/goPlugin',['orion/plugin', 'orion/form', 'orion/editor/stylers/text_x-go/syntax'], function (PluginProvider, form, mGo) {

	function connect() {
		var headers = {
			name: "Orion Go Language Tool Support",
			version: "1.0",
			description: "This plugin provides Go language tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-go",
					"extends": "text/plain",
					name: "Go",
					extension: ["go"]
				}
			] 
		});
		mGo.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
		pluginProvider.registerServiceProvider("orion.edit.contentAssist", {
			computeProposals: function (buffer, offset, context) {
				var NL = context.delimiter;
				var INDENT = context.indentation;
				var TAB = context.tab;

				var constructs = [
					"inner", "if", "if",            "if ${cond} {"                         +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "func", "func",        "func ${name}() (${retval} ${type}) {" +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"inner", "for", "for",          "for ${cond} {"                        +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"inner", "switch", "switch",    "switch {"                             +NL+
					    INDENT+                     "case ${cond}:"                        +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "default:"                             +NL+
					    INDENT+                     "}",

					"inner", "select", "select",    "select {"                             +NL+
					    INDENT+                     "case ${cond}:"                        +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "default:"                             +NL+
					    INDENT+                     "}",

					"outer", "var", "var",          "var ("                                +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "const", "const",      "const ("                              +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "import", "import",    "import ("                             +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "", "method",          "func (this *${type}) ${name}() (${retval} ${type}) {"+NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "", "struct",          "type ${name} struct {"                +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"outer", "", "interface",       "type ${name} interface {"             +NL+
					    INDENT+                     TAB+"${cursor}"                        +NL+
					    INDENT+                     "}",

					"inner", "", "make channel",    "ch := make(chan ${type}, 0)",

					"inner", "", "make array",      "arr := make([]${type}, 1, 1)",

					"outer", "", "main",            "func main() {"                        +NL+
					    INDENT+                      TAB+"${cursor}"                       +NL+
					    INDENT+                     "}"
				];

				var proposals = [];

				if (buffer.length === 0 && offset === 0) {
					proposals.push({
						description: "new file template",
						proposal:	"// COPYRIGHT"          +NL+
									""                      +NL+
									"// GODOC"              +NL+
									"package ${name}"       +NL+
									                         NL+
									"import ("              +NL+
									TAB+"${import}"         +NL+
									")"                     +NL+
									                         NL+
									"func main() {"         +NL+
									TAB                     +NL+
									"}"                     +NL,
						escapePosition: 68 + NL.length * 10,
						positions: [{offset: 28 + NL.length * 3, length: 7}, {offset: 43 +  NL.length * 7, length: 9}]
					});

					proposals.push({
						description: "new test template",
						proposal:	"// COPYRIGHT"               +NL+
									""                           +NL+
									"package main"               +NL+
									                              NL+
									"import ("                   +NL+
									TAB+"testing"                +NL+
									")"                          +NL+
									                              NL+
									"func Test1(t *testing.T) {" +NL+
									TAB                          +NL+
									"}"                          +NL,
						escapePosition: 68 + NL.length * 9,
						positions: []
					});
				}

				for (var i = 0; i < constructs.length; i = i + 4) {
					var type = constructs[i];
					var matcher = constructs[i + 1];
					var keyword = constructs[i + 2];
					var proposal = constructs[i + 3];

					if (matcher.indexOf(context.prefix) !== 0) {
						continue;
					}

					// Check whether this proposal is an "outer" (outside of a var, const, func block)
					// or an "inner"
					if (type === "inner" && INDENT === "") {
						continue;
					}
					if (type === "outer" && INDENT !== "") {
						continue;
					}

					// Ellide the proposal with the existing prefix text where possible
					if (proposal.indexOf(context.prefix) === 0) {
						proposal = proposal.substring(context.prefix.length);
					}

					var propObj = {description: keyword, positions: []};

					// Calculate positions for the variables and the cursor
					var cursorIdx = proposal.indexOf("${cursor}");
					if (cursorIdx !== -1) {
						propObj.escapePosition = cursorIdx + offset;
						proposal = proposal.replace("${cursor}", "");
					}

					propObj.proposal = proposal;

					var idx = 0;
					while (idx !== -1 && idx < proposal.length - 4) {
						idx = proposal.indexOf("${", idx + 1);
						if (idx !== -1) {
							var off = idx + offset;
							var len = proposal.indexOf("}", idx + 1) + 1 - idx;
							if (len > 0) {
								propObj.positions.push({offset: off, length: len});
							}
						}
					}
	
					proposals.push(propObj);
				}

				return proposals;
			}
		}, {
			name: "Go content assist",
			contentType: ["text/x-go"]
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/text_x-git-ignore/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.gitignore",
		contentTypes: ["text/x-git-ignore"],
		patterns: [
			// http://git-scm.com/docs/gitignore
			{
				match: {match: "^\\s*#[^$]*", literal: "#"},
				name: "comment.line.number-sign.ignore"
			}, {
				match: "^\\s*!",
				name: "punctuation.operator.negation.ignore"
			}
		]
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/text_x-git-config/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.gitconfig",
		contentTypes: ["text/x-git-config"],
		patterns: [
			/* http://git-scm.com/docs/git-config */
			{
				match: "[;#][^$]*",
				name: "comment.line.gitconfig"
			}, {
				match: "=",
				name: "punctuation.operator.equals.gitconfig"
			}, {
				match: "^\\s*\\[[^\\]]*\\]\\s*$",
				name: "entity.section.name.gitconfig"
			}
		]
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/git/gitFilePlugin',['orion/plugin', 'orion/editor/stylers/text_x-git-ignore/syntax', 'orion/editor/stylers/text_x-git-config/syntax'], function(PluginProvider, mGitIgnore, mGitConfig) {

	function connect() {
		var headers = {
			name: "Orion Git File Support",
			version: "1.0",
			description: "This plugin provides Git file syntax highlighting for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-git-ignore",
					"extends": "text/plain",
					name: "Git Ignore",
					extension: ["gitignore"],
				}, {id: "text/x-git-config",
					"extends": "text/plain",
					name: "Git Config",
					filename: ["config"],
				}, {id: "text/x-cf-ignore",
					"extends": "text/x-git-ignore",
					name: "cf Ignore",
					extension: ["cfignore"],
				}
			] 
		});
		mGitIgnore.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
		mGitConfig.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-ruby/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"alias_method", "alias", "attr_accessor", "attr_reader", "attr_writer", "attr",
		"BEGIN",
		"class",
		"defined?", "def",
		"END", "extend",
		"gem",
		"include", "initialize", "in",
		"load",  "lambda",
		"module_function", "module",
		"new", "not",
		"public", "prepend", "private", "protected",
		"require_relative", "require",
		"undef",
		"__ENCODING__", "__END__", "__FILE__", "__LINE__"
	];
	var controlKeywords = [
		"and",
		"begin", "break",
		"case", "catch",
		"do",
		"else", "elsif", "end", "ensure",
		"fail", "for",
		"if",
		"loop",
		"next",
		"or",
		"raise", "redo", "rescue", "retry", "return",
		"then", "throw",
		"unless", "until",
		"when", "while",
		"yield"
	];
	var constants = [
		"false", "nil", "true"
	];
	var languageVariables = ["self", "super"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.ruby",
		contentTypes: ["text/x-ruby"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "#symbol_quoted_single"},
			{include: "#symbol_quoted_double"},
			{include: "#symbol"},
			{include: "#classRef"},
			{
				match: "/(?![\\s])(?:\\\\.|[^/])+/(?:[ioxmuesn]\\b)?",
				name: "string.regexp.ruby"
			}, {
				match: {match: "#.*", literal: "#"},
				name: "comment.line.number-sign.ruby",
				patterns: [
					{include: "orion.lib#todo_comment_singleLine"}
				]
			}, {
				begin: {match: "^=begin\\b", literal: "\n=begin "},
				end: {match: "^=end\\b", literal: "\n=end "},
				name: "comment.block.ruby",
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.block"}
						}
					}
				]
			},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{include: "#variable"},
			{
				match: "\\b0[bB][01]+\\b",
				name: "constant.numeric.binary.ruby"
			},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.ruby"
			},
			{
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.ruby"
			},
			{
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.ruby"
			},
			{
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.ruby"
			}
		],
		repository: {
			classRef: {
				match: "\\w+::\\w+"
			},
			symbol: {
				match: ":\\w+",
				name: "entity.name.symbol.ruby"
			},
			symbol_quoted_single: {
				match: ":'[^']*'",
				name: "entity.name.symbol.quoted.single.ruby"
			},
			symbol_quoted_double: {
				match: ':"[^"]*"',
				name: "entity.name.symbol.quoted.double.ruby"
			},
			variable: {
				match: "@\\w+",
				name: "entity.name.variable.ruby"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(constants).concat(languageVariables)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-haml/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/text_x-ruby/syntax"],
	function(mLib, mRuby) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mRuby.grammars);
	grammars.push({
		id: "orion.haml",
		contentTypes: ["text/x-haml"],
		patterns: [
			{include: "#inlineRuby"},
			{include: "#interpolatedRuby"},
			{include: "#tagWithRubySymbols"},
			{include: "#tagWithHTMLAttributes"},
			{include: "#doctype"},
			{include: "#tag"},
			{include: "#htmlComment"},
		],
		repository: {
			doctype: {
				match: "^!!!.*$",
				name: "meta.tag.doctype.haml",
			},
			htmlComment: {
				// TODO multi-line comments
				match: {match: "/[^[].*$", literal: "/"},
				name: "comment.line.html.haml"
			},
			inlineRuby: {
				// TODO end match should not eat the last content character, really need a negative look-behind
				begin: "(?:^|[^\\\\])(?:=|-|~|&==?|!==?)",
				end: "(?:^|[^,])$",
				contentName: "source.ruby.embedded.haml",
				patterns: [
					{include: "orion.ruby"},
				]
			},
			interpolatedRuby: {
				begin: "#{",
				end: "}",
				contentName: "source.ruby.embedded.haml",
				patterns: [
					{include: "orion.ruby"},
				]
			},
			tag: {
				match: "^\\s*%[^\\b]+?\\b",
				name: "meta.tag.haml",
			},
			tagWithHTMLAttributes: {
				begin: "(^\\s*)(%[^\\s(]+?)\\(",
				end: "\\)\\s*$",
				beginCaptures: {
					2: {name: "meta.tag.haml"}
				},
				patterns: [
					{
						match: "[^\\s=]+(?==)",
						name: "entity.name.attribute.html.haml"
					},
					{include: "orion.ruby#variable"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
				]
			},
			tagWithRubySymbols: {
				begin: "(^\\s*)(%[^\\b]+?)\\b{",
				end: "}\\s*$",
				beginCaptures: {
					2: {name: "meta.tag.haml"}
				},
				patterns: [
					{include: "orion.ruby#symbol"},
					{include: "orion.ruby#variable"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
				]
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/haml/hamlPlugin',['orion/plugin', 'orion/editor/stylers/text_x-haml/syntax'], function(PluginProvider, mHaml) {

	function connect() {
		var headers = {
			name: "Orion Haml Tool Support",
			version: "1.0",
			description: "This plugin provides Haml tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-haml",
					"extends": "text/plain",
					name: "Haml",
					extension: ["haml"]
				}
			] 
		});
		mHaml.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-jade/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/application_javascript/syntax"], function(mLib, mJS) {
	var keywords = [
		"&attributes",
		"block",
		"case",
		"default", "doctype",
		"each", "else", "extends",
		"for",
		"if", "include",
		"mixin",
		"unless",
		"when", "while"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push({
		id: "orion.jade",
		contentTypes: ["text/x-jade"],
		patterns: [
			{include: "#comment_singleLine"},
			{include: "#code"},
			{include: "#control"},
			{include: "#caseBranch"},
			{include: "#mixinWithParameters"},
			{include: "#mixinRefWithArguments"},
			{include: "#tagWithAttributes"},
			{include: "#interpolatedJS"},
			{include: "#interpolatedTag"},
			{include: "#mixin"},
			{include: "#mixinRef"},
			{include: "#doctype"},
			{include: "#filter"},
			{include: "#case"},
			{include: "#andAttributes"},
			{include: "#otherKeywords"},
			{include: "#tag"},
		],
		repository: {
			andAttributes: {
				match: "&attributes\\b",
				name: "keyword.operator.jade"
			},
			"case": {
				match: "(^\\s*)(case)\\b",
				captures: {
					2: {name: "keyword.control.jade"}
				}				
			},
			caseBranch: {
				begin: "(^\\s*)(when|default)\\s*",
				end: ":|$",
				contentName: "source.js.embedded.jade",
				patterns: [
					{include: "orion.js"}
				],
				beginCaptures: {
					2: {name: "keyword.control.jade"}
				},
			},
			code: {
				match: "(^\\s*- |= |!= ).*$",
				name: "source.js.embedded.jade",
				patterns: [
					{include: "orion.js"}
				]
			},
			comment_singleLine: {
				match: {match: "(^\\s*)(//).*", literal: "//"},
				name: "comment.line.double-slash.jade",
				captures: {
					2: {name: "comment.line.double-slash.start.jade"}
				},
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			control: {
				begin: "(^\\s*)(if|else( if)?|each|for|unless|while)\\b",
				end: "$",
				beginCaptures: {
					2: {name: "keyword.control.jade"}
				},
				contentName: "source.js.embedded.jade",
				patterns: [
					{include: "orion.js"}
				]
			},
			doctype: {
				match: "(^\\s*)(doctype.+$)",
				captures: {
					2: {name: "meta.tag.doctype.jade"}
				}
			},
			filter: {
				match: "(^\\s*)(:\\w+)",
				captures: {
					2: {name: "entity.other.filter.jade"}
				}
			},
			interpolatedJS: {
				begin: "(#{)",
				end: "(})",
				captures: {
					1: {name: "string.interpolated.js.jade"}
				},
				contentName: "source.js.embedded.jade",
				patterns: [
					{include: "orion.js"}
				]
			},
			interpolatedTag: {
				begin: "(#\\[)",
				end: "(\\])",
				captures: {
					1: {name: "string.interpolated.tag.jade"}
				},
				patterns: [
					{
						begin: "(\\.|\\w+)\\s*\\(",
						end: "(\\))(/)?",
						beginCaptures: {
							1: {name: "meta.tag.jade"}
						},
						endCaptures: {
							2: {name: "meta.tag.jade"}
						},
						contentName: "source.js.embedded.jade",
						patterns: [
							{include: "orion.js"}
						]
					}
				]
			},
			mixin: {
				match: "(^\\s*)(mixin)(\\s+)(\\w+)",
				captures: {
					2: {name: "keyword.operator.jade"},
					4: {name: "entity.name.mixin.jade"}
				}
			},
			mixinRef: {
				match: "(^\\s*)(\\+\\w+)",
				captures: {
					2: {name: "entity.name.mixin.jade"}
				}
			},
			mixinRefWithArguments: {
				begin: "(^\\s*)(\\+\\w+)\\s*\\(",
				end: "\\)|$",
				captures: {
					2: {name: "entity.name.mixin.jade"}
				},
				patterns: [
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
					{include: "orion.lib#number_decimal"}
				]
			},
			mixinWithParameters: {
				begin: "(^\\s*)(mixin)(\\s+)(\\w+)\\s*\\(",
				end: "\\)|$",
				beginCaptures: {
					2: {name: "keyword.operator.jade"},
					4: {name: "entity.name.mixin.jade"}
				},
				patterns: [
					{
						match: "[^\\s,]+",
						name: "variable.parameter.jade"
					}	
				]
			},
			otherKeywords: {
				match: "(^\\s*)(block|extends|include)\\b",
				captures: {
					2: {name: "keyword.operator.jade"}
				}
			},
			tag: {
				match: "(^\\s*)(\\w+|(?=\\.)|(?=#))(#\\w*|\\.\\w*)*(/?)",
				captures: {
					2: {name: "meta.tag.jade"},
					4: {name: "meta.tag.jade"}
				}
			},
			tagWithAttributes: {
				begin: "(^\\s*)(\\w+|(?=\\.)|(?=#))(#\\w*|\\.\\w*)*\\s*\\(",
				end: "(\\))(/)?",
				beginCaptures: {
					2: {name: "meta.tag.jade"}
				},
				endCaptures: {
					2: {name: "meta.tag.jade"}
				},
				contentName: "source.js.embedded.jade",
				patterns: [
					{include: "orion.js"}
				]
			}
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/jade/jadePlugin',['orion/plugin', 'orion/editor/stylers/text_x-jade/syntax'], function(PluginProvider, mJade) {

	function connect() {
		var headers = {
			name: "Orion Jade Tool Support",
			version: "1.0",
			description: "This plugin provides Jade tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-jade",
					"extends": "text/plain",
					name: "Jade",
					extension: ["jade"]
				}
			] 
		});
		mJade.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-java-source/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"abstract",
		"boolean", "byte",
		"char", "class",
		"double",
		"extends",
		"final", "float",
		"implements", "import", "instanceof", "int", "interface",
		"long",
		"native", "new",
		"package", "private", "protected", "public",
		"short", "static", "synchronized",
		"throws", "transient",
		"void", "volatile"
	];
	var controlKeywords = [
		"break",
		"case", "catch", "continue",
		"default", "do",
		"else",
		"finally", "for",
		"if",
		"return",
		"switch",
		"throw", "try",
		"while"
	];
	var constants = [
		"false", "null", "true"
	];
	var languageVariables = ["this", "super"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.java",
		contentTypes: ["text/x-java-source"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.java"
			},
			{
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.java"
			},
			{
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.java"
			},
			{
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.java"
			}
		]
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(constants).concat(languageVariables)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/application_x-jsp/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/text_x-java-source/syntax", "orion/editor/stylers/text_html/syntax"],
	function(mLib, mJava, mHTML) {

	var pageAttributeNames = [
		"autoFlush", "buffer", "contentType", "errorPage", "extends",
		"import", "info", "isErrorPage", "isThreadSafe", "language", "pageEncoding", "session"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mJava.grammars);
	grammars.push.apply(grammars, mHTML.grammars);
	grammars.push({
		id: "orion.jsp",
		contentTypes: ["application/x-jsp"],
		patterns: [
			{include: "orion.html"},
			{include: "#jspComment"},
			{include: "#jspJavaFragment"},
			{include: "#jspDirectiveInclude"},
			{include: "#jspDirectivePage"},
			{include: "#jspDirectiveTaglib"}
		],
		repository: {
			jspComment: {
				begin: {match: "<%--", literal: "<%--"},
				end: {match: "--%>", literal: "<%--"},
				name: "comment.block.jsp",
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!--%>).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.line"}
						}
					}
				]
			},
			jspDirectiveInclude: {
				begin: "<%@\\s+include(?:\\s|$)",
				end: "%>",
				captures: {
					0: {name: "entity.name.directive.include.jsp"}
				},
				patterns: [
					{
						match: "\\bfile\\b",
						name: "entity.other.attribute-name.jsp"
					},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
				]
			},
			jspDirectivePage: {
				begin: "<%@\\s+page(?:\\s|$)",
				end: "%>",
				captures: {
					0: {name: "entity.name.directive.page.jsp"}
				},
				patterns: [
					{
						match: "\\b(?:" + pageAttributeNames.join("|") + ")\\b",
						name: "entity.other.attribute-name.jsp"
					},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
				]
			},
			jspDirectiveTaglib: {
				begin: "<%@\\s+taglib(?:\\s|$)",
				end: "%>",
				captures: {
					0: {name: "entity.name.directive.taglib.jsp"}
				},
				patterns: [
					{
						match: "\\b(?:uri|prefix)\\b",
						name: "entity.other.attribute-name.jsp"
					},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
				]
			},
			jspJavaFragment: {
				begin: "<%(?:=|!)?(?:\\s|$)",
				end: "%>",
				captures: {
					0: {name: "entity.name.declaration.java"}
				},
				contentName: "source.java.embedded.jsp",
				patterns: [
					{include: "orion.java"}
				]
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/serviceregistry',["orion/Deferred", "orion/EventTarget"], function(Deferred, EventTarget) {

	/**
	 * @name orion.serviceregistry.ServiceReference
	 * @description Creates a new service reference.
	 * @class A reference to a service in the Orion service registry
	 * @param {String} serviceId The symbolic id of this service instance
	 * @param {String} name The service name
	 * @param {Object} properties A JSON object containing the service's declarative properties
	 */
	function ServiceReference(serviceId, objectClass, properties) {
		this._properties = properties || {};
		this._properties["service.id"] = serviceId;
		this._properties.objectClass = objectClass;
		this._properties["service.names"] = objectClass;
	}

	ServiceReference.prototype = /** @lends orion.serviceregistry.ServiceReference.prototype */
	{
		/**
		 * @name getPropertyKeys
		 * @description Returns the names of the declarative properties of this service.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceReference.prototype
		 * @returns the names of the declarative properties of this service
		 */
		getPropertyKeys: function() {
			var result = [];
			var name;
			for (name in this._properties) {
				if (this._properties.hasOwnProperty(name)) {
					result.push(name);
				}
			}
			return result;
		},
		/**
		 * @name getProperty
		 * @description Returns the declarative service property with the given name, or <code>undefined</code>
		 * if this service does not have such a property.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceReference.prototype
		 * @param {String} propertyName The name of the service property to return
		 * @returns The {String} property with the given name or <code>undefined</code>
		 */
		getProperty: function(propertyName) {
			return this._properties[propertyName];
		}
	};
	ServiceReference.prototype.constructor = ServiceReference;

	/**
	 * @name orion.serviceregistry.ServiceRegistration
	 * @description Creates a new service registration. This constructor is private and should only be called by the service registry.
	 * @class A reference to a registered service in the Orion service registry
	 * @param {String} serviceId The symbolic id of this service
	 * @param {String} serviceReference A reference to the service
	 * @param {Object} internalRegistry A JSON object containing the service's declarative properties
	 */
	function ServiceRegistration(serviceId, serviceReference, internalRegistry) {
		this._serviceId = serviceId;
		this._serviceReference = serviceReference;
		this._internalRegistry = internalRegistry;
	}

	ServiceRegistration.prototype = /** @lends orion.serviceregistry.ServiceRegistration.prototype */
	{
		/**
		 * @name unregister
		 * @description Unregister this service registration. Clients registered for <code>unregistering</code> service events
		 * will be notified of this change.
		 * @function
		 * @private
		 * @memberof orion.serviceregistry.ServiceRegistration.prototype
		 */
		unregister: function() {
			this._internalRegistry.unregisterService(this._serviceId);
		},

		/**
		 * @name getReference
		 * @description Returns the {@link orion.serviceregistry.ServiceReference} in this registration
		 * @function
		 * @private
		 * @memberof orion.serviceregistry.ServiceRegistration.prototype
		 * @param properties
		 * @returns the {@link orion.serviceregistry.ServiceReference} in this registration
		 * @throws An error is the service has been unregistered
		 */
		getReference: function() {
			if (!this._internalRegistry.isRegistered(this._serviceId)) {
				throw new Error("Service has already been unregistered: "+this._serviceId);
			}
			return this._serviceReference;
		},
		/**
		 * @name setProperties
		 * @description Sets the properties of this registration to the new given properties. Clients registered for <code>modified</code> service events
		 * will be notified of the change.
		 * @function
		 * @private
		 * @memberof orion.serviceregistry.ServiceRegistration.prototype
		 * @param {Object} properties
		 */
		setProperties: function(properties) {
			var oldProperties = this._serviceReference._properties;
			this._serviceReference._properties = properties || {};
			properties["service.id"] = this._serviceId;
			properties.objectClass = oldProperties.objectClass;
			properties["service.names"] = oldProperties["service.names"];
			this._internalRegistry.modifyService(this._serviceId);
		}
	};
	ServiceRegistration.prototype.constructor = ServiceRegistration;

	/**
	 * @name orion.serviceregistry.DeferredService
	 * @description Creates a new service promise to resolve the service at a later time.
	 * @class A service that is resolved later
	 * @private
	 * @param {orion.serviceregistry.ServiceReference} implementation The implementation of the service
	 * @param {Function} isRegistered A function to call to know if the service is already registered
	 */
	function DeferredService(implementation, isRegistered) {

		function _createServiceCall(methodName) {
			return function() {
					var d;
					try {
						if (!isRegistered()) {
							throw new Error("Service was unregistered");
						}
						var result = implementation[methodName].apply(implementation, Array.prototype.slice.call(arguments));
						if (result && typeof result.then === "function") {
							return result;
						} else {
							d = new Deferred();
							d.resolve(result);
						}
					} catch (e) {
							d = new Deferred();
							d.reject(e);
					}
					return d.promise;
			};
		}

		var method;
		for (method in implementation) {
			if (typeof implementation[method] === 'function') {
				this[method] = _createServiceCall(method);
			}
		}
	}

	/**
	 * @name orion.serviceregistry.ServiceEvent
	 * @description An event that is fired from the service registry. Clients must register to listen to events using 
	 * the {@link orion.serviceregistry.ServiceRegistry#addEventListener} function.
	 * <br> 
	 * There are three types of events that this registry will send:
	 * <ul>
	 * <li>modified - the service has been modified</li> 
	 * <li>registered - the service has been registered</li> 
	 * <li>unregistering - the service is unregistering</li>
	 * </ul> 
	 * @class A service event
	 * @param {String} type The type of the event, one of <code>modified</code>, <code>registered</code> or <code>unregistered</code>
	 * @param {orion.serviceregistry.ServiceReference} serviceReference the service reference the event is for
	 */
	function ServiceEvent(type, serviceReference) {
		this.type = type;
		this.serviceReference = serviceReference;
	}

	/**
	 * @name orion.serviceregistry.ServiceRegistry
	 * @description Creates a new service registry
	 * @class The Orion service registry
	 */
	function ServiceRegistry() {
		this._entries = [];
		this._namedReferences = {};
		this._serviceEventTarget = new EventTarget();
		var _this = this;
		this._internalRegistry = {
			/**
			 * @name isRegistered
			 * @description Returns if the service with the given identifier is registered or not.
			 * @function
			 * @private
			 * @memberof orion.serviceregistry.ServiceRegistry
			 * @param {String} serviceId the identifier of the service
			 * @returns <code>true</code> if the service with the given identifier is registered, <code>false</code> otherwise
			 */
			isRegistered: function(serviceId) {
				return _this._entries[serviceId] ? true : false;
			},
			
			/**
			 * @name unregisterService
			 * @description Unregisters a service with the given identifier. This function will notify
			 * clients registered for <code>unregistering</code> service events.
			 * @function
			 * @private
			 * @memberof orion.serviceregistry.ServiceRegistry
			 * @param {String} serviceId the identifier of the service
			 * @throws An error if the service has already been unregistered
			 * @see orion.serviceregistry.ServiceEvent
			 */
			unregisterService: function(serviceId) {
				var entry = _this._entries[serviceId];
				if (!entry) {
					throw new Error("Service has already been unregistered: "+serviceId);
				}
				var reference = entry.reference;
				_this._serviceEventTarget.dispatchEvent(new ServiceEvent("unregistering", reference));
				_this._entries[serviceId] = null;
				var objectClass = reference.getProperty("objectClass");
				objectClass.forEach(function(type) {
					var namedReferences = _this._namedReferences[type];
					for (var i = 0; i < namedReferences.length; i++) {
						if (namedReferences[i] === reference) {
							if (namedReferences.length === 1) {
								delete _this._namedReferences[type];
							} else {
								namedReferences.splice(i, 1);
							}
							break;
						}
					}
				});
			},
			/**
			 * @name modifyService
			 * @description Notifies that the service with the given identifier has been modified. This function will notify clients
			 * registered for <code>modified</code> service events.
			 * @function
			 * @private
			 * @memberof orion.serviceregistry.ServiceRegistry
			 * @param {String} serviceId the identifier of the service
			 * @throws An error if the service for the given identifier does not exist
			 * @see orion.serviceregistry.ServiceEvent
			 */
			modifyService: function(serviceId) {
				var entry = _this._entries[serviceId];
				if (!entry) {
					throw new Error("Service not found while trying to send modified event: "+serviceId);
				}
				var reference = entry.reference;
				_this._serviceEventTarget.dispatchEvent(new ServiceEvent("modified", reference));
			}
		};
	}

	ServiceRegistry.prototype = /** @lends orion.serviceregistry.ServiceRegistry.prototype */
	{
		/**
		 * @name getService
		 * @description Returns the service with the given name or reference.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String|orion.serviceregistry.ServiceReference} nameOrServiceReference The service name or a service reference
		 * @returns {orion.serviceregistry.ServiceReference|null} The service implementation, or <code>null</code> if no such service was found.
		 */
		getService: function(typeOrServiceReference) {
			var service;
			if (typeof typeOrServiceReference === "string") {
				var references = this._namedReferences[typeOrServiceReference];
				if (references) {
					references.some(function(reference) {
						service = this._entries[reference.getProperty("service.id")].service;
						return !!service;
					}, this);
				}
			} else {
				var entry = this._entries[typeOrServiceReference.getProperty("service.id")];
				if (entry) {
					service = entry.service;
				}
			}
			return service || null;
		},

		/**
		 * @name getServiceReferences
		 * @description Returns all references to the service with the given name.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String} name The name of the service to return
		 * @returns {orion.serviceregistry.ServiceReference[]} An array of service references
		 */
		getServiceReferences: function(name) {
			if (name) {
				return this._namedReferences[name] ? this._namedReferences[name] : [];
			}
			var result = [];
			this._entries.forEach(function(entry) {
				if (entry) {
					result.push(entry.reference);
				}
			});
			return result;
		},
		
		/**
		 * @name registerService
		 * @description Registers a service with this registry. This function will notify clients registered
		 * for <code>registered</code> service events.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String|String[]} names the name or names of the service being registered
		 * @param {Object} service The service implementation
		 * @param {Object} properties A JSON collection of declarative service properties
		 * @returns {orion.serviceregistry.ServiceRegistration} A service registration object for the service.
		 * @see orion.serviceregistry.ServiceEvent
		 */
		registerService: function(names, service, properties) {
			if (typeof service === "undefined" ||  service === null) {
				throw new Error("invalid service");
			}

			if (typeof names === "string") {
				names = [names];
			} else if (!Array.isArray(names)) {
				names = [];
			}

			var serviceId = this._entries.length;
			var reference = new ServiceReference(serviceId, names, properties);
			var namedReferences = this._namedReferences;
			names.forEach(function(name) {
				namedReferences[name] = namedReferences[name] || [];
				namedReferences[name].push(reference);
			});
			var deferredService = new DeferredService(service, this._internalRegistry.isRegistered.bind(null, serviceId));
			this._entries.push({
				reference: reference,
				service: deferredService
			});
			var internalRegistry = this._internalRegistry;
			this._serviceEventTarget.dispatchEvent(new ServiceEvent("registered", reference));
			return new ServiceRegistration(serviceId, reference, internalRegistry);
		},

		/**
		 * @name addEventListener
		 * @description Adds a listener for events on this registry.
		 * <br> 
		 * The events that this registry notifies about:
		 * <ul>
		 * <li>modified - the service has been modified</li> 
		 * <li>registered - the service has been registered</li> 
		 * <li>unregistering - the service is unregistering</li> 
		 * </ul> 
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String} eventName The name of the event to be notified about.
		 * @param {Function} listener The listener to add
		 * @see orion.serviceregistry.ServiceEvent
		 */
		addEventListener: function(eventName, listener) {
			this._serviceEventTarget.addEventListener(eventName, listener);
		},

		/**
		 * @name removeEventListener
		 * @description Removes a listener for service events in this registry.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String} eventName The name of the event to stop listening for
		 * @param {Function} listener The listener to remove
		 * @see orion.serviceregistry.ServiceEvent
		 */
		removeEventListener: function(eventName, listener) {
			this._serviceEventTarget.removeEventListener(eventName, listener);
		}
	};
	ServiceRegistry.prototype.constructor = ServiceRegistry;

	//return the module exports
	return {
		ServiceRegistry: ServiceRegistry
	};
});
!function(t,e){"object"==typeof exports&&"object"==typeof module?module.exports=e():"function"==typeof define&&define.amd?define('socketio/socket.io',[],e):"object"==typeof exports?exports.io=e():t.io=e()}(this,function(){return function(t){function e(r){if(n[r])return n[r].exports;var o=n[r]={exports:{},id:r,loaded:!1};return t[r].call(o.exports,o,o.exports,e),o.loaded=!0,o.exports}var n={};return e.m=t,e.c=n,e.p="",e(0)}([function(t,e,n){"use strict";function r(t,e){"object"===("undefined"==typeof t?"undefined":o(t))&&(e=t,t=void 0),e=e||{};var n,r=i(t),s=r.source,u=r.id,h=r.path,f=p[u]&&h in p[u].nsps,l=e.forceNew||e["force new connection"]||!1===e.multiplex||f;return l?(c("ignoring socket cache for %s",s),n=a(s,e)):(p[u]||(c("new io instance for %s",s),p[u]=a(s,e)),n=p[u]),r.query&&!e.query&&(e.query=r.query),n.socket(r.path,e)}var o="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function(t){return typeof t}:function(t){return t&&"function"==typeof Symbol&&t.constructor===Symbol&&t!==Symbol.prototype?"symbol":typeof t},i=n(1),s=n(7),a=n(13),c=n(3)("socket.io-client");t.exports=e=r;var p=e.managers={};e.protocol=s.protocol,e.connect=r,e.Manager=n(13),e.Socket=n(39)},function(t,e,n){(function(e){"use strict";function r(t,n){var r=t;n=n||e.location,null==t&&(t=n.protocol+"//"+n.host),"string"==typeof t&&("/"===t.charAt(0)&&(t="/"===t.charAt(1)?n.protocol+t:n.host+t),/^(https?|wss?):\/\//.test(t)||(i("protocol-less url %s",t),t="undefined"!=typeof n?n.protocol+"//"+t:"https://"+t),i("parse %s",t),r=o(t)),r.port||(/^(http|ws)$/.test(r.protocol)?r.port="80":/^(http|ws)s$/.test(r.protocol)&&(r.port="443")),r.path=r.path||"/";var s=r.host.indexOf(":")!==-1,a=s?"["+r.host+"]":r.host;return r.id=r.protocol+"://"+a+":"+r.port,r.href=r.protocol+"://"+a+(n&&n.port===r.port?"":":"+r.port),r}var o=n(2),i=n(3)("socket.io-client:url");t.exports=r}).call(e,function(){return this}())},function(t,e){var n=/^(?:(?![^:@]+:[^:@\/]*@)(http|https|ws|wss):\/\/)?((?:(([^:@]*)(?::([^:@]*))?)?@)?((?:[a-f0-9]{0,4}:){2,7}[a-f0-9]{0,4}|[^:\/?#]*)(?::(\d*))?)(((\/(?:[^?#](?![^?#\/]*\.[^?#\/.]+(?:[?#]|$)))*\/?)?([^?#\/]*))(?:\?([^#]*))?(?:#(.*))?)/,r=["source","protocol","authority","userInfo","user","password","host","port","relative","path","directory","file","query","anchor"];t.exports=function(t){var e=t,o=t.indexOf("["),i=t.indexOf("]");o!=-1&&i!=-1&&(t=t.substring(0,o)+t.substring(o,i).replace(/:/g,";")+t.substring(i,t.length));for(var s=n.exec(t||""),a={},c=14;c--;)a[r[c]]=s[c]||"";return o!=-1&&i!=-1&&(a.source=e,a.host=a.host.substring(1,a.host.length-1).replace(/;/g,":"),a.authority=a.authority.replace("[","").replace("]","").replace(/;/g,":"),a.ipv6uri=!0),a}},function(t,e,n){(function(r){function o(){return!("undefined"==typeof window||!window.process||"renderer"!==window.process.type)||("undefined"!=typeof document&&document.documentElement&&document.documentElement.style&&document.documentElement.style.WebkitAppearance||"undefined"!=typeof window&&window.console&&(window.console.firebug||window.console.exception&&window.console.table)||"undefined"!=typeof navigator&&navigator.userAgent&&navigator.userAgent.toLowerCase().match(/firefox\/(\d+)/)&&parseInt(RegExp.$1,10)>=31||"undefined"!=typeof navigator&&navigator.userAgent&&navigator.userAgent.toLowerCase().match(/applewebkit\/(\d+)/))}function i(t){var n=this.useColors;if(t[0]=(n?"%c":"")+this.namespace+(n?" %c":" ")+t[0]+(n?"%c ":" ")+"+"+e.humanize(this.diff),n){var r="color: "+this.color;t.splice(1,0,r,"color: inherit");var o=0,i=0;t[0].replace(/%[a-zA-Z%]/g,function(t){"%%"!==t&&(o++,"%c"===t&&(i=o))}),t.splice(i,0,r)}}function s(){return"object"==typeof console&&console.log&&Function.prototype.apply.call(console.log,console,arguments)}function a(t){try{null==t?e.storage.removeItem("debug"):e.storage.debug=t}catch(n){}}function c(){var t;try{t=e.storage.debug}catch(n){}return!t&&"undefined"!=typeof r&&"env"in r&&(t=r.env.DEBUG),t}function p(){try{return window.localStorage}catch(t){}}e=t.exports=n(5),e.log=s,e.formatArgs=i,e.save=a,e.load=c,e.useColors=o,e.storage="undefined"!=typeof chrome&&"undefined"!=typeof chrome.storage?chrome.storage.local:p(),e.colors=["lightseagreen","forestgreen","goldenrod","dodgerblue","darkorchid","crimson"],e.formatters.j=function(t){try{return JSON.stringify(t)}catch(e){return"[UnexpectedJSONParseError]: "+e.message}},e.enable(c())}).call(e,n(4))},function(t,e){function n(){throw new Error("setTimeout has not been defined")}function r(){throw new Error("clearTimeout has not been defined")}function o(t){if(u===setTimeout)return setTimeout(t,0);if((u===n||!u)&&setTimeout)return u=setTimeout,setTimeout(t,0);try{return u(t,0)}catch(e){try{return u.call(null,t,0)}catch(e){return u.call(this,t,0)}}}function i(t){if(h===clearTimeout)return clearTimeout(t);if((h===r||!h)&&clearTimeout)return h=clearTimeout,clearTimeout(t);try{return h(t)}catch(e){try{return h.call(null,t)}catch(e){return h.call(this,t)}}}function s(){y&&l&&(y=!1,l.length?d=l.concat(d):m=-1,d.length&&a())}function a(){if(!y){var t=o(s);y=!0;for(var e=d.length;e;){for(l=d,d=[];++m<e;)l&&l[m].run();m=-1,e=d.length}l=null,y=!1,i(t)}}function c(t,e){this.fun=t,this.array=e}function p(){}var u,h,f=t.exports={};!function(){try{u="function"==typeof setTimeout?setTimeout:n}catch(t){u=n}try{h="function"==typeof clearTimeout?clearTimeout:r}catch(t){h=r}}();var l,d=[],y=!1,m=-1;f.nextTick=function(t){var e=new Array(arguments.length-1);if(arguments.length>1)for(var n=1;n<arguments.length;n++)e[n-1]=arguments[n];d.push(new c(t,e)),1!==d.length||y||o(a)},c.prototype.run=function(){this.fun.apply(null,this.array)},f.title="browser",f.browser=!0,f.env={},f.argv=[],f.version="",f.versions={},f.on=p,f.addListener=p,f.once=p,f.off=p,f.removeListener=p,f.removeAllListeners=p,f.emit=p,f.prependListener=p,f.prependOnceListener=p,f.listeners=function(t){return[]},f.binding=function(t){throw new Error("process.binding is not supported")},f.cwd=function(){return"/"},f.chdir=function(t){throw new Error("process.chdir is not supported")},f.umask=function(){return 0}},function(t,e,n){function r(t){var n,r=0;for(n in t)r=(r<<5)-r+t.charCodeAt(n),r|=0;return e.colors[Math.abs(r)%e.colors.length]}function o(t){function n(){if(n.enabled){var t=n,r=+new Date,o=r-(p||r);t.diff=o,t.prev=p,t.curr=r,p=r;for(var i=new Array(arguments.length),s=0;s<i.length;s++)i[s]=arguments[s];i[0]=e.coerce(i[0]),"string"!=typeof i[0]&&i.unshift("%O");var a=0;i[0]=i[0].replace(/%([a-zA-Z%])/g,function(n,r){if("%%"===n)return n;a++;var o=e.formatters[r];if("function"==typeof o){var s=i[a];n=o.call(t,s),i.splice(a,1),a--}return n}),e.formatArgs.call(t,i);var c=n.log||e.log||console.log.bind(console);c.apply(t,i)}}return n.namespace=t,n.enabled=e.enabled(t),n.useColors=e.useColors(),n.color=r(t),"function"==typeof e.init&&e.init(n),n}function i(t){e.save(t),e.names=[],e.skips=[];for(var n=("string"==typeof t?t:"").split(/[\s,]+/),r=n.length,o=0;o<r;o++)n[o]&&(t=n[o].replace(/\*/g,".*?"),"-"===t[0]?e.skips.push(new RegExp("^"+t.substr(1)+"$")):e.names.push(new RegExp("^"+t+"$")))}function s(){e.enable("")}function a(t){var n,r;for(n=0,r=e.skips.length;n<r;n++)if(e.skips[n].test(t))return!1;for(n=0,r=e.names.length;n<r;n++)if(e.names[n].test(t))return!0;return!1}function c(t){return t instanceof Error?t.stack||t.message:t}e=t.exports=o.debug=o["default"]=o,e.coerce=c,e.disable=s,e.enable=i,e.enabled=a,e.humanize=n(6),e.names=[],e.skips=[],e.formatters={};var p},function(t,e){function n(t){if(t=String(t),!(t.length>100)){var e=/^((?:\d+)?\.?\d+) *(milliseconds?|msecs?|ms|seconds?|secs?|s|minutes?|mins?|m|hours?|hrs?|h|days?|d|years?|yrs?|y)?$/i.exec(t);if(e){var n=parseFloat(e[1]),r=(e[2]||"ms").toLowerCase();switch(r){case"years":case"year":case"yrs":case"yr":case"y":return n*u;case"days":case"day":case"d":return n*p;case"hours":case"hour":case"hrs":case"hr":case"h":return n*c;case"minutes":case"minute":case"mins":case"min":case"m":return n*a;case"seconds":case"second":case"secs":case"sec":case"s":return n*s;case"milliseconds":case"millisecond":case"msecs":case"msec":case"ms":return n;default:return}}}}function r(t){return t>=p?Math.round(t/p)+"d":t>=c?Math.round(t/c)+"h":t>=a?Math.round(t/a)+"m":t>=s?Math.round(t/s)+"s":t+"ms"}function o(t){return i(t,p,"day")||i(t,c,"hour")||i(t,a,"minute")||i(t,s,"second")||t+" ms"}function i(t,e,n){if(!(t<e))return t<1.5*e?Math.floor(t/e)+" "+n:Math.ceil(t/e)+" "+n+"s"}var s=1e3,a=60*s,c=60*a,p=24*c,u=365.25*p;t.exports=function(t,e){e=e||{};var i=typeof t;if("string"===i&&t.length>0)return n(t);if("number"===i&&isNaN(t)===!1)return e["long"]?o(t):r(t);throw new Error("val is not a non-empty string or a valid number. val="+JSON.stringify(t))}},function(t,e,n){function r(){}function o(t){var n=""+t.type;return e.BINARY_EVENT!==t.type&&e.BINARY_ACK!==t.type||(n+=t.attachments+"-"),t.nsp&&"/"!==t.nsp&&(n+=t.nsp+","),null!=t.id&&(n+=t.id),null!=t.data&&(n+=JSON.stringify(t.data)),h("encoded %j as %s",t,n),n}function i(t,e){function n(t){var n=d.deconstructPacket(t),r=o(n.packet),i=n.buffers;i.unshift(r),e(i)}d.removeBlobs(t,n)}function s(){this.reconstructor=null}function a(t){var n=0,r={type:Number(t.charAt(0))};if(null==e.types[r.type])return u();if(e.BINARY_EVENT===r.type||e.BINARY_ACK===r.type){for(var o="";"-"!==t.charAt(++n)&&(o+=t.charAt(n),n!=t.length););if(o!=Number(o)||"-"!==t.charAt(n))throw new Error("Illegal attachments");r.attachments=Number(o)}if("/"===t.charAt(n+1))for(r.nsp="";++n;){var i=t.charAt(n);if(","===i)break;if(r.nsp+=i,n===t.length)break}else r.nsp="/";var s=t.charAt(n+1);if(""!==s&&Number(s)==s){for(r.id="";++n;){var i=t.charAt(n);if(null==i||Number(i)!=i){--n;break}if(r.id+=t.charAt(n),n===t.length)break}r.id=Number(r.id)}return t.charAt(++n)&&(r=c(r,t.substr(n))),h("decoded %s as %j",t,r),r}function c(t,e){try{t.data=JSON.parse(e)}catch(n){return u()}return t}function p(t){this.reconPack=t,this.buffers=[]}function u(){return{type:e.ERROR,data:"parser error"}}var h=n(3)("socket.io-parser"),f=n(8),l=n(9),d=n(11),y=n(12);e.protocol=4,e.types=["CONNECT","DISCONNECT","EVENT","ACK","ERROR","BINARY_EVENT","BINARY_ACK"],e.CONNECT=0,e.DISCONNECT=1,e.EVENT=2,e.ACK=3,e.ERROR=4,e.BINARY_EVENT=5,e.BINARY_ACK=6,e.Encoder=r,e.Decoder=s,r.prototype.encode=function(t,n){if(t.type!==e.EVENT&&t.type!==e.ACK||!l(t.data)||(t.type=t.type===e.EVENT?e.BINARY_EVENT:e.BINARY_ACK),h("encoding packet %j",t),e.BINARY_EVENT===t.type||e.BINARY_ACK===t.type)i(t,n);else{var r=o(t);n([r])}},f(s.prototype),s.prototype.add=function(t){var n;if("string"==typeof t)n=a(t),e.BINARY_EVENT===n.type||e.BINARY_ACK===n.type?(this.reconstructor=new p(n),0===this.reconstructor.reconPack.attachments&&this.emit("decoded",n)):this.emit("decoded",n);else{if(!y(t)&&!t.base64)throw new Error("Unknown type: "+t);if(!this.reconstructor)throw new Error("got binary data when not reconstructing a packet");n=this.reconstructor.takeBinaryData(t),n&&(this.reconstructor=null,this.emit("decoded",n))}},s.prototype.destroy=function(){this.reconstructor&&this.reconstructor.finishedReconstruction()},p.prototype.takeBinaryData=function(t){if(this.buffers.push(t),this.buffers.length===this.reconPack.attachments){var e=d.reconstructPacket(this.reconPack,this.buffers);return this.finishedReconstruction(),e}return null},p.prototype.finishedReconstruction=function(){this.reconPack=null,this.buffers=[]}},function(t,e,n){function r(t){if(t)return o(t)}function o(t){for(var e in r.prototype)t[e]=r.prototype[e];return t}t.exports=r,r.prototype.on=r.prototype.addEventListener=function(t,e){return this._callbacks=this._callbacks||{},(this._callbacks["$"+t]=this._callbacks["$"+t]||[]).push(e),this},r.prototype.once=function(t,e){function n(){this.off(t,n),e.apply(this,arguments)}return n.fn=e,this.on(t,n),this},r.prototype.off=r.prototype.removeListener=r.prototype.removeAllListeners=r.prototype.removeEventListener=function(t,e){if(this._callbacks=this._callbacks||{},0==arguments.length)return this._callbacks={},this;var n=this._callbacks["$"+t];if(!n)return this;if(1==arguments.length)return delete this._callbacks["$"+t],this;for(var r,o=0;o<n.length;o++)if(r=n[o],r===e||r.fn===e){n.splice(o,1);break}return this},r.prototype.emit=function(t){this._callbacks=this._callbacks||{};var e=[].slice.call(arguments,1),n=this._callbacks["$"+t];if(n){n=n.slice(0);for(var r=0,o=n.length;r<o;++r)n[r].apply(this,e)}return this},r.prototype.listeners=function(t){return this._callbacks=this._callbacks||{},this._callbacks["$"+t]||[]},r.prototype.hasListeners=function(t){return!!this.listeners(t).length}},function(t,e,n){(function(e){function r(t){if(!t||"object"!=typeof t)return!1;if(o(t)){for(var n=0,i=t.length;n<i;n++)if(r(t[n]))return!0;return!1}if("function"==typeof e.Buffer&&e.Buffer.isBuffer&&e.Buffer.isBuffer(t)||"function"==typeof e.ArrayBuffer&&t instanceof ArrayBuffer||s&&t instanceof Blob||a&&t instanceof File)return!0;if(t.toJSON&&"function"==typeof t.toJSON&&1===arguments.length)return r(t.toJSON(),!0);for(var c in t)if(Object.prototype.hasOwnProperty.call(t,c)&&r(t[c]))return!0;return!1}var o=n(10),i=Object.prototype.toString,s="function"==typeof e.Blob||"[object BlobConstructor]"===i.call(e.Blob),a="function"==typeof e.File||"[object FileConstructor]"===i.call(e.File);t.exports=r}).call(e,function(){return this}())},function(t,e){var n={}.toString;t.exports=Array.isArray||function(t){return"[object Array]"==n.call(t)}},function(t,e,n){(function(t){function r(t,e){if(!t)return t;if(s(t)){var n={_placeholder:!0,num:e.length};return e.push(t),n}if(i(t)){for(var o=new Array(t.length),a=0;a<t.length;a++)o[a]=r(t[a],e);return o}if("object"==typeof t&&!(t instanceof Date)){var o={};for(var c in t)o[c]=r(t[c],e);return o}return t}function o(t,e){if(!t)return t;if(t&&t._placeholder)return e[t.num];if(i(t))for(var n=0;n<t.length;n++)t[n]=o(t[n],e);else if("object"==typeof t)for(var r in t)t[r]=o(t[r],e);return t}var i=n(10),s=n(12),a=Object.prototype.toString,c="function"==typeof t.Blob||"[object BlobConstructor]"===a.call(t.Blob),p="function"==typeof t.File||"[object FileConstructor]"===a.call(t.File);e.deconstructPacket=function(t){var e=[],n=t.data,o=t;return o.data=r(n,e),o.attachments=e.length,{packet:o,buffers:e}},e.reconstructPacket=function(t,e){return t.data=o(t.data,e),t.attachments=void 0,t},e.removeBlobs=function(t,e){function n(t,a,u){if(!t)return t;if(c&&t instanceof Blob||p&&t instanceof File){r++;var h=new FileReader;h.onload=function(){u?u[a]=this.result:o=this.result,--r||e(o)},h.readAsArrayBuffer(t)}else if(i(t))for(var f=0;f<t.length;f++)n(t[f],f,t);else if("object"==typeof t&&!s(t))for(var l in t)n(t[l],l,t)}var r=0,o=t;n(o),r||e(o)}}).call(e,function(){return this}())},function(t,e){(function(e){function n(t){return e.Buffer&&e.Buffer.isBuffer(t)||e.ArrayBuffer&&t instanceof ArrayBuffer}t.exports=n}).call(e,function(){return this}())},function(t,e,n){"use strict";function r(t,e){if(!(this instanceof r))return new r(t,e);t&&"object"===("undefined"==typeof t?"undefined":o(t))&&(e=t,t=void 0),e=e||{},e.path=e.path||"/socket.io",this.nsps={},this.subs=[],this.opts=e,this.reconnection(e.reconnection!==!1),this.reconnectionAttempts(e.reconnectionAttempts||1/0),this.reconnectionDelay(e.reconnectionDelay||1e3),this.reconnectionDelayMax(e.reconnectionDelayMax||5e3),this.randomizationFactor(e.randomizationFactor||.5),this.backoff=new l({min:this.reconnectionDelay(),max:this.reconnectionDelayMax(),jitter:this.randomizationFactor()}),this.timeout(null==e.timeout?2e4:e.timeout),this.readyState="closed",this.uri=t,this.connecting=[],this.lastPing=null,this.encoding=!1,this.packetBuffer=[];var n=e.parser||c;this.encoder=new n.Encoder,this.decoder=new n.Decoder,this.autoConnect=e.autoConnect!==!1,this.autoConnect&&this.open()}var o="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function(t){return typeof t}:function(t){return t&&"function"==typeof Symbol&&t.constructor===Symbol&&t!==Symbol.prototype?"symbol":typeof t},i=n(14),s=n(39),a=n(8),c=n(7),p=n(41),u=n(42),h=n(3)("socket.io-client:manager"),f=n(37),l=n(43),d=Object.prototype.hasOwnProperty;t.exports=r,r.prototype.emitAll=function(){this.emit.apply(this,arguments);for(var t in this.nsps)d.call(this.nsps,t)&&this.nsps[t].emit.apply(this.nsps[t],arguments)},r.prototype.updateSocketIds=function(){for(var t in this.nsps)d.call(this.nsps,t)&&(this.nsps[t].id=this.generateId(t))},r.prototype.generateId=function(t){return("/"===t?"":t+"#")+this.engine.id},a(r.prototype),r.prototype.reconnection=function(t){return arguments.length?(this._reconnection=!!t,this):this._reconnection},r.prototype.reconnectionAttempts=function(t){return arguments.length?(this._reconnectionAttempts=t,this):this._reconnectionAttempts},r.prototype.reconnectionDelay=function(t){return arguments.length?(this._reconnectionDelay=t,this.backoff&&this.backoff.setMin(t),this):this._reconnectionDelay},r.prototype.randomizationFactor=function(t){return arguments.length?(this._randomizationFactor=t,this.backoff&&this.backoff.setJitter(t),this):this._randomizationFactor},r.prototype.reconnectionDelayMax=function(t){return arguments.length?(this._reconnectionDelayMax=t,this.backoff&&this.backoff.setMax(t),this):this._reconnectionDelayMax},r.prototype.timeout=function(t){return arguments.length?(this._timeout=t,this):this._timeout},r.prototype.maybeReconnectOnOpen=function(){!this.reconnecting&&this._reconnection&&0===this.backoff.attempts&&this.reconnect()},r.prototype.open=r.prototype.connect=function(t,e){if(h("readyState %s",this.readyState),~this.readyState.indexOf("open"))return this;h("opening %s",this.uri),this.engine=i(this.uri,this.opts);var n=this.engine,r=this;this.readyState="opening",this.skipReconnect=!1;var o=p(n,"open",function(){r.onopen(),t&&t()}),s=p(n,"error",function(e){if(h("connect_error"),r.cleanup(),r.readyState="closed",r.emitAll("connect_error",e),t){var n=new Error("Connection error");n.data=e,t(n)}else r.maybeReconnectOnOpen()});if(!1!==this._timeout){var a=this._timeout;h("connect attempt will timeout after %d",a);var c=setTimeout(function(){h("connect attempt timed out after %d",a),o.destroy(),n.close(),n.emit("error","timeout"),r.emitAll("connect_timeout",a)},a);this.subs.push({destroy:function(){clearTimeout(c)}})}return this.subs.push(o),this.subs.push(s),this},r.prototype.onopen=function(){h("open"),this.cleanup(),this.readyState="open",this.emit("open");var t=this.engine;this.subs.push(p(t,"data",u(this,"ondata"))),this.subs.push(p(t,"ping",u(this,"onping"))),this.subs.push(p(t,"pong",u(this,"onpong"))),this.subs.push(p(t,"error",u(this,"onerror"))),this.subs.push(p(t,"close",u(this,"onclose"))),this.subs.push(p(this.decoder,"decoded",u(this,"ondecoded")))},r.prototype.onping=function(){this.lastPing=new Date,this.emitAll("ping")},r.prototype.onpong=function(){this.emitAll("pong",new Date-this.lastPing)},r.prototype.ondata=function(t){this.decoder.add(t)},r.prototype.ondecoded=function(t){this.emit("packet",t)},r.prototype.onerror=function(t){h("error",t),this.emitAll("error",t)},r.prototype.socket=function(t,e){function n(){~f(o.connecting,r)||o.connecting.push(r)}var r=this.nsps[t];if(!r){r=new s(this,t,e),this.nsps[t]=r;var o=this;r.on("connecting",n),r.on("connect",function(){r.id=o.generateId(t)}),this.autoConnect&&n()}return r},r.prototype.destroy=function(t){var e=f(this.connecting,t);~e&&this.connecting.splice(e,1),this.connecting.length||this.close()},r.prototype.packet=function(t){h("writing packet %j",t);var e=this;t.query&&0===t.type&&(t.nsp+="?"+t.query),e.encoding?e.packetBuffer.push(t):(e.encoding=!0,this.encoder.encode(t,function(n){for(var r=0;r<n.length;r++)e.engine.write(n[r],t.options);e.encoding=!1,e.processPacketQueue()}))},r.prototype.processPacketQueue=function(){if(this.packetBuffer.length>0&&!this.encoding){var t=this.packetBuffer.shift();this.packet(t)}},r.prototype.cleanup=function(){h("cleanup");for(var t=this.subs.length,e=0;e<t;e++){var n=this.subs.shift();n.destroy()}this.packetBuffer=[],this.encoding=!1,this.lastPing=null,this.decoder.destroy()},r.prototype.close=r.prototype.disconnect=function(){h("disconnect"),this.skipReconnect=!0,this.reconnecting=!1,"opening"===this.readyState&&this.cleanup(),this.backoff.reset(),this.readyState="closed",this.engine&&this.engine.close()},r.prototype.onclose=function(t){h("onclose"),this.cleanup(),this.backoff.reset(),this.readyState="closed",this.emit("close",t),this._reconnection&&!this.skipReconnect&&this.reconnect()},r.prototype.reconnect=function(){if(this.reconnecting||this.skipReconnect)return this;var t=this;if(this.backoff.attempts>=this._reconnectionAttempts)h("reconnect failed"),this.backoff.reset(),this.emitAll("reconnect_failed"),this.reconnecting=!1;else{var e=this.backoff.duration();h("will wait %dms before reconnect attempt",e),this.reconnecting=!0;var n=setTimeout(function(){t.skipReconnect||(h("attempting reconnect"),t.emitAll("reconnect_attempt",t.backoff.attempts),t.emitAll("reconnecting",t.backoff.attempts),t.skipReconnect||t.open(function(e){e?(h("reconnect attempt error"),t.reconnecting=!1,t.reconnect(),t.emitAll("reconnect_error",e.data)):(h("reconnect success"),t.onreconnect())}))},e);this.subs.push({destroy:function(){clearTimeout(n)}})}},r.prototype.onreconnect=function(){var t=this.backoff.attempts;this.reconnecting=!1,this.backoff.reset(),this.updateSocketIds(),this.emitAll("reconnect",t)}},function(t,e,n){t.exports=n(15)},function(t,e,n){t.exports=n(16),t.exports.parser=n(23)},function(t,e,n){(function(e){function r(t,n){if(!(this instanceof r))return new r(t,n);n=n||{},t&&"object"==typeof t&&(n=t,t=null),t?(t=u(t),n.hostname=t.host,n.secure="https"===t.protocol||"wss"===t.protocol,n.port=t.port,t.query&&(n.query=t.query)):n.host&&(n.hostname=u(n.host).host),this.secure=null!=n.secure?n.secure:e.location&&"https:"===location.protocol,n.hostname&&!n.port&&(n.port=this.secure?"443":"80"),this.agent=n.agent||!1,this.hostname=n.hostname||(e.location?location.hostname:"localhost"),this.port=n.port||(e.location&&location.port?location.port:this.secure?443:80),this.query=n.query||{},"string"==typeof this.query&&(this.query=f.decode(this.query)),this.upgrade=!1!==n.upgrade,this.path=(n.path||"/engine.io").replace(/\/$/,"")+"/",this.forceJSONP=!!n.forceJSONP,this.jsonp=!1!==n.jsonp,this.forceBase64=!!n.forceBase64,this.enablesXDR=!!n.enablesXDR,this.timestampParam=n.timestampParam||"t",this.timestampRequests=n.timestampRequests,this.transports=n.transports||["polling","websocket"],this.transportOptions=n.transportOptions||{},this.readyState="",this.writeBuffer=[],this.prevBufferLen=0,this.policyPort=n.policyPort||843,this.rememberUpgrade=n.rememberUpgrade||!1,this.binaryType=null,this.onlyBinaryUpgrades=n.onlyBinaryUpgrades,this.perMessageDeflate=!1!==n.perMessageDeflate&&(n.perMessageDeflate||{}),!0===this.perMessageDeflate&&(this.perMessageDeflate={}),this.perMessageDeflate&&null==this.perMessageDeflate.threshold&&(this.perMessageDeflate.threshold=1024),this.pfx=n.pfx||null,this.key=n.key||null,this.passphrase=n.passphrase||null,this.cert=n.cert||null,this.ca=n.ca||null,this.ciphers=n.ciphers||null,this.rejectUnauthorized=void 0===n.rejectUnauthorized||n.rejectUnauthorized,this.forceNode=!!n.forceNode;var o="object"==typeof e&&e;o.global===o&&(n.extraHeaders&&Object.keys(n.extraHeaders).length>0&&(this.extraHeaders=n.extraHeaders),n.localAddress&&(this.localAddress=n.localAddress)),this.id=null,this.upgrades=null,this.pingInterval=null,this.pingTimeout=null,this.pingIntervalTimer=null,this.pingTimeoutTimer=null,this.open()}function o(t){var e={};for(var n in t)t.hasOwnProperty(n)&&(e[n]=t[n]);return e}var i=n(17),s=n(8),a=n(3)("engine.io-client:socket"),c=n(37),p=n(23),u=n(2),h=n(38),f=n(31);t.exports=r,r.priorWebsocketSuccess=!1,s(r.prototype),r.protocol=p.protocol,r.Socket=r,r.Transport=n(22),r.transports=n(17),r.parser=n(23),r.prototype.createTransport=function(t){a('creating transport "%s"',t);var e=o(this.query);e.EIO=p.protocol,e.transport=t;var n=this.transportOptions[t]||{};this.id&&(e.sid=this.id);var r=new i[t]({query:e,socket:this,agent:n.agent||this.agent,hostname:n.hostname||this.hostname,port:n.port||this.port,secure:n.secure||this.secure,path:n.path||this.path,forceJSONP:n.forceJSONP||this.forceJSONP,jsonp:n.jsonp||this.jsonp,forceBase64:n.forceBase64||this.forceBase64,enablesXDR:n.enablesXDR||this.enablesXDR,timestampRequests:n.timestampRequests||this.timestampRequests,timestampParam:n.timestampParam||this.timestampParam,policyPort:n.policyPort||this.policyPort,pfx:n.pfx||this.pfx,key:n.key||this.key,passphrase:n.passphrase||this.passphrase,cert:n.cert||this.cert,ca:n.ca||this.ca,ciphers:n.ciphers||this.ciphers,rejectUnauthorized:n.rejectUnauthorized||this.rejectUnauthorized,perMessageDeflate:n.perMessageDeflate||this.perMessageDeflate,extraHeaders:n.extraHeaders||this.extraHeaders,forceNode:n.forceNode||this.forceNode,localAddress:n.localAddress||this.localAddress,requestTimeout:n.requestTimeout||this.requestTimeout,protocols:n.protocols||void 0});return r},r.prototype.open=function(){var t;if(this.rememberUpgrade&&r.priorWebsocketSuccess&&this.transports.indexOf("websocket")!==-1)t="websocket";else{if(0===this.transports.length){var e=this;return void setTimeout(function(){e.emit("error","No transports available")},0)}t=this.transports[0]}this.readyState="opening";try{t=this.createTransport(t)}catch(n){return this.transports.shift(),void this.open()}t.open(),this.setTransport(t)},r.prototype.setTransport=function(t){a("setting transport %s",t.name);var e=this;this.transport&&(a("clearing existing transport %s",this.transport.name),this.transport.removeAllListeners()),this.transport=t,t.on("drain",function(){e.onDrain()}).on("packet",function(t){e.onPacket(t)}).on("error",function(t){e.onError(t)}).on("close",function(){e.onClose("transport close")})},r.prototype.probe=function(t){function e(){if(f.onlyBinaryUpgrades){var e=!this.supportsBinary&&f.transport.supportsBinary;h=h||e}h||(a('probe transport "%s" opened',t),u.send([{type:"ping",data:"probe"}]),u.once("packet",function(e){if(!h)if("pong"===e.type&&"probe"===e.data){if(a('probe transport "%s" pong',t),f.upgrading=!0,f.emit("upgrading",u),!u)return;r.priorWebsocketSuccess="websocket"===u.name,a('pausing current transport "%s"',f.transport.name),f.transport.pause(function(){h||"closed"!==f.readyState&&(a("changing transport and sending upgrade packet"),p(),f.setTransport(u),u.send([{type:"upgrade"}]),f.emit("upgrade",u),u=null,f.upgrading=!1,f.flush())})}else{a('probe transport "%s" failed',t);var n=new Error("probe error");n.transport=u.name,f.emit("upgradeError",n)}}))}function n(){h||(h=!0,p(),u.close(),u=null)}function o(e){var r=new Error("probe error: "+e);r.transport=u.name,n(),a('probe transport "%s" failed because of error: %s',t,e),f.emit("upgradeError",r)}function i(){o("transport closed")}function s(){o("socket closed")}function c(t){u&&t.name!==u.name&&(a('"%s" works - aborting "%s"',t.name,u.name),n())}function p(){u.removeListener("open",e),u.removeListener("error",o),u.removeListener("close",i),f.removeListener("close",s),f.removeListener("upgrading",c)}a('probing transport "%s"',t);var u=this.createTransport(t,{probe:1}),h=!1,f=this;r.priorWebsocketSuccess=!1,u.once("open",e),u.once("error",o),u.once("close",i),this.once("close",s),this.once("upgrading",c),u.open()},r.prototype.onOpen=function(){if(a("socket open"),this.readyState="open",r.priorWebsocketSuccess="websocket"===this.transport.name,this.emit("open"),this.flush(),"open"===this.readyState&&this.upgrade&&this.transport.pause){a("starting upgrade probes");for(var t=0,e=this.upgrades.length;t<e;t++)this.probe(this.upgrades[t])}},r.prototype.onPacket=function(t){if("opening"===this.readyState||"open"===this.readyState||"closing"===this.readyState)switch(a('socket receive: type "%s", data "%s"',t.type,t.data),this.emit("packet",t),this.emit("heartbeat"),t.type){case"open":this.onHandshake(h(t.data));break;case"pong":this.setPing(),this.emit("pong");break;case"error":var e=new Error("server error");e.code=t.data,this.onError(e);break;case"message":this.emit("data",t.data),this.emit("message",t.data)}else a('packet received with socket readyState "%s"',this.readyState)},r.prototype.onHandshake=function(t){this.emit("handshake",t),this.id=t.sid,this.transport.query.sid=t.sid,this.upgrades=this.filterUpgrades(t.upgrades),this.pingInterval=t.pingInterval,this.pingTimeout=t.pingTimeout,this.onOpen(),"closed"!==this.readyState&&(this.setPing(),this.removeListener("heartbeat",this.onHeartbeat),this.on("heartbeat",this.onHeartbeat))},r.prototype.onHeartbeat=function(t){clearTimeout(this.pingTimeoutTimer);var e=this;e.pingTimeoutTimer=setTimeout(function(){"closed"!==e.readyState&&e.onClose("ping timeout")},t||e.pingInterval+e.pingTimeout)},r.prototype.setPing=function(){var t=this;clearTimeout(t.pingIntervalTimer),t.pingIntervalTimer=setTimeout(function(){a("writing ping packet - expecting pong within %sms",t.pingTimeout),t.ping(),t.onHeartbeat(t.pingTimeout)},t.pingInterval)},r.prototype.ping=function(){var t=this;this.sendPacket("ping",function(){t.emit("ping")})},r.prototype.onDrain=function(){this.writeBuffer.splice(0,this.prevBufferLen),this.prevBufferLen=0,0===this.writeBuffer.length?this.emit("drain"):this.flush()},r.prototype.flush=function(){"closed"!==this.readyState&&this.transport.writable&&!this.upgrading&&this.writeBuffer.length&&(a("flushing %d packets in socket",this.writeBuffer.length),this.transport.send(this.writeBuffer),this.prevBufferLen=this.writeBuffer.length,this.emit("flush"))},r.prototype.write=r.prototype.send=function(t,e,n){return this.sendPacket("message",t,e,n),this},r.prototype.sendPacket=function(t,e,n,r){if("function"==typeof e&&(r=e,e=void 0),"function"==typeof n&&(r=n,n=null),"closing"!==this.readyState&&"closed"!==this.readyState){n=n||{},n.compress=!1!==n.compress;var o={type:t,data:e,options:n};this.emit("packetCreate",o),this.writeBuffer.push(o),r&&this.once("flush",r),this.flush()}},r.prototype.close=function(){function t(){r.onClose("forced close"),a("socket closing - telling transport to close"),r.transport.close()}function e(){r.removeListener("upgrade",e),r.removeListener("upgradeError",e),t()}function n(){r.once("upgrade",e),r.once("upgradeError",e)}if("opening"===this.readyState||"open"===this.readyState){this.readyState="closing";var r=this;this.writeBuffer.length?this.once("drain",function(){this.upgrading?n():t()}):this.upgrading?n():t()}return this},r.prototype.onError=function(t){a("socket error %j",t),r.priorWebsocketSuccess=!1,this.emit("error",t),this.onClose("transport error",t)},r.prototype.onClose=function(t,e){if("opening"===this.readyState||"open"===this.readyState||"closing"===this.readyState){a('socket close with reason: "%s"',t);var n=this;clearTimeout(this.pingIntervalTimer),clearTimeout(this.pingTimeoutTimer),this.transport.removeAllListeners("close"),this.transport.close(),this.transport.removeAllListeners(),this.readyState="closed",this.id=null,this.emit("close",t,e),n.writeBuffer=[],n.prevBufferLen=0}},r.prototype.filterUpgrades=function(t){for(var e=[],n=0,r=t.length;n<r;n++)~c(this.transports,t[n])&&e.push(t[n]);return e}}).call(e,function(){return this}())},function(t,e,n){(function(t){function r(e){var n,r=!1,a=!1,c=!1!==e.jsonp;if(t.location){var p="https:"===location.protocol,u=location.port;u||(u=p?443:80),r=e.hostname!==location.hostname||u!==e.port,a=e.secure!==p}if(e.xdomain=r,e.xscheme=a,n=new o(e),"open"in n&&!e.forceJSONP)return new i(e);if(!c)throw new Error("JSONP disabled");return new s(e)}var o=n(18),i=n(20),s=n(34),a=n(35);e.polling=r,e.websocket=a}).call(e,function(){return this}())},function(t,e,n){(function(e){var r=n(19);t.exports=function(t){var n=t.xdomain,o=t.xscheme,i=t.enablesXDR;try{if("undefined"!=typeof XMLHttpRequest&&(!n||r))return new XMLHttpRequest}catch(s){}try{if("undefined"!=typeof XDomainRequest&&!o&&i)return new XDomainRequest}catch(s){}if(!n)try{
return new(e[["Active"].concat("Object").join("X")])("Microsoft.XMLHTTP")}catch(s){}}}).call(e,function(){return this}())},function(t,e){try{t.exports="undefined"!=typeof XMLHttpRequest&&"withCredentials"in new XMLHttpRequest}catch(n){t.exports=!1}},function(t,e,n){(function(e){function r(){}function o(t){if(c.call(this,t),this.requestTimeout=t.requestTimeout,this.extraHeaders=t.extraHeaders,e.location){var n="https:"===location.protocol,r=location.port;r||(r=n?443:80),this.xd=t.hostname!==e.location.hostname||r!==t.port,this.xs=t.secure!==n}}function i(t){this.method=t.method||"GET",this.uri=t.uri,this.xd=!!t.xd,this.xs=!!t.xs,this.async=!1!==t.async,this.data=void 0!==t.data?t.data:null,this.agent=t.agent,this.isBinary=t.isBinary,this.supportsBinary=t.supportsBinary,this.enablesXDR=t.enablesXDR,this.requestTimeout=t.requestTimeout,this.pfx=t.pfx,this.key=t.key,this.passphrase=t.passphrase,this.cert=t.cert,this.ca=t.ca,this.ciphers=t.ciphers,this.rejectUnauthorized=t.rejectUnauthorized,this.extraHeaders=t.extraHeaders,this.create()}function s(){for(var t in i.requests)i.requests.hasOwnProperty(t)&&i.requests[t].abort()}var a=n(18),c=n(21),p=n(8),u=n(32),h=n(3)("engine.io-client:polling-xhr");t.exports=o,t.exports.Request=i,u(o,c),o.prototype.supportsBinary=!0,o.prototype.request=function(t){return t=t||{},t.uri=this.uri(),t.xd=this.xd,t.xs=this.xs,t.agent=this.agent||!1,t.supportsBinary=this.supportsBinary,t.enablesXDR=this.enablesXDR,t.pfx=this.pfx,t.key=this.key,t.passphrase=this.passphrase,t.cert=this.cert,t.ca=this.ca,t.ciphers=this.ciphers,t.rejectUnauthorized=this.rejectUnauthorized,t.requestTimeout=this.requestTimeout,t.extraHeaders=this.extraHeaders,new i(t)},o.prototype.doWrite=function(t,e){var n="string"!=typeof t&&void 0!==t,r=this.request({method:"POST",data:t,isBinary:n}),o=this;r.on("success",e),r.on("error",function(t){o.onError("xhr post error",t)}),this.sendXhr=r},o.prototype.doPoll=function(){h("xhr poll");var t=this.request(),e=this;t.on("data",function(t){e.onData(t)}),t.on("error",function(t){e.onError("xhr poll error",t)}),this.pollXhr=t},p(i.prototype),i.prototype.create=function(){var t={agent:this.agent,xdomain:this.xd,xscheme:this.xs,enablesXDR:this.enablesXDR};t.pfx=this.pfx,t.key=this.key,t.passphrase=this.passphrase,t.cert=this.cert,t.ca=this.ca,t.ciphers=this.ciphers,t.rejectUnauthorized=this.rejectUnauthorized;var n=this.xhr=new a(t),r=this;try{h("xhr open %s: %s",this.method,this.uri),n.open(this.method,this.uri,this.async);try{if(this.extraHeaders){n.setDisableHeaderCheck&&n.setDisableHeaderCheck(!0);for(var o in this.extraHeaders)this.extraHeaders.hasOwnProperty(o)&&n.setRequestHeader(o,this.extraHeaders[o])}}catch(s){}if("POST"===this.method)try{this.isBinary?n.setRequestHeader("Content-type","application/octet-stream"):n.setRequestHeader("Content-type","text/plain;charset=UTF-8")}catch(s){}try{n.setRequestHeader("Accept","*/*")}catch(s){}"withCredentials"in n&&(n.withCredentials=!0),this.requestTimeout&&(n.timeout=this.requestTimeout),this.hasXDR()?(n.onload=function(){r.onLoad()},n.onerror=function(){r.onError(n.responseText)}):n.onreadystatechange=function(){if(2===n.readyState){var t;try{t=n.getResponseHeader("Content-Type")}catch(e){}"application/octet-stream"===t&&(n.responseType="arraybuffer")}4===n.readyState&&(200===n.status||1223===n.status?r.onLoad():setTimeout(function(){r.onError(n.status)},0))},h("xhr data %s",this.data),n.send(this.data)}catch(s){return void setTimeout(function(){r.onError(s)},0)}e.document&&(this.index=i.requestsCount++,i.requests[this.index]=this)},i.prototype.onSuccess=function(){this.emit("success"),this.cleanup()},i.prototype.onData=function(t){this.emit("data",t),this.onSuccess()},i.prototype.onError=function(t){this.emit("error",t),this.cleanup(!0)},i.prototype.cleanup=function(t){if("undefined"!=typeof this.xhr&&null!==this.xhr){if(this.hasXDR()?this.xhr.onload=this.xhr.onerror=r:this.xhr.onreadystatechange=r,t)try{this.xhr.abort()}catch(n){}e.document&&delete i.requests[this.index],this.xhr=null}},i.prototype.onLoad=function(){var t;try{var e;try{e=this.xhr.getResponseHeader("Content-Type")}catch(n){}t="application/octet-stream"===e?this.xhr.response||this.xhr.responseText:this.xhr.responseText}catch(n){this.onError(n)}null!=t&&this.onData(t)},i.prototype.hasXDR=function(){return"undefined"!=typeof e.XDomainRequest&&!this.xs&&this.enablesXDR},i.prototype.abort=function(){this.cleanup()},i.requestsCount=0,i.requests={},e.document&&(e.attachEvent?e.attachEvent("onunload",s):e.addEventListener&&e.addEventListener("beforeunload",s,!1))}).call(e,function(){return this}())},function(t,e,n){function r(t){var e=t&&t.forceBase64;u&&!e||(this.supportsBinary=!1),o.call(this,t)}var o=n(22),i=n(31),s=n(23),a=n(32),c=n(33),p=n(3)("engine.io-client:polling");t.exports=r;var u=function(){var t=n(18),e=new t({xdomain:!1});return null!=e.responseType}();a(r,o),r.prototype.name="polling",r.prototype.doOpen=function(){this.poll()},r.prototype.pause=function(t){function e(){p("paused"),n.readyState="paused",t()}var n=this;if(this.readyState="pausing",this.polling||!this.writable){var r=0;this.polling&&(p("we are currently polling - waiting to pause"),r++,this.once("pollComplete",function(){p("pre-pause polling complete"),--r||e()})),this.writable||(p("we are currently writing - waiting to pause"),r++,this.once("drain",function(){p("pre-pause writing complete"),--r||e()}))}else e()},r.prototype.poll=function(){p("polling"),this.polling=!0,this.doPoll(),this.emit("poll")},r.prototype.onData=function(t){var e=this;p("polling got data %s",t);var n=function(t,n,r){return"opening"===e.readyState&&e.onOpen(),"close"===t.type?(e.onClose(),!1):void e.onPacket(t)};s.decodePayload(t,this.socket.binaryType,n),"closed"!==this.readyState&&(this.polling=!1,this.emit("pollComplete"),"open"===this.readyState?this.poll():p('ignoring poll - transport state "%s"',this.readyState))},r.prototype.doClose=function(){function t(){p("writing close packet"),e.write([{type:"close"}])}var e=this;"open"===this.readyState?(p("transport open - closing"),t()):(p("transport not open - deferring close"),this.once("open",t))},r.prototype.write=function(t){var e=this;this.writable=!1;var n=function(){e.writable=!0,e.emit("drain")};s.encodePayload(t,this.supportsBinary,function(t){e.doWrite(t,n)})},r.prototype.uri=function(){var t=this.query||{},e=this.secure?"https":"http",n="";!1!==this.timestampRequests&&(t[this.timestampParam]=c()),this.supportsBinary||t.sid||(t.b64=1),t=i.encode(t),this.port&&("https"===e&&443!==Number(this.port)||"http"===e&&80!==Number(this.port))&&(n=":"+this.port),t.length&&(t="?"+t);var r=this.hostname.indexOf(":")!==-1;return e+"://"+(r?"["+this.hostname+"]":this.hostname)+n+this.path+t}},function(t,e,n){function r(t){this.path=t.path,this.hostname=t.hostname,this.port=t.port,this.secure=t.secure,this.query=t.query,this.timestampParam=t.timestampParam,this.timestampRequests=t.timestampRequests,this.readyState="",this.agent=t.agent||!1,this.socket=t.socket,this.enablesXDR=t.enablesXDR,this.pfx=t.pfx,this.key=t.key,this.passphrase=t.passphrase,this.cert=t.cert,this.ca=t.ca,this.ciphers=t.ciphers,this.rejectUnauthorized=t.rejectUnauthorized,this.forceNode=t.forceNode,this.extraHeaders=t.extraHeaders,this.localAddress=t.localAddress}var o=n(23),i=n(8);t.exports=r,i(r.prototype),r.prototype.onError=function(t,e){var n=new Error(t);return n.type="TransportError",n.description=e,this.emit("error",n),this},r.prototype.open=function(){return"closed"!==this.readyState&&""!==this.readyState||(this.readyState="opening",this.doOpen()),this},r.prototype.close=function(){return"opening"!==this.readyState&&"open"!==this.readyState||(this.doClose(),this.onClose()),this},r.prototype.send=function(t){if("open"!==this.readyState)throw new Error("Transport not open");this.write(t)},r.prototype.onOpen=function(){this.readyState="open",this.writable=!0,this.emit("open")},r.prototype.onData=function(t){var e=o.decodePacket(t,this.socket.binaryType);this.onPacket(e)},r.prototype.onPacket=function(t){this.emit("packet",t)},r.prototype.onClose=function(){this.readyState="closed",this.emit("close")}},function(t,e,n){(function(t){function r(t,n){var r="b"+e.packets[t.type]+t.data.data;return n(r)}function o(t,n,r){if(!n)return e.encodeBase64Packet(t,r);var o=t.data,i=new Uint8Array(o),s=new Uint8Array(1+o.byteLength);s[0]=v[t.type];for(var a=0;a<i.length;a++)s[a+1]=i[a];return r(s.buffer)}function i(t,n,r){if(!n)return e.encodeBase64Packet(t,r);var o=new FileReader;return o.onload=function(){t.data=o.result,e.encodePacket(t,n,!0,r)},o.readAsArrayBuffer(t.data)}function s(t,n,r){if(!n)return e.encodeBase64Packet(t,r);if(g)return i(t,n,r);var o=new Uint8Array(1);o[0]=v[t.type];var s=new k([o.buffer,t.data]);return r(s)}function a(t){try{t=d.decode(t,{strict:!1})}catch(e){return!1}return t}function c(t,e,n){for(var r=new Array(t.length),o=l(t.length,n),i=function(t,n,o){e(n,function(e,n){r[t]=n,o(e,r)})},s=0;s<t.length;s++)i(s,t[s],o)}var p,u=n(24),h=n(9),f=n(25),l=n(26),d=n(27);t&&t.ArrayBuffer&&(p=n(29));var y="undefined"!=typeof navigator&&/Android/i.test(navigator.userAgent),m="undefined"!=typeof navigator&&/PhantomJS/i.test(navigator.userAgent),g=y||m;e.protocol=3;var v=e.packets={open:0,close:1,ping:2,pong:3,message:4,upgrade:5,noop:6},b=u(v),w={type:"error",data:"parser error"},k=n(30);e.encodePacket=function(e,n,i,a){"function"==typeof n&&(a=n,n=!1),"function"==typeof i&&(a=i,i=null);var c=void 0===e.data?void 0:e.data.buffer||e.data;if(t.ArrayBuffer&&c instanceof ArrayBuffer)return o(e,n,a);if(k&&c instanceof t.Blob)return s(e,n,a);if(c&&c.base64)return r(e,a);var p=v[e.type];return void 0!==e.data&&(p+=i?d.encode(String(e.data),{strict:!1}):String(e.data)),a(""+p)},e.encodeBase64Packet=function(n,r){var o="b"+e.packets[n.type];if(k&&n.data instanceof t.Blob){var i=new FileReader;return i.onload=function(){var t=i.result.split(",")[1];r(o+t)},i.readAsDataURL(n.data)}var s;try{s=String.fromCharCode.apply(null,new Uint8Array(n.data))}catch(a){for(var c=new Uint8Array(n.data),p=new Array(c.length),u=0;u<c.length;u++)p[u]=c[u];s=String.fromCharCode.apply(null,p)}return o+=t.btoa(s),r(o)},e.decodePacket=function(t,n,r){if(void 0===t)return w;if("string"==typeof t){if("b"===t.charAt(0))return e.decodeBase64Packet(t.substr(1),n);if(r&&(t=a(t),t===!1))return w;var o=t.charAt(0);return Number(o)==o&&b[o]?t.length>1?{type:b[o],data:t.substring(1)}:{type:b[o]}:w}var i=new Uint8Array(t),o=i[0],s=f(t,1);return k&&"blob"===n&&(s=new k([s])),{type:b[o],data:s}},e.decodeBase64Packet=function(t,e){var n=b[t.charAt(0)];if(!p)return{type:n,data:{base64:!0,data:t.substr(1)}};var r=p.decode(t.substr(1));return"blob"===e&&k&&(r=new k([r])),{type:n,data:r}},e.encodePayload=function(t,n,r){function o(t){return t.length+":"+t}function i(t,r){e.encodePacket(t,!!s&&n,!1,function(t){r(null,o(t))})}"function"==typeof n&&(r=n,n=null);var s=h(t);return n&&s?k&&!g?e.encodePayloadAsBlob(t,r):e.encodePayloadAsArrayBuffer(t,r):t.length?void c(t,i,function(t,e){return r(e.join(""))}):r("0:")},e.decodePayload=function(t,n,r){if("string"!=typeof t)return e.decodePayloadAsBinary(t,n,r);"function"==typeof n&&(r=n,n=null);var o;if(""===t)return r(w,0,1);for(var i,s,a="",c=0,p=t.length;c<p;c++){var u=t.charAt(c);if(":"===u){if(""===a||a!=(i=Number(a)))return r(w,0,1);if(s=t.substr(c+1,i),a!=s.length)return r(w,0,1);if(s.length){if(o=e.decodePacket(s,n,!1),w.type===o.type&&w.data===o.data)return r(w,0,1);var h=r(o,c+i,p);if(!1===h)return}c+=i,a=""}else a+=u}return""!==a?r(w,0,1):void 0},e.encodePayloadAsArrayBuffer=function(t,n){function r(t,n){e.encodePacket(t,!0,!0,function(t){return n(null,t)})}return t.length?void c(t,r,function(t,e){var r=e.reduce(function(t,e){var n;return n="string"==typeof e?e.length:e.byteLength,t+n.toString().length+n+2},0),o=new Uint8Array(r),i=0;return e.forEach(function(t){var e="string"==typeof t,n=t;if(e){for(var r=new Uint8Array(t.length),s=0;s<t.length;s++)r[s]=t.charCodeAt(s);n=r.buffer}e?o[i++]=0:o[i++]=1;for(var a=n.byteLength.toString(),s=0;s<a.length;s++)o[i++]=parseInt(a[s]);o[i++]=255;for(var r=new Uint8Array(n),s=0;s<r.length;s++)o[i++]=r[s]}),n(o.buffer)}):n(new ArrayBuffer(0))},e.encodePayloadAsBlob=function(t,n){function r(t,n){e.encodePacket(t,!0,!0,function(t){var e=new Uint8Array(1);if(e[0]=1,"string"==typeof t){for(var r=new Uint8Array(t.length),o=0;o<t.length;o++)r[o]=t.charCodeAt(o);t=r.buffer,e[0]=0}for(var i=t instanceof ArrayBuffer?t.byteLength:t.size,s=i.toString(),a=new Uint8Array(s.length+1),o=0;o<s.length;o++)a[o]=parseInt(s[o]);if(a[s.length]=255,k){var c=new k([e.buffer,a.buffer,t]);n(null,c)}})}c(t,r,function(t,e){return n(new k(e))})},e.decodePayloadAsBinary=function(t,n,r){"function"==typeof n&&(r=n,n=null);for(var o=t,i=[];o.byteLength>0;){for(var s=new Uint8Array(o),a=0===s[0],c="",p=1;255!==s[p];p++){if(c.length>310)return r(w,0,1);c+=s[p]}o=f(o,2+c.length),c=parseInt(c);var u=f(o,0,c);if(a)try{u=String.fromCharCode.apply(null,new Uint8Array(u))}catch(h){var l=new Uint8Array(u);u="";for(var p=0;p<l.length;p++)u+=String.fromCharCode(l[p])}i.push(u),o=f(o,c)}var d=i.length;i.forEach(function(t,o){r(e.decodePacket(t,n,!0),o,d)})}}).call(e,function(){return this}())},function(t,e){t.exports=Object.keys||function(t){var e=[],n=Object.prototype.hasOwnProperty;for(var r in t)n.call(t,r)&&e.push(r);return e}},function(t,e){t.exports=function(t,e,n){var r=t.byteLength;if(e=e||0,n=n||r,t.slice)return t.slice(e,n);if(e<0&&(e+=r),n<0&&(n+=r),n>r&&(n=r),e>=r||e>=n||0===r)return new ArrayBuffer(0);for(var o=new Uint8Array(t),i=new Uint8Array(n-e),s=e,a=0;s<n;s++,a++)i[a]=o[s];return i.buffer}},function(t,e){function n(t,e,n){function o(t,r){if(o.count<=0)throw new Error("after called too many times");--o.count,t?(i=!0,e(t),e=n):0!==o.count||i||e(null,r)}var i=!1;return n=n||r,o.count=t,0===t?e():o}function r(){}t.exports=n},function(t,e,n){var r;(function(t,o){!function(i){function s(t){for(var e,n,r=[],o=0,i=t.length;o<i;)e=t.charCodeAt(o++),e>=55296&&e<=56319&&o<i?(n=t.charCodeAt(o++),56320==(64512&n)?r.push(((1023&e)<<10)+(1023&n)+65536):(r.push(e),o--)):r.push(e);return r}function a(t){for(var e,n=t.length,r=-1,o="";++r<n;)e=t[r],e>65535&&(e-=65536,o+=w(e>>>10&1023|55296),e=56320|1023&e),o+=w(e);return o}function c(t,e){if(t>=55296&&t<=57343){if(e)throw Error("Lone surrogate U+"+t.toString(16).toUpperCase()+" is not a scalar value");return!1}return!0}function p(t,e){return w(t>>e&63|128)}function u(t,e){if(0==(4294967168&t))return w(t);var n="";return 0==(4294965248&t)?n=w(t>>6&31|192):0==(4294901760&t)?(c(t,e)||(t=65533),n=w(t>>12&15|224),n+=p(t,6)):0==(4292870144&t)&&(n=w(t>>18&7|240),n+=p(t,12),n+=p(t,6)),n+=w(63&t|128)}function h(t,e){e=e||{};for(var n,r=!1!==e.strict,o=s(t),i=o.length,a=-1,c="";++a<i;)n=o[a],c+=u(n,r);return c}function f(){if(b>=v)throw Error("Invalid byte index");var t=255&g[b];if(b++,128==(192&t))return 63&t;throw Error("Invalid continuation byte")}function l(t){var e,n,r,o,i;if(b>v)throw Error("Invalid byte index");if(b==v)return!1;if(e=255&g[b],b++,0==(128&e))return e;if(192==(224&e)){if(n=f(),i=(31&e)<<6|n,i>=128)return i;throw Error("Invalid continuation byte")}if(224==(240&e)){if(n=f(),r=f(),i=(15&e)<<12|n<<6|r,i>=2048)return c(i,t)?i:65533;throw Error("Invalid continuation byte")}if(240==(248&e)&&(n=f(),r=f(),o=f(),i=(7&e)<<18|n<<12|r<<6|o,i>=65536&&i<=1114111))return i;throw Error("Invalid UTF-8 detected")}function d(t,e){e=e||{};var n=!1!==e.strict;g=s(t),v=g.length,b=0;for(var r,o=[];(r=l(n))!==!1;)o.push(r);return a(o)}var y="object"==typeof e&&e,m=("object"==typeof t&&t&&t.exports==y&&t,"object"==typeof o&&o);m.global!==m&&m.window!==m||(i=m);var g,v,b,w=String.fromCharCode,k={version:"2.1.2",encode:h,decode:d};r=function(){return k}.call(e,n,e,t),!(void 0!==r&&(t.exports=r))}(this)}).call(e,n(28)(t),function(){return this}())},function(t,e){t.exports=function(t){return t.webpackPolyfill||(t.deprecate=function(){},t.paths=[],t.children=[],t.webpackPolyfill=1),t}},function(t,e){!function(){"use strict";for(var t="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",n=new Uint8Array(256),r=0;r<t.length;r++)n[t.charCodeAt(r)]=r;e.encode=function(e){var n,r=new Uint8Array(e),o=r.length,i="";for(n=0;n<o;n+=3)i+=t[r[n]>>2],i+=t[(3&r[n])<<4|r[n+1]>>4],i+=t[(15&r[n+1])<<2|r[n+2]>>6],i+=t[63&r[n+2]];return o%3===2?i=i.substring(0,i.length-1)+"=":o%3===1&&(i=i.substring(0,i.length-2)+"=="),i},e.decode=function(t){var e,r,o,i,s,a=.75*t.length,c=t.length,p=0;"="===t[t.length-1]&&(a--,"="===t[t.length-2]&&a--);var u=new ArrayBuffer(a),h=new Uint8Array(u);for(e=0;e<c;e+=4)r=n[t.charCodeAt(e)],o=n[t.charCodeAt(e+1)],i=n[t.charCodeAt(e+2)],s=n[t.charCodeAt(e+3)],h[p++]=r<<2|o>>4,h[p++]=(15&o)<<4|i>>2,h[p++]=(3&i)<<6|63&s;return u}}()},function(t,e){(function(e){function n(t){for(var e=0;e<t.length;e++){var n=t[e];if(n.buffer instanceof ArrayBuffer){var r=n.buffer;if(n.byteLength!==r.byteLength){var o=new Uint8Array(n.byteLength);o.set(new Uint8Array(r,n.byteOffset,n.byteLength)),r=o.buffer}t[e]=r}}}function r(t,e){e=e||{};var r=new i;n(t);for(var o=0;o<t.length;o++)r.append(t[o]);return e.type?r.getBlob(e.type):r.getBlob()}function o(t,e){return n(t),new Blob(t,e||{})}var i=e.BlobBuilder||e.WebKitBlobBuilder||e.MSBlobBuilder||e.MozBlobBuilder,s=function(){try{var t=new Blob(["hi"]);return 2===t.size}catch(e){return!1}}(),a=s&&function(){try{var t=new Blob([new Uint8Array([1,2])]);return 2===t.size}catch(e){return!1}}(),c=i&&i.prototype.append&&i.prototype.getBlob;t.exports=function(){return s?a?e.Blob:o:c?r:void 0}()}).call(e,function(){return this}())},function(t,e){e.encode=function(t){var e="";for(var n in t)t.hasOwnProperty(n)&&(e.length&&(e+="&"),e+=encodeURIComponent(n)+"="+encodeURIComponent(t[n]));return e},e.decode=function(t){for(var e={},n=t.split("&"),r=0,o=n.length;r<o;r++){var i=n[r].split("=");e[decodeURIComponent(i[0])]=decodeURIComponent(i[1])}return e}},function(t,e){t.exports=function(t,e){var n=function(){};n.prototype=e.prototype,t.prototype=new n,t.prototype.constructor=t}},function(t,e){"use strict";function n(t){var e="";do e=s[t%a]+e,t=Math.floor(t/a);while(t>0);return e}function r(t){var e=0;for(u=0;u<t.length;u++)e=e*a+c[t.charAt(u)];return e}function o(){var t=n(+new Date);return t!==i?(p=0,i=t):t+"."+n(p++)}for(var i,s="0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz-_".split(""),a=64,c={},p=0,u=0;u<a;u++)c[s[u]]=u;o.encode=n,o.decode=r,t.exports=o},function(t,e,n){(function(e){function r(){}function o(t){i.call(this,t),this.query=this.query||{},a||(e.___eio||(e.___eio=[]),a=e.___eio),this.index=a.length;var n=this;a.push(function(t){n.onData(t)}),this.query.j=this.index,e.document&&e.addEventListener&&e.addEventListener("beforeunload",function(){n.script&&(n.script.onerror=r)},!1)}var i=n(21),s=n(32);t.exports=o;var a,c=/\n/g,p=/\\n/g;s(o,i),o.prototype.supportsBinary=!1,o.prototype.doClose=function(){this.script&&(this.script.parentNode.removeChild(this.script),this.script=null),this.form&&(this.form.parentNode.removeChild(this.form),this.form=null,this.iframe=null),i.prototype.doClose.call(this)},o.prototype.doPoll=function(){var t=this,e=document.createElement("script");this.script&&(this.script.parentNode.removeChild(this.script),this.script=null),e.async=!0,e.src=this.uri(),e.onerror=function(e){t.onError("jsonp poll error",e)};var n=document.getElementsByTagName("script")[0];n?n.parentNode.insertBefore(e,n):(document.head||document.body).appendChild(e),this.script=e;var r="undefined"!=typeof navigator&&/gecko/i.test(navigator.userAgent);r&&setTimeout(function(){var t=document.createElement("iframe");document.body.appendChild(t),document.body.removeChild(t)},100)},o.prototype.doWrite=function(t,e){function n(){r(),e()}function r(){if(o.iframe)try{o.form.removeChild(o.iframe)}catch(t){o.onError("jsonp polling iframe removal error",t)}try{var e='<iframe src="javascript:0" name="'+o.iframeId+'">';i=document.createElement(e)}catch(t){i=document.createElement("iframe"),i.name=o.iframeId,i.src="javascript:0"}i.id=o.iframeId,o.form.appendChild(i),o.iframe=i}var o=this;if(!this.form){var i,s=document.createElement("form"),a=document.createElement("textarea"),u=this.iframeId="eio_iframe_"+this.index;s.className="socketio",s.style.position="absolute",s.style.top="-1000px",s.style.left="-1000px",s.target=u,s.method="POST",s.setAttribute("accept-charset","utf-8"),a.name="d",s.appendChild(a),document.body.appendChild(s),this.form=s,this.area=a}this.form.action=this.uri(),r(),t=t.replace(p,"\\\n"),this.area.value=t.replace(c,"\\n");try{this.form.submit()}catch(h){}this.iframe.attachEvent?this.iframe.onreadystatechange=function(){"complete"===o.iframe.readyState&&n()}:this.iframe.onload=n}}).call(e,function(){return this}())},function(t,e,n){(function(e){function r(t){var e=t&&t.forceBase64;e&&(this.supportsBinary=!1),this.perMessageDeflate=t.perMessageDeflate,this.usingBrowserWebSocket=h&&!t.forceNode,this.protocols=t.protocols,this.usingBrowserWebSocket||(l=o),i.call(this,t)}var o,i=n(22),s=n(23),a=n(31),c=n(32),p=n(33),u=n(3)("engine.io-client:websocket"),h=e.WebSocket||e.MozWebSocket;if("undefined"==typeof window)try{o=n(36)}catch(f){}var l=h;l||"undefined"!=typeof window||(l=o),t.exports=r,c(r,i),r.prototype.name="websocket",r.prototype.supportsBinary=!0,r.prototype.doOpen=function(){if(this.check()){var t=this.uri(),e=this.protocols,n={agent:this.agent,perMessageDeflate:this.perMessageDeflate};n.pfx=this.pfx,n.key=this.key,n.passphrase=this.passphrase,n.cert=this.cert,n.ca=this.ca,n.ciphers=this.ciphers,n.rejectUnauthorized=this.rejectUnauthorized,this.extraHeaders&&(n.headers=this.extraHeaders),this.localAddress&&(n.localAddress=this.localAddress);try{this.ws=this.usingBrowserWebSocket?e?new l(t,e):new l(t):new l(t,e,n)}catch(r){return this.emit("error",r)}void 0===this.ws.binaryType&&(this.supportsBinary=!1),this.ws.supports&&this.ws.supports.binary?(this.supportsBinary=!0,this.ws.binaryType="nodebuffer"):this.ws.binaryType="arraybuffer",this.addEventListeners()}},r.prototype.addEventListeners=function(){var t=this;this.ws.onopen=function(){t.onOpen()},this.ws.onclose=function(){t.onClose()},this.ws.onmessage=function(e){t.onData(e.data)},this.ws.onerror=function(e){t.onError("websocket error",e)}},r.prototype.write=function(t){function n(){r.emit("flush"),setTimeout(function(){r.writable=!0,r.emit("drain")},0)}var r=this;this.writable=!1;for(var o=t.length,i=0,a=o;i<a;i++)!function(t){s.encodePacket(t,r.supportsBinary,function(i){if(!r.usingBrowserWebSocket){var s={};if(t.options&&(s.compress=t.options.compress),r.perMessageDeflate){var a="string"==typeof i?e.Buffer.byteLength(i):i.length;a<r.perMessageDeflate.threshold&&(s.compress=!1)}}try{r.usingBrowserWebSocket?r.ws.send(i):r.ws.send(i,s)}catch(c){u("websocket closed before onclose event")}--o||n()})}(t[i])},r.prototype.onClose=function(){i.prototype.onClose.call(this)},r.prototype.doClose=function(){"undefined"!=typeof this.ws&&this.ws.close()},r.prototype.uri=function(){var t=this.query||{},e=this.secure?"wss":"ws",n="";this.port&&("wss"===e&&443!==Number(this.port)||"ws"===e&&80!==Number(this.port))&&(n=":"+this.port),this.timestampRequests&&(t[this.timestampParam]=p()),this.supportsBinary||(t.b64=1),t=a.encode(t),t.length&&(t="?"+t);var r=this.hostname.indexOf(":")!==-1;return e+"://"+(r?"["+this.hostname+"]":this.hostname)+n+this.path+t},r.prototype.check=function(){return!(!l||"__initialize"in l&&this.name===r.prototype.name)}}).call(e,function(){return this}())},function(t,e){},function(t,e){var n=[].indexOf;t.exports=function(t,e){if(n)return t.indexOf(e);for(var r=0;r<t.length;++r)if(t[r]===e)return r;return-1}},function(t,e){(function(e){var n=/^[\],:{}\s]*$/,r=/\\(?:["\\\/bfnrt]|u[0-9a-fA-F]{4})/g,o=/"[^"\\\n\r]*"|true|false|null|-?\d+(?:\.\d*)?(?:[eE][+\-]?\d+)?/g,i=/(?:^|:|,)(?:\s*\[)+/g,s=/^\s+/,a=/\s+$/;t.exports=function(t){return"string"==typeof t&&t?(t=t.replace(s,"").replace(a,""),e.JSON&&JSON.parse?JSON.parse(t):n.test(t.replace(r,"@").replace(o,"]").replace(i,""))?new Function("return "+t)():void 0):null}}).call(e,function(){return this}())},function(t,e,n){"use strict";function r(t,e,n){this.io=t,this.nsp=e,this.json=this,this.ids=0,this.acks={},this.receiveBuffer=[],this.sendBuffer=[],this.connected=!1,this.disconnected=!0,n&&n.query&&(this.query=n.query),this.io.autoConnect&&this.open()}var o="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function(t){return typeof t}:function(t){return t&&"function"==typeof Symbol&&t.constructor===Symbol&&t!==Symbol.prototype?"symbol":typeof t},i=n(7),s=n(8),a=n(40),c=n(41),p=n(42),u=n(3)("socket.io-client:socket"),h=n(31);t.exports=e=r;var f={connect:1,connect_error:1,connect_timeout:1,connecting:1,disconnect:1,error:1,reconnect:1,reconnect_attempt:1,reconnect_failed:1,reconnect_error:1,reconnecting:1,ping:1,pong:1},l=s.prototype.emit;s(r.prototype),r.prototype.subEvents=function(){if(!this.subs){var t=this.io;this.subs=[c(t,"open",p(this,"onopen")),c(t,"packet",p(this,"onpacket")),c(t,"close",p(this,"onclose"))]}},r.prototype.open=r.prototype.connect=function(){return this.connected?this:(this.subEvents(),this.io.open(),"open"===this.io.readyState&&this.onopen(),this.emit("connecting"),this)},r.prototype.send=function(){var t=a(arguments);return t.unshift("message"),this.emit.apply(this,t),this},r.prototype.emit=function(t){if(f.hasOwnProperty(t))return l.apply(this,arguments),this;var e=a(arguments),n={type:i.EVENT,data:e};return n.options={},n.options.compress=!this.flags||!1!==this.flags.compress,"function"==typeof e[e.length-1]&&(u("emitting packet with ack id %d",this.ids),this.acks[this.ids]=e.pop(),n.id=this.ids++),this.connected?this.packet(n):this.sendBuffer.push(n),delete this.flags,this},r.prototype.packet=function(t){t.nsp=this.nsp,this.io.packet(t)},r.prototype.onopen=function(){if(u("transport is open - connecting"),"/"!==this.nsp)if(this.query){var t="object"===o(this.query)?h.encode(this.query):this.query;u("sending connect packet with query %s",t),this.packet({type:i.CONNECT,query:t})}else this.packet({type:i.CONNECT})},r.prototype.onclose=function(t){u("close (%s)",t),this.connected=!1,this.disconnected=!0,delete this.id,this.emit("disconnect",t)},r.prototype.onpacket=function(t){if(t.nsp===this.nsp)switch(t.type){case i.CONNECT:this.onconnect();break;case i.EVENT:this.onevent(t);break;case i.BINARY_EVENT:this.onevent(t);break;case i.ACK:this.onack(t);break;case i.BINARY_ACK:this.onack(t);break;case i.DISCONNECT:this.ondisconnect();break;case i.ERROR:this.emit("error",t.data)}},r.prototype.onevent=function(t){var e=t.data||[];u("emitting event %j",e),null!=t.id&&(u("attaching ack callback to event"),e.push(this.ack(t.id))),this.connected?l.apply(this,e):this.receiveBuffer.push(e)},r.prototype.ack=function(t){var e=this,n=!1;return function(){if(!n){n=!0;var r=a(arguments);u("sending ack %j",r),e.packet({type:i.ACK,id:t,data:r})}}},r.prototype.onack=function(t){var e=this.acks[t.id];"function"==typeof e?(u("calling ack %s with %j",t.id,t.data),e.apply(this,t.data),delete this.acks[t.id]):u("bad ack %s",t.id)},r.prototype.onconnect=function(){this.connected=!0,this.disconnected=!1,this.emit("connect"),this.emitBuffered()},r.prototype.emitBuffered=function(){var t;for(t=0;t<this.receiveBuffer.length;t++)l.apply(this,this.receiveBuffer[t]);for(this.receiveBuffer=[],t=0;t<this.sendBuffer.length;t++)this.packet(this.sendBuffer[t]);this.sendBuffer=[]},r.prototype.ondisconnect=function(){u("server disconnect (%s)",this.nsp),this.destroy(),this.onclose("io server disconnect")},r.prototype.destroy=function(){if(this.subs){for(var t=0;t<this.subs.length;t++)this.subs[t].destroy();this.subs=null}this.io.destroy(this)},r.prototype.close=r.prototype.disconnect=function(){return this.connected&&(u("performing disconnect (%s)",this.nsp),this.packet({type:i.DISCONNECT})),this.destroy(),this.connected&&this.onclose("io client disconnect"),this},r.prototype.compress=function(t){return this.flags=this.flags||{},this.flags.compress=t,this}},function(t,e){function n(t,e){var n=[];e=e||0;for(var r=e||0;r<t.length;r++)n[r-e]=t[r];return n}t.exports=n},function(t,e){"use strict";function n(t,e,n){return t.on(e,n),{destroy:function(){t.removeListener(e,n)}}}t.exports=n},function(t,e){var n=[].slice;t.exports=function(t,e){if("string"==typeof e&&(e=t[e]),"function"!=typeof e)throw new Error("bind() requires a function");var r=n.call(arguments,2);return function(){return e.apply(t,r.concat(n.call(arguments)))}}},function(t,e){function n(t){t=t||{},this.ms=t.min||100,this.max=t.max||1e4,this.factor=t.factor||2,this.jitter=t.jitter>0&&t.jitter<=1?t.jitter:0,this.attempts=0}t.exports=n,n.prototype.duration=function(){var t=this.ms*Math.pow(this.factor,this.attempts++);if(this.jitter){var e=Math.random(),n=Math.floor(e*this.jitter*t);t=0==(1&Math.floor(10*e))?t-n:t+n}return 0|Math.min(t,this.max)},n.prototype.reset=function(){this.attempts=0},n.prototype.setMin=function(t){this.ms=t},n.prototype.setMax=function(t){this.max=t},n.prototype.setJitter=function(t){this.jitter=t}}])});
//# sourceMappingURL=socket.io.js.map;
/*******************************************************************************
 * @license
 * Copyright (c) 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd, browser*/
define('plugins/languages/java/ipc',[
	"orion/Deferred",
	"socketio/socket.io"
], function(Deferred, io) {
	
	var messageTypes = {
		/**
		 * @description The initialize request is sent as the first request from the client to the server.
		 * @kind request
		 * @see https://github.com/Microsoft/language-server-protocol/blob/master/protocol.md#initialize-request
		 */
		initialize: 'initialize',
		/**
		 * @kind notification
		 * @see https://github.com/Microsoft/language-server-protocol/blob/master/protocol.md#logmessage-notification
		 */
		logMessage: 'window/logMessage',
		/**
		 * @kind notification
		 * @see https://github.com/Microsoft/language-server-protocol/blob/master/protocol.md#status-notification
		 */
		status: 'language/status'
	};
	
	/**
	 * @name IPC
	 * @description Creates a new IPC class
	 * @param {String} channel The channel
	 * @param {String} language The id of the language 
	 * @returns {IPC} A new class
	 * @since 13.0
	 */
	function IPC(channel) {
		this.socket = null;
		this.channel = channel;
		this.id = 1;
		this.requests = {};
		this.initialized = false;
		this.queue = [];
	}
	/**
	 * The collection of message types corresponding to the launguage server protocol
	 * @see https://github.com/Microsoft/language-server-protocol/blob/master/protocol.md
	 */
	IPC.prototype.MESSAGE_TYPES = messageTypes; //TODO should be a clone, not a live copy

	/**
	 * @name IPC.prototype.sendMessage
	 * @description Send a message over the socket
	 * @function
	 * @param {number} id The id of the message
	 * @param {String} message The name of the message to send
	 * @param {?} params The object of parameters for the message
	 */
	IPC.prototype.sendMessage = function sendMessage(id, message, params) {
		var json = {
			"jsonrpc": "2.0",
			"method": message,
			"params": params
		};
		if (id) {
			json.id = id;
		}
		if (!this.initialized && message !== this.MESSAGE_TYPES.initialize) {
			this.queue.push(json);
		} else {
			this.socket.emit('data', json);
		}
		if (id) {
			return this.requests[id] = new Deferred();
		}
	};
	
	/**
	 * @name _notifyListeners
	 * @description Notify the given list of listeners with the given data. If no type is given, 'data.method' will be queried. If there is no
	 * 'data.method' property, no work is done
	 * @private
	 * @param {Array.<{}>} listeners The list of listeners to notify
	 * @param {String} type The type of listener to notify 
	 * @param {?} data The data to tell the listeneres about
	 */
	IPC.prototype._notifyListeners = function _notifyListeners(type, data) {
		if (type) {
			var evt = {
				type: type,
				data: data
			};
			this.lspService.dispatchEvent(evt);
		}
	}

	/**
	 * @name IPC.prototype.connect
	 * @description Connects to the class channel name 
	 * @function
	 */
	IPC.prototype.connect = function connect() {
		var d = new Deferred();
		this.socket = io('/jdt')//.connect(this.channel);

		this.socket.on('connect', function() {
			this.socket.emit('start');
		}.bind(this));
		this.socket.on('fail', function(error) {
			console.log(error);
		});
		this.socket.on('error', function(error) {
			console.log(error);
		});
		this.socket.on('data', function(data) {
			try {
				if(!data) {
					this._notifyListeners(this.MESSAGE_TYPES.logMessage, "Dropped response with null data.");
					return;
				}
				if (data && data.id) {
					var deferred = this.requests[data.id];
					if(deferred) {
						if(data.error) {
							deferred.reject(data.error);
						} else {
							deferred.resolve(data.result);
						}
						delete this.requests[data.id];
					}
				}
				this._notifyListeners(data.method, data);
			} catch(err) {
				this._notifyListeners(this.MESSAGE_TYPES.logMessage, err.toString());
			}
		}.bind(this));
		this.socket.on('ready', function(data) {
			var pid;
			try {
				var json  = JSON.parse(data);
				this.workspaceDir = json.workspaceDir;
				pid = json.processId;
			} catch(err) {
				this._notifyListeners(this.MESSAGE_TYPES.logMessage, err.toString());
			}
			this.initialize(pid, this.workspaceDir).then(/* @callback */ function initializeCallback(result) {
				this.initialized = true;
				this.capabilities = result.capabilities;
				d.resolve(this.capabilities);
				this.queue.forEach(function queueFlushCallback(item) {
					this.socket.emit('data', item);
					this._notifyListeners(this.MESSAGE_TYPES.logMessage, JSON.stringify(item));
				}.bind(this));
				this.queue = [];
			}.bind(this));
		}.bind(this));
		return d;
	};
	
		/**
	 * @name IPC.prototype.initialize
	 * @description The initialize request is sent as the first request from the client to the server.
	 * @param {String} processId The id of the process
	 * @param {String} workspaceDir The root of the current workspace 
	 * @function
	 * @returns {Deferred} The deferred that resolves to the result of the request
	 */
	IPC.prototype.initialize = function initialize(processId, workspaceDir) {
		return this.sendMessage(this.id++, messageTypes.initialize, {
			rootPath: workspaceDir,
			processId: processId,
			capabilities: {
				textDocument: {
					completion: {
						completionItem: {
							snippetSupport: true
						}
					}
				}
			}
		});
	};

	return IPC;
});
/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2018 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/java/javaPlugin',[
	'orion/plugin',
	'orion/editor/stylers/text_x-java-source/syntax', 
	'orion/editor/stylers/application_x-jsp/syntax',
	'orion/serviceregistry',
	'plugins/languages/java/ipc',
	'orion/EventTarget'
], function(PluginProvider, mJava, mJSP, mServiceRegistry, IPC, EventTarget) {

	return {
		connect: function connect() {
			var headers = {
				name: "Orion Java Tool Support",
				version: "1.0",
				description: "This plugin provides Java tools support for Orion."
			};
			var pluginProvider = new PluginProvider(headers);
			registerServiceProviders(pluginProvider);
			pluginProvider.connect();
		},
		/**
		 * @callback
		 */
		registerServiceProviders: function registerServiceProviders(pluginProvider) {
			// register the content type
			pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
				contentTypes: [
					{
						id: "text/x-java-source",
						"extends": "text/plain",
						name: "Java",
						extension: ["java"]
					},
					{
						id: "application/x-jsp",
						"extends": "text/plain",
						name: "Java Server Page",
						extension: ["jsp"]
					}
				]
			});
			// syntax highlighting
			mJava.grammars.forEach(function(current) {
				pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
			});
			mJSP.grammars.forEach(function(current) {
				pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
			});
			// var ipc = new IPC("/jdt");
			// function LSPService() {
			// 	EventTarget.attach(this);
			// 	ipc.lspService = this;
			// }
			// LSPService.prototype = {
			// 	sendMessage: function(id, message, params) {
			// 		return ipc.sendMessage(id, message, params);
			// 	},
			// 	start: function () {
			// 		return ipc.connect();
			// 	}
			// };
			// pluginProvider.registerService("orion.languages.server", //$NON-NLS-1$
			// 	new LSPService(),
			// 	{
			// 		languageId: "java",
			// 		name: "Java Symbol Outline",
			// 		title: "Java Symbols",
			// 		contentType: ["text/x-java-source", "application/x-jsp"]  //$NON-NLS-1$ //$NON-NLS-2$
			// 	}
			// );
		}
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_jsx/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/application_javascript/syntax", "orion/editor/stylers/application_xml/syntax"],
	/* eslint-disable missing-nls */
	function(mLib, mJS, mXML) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push.apply(grammars, mXML.grammars);
	grammars.push({
		id: "orion.jsx",
		contentTypes: ["text/jsx"],
		patterns: [
			{
				begin: "</?[A-Za-z0-9]+",
				end: "/?>",
				captures: {
					0: {name: "meta.tag.jsx"}
				},
				patterns: [
					{include: "#jsExpression"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
					{include: "orion.c-like#comment_singleLine"},
					{include: "orion.c-like#comment_block"}
				]
			}, {
				include: "orion.xml#ampersandEscape"
			}, {
				include: "orion.js"
			}
		],
		repository: {
			jsExpression: {
				begin: "\\{",
				end: "\\}",
				contentName: "source.js.embedded.jsx",
				patterns: [
					{include: "orion.js"}
				]
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/jsx/jsxPlugin',['orion/plugin', 'orion/editor/stylers/text_jsx/syntax'], function(PluginProvider, mJSX) {

	function connect() {
		var headers = {
			name: "Orion JSX Tool Support",
			version: "1.0",
			description: "This plugin provides JSX tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/jsx",
					"extends": "text/plain",
					name: "JSX",
					extension: ["jsx"]
				}
			] 
		});
		mJSX.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/launch/launchPlugin',['orion/plugin'], function(PluginProvider) {

	function connect() {
		var headers = {
			name: "Orion Launch File Tool Support",
			version: "1.0",
			description: "This plugin provides tools support for Orion Launch files."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-launch",
					"extends": "application/json",
					name: "Launch file",
					extension: ["launch"]
				}
			] 
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-less/syntax", ["orion/editor/stylers/text_css/syntax", "orion/editor/stylers/lib/syntax"], function(mCSS, mLib) {
	var flags = ["important"];
	var constants = ["true"];
	var directives = ["arguments"];
	var keywords = ["all", "and", "not", "when"];
	var importKeywords = [
		"css",
		"inline",
		"less",
		"multiple",
		"once", "optional",
		"reference"
	];

	var grammars = [];
	grammars.push.apply(grammars, mCSS.grammars);
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.less",
		contentTypes: ["text/x-less"],
		patterns: [
			{include: "orion.css#string_single_multiline"},
			{include: "orion.css#string_double_multiline"},
			{include: "orion.c-like#comment_block"},
			{include: "#string_doubleQuote"},
			{include: "#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "#importDirective"},
			{include: "orion.css#directive"},
			{include: "#directive"},
			{include: "#variable"},
			{include: "#interpolated"},
			{include: "#constant"},
			{include: "#flag"},
			{include: "#operator"},
			{include: "#keyword"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.css#number_hex"},
			{include: "orion.css#numeric_value"},
			{include: "orion.css#color"},
			{include: "orion.css#keyword"}
		],
		repository: {
			constant: {
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.less"
			},
			directive: {
				match: "(^|\\s)(@("  + directives.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.directive.less"}
				}
			},
			flag: {
				match: "(^|\\s)(!("  + flags.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.flag.less"}
				}
			},
			importDirective: {
				begin: "(@import)\\s*\\(",
				end: "\\)",
				beginCaptures: {
					1: {name: "keyword.other.directive.less"}
				},
				patterns: [{
					match: "\\b(?:" + importKeywords.join("|") + ")\\b",
					name: "keyword.operator.less"
				}]
			},
			interpolated: {
				match: "@\\{[^}]*\\}",
				name: "string.interpolated.less"
			},
			keyword: {
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.less"
			},
			operator: {
				match: "\\+_?|\\*|\\/|=|>=?|<=?|&",
				name: "punctuation.operator.less"
			},
			string_doubleQuote: {
				match: '~?"(?:\\\\.|[^"])*"?',
				name: "string.quoted.double"
			},
			string_singleQuote: {
				match: "~?'(?:\\\\.|[^'])*'?",
				name: "string.quoted.single"
			},
			variable: {
				match: "@[\\w-]+(\\.{3})?",
				name: "variable.other.less"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: directives.concat(flags).concat(importKeywords).concat(keywords)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/less/lessPlugin',['orion/plugin', 'orion/editor/stylers/text_x-less/syntax'], function(PluginProvider, mLess) {

	function connect() {
		var headers = {
			name: "Orion Less Tool Support",
			version: "1.0",
			description: "This plugin provides Less tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-less",
					"extends": "text/plain",
					name: "Less",
					extension: ["less"]
				}
			] 
		});
		mLess.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-lua/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	// from http://www.lua.org/ftp/refman-5.0.pdf
	var keywords = [
		"and",
		"break",
		"do",
		"else", "elseif", "end",
		"false", "for", "function",
		"if", "in",
		"local",
		"nil", "not",
		"or",
		"repeat", "return",
		"then", "true",
		"until",
		"while",
	];

	// base library functions and variables
	// from http://lua-users.org/files/wiki_insecure/users/thomasl/luarefv51.pdf
	var base_functions = [
		"assert", "arg",
		"collectgarbage",
		"dofile",
		"error",
		"getfenv", "getmetatable",
		"ipairs",
		"load", "loadfile", "loadstring",
		"next",
		"pairs", "pcall", "print",
		"rawequal", "rawget", "rawset", "require",
		"select", "setfenv", "setmetatable",
		"tonumber", "tostring", "type",
		"unpack",
		"xpcall"
	];
	var base_variables = [
		"_G",
		"LUA_INIT", "LUA_PATH", "LUA_CPATH",
		"_VERSION"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.lua",
		contentTypes: ["text/x-luasrc", "text/x-lua"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "#comment_block_dash_dash"},
			{include: "#comment_singleLine_dash_dash"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{include: "#base_functions"},
			{include: "#base_variables"},
			{include: "#reserved_underscore_capital"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.lua"
			}
		],
		repository: {
			comment_block_dash_dash: {
				begin: {match: "--\\[\\[", literal: "--[["},
				end: {match: "\\]\\]", literal: "]]"},
				name: "comment.block.dash-dash.lua",
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!\\]\\]).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.block"}
						}
					}
				]
			},
			comment_singleLine_dash_dash: {
				begin: {match: "--", literal: "--"},
				end: {match: "$", literal: ""},
				name: "comment.line.dash-dash.lua",
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			base_functions: {
				match: "\\b(?:" + base_functions.join("|") + ")\\b",
				name: "support.function.lua"
			},
			base_variables: {
				match: "\\b(?:" + base_variables.join("|") + ")\\b",
				name: "support.variable.lua"
			},
			// _ANYCAPITALS is reserved in Lua
			reserved_underscore_capital: {
				match: "\\b_[A-Z]*\\b",
				name: "constant.other.userdefined.lua"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/lua/luaPlugin',['orion/plugin', 'orion/editor/stylers/text_x-lua/syntax'], function(PluginProvider, mLua) {

	function connect() {
		var headers = {
			name: "Orion Lua Tool Support",
			version: "1.0",
			description: "This plugin provides Lua tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-lua",
					"extends": "text/plain",
					name: "Lua",
					extension: ["lua"]
				}
			] 
		});
		mLua.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/nls/messages',["module"],function(module){
    var config = module.config();
    return config && config.root ? config : {root:true};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2012, 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/nls/root/messages',{//Default message bundle
	"Navigator": "Navigator",
	"Sites": "Sites",
	"Shell": "Shell",
	"Get Plugins": "Get Plug-ins",
	"Global": "Global",
	"Editor": "Editor",
	"EditorRelatedLink": "Show Current Folder",
	"EditorRelatedLinkParent": "Show Enclosing Folder",
	"EditorLinkWorkspace": "Edit",
	"EditorRelatedLinkProj": "Show Project",
	"sidebar": "Sidebar",
	"toolbar": "Toolbar",
	"Filter bindings:": "Filter bindings:",
	"Filter bindings": "Type characters to filter by name or key combination",
	"BindingPrompt": "Enter the new binding",
	"NoBinding": "---",
	"orionClientLabel": "Orion client repository",
	"Orion Editor": "Text Editor",
	"Orion Image Viewer": "Image Viewer",
	"Orion Markdown Editor": "Markdown Editor",
	"Orion Markdown Viewer": "Markdown Viewer",
	"Orion JSON Editor": "JSON Editor",
	"Orion System Editor": "System Editor",
	"View on Site": "View on Site",
	"View this file or folder on a web site hosted by Orion": "View this file or folder on a web site hosted by Orion.",
	"ShowAllKeyBindings": "Show a list of all the keybindings on this page",
	"Show Keys": "Show Keys",
	"HideShowBannerFooter": "Hide or show the page banner and footer",
	"Toggle Banner and Footer": "Toggle Banner and Footer",
	"ChooseFileOpenEditor": "Choose a file by name and open an editor on it",
	"FindFile": "Open File...",
	"System Configuration Details": "System Configuration Details",
	"System Config Tooltip": "Go to the System Configuration Details page",
	"System Editor Tooltip": "Open this file type in the desktop's default manner",
	"Background Operations": "Background Operations",
	"Background Operations Tooltip": "Go to the Background Operations page",
	"Operation status is unknown": "Operation status is unknown",
	"Unknown item": "Unknown item",
	"NoSearchAvailableErr": "Can't search: no search service is available",
	"Related": "Related",
	"Options": "Options",
	"LOG: ": "LOG: ",
	"View": "View",
	"SplashTitle": "Setting up Workspace",
	"SplashTitleSettings": "Loading Settings",
	"SplashTitleGit": "Loading Git Repositories",
	"LoadingPage": "Loading Page",
	"LoadingPlugins": "Loading Plugins",
	"AuthenticatingUser": "Authenticating user...",
	"AuthenticatedUser": "Authenticated user",
	"LoadingResources": "Loading Resources",
	"plugin_started": "\"${0}\" started",
	"plugin_lazy activation": "\"${0}\" lazily activated",
	"plugin_starting": "\"${0}\" starting",
	"no parent": "no parent",
	"no tree model": "no tree model",
	"no renderer": "no renderer",
	"could not find table row ": "could not find table row ",
	"Operations": "Operations",
	"Operations running": "Operations running",
	"SomeOpWarning": "Some operations finished with warning",
	"SomeOpErr": "Some operations finished with error",
	"no service registry": "no service registry",
	"Tasks": "Tasks",
	"Close": "Close",
	"Expand all": "Expand all",
	"Collapse all": "Collapse all",
	"Search" : "Search",
	"Advanced search" : "Advanced search",
	"Submit" : "Submit",
	"More" : "More",
	"Recent searches" : "Recent searches",
	"Regular expression" : "Regular expression",
	"Search options" : "Search options",
	"Global search" : "Global search",
	"Orion Home" : "Home",
	"Close notification" : "Close notification",
	"Toggle Side Panel" : "Toggle Side Panel",
	"Open or close the side panel": "Open or close the side panel",
	"Projects" : "Projects",
	"Toggle Sidebar" : "Toggle Sidebar",
	"Sample HTML5 Site": "Sample HTML5 Site",
	"Generate an HTML5 'Hello World' website, including JavaScript, HTML, and CSS files.": "Generate an HTML5 'Hello World' website, including JavaScript, HTML, and CSS files.",
	"Sample Orion Plugin": "Sample Orion Plug-in",
	"Generate a sample plugin for integrating with Orion.": "Generate a sample plug-in for integrating with Orion.",
	"Browser": "Web Browser",
	"OutlineProgress": "Getting outline for ${0} from ${1}",
	"FormatProgress" : "Formatting ${0} from ${1}",
	"outlineTimeout": "Outline service timed out. Try reloading the page and opening the outline again.",
	"UnknownError": "An unknown error occurred.",
	"Filter outline:": "Filter outline:",
	"Filter": "Type characters (* = any string, ? = any character)",
	"TemplateExplorerLabel": "Templates",
	"OpenTemplateExplorer": "Open Template Explorer",
	"Edit": "Edit",
	"CentralNavTooltip": "Toggle Navigation Menu",
	"Wrote: ${0}": "Wrote: ${0}",
	"GenerateHTML": "Generate HTML file",
	"GenerateHTMLTooltip": "Write an HTML file generated from the current Markdown editor content",
	"alt text": "alt text",
	"blockquote": "blockquote",
	"code": "code",
	"code (block)": "code (block)",
	"code (span)": "code (span)",
	"emphasis": "emphasis",
	"fenced code (${0})": "fenced code (${0})",
	"header (${0})": "header (${0})",
	"horizontal rule": "horizontal rule",
	"label": "label",
	"link (auto)": "link (auto)",
	"link (image)": "link (image)",
	"link (inline)": "link (inline)",
	"link label": "link label",
	"link label (optional)": "link label (optional)",
	"link (ref)": "link (ref)",
	"list item (bullet)": "list item (bullet)",
	"list item (numbered)": "list item (numbered)",
	"strikethrough (${0})": "strikethrough (${0})",
	"strong": "strong",
	"table (${0})": "table (${0})",
	"text": "text",
	"title (optional)": "title (optional)",
	"url": "url",
	"workedProgress": "${0} (${1}/${2})",
	"ConfirmLogout": "Do you want to logout?",
	"VerticalPaneOrientation": "Vertical pane orientation",
	"TogglePaneOrientationTooltip": "Toggle split pane orientation",
	"WarningDuplicateLinkId": "Duplicate link ID: ${0} (link IDs are not case-sensitive)",
	"WarningHeaderTooDeep": "Header level cannot exceed 6",
	"WarningLinkHasNoText": "Link has no text",
	"WarningLinkHasNoURL": "Link has no URL",
	"WarningOrderedListItem": "Ordered list item within unordered list",
	"WarningOrderedListShouldStartAt1": "The first item in an ordered list should have index 1",
	"WarningUndefinedLinkId": "Undefined link ID: ${0}",
	"WarningUnorderedListItem": "Unordered list item within ordered list",
	"PageTitleFormat": "${0} - ${1}", // ${0} is the file or resource being edited; ${1} is the task (eg. "Editor")
	// Display names for keys:
	"KeyCTRL": "Ctrl",
	"KeySHIFT": "Shift",
	"KeyALT": "Alt",
	"KeyBKSPC": "Backspace",
	"KeyDEL": "Del",
	"KeyEND": "End",
	"KeyENTER": "Enter",
	"KeyESCAPE": "Esc",
	"KeyHOME": "Home",
	"KeyINSERT": "Ins",
	"KeyPAGEDOWN": "Page Down",
	"KeyPAGEUP": "Page Up",
	"KeySPACE": "Space",
	"KeyTAB": "Tab",
	// Display elapsed time:
	"a year": "a year",
	"years": "${0} years",
	"a month": "a month",
	"months": "${0} months",
	"a day": "a day",
	"days": "${0} days",
	"an hour": "an hour",
	"hours": "${0} hours",
	"a minute": "a minute",
	"minutes": "${0} minutes",
	"timeAgo": "${0} ago", //${0} represent the time elapsed
	"justNow": "just now", //Represent that the time elapsed is less than 1 minute
	"fixAll": "Fix all",
	"nextSplitter" : "Next Splitter",
	"nextSplitterTooltip": "Move focus through the available splitters",
	"Confirm": "Confirm"
});


/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser,amd*/
define('plugins/languages/markdown/markdownPlugin',['orion/plugin', 'orion/i18nUtil', 'i18n!orion/nls/messages'], function (PluginProvider, i18nUtil, messages) { //$NON-NLS-0$

	function connect() {
		var headers = {
			name: "Orion Extended Markdown Language Tool Support",
			version: "1.0",
			description: "This plugin provides extended Markdown language tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-markdown",
					"extends": "text/plain",
					name: "Markdown",
					extension: ["md"]
				}
			] 
		});
		pluginProvider.registerServiceProvider("orion.edit.contentAssist", { //$NON-NLS-0$
			computeProposals: function (buffer, offset, context) {
				var selection = context.selection.start !== context.selection.end ?
					buffer.substring(context.selection.start, context.selection.end) :
					null;
				var NL = context.delimiter;
				var multilineSelection = selection && selection.indexOf(NL) !== -1;
				var onEmptyLine = /^[ ]{0,3}$/.test(context.line);
				var result = [];

				result.push({
					description: messages["emphasis"],
					escapePosition: selection ? null : offset + 1,
					proposal: "*" + (selection ? selection : "") + "*" //$NON-NLS-1$ //$NON-NLS-0$
				});

				result.push({
					description: messages["strong"],
					escapePosition: selection ? null : offset + 2,
					proposal: "**" + (selection ? selection : "") + "**" //$NON-NLS-1$ //$NON-NLS-0$
				});

				if (!multilineSelection) {
					var headerText = onEmptyLine && !selection ? messages["text"] : "";
					result.push({
						description: i18nUtil.formatMessage(messages["header (${0})"], "atx"), //$NON-NLS-0$
						positions: onEmptyLine ? [{offset: offset + 1, length: headerText.length}] : null,
						proposal: (onEmptyLine ? "" : NL) + "#" + headerText + (selection ? selection : "") //$NON-NLS-0$
					});
				}

				if (!multilineSelection) {
					result.push({
						description: messages["link (auto)"],
						escapePosition: selection ? null : offset + 1,
						proposal: "<" + (selection ? selection : "") + ">" //$NON-NLS-1$ //$NON-NLS-0$
					});
				}

				var inlineLinkText = selection || messages["text"];
				var inlineLinkUrl = messages["url"];
				var inlineLinkTitle = messages["title (optional)"];
				result.push({
					description: messages["link (inline)"],
					positions: [
						{offset: offset + 1, length: inlineLinkText.length},
						{offset: offset + 3 + inlineLinkText.length, length: inlineLinkUrl.length},
						{offset: offset + 5 + inlineLinkText.length + inlineLinkUrl.length, length: inlineLinkTitle.length}],
					proposal: "[" + inlineLinkText + "](" + inlineLinkUrl + " \"" + inlineLinkTitle + "\")" //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
				});

				var imageLinkText = selection || messages["alt text"];
				var imageLinkUrl = messages["url"];
				var imageLinkTitle = messages["title (optional)"];
				result.push({
					description: messages["link (image)"],
					positions: [
						{offset: offset + 2, length: imageLinkText.length},
						{offset: offset + 4 + imageLinkText.length, length: imageLinkUrl.length},
						{offset: offset + 6 + imageLinkText.length + imageLinkUrl.length, length: imageLinkTitle.length}],
					proposal: "![" + imageLinkText + "](" + imageLinkUrl + " \"" + imageLinkTitle + "\")" //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
				});
	
				var refLinkText = selection || messages["text"];
				var refLinkLabel = messages["link label (optional)"];
				result.push({
					description: messages["link (ref)"],
					positions: [
						{offset: offset + 1, length: refLinkText.length},
						{offset: offset + 3 + refLinkText.length, length: refLinkLabel.length}],
					proposal: "[" + refLinkText + "][" + refLinkLabel + "]" //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
				});

				if (!multilineSelection) {
					var linkLabelText = selection || messages["label"];
					var linkLabelUrl = messages["url"];
					var linkLabelTitle = messages["title (optional)"];
					var linkLabelNL = onEmptyLine ? "" : NL;
					result.push({
						description: messages["link label"],
						positions: [
							{offset: offset + linkLabelNL.length + 1, length: linkLabelText.length},
							{offset: offset + linkLabelNL.length + 4 + linkLabelText.length, length: linkLabelUrl.length},
							{offset: offset + linkLabelNL.length + 5 + linkLabelText.length + linkLabelUrl.length, length: linkLabelTitle.length}],
						proposal: linkLabelNL + "[" + linkLabelText + "]: " + linkLabelUrl + " " + linkLabelTitle + NL //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
					});
				}

				if (!multilineSelection) {
					var codeBlockText = onEmptyLine && !selection ? messages["code"] : "";
					var indent = onEmptyLine ? "    ".substring(0, 4 - context.indentation.length) : NL + NL + "    "; //$NON-NLS-1$ //$NON-NLS-0$
					result.push({
						description: messages["code (block)"],
						positions: codeBlockText ? [{offset: offset + indent.length, length: codeBlockText.length}] : null,
						proposal: indent + codeBlockText + (selection ? selection : "")
					});
				}

				result.push({
					description: messages["code (span)"],
					escapePosition: selection ? null : offset + 1,
					proposal: "`" + (selection ? selection : "") + "`" //$NON-NLS-1$ //$NON-NLS-0$
				});

				result.push({
					description: messages["horizontal rule"],
					proposal: (onEmptyLine ? "" : NL) + "- - -" + NL + (selection ? selection : "") //$NON-NLS-0$
				});

				if (!multilineSelection) {
					var blockquoteText = onEmptyLine && !selection ? messages["text"] : "";
					result.push({
						description: messages["blockquote"],
						positions: onEmptyLine ? [{offset: offset + 2, length: blockquoteText.length}] : null,
						proposal: (onEmptyLine ? "" : NL) + "> " + blockquoteText + (selection ? selection : "") //$NON-NLS-0$
					});
				}

				var listItemText = onEmptyLine && !selection ? messages["text"] : "";
				result.push({
					description: messages["list item (numbered)"],
					positions: onEmptyLine ? [{offset: offset + 3, length: listItemText.length}] : null,
					proposal: (onEmptyLine ? "" : NL) + "1. " + listItemText + (selection ? selection : "") //$NON-NLS-0$
				});

				result.push({
					description: messages["list item (bullet)"],
					positions: onEmptyLine ? [{offset: offset + 2, length: listItemText.length}] : null,
					proposal: (onEmptyLine ? "" : NL) + "* " + listItemText + (selection ? selection : "") //$NON-NLS-0$
				});

				result.push({ /* gfm items separator */
					style: "hr" //$NON-NLS-0$
				});
	
				result.push({
					description: i18nUtil.formatMessage(messages["strikethrough (${0})"], "gfm"), //$NON-NLS-0$
					escapePosition: selection ? null : offset + 2,
					proposal: "~~" + (selection ? selection : "") + "~~" //$NON-NLS-1$ //$NON-NLS-0$
				});

				var tableNL = onEmptyLine ? "" : NL;
				result.push({
					description: i18nUtil.formatMessage(messages["table (${0})"], "gfm"), //$NON-NLS-0$
					positions: [
						{offset: offset + tableNL.length, length: 5},
						{offset: offset + tableNL.length + 7, length: 9},
						{offset: offset + tableNL.length + 16 + NL.length, length: 6},
						{offset: offset + tableNL.length + 23 + NL.length, length: 9},
						{offset: offset + tableNL.length + 32 + 2 * NL.length, length: 4},
						{offset: offset + tableNL.length + 39 + 2 * NL.length, length: 4}],
					proposal: tableNL + "hLeft |hCentered" + NL + ":-----|:-------:" + NL + "item  |item     " + NL + (selection ? selection : "") //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
				});
	
				result.push({
					description: i18nUtil.formatMessage(messages["fenced code (${0})"], "gfm"), //$NON-NLS-0$
					escapePosition: selection ? null : offset + 3,
					proposal: "```" + (selection ? selection : "") + "```" //$NON-NLS-1$ //$NON-NLS-0$
				});
	
				return result;
			}
		}, {
			name: "Markdown content assist", //$NON-NLS-0$
			contentType: ["text/x-markdown"] //$NON-NLS-0$
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-objective-c/syntax", ["orion/editor/stylers/text_x-csrc/syntax"], function(mC) {
	var keywords = [
		"atomic",
		"BOOL", "bycopy", "byref",
		"Class",
		"id", "IMP", "inout", "in",
		"nil", "nonatomic", "NO", "NULL",
		"oneway", "out",
		"Protocol",
		"retain",
		"SEL", "self", "super",
		"YES"
	];

	var atKeywords = [
		"@catch", "@class",
		"@dynamic",
		"@end",
		"@finally",
		"@implementation", "@interface",
		"@private", "@property", "@protected", "@protocol", "@public",
		"@selector",
		"@synthesize",
		"@throw", "@try"
	];

	var directives = ["import"];

	var grammars = [];
	grammars.push.apply(grammars, mC.grammars);
	grammars.push({
		id: "orion.objectiveC",
		contentTypes: ["text/x-objective-c"],
		patterns: [
			{include: "#objectiveCString"},
			{include: "orion.c"},
			{include: "#objectiveCDirective"},
			{include: "#objectiveCKeyword"}
		],
		repository: {
			objectiveCDirective: {
				match: "#\\s*(?:" + directives.join("|") + ")\\b[^$]*",
				name: "meta.preprocessor.objective-c"
			},
			objectiveCKeyword: {
				match: "(\\b(?:" + keywords.join("|") + ")|(?:" + atKeywords.join("|") + "))\\b",
				name: "keyword.operator.objective-c"
			},
			objectiveCString: {
				match: '@"(?:\\\\.|[^"])*"?',
				name: "string.quoted.double.objective-c"
			},
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: mC.keywords.concat(keywords).concat(directives).concat(atKeywords)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/objectiveC/objectiveCPlugin',['orion/plugin', 'orion/editor/stylers/text_x-objective-c/syntax'], function(PluginProvider, mObjectiveC) {

	function connect() {
		var headers = {
			name: "Orion Objective-C Tool Support",
			version: "1.0",
			description: "This plugin provides Objective-C tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-objective-c",
					"extends": "text/plain",
					name: "Objective-C",
					extension: ["m", "mm", "h"]
				}
			] 
		});
		mObjectiveC.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-php/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"abstract", "and", "array", "as",
		"callable", "class", "clone", "const",
		"declare",
		"echo", "empty", "eval", "extends",
		"final", "function",
		"global",
		"implements", "include", "include_once", "insteadof", "interface", "instanceof", "isset",
		"list",
		"namespace", "new",
		"or",
		"parent", "print", "private", "protected", "public",
		"require", "require_once",
		"static",
		"trait",
		"unset", "use",
		"var",
		"xor",
		"__halt_compiler", "__CLASS__", "__DIR__", "__FILE__", "__FUNCTION__",
		"__LINE__", "__METHOD__", "__NAMESPACE__", "__TRAIT__"
	];
	var controlKeywords = [
		"break",
		"case", "catch", "continue",
		"default", "die", "do",
		"else", "elseif", "enddeclare", "endfor", "endforeach", "endif", "endswitch", "endwhile", "exit",
		"finally", "for", "foreach",
		"goto",
		"if",
		"return",
		"switch",
		"throw", "try",
		"while",
		"yield"
	];
	var constants = [
		"false", "FALSE", "null", "NULL", "true", "TRUE"
	];
	var languageVariables = ["self"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.php-core",
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{
				match: {match: "(#).*", literal: "#"},
				name: "comment.line.number-sign.php",
				captures: {
					1: {name: "comment.line.number-sign.start.php"}
				},
				patterns: [
					{include: "orion.lib#todo_comment_singleLine"}
				]
			}, {
				begin: "<<<(\\w+)$",
				end: "^\\1;$",
				name: "string.unquoted.heredoc.php"
			}, {
				begin: "<<<'(\\w+)'$",
				end: "^\\1;$",
				name: "string.unquoted.heredoc.nowdoc.php"
			},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{
				match: "\\b0[bB][01]+\\b",
				name: "constant.numeric.binary.php"
			},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.php"
			},
			{
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.php"
			},
			{
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.php"
			},
			{
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.php"
			}
		]
	});
	
	grammars.push({
		id: "orion.php",
		contentTypes: ["text/x-php"],
		patterns: [
			{
				begin: "(?i)<(\\?|%(?!php))(?:=|php(?:\\s|$))?",
				end: "[\\1]>",
				captures: {
					0: {name: "entity.name.declaration.php"}
				},
				contentName: "source.php.embedded",
				patterns: [
					{include: "orion.php-core"}
				]
			}, {
				include: "orion.html"
			}
		]
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(constants).concat(languageVariables)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/php/phpPlugin',['orion/plugin', 'orion/editor/stylers/text_x-php/syntax'], function(PluginProvider, mPHP) {

	function connect() {
		var headers = {
			name: "Orion PHP Tool Support",
			version: "1.0",
			description: "This plugin provides PHP tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-php",
					"extends": "text/plain",
					name: "PHP",
					extension: ["php", "php3", "php4", "php5", "phpt", "phtml", "aw", "ctp"]
				}
			] 
		});
		mPHP.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-properties/syntax", [], function() {
	var grammars = [];
	grammars.push({
		id: "orion.properties",
		contentTypes: ["text/x-properties"],
		patterns: [
			{
				match: "^\\s*#[^$]*",
				name: "comment.line.properties"
			}, {
				begin: "(^[^=]*)(=)",
				end: ".*[^\\\\]$",
				beginCaptures: {
					1: {name: "entity.name.properties"},
					2: {name: "punctuation.operator.properties"}
				},
				endCaptures: {
					0: {name: "string.unquoted.properties"}
				},
				contentName: "string.unquoted.properties"
			}
		]
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/properties/propertiesPlugin',['orion/plugin', 'orion/editor/stylers/text_x-properties/syntax'], function(PluginProvider, mProperties) {

	function connect() {
		var headers = {
			name: "Orion Properties File Support",
			version: "1.0",
			description: "This plugin provides Properties file support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-properties",
					"extends": "text/plain",
					name: "Properties",
					extension: ["properties"]
				}
			] 
		});
		mProperties.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-python/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"and", "as", "assert",
		"break",
		"class", "continue",
		"def", "del",
		"exec", "elif", "else", "except", "Ellipsis",
		"False", "finally", "for", "from",
		"global",
		"if", "import", "in", "is",
		"lambda",
		"not", "None", "NotImplemented",
		"or",
		"pass", "print",
		"raise", "return",
		"try", "True",
		"while", "with",
		"yield"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.python",
		contentTypes: ["text/x-python"],
		patterns: [
			{
				begin: "(['\"])\\1\\1",
				end: "\\1\\1\\1",
				name: "string.quoted.triple.python"
			}, 
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{
				begin: {match: "#", literal: "#"},
				end: {match: "$", literal: ""},
				name: "comment.line.number-sign.python",
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "#number_decimal"},
			{include: "orion.lib#number_hex"}, 
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.python"
			}
		],
		repository: {
			number_decimal: {
				match: "\\b-?(?:\\.\\d+|\\d+\\.?\\d*)[lL]?\\b",
				name: "constant.numeric.number.python"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/python/pythonPlugin',['orion/plugin', 'orion/editor/stylers/text_x-python/syntax'], function(PluginProvider, mPython) {

	function connect() {
		var headers = {
			name: "Orion Python Tool Support",
			version: "1.0",
			description: "This plugin provides Python tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-python",
					"extends": "text/plain",
					name: "Python",
					extension: ["py", "rpy", "pyw", "cpy", "SConstruct", "Sconstruct", "sconstruct", "SConscript", "gyp", "gypi"]
				}
			] 
		});
		mPython.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/ruby/rubyPlugin',['orion/plugin', 'orion/editor/stylers/text_x-ruby/syntax'], function(PluginProvider, mRuby) {

	function connect() {
		var headers = {
			name: "Orion Ruby Tool Support",
			version: "1.0",
			description: "This plugin provides Ruby tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-ruby",
					"extends": "text/plain",
					name: "Ruby",
					extension: ["rb", "rbx", "rjs", "Rakefile", "rake", "cgi", "fcgi", "gemspec", "irbrc", "capfile", "ru", "prawn", "Gemfile", "Guardfile", "Vagrantfile", "Appraisals", "Rantfile"]
				}
			] 
		});
		mRuby.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-scss/syntax", ["orion/editor/stylers/text_css/syntax", "orion/editor/stylers/lib/syntax"], function(mCSS, mLib) {
	var flags = [
		"default", "global", "optional"
	];
	var constants = [
		"false", "null", "true"
	];
	var directives = [
		"at-root",
		"content",
		"debug",
		"each", "else", "error", "extend",
		"for", "function",
		"if", "include",
		"mixin",
		"return",
		"warn", "while"
	];

	var grammars = [];
	grammars.push.apply(grammars, mCSS.grammars);
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.scss",
		contentTypes: ["text/x-scss"],
		patterns: [
			{include: "orion.css#string_single_multiline"},
			{include: "orion.css#string_double_multiline"},
			{include: "orion.c-like#comment_block"},
			{include: "#eachin"},
			{include: "#forto"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "#variable"},
			{include: "#placeholder"},
			{include: "#flag"},
			{include: "#directive"},
			{include: "orion.css#directive"},
			{include: "#constant"},
			{include: "#interpolated"},
			{include: "#operator"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.css#number_hex"},
			{include: "orion.css#numeric_value"},
			{include: "orion.css#color"},
			{include: "orion.css#keyword"}
		],
		repository: {
			constant: {
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.scss"
			},
			directive: {
				match: "(^|\\s)(@("  + directives.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.directive.scss"}
				}
			},
			eachin: {
				begin: "@each\\s",
				end: "\\sin\\b",
				captures: {
					0: {name: "keyword.other.directive.scss"},
				},
				patterns: [
					{include: "#variable"}
				]
			},
			flag: {
				match: "(^|\\s)(!("  + flags.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.flag.scss"}
				}
			},
			forto: {
				begin: "@for\\s",
				end: "(^|\\s)(t(o|hrough))(\\s|$)",
				beginCaptures: {
					0: {name: "keyword.other.directive.scss"},
				},
				endCaptures: {
					2: {name: "keyword.other.directive.scss"}
				},
				patterns: [
					{include: "#variable"},
					{include: "orion.lib#number_decimal"},
					{
						match: "(^|\\s)(from)(\\s|$)",
						name: "keyword.other.directive.scss"
					}
				]
			},
			interpolated: {
				match: "#\\{[^}]*\\}",
				name: "string.interpolated.scss"
			},
			operator: {
				match: "\\+|\\*|\\/|%|==?|!=|&|<=?|=?>|!",
				name: "punctuation.operator.scss"
			},
			placeholder: {
				match: "%[\\w-]+",
				name: "variable.other.placeholder.sas"
			},
			variable: {
				match: "\\$[\\w-]+(\\.{3})?",
				name: "variable.other.scss"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: directives.concat(flags)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/scss/scssPlugin',['orion/plugin', 'orion/editor/stylers/text_x-scss/syntax'], function(PluginProvider, mScss) {

	function connect() {
		var headers = {
			name: "Orion SCSS Tool Support",
			version: "1.0",
			description: "This plugin provides SCSS tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-scss",
					"extends": "text/plain",
					name: "SCSS",
					extension: ["scss", "sass"]
				}
			] 
		});
		mScss.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/text_x-smarty/syntax", ["orion/editor/stylers/lib/syntax", "orion/editor/stylers/text_html/syntax", "orion/editor/stylers/text_x-php/syntax"],
	function(mLib, mHTML, mPHP) {

	var constants = [
		"false", "no", "off", "on", "true", "yes"
	];

	/* these can be redefined in the file, this is not currently handled */
	var DELIMITER_OPEN = "{";
	var DELIMITER_CLOSE = "}";
	
	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push.apply(grammars, mHTML.grammars);
	grammars.push.apply(grammars, mPHP.grammars);
	grammars.push({
		id: "orion.smarty",
		contentTypes: ["text/x-smarty"],
		patterns: [
			{include: "orion.html"},
			{include: "#smartyCommentBlock"},
			{include: "#literalBlock"},
			{include: "#phpBlock"},
			{include: "#smartyBlock"}
		],
		repository: {
			literalBlock: {
				begin: "({)literal(})",
				end: "({)/literal(})",
				captures: {
					1: "punctuation.brace.begin.smarty",
					2: "punctuation.brace.end.smarty"
				}
			},
			phpBlock: {
				begin: "({)php(})",
				end: "({)/php(})",
				captures: {
					1: "punctuation.brace.begin.smarty",
					2: "punctuation.brace.end.smarty"
				},
				contentName: "source.php.embedded.smarty",
				patterns: [
					{include: "orion.php-core"}
				]
			},
			smartyBlock: {
				begin: "(" + DELIMITER_OPEN + ")",
				end: "(" + DELIMITER_CLOSE + ")",
				captures: {
					1: "punctuation.brace.begin.smarty",
					2: "punctuation.brace.end.smarty"
				},
				patterns: [
					{include: "orion.lib#string_singleQuote"},
					{include: "#smartyString_doubleQuote"},
					{include: "#smartyVariable"},
					{include: "#smartyConfigVariable"},
					{include: "#smartyConstant"},
					{include: "orion.lib#number_decimal"},
				]
			},
			smartyCommentBlock: {
				begin: {match: DELIMITER_OPEN + "\\*", literal: DELIMITER_OPEN + "*"},
				end: {match: "\\*" + DELIMITER_CLOSE, literal: "*" + DELIMITER_CLOSE},
				name: "comment.block.smarty",
			},
			smartyConfigVariable: {
				match: "#\\w+#",
				name: "variable.other.config.smarty",
			},
			smartyConstant: {
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.smarty"
			},
			smartyEscapedVariable: {
				match: "`\\$[^`]+`",
				name: "variable.other.escaped.smarty",
			},
			smartyString_doubleQuote: {
				begin: '"',
				end: '"',
				name: "string.quoted.double.smarty",
				patterns: [
					{include: "#smartyEscapedVariable"},
					{include: "#smartyVariable"},
					{include: "#smartyConfigVariable"}
				]
			},
			smartyVariable: {
				match: "\\$(?:smarty\\.(?:config|server)\\.)?\\w+",
				name: "variable.other.smarty",
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/smarty/smartyPlugin',['orion/plugin', 'orion/editor/stylers/text_x-smarty/syntax'], function(PluginProvider, mSmarty) {

	function connect() {
		var headers = {
			name: "Orion Smarty Tool Support",
			version: "1.0",
			description: "This plugin provides Smarty tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-smarty",
					"extends": "text/plain",
					name: "Smarty",
					extension: ["tpl"]
				}
			] 
		});
		mSmarty.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/application_sql/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"ADD", "ALTER\\s+TABLE", "AND", "ASC",
		"BETWEEN",
		"COMMIT", "CONTAINS", "CREATE",
		"DELETE\\s+FROM", "DESC", "DROP(\\s+(INDEX|TABLE|DATABASE))?",
		"FROM",
		"GROUP\\s+BY",
		"HAVING",
		"INSERT\\s+INTO", "IN",
		"LIKE",
		"MODIFY",
		"ON", "ORDER\\s+BY", "OR",
		"PRIMARY\\s+KEY",
		"ROLLBACK",
		"SELECT(\\s+(COUNT|DISTINCT|SUM))?", "SET",
		"TO", "TRUNCATE\\s+TABLE",
		"UPDATE", "USE",
		"VALUES",
		"WHERE"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.sql",
		contentTypes: ["application/sql"],
		patterns: [
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#number_decimal"},
			{
				match: "(?i)\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.sql"
			}, {
				match: "<>|>=?|<=?|=",
				name: "punctuation.operator.sql"
			}
		],
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});


/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/sql/sqlPlugin',['orion/plugin', 'orion/editor/stylers/application_sql/syntax'], function(PluginProvider, mSql) {

	function connect() {
		var headers = {
			name: "Orion SQL Tool Support",
			version: "1.0",
			description: "This plugin provides SQL tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "application/sql",
					"extends": "text/plain",
					name: "SQL",
					extension: ["sql"]
				}
			] 
		});
		mSql.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-swift/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"associativity", "as",
		"class", "convenience",
		"deinit", "didSet", "dynamicType", "dynamic",
		"enum", "extension",
		"final", "func",
		"get",
		"import", "infix", "init", "inout", "internal", "in", "is",
		"lazy", "left", "let",
		"mutating",
		"none", "nonmutating",
		"operator", "optional", "override",
		"postfix", "precedence", "prefix", "private", "protocol", "Protocol", "public",
		"required", "right",
		"Self", "set", "static", "struct", "subscript",
		"typealias", "Type",
		"unowned",
		"var",
		"weak", "willSet",
		"@objc"
	];
	var controlKeywords = [
		"break",
		"case", "continue",
		"default", "do",
		"else",
		"fallthrough", "for",
		"if",
		"return",
		"switch",
		"where", "while"
	];
	var constants = [
		"false", "nil", "true"
	];
	var languageVars1 = [
		"self", "super"
	];
	var languageVars2 = [
		"__COLUMN__", "__FILE__", "__FUNCTION__", "__LINE__"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.swift",
		contentTypes: ["text/x-swift"],
		patterns: [
			{include: "#string_doubleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{include: "#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "#number_binary"},
			{include: "#number_hex"},
			{include: "#number_octal"},
			{include: "#number_decimal"},
			{include: "#keywords_operator"},
			{include: "#keywords_control"},
			{include: "#constants"},
			{include: "#languageVars1"},
			{include: "#languageVars2"},
		],
		repository: {
			comment_block: {
				begin: {match: "/\\*", literal: "/*"},
				end: {match: "\\*/", literal: "*/"}, 
				name: "comment.block.swift",
				patterns: [
					{include: "#comment_block"},
					{
						match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task.swift"},
							4: {name: "comment.block.swift"}
						}
					}
				]
			},
			constants: {
				match: "(^|[^\\w`])(" + constants.join("|") + ")\\b",
				captures: {
					2: {name: "constant.language.swift"}
				}
			},
			keywords_operator: {
				match: "(^|[^\\w`])(" + keywords.join("|") + ")\\b",
				captures: {
					2: {name: "keyword.operator.swift"}
				}
			},
			keywords_control: {
				match: "(^|[^\\w`])(" + controlKeywords.join("|") + ")\\b",
				captures: {
					2: {name: "keyword.control.swift"}
				}
			},
			languageVars1: {
				match: "(^|[^\\w`])(" + languageVars1.join("|") + ")\\b",
				captures: {
					2: {name: "variable.language.swift"}
				}
			},
			languageVars2: {
				match: "(^|[^\\w`])(" + languageVars2.join("|") + ")(?:$|[^\\w])",
				captures: {
					2: {name: "variable.language.swift"}
				}
			},
			number_binary: {
				match: "\\b0b[01]+\\b",
				name: "constant.numeric.binary.swift"
			},
			number_decimal: {
				match: "\\b-?(?:\\.\\d[\\d_]*|\\d[\\d_]*\\.?[\\d_]*)(?:[eE][+-]?\\d[\\d_]*)?\\b",
				name: "constant.numeric.decimal.swift"
			},
			number_hex: {
				match: "\\b0[xX](?:\\.[0-9A-Fa-f][0-9A-Fa-f_]*|[0-9A-Fa-f][0-9A-Fa-f_]*\\.?[0-9A-Fa-f_]*)(?:[pP][+-]?\\d[\\d_]*)?\\b",
				name: "constant.numeric.hex.swift"
			},
			number_octal: {
				match: "\\b0o[01234567][01234567_]*\\b",
				name: "constant.numeric.octal.swift"
			},
			segment: {
				begin: "\\(",
				end: "\\)",
				patterns: [
					{include: "#segment"},
					{include: "#comment_block"},
					{include: "#number_binary"},
					{include: "#number_hex"},
					{include: "#number_octal"},
					{include: "#number_decimal"},
					{include: "#keywords_operator"},
					{include: "#keywords_control"},
					{include: "#constants"},
					{include: "#languageVars1"},
					{include: "#languageVars2"}
				]
			},
			string_doubleQuote: {
				match: '"(?:\\\\.|[^"])*"?',
				name: "string.quoted.double.swift",
				patterns: [
					{
						begin: "\\\\\\(",
						end: "\\)",
						name: "string.interpolated.swift",
						patterns: [
							{include: "#segment"},
							{include: "#comment_block"},
							{include: "#number_binary"},
							{include: "#number_hex"},
							{include: "#number_octal"},
							{include: "#number_decimal"},
							{include: "#keywords_operator"},
							{include: "#keywords_control"},
							{include: "#constants"},
							{include: "#languageVars1"},
							{include: "#languageVars2"}
						]
					}
				]
			}
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(constants).concat(languageVars1).concat(languageVars2)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/swift/swiftPlugin',['orion/plugin', 'orion/editor/stylers/text_x-swift/syntax'], function(PluginProvider, mSwift) {

	function connect() {
		var headers = {
			name: "Orion Swift Tool Support",
			version: "1.0",
			description: "This plugin provides Swift tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-swift",
					"extends": "text/plain",
					name: "Swift",
					extension: ["swift"]
				}
			] 
		});
		mSwift.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-typescript/syntax", ["orion/editor/stylers/application_javascript/syntax"], function(mJS) {
	var keywords = ["constructor", "declare", "module"];
	var types = ["any", "boolean", "number", "string"];

	var grammars = [];
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push({
		id: "orion.typescript",
		contentTypes: ["text/x-typescript"],
		patterns: [
			{include: "orion.js"},
			{include: "#keywords"},
			{include: "#types"},
		],
		repository: {
			keywords: {
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.typescript"
			},
			types: {
				match: "\\b(?:" + types.join("|") + ")\\b",
				name: "storage.type.typescript"
			}
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(types)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/typescript/typescriptPlugin',['orion/plugin', 'orion/editor/stylers/text_x-typescript/syntax'], function(PluginProvider, mTypescript) {

	function connect() {
		var headers = {
			name: "Orion Typescript Tool Support",
			version: "1.0",
			description: "This plugin provides Typescript tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-typescript",
					"extends": "text/plain",
					name: "Typescript",
					extension: ["ts"]
				}
			] 
		});
		mTypescript.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-vb/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
	    "AddHandler", "AddressOf", "Aggregate", "Alias", "AndAlso",
	    "And", "Ansi", "Assembly", "Async", "As", "Auto", "Await",
	    "Binary", "Boolean", "ByRef", "Byte", "ByVal",
	    "Call", "Case", "Catch", "CBool", "CByte", "CChar", "CDate",
	    "CDbl", "CDec", "Char", "CInt", "Class", "CLng", "CObj", "Compare",
	    "Const", "CShort", "CSng", "CStr", "CType", "Custom",
	    "Date", "Decimal", "Declare", "Default", "Delegate", "Dim",
	    "DirectCast", "Distinct", "Double", "Do",
	    "Each", "ElseIf", "Else", "EndIf", "End", "Enum", "Equals",
	    "Erase", "Error", "Event", "Exit", "Explicit",
	    "False", "Finally", "For", "Friend", "From", "Function",
	    "GetType", "Get", "GoSub", "GoTo", "Group By", "Group Join",
	    "Handles",
	    "If", "Implements", "Imports", "Inherits", "Integer", "Interface",
	    "Into", "In", "IsFalse", "IsTrue", "Is", "Iterator",
	    "Join",
	    "Key",
	    "Let", "Lib", "Like", "Long", "Loop",
	    "Me", "Mid", "Module", "Mod", "MustInherit", "MustOverride",
	    "MyBase", "MyClass",
	    "Namespace", "New", "Next", "Nothing", "NotInheritable",
	    "NotOverridable", "Not",
	    "Object", "Off", "On", "Optional", "Option", "Order By", "OrElse",
	    "Or", "Overloads", "Overridable", "Overrides",
	    "ParamArray", "Preserve", "Private", "Property", "Protected", "Public",
	    "RaiseEvent", "ReadOnly", "ReDim", "REM", "RemoveHandler", "Resume", "Return",
	    "Select", "Set", "Shadows", "Shared", "Short", "Single", "Skip While", "Skip",
	    "Static", "Step", "Stop", "Strict", "String", "Structure", "Sub", "SyncLock",
	    "Take While", "Take", "Text", "Then", "Throw", "To", "True", "Try", "TypeOf",
	    "Unicode", "Until",
	    "Variant",
	    "Wend", "When", "Where", "While", "WithEvents", "With", "WriteOnly",
	    "Xor",
	    "Yield"
	];

	var preprocessorDirectives = ["Const", "ElseIf", "Else", "End", "ExternalSource", "If", "Region"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.vb",
		contentTypes: ["text/x-vb"],
		patterns: [
			{
				match: "^\\s*#(?:" + preprocessorDirectives.join("|") + ")\\b[^$]*",
				name: "meta.preprocessor.vb"
			},
			{include: "orion.lib#string_doubleQuote"},
			{include: "#doc"},
			{include: "#comment"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.vb"
			}
		],
		repository: {
			comment: {
				begin: {match: "'", literal: "'"},
				end: {match: "$", literal: ""},
				name: "comment.line.vb",
				patterns: [
					{
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			doc: {
				match: {match: "'''.*", literal: "'''"},
				name: "comment.line.documentation.vb",
				patterns: [
					{
						match: "<[^\\s>]*>",
						name: "meta.documentation.tag"
					}, {
						include: "orion.lib#todo_comment_singleLine"
					}
				]
			},
			number_hex: {
				match: "&[hH][0-9A-Fa-f]+\\b",
				name: "constant.numeric.hex.vb"
			},
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-vbhtml/syntax", [
	"orion/editor/stylers/application_xml/syntax",
	"orion/editor/stylers/text_html/syntax",
	"orion/editor/stylers/text_x-vb/syntax"], function(mXML, mHTML, mVB) {

	var grammars = [];
	grammars.push.apply(grammars, mXML.grammars);
	grammars.push.apply(grammars, mHTML.grammars);
	grammars.push.apply(grammars, mVB.grammars);
	grammars.push({
		id: "orion.vbhtml",
		contentTypes: ["text/x-vbhtml"],
		patterns: [
			{include: "#vbhtmlComment"},
			{include: "#codeBlock"},
			{include: "#expression"},
			{include: "#reference"},
			{include: "orion.html"},
		],
		repository: {
			vbhtmlComment: {
				begin: {match: "@\\*", literal: "@*"},
				end: {match: "\\*@", literal: "*@"},
				name: "comment.block.vbhtml",
			},
			codeBlock: {
				begin: "(?i)^\\s*@code",
				end: "(?i)end code",
				captures: {
					0: {name: "entity.name.declaration.vb"}
				},
				contentName: "source.vb.embedded.vbhtml",
				patterns: [
				    {include: "orion.xml#tag"},
				    {include: "#reference"},
					{include: "orion.vb"},
				]
			},
			expression: {
				match: "(?i)^\\s*@(?!code)[^$]*",
				name: "source.vb.embedded.vbhtml",
				patterns: [
				    {include: "#reference"},
					{include: "orion.vb"},
				]
			},
			reference: {
				match: "@",
				name: "entity.name.declaration.vb"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/vb/vbPlugin',['orion/plugin', 'orion/editor/stylers/text_x-vb/syntax', 'orion/editor/stylers/text_x-vbhtml/syntax'], function(PluginProvider, mVB, mVBHtml) {

	function connect() {
		var headers = {
			name: "Orion VB.NET Tool Support",
			version: "1.0",
			description: "This plugin provides VB.NET tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-vb",
					"extends": "text/plain",
					name: "VB.NET",
					extension: ["vb"]
				}, {id: "text/x-vbhtml",
					"extends": "text/plain",
					name: "vbhtml",
					extension: ["vbhtml"]
				}
			]
		});
		mVB.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
		mVBHtml.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/xml/xmlPlugin',['orion/plugin', 'orion/editor/stylers/application_xml/syntax'], function(PluginProvider, mXML) {

	function connect() {
		var headers = {
			name: "Orion XML Tool Support",
			version: "1.0",
			description: "This plugin provides XML tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "application/xml",
					"extends": "text/plain",
					name: "XML",
					extension: ["xml", "xib"],
					imageClass: "file-sprite-xml"
				}, {id: "application/xhtml+xml",
					"extends": "text/plain",
					name: "XHTML",
					extension: ["xhtml", "xht"],
					imageClass: "file-sprite-xml"
				}
			] 
		});
		mXML.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
 
/*eslint-env browser, amd*/
define("orion/editor/stylers/application_xquery/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"zero-digit",
		"xquery",
		"window", "where", "when",
		"version", "variable", "validate",
		"unordered", "union",
		"typeswitch", "type", "tumbling", "try",
		"treat", "to", "then", "text",
		"switch","strip", "strict", "start", "stable", "some",
		"sliding", "self", "schema-element", "schema-attribute", "schema", "satisfies",
		"return",
		"processing-instruction", "previous", "preserve", "preceding-sibling",
		"preceding", "percent", "per-mille", "pattern-separator", "parent",
		"ordering", "order", "or", "option", "only", "of",
		"node", "no-preserve", "no-inherit", "next",
		"ne", "NaN", "namespace-node", "namespace",
		"module", "mod", "minus-sign",
		"lt", "let", "least", "le", "lax",
		"item", "is", "intersect", "instance", "inherit",
		"infinity", "in", "import", "if", "idiv",
		"gt", "grouping-separator", "group", "greatest", "ge",
		"function", "for", "following-sibling", "following",
		"external", "except", "every", "eq", "end",
		"encoding", "empty-sequence", "empty", "else", "element",
		"document-node", "div", "digit", "descending", "descendant-or-self",
		"descendant", "default", "declare", "decimal-separator", "decimal-format",
		"count", "copy-namespaces", "context", "construction", "comment",
		"collation", "child", "catch", "castable", "cast", "case",
		"by", "boundary-space", "base-uri",
		"attribute", "at", "ascending", "as",
		"and", "ancestor-or-self", "ancestor", "allowing"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.xquery",
		contentTypes: ["application/xquery"],
		patterns: [
			{include: "#comment"},
			{include: "#variable"},
			{include: "#multiLineDoubleQuote"}, 
			{include: "#multiLineSingleQuote"}, 
			{include: "orion.xml#tag"}, 
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.js"
			}
		],
		repository: {
			comment: {
				begin: {match: "\\(:", literal: "(:"},
				end: {match: ":\\)", literal: ":)"}, 
				name: "comment.block.xquery",
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!:\\)).)*)", // match: "(\\b)(TODO)(\\b)(((?!-->).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.block"}
						}
					}
				]
			},
			variable: {
				match: "\\$[a-zA-z0-9_]+",
				name: "variable.other.xquery" 
			},
			multiLineDoubleQuote: {
				begin: "\"",
				end: "\"",
				name: "string.quoted.double"
			},
			multiLineSingleQuote: {
				begin: "\'",
				end: "\'",
				name: "string.quoted.single"
			}
		}

	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/xquery/xqueryPlugin',['orion/plugin', 'orion/editor/stylers/application_xquery/syntax'], function(PluginProvider, mXQuery) {
	
	function connect() {
		var headers = {
			name: "Orion XQuery Tool Support",
			version: "1.0",
			description: "This plugin provides XQuery tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "application/xquery",
					"extends": "text/plain",
					name: "XQuery",
					extension: ["xq", "xqy", "xquery"]
				}
			] 
		});
		mXQuery.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-yaml/syntax", [], function() {
	var keywords = ["false", "null", "true"];
	var casts = ["!!bool", "!!float", "!!int", "!!map", "!!null", "!!omap", "!!seq", "!!str"];

	var id = "orion.yaml";
	var grammar = {
		id: id,
		contentTypes: ["text/x-yaml"],
		patterns: [
			{include: "#numberSignComment"},
			{
				match: "^%(?:YAML|TAG)\\s.*",
				name: "meta.directive.yaml"
			}, {
				begin: "^.*?:(?:[\\t ]|$)",
				end: "$",
				contentName: "string.unquoted.yaml",
				beginCaptures: {
					0: {
						name: "entity.name.key.yaml"
					}
				},
				patterns: [
					{include: "#numberSignComment"},
					{
						match: "^\\s*[&*]\\s*$",
						name: "entity.name.tag.yaml"
					}, {
						match: "(?i)^\\s*(?:" + keywords.join("|") + ")\\s*$",
						name: "keyword.operator.yaml"
					}, {
						match: "(?i)^\\s*(?:" + casts.join("|") + ")\\b",
						name: "keyword.operator.yaml"
					}, {
						match: "(?i)^\\s*(?:-?[0-9]*(?:\\.[0-9]+)?(?:e[-+][1-9][0-9]*)?)\\s*$",
						name: "constant.numeric.yaml"
					}, {
						match: "(?i)^\\s*(?:-?[1-9][0-9]*|0|-?\\.inf|\\.nan)\\s*$",
						name: "constant.numeric.yaml"
					}
				]
			}, {
				match: "---|\\.\\.\\.",
				name: "meta.separator.yaml"
			}
		],
		repository: {
			numberSignComment: {
				begin: {match: "(?:^|\\s+)#", literal: "#"},
				end: {match: "$", literal: ""},
				name: "comment.line.number-sign.yaml",
				patterns: [
					{include: "orion.lib#todo_comment_singleLine"}
				]
			}
		}
	};

	return {
		id: id,
		grammars: [grammar],
		keywords: casts.concat(keywords)
	};
});

/* js-yaml 3.6.1 https://github.com/nodeca/js-yaml */
!function(e){if("object"==typeof exports&&"undefined"!=typeof module)module.exports=e();else if("function"==typeof define&&define.amd)define('js-yaml/js-yaml',[],e);else{var t;t="undefined"!=typeof window?window:"undefined"!=typeof global?global:"undefined"!=typeof self?self:this,t.jsyaml=e()}}(function(){return function e(t,n,i){function r(a,s){if(!n[a]){if(!t[a]){var c="function"==typeof require&&require;if(!s&&c)return c(a,!0);if(o)return o(a,!0);var u=new Error("Cannot find module '"+a+"'");throw u.code="MODULE_NOT_FOUND",u}var l=n[a]={exports:{}};t[a][0].call(l.exports,function(e){var n=t[a][1][e];return r(n?n:e)},l,l.exports,e,t,n,i)}return n[a].exports}for(var o="function"==typeof require&&require,a=0;a<i.length;a++)r(i[a]);return r}({1:[function(e,t,n){"use strict";function i(e){return function(){throw new Error("Function "+e+" is deprecated and cannot be used.")}}var r=e("./js-yaml/loader"),o=e("./js-yaml/dumper");t.exports.Type=e("./js-yaml/type"),t.exports.Schema=e("./js-yaml/schema"),t.exports.FAILSAFE_SCHEMA=e("./js-yaml/schema/failsafe"),t.exports.JSON_SCHEMA=e("./js-yaml/schema/json"),t.exports.CORE_SCHEMA=e("./js-yaml/schema/core"),t.exports.DEFAULT_SAFE_SCHEMA=e("./js-yaml/schema/default_safe"),t.exports.DEFAULT_FULL_SCHEMA=e("./js-yaml/schema/default_full"),t.exports.load=r.load,t.exports.loadAll=r.loadAll,t.exports.safeLoad=r.safeLoad,t.exports.safeLoadAll=r.safeLoadAll,t.exports.dump=o.dump,t.exports.safeDump=o.safeDump,t.exports.YAMLException=e("./js-yaml/exception"),t.exports.MINIMAL_SCHEMA=e("./js-yaml/schema/failsafe"),t.exports.SAFE_SCHEMA=e("./js-yaml/schema/default_safe"),t.exports.DEFAULT_SCHEMA=e("./js-yaml/schema/default_full"),t.exports.scan=i("scan"),t.exports.parse=i("parse"),t.exports.compose=i("compose"),t.exports.addConstructor=i("addConstructor")},{"./js-yaml/dumper":3,"./js-yaml/exception":4,"./js-yaml/loader":5,"./js-yaml/schema":7,"./js-yaml/schema/core":8,"./js-yaml/schema/default_full":9,"./js-yaml/schema/default_safe":10,"./js-yaml/schema/failsafe":11,"./js-yaml/schema/json":12,"./js-yaml/type":13}],2:[function(e,t,n){"use strict";function i(e){return"undefined"==typeof e||null===e}function r(e){return"object"==typeof e&&null!==e}function o(e){return Array.isArray(e)?e:i(e)?[]:[e]}function a(e,t){var n,i,r,o;if(t)for(o=Object.keys(t),n=0,i=o.length;i>n;n+=1)r=o[n],e[r]=t[r];return e}function s(e,t){var n,i="";for(n=0;t>n;n+=1)i+=e;return i}function c(e){return 0===e&&Number.NEGATIVE_INFINITY===1/e}t.exports.isNothing=i,t.exports.isObject=r,t.exports.toArray=o,t.exports.repeat=s,t.exports.isNegativeZero=c,t.exports.extend=a},{}],3:[function(e,t,n){"use strict";function i(e,t){var n,i,r,o,a,s,c;if(null===t)return{};for(n={},i=Object.keys(t),r=0,o=i.length;o>r;r+=1)a=i[r],s=String(t[a]),"!!"===a.slice(0,2)&&(a="tag:yaml.org,2002:"+a.slice(2)),c=e.compiledTypeMap[a],c&&L.call(c.styleAliases,s)&&(s=c.styleAliases[s]),n[a]=s;return n}function r(e){var t,n,i;if(t=e.toString(16).toUpperCase(),255>=e)n="x",i=2;else if(65535>=e)n="u",i=4;else{if(!(4294967295>=e))throw new N("code point within a string may not be greater than 0xFFFFFFFF");n="U",i=8}return"\\"+n+F.repeat("0",i-t.length)+t}function o(e){this.schema=e.schema||M,this.indent=Math.max(1,e.indent||2),this.skipInvalid=e.skipInvalid||!1,this.flowLevel=F.isNothing(e.flowLevel)?-1:e.flowLevel,this.styleMap=i(this.schema,e.styles||null),this.sortKeys=e.sortKeys||!1,this.lineWidth=e.lineWidth||80,this.noRefs=e.noRefs||!1,this.noCompatMode=e.noCompatMode||!1,this.implicitTypes=this.schema.compiledImplicit,this.explicitTypes=this.schema.compiledExplicit,this.tag=null,this.result="",this.duplicates=[],this.usedDuplicates=null}function a(e,t){for(var n,i=F.repeat(" ",t),r=0,o=-1,a="",s=e.length;s>r;)o=e.indexOf("\n",r),-1===o?(n=e.slice(r),r=s):(n=e.slice(r,o+1),r=o+1),n.length&&"\n"!==n&&(a+=i),a+=n;return a}function s(e,t){return"\n"+F.repeat(" ",e.indent*t)}function c(e,t){var n,i,r;for(n=0,i=e.implicitTypes.length;i>n;n+=1)if(r=e.implicitTypes[n],r.resolve(t))return!0;return!1}function u(e){return e===q||e===D}function l(e){return e>=32&&126>=e||e>=161&&55295>=e&&8232!==e&&8233!==e||e>=57344&&65533>=e&&65279!==e||e>=65536&&1114111>=e}function p(e){return l(e)&&65279!==e&&e!==H&&e!==Q&&e!==X&&e!==te&&e!==ie&&e!==V&&e!==B}function f(e){return l(e)&&65279!==e&&!u(e)&&e!==G&&e!==z&&e!==V&&e!==H&&e!==Q&&e!==X&&e!==te&&e!==ie&&e!==B&&e!==W&&e!==$&&e!==Y&&e!==ne&&e!==Z&&e!==K&&e!==R&&e!==P&&e!==J&&e!==ee}function d(e,t,n,i,r){var o,a,s=!1,c=!1,d=-1!==i,h=-1,m=f(e.charCodeAt(0))&&!u(e.charCodeAt(e.length-1));if(t)for(o=0;o<e.length;o++){if(a=e.charCodeAt(o),!l(a))return le;m=m&&p(a)}else{for(o=0;o<e.length;o++){if(a=e.charCodeAt(o),a===U)s=!0,d&&(c=c||o-h-1>i&&" "!==e[h+1],h=o);else if(!l(a))return le;m=m&&p(a)}c=c||d&&o-h-1>i&&" "!==e[h+1]}return s||c?" "===e[0]&&n>9?le:c?ue:ce:m&&!r(e)?ae:se}function h(e,t,n,i){e.dump=function(){function r(t){return c(e,t)}if(0===t.length)return"''";if(!e.noCompatMode&&-1!==oe.indexOf(t))return"'"+t+"'";var o=e.indent*Math.max(1,n),s=-1===e.lineWidth?-1:Math.max(Math.min(e.lineWidth,40),e.lineWidth-o),u=i||e.flowLevel>-1&&n>=e.flowLevel;switch(d(t,u,e.indent,s,r)){case ae:return t;case se:return"'"+t.replace(/'/g,"''")+"'";case ce:return"|"+m(t,e.indent)+g(a(t,o));case ue:return">"+m(t,e.indent)+g(a(y(t,s),o));case le:return'"'+v(t,s)+'"';default:throw new N("impossible error: invalid scalar style")}}()}function m(e,t){var n=" "===e[0]?String(t):"",i="\n"===e[e.length-1],r=i&&("\n"===e[e.length-2]||"\n"===e),o=r?"+":i?"":"-";return n+o+"\n"}function g(e){return"\n"===e[e.length-1]?e.slice(0,-1):e}function y(e,t){for(var n,i,r=/(\n+)([^\n]*)/g,o=function(){var n=e.indexOf("\n");return n=-1!==n?n:e.length,r.lastIndex=n,x(e.slice(0,n),t)}(),a="\n"===e[0]||" "===e[0];i=r.exec(e);){var s=i[1],c=i[2];n=" "===c[0],o+=s+(a||n||""===c?"":"\n")+x(c,t),a=n}return o}function x(e,t){if(""===e||" "===e[0])return e;for(var n,i,r=/ [^ ]/g,o=0,a=0,s=0,c="";n=r.exec(e);)s=n.index,s-o>t&&(i=a>o?a:s,c+="\n"+e.slice(o,i),o=i+1),a=s;return c+="\n",c+=e.length-o>t&&a>o?e.slice(o,a)+"\n"+e.slice(a+1):e.slice(o),c.slice(1)}function v(e){for(var t,n,i="",o=0;o<e.length;o++)t=e.charCodeAt(o),n=re[t],i+=!n&&l(t)?e[o]:n||r(t);return i}function A(e,t,n){var i,r,o="",a=e.tag;for(i=0,r=n.length;r>i;i+=1)j(e,t,n[i],!1,!1)&&(0!==i&&(o+=", "),o+=e.dump);e.tag=a,e.dump="["+o+"]"}function b(e,t,n,i){var r,o,a="",c=e.tag;for(r=0,o=n.length;o>r;r+=1)j(e,t+1,n[r],!0,!0)&&(i&&0===r||(a+=s(e,t)),a+="- "+e.dump);e.tag=c,e.dump=a||"[]"}function w(e,t,n){var i,r,o,a,s,c="",u=e.tag,l=Object.keys(n);for(i=0,r=l.length;r>i;i+=1)s="",0!==i&&(s+=", "),o=l[i],a=n[o],j(e,t,o,!1,!1)&&(e.dump.length>1024&&(s+="? "),s+=e.dump+": ",j(e,t,a,!1,!1)&&(s+=e.dump,c+=s));e.tag=u,e.dump="{"+c+"}"}function C(e,t,n,i){var r,o,a,c,u,l,p="",f=e.tag,d=Object.keys(n);if(e.sortKeys===!0)d.sort();else if("function"==typeof e.sortKeys)d.sort(e.sortKeys);else if(e.sortKeys)throw new N("sortKeys must be a boolean or a function");for(r=0,o=d.length;o>r;r+=1)l="",i&&0===r||(l+=s(e,t)),a=d[r],c=n[a],j(e,t+1,a,!0,!0,!0)&&(u=null!==e.tag&&"?"!==e.tag||e.dump&&e.dump.length>1024,u&&(l+=e.dump&&U===e.dump.charCodeAt(0)?"?":"? "),l+=e.dump,u&&(l+=s(e,t)),j(e,t+1,c,!0,u)&&(l+=e.dump&&U===e.dump.charCodeAt(0)?":":": ",l+=e.dump,p+=l));e.tag=f,e.dump=p||"{}"}function k(e,t,n){var i,r,o,a,s,c;for(r=n?e.explicitTypes:e.implicitTypes,o=0,a=r.length;a>o;o+=1)if(s=r[o],(s.instanceOf||s.predicate)&&(!s.instanceOf||"object"==typeof t&&t instanceof s.instanceOf)&&(!s.predicate||s.predicate(t))){if(e.tag=n?s.tag:"?",s.represent){if(c=e.styleMap[s.tag]||s.defaultStyle,"[object Function]"===T.call(s.represent))i=s.represent(t,c);else{if(!L.call(s.represent,c))throw new N("!<"+s.tag+'> tag resolver accepts not "'+c+'" style');i=s.represent[c](t,c)}e.dump=i}return!0}return!1}function j(e,t,n,i,r,o){e.tag=null,e.dump=n,k(e,n,!1)||k(e,n,!0);var a=T.call(e.dump);i&&(i=e.flowLevel<0||e.flowLevel>t);var s,c,u="[object Object]"===a||"[object Array]"===a;if(u&&(s=e.duplicates.indexOf(n),c=-1!==s),(null!==e.tag&&"?"!==e.tag||c||2!==e.indent&&t>0)&&(r=!1),c&&e.usedDuplicates[s])e.dump="*ref_"+s;else{if(u&&c&&!e.usedDuplicates[s]&&(e.usedDuplicates[s]=!0),"[object Object]"===a)i&&0!==Object.keys(e.dump).length?(C(e,t,e.dump,r),c&&(e.dump="&ref_"+s+e.dump)):(w(e,t,e.dump),c&&(e.dump="&ref_"+s+" "+e.dump));else if("[object Array]"===a)i&&0!==e.dump.length?(b(e,t,e.dump,r),c&&(e.dump="&ref_"+s+e.dump)):(A(e,t,e.dump),c&&(e.dump="&ref_"+s+" "+e.dump));else{if("[object String]"!==a){if(e.skipInvalid)return!1;throw new N("unacceptable kind of an object to dump "+a)}"?"!==e.tag&&h(e,e.dump,t,o)}null!==e.tag&&"?"!==e.tag&&(e.dump="!<"+e.tag+"> "+e.dump)}return!0}function I(e,t){var n,i,r=[],o=[];for(S(e,r,o),n=0,i=o.length;i>n;n+=1)t.duplicates.push(r[o[n]]);t.usedDuplicates=new Array(i)}function S(e,t,n){var i,r,o;if(null!==e&&"object"==typeof e)if(r=t.indexOf(e),-1!==r)-1===n.indexOf(r)&&n.push(r);else if(t.push(e),Array.isArray(e))for(r=0,o=e.length;o>r;r+=1)S(e[r],t,n);else for(i=Object.keys(e),r=0,o=i.length;o>r;r+=1)S(e[i[r]],t,n)}function O(e,t){t=t||{};var n=new o(t);return n.noRefs||I(e,n),j(n,0,e,!0,!0)?n.dump+"\n":""}function E(e,t){return O(e,F.extend({schema:_},t))}var F=e("./common"),N=e("./exception"),M=e("./schema/default_full"),_=e("./schema/default_safe"),T=Object.prototype.toString,L=Object.prototype.hasOwnProperty,D=9,U=10,q=32,Y=33,R=34,B=35,P=37,W=38,K=39,$=42,H=44,G=45,V=58,Z=62,z=63,J=64,Q=91,X=93,ee=96,te=123,ne=124,ie=125,re={};re[0]="\\0",re[7]="\\a",re[8]="\\b",re[9]="\\t",re[10]="\\n",re[11]="\\v",re[12]="\\f",re[13]="\\r",re[27]="\\e",re[34]='\\"',re[92]="\\\\",re[133]="\\N",re[160]="\\_",re[8232]="\\L",re[8233]="\\P";var oe=["y","Y","yes","Yes","YES","on","On","ON","n","N","no","No","NO","off","Off","OFF"],ae=1,se=2,ce=3,ue=4,le=5;t.exports.dump=O,t.exports.safeDump=E},{"./common":2,"./exception":4,"./schema/default_full":9,"./schema/default_safe":10}],4:[function(e,t,n){"use strict";function i(e,t){Error.call(this),Error.captureStackTrace?Error.captureStackTrace(this,this.constructor):this.stack=(new Error).stack||"",this.name="YAMLException",this.reason=e,this.mark=t,this.message=(this.reason||"(unknown reason)")+(this.mark?" "+this.mark.toString():"")}i.prototype=Object.create(Error.prototype),i.prototype.constructor=i,i.prototype.toString=function(e){var t=this.name+": ";return t+=this.reason||"(unknown reason)",!e&&this.mark&&(t+=" "+this.mark.toString()),t},t.exports=i},{}],5:[function(e,t,n){"use strict";function i(e){return 10===e||13===e}function r(e){return 9===e||32===e}function o(e){return 9===e||32===e||10===e||13===e}function a(e){return 44===e||91===e||93===e||123===e||125===e}function s(e){var t;return e>=48&&57>=e?e-48:(t=32|e,t>=97&&102>=t?t-97+10:-1)}function c(e){return 120===e?2:117===e?4:85===e?8:0}function u(e){return e>=48&&57>=e?e-48:-1}function l(e){return 48===e?"\x00":97===e?"":98===e?"\b":116===e?"	":9===e?"	":110===e?"\n":118===e?"\x0B":102===e?"\f":114===e?"\r":101===e?"":32===e?" ":34===e?'"':47===e?"/":92===e?"\\":78===e?"Â…":95===e?"Â ":76===e?"\u2028":80===e?"\u2029":""}function p(e){return 65535>=e?String.fromCharCode(e):String.fromCharCode((e-65536>>10)+55296,(e-65536&1023)+56320)}function f(e,t){this.input=e,this.filename=t.filename||null,this.schema=t.schema||K,this.onWarning=t.onWarning||null,this.legacy=t.legacy||!1,this.json=t.json||!1,this.listener=t.listener||null,this.implicitTypes=this.schema.compiledImplicit,this.typeMap=this.schema.compiledTypeMap,this.length=e.length,this.position=0,this.line=0,this.lineStart=0,this.lineIndent=0,this.documents=[]}function d(e,t){return new B(t,new P(e.filename,e.input,e.position,e.line,e.position-e.lineStart))}function h(e,t){throw d(e,t)}function m(e,t){e.onWarning&&e.onWarning.call(null,d(e,t))}function g(e,t,n,i){var r,o,a,s;if(n>t){if(s=e.input.slice(t,n),i)for(r=0,o=s.length;o>r;r+=1)a=s.charCodeAt(r),9===a||a>=32&&1114111>=a||h(e,"expected valid JSON character");else X.test(s)&&h(e,"the stream contains non-printable characters");e.result+=s}}function y(e,t,n,i){var r,o,a,s;for(R.isObject(n)||h(e,"cannot merge mappings; the provided source object is unacceptable"),r=Object.keys(n),a=0,s=r.length;s>a;a+=1)o=r[a],$.call(t,o)||(t[o]=n[o],i[o]=!0)}function x(e,t,n,i,r,o){var a,s;if(r=String(r),null===t&&(t={}),"tag:yaml.org,2002:merge"===i)if(Array.isArray(o))for(a=0,s=o.length;s>a;a+=1)y(e,t,o[a],n);else y(e,t,o,n);else e.json||$.call(n,r)||!$.call(t,r)||h(e,"duplicated mapping key"),t[r]=o,delete n[r];return t}function v(e){var t;t=e.input.charCodeAt(e.position),10===t?e.position++:13===t?(e.position++,10===e.input.charCodeAt(e.position)&&e.position++):h(e,"a line break is expected"),e.line+=1,e.lineStart=e.position}function A(e,t,n){for(var o=0,a=e.input.charCodeAt(e.position);0!==a;){for(;r(a);)a=e.input.charCodeAt(++e.position);if(t&&35===a)do a=e.input.charCodeAt(++e.position);while(10!==a&&13!==a&&0!==a);if(!i(a))break;for(v(e),a=e.input.charCodeAt(e.position),o++,e.lineIndent=0;32===a;)e.lineIndent++,a=e.input.charCodeAt(++e.position)}return-1!==n&&0!==o&&e.lineIndent<n&&m(e,"deficient indentation"),o}function b(e){var t,n=e.position;return t=e.input.charCodeAt(n),(45===t||46===t)&&t===e.input.charCodeAt(n+1)&&t===e.input.charCodeAt(n+2)&&(n+=3,t=e.input.charCodeAt(n),0===t||o(t))}function w(e,t){1===t?e.result+=" ":t>1&&(e.result+=R.repeat("\n",t-1))}function C(e,t,n){var s,c,u,l,p,f,d,h,m,y=e.kind,x=e.result;if(m=e.input.charCodeAt(e.position),o(m)||a(m)||35===m||38===m||42===m||33===m||124===m||62===m||39===m||34===m||37===m||64===m||96===m)return!1;if((63===m||45===m)&&(c=e.input.charCodeAt(e.position+1),o(c)||n&&a(c)))return!1;for(e.kind="scalar",e.result="",u=l=e.position,p=!1;0!==m;){if(58===m){if(c=e.input.charCodeAt(e.position+1),o(c)||n&&a(c))break}else if(35===m){if(s=e.input.charCodeAt(e.position-1),o(s))break}else{if(e.position===e.lineStart&&b(e)||n&&a(m))break;if(i(m)){if(f=e.line,d=e.lineStart,h=e.lineIndent,A(e,!1,-1),e.lineIndent>=t){p=!0,m=e.input.charCodeAt(e.position);continue}e.position=l,e.line=f,e.lineStart=d,e.lineIndent=h;break}}p&&(g(e,u,l,!1),w(e,e.line-f),u=l=e.position,p=!1),r(m)||(l=e.position+1),m=e.input.charCodeAt(++e.position)}return g(e,u,l,!1),e.result?!0:(e.kind=y,e.result=x,!1)}function k(e,t){var n,r,o;if(n=e.input.charCodeAt(e.position),39!==n)return!1;for(e.kind="scalar",e.result="",e.position++,r=o=e.position;0!==(n=e.input.charCodeAt(e.position));)if(39===n){if(g(e,r,e.position,!0),n=e.input.charCodeAt(++e.position),39!==n)return!0;r=o=e.position,e.position++}else i(n)?(g(e,r,o,!0),w(e,A(e,!1,t)),r=o=e.position):e.position===e.lineStart&&b(e)?h(e,"unexpected end of the document within a single quoted scalar"):(e.position++,o=e.position);h(e,"unexpected end of the stream within a single quoted scalar")}function j(e,t){var n,r,o,a,u,l;if(l=e.input.charCodeAt(e.position),34!==l)return!1;for(e.kind="scalar",e.result="",e.position++,n=r=e.position;0!==(l=e.input.charCodeAt(e.position));){if(34===l)return g(e,n,e.position,!0),e.position++,!0;if(92===l){if(g(e,n,e.position,!0),l=e.input.charCodeAt(++e.position),i(l))A(e,!1,t);else if(256>l&&re[l])e.result+=oe[l],e.position++;else if((u=c(l))>0){for(o=u,a=0;o>0;o--)l=e.input.charCodeAt(++e.position),(u=s(l))>=0?a=(a<<4)+u:h(e,"expected hexadecimal character");e.result+=p(a),e.position++}else h(e,"unknown escape sequence");n=r=e.position}else i(l)?(g(e,n,r,!0),w(e,A(e,!1,t)),n=r=e.position):e.position===e.lineStart&&b(e)?h(e,"unexpected end of the document within a double quoted scalar"):(e.position++,r=e.position)}h(e,"unexpected end of the stream within a double quoted scalar")}function I(e,t){var n,i,r,a,s,c,u,l,p,f,d,m=!0,g=e.tag,y=e.anchor,v={};if(d=e.input.charCodeAt(e.position),91===d)a=93,u=!1,i=[];else{if(123!==d)return!1;a=125,u=!0,i={}}for(null!==e.anchor&&(e.anchorMap[e.anchor]=i),d=e.input.charCodeAt(++e.position);0!==d;){if(A(e,!0,t),d=e.input.charCodeAt(e.position),d===a)return e.position++,e.tag=g,e.anchor=y,e.kind=u?"mapping":"sequence",e.result=i,!0;m||h(e,"missed comma between flow collection entries"),p=l=f=null,s=c=!1,63===d&&(r=e.input.charCodeAt(e.position+1),o(r)&&(s=c=!0,e.position++,A(e,!0,t))),n=e.line,_(e,t,H,!1,!0),p=e.tag,l=e.result,A(e,!0,t),d=e.input.charCodeAt(e.position),!c&&e.line!==n||58!==d||(s=!0,d=e.input.charCodeAt(++e.position),A(e,!0,t),_(e,t,H,!1,!0),f=e.result),u?x(e,i,v,p,l,f):s?i.push(x(e,null,v,p,l,f)):i.push(l),A(e,!0,t),d=e.input.charCodeAt(e.position),44===d?(m=!0,d=e.input.charCodeAt(++e.position)):m=!1}h(e,"unexpected end of the stream within a flow collection")}function S(e,t){var n,o,a,s,c=z,l=!1,p=!1,f=t,d=0,m=!1;if(s=e.input.charCodeAt(e.position),124===s)o=!1;else{if(62!==s)return!1;o=!0}for(e.kind="scalar",e.result="";0!==s;)if(s=e.input.charCodeAt(++e.position),43===s||45===s)z===c?c=43===s?Q:J:h(e,"repeat of a chomping mode identifier");else{if(!((a=u(s))>=0))break;0===a?h(e,"bad explicit indentation width of a block scalar; it cannot be less than one"):p?h(e,"repeat of an indentation width identifier"):(f=t+a-1,p=!0)}if(r(s)){do s=e.input.charCodeAt(++e.position);while(r(s));if(35===s)do s=e.input.charCodeAt(++e.position);while(!i(s)&&0!==s)}for(;0!==s;){for(v(e),e.lineIndent=0,s=e.input.charCodeAt(e.position);(!p||e.lineIndent<f)&&32===s;)e.lineIndent++,s=e.input.charCodeAt(++e.position);if(!p&&e.lineIndent>f&&(f=e.lineIndent),i(s))d++;else{if(e.lineIndent<f){c===Q?e.result+=R.repeat("\n",l?1+d:d):c===z&&l&&(e.result+="\n");break}for(o?r(s)?(m=!0,e.result+=R.repeat("\n",l?1+d:d)):m?(m=!1,e.result+=R.repeat("\n",d+1)):0===d?l&&(e.result+=" "):e.result+=R.repeat("\n",d):e.result+=R.repeat("\n",l?1+d:d),l=!0,p=!0,d=0,n=e.position;!i(s)&&0!==s;)s=e.input.charCodeAt(++e.position);g(e,n,e.position,!1)}}return!0}function O(e,t){var n,i,r,a=e.tag,s=e.anchor,c=[],u=!1;for(null!==e.anchor&&(e.anchorMap[e.anchor]=c),r=e.input.charCodeAt(e.position);0!==r&&45===r&&(i=e.input.charCodeAt(e.position+1),o(i));)if(u=!0,e.position++,A(e,!0,-1)&&e.lineIndent<=t)c.push(null),r=e.input.charCodeAt(e.position);else if(n=e.line,_(e,t,V,!1,!0),c.push(e.result),A(e,!0,-1),r=e.input.charCodeAt(e.position),(e.line===n||e.lineIndent>t)&&0!==r)h(e,"bad indentation of a sequence entry");else if(e.lineIndent<t)break;return u?(e.tag=a,e.anchor=s,e.kind="sequence",e.result=c,!0):!1}function E(e,t,n){var i,a,s,c,u=e.tag,l=e.anchor,p={},f={},d=null,m=null,g=null,y=!1,v=!1;for(null!==e.anchor&&(e.anchorMap[e.anchor]=p),c=e.input.charCodeAt(e.position);0!==c;){if(i=e.input.charCodeAt(e.position+1),s=e.line,63!==c&&58!==c||!o(i)){if(!_(e,n,G,!1,!0))break;if(e.line===s){for(c=e.input.charCodeAt(e.position);r(c);)c=e.input.charCodeAt(++e.position);if(58===c)c=e.input.charCodeAt(++e.position),o(c)||h(e,"a whitespace character is expected after the key-value separator within a block mapping"),y&&(x(e,p,f,d,m,null),d=m=g=null),v=!0,y=!1,a=!1,d=e.tag,m=e.result;else{if(!v)return e.tag=u,e.anchor=l,!0;h(e,"can not read an implicit mapping pair; a colon is missed")}}else{if(!v)return e.tag=u,e.anchor=l,!0;h(e,"can not read a block mapping entry; a multiline key may not be an implicit key")}}else 63===c?(y&&(x(e,p,f,d,m,null),d=m=g=null),v=!0,y=!0,a=!0):y?(y=!1,a=!0):h(e,"incomplete explicit mapping pair; a key node is missed"),e.position+=1,c=i;if((e.line===s||e.lineIndent>t)&&(_(e,t,Z,!0,a)&&(y?m=e.result:g=e.result),y||(x(e,p,f,d,m,g),d=m=g=null),A(e,!0,-1),c=e.input.charCodeAt(e.position)),e.lineIndent>t&&0!==c)h(e,"bad indentation of a mapping entry");else if(e.lineIndent<t)break}return y&&x(e,p,f,d,m,null),v&&(e.tag=u,e.anchor=l,e.kind="mapping",e.result=p),v}function F(e){var t,n,i,r,a=!1,s=!1;if(r=e.input.charCodeAt(e.position),33!==r)return!1;if(null!==e.tag&&h(e,"duplication of a tag property"),r=e.input.charCodeAt(++e.position),60===r?(a=!0,r=e.input.charCodeAt(++e.position)):33===r?(s=!0,n="!!",r=e.input.charCodeAt(++e.position)):n="!",t=e.position,a){do r=e.input.charCodeAt(++e.position);while(0!==r&&62!==r);e.position<e.length?(i=e.input.slice(t,e.position),r=e.input.charCodeAt(++e.position)):h(e,"unexpected end of the stream within a verbatim tag")}else{for(;0!==r&&!o(r);)33===r&&(s?h(e,"tag suffix cannot contain exclamation marks"):(n=e.input.slice(t-1,e.position+1),ne.test(n)||h(e,"named tag handle cannot contain such characters"),s=!0,t=e.position+1)),r=e.input.charCodeAt(++e.position);i=e.input.slice(t,e.position),te.test(i)&&h(e,"tag suffix cannot contain flow indicator characters")}return i&&!ie.test(i)&&h(e,"tag name cannot contain such characters: "+i),a?e.tag=i:$.call(e.tagMap,n)?e.tag=e.tagMap[n]+i:"!"===n?e.tag="!"+i:"!!"===n?e.tag="tag:yaml.org,2002:"+i:h(e,'undeclared tag handle "'+n+'"'),!0}function N(e){var t,n;if(n=e.input.charCodeAt(e.position),38!==n)return!1;for(null!==e.anchor&&h(e,"duplication of an anchor property"),n=e.input.charCodeAt(++e.position),t=e.position;0!==n&&!o(n)&&!a(n);)n=e.input.charCodeAt(++e.position);return e.position===t&&h(e,"name of an anchor node must contain at least one character"),e.anchor=e.input.slice(t,e.position),!0}function M(e){var t,n,i;if(i=e.input.charCodeAt(e.position),42!==i)return!1;for(i=e.input.charCodeAt(++e.position),t=e.position;0!==i&&!o(i)&&!a(i);)i=e.input.charCodeAt(++e.position);return e.position===t&&h(e,"name of an alias node must contain at least one character"),n=e.input.slice(t,e.position),e.anchorMap.hasOwnProperty(n)||h(e,'unidentified alias "'+n+'"'),e.result=e.anchorMap[n],A(e,!0,-1),!0}function _(e,t,n,i,r){var o,a,s,c,u,l,p,f,d=1,m=!1,g=!1;if(null!==e.listener&&e.listener("open",e),e.tag=null,e.anchor=null,e.kind=null,e.result=null,o=a=s=Z===n||V===n,i&&A(e,!0,-1)&&(m=!0,e.lineIndent>t?d=1:e.lineIndent===t?d=0:e.lineIndent<t&&(d=-1)),1===d)for(;F(e)||N(e);)A(e,!0,-1)?(m=!0,s=o,e.lineIndent>t?d=1:e.lineIndent===t?d=0:e.lineIndent<t&&(d=-1)):s=!1;if(s&&(s=m||r),1!==d&&Z!==n||(p=H===n||G===n?t:t+1,f=e.position-e.lineStart,1===d?s&&(O(e,f)||E(e,f,p))||I(e,p)?g=!0:(a&&S(e,p)||k(e,p)||j(e,p)?g=!0:M(e)?(g=!0,null===e.tag&&null===e.anchor||h(e,"alias node should not have any properties")):C(e,p,H===n)&&(g=!0,null===e.tag&&(e.tag="?")),null!==e.anchor&&(e.anchorMap[e.anchor]=e.result)):0===d&&(g=s&&O(e,f))),null!==e.tag&&"!"!==e.tag)if("?"===e.tag){for(c=0,u=e.implicitTypes.length;u>c;c+=1)if(l=e.implicitTypes[c],l.resolve(e.result)){e.result=l.construct(e.result),e.tag=l.tag,null!==e.anchor&&(e.anchorMap[e.anchor]=e.result);break}}else $.call(e.typeMap,e.tag)?(l=e.typeMap[e.tag],null!==e.result&&l.kind!==e.kind&&h(e,"unacceptable node kind for !<"+e.tag+'> tag; it should be "'+l.kind+'", not "'+e.kind+'"'),l.resolve(e.result)?(e.result=l.construct(e.result),null!==e.anchor&&(e.anchorMap[e.anchor]=e.result)):h(e,"cannot resolve a node with !<"+e.tag+"> explicit tag")):h(e,"unknown tag !<"+e.tag+">");return null!==e.listener&&e.listener("close",e),null!==e.tag||null!==e.anchor||g}function T(e){var t,n,a,s,c=e.position,u=!1;for(e.version=null,e.checkLineBreaks=e.legacy,e.tagMap={},e.anchorMap={};0!==(s=e.input.charCodeAt(e.position))&&(A(e,!0,-1),s=e.input.charCodeAt(e.position),!(e.lineIndent>0||37!==s));){for(u=!0,s=e.input.charCodeAt(++e.position),t=e.position;0!==s&&!o(s);)s=e.input.charCodeAt(++e.position);for(n=e.input.slice(t,e.position),a=[],n.length<1&&h(e,"directive name must not be less than one character in length");0!==s;){for(;r(s);)s=e.input.charCodeAt(++e.position);if(35===s){do s=e.input.charCodeAt(++e.position);while(0!==s&&!i(s));break}if(i(s))break;for(t=e.position;0!==s&&!o(s);)s=e.input.charCodeAt(++e.position);a.push(e.input.slice(t,e.position))}0!==s&&v(e),$.call(se,n)?se[n](e,n,a):m(e,'unknown document directive "'+n+'"')}return A(e,!0,-1),0===e.lineIndent&&45===e.input.charCodeAt(e.position)&&45===e.input.charCodeAt(e.position+1)&&45===e.input.charCodeAt(e.position+2)?(e.position+=3,A(e,!0,-1)):u&&h(e,"directives end mark is expected"),_(e,e.lineIndent-1,Z,!1,!0),A(e,!0,-1),e.checkLineBreaks&&ee.test(e.input.slice(c,e.position))&&m(e,"non-ASCII line breaks are interpreted as content"),e.documents.push(e.result),e.position===e.lineStart&&b(e)?void(46===e.input.charCodeAt(e.position)&&(e.position+=3,A(e,!0,-1))):void(e.position<e.length-1&&h(e,"end of the stream or a document separator is expected"))}function L(e,t){e=String(e),t=t||{},0!==e.length&&(10!==e.charCodeAt(e.length-1)&&13!==e.charCodeAt(e.length-1)&&(e+="\n"),65279===e.charCodeAt(0)&&(e=e.slice(1)));var n=new f(e,t);for(n.input+="\x00";32===n.input.charCodeAt(n.position);)n.lineIndent+=1,n.position+=1;for(;n.position<n.length-1;)T(n);return n.documents}function D(e,t,n){var i,r,o=L(e,n);for(i=0,r=o.length;r>i;i+=1)t(o[i])}function U(e,t){var n=L(e,t);if(0!==n.length){if(1===n.length)return n[0];throw new B("expected a single document in the stream, but found more")}}function q(e,t,n){D(e,t,R.extend({schema:W},n))}function Y(e,t){return U(e,R.extend({schema:W},t))}for(var R=e("./common"),B=e("./exception"),P=e("./mark"),W=e("./schema/default_safe"),K=e("./schema/default_full"),$=Object.prototype.hasOwnProperty,H=1,G=2,V=3,Z=4,z=1,J=2,Q=3,X=/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uFFFE\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/,ee=/[\x85\u2028\u2029]/,te=/[,\[\]\{\}]/,ne=/^(?:!|!!|![a-z\-]+!)$/i,ie=/^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i,re=new Array(256),oe=new Array(256),ae=0;256>ae;ae++)re[ae]=l(ae)?1:0,oe[ae]=l(ae);var se={YAML:function(e,t,n){var i,r,o;null!==e.version&&h(e,"duplication of %YAML directive"),1!==n.length&&h(e,"YAML directive accepts exactly one argument"),i=/^([0-9]+)\.([0-9]+)$/.exec(n[0]),null===i&&h(e,"ill-formed argument of the YAML directive"),r=parseInt(i[1],10),o=parseInt(i[2],10),1!==r&&h(e,"unacceptable YAML version of the document"),e.version=n[0],e.checkLineBreaks=2>o,1!==o&&2!==o&&m(e,"unsupported YAML version of the document")},TAG:function(e,t,n){var i,r;2!==n.length&&h(e,"TAG directive accepts exactly two arguments"),i=n[0],r=n[1],ne.test(i)||h(e,"ill-formed tag handle (first argument) of the TAG directive"),$.call(e.tagMap,i)&&h(e,'there is a previously declared suffix for "'+i+'" tag handle'),ie.test(r)||h(e,"ill-formed tag prefix (second argument) of the TAG directive"),e.tagMap[i]=r}};t.exports.loadAll=D,t.exports.load=U,t.exports.safeLoadAll=q,t.exports.safeLoad=Y},{"./common":2,"./exception":4,"./mark":6,"./schema/default_full":9,"./schema/default_safe":10}],6:[function(e,t,n){"use strict";function i(e,t,n,i,r){this.name=e,this.buffer=t,this.position=n,this.line=i,this.column=r}var r=e("./common");i.prototype.getSnippet=function(e,t){var n,i,o,a,s;if(!this.buffer)return null;for(e=e||4,t=t||75,n="",i=this.position;i>0&&-1==="\x00\r\nÂ…\u2028\u2029".indexOf(this.buffer.charAt(i-1));)if(i-=1,this.position-i>t/2-1){n=" ... ",i+=5;break}for(o="",a=this.position;a<this.buffer.length&&-1==="\x00\r\nÂ…\u2028\u2029".indexOf(this.buffer.charAt(a));)if(a+=1,a-this.position>t/2-1){o=" ... ",a-=5;break}return s=this.buffer.slice(i,a),r.repeat(" ",e)+n+s+o+"\n"+r.repeat(" ",e+this.position-i+n.length)+"^"},i.prototype.toString=function(e){var t,n="";return this.name&&(n+='in "'+this.name+'" '),n+="at line "+(this.line+1)+", column "+(this.column+1),e||(t=this.getSnippet(),t&&(n+=":\n"+t)),n},t.exports=i},{"./common":2}],7:[function(e,t,n){"use strict";function i(e,t,n){var r=[];return e.include.forEach(function(e){n=i(e,t,n)}),e[t].forEach(function(e){n.forEach(function(t,n){t.tag===e.tag&&r.push(n)}),n.push(e)}),n.filter(function(e,t){return-1===r.indexOf(t)})}function r(){function e(e){i[e.tag]=e}var t,n,i={};for(t=0,n=arguments.length;n>t;t+=1)arguments[t].forEach(e);return i}function o(e){this.include=e.include||[],this.implicit=e.implicit||[],this.explicit=e.explicit||[],this.implicit.forEach(function(e){if(e.loadKind&&"scalar"!==e.loadKind)throw new s("There is a non-scalar type in the implicit list of a schema. Implicit resolving of such types is not supported.")}),this.compiledImplicit=i(this,"implicit",[]),this.compiledExplicit=i(this,"explicit",[]),this.compiledTypeMap=r(this.compiledImplicit,this.compiledExplicit)}var a=e("./common"),s=e("./exception"),c=e("./type");o.DEFAULT=null,o.create=function(){var e,t;switch(arguments.length){case 1:e=o.DEFAULT,t=arguments[0];break;case 2:e=arguments[0],t=arguments[1];break;default:throw new s("Wrong number of arguments for Schema.create function")}if(e=a.toArray(e),t=a.toArray(t),!e.every(function(e){return e instanceof o}))throw new s("Specified list of super schemas (or a single Schema object) contains a non-Schema object.");if(!t.every(function(e){return e instanceof c}))throw new s("Specified list of YAML types (or a single Type object) contains a non-Type object.");return new o({include:e,explicit:t})},t.exports=o},{"./common":2,"./exception":4,"./type":13}],8:[function(e,t,n){"use strict";var i=e("../schema");t.exports=new i({include:[e("./json")]})},{"../schema":7,"./json":12}],9:[function(e,t,n){"use strict";var i=e("../schema");t.exports=i.DEFAULT=new i({include:[e("./default_safe")],explicit:[e("../type/js/undefined"),e("../type/js/regexp"),e("../type/js/function")]})},{"../schema":7,"../type/js/function":18,"../type/js/regexp":19,"../type/js/undefined":20,"./default_safe":10}],10:[function(e,t,n){"use strict";var i=e("../schema");t.exports=new i({include:[e("./core")],implicit:[e("../type/timestamp"),e("../type/merge")],explicit:[e("../type/binary"),e("../type/omap"),e("../type/pairs"),e("../type/set")]})},{"../schema":7,"../type/binary":14,"../type/merge":22,"../type/omap":24,"../type/pairs":25,"../type/set":27,"../type/timestamp":29,"./core":8}],11:[function(e,t,n){"use strict";var i=e("../schema");t.exports=new i({explicit:[e("../type/str"),e("../type/seq"),e("../type/map")]})},{"../schema":7,"../type/map":21,"../type/seq":26,"../type/str":28}],12:[function(e,t,n){"use strict";var i=e("../schema");t.exports=new i({include:[e("./failsafe")],implicit:[e("../type/null"),e("../type/bool"),e("../type/int"),e("../type/float")]})},{"../schema":7,"../type/bool":15,"../type/float":16,"../type/int":17,"../type/null":23,"./failsafe":11}],13:[function(e,t,n){"use strict";function i(e){var t={};return null!==e&&Object.keys(e).forEach(function(n){e[n].forEach(function(e){t[String(e)]=n})}),t}function r(e,t){if(t=t||{},Object.keys(t).forEach(function(t){if(-1===a.indexOf(t))throw new o('Unknown option "'+t+'" is met in definition of "'+e+'" YAML type.')}),this.tag=e,this.kind=t.kind||null,this.resolve=t.resolve||function(){return!0},this.construct=t.construct||function(e){return e},this.instanceOf=t.instanceOf||null,this.predicate=t.predicate||null,this.represent=t.represent||null,this.defaultStyle=t.defaultStyle||null,this.styleAliases=i(t.styleAliases||null),-1===s.indexOf(this.kind))throw new o('Unknown kind "'+this.kind+'" is specified for "'+e+'" YAML type.')}var o=e("./exception"),a=["kind","resolve","construct","instanceOf","predicate","represent","defaultStyle","styleAliases"],s=["scalar","sequence","mapping"];t.exports=r},{"./exception":4}],14:[function(e,t,n){"use strict";function i(e){if(null===e)return!1;var t,n,i=0,r=e.length,o=p;for(n=0;r>n;n++)if(t=o.indexOf(e.charAt(n)),!(t>64)){if(0>t)return!1;i+=6}return i%8===0}function r(e){var t,n,i=e.replace(/[\r\n=]/g,""),r=i.length,o=p,a=0,c=[];for(t=0;r>t;t++)t%4===0&&t&&(c.push(a>>16&255),c.push(a>>8&255),c.push(255&a)),a=a<<6|o.indexOf(i.charAt(t));return n=r%4*6,0===n?(c.push(a>>16&255),c.push(a>>8&255),c.push(255&a)):18===n?(c.push(a>>10&255),c.push(a>>2&255)):12===n&&c.push(a>>4&255),s?new s(c):c}function o(e){var t,n,i="",r=0,o=e.length,a=p;for(t=0;o>t;t++)t%3===0&&t&&(i+=a[r>>18&63],i+=a[r>>12&63],i+=a[r>>6&63],i+=a[63&r]),r=(r<<8)+e[t];return n=o%3,0===n?(i+=a[r>>18&63],i+=a[r>>12&63],i+=a[r>>6&63],i+=a[63&r]):2===n?(i+=a[r>>10&63],i+=a[r>>4&63],i+=a[r<<2&63],i+=a[64]):1===n&&(i+=a[r>>2&63],i+=a[r<<4&63],i+=a[64],i+=a[64]),i}function a(e){return s&&s.isBuffer(e)}var s;try{var c=e;s=c("buffer").Buffer;
}catch(u){}var l=e("../type"),p="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";t.exports=new l("tag:yaml.org,2002:binary",{kind:"scalar",resolve:i,construct:r,predicate:a,represent:o})},{"../type":13}],15:[function(e,t,n){"use strict";function i(e){if(null===e)return!1;var t=e.length;return 4===t&&("true"===e||"True"===e||"TRUE"===e)||5===t&&("false"===e||"False"===e||"FALSE"===e)}function r(e){return"true"===e||"True"===e||"TRUE"===e}function o(e){return"[object Boolean]"===Object.prototype.toString.call(e)}var a=e("../type");t.exports=new a("tag:yaml.org,2002:bool",{kind:"scalar",resolve:i,construct:r,predicate:o,represent:{lowercase:function(e){return e?"true":"false"},uppercase:function(e){return e?"TRUE":"FALSE"},camelcase:function(e){return e?"True":"False"}},defaultStyle:"lowercase"})},{"../type":13}],16:[function(e,t,n){"use strict";function i(e){return null===e?!1:!!u.test(e)}function r(e){var t,n,i,r;return t=e.replace(/_/g,"").toLowerCase(),n="-"===t[0]?-1:1,r=[],"+-".indexOf(t[0])>=0&&(t=t.slice(1)),".inf"===t?1===n?Number.POSITIVE_INFINITY:Number.NEGATIVE_INFINITY:".nan"===t?NaN:t.indexOf(":")>=0?(t.split(":").forEach(function(e){r.unshift(parseFloat(e,10))}),t=0,i=1,r.forEach(function(e){t+=e*i,i*=60}),n*t):n*parseFloat(t,10)}function o(e,t){var n;if(isNaN(e))switch(t){case"lowercase":return".nan";case"uppercase":return".NAN";case"camelcase":return".NaN"}else if(Number.POSITIVE_INFINITY===e)switch(t){case"lowercase":return".inf";case"uppercase":return".INF";case"camelcase":return".Inf"}else if(Number.NEGATIVE_INFINITY===e)switch(t){case"lowercase":return"-.inf";case"uppercase":return"-.INF";case"camelcase":return"-.Inf"}else if(s.isNegativeZero(e))return"-0.0";return n=e.toString(10),l.test(n)?n.replace("e",".e"):n}function a(e){return"[object Number]"===Object.prototype.toString.call(e)&&(e%1!==0||s.isNegativeZero(e))}var s=e("../common"),c=e("../type"),u=new RegExp("^(?:[-+]?(?:[0-9][0-9_]*)\\.[0-9_]*(?:[eE][-+][0-9]+)?|\\.[0-9_]+(?:[eE][-+][0-9]+)?|[-+]?[0-9][0-9_]*(?::[0-5]?[0-9])+\\.[0-9_]*|[-+]?\\.(?:inf|Inf|INF)|\\.(?:nan|NaN|NAN))$"),l=/^[-+]?[0-9]+e/;t.exports=new c("tag:yaml.org,2002:float",{kind:"scalar",resolve:i,construct:r,predicate:a,represent:o,defaultStyle:"lowercase"})},{"../common":2,"../type":13}],17:[function(e,t,n){"use strict";function i(e){return e>=48&&57>=e||e>=65&&70>=e||e>=97&&102>=e}function r(e){return e>=48&&55>=e}function o(e){return e>=48&&57>=e}function a(e){if(null===e)return!1;var t,n=e.length,a=0,s=!1;if(!n)return!1;if(t=e[a],"-"!==t&&"+"!==t||(t=e[++a]),"0"===t){if(a+1===n)return!0;if(t=e[++a],"b"===t){for(a++;n>a;a++)if(t=e[a],"_"!==t){if("0"!==t&&"1"!==t)return!1;s=!0}return s}if("x"===t){for(a++;n>a;a++)if(t=e[a],"_"!==t){if(!i(e.charCodeAt(a)))return!1;s=!0}return s}for(;n>a;a++)if(t=e[a],"_"!==t){if(!r(e.charCodeAt(a)))return!1;s=!0}return s}for(;n>a;a++)if(t=e[a],"_"!==t){if(":"===t)break;if(!o(e.charCodeAt(a)))return!1;s=!0}return s?":"!==t?!0:/^(:[0-5]?[0-9])+$/.test(e.slice(a)):!1}function s(e){var t,n,i=e,r=1,o=[];return-1!==i.indexOf("_")&&(i=i.replace(/_/g,"")),t=i[0],"-"!==t&&"+"!==t||("-"===t&&(r=-1),i=i.slice(1),t=i[0]),"0"===i?0:"0"===t?"b"===i[1]?r*parseInt(i.slice(2),2):"x"===i[1]?r*parseInt(i,16):r*parseInt(i,8):-1!==i.indexOf(":")?(i.split(":").forEach(function(e){o.unshift(parseInt(e,10))}),i=0,n=1,o.forEach(function(e){i+=e*n,n*=60}),r*i):r*parseInt(i,10)}function c(e){return"[object Number]"===Object.prototype.toString.call(e)&&e%1===0&&!u.isNegativeZero(e)}var u=e("../common"),l=e("../type");t.exports=new l("tag:yaml.org,2002:int",{kind:"scalar",resolve:a,construct:s,predicate:c,represent:{binary:function(e){return"0b"+e.toString(2)},octal:function(e){return"0"+e.toString(8)},decimal:function(e){return e.toString(10)},hexadecimal:function(e){return"0x"+e.toString(16).toUpperCase()}},defaultStyle:"decimal",styleAliases:{binary:[2,"bin"],octal:[8,"oct"],decimal:[10,"dec"],hexadecimal:[16,"hex"]}})},{"../common":2,"../type":13}],18:[function(e,t,n){"use strict";function i(e){if(null===e)return!1;try{var t="("+e+")",n=s.parse(t,{range:!0});return"Program"===n.type&&1===n.body.length&&"ExpressionStatement"===n.body[0].type&&"FunctionExpression"===n.body[0].expression.type}catch(i){return!1}}function r(e){var t,n="("+e+")",i=s.parse(n,{range:!0}),r=[];if("Program"!==i.type||1!==i.body.length||"ExpressionStatement"!==i.body[0].type||"FunctionExpression"!==i.body[0].expression.type)throw new Error("Failed to resolve function");return i.body[0].expression.params.forEach(function(e){r.push(e.name)}),t=i.body[0].expression.body.range,new Function(r,n.slice(t[0]+1,t[1]-1))}function o(e){return e.toString()}function a(e){return"[object Function]"===Object.prototype.toString.call(e)}var s;try{var c=e;s=c("esprima")}catch(u){"undefined"!=typeof window&&(s=window.esprima)}var l=e("../../type");t.exports=new l("tag:yaml.org,2002:js/function",{kind:"scalar",resolve:i,construct:r,predicate:a,represent:o})},{"../../type":13}],19:[function(e,t,n){"use strict";function i(e){if(null===e)return!1;if(0===e.length)return!1;var t=e,n=/\/([gim]*)$/.exec(e),i="";if("/"===t[0]){if(n&&(i=n[1]),i.length>3)return!1;if("/"!==t[t.length-i.length-1])return!1}return!0}function r(e){var t=e,n=/\/([gim]*)$/.exec(e),i="";return"/"===t[0]&&(n&&(i=n[1]),t=t.slice(1,t.length-i.length-1)),new RegExp(t,i)}function o(e){var t="/"+e.source+"/";return e.global&&(t+="g"),e.multiline&&(t+="m"),e.ignoreCase&&(t+="i"),t}function a(e){return"[object RegExp]"===Object.prototype.toString.call(e)}var s=e("../../type");t.exports=new s("tag:yaml.org,2002:js/regexp",{kind:"scalar",resolve:i,construct:r,predicate:a,represent:o})},{"../../type":13}],20:[function(e,t,n){"use strict";function i(){return!0}function r(){}function o(){return""}function a(e){return"undefined"==typeof e}var s=e("../../type");t.exports=new s("tag:yaml.org,2002:js/undefined",{kind:"scalar",resolve:i,construct:r,predicate:a,represent:o})},{"../../type":13}],21:[function(e,t,n){"use strict";var i=e("../type");t.exports=new i("tag:yaml.org,2002:map",{kind:"mapping",construct:function(e){return null!==e?e:{}}})},{"../type":13}],22:[function(e,t,n){"use strict";function i(e){return"<<"===e||null===e}var r=e("../type");t.exports=new r("tag:yaml.org,2002:merge",{kind:"scalar",resolve:i})},{"../type":13}],23:[function(e,t,n){"use strict";function i(e){if(null===e)return!0;var t=e.length;return 1===t&&"~"===e||4===t&&("null"===e||"Null"===e||"NULL"===e)}function r(){return null}function o(e){return null===e}var a=e("../type");t.exports=new a("tag:yaml.org,2002:null",{kind:"scalar",resolve:i,construct:r,predicate:o,represent:{canonical:function(){return"~"},lowercase:function(){return"null"},uppercase:function(){return"NULL"},camelcase:function(){return"Null"}},defaultStyle:"lowercase"})},{"../type":13}],24:[function(e,t,n){"use strict";function i(e){if(null===e)return!0;var t,n,i,r,o,c=[],u=e;for(t=0,n=u.length;n>t;t+=1){if(i=u[t],o=!1,"[object Object]"!==s.call(i))return!1;for(r in i)if(a.call(i,r)){if(o)return!1;o=!0}if(!o)return!1;if(-1!==c.indexOf(r))return!1;c.push(r)}return!0}function r(e){return null!==e?e:[]}var o=e("../type"),a=Object.prototype.hasOwnProperty,s=Object.prototype.toString;t.exports=new o("tag:yaml.org,2002:omap",{kind:"sequence",resolve:i,construct:r})},{"../type":13}],25:[function(e,t,n){"use strict";function i(e){if(null===e)return!0;var t,n,i,r,o,s=e;for(o=new Array(s.length),t=0,n=s.length;n>t;t+=1){if(i=s[t],"[object Object]"!==a.call(i))return!1;if(r=Object.keys(i),1!==r.length)return!1;o[t]=[r[0],i[r[0]]]}return!0}function r(e){if(null===e)return[];var t,n,i,r,o,a=e;for(o=new Array(a.length),t=0,n=a.length;n>t;t+=1)i=a[t],r=Object.keys(i),o[t]=[r[0],i[r[0]]];return o}var o=e("../type"),a=Object.prototype.toString;t.exports=new o("tag:yaml.org,2002:pairs",{kind:"sequence",resolve:i,construct:r})},{"../type":13}],26:[function(e,t,n){"use strict";var i=e("../type");t.exports=new i("tag:yaml.org,2002:seq",{kind:"sequence",construct:function(e){return null!==e?e:[]}})},{"../type":13}],27:[function(e,t,n){"use strict";function i(e){if(null===e)return!0;var t,n=e;for(t in n)if(a.call(n,t)&&null!==n[t])return!1;return!0}function r(e){return null!==e?e:{}}var o=e("../type"),a=Object.prototype.hasOwnProperty;t.exports=new o("tag:yaml.org,2002:set",{kind:"mapping",resolve:i,construct:r})},{"../type":13}],28:[function(e,t,n){"use strict";var i=e("../type");t.exports=new i("tag:yaml.org,2002:str",{kind:"scalar",construct:function(e){return null!==e?e:""}})},{"../type":13}],29:[function(e,t,n){"use strict";function i(e){return null===e?!1:null!==s.exec(e)?!0:null!==c.exec(e)}function r(e){var t,n,i,r,o,a,u,l,p,f,d=0,h=null;if(t=s.exec(e),null===t&&(t=c.exec(e)),null===t)throw new Error("Date resolve error");if(n=+t[1],i=+t[2]-1,r=+t[3],!t[4])return new Date(Date.UTC(n,i,r));if(o=+t[4],a=+t[5],u=+t[6],t[7]){for(d=t[7].slice(0,3);d.length<3;)d+="0";d=+d}return t[9]&&(l=+t[10],p=+(t[11]||0),h=6e4*(60*l+p),"-"===t[9]&&(h=-h)),f=new Date(Date.UTC(n,i,r,o,a,u,d)),h&&f.setTime(f.getTime()-h),f}function o(e){return e.toISOString()}var a=e("../type"),s=new RegExp("^([0-9][0-9][0-9][0-9])-([0-9][0-9])-([0-9][0-9])$"),c=new RegExp("^([0-9][0-9][0-9][0-9])-([0-9][0-9]?)-([0-9][0-9]?)(?:[Tt]|[ \\t]+)([0-9][0-9]?):([0-9][0-9]):([0-9][0-9])(?:\\.([0-9]*))?(?:[ \\t]*(Z|([-+])([0-9][0-9]?)(?::([0-9][0-9]))?))?$");t.exports=new a("tag:yaml.org,2002:timestamp",{kind:"scalar",resolve:i,construct:r,instanceOf:Date,represent:o})},{"../type":13}],"/":[function(e,t,n){"use strict";var i=e("./lib/js-yaml.js");t.exports=i},{"./lib/js-yaml.js":1}]},{},[])("/")});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languages/yaml/yamlPlugin',[
	'orion/plugin', 
	'orion/editor/stylers/text_x-yaml/syntax',
	'js-yaml/js-yaml'
], function(PluginProvider, mYAML, JsYaml) {

	function connect() {
		var headers = {
			name: "Orion YAML Tool Support",
			version: "1.0",
			description: "This plugin provides YAML tools support for Orion."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-yaml",
					"extends": "text/plain",
					name: "YAML",
					extension: ["yaml", "yml"]
				}
			] 
		});
		mYAML.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
		
		pluginProvider.registerService("orion.edit.validator", {
			/**
			 * @callback
			 */
			computeProblems: function(editorContext , context, config) {
				return editorContext.getText().then(function(text) {
					if(text) {
						var errors = [];
						try {
							JsYaml.safeLoad(text,
								{
									onWarning: function onWarning(err) {
										errors.push(err);
									},
									/**
									 * @callback
									 */
									listener: function listener(type, node) {
										//TODO create an AST here for more advanced support
									}
								});
						}
						catch(yerr) {
							if(yerr && yerr.name === 'YAMLException') {
								errors.push(yerr);
							}						
						}
						return errors.map(function(err) {
							var oerr = {
								id: "yaml.syntax.error",
								severity: "error",
								description: err.reason || err.message
							};
							if(err.name === 'YAMLException') {
								oerr.line = err.mark.line+1;
								oerr.start = err.mark.column;
								oerr.end = err.mark.position || err.mark.column+1;
							}
							return oerr;
						});
					}
					return [];
				});
			}
		},
		{
			contentType: ["text/x-yaml"],
			pid: 'yaml.validator.config'
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('plugins/languageToolsPlugin',[
	'orion/plugin',
	'plugins/languages/arduino/arduinoPlugin',
	'plugins/languages/bash/bashPlugin',
	'plugins/languages/c/cPlugin',
	'plugins/languages/coffeescript/coffeescriptPlugin',
	'plugins/languages/cpp/cppPlugin',
	'plugins/languages/csharp/csharpPlugin',
	'plugins/languages/docker/dockerPlugin',
	'plugins/languages/erlang/erlangPlugin',
	'plugins/languages/go/goPlugin',
	'plugins/languages/git/gitFilePlugin',
	'plugins/languages/haml/hamlPlugin',
	'plugins/languages/jade/jadePlugin',
	'plugins/languages/java/javaPlugin',
	'plugins/languages/jsx/jsxPlugin',
	'plugins/languages/launch/launchPlugin',
	'plugins/languages/less/lessPlugin',
	'plugins/languages/lua/luaPlugin',
	'plugins/languages/markdown/markdownPlugin',
	'plugins/languages/objectiveC/objectiveCPlugin',
	'plugins/languages/php/phpPlugin',
	'plugins/languages/properties/propertiesPlugin',
	'plugins/languages/python/pythonPlugin',
	'plugins/languages/ruby/rubyPlugin',
	'plugins/languages/scss/scssPlugin',
	'plugins/languages/smarty/smartyPlugin',
	'plugins/languages/sql/sqlPlugin',
	'plugins/languages/swift/swiftPlugin',
	'plugins/languages/typescript/typescriptPlugin',
	'plugins/languages/vb/vbPlugin',
	'plugins/languages/xml/xmlPlugin',
	'plugins/languages/xquery/xqueryPlugin',
	'plugins/languages/yaml/yamlPlugin'
], function(PluginProvider) {

	var plugins = Array.prototype.slice.call(arguments);
	plugins.shift();

	function connect() {
		var headers = {
			name: "Orion Languages Tool Support",
			version: "1.0",
			description: "This plugin provides tooling for languages that are not included in other core Orion plugins."
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(provider) {
		plugins.forEach(function(plugin) {
			plugin.registerServiceProviders(provider);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

