/*******************************************************************************
 * Copyright (c) 2011 Google, Inc.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Google, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.wb.internal.core.model.property.editor;

import org.eclipse.wb.internal.core.DesignerPlugin;
import org.eclipse.wb.internal.core.model.ModelMessages;
import org.eclipse.wb.internal.core.model.property.Property;
import org.eclipse.wb.internal.core.utils.ui.dialogs.StringsDialog;

import org.eclipse.jface.window.Window;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * {@link PropertyEditor} for array of {@link String}'s.
 *
 * @author scheglov_ke
 * @coverage core.model.property.editor
 */
public final class StringArrayPropertyEditor extends TextDialogPropertyEditor {
	////////////////////////////////////////////////////////////////////////////
	//
	// Instance
	//
	////////////////////////////////////////////////////////////////////////////
	public static final PropertyEditor INSTANCE = new StringArrayPropertyEditor();

	private StringArrayPropertyEditor() {
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// Presentation
	//
	////////////////////////////////////////////////////////////////////////////
	@Override
	protected String getText(Property property) throws Exception {
		String[] items = getItems(property);
		return "[" + StringUtils.join(items, ", ") + "]";
	}

	/**
	 * @return the items specified in value of given {@link Property}.
	 */
	private static String[] getItems(Property property) throws Exception {
		Object value = property.getValue();
		if (value instanceof String[]) {
			return (String[]) value;
		}
		// no items
		return ArrayUtils.EMPTY_STRING_ARRAY;
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// Editing
	//
	////////////////////////////////////////////////////////////////////////////
	@Override
	protected void openDialog(Property property) throws Exception {
		StringsDialog itemsDialog =
				new StringsDialog(DesignerPlugin.getShell(),
						DesignerPlugin.getDefault(),
						property.getTitle(),
						ModelMessages.StringArrayPropertyEditor_itemsLabel,
						ModelMessages.StringArrayPropertyEditor_hint);
		itemsDialog.setItems(getItems(property));
		// open dialog
		if (itemsDialog.open() == Window.OK) {
			property.setValue(itemsDialog.getItems());
		}
	}
}
