/*
   This file is part of TALER
   Copyright (C) 2022 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file exchangedb/pg_get_refresh.c
 * @brief Implementation of the get_refresh function for Postgres
 * @author get_refresh
 */
#include "taler/platform.h"
#include "taler/taler_error_codes.h"
#include "taler/taler_dbevents.h"
#include "taler/taler_pq_lib.h"
#include "pg_get_refresh.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TEH_PG_get_refresh (void *cls,
                    const struct TALER_RefreshCommitmentP *rc,
                    struct TALER_EXCHANGEDB_Refresh_vDOLDPLUS *refresh)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_auto_from_type (rc),
    GNUNET_PQ_query_param_end
  };
  bool no_cs_r_values;
  bool no_cs_r_choices;
  bool no_transfer_pubs;
  size_t num_denom_sigs;
  size_t num_transfer_pubs;
  struct TALER_BlindedDenominationSignature *denom_sigs = NULL;
  struct GNUNET_CRYPTO_CSPublicRPairP *cs_r_values = NULL;
  struct TALER_TransferPublicKeyP *transfer_pubs = NULL;
  uint64_t *denom_serials = NULL;
  struct GNUNET_PQ_ResultSpec rs[] = {
    TALER_PQ_RESULT_SPEC_AMOUNT ("amount_with_fee",
                                 &refresh->amount_with_fee),
    GNUNET_PQ_result_spec_auto_from_type ("old_coin_pub",
                                          &refresh->coin.coin_pub),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_auto_from_type ("age_commitment_hash",
                                            &refresh->coin.h_age_commitment),
      &refresh->coin.no_age_commitment),
    GNUNET_PQ_result_spec_auto_from_type ("old_coin_sig",
                                          &refresh->coin_sig),
    GNUNET_PQ_result_spec_auto_from_type ("refresh_seed",
                                          &refresh->refresh_seed),
    GNUNET_PQ_result_spec_uint32 ("noreveal_index",
                                  &refresh->noreveal_index),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_auto_from_type ("blinding_seed",
                                            &refresh->blinding_seed),
      &refresh->no_blinding_seed),
    GNUNET_PQ_result_spec_allow_null (
      TALER_PQ_result_spec_array_cs_r_pub (pg->conn,
                                           "cs_r_values",
                                           &refresh->num_cs_r_values,
                                           &cs_r_values),
      &no_cs_r_values),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_uint64 ("cs_r_choices",
                                    &refresh->cs_r_choices),
      &no_cs_r_choices),
    GNUNET_PQ_result_spec_auto_from_type ("planchets_h",
                                          &refresh->planchets_h),
    GNUNET_PQ_result_spec_auto_from_type ("selected_h",
                                          &refresh->selected_h),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_array_fixed_size (pg->conn,
                                              "transfer_pubs",
                                              sizeof(*transfer_pubs),
                                              &num_transfer_pubs,
                                              (void **) &transfer_pubs),
      &no_transfer_pubs),
    GNUNET_PQ_result_spec_array_uint64 (pg->conn,
                                        "denom_serials",
                                        &refresh->num_coins,
                                        &denom_serials),
    TALER_PQ_result_spec_array_blinded_denom_sig (pg->conn,
                                                  "denom_sigs",
                                                  &num_denom_sigs,
                                                  &denom_sigs),
    GNUNET_PQ_result_spec_bool ("revealed",
                                &refresh->revealed),
    GNUNET_PQ_result_spec_end
  };
  enum GNUNET_DB_QueryStatus qs;

  memset (&refresh->coin.denom_sig,
          0,
          sizeof (refresh->coin.denom_sig));
  PREPARE (pg,
           "get_refresh",
           "SELECT"
           " amount_with_fee"
           ",old_coin_pub"
           ",kc.age_commitment_hash AS age_commitment_hash"
           ",old_coin_sig"
           ",refresh_seed"
           ",noreveal_index"
           ",blinding_seed"
           ",cs_r_values"
           ",cs_r_choices"
           ",planchets_h"
           ",transfer_pubs"
           ",selected_h"
           ",denom_serials"
           ",denom_sigs"
           ",revealed"
           " FROM refresh"
           " JOIN known_coins kc"
           " ON (old_coin_pub = kc.coin_pub)"
           " WHERE rc = $1;"
           );
  qs = GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                 "get_refresh",
                                                 params,
                                                 rs);
  GNUNET_PQ_cleanup_query_params_closures (params);
  if (0 > qs)
  {
    GNUNET_break (0);
    GNUNET_PQ_cleanup_result (rs);
    return qs;
  }
  if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == qs)
  {
    GNUNET_PQ_cleanup_result (rs);
    return qs;
  }
  if (refresh->num_coins != num_denom_sigs)
  {
    GNUNET_break (0);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "got inconsistent number of entries in refresh from DB: "
                "num_coins=%ld, num_denom_sigs=%ld\n",
                refresh->num_coins,
                num_denom_sigs);
    GNUNET_PQ_cleanup_result (rs);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (no_transfer_pubs)
  {
    refresh->is_v27_refresh = true;
    refresh->transfer_pubs = NULL;
  }
  else
  {
    if (num_transfer_pubs != refresh->num_coins)
    {
      GNUNET_break (0);
      GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                  "got inconsistent number of transfer_pubs in refresh from DB: "
                  "num_coins=%ld, num_transfer_pubs=%ld\n",
                  refresh->num_coins,
                  num_transfer_pubs);
      GNUNET_PQ_cleanup_result (rs);
      return GNUNET_DB_STATUS_HARD_ERROR;
    }
    refresh->is_v27_refresh = false;
    refresh->transfer_pubs = transfer_pubs;
  }
  if (refresh->no_blinding_seed != no_cs_r_values)
  {
    GNUNET_break (0);
    GNUNET_PQ_cleanup_result (rs);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (no_cs_r_choices != no_cs_r_values)
  {
    GNUNET_break (0);
    GNUNET_PQ_cleanup_result (rs);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (no_cs_r_values)
  {
    refresh->cs_r_values = NULL;
    refresh->num_cs_r_values = 0;
  }
  if (refresh->coin.no_age_commitment)
    memset (&refresh->coin.h_age_commitment,
            0,
            sizeof(refresh->coin.h_age_commitment));
  refresh->rc = *rc;
  /* move the result arrays */
  refresh->denom_sigs = denom_sigs;
  refresh->denom_serials = denom_serials;
  refresh->cs_r_values = cs_r_values;
  transfer_pubs = NULL;
  denom_sigs = NULL;
  denom_serials = NULL;
  cs_r_values = NULL;
  GNUNET_PQ_cleanup_result (rs);
  return qs;
}
