/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import type { AbsoluteTime } from "@gnu-taler/taler-util";
import type { InternationalizationAPI } from "@gnu-taler/web-util/browser";
import { BaseForm } from "../context/ui-forms.js";
import { resolutionSection } from "./simplest.js";

export const v1 = (i18n: InternationalizationAPI) => ({
  design: [
    {
      title: i18n.str`Information on customer`,
      description: i18n.str`The customer is the person with whom the member concludes the contract with regard to the financial service provided (civil law). Does the member act as director of a domiciliary company, this domiciliary company is the customer.`,
      fields: [
        {
          type: "choiceStacked",
          properties: {
            name: "customerType",
            label: i18n.str`Type of customer`,
            required: true,
            choices: [
              {
                label: i18n.str`Natural person`,
                value: "natural",
              },
              {
                label: i18n.str`Legal entity`,
                value: "legal",
              },
            ],
          },
        },
        {
          type: "text",
          properties: {
            name: "naturalCustomer.fullName",
            label: i18n.str`Full name`,
            required: true,
          },
        },
        {
          type: "text",
          properties: {
            name: "naturalCustomer.address",
            label: i18n.str`Residential address`,
            required: true,
          },
        },
        {
          type: "integer",
          properties: {
            name: "naturalCustomer.telephone",
            label: i18n.str`Telephone`,
          },
        },
        {
          type: "text",
          properties: {
            name: "naturalCustomer.email",
            label: i18n.str`E-mail`,
          },
        },
        {
          type: "absoluteTime",
          properties: {
            name: "naturalCustomer.dateOfBirth",
            label: i18n.str`Date of birth`,
            required: true,
            // help: i18n.str`format 'dd/MM/yyyy'`,
          },
        },
        {
          type: "text",
          properties: {
            name: "naturalCustomer.nationality",
            label: i18n.str`Nationality`,
            required: true,
          },
        },
        {
          type: "text",
          properties: {
            name: "naturalCustomer.document",
            label: i18n.str`Identification document`,
            required: true,
          },
        },
        {
          type: "file",
          properties: {
            name: "naturalCustomer.documentAttachment",
            label: i18n.str`Document attachment`,
            required: true,
            maxBites: 2 * 1024 * 1024,
            accept: ".png",
            help: i18n.str`Max size of 2 mega bytes`,
          },
        },
        {
          type: "text",
          properties: {
            name: "naturalCustomer.companyName",
            label: i18n.str`Company name`,
          },
        },
        {
          type: "text",
          properties: {
            name: "naturalCustomer.office",
            label: i18n.str`Registered office`,
          },
        },
        {
          type: "text",
          properties: {
            name: "naturalCustomer.companyDocument",
            label: i18n.str`Company identification document`,
          },
        },
        {
          type: "file",
          properties: {
            name: "naturalCustomer.companyDocumentAttachment",
            label: i18n.str`Document attachment`,
            required: true,
            maxBites: 2 * 1024 * 1024,
            accept: ".png",
            help: i18n.str`Max size of 2 mega bytes`,
          },
        },
        {
          type: "text",
          properties: {
            name: "legalCustomer.companyName",
            label: i18n.str`Company name`,
            required: true,
          },
        },
        {
          type: "text",
          properties: {
            name: "legalCustomer.domicile",
            label: i18n.str`Domicile`,
            required: true,
          },
        },
        {
          type: "text",
          properties: {
            name: "legalCustomer.contactPerson",
            label: i18n.str`Contact person`,
          },
        },
        {
          type: "text",
          properties: {
            name: "legalCustomer.telephone",
            label: i18n.str`Telephone`,
          },
        },
        {
          type: "text",
          properties: {
            name: "legalCustomer.email",
            label: i18n.str`E-mail`,
          },
        },
        {
          type: "text",
          properties: {
            name: "legalCustomer.document",
            label: i18n.str`Identification document`,
            help: i18n.str`Not older than 12 month`,
          },
        },
        {
          type: "file",
          properties: {
            name: "legalCustomer.documentAttachment",
            label: i18n.str`Document attachment`,
            required: true,
            maxBites: 2 * 1024 * 1024,
            accept: ".png",
            help: i18n.str`Max size of 2 mega bytes`,
          },
        },
      ],
    },
    {
      title: i18n.str`Information on the natural persons who establish the business relationship for legal entities and partnerships`,
      description: i18n.str`For legal entities and partnerships the identity of the natural persons who establish the business relationship must be verified.`,
      fields: [
        {
          type: "array",
          properties: {
            name: "businessEstablisher",
            label: i18n.str`Persons`,
            required: true,
            placeholder: i18n.str`this is the placeholder`,
            fields: [
              {
                type: "text",
                properties: {
                  name: "fullName",
                  label: i18n.str`Full name`,
                  required: true,
                },
              },
              {
                type: "text",
                properties: {
                  name: "address",
                  label: i18n.str`Residential address`,
                  required: true,
                },
              },
              {
                type: "absoluteTime",
                properties: {
                  name: "dateOfBirth",
                  label: i18n.str`Date of birth`,
                  required: true,
                  // help: i18n.str`format 'dd/MM/yyyy'`,
                },
              },
              {
                type: "text",
                properties: {
                  name: "nationality",
                  label: i18n.str`Nationality`,
                  required: true,
                },
              },
              {
                type: "text",
                properties: {
                  name: "typeOfAuthorization",
                  label: i18n.str`Type of authorization (signatory of representation)`,
                  required: true,
                },
              },
              {
                type: "file",
                properties: {
                  name: "documentAttachment",
                  label: i18n.str`Identification document attachment`,
                  required: true,
                  maxBites: 2 * 1024 * 1024,
                  accept: ".png",
                  help: i18n.str`Max size of 2 mega bytes`,
                },
              },
              {
                type: "choiceStacked",
                properties: {
                  name: "powerOfAttorneyArrangements",
                  label: i18n.str`Power of attorney arrangements`,
                  required: true,
                  choices: [
                    {
                      label: i18n.str`CR extract`,
                      value: "cr",
                    },
                    {
                      label: i18n.str`Mandate`,
                      value: "mandate",
                    },
                    {
                      label: i18n.str`Other`,
                      value: "other",
                    },
                  ],
                },
              },
              {
                type: "text",
                properties: {
                  name: "powerOfAttorneyArrangementsOther",
                  label: i18n.str`Power of attorney arrangements`,
                  required: true,
                },
              },
            ],
            labelField: "fullName",
          },
        },
      ],
    },
    {
      title: i18n.str`Acceptance of business relationship`,
      fields: [
        {
          type: "absoluteTime",
          properties: {
            name: "acceptance.when",
            pattern: "dd/MM/yyyy",
            label: i18n.str`Date (conclusion of contract)`,
            // help: i18n.str`format 'dd/MM/yyyy'`,
          },
        },
        {
          type: "choiceStacked",
          properties: {
            name: "acceptance.acceptedBy",
            label: i18n.str`Accepted by`,
            required: true,
            choices: [
              {
                label: i18n.str`Face-to-face meeting with customer`,
                value: "face-to-face",
              },
              {
                label: i18n.str`Correspondence: authenticated copy of identification document obtained`,
                value: "correspondence-document",
              },
              {
                label: i18n.str`Correspondence: residential address validated`,
                value: "correspondence-address",
              },
            ],
          },
        },
        {
          type: "choiceStacked",
          properties: {
            name: "acceptance.typeOfCorrespondence",
            label: i18n.str`Type of correspondence service`,
            choices: [
              {
                label: i18n.str`to the customer`,
                value: "customer",
              },
              {
                label: i18n.str`hold at bank`,
                value: "bank",
              },
              {
                label: i18n.str`to the member`,
                value: "member",
              },
              {
                label: i18n.str`to a third party`,
                value: "third-party",
              },
            ],
          },
        },
        {
          type: "text",
          properties: {
            name: "acceptance.thirdPartyFullName",
            label: i18n.str`Third party full name`,
            required: true,
          },
        },
        {
          type: "text",
          properties: {
            name: "acceptance.thirdPartyAddress",
            label: i18n.str`Third party  address`,
            required: true,
          },
        },
        {
          type: "selectMultiple",
          properties: {
            name: "acceptance.language",
            label: i18n.str`Languages`,
            choices: ["asd"],
            unique: true,
          },
        },
        {
          type: "textArea",
          properties: {
            name: "acceptance.furtherInformation",
            label: i18n.str`Further information`,
          },
        },
      ],
    },
    {
      title: i18n.str`Information on the beneficial owner of the assets and/or controlling person`,
      description: i18n.str`Establishment of the beneficial owner of the assets and/or controlling person`,
      fields: [
        {
          type: "choiceStacked",
          properties: {
            name: "establishment",
            label: i18n.str`The customer is`,
            required: true,
            choices: [
              {
                label: i18n.str`a natural person and there are no doubts that this person is the sole beneficial owner of the assets`,
                value: "natural",
              },
              {
                label: i18n.str`a foundation (or a similar construct; incl. underlying companies)`,
                value: "foundation",
              },
              {
                label: i18n.str`a trust (incl. underlying companies)`,
                value: "trust",
              },
              {
                label: i18n.str`a life insurance policy with separately managed accounts/securities accounts`,
                value: "insurance-wrapper",
              },
              {
                label: i18n.str`all other cases`,
                value: "other",
              },
            ],
          },
        },
      ],
    },
    {
      title: i18n.str`Evaluation with regard to embargo procedures/terrorism lists on establishing the business relationship`,
      description: i18n.str`Verification whether the customer, beneficial owners of the assets, controlling persons, authorized representatives or other involved persons are listed on an embargo/terrorism list (date of verification/result)`,
      fields: [
        {
          type: "textArea",
          properties: {
            name: "embargoEvaluation",
            help: i18n.str`The evaluation must be made at the beginning of the business relationship and has to be repeated in the case of permanent business relationship every time the according lists are updated.`,
            label: i18n.str`Evaluation`,
          },
        },
      ],
    },
    {
      title: i18n.str`In the case of cash transactions/occasional customers: Information on type and purpose of business relationship`,
      description: i18n.str`These details are only necessary for occasional customers, i.e. money exchange, money and asset transfer or other cash transactions provided that no customer profile (VQF doc. No. 902.5) is created`,
      fields: [
        {
          type: "choiceStacked",
          properties: {
            name: "cashTransactions.typeOfBusiness",
            label: i18n.str`Type of business relationship`,
            choices: [
              {
                label: i18n.str`Money exchange`,
                value: "money-exchange",
              },
              {
                label: i18n.str`Money and asset transfer`,
                value: "money-and-asset-transfer",
              },
              {
                label: i18n.str`Other cash transactions. Specify below`,
                value: "other",
              },
            ],
          },
        },
        {
          type: "text",
          properties: {
            name: "cashTransactions.otherTypeOfBusiness",
            required: true,
            label: i18n.str`Specify other cash transactions:`,
          },
        },
        {
          type: "textArea",
          properties: {
            name: "cashTransactions.purpose",
            label: i18n.str`Purpose of the business relationship (purpose of service requested)`,
          },
        },
      ],
    },
    resolutionSection(i18n),
  ],
  // behavior: function formBehavior(
  //   v: Partial<Form902_1.Form>,
  // ): FormState<Form902_1.Form> {
  //   return {
  //     fullName: {
  //       disabled: true,
  //     },
  //     businessEstablisher: {
  //       elements: (v.businessEstablisher ?? []).map((be) => {
  //         return {
  //           powerOfAttorneyArrangementsOther: {
  //             hidden: be.powerOfAttorneyArrangements !== "other",
  //           },
  //         };
  //       }),
  //     },
  //     acceptance: {
  //       thirdPartyFullName: {
  //         hidden: v.acceptance?.typeOfCorrespondence !== "third-party",
  //       },
  //       thirdPartyAddress: {
  //         hidden: v.acceptance?.typeOfCorrespondence !== "third-party",
  //       },
  //     },
  //     cashTransactions: {
  //       otherTypeOfBusiness: {
  //         hidden: v.cashTransactions?.typeOfBusiness !== "other",
  //       },
  //     },
  //     naturalCustomer: {
  //       fullName: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       address: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       telephone: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       email: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       dateOfBirth: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       nationality: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       document: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       companyName: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       office: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       companyDocument: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       companyDocumentAttachment: {
  //         hidden: v.customerType !== "natural",
  //       },
  //       documentAttachment: {
  //         hidden: v.customerType !== "natural",
  //       },
  //     },
  //     legalCustomer: {
  //       companyName: {
  //         hidden: v.customerType !== "legal",
  //       },
  //       contactPerson: {
  //         hidden: v.customerType !== "legal",
  //       },
  //       document: {
  //         hidden: v.customerType !== "legal",
  //       },
  //       domicile: {
  //         hidden: v.customerType !== "legal",
  //       },
  //       email: {
  //         hidden: v.customerType !== "legal",
  //       },
  //       telephone: {
  //         hidden: v.customerType !== "legal",
  //       },
  //       documentAttachment: {
  //         hidden: v.customerType !== "legal",
  //       },
  //     },
  //   };
  // },
});

namespace Form902_1 {
  interface LegalEntityCustomer {
    companyName: string;
    domicile: string;
    contactPerson: string;
    telephone: string;
    email: string;
    document: string;
    documentAttachment: string;
  }
  interface NaturalCustomer {
    fullName: string;
    address: string;
    telephone: string;
    email: string;
    dateOfBirth: AbsoluteTime;
    nationality: string;
    document: string;
    documentAttachment: string;
    companyName: string;
    office: string;
    companyDocument: string;
    companyDocumentAttachment: string;
  }

  interface Person {
    fullName: string;
    address: string;
    dateOfBirth: AbsoluteTime;
    nationality: string;
    typeOfAuthorization: string;
    document: string;
    documentAttachment: string;
    powerOfAttorneyArrangements: "cr" | "mandate" | "other";
    powerOfAttorneyArrangementsOther: string;
  }

  interface Acceptance {
    when: AbsoluteTime;
    acceptedBy: "face-to-face" | "authenticated-copy";
    typeOfCorrespondence: string;
    language: string[];
    furtherInformation: string;
    thirdPartyFullName: string;
    thirdPartyAddress: string;
  }

  interface BeneficialOwner {
    establishment:
      | "natural-person"
      | "foundation"
      | "trust"
      | "insurance-wrapper"
      | "other";
  }

  interface CashTransactions {
    typeOfBusiness: "money-exchange" | "money-and-asset-transfer" | "other";
    otherTypeOfBusiness: string;
    purpose: string;
  }

  export interface Form extends BaseForm {
    fullName: string;
    customerType: "natural" | "legal";
    naturalCustomer: NaturalCustomer;
    legalCustomer: LegalEntityCustomer;
    businessEstablisher: Array<Person>;
    acceptance: Acceptance;
    beneficialOwner: BeneficialOwner;
    embargoEvaluation: string;
    cashTransactions: CashTransactions;
    // enclosures: Enclosures;
  }
}
